/* ========================================================================
 * PlantUML : a free UML diagram generator
 * ========================================================================
 *
 * (C) Copyright 2009-2017, Arnaud Roques
 *
 * Project Info:  http://plantuml.com
 * 
 * This file is part of PlantUML.
 *
 * Licensed under The MIT License (Massachusetts Institute of Technology License)
 * 
 * See http://opensource.org/licenses/MIT
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR
 * IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 *
 * Original Author:  Arnaud Roques
 */
package net.sourceforge.plantuml.sequencediagram.graphic;

import java.util.AbstractCollection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import net.sourceforge.plantuml.Dimension2DDouble;
import net.sourceforge.plantuml.graphic.StringBounder;
import net.sourceforge.plantuml.skin.Area;
import net.sourceforge.plantuml.skin.Component;
import net.sourceforge.plantuml.skin.SimpleContext2D;
import net.sourceforge.plantuml.ugraphic.UGraphic;
import net.sourceforge.plantuml.ugraphic.UTranslate;

public class ParticipantBox implements Pushable {

	private static int CPT = 0;

	private final int outMargin;

	private double startingX;

	private final Component head;
	private final Component line;
	private final Component tail;
	private final Component delayLine;

	private int cpt = CPT++;

	public ParticipantBox(Component head, Component line, Component tail, Component delayLine, double startingX, int outMargin) {
		this.outMargin = outMargin;
		this.startingX = startingX;
		this.head = head;
		this.line = line;
		this.tail = tail;
		this.delayLine = delayLine;
	}

	@Override
	public String toString() {
		return "PB" + cpt;
	}

	public double getMinX() {
		return startingX + outMargin;
	}

	public double getMaxX(StringBounder stringBounder) {
		return startingX + head.getPreferredWidth(stringBounder) + 2 * outMargin;
	}

	public double getCenterX(StringBounder stringBounder) {
		return startingX + head.getPreferredWidth(stringBounder) / 2.0 + outMargin;
	}

	public double getHeadHeight(StringBounder stringBounder) {
		return head.getPreferredHeight(stringBounder) + line.getPreferredHeight(stringBounder) / 2.0;
	}

	public double getHeadHeightOnly(StringBounder stringBounder) {
		return head.getPreferredHeight(stringBounder);
	}

	public double getPreferredWidth(StringBounder stringBounder) {
		return head.getPreferredWidth(stringBounder);
	}

	public double getTailHeight(StringBounder stringBounder) {
		return tail.getPreferredHeight(stringBounder) + line.getPreferredHeight(stringBounder) / 2.0;
	}

	public void pushToLeft(double deltaX) {
		startingX += deltaX;
	}

	public void drawHeadTailU(UGraphic ug, double topStartingY, boolean showHead, double positionTail) {
		if (topStartingY == 0) {
			throw new IllegalStateException("setTopStartingY cannot be zero");
		}

		// final double atX = ug.getTranslateX();
		// final double atY = ug.getTranslateY();
		final StringBounder stringBounder = ug.getStringBounder();

		if (showHead) {
			final double y1 = topStartingY - head.getPreferredHeight(stringBounder)
					- line.getPreferredHeight(stringBounder) / 2;
			head.drawU(
					ug.apply(new UTranslate(getMinX(), y1)),
					new Area(new Dimension2DDouble(head.getPreferredWidth(stringBounder), head
							.getPreferredHeight(stringBounder))), new SimpleContext2D(false));
			// ug.setTranslate(atX, atY);
		}

		if (positionTail > 0) {
			// final double y2 = positionTail - topStartingY +
			// line.getPreferredHeight(stringBounder) / 2 - 1;
			positionTail += line.getPreferredHeight(stringBounder) / 2 - 1;
			// if (y2 != y22) {
			// throw new IllegalStateException();
			// }
			ug = ug.apply(new UTranslate(getMinX(), positionTail));
			tail.drawU(
					ug,
					new Area(new Dimension2DDouble(tail.getPreferredWidth(stringBounder), tail
							.getPreferredHeight(stringBounder))), new SimpleContext2D(false));
			// ug.setTranslate(atX, atY);
		}
	}

	public void drawParticipantHead(UGraphic ug) {
		// ug.translate(outMargin, 0);
		final StringBounder stringBounder = ug.getStringBounder();
		head.drawU(
				ug.apply(new UTranslate(outMargin, 0)),
				new Area(new Dimension2DDouble(head.getPreferredWidth(stringBounder), head
						.getPreferredHeight(stringBounder))), new SimpleContext2D(false));
		// ug.translate(-outMargin, 0);
	}

	public void drawLineUTOBEREMOVED_4243(UGraphic ug, double startingY, double endingY, boolean showTail, double myDelta) {
		throw new UnsupportedOperationException();
//		ug = ug.apply(new UTranslate(startingX, 0));
//		if (delays.size() > 0) {
//			final StringBounder stringBounder = ug.getStringBounder();
//			for (GraphicalDelayText delay : delays) {
//				if (delay.getStartingY() - myDelta >= startingY) {
//					drawLine(ug, startingY, delay.getStartingY() - myDelta, line);
//					drawLine(ug, delay.getStartingY() - myDelta, delay.getEndingY(stringBounder) - myDelta, delayLine);
//					startingY = delay.getEndingY(stringBounder) - myDelta;
//				}
//			}
//			if (delays.get(delays.size() - 1).getEndingY(stringBounder) - myDelta > startingY) {
//				startingY = delays.get(delays.size() - 1).getEndingY(stringBounder) - myDelta;
//			}
//		}
//		drawLine(ug, startingY, endingY, line);
	}

	public void drawLineU22(UGraphic ug, double startingY, final double endingY, boolean showTail, double myDelta) {
		ug = ug.apply(new UTranslate(startingX, 0));
		if (delays.size() > 0) {
			final StringBounder stringBounder = ug.getStringBounder();
			for (GraphicalDelayText delay : delays) {
				if (delay.getStartingY() - myDelta >= startingY) {
					drawLineIfLowerThan(ug, startingY, delay.getStartingY() - myDelta, line, endingY);
					drawLineIfLowerThan(ug, delay.getStartingY() - myDelta, delay.getEndingY(stringBounder) - myDelta,
							delayLine, endingY);
					startingY = delay.getEndingY(stringBounder) - myDelta;
				}
			}
			if (delays.get(delays.size() - 1).getEndingY(stringBounder) - myDelta > startingY) {
				startingY = delays.get(delays.size() - 1).getEndingY(stringBounder) - myDelta;
			}
		}
		drawLineIfLowerThan(ug, startingY, endingY, line, endingY);
	}

	private void drawLineIfLowerThan(UGraphic ug, double startingY, double endingY, Component comp, double limitY) {
		startingY = Math.min(startingY, limitY);
		endingY = Math.min(endingY, limitY);
		if (startingY < limitY || endingY < limitY) {
			drawLine(ug, startingY, endingY, comp);
		}

	}

	private void drawLine(UGraphic ug, double startingY, double endingY, Component comp) {
		final StringBounder stringBounder = ug.getStringBounder();
		comp.drawU(ug.apply(new UTranslate(0, startingY)),
				new Area(new Dimension2DDouble(head.getPreferredWidth(stringBounder) + outMargin * 2, endingY
						- startingY)), new SimpleContext2D(false));
	}

	public double magicMargin(StringBounder stringBounder) {
		return line.getPreferredHeight(stringBounder) / 2;
	}

	public double getStartingX() {
		return startingX;
	}

	private final List<GraphicalDelayText> delays = new ArrayList<GraphicalDelayText>();

	public void addDelay(GraphicalDelayText delay) {
		this.delays.add(delay);
	}

	public Collection<Segment> getDelays(final StringBounder stringBounder) {
		return new AbstractCollection<Segment>() {

			@Override
			public Iterator<Segment> iterator() {
				return new Iterator<Segment>() {

					private final Iterator<GraphicalDelayText> it = delays.iterator();

					public boolean hasNext() {
						return it.hasNext();
					}

					public Segment next() {
						final GraphicalDelayText d = it.next();
						return new Segment(d.getStartingY(), d.getEndingY(stringBounder));
					}

					public void remove() {
						throw new UnsupportedOperationException();
					}
				};
			}

			@Override
			public int size() {
				return delays.size();
			}
		};
	}

}
