//-----------------------------------------------------------------------------
//
//	PICSTART Plus programming interface
//
//-----------------------------------------------------------------------------
//
//	Cosmodog, Ltd.
//	415 West Superior Street
//	Chicago, IL   60610-3429
//	http://www.cosmodog.com
//
//-----------------------------------------------------------------------------
//
//	this interface to the PICSTART Plus programmer is provided in an effort
//	to make the PICSTART (and thus, the PIC family of microcontrollers) useful
//	and accessible across multiple platforms, not just one particularly well-
//	known, unstable one.
//
//-----------------------------------------------------------------------------
//
//	Copyright (C) 1999-2001 Cosmodog, Ltd.
//
//	This program is free software; you can redistribute it and/or
//	modify it under the terms of the GNU General Public License
//	as published by the Free Software Foundation; either version 2
//	of the License, or (at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//-----------------------------------------------------------------------------
//
// revison history
//
//	0.2  (10 Sept. 1999) -- first real release
//
//	0.2a (11 Sept. 1999)
//		removed timebits.h from includes.h (wasn't actually used anywhere)
//
//	0.3  (12 Sept. 1999)
//		cleaned up a couple of mistakes in the usage text
//		added support for many more parts (everything supported by MPLAB)
//
//	0.4  (1 Oct. 1999)
//		cleaned up some argument handling (better, more generalized)
//		broke picstart.c up into smaller modules
//		added support for -wo flag (only works on single-word calibration spaces)
//		added oscillator calibration space size checking
//		added dump of device info if invoked without any flags
//
//	0.4a (28 Oct. 1999)
//		improved hex record parsing
//		added timeout to InitDevice() to allow the PICSTART to respond to DTR
//
//	0.4b (14 July 2000)
//		added support for several additional parts (culled from MPLAB 5.00.00)
//		added extended device profile (or something) to initialization -- newer
//			versions of MPLAB send more stuff at device init; these are now sent
//			by picp as well
//
//	0.4c (14 July, 2000)
//		added support for PIC18C442 and PIC18C452
//
//	0.4d (20 February, 2001)
//		changed CMD_REQUEST_ACK to CMD_REQUEST_MODEL (actually asking for programmer
//			model, not just an acknowledge)
//		dropped baud rates over 115200 from serial.c, they aren't useful here and
//			were limiting cross compatibility
//		added bool as a typedef so picp can be compiled under straight c instead of c++
//		changed Makefile so picp compiles using gcc instead of c++
//		modified OpenDevice() and CloseDevice() to save the previous serial port
//			settings and restore them on exit
//		made some improvements to ConfigureDevice() courtesy Brian Chamberlain
//
//-----------------------------------------------------------------------------
//
// Special thanks to Todd Squires, Brian Chamberlain, and Scott Johnston for
//  various suggestions and contributions
//
//-----------------------------------------------------------------------------
//
//  to do:
//
//		support previous PICSTART versions (maybe)
//		read/write/erase ID locations
//		read/write/erase/verify data space
//		erase config bits
//		erase oscillator calibration (probably unnecessary since none of the current flash devices have calibration space)
//		verify program space
//		support osc. calibration space sizes other than 1 (PIC14000 only?)
//		seems to have trouble verifying oscillator calibration when stringing several operations together
//		seems to have a problem with 16C63A (but not 16C63, which MPLAB appears to treat as identical)
//		need to be able to write motorola S records
//
//-----------------------------------------------------------------------------

#include "includes.h"

#define CHAR_TIMEOUT	500000				// time to wait for a character before giving up (in microseconds)
#define BUFFERSIZE		0x2000				// buffer bigger than anyone should need

// COMMANDS (sent to PICSTART)
//
#define CMD_BLANK_CHECK		'B'				// blank check (0x0f = F not blank, 0x17 = not blank, 0x10 = blank?)
#define CMD_WRITE_PGM		'Q'				// write program memory
#define CMD_READ_PGM		'T'				// read program memory
#define CMD_READ_DATA		'd'				// read data (EEPROM) memory
#define CMD_READ_ID			'e'				// read ID locations
#define CMD_READ_CFG		'f'				// request configuration bits
#define CMD_READ_OSC		'c'				// read oscillator calibration memory
#define CMD_WRITE_CFG		'g'				// write configuration bits
#define CMD_WRITE_OSC		'q'				// write oscillator calibration memory
#define CMD_LOAD_INFO		0x81			// send processor-specific info
#define CMD_LOAD_EXT_INFO	0x82			// send more processor-specific info (this is new as of 0.4b)
#define CMD_REQUEST_MODEL	0x88			// request programmer model
#define CMD_REQUEST_VERSION	0x8d			// request firmware version
#define CMD_SET_ADDR		0x8e			// set address range (start address, size)

#define PIC_ACK				0xab			// response from picstart plus to model request (only programmer supported)



static const char versionString[] =	"0.4d";	// this program's version number

#define MIN_MAJORVER	2					// version number detected must be at least this high
#define MIN_MIDVER		1
#define MIN_MINORVER	0

#define MINIMUM_VER		(MIN_MAJORVER*65536 + MIN_MIDVER*256 + MIN_MINORVER)

#define HASH_WIDTH_DEFAULT		40			// default width of the status bar

typedef struct
{
	UINT8 major;
	UINT8 middle;
	UINT8 minor;
}VERSION;

static VERSION PICversion;


extern int errno;

char *programName;
char *deviceName;
char*picName;

bool done;
int terminalDevice;

struct termios
	terminalParams,
	TTYParams,
	oldTTYInParams,
	oldTTYOutParams;

static bool ignoreVerfErr;
static int theDevice;
static UINT16 readConfigBits;	// config bits read back from device
static UINT32 hashWidth;		// width of status bar (0 = none)


//-----------------------------------------------------------------------------
//
// Find the requested PIC device from the list of supported devices.
// If the user didn't prepend 'PIC' to the device name skip the
// first three letters of the devices when comparing.  This allows
// both PIC16C505 or 16C505 to refer to the same part.
//
//-----------------------------------------------------------------------------

const static PIC_DEFINITION *GetPICDefinition(char *name)
{
	int idx = 0;
	int offset = 0;							// assume no PIC at the front

	while(name[idx])						// ensure it's all upper case
	{
		name[idx] = toupper(name[idx]);
		idx++;
	}
	if(strncmp(name,"PIC",3) == 0)			// if the user's argument has 'PIC' at the front,
	{
		offset = 3;							// skip the first three letters of the user's argument
	}
	idx = 0;
	while(deviceList[idx])
	{
		if(strcmp(deviceList[idx]->name,name+offset) == 0)
		{
			break;
		}
		idx++;
	}
	return(deviceList[idx]);		// return 0 if no match
}


//-----------------------------------------------------------------------------
//
// return the size of the program space of the specified device (in words)
//
//-----------------------------------------------------------------------------

static UINT16 GetPgmSize(const PIC_DEFINITION *picDevice)
{
	return(picDevice->def[PD_PGM_SIZEH]*256 + picDevice->def[PD_PGM_SIZEL]);
}


//-----------------------------------------------------------------------------
//
// return the size of the data space of the specified device (in bytes)
//
//-----------------------------------------------------------------------------

static UINT16 GetDataSize(const PIC_DEFINITION *picDevice)
{
	return(picDevice->def[PD_DATA_SIZEH]*256 + picDevice->def[PD_DATA_SIZEL]);
}


//-----------------------------------------------------------------------------
//
// return the start address of the data space of the specified device
//
//-----------------------------------------------------------------------------

static UINT16 GetDataStart(const PIC_DEFINITION *picDevice)
{
	return(picDevice->def[PD_DATA_ADDRH]*256 + picDevice->def[PD_DATA_ADDRL]);
}


//-----------------------------------------------------------------------------
//
// return the size of the oscillator calibration space of the specified device (in words)
//
//-----------------------------------------------------------------------------

static UINT16 GetOscCalSize(const PIC_DEFINITION *picDevice)
{
	return(picDevice->def[PD_CLK_SIZEH]*256 + picDevice->def[PD_CLK_SIZEL]);
}


//-----------------------------------------------------------------------------
//
// return the start address of the oscillator calibration space of the specified device
//
//-----------------------------------------------------------------------------

static UINT16 GetOscCalStart(const PIC_DEFINITION *picDevice)
{
	return(picDevice->def[PD_CLK_ADDRH]*256 + picDevice->def[PD_CLK_ADDRL]);
}

//-----------------------------------------------------------------------------
//
// return the start address of the configuration bits
//
//-----------------------------------------------------------------------------

static UINT16 GetConfigStart(const PIC_DEFINITION *picDevice)
{
	return(picDevice->def[PD_CFG_ADDRH]*256 + picDevice->def[PD_CFG_ADDRL]);
}



//-----------------------------------------------------------------------------
//
//	send a message to the PICSTART, wait for a specified number of bytes to be
//		returned
//
//-----------------------------------------------------------------------------

static bool SendMsg(const UINT8 *cmdBuff, UINT32 cmdBytes, UINT8 *rtnBuff, UINT32 rtnBytes)
{
	bool fail = false;
	int numRead;
	int bytesRemaining = rtnBytes;

	if(cmdBytes)
	{
		write(theDevice,&cmdBuff[0],cmdBytes);								// send out the command
	}

	while(bytesRemaining && !fail)
	{
		numRead = ReadBytes(theDevice,&rtnBuff[rtnBytes-bytesRemaining],bytesRemaining,CHAR_TIMEOUT);

		if(numRead < 0)
		{
			ReportError("error %d, %s\n",errno,strerror(errno));
			fail = true;
		}
		else if(numRead == 0)		// timed out
		{
			ReportError("no response from PICSTART\n");
			fail = true;
		}
		bytesRemaining = (bytesRemaining > numRead) ? bytesRemaining - numRead : 0;		// subtract bytes read in, don't allow to underflow (shouldn't happen)
	}
	return(!fail);
}


//-----------------------------------------------------------------------------
//
//	ask what type of programmer is attached (if any)
//
//  currently only the PICSTART Plus is supported; this is where MPLAB tries
//    to identify what type of programmer is attached and selects the
//    appropriate protocol based on the response.  Since only the PICSTART
//    Plus is supported we just check for its response code and fail if we
//    don't get it.
//
//-----------------------------------------------------------------------------

static bool DoGetProgrammerType()
{
	bool fail;
	UINT8 theBuffer[1];

	theBuffer[0] = CMD_REQUEST_MODEL;
	fail = !SendMsg(theBuffer,1,theBuffer,1);
	fail |= (theBuffer[0] != PIC_ACK);			// fail if not PICSTART Plus
	return(!fail);
}


//-----------------------------------------------------------------------------
//
//	request version from the PICSTART
//
//-----------------------------------------------------------------------------

static bool DoGetVersion()
{
	bool fail = false;
	UINT8 theBuffer[4];

	theBuffer[0] = CMD_REQUEST_VERSION;
	if( !(fail = !SendMsg(theBuffer,1,theBuffer,4)) )
	{
		if(theBuffer[0] != CMD_REQUEST_VERSION)
		{
			fail = true;
		}
		else
		{
			PICversion.major = theBuffer[1];
			PICversion.middle = theBuffer[2];
			PICversion.minor = theBuffer[3];
		}
	}
	return(!fail);
}


//-----------------------------------------------------------------------------
//
//	get the version from the PICSTART and display it
//
//-----------------------------------------------------------------------------

static bool DoShowVersion()
{
	bool fail = !DoGetVersion();

	if(!fail)
	{
		ReportMessage("PICSTART Plus firmware version %d.%02d.%02d\n",PICversion.major,PICversion.middle,PICversion.minor);
	}
	return(!fail);
}


//-----------------------------------------------------------------------------
//
//  QualifyVersion -- verify that the version number of the attached PICSTART
//		is high enough to work with this program, return false if it is not
//
//-----------------------------------------------------------------------------

static bool QualifyVersion(VERSION *PICversion)
{
	return( (PICversion->major*65536 + PICversion->middle*256 + PICversion->minor) >= MINIMUM_VER);	// true if >= min
}





typedef UINT16 SIZEFNCT(const PIC_DEFINITION *);

typedef struct
{
	UINT8 mask;
	SIZEFNCT *sizef;				// pointer to function to return the size of the space (0=no function)
	char *name;
} BLANK_MSG;


const static BLANK_MSG blankList[] =
{
	{BLANK_PGM,		&GetPgmSize,	"program memory"},
	{BLANK_CFG,		0,				"configuration bits"},
	{BLANK_ID,		0,				"ID locations"},
	{BLANK_DATA,	&GetDataSize,	"data memory"},
	{0,0,""},
};


static bool DoBlankCheck(const PIC_DEFINITION *picDevice, UINT8 blankMode)
{
	bool fail = false;
	UINT8 theBuffer[3];

	theBuffer[0] = CMD_BLANK_CHECK;
	fail = !SendMsg(theBuffer,1,theBuffer,2);
	if(!fail)
	{
		int idx = 0;
		theBuffer[1] &= blankMode;										// look only at what we were asked to look at
		if(!GetMessageDisplay())
		{
			fprintf(stdout,"0x%02x\n",theBuffer[1]);					// quiet mode will just show the return code
		}
		else
		{
			while(blankList[idx].mask)									// report anything that isn't blank
			{
				if(blankMode & blankList[idx].mask)						// check only what we were asked
				{
					if(!blankList[idx].sizef || (blankList[idx].sizef(picDevice) > 0))
					{
						ReportMessage("%s: ",blankList[idx].name);
						if(theBuffer[1] & blankList[idx].mask)			// if a bit is set,
						{
							ReportMessage("not blank\n");
							fail = true;								// something wasn't blank
						}
						else
						{
							ReportMessage("blank\n");
						}
					}
				}
				idx++;
			}
		}
	}
	return(!fail);
}



#if 0		// this waits until we know how big the EEPROM space should be
static bool DoReadData(const PIC_DEFINITION *picDevice, FILE *theFile)
{
	bool fail = false;
	UINT8 *theBuffer;

	theBuffer = (UINT8 *)malloc(BUFFERSIZE);					// get a needlessly large buffer
	if(theBuffer)
	{
		theBuffer[0] = CMD_READ_DATA;
		fail = !SendMsg(theBuffer,1,theBuffer,?);			// ask it to fill the buffer (plus the command plus a terminating zero)
		if(!fail)
		{
			WriteHexRecord(theFile,&theBuffer[1],0,size);			// write hex records to selected stream
		}
		free(theBuffer);
	}
	else
	{
		fail = true;				// failed to malloc
	}
	return(!fail);
}
#endif




//--------------------------------------------------------------------
//
//  status bar handling
//

static UINT16 hashMod;
static UINT16 hashNum;

// initialize a status bar, given the number
// of operations expected
//
static void InitHashMark(UINT16 numOps, UINT16 hashWidth)
{
	hashNum = 0;
	if(hashWidth)							// if width = zero do nothing (no bar)
	{
		if(hashWidth <= numOps)				// if it will create less than 1 mark per operation
		{
			hashMod = numOps/hashWidth;		// mod is the number of operations divided by the bar width
		}
		else
		{
			hashMod = 1;					// don't allow the bar to be longer than the number of operations
		}
	}
	else
	{
		hashMod = 0;						// no bar, no mod
	}
}


// advance the status bar if the current number
// of operations warrants it
//
static void ShowHashMark(UINT16 curOps)
{
	if(hashMod && (curOps/hashMod > hashNum) )
	{
		ReportMessage("#");
		fflush(stdout);						// be sure it gets displayed right away
		hashNum++;
	}
}

// finish up a status bar
//
static void UnInitHashMark()
{
	if(hashMod)
	{
		ReportMessage("\n");
	}
}
//
//--------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// read oscillator calibration
//
static bool DoReadOscCal(const PIC_DEFINITION *picDevice)
{
	bool fail = false;
	UINT8 rangeBuffer[] =
	{
		CMD_SET_ADDR,
		picDevice->def[PD_CLK_ADDRH],
		picDevice->def[PD_CLK_ADDRL],
		picDevice->def[PD_CLK_SIZEH],
		picDevice->def[PD_CLK_SIZEL]
	};
	UINT8 rtnBuffer[sizeof(rangeBuffer)];
	UINT8 *theBuffer;
	UINT16 size = GetOscCalSize(picDevice);

	if(size)
	{
		fail = !SendMsg(rangeBuffer,sizeof(rangeBuffer),rtnBuffer,sizeof(rtnBuffer));
		if(!fail)
		{
			fail = memcmp(rangeBuffer,rtnBuffer,sizeof(rangeBuffer)) != 0;
			if(!fail)
			{
				theBuffer = (UINT8 *)malloc(size+2);						// get a buffer this big plus one char for the command and a 0 at the end
				if(theBuffer)
				{
					theBuffer[0] = CMD_READ_OSC;
					fail = !SendMsg(theBuffer,1,theBuffer,size+2);			// ask it to fill the buffer (plus the command plus a terminating zero)
					if(!fail)
					{
						int idx;
						ReportMessage("oscillator calibration: ");
						for(idx = 1; idx<size+1; idx+=2)
						{
							if( (((idx-1)/2)&8) == 0 )
							{
								ReportMessage("\n");
							}
							ReportMessage(" 0x%02x%02x",theBuffer[idx],theBuffer[idx+1]);
						}
						ReportMessage("\n");
					}
				}
				else
				{
					ReportError("failed to malloc\n");
					fail = true;
				}
			}
		}
	}
	else
	{
		ReportError("device %s has no oscillator calibration space\n",picDevice->name);
		fail = true;
	}
	return(!fail);
}


//-----------------------------------------------------------------------------
//
// write oscillator calibration
//
static bool DoWriteOscCalBits(const PIC_DEFINITION *picDevice, UINT16 oscCalBits)
{
	bool fail = false;
	UINT8 theBuffer[3];
	UINT8 rtnBuffer[4];
	UINT16 size = GetOscCalSize(picDevice);		// size of the calibration space

	if(size == 1)
	{
		theBuffer[0] = CMD_WRITE_OSC;
		theBuffer[1] = (oscCalBits>>8) & 0xff;
		theBuffer[2] = oscCalBits & 0xff;
		fail = !SendMsg(theBuffer,3,rtnBuffer,4);
		if(!fail)
		{
			if(theBuffer[0] != rtnBuffer[0] || theBuffer[1] != rtnBuffer[1] || theBuffer[2] != rtnBuffer[2] || rtnBuffer[3] != 0)
			{
				fail = true;
				ReportError("failed to verify while writing oscillator calibration data\n");
			}
		}
		else
		{
			fail = true;
		}
	}
	else if(size == 0)
	{
		ReportError("device %s has no oscillator calibration space\n",picDevice->name);
		fail = true;
	}
	else
	{
		ReportError("oscillator calibration space sizes other than 1 not supported yet\n");
		fail = true;
	}
	return(!fail);
}


//-----------------------------------------------------------------------------
//
// read configuration bits
//
static bool DoReadCfg(bool verbose)
{
	bool fail = false;
	UINT8 theBuffer[4];

	theBuffer[0] = CMD_READ_CFG;
	fail = !SendMsg(theBuffer,1,theBuffer,4);
	if(!fail)
	{
		if( (theBuffer[0] != CMD_READ_CFG) || (theBuffer[3] != 0) )
		{
			ReportError("failed to read configuration bits\n");
			fail = true;
		}
		else
		{
			readConfigBits = theBuffer[1]*256 + theBuffer[2];
			if(verbose)
			{
				ReportMessage("configuration bits: ");
				fprintf(stdout,"0x%04x\n",readConfigBits);
			}
		}
	}
	return(!fail);
}

// erase the program space of a part that can be erased (PIC16Fxx, etc)
//
static bool DoErasePgm(const PIC_DEFINITION *picDevice)
{
	bool fail = false;
	UINT8 rangeBuffer[] = {CMD_SET_ADDR,0,0,picDevice->def[PD_PGM_SIZEH],picDevice->def[PD_PGM_SIZEL]};		// write to the whole program space
	UINT8 rtnBuffer[sizeof(rangeBuffer)];
	UINT8 theBuffer[4];
	UINT16 size = GetPgmSize(picDevice)*2;			// size of the device's program memory (in bytes)
	int byteCnt;

	InitHashMark(size, hashWidth);
	fail = !SendMsg(rangeBuffer,sizeof(rangeBuffer),rtnBuffer,sizeof(rtnBuffer));
	if(!fail)
	{
		if(memcmp(rangeBuffer,rtnBuffer,sizeof(rangeBuffer)) == 0)
		{
			theBuffer[0] = CMD_WRITE_PGM;
			fail = !SendMsg(&theBuffer[0],1,rtnBuffer,1);		// send the command, watch for it to bounce back
			if(*rtnBuffer != CMD_WRITE_PGM)
			{
				fail = true;
			}
			theBuffer[0] = 0xff;							// send as all 1's, even if device isn't 16 bits wide
			theBuffer[1] = 0xff;
			byteCnt = 0;
			while( (byteCnt < size) && !fail)
			{
				fail = !SendMsg(theBuffer,2,rtnBuffer,2);	// write the bytes, ignore the return value (check results later)
				byteCnt+=2;
				ShowHashMark(byteCnt);
			}
			UnInitHashMark();
			fail = !SendMsg(theBuffer,0,rtnBuffer,1);		// eat the trailing zero
			if(*rtnBuffer != 0)
			{
				fail = true;								// fail if it's not zero
			}
			else
			{
				fail = !DoBlankCheck(picDevice, BLANK_PGM);	// see if it's blank
				if(fail)
				{
					ReportError("failed to erase program space\n");
				}
			}
		}
		else
		{
			ReportError("didn't echo range properly\n");
			fail = true;									// didn't echo everthing back like it should have
		}
	}
	else
	{
		ReportError("failed to send range\n");
		fail = true;										// didn't echo everthing back like it should have
	}
	return(!fail);
}


static bool DoWriteConfigBits(const PIC_DEFINITION *picDevice, UINT16 configBits)
{
	bool fail = false;
	UINT8 theBuffer[3];
	UINT8 rtnBuffer[3];

	// DEBUG should do processor-specific range check here, if appropriate
	theBuffer[0] = CMD_WRITE_CFG;
	theBuffer[1] = (configBits>>8) & 0xff;
	theBuffer[2] = configBits & 0xff;
	fail = !SendMsg(theBuffer,3,rtnBuffer,3);
	if(!fail)
	{
		if(theBuffer[0] != rtnBuffer[0] || theBuffer[1] != rtnBuffer[1] || theBuffer[2] != rtnBuffer[2])
		{
			fail = true;
			ReportError("failed to verify while writing configuration bits\n");
		}
	}
	else
	{
		fail = true;
	}
	return(!fail);
}


#if 0	// not done yet...
static bool DoWriteIDLocs(const PIC_DEFINITION *picDevice, UINT16 configBits)
{
	bool fail = false;
	UINT8 theBuffer[3];
	UINT8 rtnBuffer[3];

	// DEBUG should do processor-specific range check here, if appropriate
	theBuffer[0] = CMD_WRITE_CFG;
	theBuffer[1] = (configBits>>8) & 0xff;
	theBuffer[2] = configBits & 0xff;
	fail = !SendMsg(theBuffer,3,rtnBuffer,3);
	if(!fail)
	{
		if(theBuffer[0] != rtnBuffer[0] || theBuffer[1] != rtnBuffer[1] || theBuffer[2] != rtnBuffer[2])
		{
			fail = true;
			ReportError("failed to verify while writing configuration bits\n");
		}
	}
	else
	{
		fail = true;
		ReportError("Configuration bits value out of range: 0x%04x\n",configBits);
	}
	return(!fail);
}
#endif



// copy buffer to device, starting at word address startAddr_w, running for size_w words
//  DOES NOT boundary-check range -- will attempt to write outside of device's memory
//  Returns true if okay, false if failed
//  Verify error counts as failure only if failOnVerf = true
//
static bool WritePgmRange(UINT16 startAddr_w, UINT16 size_w, UINT8 *buffer)
{
	bool fail = false;
	bool verifyFail = false;
	UINT8 rangeBuffer[5];
	UINT8 rtnBuffer[sizeof(rangeBuffer)];

	rangeBuffer[0] = CMD_SET_ADDR;
	rangeBuffer[1] = (startAddr_w>>8) & 0xff;
	rangeBuffer[2] = startAddr_w & 0xff;
	rangeBuffer[3] = (size_w>>8) & 0xff;
	rangeBuffer[4] = size_w & 0xff;

	fail = !SendMsg(rangeBuffer,sizeof(rangeBuffer),rtnBuffer,sizeof(rtnBuffer));
	if(!fail)
	{
		fail = !(memcmp(rangeBuffer,rtnBuffer,sizeof(rangeBuffer)) == 0);
		if(!fail)
		{
			int idx = 0;

			while(idx < (size_w*2) )
			{
				UINT8 temp;
				temp = buffer[idx+1];
				buffer[idx+1] = buffer[idx];				// swap byte order (make it little endian)
				buffer[idx] = temp;
				idx += 2;
			}

			*rangeBuffer = CMD_WRITE_PGM;					// add in the command
			fail = !SendMsg(rangeBuffer,1,rtnBuffer,1);		// send the command, watch for it to bounce back
			if(*rtnBuffer != CMD_WRITE_PGM)
			{
				fail = true;
			}

			idx = 0;
			while( !fail && (idx < (size_w*2)) )
			{
				fail = !SendMsg(&buffer[idx],2,rtnBuffer,2);
				if( (buffer[idx] != rtnBuffer[0]) || (buffer[idx+1] != rtnBuffer[1]) )
				{
					verifyFail = true;						// didn't get back what we sent
				}
				idx+=2;
				ShowHashMark(idx);
			}
			fail = !SendMsg(buffer,0,rtnBuffer,1);			// eat the trailing zero
			if(verifyFail)
			{
				if(!ignoreVerfErr)
				{
					ReportError("failed to verify while writing to program space\n");
				}
				else
				{
					fprintf(stderr,"Warning: failed to verify while writing to program space\n");	// report it but don't fail on it
				}
			}
		}
		else
		{
			ReportError("didn't echo range properly\n");
		}
	}
	return( !(fail || (!ignoreVerfErr && verifyFail)) );
}


static bool DoWritePgm(const PIC_DEFINITION *picDevice, FILE *theFile)
{
	bool fail = false;
	bool fileDone = false;
	UINT8 *theBuffer;
	UINT16 startAddr;		// hex record returns byte address, device expects word address -- always need to translate
	UINT16 curAddr;
	UINT32 nextAddr;
	UINT16 size;
	UINT8 data;

	theBuffer = (UINT8 *)malloc(BUFFERSIZE);
	fail = (theBuffer == 0);
	if(!fail)
	{
		InitHashMark(picDevice->def[PD_PGM_SIZEH]*256+picDevice->def[PD_PGM_SIZEL], hashWidth);		// go to too much effort to set the width

		InitParse();														// get ready to start reading the hex file

		fileDone = !GetNextByte(theFile, &nextAddr, &data);					// get a byte and the initial address
		while(!fileDone && !fail)
		{
			startAddr = nextAddr;											// the first address of the new block
			curAddr = startAddr;
			theBuffer[0] = data;											// the first data byte of the new block
			size = 1;														// number of bytes waiting to be sent
			while( !(fileDone = !GetNextByte(theFile, &nextAddr, &data)) )	// get next byte
			{
				if(curAddr+1 == (UINT16)nextAddr)							// use this byte now only if it's sequential
				{
					curAddr++;
					theBuffer[size] = data;
					size++;
				}
				else
				{
					break;													// non-sequential, write this buffer then start another
				}
			}
			if(size & 1)
			{
				size++;														// don't allow odd sizes
				theBuffer[size] = 0xff;										// pad with 0xff (erased)
			}
			{
				// DEBUG check if config address is embedded in range
				//
				UINT16 devCfgAddr = picDevice->def[PD_CFG_ADDRH]*256 + picDevice->def[PD_CFG_ADDRL];	// word address of device's config memory
				if( startAddr/2 == devCfgAddr)															// at config bits address?
				{
					if(size == 2)																		// must be exactly one word
					{																					// inside configuration space, write as configuration
						fail = !DoWriteConfigBits(picDevice, theBuffer[0] + theBuffer[1]*256);			// (little endian, data already swapped)
					}
					else
					{
						ReportError("configuration word must be exactly 16 bits\n");
					}
				}
				else
				{
					fail = !WritePgmRange(startAddr/2, size/2, theBuffer);	// if not config bits try to write where ever it lands
				}
			}
		}
		UnInitHashMark();
		free(theBuffer);
	}
	else
	{
		ReportError("failed to malloc %d bytes\n",BUFFERSIZE);
	}
	return(!fail);
}


// DEBUG always reads entire device -- may want to allow reading of a specific range
//
static bool DoReadPgm(const PIC_DEFINITION *picDevice, FILE *theFile)
{
	bool fail = false;
	UINT8 rangeBuffer[] = {CMD_SET_ADDR,0,0,picDevice->def[PD_PGM_SIZEH],picDevice->def[PD_PGM_SIZEL]};		// DEBUG this is default -- user selectable?
	UINT8 rtnBuffer[sizeof(rangeBuffer)];
	UINT8 *theBuffer;
	UINT16 size = GetPgmSize(picDevice)*2;								// size of the device's program memory (in bytes)

	fail = !DoReadCfg(false);											// read config bits silently
	if(!fail)
	{
		if(~readConfigBits & picDevice->cpbits)
		{
			fprintf(stderr,"Warning: device is code protected: configuration bits = 0x%04x\n",readConfigBits);
		}
		fail = !SendMsg(rangeBuffer,sizeof(rangeBuffer),rtnBuffer,sizeof(rtnBuffer));
		if(!fail)
		{
			if(memcmp(rangeBuffer,rtnBuffer,sizeof(rangeBuffer)) == 0)
			{
				theBuffer = (UINT8 *)malloc(size+2);						// get a buffer this big plus one char for the command and a 0 at the end
				if(theBuffer)
				{
					theBuffer[0] = CMD_READ_PGM;
					fail = !SendMsg(theBuffer,1,theBuffer,size+2);			// ask it to fill the buffer (plus the command plus a terminating zero)
					if(!fail)
					{
						// DEBUG shouldn't need to swap byte order here but we do
						int idx;
						UINT8 temp;
						for(idx = 1; idx<size+1; idx+=2)
						{
							temp = theBuffer[idx+1];
							theBuffer[idx+1] = theBuffer[idx];				// swap byte order (make it little endian)
							theBuffer[idx] = temp;
						}
						WriteHexRecord(theFile,&theBuffer[1],0,size);		// write hex records to selected stream
					}
					free(theBuffer);
				}
				else
				{
					fail = true;				// failed to malloc
				}
			}
			else
			{
				fail = true;					// didn't echo everthing back like it should have
			}
		}
	}
	return(!fail);
}


static bool DoReadID()
{
	bool fail = false;
	UINT8 theBuffer[10];

	theBuffer[0] = CMD_READ_ID;
	fail = !SendMsg(theBuffer,1,theBuffer,10);
	if(!fail)
	{
		if( (theBuffer[0] != CMD_READ_ID) || (theBuffer[9] != 0) )
		{
			ReportError("failed to read ID locations\n");
			fail = true;
		}
		else
		{
			ReportMessage("ID locations: ");							// if in quiet mode, only the values will be returned
			fprintf(stdout,"0x%02x%02x ",theBuffer[1],theBuffer[2]);
			fprintf(stdout,"0x%02x%02x ",theBuffer[3],theBuffer[4]);
			fprintf(stdout,"0x%02x%02x ",theBuffer[5],theBuffer[6]);
			fprintf(stdout,"0x%02x%02x",theBuffer[7],theBuffer[8]);
			ReportMessage("\n");
		}
	}
	return(!fail);
}




//-----------------------------------------------------------------------------
//
// initialize for specified part, return true if succeeded
//
static bool DoInitPIC(const PIC_DEFINITION *picDevice)
{
	bool fail;
	UINT8 theBuffer[3];
	UINT8 *cmdBuffer;

	theBuffer[0] = CMD_LOAD_INFO;
	fail = !SendMsg(theBuffer,1,theBuffer,1);								// send load processor info command, wait for command to echo back
	if(!fail && (theBuffer[0] == CMD_LOAD_INFO) )
	{
		if( (cmdBuffer = (UINT8 *)malloc(PICDEV_DEFSIZE+1)) )
		{
			int idx;
			memcpy(cmdBuffer,picDevice->def,PICDEV_DEFSIZE);			// copy definition into the new buffer
			cmdBuffer[PICDEV_DEFSIZE] = 0;								// initialize the checksum
			for(idx = 0; idx < PICDEV_DEFSIZE; idx++)					// calculate the checksum, store as last byte in buffer
			{
				cmdBuffer[PICDEV_DEFSIZE] += cmdBuffer[idx];
			}
			fail = !SendMsg(cmdBuffer,PICDEV_DEFSIZE+1,theBuffer,1);	// send whole buffer including checksum
			if(!fail)
			{
				if(theBuffer[0] != 0)										// zero = checksum okay (data received okay)
				{
					fail = true;											// didn't receive zero, fail
				}
			}
			free(cmdBuffer);												// release the buffer

			if(!fail)
			{
				theBuffer[0] = CMD_LOAD_EXT_INFO;
				fail = !SendMsg(theBuffer,1,theBuffer,1);					// send load extended processor info command, wait for command to echo back
				if(!fail && (theBuffer[0] == CMD_LOAD_EXT_INFO) )
				{
					if( (cmdBuffer = (UINT8 *)malloc(PICDEV_DEFXSIZE+1)) )
					{
						int idx;
						memcpy(cmdBuffer,picDevice->defx,PICDEV_DEFXSIZE);			// copy definition into the new buffer
						cmdBuffer[PICDEV_DEFXSIZE] = 0;								// initialize the checksum
						for(idx = 0; idx < PICDEV_DEFXSIZE; idx++)					// calculate the checksum, store as last byte in buffer
						{
							cmdBuffer[PICDEV_DEFXSIZE] += cmdBuffer[idx];
						}
						fail = !SendMsg(cmdBuffer,PICDEV_DEFXSIZE+1,theBuffer,1);	// send whole buffer including checksum
						if(!fail)
						{
							if(theBuffer[0] != 0)									// zero = checksum okay (data received okay)
							{
								fail = true;										// didn't receive zero, fail
							}
						}
						free(cmdBuffer);											// release the buffer
					}
					else
					{
						fail = true;												// failed to malloc
					}
				}
			}
		}
		else
		{
			fail = true;													// failed to malloc
		}
	}
	return(!fail);
}


//--------------------------------------------------------------------------
// find out if the next argument is a flag (not preceeded by '-').
// 	if it is, return a pointer to it and advance to the next argument, otherwise
//	return null.

static char *GetNextFlag(int *argc, char **argv[])
{
	if(*argc  && (***argv != '-'))						// if the next argument isn't preceeded by a '-'
	{
		(*argv)++;										// advance to next argument
		(*argc)--;
		return(*(*argv-1));								// but return pointer to this one
	}
	else
	{
		return(NULL);
	}
}


static bool DoTasks(int *argc,char **argv[],const PIC_DEFINITION *picDevice,char *flags)
{
	bool fail = false;
	char *fileName = (char *)0;
	FILE *theFile;
	UINT8 blankMode;


	switch(*flags)
	{
		case 'b':				// blank check
			flags++;
			if(*flags)
			{
				blankMode = 0;
				while(*flags)
				{
					switch(*flags)
					{
						case 'p':
							blankMode |= BLANK_PGM;
							break;

						case 'c':
							blankMode |= BLANK_CFG;
							break;

						case 'i':
							blankMode |= BLANK_ID;
							break;

						case 'd':
							blankMode |= BLANK_DATA;
							break;

						default:
							break;				// ignore undefined flags

					}
					flags++;
				}
			}
			else
			{
				blankMode = BLANK_PGM | BLANK_CFG | BLANK_ID | BLANK_DATA;	// no mode flags means check them all
			}
			fail = !DoBlankCheck(picDevice,blankMode);
			break;

		case 'e':
			flags++;
			if(*flags)
			{
				while(*flags && !fail)
				{
					switch(*flags)
					{
						case 'p':
							fail = !DoErasePgm(picDevice);
							break;

						case 'c':
							ReportError("erase config bits not implemented yet\n");
							break;

						case 'i':
							ReportError("erase ID locations not implemented yet\n");
							break;

						case 'd':
							ReportError("erase data memory not implemented yet\n");
							break;

						case 'o':
							ReportError("erase oscillator calibration not implemented yet\n");	// DEBUG can osc cal be erased/written on flash parts?
							break;

					}
					flags++;
				}
			}
			else
			{
				ReportError("specify one or more regions to erase (p|c|i|d|o)\n");
			}
			break;

		case 'r':
			flags++;
			if(*flags)
			{
				while(*flags && !fail)
				{
					switch(*flags)
					{
						case 'p':
							if( (fileName = GetNextFlag(argc,argv)) )
							{
								theFile = OpenWriteFile(fileName,0x10000);
							}
							else
							{
								theFile = stdout;
							}
							if(theFile)
							{
								fail = !DoReadPgm(picDevice, theFile);		// read program data, write to stream
								if(theFile != stdout)						// if we wrote it to a file,
								{
									CloseFile(theFile);						// close the file
								}
							}
							else
							{
								ReportError("unable to open output file: '%s'\n",fileName);
							}
							break;

						case 'c':
							fail = !DoReadCfg(true);		// read configuration bits, display them
							break;

						case 'i':
							fail = !DoReadID();				// read ID locations
							break;

						case 'd':
							ReportError("reading data memory not implemented yet\n");
							// read data memory
							break;

						case 'o':
							fail = !DoReadOscCal(picDevice);
							break;

					}
					flags++;
				}
			}
			else
			{
				ReportError("specify one or more regions to read (p|c|i|d|o)\n");
			}
			break;

		case 'w':
			flags++;
			if(*flags)
			{
				switch(*flags)
				{
					case 'p':
						if( (fileName = GetNextFlag(argc,argv)) )
						{
							theFile = OpenReadFile(fileName,0x10000);
						}
						else
						{
							theFile = stdin;
						}
						if(theFile)
						{
							fail = !DoWritePgm(picDevice, theFile);
							if(theFile != stdin)						// if we read it from a file,
							{
								CloseFile(theFile);						// close the file
							}
						}
						else
						{
							ReportError("unable to open input file: '%s'\n",fileName);
						}
						break;

					case 'c':
						if( (fileName = GetNextFlag(argc,argv)) )	// 'fileName' is actually the next argument
						{
							UINT32 configBits;
							fail = !atoi_base(fileName,&configBits);
							if(!fail)
							{
								if(configBits < 0x10000)
								{
									fail = !DoWriteConfigBits(picDevice, (UINT16)configBits);
								}
								else
								{
									ReportError("Value out of range: '%s'\n",fileName);
									fail = true;
								}

							}
							else
							{
								ReportError("Unable to interpret '%s' as a numerical value\n",fileName);
							}
						}
						else
						{
							ReportError("write configuration bits must be followed by a numerical value\n");
						}
						break;

					case 'i':			// if writing ID locations, the next four arguments must be numbers
						ReportError("writing ID locations not implemented yet\n");
						break;

					case 'd':
						ReportError("writing data memory not implemented yet\n");
						break;

					case 'o':
						if( (fileName = GetNextFlag(argc,argv)) )	// 'fileName' is actually the next argument
						{
							UINT32 oscCalBits;
							fail = !atoi_base(fileName,&oscCalBits);
							if(!fail)
							{
								if(oscCalBits < 0x10000)
								{
									fail = !DoWriteOscCalBits(picDevice, (UINT16)oscCalBits);
								}
								else
								{
									ReportError("Value out of range: '%s'\n",fileName);
									fail = true;
								}

							}
							else
							{
								ReportError("Unable to interpret '%s' as a numerical value\n",fileName);
							}
						}
						else
						{
							ReportError("write oscillator calibration must be followed by a numerical value\n");
						}
						break;

					default:
						ReportError("must specify a region to write\n");
						break;

				}
			}
			else
			{
				ReportError("specify a region to write (p|c|i|d|o)\n");
			}
			break;

		case 'v':
			ReportError("DEBUG verify is not implemented yet\n");
			// DEBUG verify device goes here
			break;

		default:
			break;

	}
	return(!fail);
}


//-----------------------------------------------------------------------------
//
//	InitDevice -- initialize the serial port
//
//		Once the device is opened and locked, this sets
//		up the port, and makes sure the handshake looks good.
//
//-----------------------------------------------------------------------------

static bool InitDevice(int theDevice,UINT32 baudRate,UINT8 dataBits,UINT8 stopBits,UINT8 parity)
{
	bool fail = false;								// haven't failed (yet)
	bool CTS;
	bool DCD;
	UINT16 ctsTimeOut;

	if(ConfigureDevice(theDevice,baudRate,dataBits,stopBits,parity,false))	// set up the device
	{
		if(ConfigureFlowControl(theDevice,false))		// no flow control at the moment (raise RTS)
		{
			SetDTR(theDevice,true);						// raise DTR
			
			ctsTimeOut = 100;							// allow about 100 ms (0.1 sec) for CTS to show up
			do
			{
				GetDeviceStatus(theDevice,&CTS,&DCD);	// see if CTS is true
				if(CTS)
				{
					break;								// break out if it is
				}
				usleep(1000);							// wait 1 ms (more or less), try again
			}
			while(ctsTimeOut--);
			
			if(!CTS)
			{
				ReportError("programmer not detected (CTS is false)\n");
				fail = true;							// didn't see CTS, assume device is not present or not ready, fail
			}
			else
			{
				ConfigureFlowControl(theDevice,true);	// looks ok to use flow control, so allow it
			}
			FlushBytes(theDevice);						// get rid of any pending data
		}
		else
		{
			ReportError("could not configure flow control\n");
			fail = true;
		}
	}
	else
	{
		ReportError("could not configure device parameters\n");
		fail = true;
	}
	return(!fail);
}


static void ShowStartSize(UINT16 start, UINT16 size)
{
	if(size)
	{
		ReportMessage("    0x%04x-0x%04x (0x%04x word%c)\n",start,start+size-1,size,((size != 1)?'s':'\0'));
	}
	else
	{
		ReportMessage("    none\n");
	}
}

static void ShowDeviceInfo(const PIC_DEFINITION *picDevice)
{
	ReportMessage("device name: %s\n",picDevice->name);					// show the name

	ReportMessage("  program space:\n");								// show range of program space
	ShowStartSize(0,GetPgmSize(picDevice));

	ReportMessage("  data space:\n");									// show range of data space, if any
	ShowStartSize(GetDataStart(picDevice),GetDataSize(picDevice));

	ReportMessage("  oscillator calibration space:\n");					// show range of calibration space, if any
	ShowStartSize(GetOscCalStart(picDevice),GetOscCalSize(picDevice));

	ReportMessage("  configuration bits:\n");
	ReportMessage("    address: 0x%04x\n",GetConfigStart(picDevice));	// show address of configuration bits
	ReportMessage("    protect mask:  0x%04x\n",picDevice->cpbits);		// mask of code protect bits
	ReportMessage("    watchdog mask: 0x%04x\n",picDevice->wdbit);		// mask of watchdog enable bit
}


//-----------------------------------------------------------------------------
//
//	ShowDevices -- display all supported devices
//
//-----------------------------------------------------------------------------

#define MAXNAMESLEN 80				// max number of characters on a line

static void ShowDevices()
{
	int idx = 0;
	int length = 0;
	int thisLength;

	ReportMessage("supported devices:\n");
	while(deviceList[idx])
	{
		thisLength = strlen(deviceList[idx]->name);			// length of this device's name
		length += thisLength;								// add to length of this line
		if(length+2 < MAXNAMESLEN)							// ensure there's room for the space and comma, too
		{
			ReportMessage("%s",deviceList[idx]->name);		// put it on this line
		}
		else
		{
			ReportMessage("\n%s",deviceList[idx]->name);	// put it on the next line
			length = thisLength;
		}
		idx++;
		if(deviceList[idx])									// if more devices are in the list,
		{
			ReportMessage(", ");							// add a comma and a space
			length+=2;
		}
		else
		{
			ReportMessage("\n");							// or newline if it is the last one
		}
	}
}


// some thorny issues still exist in specifying arguments:
//
//	1) should write cause a blank check before writing?  should it program anyway if no bits that should be 1 are 0?
//	2) several different writes are needed (program, ID, data, config)
//	3) likewise, several different reads are needed
//	4) need to be able to specify a range of addresses to read, instead of reading the entire device
//	5) should erase be an option?  several different erases? erase plus a set of flags to indicate what to erase?

static void Usage()
// tell the user how to use this program
{
	ReportMessage("%s: version %s, (c) 2000 Cosmodog, Ltd. (http://www.cosmodog.com)\n",programName,versionString);
	ReportMessage("Usage: %s ttyname devtype [-h] [-q] [-v] [-s [size]] [-b|-r|-w|-e][pcido]\n",programName);
	ReportMessage(" where:\n");
	ReportMessage("  ttyname is the serial device the PICSTART is attached to (e.g., /dev/ttyS0)\n");
	ReportMessage("  devtype is the pic device to be used (12C508, 16C505, etc.)\n");
	ReportMessage("  -h shows this help\n");
	ReportMessage("  -s [size] shows a hash mark status bar of length [size] while erasing/writing\n");
	ReportMessage("  -q sets quiet mode (excess messages supressed)\n");
	ReportMessage("  -r initiates a read (Intel Hex record format)\n");
	ReportMessage("  -b blank checks the requested region or regions\n");
	ReportMessage("  -f ignores verify errors while writing\n");
	ReportMessage("  -w writes to the requested region\n");
	ReportMessage("  -e erases the requested region (flash parts only)\n");
	ReportMessage("  -v shows PICSTART Plus version number\n");
	ReportMessage("    p [filename] = program memory, optionally reading/writing filename\n");
	ReportMessage("    c [val] = configuration bits (val is a numeric word value when writing)\n");
	ReportMessage("    i [val] = ID locations\n");
	ReportMessage("    d [filename] = data memory, optionally reading/writing filename\n");
	ReportMessage("    o [val] = oscillator calibration space\n");
	ReportMessage("  filename is an optional input or output file (default is stdin/stdout)\n");
	ReportMessage("\n");
	ReportMessage("Flags are operated on in order, from left to right.  If any operation fails,\n");
	ReportMessage("further execution is aborted.  Thus, a part can be blank checked and programmed\n");
	ReportMessage("with a single command, e.g.:\n");
	ReportMessage("        %s /dev/ttyS0 16c505 -bp -wp program.hex \n",programName);
	ReportMessage("This example will blank check the program memory of a PIC16C505 then write the\n");
	ReportMessage("contents of the file program.hex to the program memory only if the blank check\n");
	ReportMessage("succeeded.\n");
	ReportMessage("The -wc, -wi, and -wo options must be followed by a numeric argument which\n");
	ReportMessage("represents the value.  The number may be binary (preceeded by 0b or 0B), hex\n");
	ReportMessage("(preceeded by 0x or 0X), or decimal (anything else).\n\n");
	ShowDevices();
}


int main(int argc,char *argv[])
// open the device, and begin terminal operations on it
{
	bool fail = false;
	UINT32
		baudRate;
	UINT8
		dataBits,
		stopBits,
		parity;

	char *flags;

	const PIC_DEFINITION *picDevice = 0;

	programName=*argv++;									// name of the application
	argc--;

	SetMessageDisplay(true);								// allow display of messages unless told otherwise
	SetErrorDisplay(true);									// allow display of errors unless told otherwise
	hashWidth = false;										// don't show hask marks by default
	ignoreVerfErr = false;									// by default stop on verify errors

	if(argc>2)												// need at least four arguments to do anything
	{
		deviceName=*argv++;									// name of the device (probably)
		argc--;
		picName=*argv++;									// name of the PIC type (probably)
		argc--;
		if( (picDevice = GetPICDefinition(picName)) )		// locate the PIC type (0 = none found)
		{
			done=false;
			if(OpenDevice(deviceName,&theDevice))				// open the serial device
			{
				baudRate=19200;
				dataBits=8;
				stopBits=1;
				parity=0;

				if(InitDevice(theDevice,baudRate,dataBits,stopBits,parity))		// initialize the serial port
				{
					if(DoGetProgrammerType())									// ask what kind of programmer is attached, fail if none or one we don't support
					{
						if(DoGetVersion())										// get the version number of the PICSTART
						{
							if(QualifyVersion(&PICversion))						// be sure it's correct
							{
								if(DoInitPIC(picDevice))						// try to load up the parameters for this device
								{
									while(argc && !fail)						// do as long as we can read some more
									{
										flags = *argv++;						// get this argument, point to the next
										argc--;
										if(*flags == '-')						// see if it's a flag
										{
											flags++;							// it is, skip the dash
											switch(*flags)
											{
												case 'v':
													DoShowVersion();
													break;

												case 'f':
													ignoreVerfErr = true;		// force, ignore verify error
													break;

												case 'q':
													SetMessageDisplay(false);	// inhibit display of messages
													break;

												case 'h':
													Usage();					// give help
													break;

												case 's':
													if(argc && **argv != '-')					// if the next argument isn't preceeded by a '-'
													{
														fail = !atoi_base(*argv,&hashWidth);	// try to read the next argument as a number
														argv++;									// skip to the next argument
														argc--;
														if(fail)
														{
															ReportError("Unable to interpret '%s' as a numerical value\n",*argv);
														}
													}
													else
													{
														hashWidth = HASH_WIDTH_DEFAULT;						// turn hash marks on to the default width
													}
													break;

												case 'b':
												case 'r':
												case 'w':
												case 'e':
													fail = !DoTasks(&argc,&argv,picDevice,flags);	// do the requested operation
													break;

												case '\0':						// ignore a stray dash
													break;

												default:
													ReportError("bad argument: '%s'\n",*(argv-1));			// back up, show the trouble spot
													break;

											}
										}
									}

								}
								else
								{
									ReportError("failed to initialize %s\n",picDevice->name);
								}
							}
							else
							{
								ReportError("firmware version must be at least %d.%02d.%02d\ngo to http://www.microchip.com to upgrade\n",MIN_MAJORVER,MIN_MIDVER,MIN_MINORVER);
							}
						}
						else
						{
							ReportError("failed to obtain PICSTART firmware version number\n");
						}
					}
					else
					{
						ReportError("failed to connect to PICSTART Plus\n");
					}
				}
				else
				{
					ReportError("failed to set up the serial port\n");
				}
				CloseDevice(theDevice);
			}
			else
			{
				ReportError("failed to open device '%s'\n",argv[1]);
			}
		}
		else
		{
			ReportError("unrecognized PIC device type: '%s'\n",argv[2]);		// don't know that one;
			ShowDevices();														// give a helpful list of supported devices
		}
	}
	else
	{
		if( (argc == 1 && (picDevice = GetPICDefinition(argv[0]))) ||			// locate the PIC type (0 = none found)
			(argc == 2 && (picDevice = GetPICDefinition(argv[1]))) )			//  (be forgiving about arg position)
		{
			ShowDeviceInfo(picDevice);
		}
		else
		{
			Usage();
			fail = true;
		}
	}
	return(fail);	// return 0 if okay (not failed)
}
