/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Uses:      ImLib                                                          *
 * Modul:     cutarea.c                                                      *
 *            Funktions to cut rectangular areas from an image               *
 * Author:    Andreas Tille                                                  *
 * Date:      27.02.1998                                                     *
 * Copyright: Andreas Tille, 1999; Gnu Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>

#include "paul.h" 

int CropImage(PICTURE *bild, int x, int y, int w, int h)
/* Implements an own crop routine if images has to be cut before ImLib structures
 * are created.  After creation of ImLib structures it is necessary to use the
 * fairly slower :-( ImLib routines, because since ImLib 1.8.2 the rgb_data pointer
 * can't be changed
 * --- Parameters: ---
 * PICTURE *bild       : image to crop
 * int      x          : left
 * int      y          : top
 * int      w          : width
 * int      h          : height
 * --- Return: ---
 * int      CropImage(): RET_ERR in case of error, else RET_OK
 */
{
  register unsigned char *ap, *bp, *fap;
  register int            sw, sW;
  unsigned char          *data;

  g_return_val_if_fail ( IS_PICTURE(bild), RET_ERR );
  if ( x < 0 || y < 0 ) {
    g_warning(_("Invalid top-left cutting point (%i,%i)"), x, y);
    return RET_ERR;
  }
  if ( w > bild->W || h > bild->H ) {
    g_warning(_("Invalid bottom-rigth cutting point (%i,%i)"), w, h);
    return RET_ERR;
  }

  if ( bild->im ) {
     gdk_imlib_crop_image(bild->im, x, y, w, h);
     bild->DATA = bild->im->rgb_data;
  } else {
    sw   = bild->storepix * w;
    sW   = bild->storepix * bild->W;
    data = g_malloc(sw*h);
  
    for ( fap = (ap = bild->DATA + bild->storepix*(y*bild->W + x)) + h*sW,
          bp  = data; ap < fap; ap += sW, bp += sw )
      memcpy(bp, ap, sw);
    FREE(bild->DATA);
    bild->DATA = data;
  }
  bild->size = (bild->W = w) * (bild->H = h);

  return RET_OK;
}


gint CutArea(PICTURE *bild, BOX *cut, glong flag, gchar *was, gchar *ext, gint del_old)
/* Cut area of an image.  Original data were destroyed
 * --- Parameter: ---
 * PICTURE *bild     : image structure
 * BOX     *cut      : corners of the area to cut
 * int      flag     : flag if normal cut or search for extrema
 * char    *was      : type of cutting (cut, min, max)
 * char    *ext      : extension to use for new filename (cut, min, max)
 * int      del_old  : flag, whether the old image is to be freed or not
 * --- Return: ---
 * int      CutArea(): RET_ERR or RET_OK
 */
{
  gchar                  *buf;

  g_return_val_if_fail ( IS_PICTURE(bild), RET_ERR );
  g_return_val_if_fail ( cut, RET_ERR );
  
  if ( SaveExtrema(flag) ) { 
             /* Create a shell script which cuts images according to the   *
              * extrema of this image.                                     */
    FILE  *fp;
    
    buf = g_strdup_printf("%s%s.sh", bild->file, ext);
    Backup(buf);
    g_return_val_if_fail ( (fp = fopen(buf, "wt")), RET_ERR );
    fprintf(fp, "#!/bin/sh\n# Crop all images according to %s\n", was);
    fprintf(fp, "%s -c%ix%i+%i+%i $@\n", 
                 exename, cut->x2-cut->x1, cut->y2-cut->y1, cut->x1, cut->y1);
    fclose(fp);
    chmod(buf,  S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
    g_free(buf);
  }  

  CropImage(bild, cut->x1, cut->y1, cut->x2 - cut->x1, cut->y2 - cut->y1);

  buf = g_strdup_printf("%s: {%i,%i}{%i,%i}", was, cut->x1, cut->y1, cut->x2, cut->y2);

                 /* there is no special cut flag. vvvvvv */
  ImgChunksUpdate(bild, TypAusschnitt, buf, ext, SAFEFLAG);
  FREE(buf);

  return RET_OK;
}

int CutAreas(GList *piclist, OPTIONS *p)
/* cut box from set of images, replace the old images by the cuts
 * --- Parameter: ---
 * GList   *piclist   : list of images
 * OPTIONS *p         : cut: coordinates of box to cut from image
 * --- Return: ---
 * int      CutAreas(): RET_ERR or RET_OK
 */
{
  char    *buf;
  PICTURE *bild;
  GList   *pl;

  g_return_val_if_fail ( CheckPicList(piclist), RET_ERR );

  for ( bild = BILD(pl = piclist); pl; bild = BILD(pl = pl->next) ) {
    buf = g_strdup_printf("Section from %s", bild->file);
    if ( CutArea(bild, p->cut, p->f, buf, APPCUT, 1) ) {
      g_warning(_("Error while cutting %s"), bild->file);
      continue;
    }
    pl->data = bild;
    FREE(buf);
  }
  return RET_OK;
}





