/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                            avtM3DC1FileFormat.C                           //
// ************************************************************************* //

// For whatever reason hdf5 1.8.4 on Ubunut defines the 1.6 API as default...
// Force using the 1.8 API in this situation. These must be defined before
// including hdf5.h
#define H5Gopen_vers 2
#define H5Dopen_vers 2
#define H5Eset_auto_vers 2

#include <avtM3DC1FileFormat.h>
#include <avtIVPM3DC1Field.h>

#include <string>

#include <vtkIntArray.h>
#include <vtkFloatArray.h>
#include <vtkDoubleArray.h>
#include <vtkUnstructuredGrid.h>
#include <vtkTriangle.h>

#include <avtDatabaseMetaData.h>

#include <DBOptionsAttributes.h>
#include <Expression.h>

#include <InvalidVariableException.h>
#include <InvalidFilesException.h>
#include <UnexpectedValueException.h>
#include <DebugStream.h>

using namespace std;

#define ELEMENT_SIZE 7
#define SCALAR_SIZE 20

#define TWO_PI 6.283185307179586476925286766559


// ****************************************************************************
//  Method: avtM3DC1FileFormat constructor
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

avtM3DC1FileFormat::avtM3DC1FileFormat(const char *filename,
                                       DBOptionsAttributes *readOpts)
  : avtMTSDFileFormat(&filename, 1),
    m_filename(filename),
    m_poloidalPlanes(1), m_refinementLevel(0), m_dataLocation(AVT_NODECENT),
    m_perturbationScale(1.5)
{
    if (readOpts != NULL) {
      for (int i=0; i<readOpts->GetNumberOfOptions(); ++i) {
        if (readOpts->GetName(i) == "Number of poloidal planes")
          m_poloidalPlanes = readOpts->GetInt("Number of poloidal planes");
        else if (readOpts->GetName(i) == "Levels of mesh refinement 0-5")
          m_refinementLevel = readOpts->GetInt("Levels of mesh refinement 0-5");
        else if (readOpts->GetName(i) == "Linear Mesh Data Location") 
        {
          int dataLocation = readOpts->GetEnum("Linear Mesh Data Location");
          
          if( dataLocation == 0 )
            m_dataLocation = AVT_NODECENT;
          else if( dataLocation == 1 )
            m_dataLocation = AVT_ZONECENT;
        }
        else if (readOpts->GetName(i) == "Perturbation scaling")
          m_perturbationScale = readOpts->GetDouble("Perturbation scaling");
      }
    }

    if( m_poloidalPlanes < 1 )       m_poloidalPlanes = 1;

    if( m_refinementLevel < 0 )      m_refinementLevel = 0;
    else if( m_refinementLevel > 5 ) m_refinementLevel = 5;

    LoadFile();
}


// ****************************************************************************
//  Method: avtEMSTDFileFormat::GetNTimesteps
//
//  Purpose:
//      Tells the rest of the code how many timesteps there are in this file.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

int
avtM3DC1FileFormat::GetNTimesteps(void)
{
    return m_timeSteps.size();
}


// ****************************************************************************
//  Method: avtM3DC1FileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

void
avtM3DC1FileFormat::FreeUpResources(void)
{
    H5Fclose( m_fileID );
}


// ****************************************************************************
//  Method: avtM3DC1FileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

void
avtM3DC1FileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md,
                                             int timeState)
{
    avtMeshMetaData *mmd;
    avtMeshType mt = AVT_UNSTRUCTURED_MESH;
    int nblocks = nelms;
    int block_origin = 0;
    int cell_origin = 0;
    int group_origin = 0;
    int spatial_dimension = 3;
    int topological_dimension = 3;
    double *extents = NULL;

    char level[4];

    if( m_refinementLevel != 0 )
    {
      sprintf( level, "_%d", m_refinementLevel );
   }
    else
      sprintf( level, "" );


    // Original meshes for the user to see.
    AddMeshToMetaData( md, "equilibrium/mesh",
                       mt, extents, nblocks*m_poloidalPlanes, block_origin,
                       spatial_dimension, topological_dimension );
    
    AddMeshToMetaData( md,"mesh", mt,
                       extents, nblocks*m_poloidalPlanes, block_origin,
                       spatial_dimension, topological_dimension );
    
    
    // Populate the scalar field vars that will be interpolate onto a
    // refined mesh.
    for ( int i = 0; i < m_fieldVarNames.size(); ++i )
    {
      string varname = "equilibrium/" + m_fieldVarNames[i];
      string meshname = string("equilibrium/mesh") + string(level);
      AddScalarVarToMetaData( md, varname, meshname, m_dataLocation );

      meshname = string("mesh") + string(level);
      AddScalarVarToMetaData( md, m_fieldVarNames[i], meshname, m_dataLocation );
    }

    // This vector is the dummy vector
    // For now the mesh is the same mesh as the original mesh because
    // of needing it for the integration.

    AddVectorVarToMetaData( md, "B", string("mesh"),
                            AVT_ZONECENT, 3);

//     AddVectorVarToMetaData( md, "B", string("mesh") + string(level),
//                             m_dataLocation, 3 );


    // Hidden refined meshes for working with the interpolated data
    if( m_refinementLevel != 0 )
    {
      nblocks = nelms * m_poloidalPlanes * pow(3.0,m_refinementLevel);

      mmd =
        new avtMeshMetaData(string("equilibrium/mesh") + string(level),
                            nblocks, block_origin,
                            cell_origin, group_origin,
                            topological_dimension, spatial_dimension, mt);
      mmd->hideFromGUI = true;
      md->Add(mmd);


      mmd =
        new avtMeshMetaData(string("mesh") + string(level),
                            nblocks, block_origin,
                            cell_origin, group_origin,
                            topological_dimension, spatial_dimension, mt);
      mmd->hideFromGUI = true;
      md->Add(mmd);
    }

    nblocks = nelms;    

    // Hidden meshes for working with the elements directly
    mmd =
      new avtMeshMetaData("hidden/equilibrium/mesh",
                          nblocks, block_origin, cell_origin, group_origin,
                          topological_dimension, spatial_dimension, mt);
    mmd->hideFromGUI = true;
    md->Add(mmd);

    mmd =
      new avtMeshMetaData("hidden/mesh",
                          nblocks, block_origin, cell_origin, group_origin,
                          topological_dimension, spatial_dimension, mt);
    mmd->hideFromGUI = true;
    md->Add(mmd);


    // Hidden scalar header vars.
    for ( int i = 0; i < m_scalarVarNames.size(); ++i )
    {
      avtScalarMetaData *smd =
        new avtScalarMetaData("hidden/" + m_scalarVarNames[i],
                              "hidden/equilibrium/mesh", AVT_ZONECENT);

      smd->hideFromGUI = true;
      md->Add(smd);
    }

    // Add the elements so we have access to them for the interpolation
    avtVectorMetaData *amd =
      new avtVectorMetaData("hidden/equilibrium/elements",
                           "hidden/equilibrium/mesh",
                           AVT_ZONECENT, ELEMENT_SIZE);

    amd->hideFromGUI = true;
    md->Add(amd);

    amd = new avtVectorMetaData("hidden/elements", "hidden/mesh",
                               AVT_ZONECENT, ELEMENT_SIZE);

    amd->hideFromGUI = true;
    md->Add(amd);

    // Hidden array field vars so we have access to them for the interpolation
    for ( int i = 0; i < m_fieldVarNames.size(); ++i )
    {
      string varname = "hidden/equilibrium/" + m_fieldVarNames[i];
      amd = new avtVectorMetaData(varname, "hidden/equilibrium/mesh",
                                 AVT_ZONECENT, SCALAR_SIZE);

      amd->hideFromGUI = true;
      md->Add(amd);

      varname = "hidden/" + m_fieldVarNames[i];
      amd = new avtVectorMetaData(varname, "hidden/mesh",
                                 AVT_ZONECENT, SCALAR_SIZE);

      amd->hideFromGUI = true;
      md->Add(amd);
    }
}


// ****************************************************************************
//  Method: avtM3DC1FileFormat::GetElements
//
//  Purpose:
//      Gets the elements assoicated for the C1 mesh.
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

float *
avtM3DC1FileFormat::GetElements(int timestate, const char *meshname)
{
  char meshStr[64];

  // Parse the mesh naem into the hdf5 group name.
  if( strncmp(meshname, "equilibrium/mesh", 16 ) == 0 )
  {
    sprintf( meshStr, "/equilibrium/mesh" );
  } else if( strncmp(meshname, "mesh", 4 ) == 0 ) {
    sprintf( meshStr, "/time_%03d/mesh", timestate );
  } else
    EXCEPTION2( UnexpectedValueException, meshname, "NOT FOUND" );

  // Open the group.
  hid_t meshId = H5Gopen( m_fileID, meshStr, H5P_DEFAULT);
  if ( meshId < 0 )
    EXCEPTION2( UnexpectedValueException, meshStr, "NOT FOUND" );
  
  // Read in the mesh information.
  int nElements;
  if ( ! ReadAttribute( meshId, "nelms", &nElements ) )
    EXCEPTION2( UnexpectedValueException, "nelms", "Not found or wrong type" );
  
  if( nElements != nelms )
    EXCEPTION2( UnexpectedValueException, "nelms",  "Time step nelms does not match equilibrium nelms" );
 
  // Open the dataset and space info for the elements.  
  hid_t datasetId = H5Dopen(meshId, "elements", H5P_DEFAULT);
  hid_t spaceId = H5Dget_space(datasetId);
  size_t rank = H5Sget_simple_extent_ndims(spaceId);
  std::vector<hsize_t> sdim(rank);
  H5Sget_simple_extent_dims(spaceId, &sdim[0], NULL);

  // Sanity check.  
  if( rank != 2 || sdim[0] != nelms || sdim[1] != ELEMENT_SIZE )
  {
      EXCEPTION2( UnexpectedValueException, "elements", "The number of elements or the element size does not match" );          
  }

  // Memory for the elements.  
  float *elements = new float[sdim[0]*sdim[1]];
  if( elements == 0 )
    EXCEPTION2( UnexpectedValueException, "Elements", "CAN NOT ALLOCATE MEMORY" );

  // Read in the elements.
  H5Dread( datasetId,
           H5T_NATIVE_FLOAT, H5S_ALL, spaceId, H5P_DEFAULT, elements );

  H5Dclose(spaceId);
  H5Dclose(datasetId);
  H5Gclose( meshId );

  return elements;
}

// ****************************************************************************
//  Method: avtM3DC1FileFormat::GetMeshPoints
//
//  Purpose: Gets the mesh points associated with this file.  The mesh
//      is returned as vtkPoints.
//
//  Arguments:
//      elements        The original C1 elements 
//      poloidalPlanes  The number poloidal planes to create
//      refinementLevel The amount of mesh refinement.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

vtkPoints *
avtM3DC1FileFormat::GetMeshPoints(float *elements,
                                  int poloidalPlanes,
                                  int refinementLevel)
{
  // Inital number of triangles before refinement but with all planes.
  int ntris = m_poloidalPlanes * nelms * 3;

  // VTK structure for holding the mesh points. 
  vtkPoints *vtkPts = vtkPoints::New();
  vtkPts->SetNumberOfPoints( ntris );
  float *pts = (float *) vtkPts->GetVoidPointer(0);

  // Create the mesh hat each requested plane 0-> 2 * Pi
  for( int p=0; p<poloidalPlanes; ++p ) 
  {
    float phi = TWO_PI * (float) p / (float) poloidalPlanes;

    // Pointer for fast indexing through the elements.
    float *element = elements;

    for( int i=0; i<nelms; ++i )
    {
      // The element values from Jardin, JCP 200:133 (2004)
      float a     = element[0];
      float b     = element[1];
      float c     = element[2];
      float theta = element[3];
      float x     = element[4];
      float z     = element[5];

      // The three points in the Y plane that define the mesh element.
      pts[0] = x;
      pts[1] = phi;
      pts[2] = z;

      pts[3] = x + (a+b)*cos(theta);
      pts[4] = phi;
      pts[5] = z + (a+b)*sin(theta);

      pts[6] = x + b*cos(theta) - c*sin(theta);
      pts[7] = phi;
      pts[8] = z + b*sin(theta) + c*cos(theta);

      // Increment for fast indexing.
      element += ELEMENT_SIZE;
      pts += 9;
    } 
  }
  
  // Now do any needed refinement.
  while( refinementLevel-- )
  {
    // Pointer to the current mesh.
    float *pts = (float *) vtkPts->GetVoidPointer(0);

    // VTK structure for holding the refined mesh points. 
    vtkPoints *rf_vtkPts = vtkPoints::New();
    rf_vtkPts->SetNumberOfPoints( ntris*3 );
    float *rf_pts = (float *) rf_vtkPts->GetVoidPointer(0);

    // For each triangle get the centroid of the triangle and use it
    // for creating three new triangles. This algorithm works but
    // gives crappy results as it generates thin triangles.
    for( int i=0; i<ntris; i+=3 )
    {
      // Get the centroid.
      float centroid[3] = {0,0,0};
      
      for( int j=0; j<3; ++j )
      {
        centroid[0] += pts[j*3  ];
        centroid[1] += pts[j*3+1];
        centroid[2] += pts[j*3+2];
      }
      
      for( int j=0; j<3; ++j )
        centroid[j] /= 3.0;

      // Create three new triagles using the centroid as one of the
      // new points.
      for( int j=0,k=1; j<3; ++j,++k )
      {
        // For wrapping around the three original points.
        if( k == 3 )
          k = 0;
        
        rf_pts[0] = centroid[0];
        rf_pts[1] = centroid[1];
        rf_pts[2] = centroid[2];
        
        rf_pts[3] = pts[j*3+0];
        rf_pts[4] = pts[j*3+1];
        rf_pts[5] = pts[j*3+2];
        
        rf_pts[6] = pts[k*3+0];
        rf_pts[7] = pts[k*3+1];
        rf_pts[8] = pts[k*3+2];
        
        rf_pts += 9;
      }

      pts += 9;
    }

    // Delete the old points.
    vtkPts->Delete();

    // Update the pointers to the new mesh
    vtkPts = rf_vtkPts;

    // Number of triangles in the refined mesh.
    ntris *= 3;
  }

  return vtkPts;  
}



// ****************************************************************************
//  Method: avtM3DC1FileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

vtkDataSet *
avtM3DC1FileFormat::GetMesh(int timestate, const char *meshname)
{
  const char *meshnamePtr = meshname;  
  char meshStr[64];

  int poloidalPlanes;

  // If hidden only one poloidal plane as the mesh will be used for
  // interpolation. There should also be no refinement.
  if( strncmp(meshname, "hidden/", 7 ) == 0 )
  {
    poloidalPlanes = 1;
    meshnamePtr = &(meshname[7]);
  }
  else
  {
    poloidalPlanes = m_poloidalPlanes;
    meshnamePtr = meshname;
  }
  
  // Parse the mesh variable name to get the true mesh name.
  if( strncmp(meshnamePtr, "equilibrium/mesh", 16 ) == 0 )
  {
    meshnamePtr = &(meshnamePtr[16]);
    sprintf( meshStr, "equilibrium/mesh" );    
  }
  else if( strncmp(meshnamePtr, "mesh", 4 ) == 0 ) {
    meshnamePtr = &(meshnamePtr[4]);
    sprintf( meshStr, "mesh" );    
  }
  else
    EXCEPTION2( UnexpectedValueException, meshname, "NOT FOUND" );


  // Parse the mesh variable name to get the refinement level.
  int refinementLevel;

  if( strlen(meshnamePtr) && atoi(&(meshnamePtr[1])) == m_refinementLevel )
      refinementLevel = m_refinementLevel;
  else
    refinementLevel = 0;

  // Get the C1 elements.
  float* elements = GetElements(timestate, meshStr);

  // Create a VTk grid for the mesh.
  vtkUnstructuredGrid *grid = vtkUnstructuredGrid::New();

  // Now get the points on the mesh based on the number of planes and
  // the refinement level.
  vtkPoints *vtkPts =
    GetMeshPoints( elements, m_poloidalPlanes, refinementLevel);

  // Add the points to the VTK grid.
  int npts = vtkPts->GetNumberOfPoints();
  grid->SetPoints( vtkPts );

  delete [] elements;

  // Add in the VTK cells. The connectivity is the same for all
  // triangles because each triangle is defined by three points that
  // are not unique. 
  vtkTriangle *tri = vtkTriangle::New();
  for( int i=0; i<npts; i+=3 )
  {
    tri->GetPointIds()->SetId( 0, i   );
    tri->GetPointIds()->SetId( 1, i+1 );
    tri->GetPointIds()->SetId( 2, i+2 );
    
    grid->InsertNextCell( tri->GetCellType(), tri->GetPointIds() );
  }
  
  tri->Delete();

  return grid;
}


// ****************************************************************************
//  Method: avtM3DC1FileFormat::GetHeaderVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

vtkDataArray *
avtM3DC1FileFormat::GetHeaderVar(int timestate, const char *varname)
{
  // Get the header variable

  // Header variables are at the top level group.
  hid_t rootID = H5Gopen( m_fileID, "/", H5P_DEFAULT);
  if ( rootID < 0 )
    EXCEPTION2( UnexpectedValueException, "Root Group", "NOT FOUND" );

  // Everything is converted to floats by visit so might as well do it
  // here and save the copying time and memory.
  float value;

  // Read in linear flag and ntor an an int
  if( strcmp(&(varname[7]), "linear") == 0 ||
      strcmp(&(varname[7]), "ntor"  ) == 0 )
  {
      int intVal;
      if ( ! ReadAttribute( rootID, &(varname[7]), &intVal ) )
        EXCEPTION2( UnexpectedValueException, &(varname[7]),
                    "Not found or wrong type" )
      else
        value = intVal;
  }
    
  // Read in bzero and rzero as a double
  else if( strcmp(&(varname[7]), "bzero") == 0 ||
           strcmp(&(varname[7]), "rzero"  ) == 0 )
  {
      double dblVal;
      if ( ! ReadAttribute( rootID, &(varname[7]), &dblVal ) )
        EXCEPTION2( UnexpectedValueException, &(varname[7]),
                    "Not found or wrong type" )
      else
        value = dblVal;
  }

  vtkFloatArray *var = vtkFloatArray::New();

  // Set the number of components before setting the number of tuples
  // for proper memory allocation.
  var->SetNumberOfComponents( 1 );
  var->SetNumberOfTuples( nelms );
    
  float* varPtr = (float *) var->GetVoidPointer(0);
    
  for( int i=0; i<nelms; ++i)
    *varPtr++ = value;
    
  H5Gclose( rootID );
    
  return var;
}


// ****************************************************************************
//  Method: avtM3DC1FileFormat::GetFieldVar
//
//  Purpose:
//      Gets a field variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

vtkDataArray *
avtM3DC1FileFormat::GetFieldVar(int timestate, const char *varname)
{
  char groupStr[64];
  char varStr[64];

  int nComponents;

  // Parse the field variable name to get the HDF5 group and dataset
  // name.
  if( strncmp(varname, "equilibrium", 11 ) == 0 )
  {
    if( strcmp(&(varname[11]), "elements" ) == 0 ) {
      nComponents = ELEMENT_SIZE;;
      sprintf( groupStr, "/equilibrium/mesh" );
    } else {
      nComponents = SCALAR_SIZE;;
      sprintf( groupStr, "/equilibrium/fields" );
    }

    strcpy( varStr, &(varname[12]) );

  } else {
    if( strcmp(varname, "elements" ) == 0 ) {
      nComponents = ELEMENT_SIZE;;
      sprintf( groupStr, "/time_%03d/mesh", timestate );
    } else {
      nComponents = SCALAR_SIZE;;
      sprintf( groupStr, "/time_%03d/fields", timestate );
    }

    strcpy( varStr, varname );
  }

  // Open the group.
  hid_t groupID = H5Gopen( m_fileID, groupStr, H5P_DEFAULT);
  if ( groupID < 0 )
    EXCEPTION2( UnexpectedValueException, groupStr, "NOT FOUND" );

  // Open the field dataset
  hid_t datasetId = H5Dopen(groupID, varStr, H5P_DEFAULT);
  if ( datasetId < 0 )
    EXCEPTION2( UnexpectedValueException, groupStr + string("/") + varStr,
                "NOT FOUND" );
  
  // Read in the dataset information.
  hid_t spaceId = H5Dget_space(datasetId);
  size_t rank = H5Sget_simple_extent_ndims(spaceId);
  std::vector<hsize_t> sdim(rank);
  H5Sget_simple_extent_dims(spaceId, &sdim[0], NULL);
  
  if( rank != 2 || sdim[0] != nelms || sdim[1] != nComponents )
      EXCEPTION2( UnexpectedValueException,
                  groupStr + string("/") + varStr,
                  "The number of elements or the component size does not match" );

  // Normally an array ould be created but instead use the VTK memory
  // directly - this usage works because the vtk and hdf5 memory
  // layout are the same.

//   float *vals = new float[sdim[0]*sdim[1]];
//   if( H5Dread( datasetId,
//             H5T_NATIVE_FLOAT, H5S_ALL, spaceId, H5P_DEFAULT, vals ) < 0 )
//     EXCEPTION2( UnexpectedValueException,
//                 groupStr + string("/") + varStr, "Can not read the data" );

  // Create the VTK structure to hole the field variable.
  vtkFloatArray *var = vtkFloatArray::New();

  // Set the number of components before setting the number of tuples
  // for proper memory allocation.
  var->SetNumberOfComponents( sdim[1] );
  var->SetNumberOfTuples( sdim[0] );

  // Pointer to the vtk memory.
  float* values = (float*) var->GetVoidPointer(0);
  
  // Read the data directly into the vtk memory - this call assume
  // that the hdfd5 and vtk memory layout are the same.
  if( H5Dread( datasetId,
               H5T_NATIVE_FLOAT, H5S_ALL, spaceId, H5P_DEFAULT, values ) < 0 )
    EXCEPTION2( UnexpectedValueException,
                groupStr + string("/") + varStr, "Can not read the data" );
  
  H5Dclose(spaceId);
  H5Dclose(datasetId);
  H5Gclose( groupID );

//  delete [] vals;
  
  return var;
}


// ****************************************************************************
//  Method: avtM3DC1FileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

vtkDataArray *
avtM3DC1FileFormat::GetVar(int timestate, const char *varname)
{
  // Hidden scalar variables are read from the header and put on the
  // C1 mesh.
  if( strncmp(varname, "hidden/", 7) == 0 )
  {
    char varStr[64];
    strcpy( varStr, &(varname[7]) );

    if( strncmp(varStr, "header", 6) == 0 )
      return GetHeaderVar( timestate, varStr);
    else
      return 0;
  }
    
  // First get the elements for this variable so that the variable can
  // be interpolated onto the linear mesh.
  float* elements;
  if( strncmp(varname, "equilibrium", 11 ) == 0 )
  {
    elements = GetElements(timestate, "equilibrium/mesh");
  } else {
    elements = GetElements(timestate, "mesh");
  }

  // Create a temporary mesh for getting the correct field variable
  // values on the linear mesh.
  vtkPoints *vtkPts = GetMeshPoints( elements,
                                     m_poloidalPlanes, m_refinementLevel);
  float* pts = (float *) vtkPts->GetVoidPointer(0);
  int npts = vtkPts->GetNumberOfPoints();

  // Get the M3D C1 field so the variables can be interpolated on the
  // linear mesh.
  avtIVPM3DC1Field m3dField(elements, nelms);

  // Get the field variable to be interpolated on the linear mesh.
  vtkDataArray* vtkVar = GetFieldVar( timestate, varname );
  float* values = (float*) vtkVar->GetVoidPointer(0);

  // Get the value at the node of each element on the linear mesh.
  int nvalues;

  if( m_dataLocation == AVT_NODECENT)
    nvalues = npts;
  // Get the value at the center of each element on the linear mesh.
  else if( m_dataLocation == AVT_ZONECENT)
    nvalues = npts / 3;
    
  // VTK structure for the field variable on the linear mesh.
  vtkFloatArray *var = vtkFloatArray::New();

  // Set the number of components before setting the number of tuples
  // for proper memory allocation.
  var->SetNumberOfComponents( 1 );
  var->SetNumberOfTuples( nvalues );

  // Pointer to the field variable on the linear mesh.
  float* varPtr = (float *) var->GetVoidPointer(0);

  int element;
  double centroid[3], xieta[2];

  if( m_dataLocation == AVT_NODECENT)
  {
    for( int i=0; i<npts; ++i )
    {
      // Find the element containing the point; get local coords
      // xi,eta. We only want the index so that we know which element
      // the nodes are part of. Do this once so that it is
      // faster. Also it ensures that the correct element is found
      // because the points lie on the element.
      // 
      if( i % 3 == 0 )
      {
        centroid[0] = (pts[0] + pts[3] + pts[6] ) / 3.0;
        centroid[1] = (pts[1] + pts[4] + pts[7] ) / 3.0;
        centroid[2] = (pts[2] + pts[5] + pts[8] ) / 3.0;

        if( (element = m3dField.get_tri_coords2D(centroid, xieta)) < 0 )
        {
          debug1 << "avtM3DC1FileFormat::GetVar - Can not find element for "
                 << centroid[0] << "  "<< centroid[1] << "  "<< centroid[2]
                 << endl;
        }
      }

      double pt[3] = {pts[0],pts[1],pts[2]};

      /* Find the element containing the point; get local coords xi,eta */
      if ((element = m3dField.get_tri_coords2D(pt, element, xieta)) >= 0)
        *varPtr++ = m3dField.interp(values, element, xieta);
      else
      {
        debug1 << "avtM3DC1FileFormat::GetVar - Can not find element for "
               << pt[0] << "  "<< pt[1] << "  "<< pt[2]
               << endl;
        
        *varPtr++ = 0;
      }

      pts += 3;
    }
  }
  
  else if( m_dataLocation == AVT_ZONECENT)
  {
    for( int i=0; i<npts; i+=3 )
    {
      centroid[0] = (pts[0] + pts[3] + pts[6] ) / 3.0;
      centroid[1] = (pts[1] + pts[4] + pts[7] ) / 3.0;
      centroid[2] = (pts[2] + pts[5] + pts[8] ) / 3.0;

      /* Find the element containing the point; get local coords xi,eta */
      if ((element = m3dField.get_tri_coords2D(centroid, xieta)) >= 0)
        *varPtr++ = m3dField.interp(values, element, xieta);
      else
      {
        debug1 << "avtM3DC1FileFormat::GetVar - Can not find element for "
               << centroid[0] << "  "<< centroid[1] << "  "<< centroid[2]
               << endl;
        
        *varPtr++ = 0;
      }
      
      pts += 9;
    }
  }

  vtkPts->Delete();
  vtkVar->Delete();
  
  return var;
}


// ****************************************************************************
//  Method: avtM3DC1FileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Fri Dec 4 15:04:15 PST 2009
//
// ****************************************************************************

vtkDataArray *
avtM3DC1FileFormat::GetVectorVar(int timestate, const char *varname)
{
  // Hidden variables are read directly set down stream.
  if( strncmp(varname, "hidden/", 7) == 0 )
  {
    char varStr[64];

    strcpy( varStr, &(varname[7]) );
    
    if( m_perturbationScale == 1.0 ||
        (strcmp(varname, "hidden/f" ) &&
         strcmp(varname, "hidden/f_i" ) &&
         strcmp(varname, "hidden/psi" ) &&
         strcmp(varname, "hidden/psi_i" ) ) )
    {
      return GetFieldVar( timestate, varStr );
    }

    // The user can scale the perturbed variables by a scaling factor
    // set when the file is openned.
    else
    {
      debug1 << "avtM3DC1FileFormat::GetVectorVar - Scaling "
             << varname << " by " << m_perturbationScale << endl;

      vtkDataArray* vtkVar = GetFieldVar( timestate, varStr );

      float* values = (float*) vtkVar->GetVoidPointer(0);
      
      unsigned int nvals =
        vtkVar->GetNumberOfTuples() * vtkVar->GetNumberOfComponents();
      
      for( unsigned int i=0; i<nvals; ++i )
        *values++ *= m_perturbationScale;

      return vtkVar;
    } 
  }
  else if( strcmp(varname, "B") == 0 )
  {
    // FIXME - TEMPORARY UNITL FIX FOR MIXED NODE AND ZONE
    avtCentering dataLocation = m_dataLocation;
    m_dataLocation = AVT_ZONECENT;
    
    // First get the elements for this variable so that the variable can
    // be interpolated onto the linear mesh.
    float* elements = GetElements(timestate, "mesh");

    // Create a temporary mesh for getting the correct field variable
    // values on the linear mesh.

    // For now the mesh is the same mesh as the original mesh because
    // of needing it for the integration.
    vtkPoints *vtkPts = GetMeshPoints( elements,
                                       1, 0);
//  vtkPoints *vtkPts = GetMeshPoints( elements,
//                                     m_poloidalPlanes, m_refinementLevel);

    float* pts = (float *) vtkPts->GetVoidPointer(0);
    int npts = vtkPts->GetNumberOfPoints();

    // Get the M3D C1 field so the variables can be interpolated on the
    // linear mesh.
    avtIVPM3DC1Field m3dField(elements, nelms);

    // Header variables are at the top level group.
    hid_t rootID = H5Gopen( m_fileID, "/", H5P_DEFAULT);
    if ( rootID < 0 )
      EXCEPTION2( UnexpectedValueException, "Root Group", "NOT FOUND" );

    // Get the field variable to be interpolated on the linear mesh.
    if ( ! ReadAttribute( rootID, "linear", &(m3dField.linflag) ) )
      EXCEPTION2( UnexpectedValueException, "linear",
                  "Not found or wrong type" );
      
    if ( ! ReadAttribute( rootID, "ntor", &(m3dField.tmode) ) )
      EXCEPTION2( UnexpectedValueException, "ntor",
                  "Not found or wrong type" );
      
    if ( ! ReadAttribute( rootID, "bzero", &(m3dField.bzero) ) )
      EXCEPTION2( UnexpectedValueException, "bzero",
                  "Not found or wrong type" );
      
    if ( ! ReadAttribute( rootID, "rzero", &(m3dField.rzero) ) )
      EXCEPTION2( UnexpectedValueException, "rzero",
                  "Not found or wrong type" );

    m3dField.F0 = -m3dField.bzero * m3dField.rzero;
      
    H5Gclose( rootID );
    
    // Variables on the mesh - N elements x 20
    vtkDataArray* vtkVarF0 = GetFieldVar( timestate, "equilibrium/f");  
    m3dField.f0 = (float*) vtkVarF0->GetVoidPointer(0);

    vtkDataArray* vtkVarPsi0 = GetFieldVar( timestate, "equilibrium/psi");
    m3dField.psi0 = (float*) vtkVarPsi0->GetVoidPointer(0);
    
    vtkDataArray* vtkVarF = GetFieldVar( timestate, "f");
    m3dField.fnr = (float*) vtkVarF->GetVoidPointer(0);

    vtkDataArray* vtkVarF_i = GetFieldVar( timestate, "f_i");
    m3dField.fni = (float*) vtkVarF_i->GetVoidPointer(0);

    vtkDataArray* vtkVarPsi = GetFieldVar( timestate, "psi");
    m3dField.psinr = (float*) vtkVarPsi->GetVoidPointer(0);

    vtkDataArray* vtkVarPsi_i = GetFieldVar( timestate, "psi_i");
    m3dField.psini = (float*) vtkVarPsi_i->GetVoidPointer(0);
    
    // Get the value at the node of each element on the linear mesh.
    int nvalues;

    if( m_dataLocation == AVT_NODECENT)
      nvalues = npts;
    // Get the value at the center of each element on the linear mesh.
    else if( m_dataLocation == AVT_ZONECENT)
      nvalues = npts / 3;
    
    // VTK structure for the field variable on the linear mesh.
    vtkFloatArray *var = vtkFloatArray::New();

    // Set the number of components before setting the number of tuples
    // for proper memory allocation.
    var->SetNumberOfComponents( 3 );
    var->SetNumberOfTuples( nvalues );

    // Pointer to the field variable on the linear mesh.
    float* varPtr = (float *) var->GetVoidPointer(0);

    int element;
    float B[3];
    double centroid[3], xieta[2];

    if( m_dataLocation == AVT_NODECENT)
    {
      for( int i=0; i<npts; ++i )
      {
        // Find the element containing the point; get local coords
        // xi,eta. We only want the index so that we know which element
        // the nodes are part of. Do this once so that it is
        // faster. Also it ensures that the correct element is found
        // because the points lie on the element.
        // 
        if( i % 3 == 0 )
        {
          centroid[0] = (pts[0] + pts[3] + pts[6] ) / 3.0;
          centroid[1] = (pts[1] + pts[4] + pts[7] ) / 3.0;
          centroid[2] = (pts[2] + pts[5] + pts[8] ) / 3.0;
          
          element = m3dField.get_tri_coords2D(centroid, xieta);
        }
        
        double pt[3] = {pts[0],pts[1],pts[2]};
            
        /* Find the element containing the point; get local coords xi,eta */
        if ((element = m3dField.get_tri_coords2D(pt, element, xieta)) < 0) 
        {
          *varPtr++ = 0; *varPtr++ = 0; *varPtr++ = 0;
        }
        else 
        {
          m3dField.interpBcomps(B, pt, element, xieta);
        
          *varPtr++ = B[0]; *varPtr++ = B[1]; *varPtr++ = B[2];
        }
        
        pts += 3;
      }
    }
  
    else if( m_dataLocation == AVT_ZONECENT)
    {
      for( int i=0; i<npts; i+=3 )
      {
        centroid[0] = (pts[0] + pts[3] + pts[6] ) / 3.0;
        centroid[1] = (pts[1] + pts[4] + pts[7] ) / 3.0;
        centroid[2] = (pts[2] + pts[5] + pts[8] ) / 3.0;
        
        /* Find the element containing the point; get local coords xi,eta */
        if ((element = m3dField.get_tri_coords2D(centroid, xieta)) < 0) 
        {
          *varPtr++ = 0; *varPtr++ = 0; *varPtr++ = 0;
        }
        else 
        {
          m3dField.interpBcomps(B, centroid, element, xieta);
        
          *varPtr++ = B[0]; *varPtr++ = B[1]; *varPtr++ = B[2];
        }

        pts += 9;
      }
    }
    
    // Set the pointers to null as the VTK delete operation will take
    // of deleting the data. Normally the M3DCIField thinks it needs
    // to delete the data.
    m3dField.f0 = 0;
    m3dField.psi0 = 0;    
    m3dField.fnr = 0;
    m3dField.fni = 0;
    m3dField.psinr = 0;
    m3dField.psini = 0;

    vtkPts->Delete();

    vtkVarF0->Delete();    
    vtkVarPsi0->Delete();
   
    vtkVarF->Delete();
    vtkVarF_i->Delete();
    vtkVarPsi->Delete();
    vtkVarPsi_i->Delete();

    // FIXME - TEMPORARY UNITL FIX FOR MIXED NODE AND ZONE 
    m_dataLocation = dataLocation;
  
    return var;
  }

  else
    return 0;
}



// ****************************************************************************
//  Method: avtM3DFileFormat::NormalizeH5Type
//
//  Purpose:
//     Convert HDF5 types to visit types if necessary.
//
//  Arguments:
//      type       Input type from file.
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Tue Sep 25 08:49:28 PDT 2007
//
// ****************************************************************************
hid_t
avtM3DC1FileFormat::NormalizeH5Type( hid_t type )
{
    H5T_class_t tclass = H5Tget_class( type );
    int size = H5Tget_size( type );

    switch ( tclass )
    {
    case H5T_INTEGER:
        if ( size == 8 )
            return H5T_NATIVE_INT64;
        else if ( size == 4 )
            return H5T_NATIVE_INT32;
        else if ( size == 1 )
            return H5T_NATIVE_CHAR;
        break;
    case H5T_FLOAT:
        if ( size == 8 )
            return H5T_NATIVE_DOUBLE;
        else if ( size == 4 )
            return H5T_NATIVE_FLOAT;
    default:
        break;
    }
    return -1;
}

// ****************************************************************************
//  Method: avtM3DFileFormat::ReadStringAttribute
//
//  Purpose:
//      Read a string attribute from an HDF5 file.
//
//  Arguments:
//      parentID   The id of the parent.
//      attr       Name of the attribute to be read.
//      value      The attribute value that was read.
//
//  Programmer: allen
//  Creation:   Tue Sep 25 08:49:28 PDT 2007
//
// ****************************************************************************
bool
avtM3DC1FileFormat::ReadStringAttribute( hid_t parentID,
                                         const char *attr,
                                         string *value )
{
    hid_t attrID = H5Aopen_name( parentID, attr );
    if ( attrID <= 0 )
        return false;

    hid_t typeID = H5Aget_type( attrID );
    if ( typeID < 0 )
        return false;
    hsize_t nelem = H5Tget_size( typeID );
    if ( nelem <= 0 )
        return false;
    char *str = new char[nelem];
    H5Aread( attrID, typeID, str );
    *value = str;
    delete [] str;

    H5Tclose( typeID );
    H5Aclose( attrID );
    return true;
}

// ****************************************************************************
//  Method: avtM3DFileFormat::ReadAttribute
//
//  Purpose:
//      Read an attribute from an HDF5 file.
//
//  Arguments:
//      parentID   The id of the parent.
//      attr       Name of the attribute to be read.
//      value      The attribute value that was read.
//
//  Programmer: allen
//  Creation:   Tue Sep 25 08:49:28 PDT 2007
//
// ****************************************************************************
bool
avtM3DC1FileFormat::ReadAttribute( hid_t parentID,
                                   const char *attr,
                                   void *value )
{
    hid_t attrID = H5Aopen_name( parentID, attr );
    if ( attrID <= 0 )
        return false;

    hid_t attrType = H5Aget_type( attrID );
    if ( attrType < 0 )
        return false;

    hid_t spaceID = H5Aget_space( attrID );
    if ( spaceID < 0 )
        return false;

    hsize_t nelem = H5Sget_simple_extent_npoints( spaceID );
    if ( nelem < 0 )
        return false;

    hid_t typeID = NormalizeH5Type( attrType );
    if ( typeID < 0 )
        return false;

    if ( H5Aread ( attrID, typeID, value ) < 0 )
        return false;

    H5Sclose( spaceID );
    H5Tclose( attrType );
    H5Aclose( attrID );

    return true;
}

// ****************************************************************************
//  Method: avtM3DFileFormat::groupIterator
//
//  Purpose:
//      Iterate through all of the grops in an HDF5 file.
//
//  Arguments:
//      parentID   The id of the parent.
//      attr       Name of the group.
//      value      Optional data - in this case the avtM3DC1FileFormat.
//
//  Programmer: allen
//  Creation:   Tue Sep 25 08:49:28 PDT 2007
//
// ****************************************************************************
herr_t groupIterator(hid_t locId, const char* name, void* opdata) {

  avtM3DC1FileFormat* M3DC1FF = static_cast< avtM3DC1FileFormat* >(opdata);

  H5G_stat_t statbuf;
  H5Gget_objinfo (locId, name, false, &statbuf);

  switch (statbuf.type) {

    case H5G_DATASET: {
      hid_t datasetId = H5Dopen(locId, name, H5P_DEFAULT);
      hid_t spaceId = H5Dget_space(datasetId);
      size_t rank = H5Sget_simple_extent_ndims(spaceId);
      std::vector<hsize_t> sdim(rank);
      H5Sget_simple_extent_dims(spaceId, &sdim[0], NULL);

      H5Dclose(spaceId);
      H5Dclose(datasetId);

      if( rank != 2 ||
          sdim[0] != M3DC1FF->nelms ||
          sdim[1] != SCALAR_SIZE )
      {
        EXCEPTION2( UnexpectedValueException, name,
                    "Wrong rank or dimensions" );
        return -1;
      }
      else
      {
        M3DC1FF->m_fieldVarNames.push_back( std::string( name ) );

        return 0;
      }
    }
  }
}


// ****************************************************************************
//  Method: avtM3DFileFormat::LoadFile
//
//  Purpose:
//      Open an m3d file and read in it's structure.
//
//  Arguments:
//
//  Programmer: allen -- generated by xml2avt
//  Creation:   Tue Sep 25 08:49:28 PDT 2007
//
//  Modifications:
//    Jeremy Meredith, Thu Jan  7 15:36:19 EST 2010
//    Close all open ids when returning an exception.
// ****************************************************************************
void
avtM3DC1FileFormat::LoadFile()
{
    debug1 << "Attempting to open M3D C1 file " << m_filename << endl;
    // Init HDF5.
    H5open();
    H5Eset_auto( NULL, NULL, NULL );

    m_fileID = H5Fopen( m_filename.c_str(), H5P_DEFAULT, H5P_DEFAULT );
    if ( m_fileID < 0 )
        EXCEPTION1( InvalidFilesException, m_filename.c_str() );

    hid_t rootID = H5Gopen( m_fileID, "/", H5P_DEFAULT);
    if ( rootID < 0 )
    {
        H5Fclose(m_fileID);        
        EXCEPTION2( UnexpectedValueException, "Root Group", "NOT FOUND" );
    }

    // HEADER

    // Read in step and time information.
    int ntime;
    if ( ! ReadAttribute( rootID, "ntime", &ntime ) )
    {
        H5Gclose(rootID);
        H5Fclose(m_fileID);
        EXCEPTION2( UnexpectedValueException, "ntime",
                    "Not found or wrong type" );             
    }
    // Read in linear flag and ntor
    int linear;
    if ( ! ReadAttribute( rootID, "linear", &linear ) )
    {
        H5Gclose(rootID);
        H5Fclose(m_fileID);
        EXCEPTION2( UnexpectedValueException, "linear",
                    "Not found or wrong type" );
    }
    
    m_scalarVarNames.push_back("header/linear");
      
    int ntor;
    if ( ! ReadAttribute( rootID, "ntor", &ntor ) )
    {
        H5Gclose(rootID);
        H5Fclose(m_fileID);
        EXCEPTION2( UnexpectedValueException, "ntor",
                    "Not found or wrong type" );
    }
    
    m_scalarVarNames.push_back("header/ntor");

    // Read in bzero and rzero
    double bzero, rzero;
    if ( ! ReadAttribute( rootID, "bzero", &bzero ) )
    {
        H5Gclose(rootID);
        H5Fclose(m_fileID);
        EXCEPTION2( UnexpectedValueException, "bzero",
                    "Not found or wrong type" );
    }

    m_scalarVarNames.push_back("header/bzero");

    if ( ! ReadAttribute( rootID, "rzero", &rzero ) )
    {
        H5Gclose(rootID);
        H5Fclose(m_fileID);
        EXCEPTION2( UnexpectedValueException, "rzero",
                    "Not found or wrong type" );
    }

    m_scalarVarNames.push_back("header/rzero");

    H5Gclose( rootID );

    // EQUILIBRIUM

    // Read in equilibrium mesh element information.
    hid_t groupId = H5Gopen( m_fileID, "/equilibrium/mesh", H5P_DEFAULT);
    if ( groupId < 0 )
    {
        H5Fclose(m_fileID);
        EXCEPTION2( UnexpectedValueException, "/equilibrium/mesh Group",
                    "NOT FOUND" );
    }

    if ( ! ReadAttribute( groupId, "nelms", &nelms ) )
        EXCEPTION2( UnexpectedValueException, "nelms",
                    "Not found or wrong type" );

    hid_t datasetId = H5Dopen(groupId, "elements", H5P_DEFAULT);
    hid_t spaceId = H5Dget_space(datasetId);
    size_t rank = H5Sget_simple_extent_ndims(spaceId);
    std::vector<hsize_t> sdim(rank);
    H5Sget_simple_extent_dims(spaceId, &sdim[0], NULL);
    
    H5Dclose(spaceId);
    H5Dclose(datasetId);
    
    if( rank != 2 ||
        sdim[0] != nelms ||
        sdim[1] != ELEMENT_SIZE )
    {
      EXCEPTION2( UnexpectedValueException, "elements",
                  "The number of elements or the element size does not match" );
    }

    H5Gclose( groupId );


    // Read in equilibrium field information.
    groupId = H5Gopen( m_fileID, "/equilibrium/fields", H5P_DEFAULT);
    if ( groupId < 0 )
        EXCEPTION2( UnexpectedValueException, "/equilibrium/fields Group",
                    "NOT FOUND" );

    int nfields;
    
    if ( ! ReadAttribute( groupId, "nfields", &nfields ) )
        EXCEPTION2( UnexpectedValueException, "nfields",
                    "Not found or wrong type" );

    // Go through the field group and collect all of the field datasets
    H5Giterate(groupId, ".", NULL, groupIterator, this);

    H5Gclose( groupId );

    if( nfields != m_fieldVarNames.size() )
        EXCEPTION2( UnexpectedValueException, "nfields",
                    "Number of fields does not match the number of datasets founds." );

    // TIME STEPS

    //Load basic info on variables for each time step.
    for ( int t=0; t<ntime; ++t )
    {
        char timeStep[64];
        sprintf( timeStep, "/time_%03d", t );

        hid_t groupID = H5Gopen( m_fileID, timeStep, H5P_DEFAULT);
        if ( groupID < 0 )
          EXCEPTION2( UnexpectedValueException, timeStep, "NOT FOUND" );

        // Read the time value
        double time;
        if ( ! ReadAttribute( groupID, "time", &time ) )
          EXCEPTION2( UnexpectedValueException, "time", "Not found or wrong type" );
        m_timeSteps.push_back( time );

        // Read in the field information.
        hid_t fieldID = H5Gopen( groupID, "fields", H5P_DEFAULT);
        if ( fieldID < 0 )
          EXCEPTION2( UnexpectedValueException, "fields", "NOT FOUND" );

        if ( ! ReadAttribute( fieldID, "nfields", &nfields ) )
          EXCEPTION2( UnexpectedValueException, "nfields",
                      "Not found or wrong type" );

        if( nfields != m_fieldVarNames.size() )
          EXCEPTION2( UnexpectedValueException, "nfields",
                      "The time step nfields does not match the equilibrium nfields" );
             
        for ( int i=0; i<m_fieldVarNames.size(); ++i )
        {
            hid_t datasetId =
              H5Dopen(fieldID, m_fieldVarNames[i].c_str(), H5P_DEFAULT);
            if ( datasetId < 0 )
              EXCEPTION2( UnexpectedValueException,
                          timeStep + string("/fields/") + m_fieldVarNames[i],
                          "NOT FOUND" );

            hid_t spaceId = H5Dget_space(datasetId);
            size_t rank = H5Sget_simple_extent_ndims(spaceId);
            std::vector<hsize_t> sdim(rank);
            H5Sget_simple_extent_dims(spaceId, &sdim[0], NULL);
            
            H5Dclose(spaceId);
            H5Dclose(datasetId);

            if( rank != 2 ||
                sdim[0] != nelms ||
                sdim[1] != SCALAR_SIZE )
              {
                EXCEPTION2( UnexpectedValueException,
                            timeStep + string("/fields/") + m_fieldVarNames[i],
                            "The number of elements or the element size does not match" );
              }
        }
            
        H5Gclose( fieldID );

        // Read in the mesh information.
        hid_t meshId = H5Gopen( groupID, "mesh", H5P_DEFAULT);

        int nElements;
        if ( ! ReadAttribute( meshId, "nelms", &nElements ) )
          EXCEPTION2( UnexpectedValueException, "nelms",
                      "Not found or wrong type" );

        if( nElements != nelms )
          EXCEPTION2( UnexpectedValueException, "nelms",
                      "Time step nelms does not match equilibrium nelms" );

        hid_t datasetId = H5Dopen(meshId, "elements", H5P_DEFAULT);
        hid_t spaceId = H5Dget_space(datasetId);
        size_t rank = H5Sget_simple_extent_ndims(spaceId);
        std::vector<hsize_t> sdim(rank);
        H5Sget_simple_extent_dims(spaceId, &sdim[0], NULL);
    
        H5Dclose(spaceId);
        H5Dclose(datasetId);
    
        if( rank != 2 ||
            sdim[0] != nelms ||
            sdim[1] != ELEMENT_SIZE )
        {
            EXCEPTION2( UnexpectedValueException, "elements",
                        "The number of elements or the element size does not match" );
        }
        
        H5Gclose( meshId );
    }

    debug1 << "SUCCESS in opening M3D C1 file " << m_filename << endl;
}

