/*

Copyright (C) 2000, 2001 Christian Kreibich <kreibich@aciri.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

#include <gtk/gtk.h>
#include <nd.h>
#include <nd_globals.h>
#include <nd_misc.h>
#include <nd_tcpdump.h>
#include <nd_packet.h>
#include <nd_tcb.h>
#include <nd_tracefile.h>
#include <nd_prefs.h>
#include <nd_recent.h>
#include <callbacks.h>
#include <support.h>
#include <interface.h>

ND_Trace    trace;

static GdkColor    green[5];
static guint       timestamp_timeout_id;
static int         timestamp_timeout_set = FALSE;

static void pcap_read_handler(u_char *data, const struct pcap_pkthdr *h,
			      const u_char *pdata);

static void
pcap_read_handler(u_char *data, const struct pcap_pkthdr *h,
		  const u_char *pdata)
{
  ND_Packet *p;

  p = nd_packet_new();
  p->ph = *h;

  p->link_data = malloc(h->caplen);
  memcpy(p->link_data, pdata, h->caplen);

  if (trace.pl_end)
    {
      trace.pl_end->next = p;
      p->prev = trace.pl_end;
      trace.pl_end = p;
    }
  else
    {
      trace.pl = trace.pl_end = p;
    }
  trace.num_packets++;

  gtk_main_iteration_do(FALSE);

  return;
  data = NULL;
}


void    
nd_trace_init(void)
{
  int            i;
  GtkWidget     *w;
  GtkStyle      *gs;
  GdkColormap   *cm;

  bzero(&trace, sizeof(ND_Trace));
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);

  gs = gtk_widget_get_style(w);
  
  for (i = 0; i < 5; i++)
    {
      green[i] = gs->bg[i];
      green[i].green = (2 * green[i].green) > 65535 ? 65535 : 2 * green[i].green;
      if (green[i].green == 0)
	green[i].green = 32768;
      green[i].red /= 2;
      green[i].blue /= 2;
    }

  cm = gtk_widget_get_colormap(w);
  gdk_color_alloc(cm, &green[GTK_STATE_NORMAL]);
}


void    
nd_trace_clear(void)
{
  ND_Packet *p, *p2;

  FREE(trace.filename);

  nd_tcb_clear();

  p = trace.pl;

  while (p)
    {
      p2 = p;
      p = p->next;
      nd_packet_free(p2);
    }

  bzero(&trace, sizeof(ND_Trace));
}


int
nd_trace_load(char *filename)
{
  ND_Packet *p;
  ND_TCB    *tcb;

  nd_trace_clear();

  if (!nd_misc_is_tcpdump_file(filename))
    {
      nd_misc_statusbar_set(_("File is not a tcpdump savefile."));
      return (FALSE);
    }

  /* Reset modificaton indicator. */
  nd_trace_set_dirty(FALSE);

  /* Update "recently-used" menu entries: */
  nd_recent_add_file(filename);

  /* Close old connection first */
  nd_tcpdump_close();

  /* Sync comms with tcpdump */
  nd_tcpdump_init_tracefile(filename);
  if (!nd_tcpdump_open())
    return (FALSE);

  trace.filename = strdup(filename);
  
  if ( (trace.pcap = pcap_open_offline(trace.filename, NULL)) == NULL)
    return (FALSE);

  nd_misc_statusbar_set(_("Loading file..."));

  /* Do it! */
  nd_misc_pbar_start_activity();  
  pcap_loop(trace.pcap, 0, pcap_read_handler, NULL);
  nd_misc_pbar_stop_activity();

  /* Adjust label in lower left */
  nd_misc_set_num_packets_label();
  nd_misc_set_windowtitle(filename);
  nd_misc_tcpdump_list_incomplete_column_visibility(FALSE);

  nd_misc_statusbar_set(_("Initializing packets..."));
  nd_misc_pbar_reset(trace.num_packets);

  /* Initialize all ND_Packets internally: */
  for (p = trace.pl; p; p = p->next)
    {
      nd_packet_init(p);

      if (p->transp_prot == ND_PROT_TCP)
	{
	  tcb = nd_tcb_lookup(p);

	  if (!tcb)
	    nd_tcb_insert(p);
	  else if (!nd_tcb_rec_known(tcb))
	    nd_tcb_set_rec(tcb, p);	    
	}

      if (!p->is_complete)
	{
	  nd_misc_tcpdump_list_incomplete_column_visibility(TRUE);
	  trace.incomplete = TRUE;
	}

      nd_misc_pbar_inc();
    }
  nd_misc_pbar_clear();  

  nd_trace_update_tcpdump_list();
  nd_misc_statusbar_set(_("File loaded."));
  return (TRUE);
}


int
nd_trace_save(void)
{
  pcap_dumper_t  *pdt;
  ND_Packet      *p;

  if ( (pdt = pcap_dump_open(trace.pcap, trace.filename)) == NULL)
    return (0);

  for (p = trace.pl; p; p = p->next)
    {
      if (!p->is_hidden)
	pcap_dump((u_char*)pdt, &(p->ph), p->link_data);
    }
  
  pcap_dump_close(pdt);
  nd_misc_statusbar_set(_("File saved."));
  nd_trace_set_dirty(FALSE);
  return (TRUE);
}


int
nd_trace_save_as(char *filename)
{
  if (!filename)
    return (FALSE);

  FREE(trace.filename);

  trace.filename = strdup(filename);
  if (nd_trace_save())
    {
      nd_misc_set_windowtitle(trace.filename);
      return (TRUE);
    }

  return (FALSE);
}


void        
nd_trace_update_tcpdump_packet(ND_Packet *p)
{
  GtkWidget   *w;
  int          index;
  char         line[LINESIZE];

  if (!p)
    return;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);
  index = nd_trace_packet_get_index(p);

  nd_tcpdump_get_packet_line(p, line, TRUE);
  gtk_clist_set_text(GTK_CLIST(w), index, 0, line);

  if (p->is_complete)
    nd_misc_tcpdump_list_set_row_incomplete(index, FALSE);
  else
    nd_misc_tcpdump_list_set_row_incomplete(index, TRUE);

}


void    
nd_trace_update_tcpdump_list(void)
{
  GtkWidget *w;
  ND_Packet *p;
  char       line[LINESIZE];
  char      *s[2];
  int        i;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);
  gtk_clist_freeze(GTK_CLIST(w));
  gtk_clist_clear(GTK_CLIST(w));    
  gtk_clist_set_reorderable(GTK_CLIST(w), 1);

  nd_misc_statusbar_set(_("Updating tcpdump output..."));
  nd_misc_pbar_reset(trace.num_packets);

  s[0] = line;
  s[1] = "";

  for (p = trace.pl, i = 0; p; p = p->next, i++)
    {
      nd_tcpdump_get_packet_line(p, line, FALSE);
      
      gtk_clist_append(GTK_CLIST(w), s);
      
      if (trace.incomplete)
	{
	  if (p->is_complete)
	    nd_misc_tcpdump_list_set_row_incomplete(i, FALSE);
	  else
	    nd_misc_tcpdump_list_set_row_incomplete(i, TRUE);
	}

      nd_misc_pbar_inc();
    }

  nd_misc_pbar_clear();    
  gtk_clist_thaw(GTK_CLIST(w));
}


void    
nd_trace_move_packet(int from_packet, int to_packet)
{
  ND_Packet *p, *p2;

  if (to_packet > from_packet)
    to_packet++;
  
  p  = nd_trace_packet_get_nth(from_packet);
  p2 = nd_trace_packet_get_nth(to_packet);

  if (p)
    {
      if (p->prev)
	{
	  if (p->next)
	    {
	      p->prev->next = p->next;
	      p->next->prev = p->prev;
	    }
	  else
	    {
	      p->prev->next = NULL;
	      trace.pl_end = p->prev;
	    }

	}
      else if (p->next)
	{
	  trace.pl = p->next;
	  p->next->prev = NULL;
	}
      else
	{
	  /* No prev and no next -- moving doesn't make
	     much sense, so we return. */
	  return;
	}

      if (p2)
	{
	  p->next = p2;
	  p->prev = p2->prev;

	  if (p2->prev)
	    p2->prev->next = p;
	  else
	    trace.pl = p;

	  p2->prev = p;
	}
      else
	{
	  trace.pl_end->next = p;
	  p->prev = trace.pl_end;
	  trace.pl_end = p;
	  p->next = NULL;
	}

      nd_trace_set_dirty(TRUE);
    }
}


void    
nd_trace_select_packet(int index)
{
  ND_Packet   *p, *s = NULL, *s_last = NULL;
  int          i;

  p = trace.pl;
  s = trace.sel;
  i = 0;

  if (!s)
    {
      /* No selected packet yet -- we can speed things up */

      p = nd_trace_packet_get_nth(index);
      if (p)
	{
	  trace.p = p;
	  trace.sel = p;
	  p->sel_next = NULL;
	  p->sel_prev = NULL;
	  trace.num_sel++;
	}

      return;
    }

  /* We already have something selected */

  while (p)
    {
      if (i == index)
	{
	  if (s_last)
	    {
	      p->sel_prev = s_last;
	      p->sel_next = s_last->sel_next;
	      if (p->sel_next)
		p->sel_next->sel_prev = p;
	      s_last->sel_next = p;
	    }
	  else /* It's before the first selected packet */
	    {
	      p->sel_next = trace.sel;
	      trace.sel->sel_prev = p;
	      p->sel_prev = NULL;
	      trace.sel = p;
	    }

	  trace.num_sel++;
	  return;
	}

      if (p == s)
	{
	  s_last = p;
	  s = p->sel_next;
	}
      
      i++;
      p = p->next;
    }
}


void    
nd_trace_unselect_packet(int index)
{
  ND_Packet *p = NULL;
  
  if ((p = nd_trace_packet_get_nth(index)))
    {      
      if (p->sel_next)
	{
	  if (p->sel_prev)
	    {
	      p->sel_next->sel_prev = p->sel_prev;
	      p->sel_prev->sel_next = p->sel_next;
	    }
	  else
	    {
	      trace.sel = p->sel_next;
	      p->sel_next->sel_prev = NULL;
	    }
	}
      else /* Last packet in selection */
	{
	  if (p->sel_prev) /* Not single selected packet */
	    p->sel_prev->sel_next = NULL;
	  else
	    trace.sel = NULL;
	}
      
      p->sel_next = p->sel_prev = NULL;
      trace.num_sel--;
    }
}


void    
nd_trace_clear_selection(void)
{
  ND_Packet *p, *p_last = NULL;

  p_last = trace.sel;

  if (p_last)
    {
      for(p = p_last->sel_next; p; p = p->sel_next)
	{
	  p_last->sel_next = p_last->sel_prev = NULL;
	  p_last = p;
	}

      p_last->sel_next = p_last->sel_prev = NULL;
    }


  trace.sel = NULL;
  trace.num_sel = 0;
}


void    
nd_trace_full_selection(void)
{
  ND_Packet *p, *p_last = NULL;
  
  if ((p = trace.pl))
    {
      p_last = p;
      trace.sel = p_last;
      p_last->sel_prev = NULL;
      p_last->sel_next = NULL;
  
      for(p = p->next; p; p = p->next)
	{
	  p_last->sel_next = p;
	  p->sel_prev = p_last;
	  p->sel_next = NULL;
	  p_last = p;
	}

      trace.num_sel = trace.num_packets;  
    }
}


int         
nd_trace_packet_get_index(ND_Packet *needle)
{
  ND_Packet *p = trace.pl;
  int        i = 0;

  if (!needle)
    return (-1);

  while (p)
    {
      if (p == needle)
	return (i);

      i++;
      p = p->next;
    }

  printf("Packet lookup failed!");
  abort();
  
  return (-1);
}

void
nd_trace_packet_insert_at_index(ND_Packet *p, int index)
{
  ND_Packet *t;
  GtkWidget *w;
  char       line[LINESIZE];
  char      *s[2];

  nd_tcpdump_get_packet_line(p, line, TRUE);

  s[0] = line;
  s[1] = "";

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);
  gtk_clist_insert(GTK_CLIST(w), index, s);

  t = nd_trace_packet_get_nth(index);
  if (t)
    {
      if (t->prev)
	{
	  t->prev->next = p;
	  p->prev = t->prev;
	  p->next = t;
	  t->prev = p;
	}
      else
	{
	  trace.pl = p;
	  p->prev = NULL;
	  p->next = trace.pl;
	  t->prev = p;
	}

      trace.num_packets++;
      nd_misc_set_num_packets_label();
      nd_trace_set_dirty(TRUE);
    }
}


ND_Packet  *
nd_trace_packet_get_nth(int n)
{
  ND_Packet *p = trace.pl;
  int        i = 0;
  
  while (p)
    {
      if (i == n)
	return (p);

      p = p->next;
      i++;
    }

  return (NULL);
}


ND_Packet  *
nd_trace_sel(void)
{
  if (trace.apply_to_all)
    return (trace.pl);

  return (trace.sel);
}


void        
nd_trace_sel_delete(void)
{
  GtkWidget        *w;
  ND_Packet        *p, *p_sel;
  int               index;

  if (trace.num_sel == 0)
    return;

  nd_trace_packet_dehilight_fragments();

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);
  index = trace.num_packets - 1;

  /* Remove lines in the gtkclist ... */
  nd_misc_tcpdump_list_remove_selected_rows();

  /* ... and in the actual packet list: */
  for (p = trace.sel; p; )
    {
      if (p->next)
	{
	  if (p->prev)
	    {
	      p->prev->next = p->next;
	      p->next->prev = p->prev;
	    }
	  else
	    {
	      trace.pl = p->next;
	      p->next->prev = NULL;
	    }
	}
      else if (p->prev)
	{
	  p->prev->next = NULL;
	  trace.pl_end = p->prev;
	}
      else
	{
	  trace.pl = NULL;
	  trace.pl_end = NULL;
	}

      p_sel = p;
      p = p->sel_next;
      nd_packet_free(p_sel);
      trace.num_packets--;
      nd_trace_set_dirty(TRUE);
    }

  nd_misc_set_num_packets_label();
  trace.num_sel = 0;
  trace.sel = NULL;
  trace.p   = NULL;
}


void        
nd_trace_sel_hide(void)
{
  ND_Packet        *p;
  GtkStyle         *gs;
  GtkWidget        *w;
  int               index;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);
  gs = gtk_widget_get_style(GTK_WIDGET(w));

  for (p = trace.sel; p; p = p->sel_next)
    {
      index = nd_trace_packet_get_index(p);
      p->is_hidden = 1;
      gtk_clist_set_foreground(GTK_CLIST(w), index, gs->mid);
    }
}


void        
nd_trace_sel_show(void)
{
  ND_Packet        *p;
  GtkStyle         *gs;
  GtkWidget        *w;
  int               index;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);
  gs = gtk_widget_get_style(GTK_WIDGET(w));
  
  for (p = trace.sel; p; p = p->sel_next)
    {
      index = nd_trace_packet_get_index(p);
      p->is_hidden = 0;
      gtk_clist_set_foreground(GTK_CLIST(w), index, gs->text);
    }
}


void        
nd_trace_packet_hilight_fragments(ND_Packet *packet)
{
  ND_Packet        *p;
  GtkWidget        *w;
  int               i;
  u_int16_t         ip_id;

  if (!packet || packet->net_prot != ND_PROT_IP)
    return;

  nd_trace_packet_dehilight_fragments();
  trace.p_frag = packet;

  ip_id = ((struct ip*)packet->net_data)->ip_id;
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);

  for (i = 0, p = trace.pl; p; i++, p = p->next)
    {
      if (p->net_data && ((struct ip*)p->net_data)->ip_id == ip_id)
	gtk_clist_set_background(GTK_CLIST(w), i, &green[GTK_STATE_NORMAL]);
    }
}


void        
nd_trace_packet_dehilight_fragments(void)
{
  ND_Packet        *p;
  GtkStyle         *gs;
  GtkWidget        *w;
  int               i;
  u_int16_t         ip_id;

  if (!trace.p_frag)
    return;

  ip_id = ((struct ip*)trace.p_frag->net_data)->ip_id;
  w = gtk_object_get_data(GTK_OBJECT(toplevel), "tcpdump_list");
  D_ASSERT(w);
  gs = gtk_widget_get_style(GTK_WIDGET(w));
  
  for (i = 0, p = trace.pl; p; i++, p = p->next)
    {
      if (p->net_data && ((struct ip*)p->net_data)->ip_id == ip_id)
	gtk_clist_set_background(GTK_CLIST(w), i, gs->light);
    }

  trace.p_frag = NULL;
}


static gint
nd_trace_show_timestamp_timeout(gpointer data)
{
  GtkWidget *w;

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "timestamp_win");
  D_ASSERT(w);
  gtk_widget_show(w);

  return (0);
  data = NULL;
}


void        
nd_trace_show_packet_tooltip(int row, int x, int y)
{
  static GtkWidget *w;
  char   tt_text[LINESIZE];
  ND_Packet *p = nd_trace_packet_get_nth(row);
  GtkWidget *w2;
  int show_ts, show_ts_abs;
  gfloat ts_delay; 

  nd_prefs_get_item("show_timestamps", &show_ts);
  nd_prefs_get_item("show_timestamps_absolute", &show_ts_abs);    
  nd_prefs_get_item("timestamps_delay", &ts_delay);

  if (!show_ts)
    return;

  if (w)
    gtk_widget_hide(w);

  if (p)
    {
      if (timestamp_timeout_set && ts_delay > 0.1)
	gtk_timeout_remove(timestamp_timeout_id);

      if (p->prev && !show_ts_abs)
	{
	  struct timeval delta_last, delta_abs;

	  TV_SUB(&p->ph.ts, &p->prev->ph.ts, &delta_last);
	  TV_SUB(&p->ph.ts, &trace.pl->ph.ts, &delta_abs);
	  
	  snprintf(tt_text, LINESIZE, "%li.%06lis, %li.%06lis",
		   delta_abs.tv_sec, delta_abs.tv_usec,
		   delta_last.tv_sec, delta_last.tv_usec);
	}
      else
	{
	  int offset;
	  time_t tt = (time_t)p->ph.ts.tv_sec;

	  snprintf(tt_text, LINESIZE, "%s", ctime(&tt));
	  offset = strlen(tt_text)-1;
	  snprintf(tt_text + offset, LINESIZE, " +  %li usec",
		   p->ph.ts.tv_usec);
	}

      if (!w)
	{
	  w = create_timestamp_window();
	  gtk_window_set_position(GTK_WINDOW(w), GTK_WIN_POS_NONE);
	  gtk_object_set_data(GTK_OBJECT(toplevel), "timestamp_win", w);
	}

      /* If the packet is only partly captured, show how many bytes
	 are missing, too:
      */
      if (p->ph.len > p->ph.caplen)
	{
	  int offset = strlen(tt_text);

	  snprintf(tt_text + offset, LINESIZE - offset,
		   _(", %i bytes missing"), p->ph.len - p->ph.caplen);
	}


      w2 = gtk_object_get_data(GTK_OBJECT(w), "timestamp_label");
      D_ASSERT(w2);
      gtk_label_set_text(GTK_LABEL(w2), tt_text);

      /* HELP! What's the correct way to ensure that w->allocation.width has the correct
	 size now? I haven't been able to figure it out. Since our window's content is
	 simple (just the line of text), let's use the width of the text string as
	 an approximation.
      */

      gtk_widget_realize(w);
      gtk_window_reposition(GTK_WINDOW(w),
			    x - gdk_text_width(w2->style->font, tt_text, strlen(tt_text))/2,
			    y + 15);      

      timestamp_timeout_id = gtk_timeout_add(ts_delay * 1000, nd_trace_show_timestamp_timeout, NULL);
      timestamp_timeout_set = TRUE;
    }
}


void        
nd_trace_hide_packet_tooltip(void)
{
  GtkWidget *w;

  if (timestamp_timeout_set)
    {
      gtk_timeout_remove(timestamp_timeout_id);
      timestamp_timeout_set = FALSE;
    }

  w = gtk_object_get_data(GTK_OBJECT(toplevel), "timestamp_win");
  if (w)
    gtk_widget_hide(w);
}


void        
nd_trace_set_dirty(int dirty)
{
  GtkWidget *w;

  if (dirty && !trace.dirty)
    {      
      trace.dirty = TRUE;

      if ((w = gtk_object_get_data(GTK_OBJECT(toplevel), "mod_label")))
	gtk_label_set_text(GTK_LABEL(w), _("Modified."));
    }
  else if (!dirty && trace.dirty)
    {
      trace.dirty = FALSE;
      
      if ((w = gtk_object_get_data(GTK_OBJECT(toplevel), "mod_label")))
	gtk_label_set_text(GTK_LABEL(w), "");
    }
}

