/* 
 * E-XML Library:  For XML, XML-RPC, HTTP, and related.
 * Copyright (C) 2002-2008  Elias Ross
 * 
 * genman@noderunner.net
 * http://noderunner.net/~genman
 * 
 * 1025 NE 73RD ST
 * SEATTLE WA 98115
 * USA
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * $Id$
 */

package net.noderunner.xmlrpc;

/**
 * Provides encoding of raw bytes to base64-encoded characters, and
 * decoding of base64 characters to raw bytes.
 * <p>
 * This class is implemented as a singleton.  There is no storage in
 * this class, so it is thread-safe.
 *
 * @author Elias Ross
 * @version 1.0
 */

class Base64
{
	static Base64 base64 = new Base64();

	/**
	 * Code characters for values 0 to 63.
	 */
	static final char[] alphabet =
		"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="
		.toCharArray();

	/**
	 * Fast lookup table for converting to 0 to 63.
	 */
	static final byte[] codes = new byte[256];
	static {
		for (int i = 0; i < 256; i++)
			codes[i] = -1;
		for (int i = 'A'; i <= 'Z'; i++)
			codes[i] = (byte)(i - 'A');
		for (int i = 'a'; i <= 'z'; i++)
			codes[i] = (byte)(26 + i - 'a');
		for (int i = '0'; i <= '9'; i++)
			codes[i] = (byte)(52 + i - '0');
		codes['+'] = 62;
		codes['/'] = 63;
	}

	protected Base64() {
	}

	public static Base64 getInstance() {
		return base64;
	}

	/**
	 * Returns an array of base64-encoded characters to represent the
	 * passed data array.
	 *
	 * @param data the array of bytes to encode
	 * @return base64-coded character array.
	 */
	public char[] encode(byte[] data) {
		char[] out = new char[((data.length + 2) / 3) * 4];
		//
		// 3 bytes encode to 4 chars.  Output is always an even
		// multiple of 4 characters.
		//
		
		for (int i = 0, index = 0; i < data.length; i += 3,
			 index += 4)
		{
			boolean quad = false;
			boolean trip = false;
			int val = (0xFF & data[i]);
			
			val <<= 8;
			if ((i + 1) < data.length) {
				val |= (0xFF & data[i + 1]);
				trip = true;
			}
			val <<= 8;
			if ((i + 2) < data.length) {
				val |= (0xFF & data[i + 2]);
				quad = true;
			}
			out[index + 3] = alphabet[(quad ? (val & 0x3F) : 64)];
			val >>= 6;
			out[index + 2] = alphabet[(trip ? (val & 0x3F) : 64)];
			val >>= 6;
			out[index + 1] = alphabet[val & 0x3F];
			val >>= 6;
			out[index + 0] = alphabet[val & 0x3F];
		}
		return out;
	}

	/**
	 * Returns an array of bytes which were encoded in the passed
	 * character array.
	 *
	 * @param data the array of base64-encoded characters, no whitespace
	 * allowed.
	 * @return decoded data array
	 * @throws Base64Exception if the array-size was miscalculated,
	 * perhaps due to invalid characters read on input.
	 */
	public byte[] decode(char[] data) 
		throws Base64Exception
	{
		int len = ((data.length + 3) / 4) * 3;
		if (data.length > 0 && data[data.length - 1] == '=')
			--len;
		if (data.length > 1 && data[data.length - 2] == '=')
			--len;
		byte[] out = new byte[len];

		int shift = 0; // # of excess bits stored in accum
		int accum = 0; // excess bits
		int index = 0;

		for (int ix = 0; ix < data.length; ix++) {
			int value = codes[data[ix] & 0xFF]; // ignore high byte of char
			if (value >= 0) { 
				// skip over non-code
				accum <<= 6; // bits shift up by 6 each time thru
				shift += 6; // loop, with new bits being put in
				accum |= value; // at the bottom.
				if (shift >= 8) { 
					// whenever there are 8 or more shifted in,
					shift -= 8; // write them out (from the top, leaving any
					// excess at the bottom for next iteration.
					out[index++] = (byte)((accum >> shift) & 0xff);
				}
			}
		}

		if (index != out.length) 
			throw new Base64Exception("Error decoding BASE64 element: miscalculated data length!");

		return out;
	}
}
