/** -*- C++ -*-
    @file predicate/combinators.h
    @author Peter Rockai <me@mornfall.net>
    @author Enrico Zini
*/

#include <algorithm>
#include <wibble/amorph.h>
#include <wibble/range.h>
#include <wibble/cast.h>

#include <ept/predicate/predicate.h>

#ifndef EPT_PREDICATE_COMBINATORS_H
#define EPT_PREDICATE_COMBINATORS_H

namespace ept {
namespace predicate {

template< typename T, typename P1, typename P2 >
struct And: CompoundMixin< T, And, P1, P2 >
{
    And( P1 p1, P2 p2 )
        : CompoundMixin< T, predicate::And, P1, P2 >( p1, p2 ) {}

    bool operator()( const T &t ) const {
        return this->first()( t ) && this->second()( t );
    }
};

template< typename T, typename P1, typename P2 >
struct Or: CompoundMixin< T, Or, P1, P2 >
{
    Or( P1 p1, P2 p2 )
        : CompoundMixin< T, predicate::Or, P1, P2 >( p1, p2 ) {}

    bool operator()( const T &t ) const {
        return this->first()( t ) || this->second()( t );
    }
};

template< typename T, typename P, typename P2 = Void< T > >
struct Not: CompoundMixin< T, Not, P, P2 >
{
    Not( P p, P2 = Void< T >() )
        : CompoundMixin< T, predicate::Not, P, P2 >( p, Void< T >() ) {}

    bool operator()( const T &t ) { return not this->first()( t ); }

protected:
    Predicate< T > m_op;
};

template<typename Op1, typename Op2>
And< typename Op1::argument_type, Op1, Op2 > operator and( Op1 op1, Op2 op2 )
{
    return And< typename Op1::argument_type, Op1, Op2 >( op1, op2 );
}

template< typename Op1, typename Op2 >
Or< typename Op1::argument_type, Op1, Op2 > operator or( Op1 op1, Op2 op2 )
{
    return Or< typename Op1::argument_type, Op1, Op2 >( op1, op2 );
}

template< typename Op >
Not< typename Op::argument_type, Op > operator not( Op op )
{
    return Not< typename Op::argument_type, Op >( op );
}

// the complication of inheriting Pred is to allow dynamic_cast to
// cross Adaptor... evil, i know
template< typename Adapt, typename Pred >
struct Adaptor : Pred, PredicateMixin< Adapt, Adaptor< Adapt, Pred >* > {
    typedef Adaptor PredicateImplementation;
    Adaptor( const Pred &op ) : Pred( op ) {}
    const Pred &predicate() const { return *static_cast< const Pred * >( this ); }
    bool operator()( Adapt a ) {
        wibble::Maybe< typename Pred::result_type > r;
        r = a.template ifType<typename Pred::argument_type>( predicate() );
        if ( r.nothing() )
            return false;
        return r.value();
    }
    bool operator<=( const Adaptor &o ) const { return predicate() <= o.predicate(); }
    Compound< Adapt > compound() const { return Compound< Adapt >(); }
};

template<typename Adapt, typename Pred>
Adaptor<Adapt, Pred> adapt( Pred op ) {
    return Adaptor<Adapt, Pred>( op );
}

template< typename P >
Predicate< typename P::argument_type > remove(
    P p, Predicate< typename P::argument_type > rem );

template< typename P, typename R >
Predicate< typename P::argument_type > remove_helper( P p, R rem )
{
    typedef typename P::argument_type T;
    Predicate< T > ret;
    Compound< T > c = p.compound();
    if ( c.isVoid() ) {
        if( p == rem )
            return Void< T >();
        return p;
    } else {
        if (c.first() == rem) {
            if (c.second() == rem)
                return Void< T >();
            return remove( c.second(), rem );
        } else {
            if ( c.second() == rem )
                return remove( c.first(), rem );
            return c.substitute( remove( c.first(), rem ), remove( c.second(), rem ) );
        }
    }
}

template< typename P >
Predicate< typename P::argument_type > remove(
    P p, Predicate< typename P::argument_type > rem )
{
    Predicate< typename P::argument_type > a, b, rmp;
    // rmp = Remove< typename P::argument_type >();
    rmp = Void< typename P::argument_type >();
    a = remove_helper( p, rem );
    while( a != b ) {
        b = a;
        a = remove_helper( a, rmp );
    }
    if( a == rmp )
        return True< typename P::argument_type >();
    return a;
}

}
}

// vim:set ts=4 sw=4:
#endif
