// -*- mode: c++; tab-width: 4; indent-tabs-mode: t -*-
/**
 * @file cache/component/debtags/serializer.h
 * @author Enrico Zini (enrico) <enrico@enricozini.org>
 */

#ifndef EPT_CACHE_DEBTAGS_SERIALIZER_H
#define EPT_CACHE_DEBTAGS_SERIALIZER_H

#include <ept/forward.h>
#include <ept/cache/package.h>
#include <ept/cache/debtags/pkgidx.h>
#include <ept/cache/debtags/vocabulary.h>
#include <wibble/mixin.h>
#include <tagcoll/patch.h>
#if 0
#include <tagcoll/coll/intdiskindex.h>
#include <ept/cache/package.h>
#include <ept/cache/apt/packages.h>
#endif

namespace ept {
namespace t {
namespace cache {
namespace debtags {

template<typename C, typename OUT>
class StringToEpt : public wibble::mixin::OutputIterator< StringToEpt<C, OUT> >
{
	typedef typename C::Aggregator Aggregator;
	typedef typename C::Vocabulary Vocabulary;

	Aggregator& packages;
	Vocabulary& tags;
	OUT out;

public:
	StringToEpt(Aggregator& packages, Vocabulary& tags, const OUT& out) : packages(packages), tags(tags), out(out) {}

	template<typename ITEMS, typename TAGS>
	StringToEpt<C, OUT>& operator=(const std::pair<ITEMS, TAGS>& data);
};

template<typename C, typename OUT>
StringToEpt<C, OUT> stringToEpt(typename C::Aggregator& packages, typename C::Vocabulary& tags, const OUT& out)
{
	return StringToEpt<C, OUT>(packages, tags, out);
}

template<typename C, typename OUT>
class IntToEpt : public wibble::mixin::OutputIterator< IntToEpt<C, OUT> >
{
	typedef typename C::Aggregator Aggregator;
	typedef typename C::Vocabulary Vocabulary;
	typedef PkgIdx<C> PkgIdx;

	Aggregator& packages;
	PkgIdx& pkgidx;
	Vocabulary& tags;
	OUT out;

public:
	IntToEpt(Aggregator& packages, PkgIdx& pkgidx, Vocabulary& tags, const OUT& out)
		: packages(packages), pkgidx(pkgidx), tags(tags), out(out) {}

	template<typename ITEMS, typename TAGS>
	IntToEpt<C, OUT>& operator=(const std::pair<ITEMS, TAGS>& data);
};

template<typename C, typename OUT>
IntToEpt<C, OUT> intToEpt(typename C::Aggregator& packages, PkgIdx<C>& pkgidx, typename C::Vocabulary& tags, const OUT& out)
{
	return IntToEpt<C, OUT>(packages, pkgidx, tags, out);
}

template<typename OUT>
class ToInt : public wibble::mixin::OutputIterator< ToInt<OUT> >
{
	OUT out;
public:
	ToInt(const OUT& out) : out(out) {}

	template<typename ITEMS, typename TAGS>
	ToInt<OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		std::set<int> iitems;
		std::set<int> itags;
		for (typename ITEMS::const_iterator i = data.first.begin();
				i != data.first.end(); ++i)
			if (i->valid())
				iitems.insert(i->ondiskId());
		for (typename TAGS::const_iterator i = data.second.begin();
				i != data.second.end(); ++i)
			if (i->valid())
				itags.insert(i->id());
		*out = make_pair(iitems, itags);
		++out;
		return *this;
	}
};

template<typename OUT>
ToInt<OUT> toInt(const OUT& out)
{
	return ToInt<OUT>(out);
}

template<typename ITEMCONV, typename TAGCONV, typename OUT>
class Converter : public wibble::mixin::OutputIterator< Converter<ITEMCONV, TAGCONV, OUT> >
{
	ITEMCONV itemconv;
	TAGCONV tagconv;
	OUT out;

public:
	Converter(const ITEMCONV& itemconv, const TAGCONV& tagconv, const OUT& out)
		: itemconv(itemconv), tagconv(tagconv), out(out) {}

	template<typename ITEMS, typename TAGS>
	Converter<ITEMCONV, TAGCONV, OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		*out = make_pair(itemconv(data.first), tagconv(data.second));
		++out;
		return *this;
	}
};

template<typename ITEMCONV, typename TAGCONV, typename OUT>
Converter<ITEMCONV, TAGCONV, OUT> converter(const ITEMCONV& itemconv, const TAGCONV& tagconv, const OUT& out)
{
	return Converter<ITEMCONV, TAGCONV, OUT>(itemconv, tagconv, out);
}

template<typename OUT>
class ToString : public wibble::mixin::OutputIterator< ToString<OUT> >
{
	OUT out;
public:
	ToString(const OUT& out) : out(out) {}

	template<typename ITEMS, typename TAGS>
	ToString<OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		std::set<std::string> sitems;
		std::set<std::string> stags;
		for (typename ITEMS::const_iterator i = data.first.begin();
				i != data.first.end(); ++i)
			if (i->valid())
				sitems.insert(i->name());
		for (typename TAGS::const_iterator i = data.second.begin();
				i != data.second.end(); ++i)
			if (i->valid())
				stags.insert(i->fullname());
		*out = make_pair(sitems, stags);
		++out;
		return *this;
	}
};

template<typename OUT>
ToString<OUT> toString(const OUT& out)
{
	return ToString<OUT>(out);
}

template<typename C, typename OUT>
class PatchStringToInt : public wibble::mixin::OutputIterator< PatchStringToInt<C, OUT> >
{
	typedef typename C::Aggregator Aggregator;
	typedef typename C::Vocabulary Vocabulary;
	typedef typename C::Package Package;
	typedef typename C::Tag Tag;
	Aggregator& pkgs;
	Vocabulary& tags;
	OUT out;

public:
	PatchStringToInt(Aggregator& pkgs, Vocabulary& tags, const OUT& out) : pkgs(pkgs), tags(tags), out(out) {}
	PatchStringToInt<C, OUT>& operator=(const tagcoll::Patch<std::string, std::string>& patch)
	{
		Package pkg = pkgs.index().packageByName(patch.item);
		if (!pkg.valid())
			return *this;

		tagcoll::Patch<int, int> res(pkg.ondiskId());
		for (std::set<std::string>::const_iterator i = patch.added.begin();
				i != patch.added.end(); ++i)
		{
			Tag tag = tags.tagByName(*i);
			if (tag.valid())
				res.add(tag.id());
		}
		for (std::set<std::string>::const_iterator i = patch.removed.begin();
				i != patch.removed.end(); ++i)
		{
			Tag tag = tags.tagByName(*i);
			if (tag.valid())
				res.remove(tag.id());
		}
		*out = res;
		++out;
		return *this;
	}
};

template<typename C, typename OUT>
PatchStringToInt<C, OUT> patchStringToInt(typename C::Aggregator& pkgs, typename C::Vocabulary& tags, const OUT& out)
{
	return PatchStringToInt<C, OUT>(pkgs, tags, out);
}

template<typename C, typename OUT>
class PatchIntToString : public wibble::mixin::OutputIterator< PatchIntToString<C, OUT> >
{
	typedef typename C::Aggregator Aggregator;
	typedef typename C::Vocabulary Vocabulary;
	typedef typename C::Package Package;
	typedef typename C::PkgIdx PkgIdx;
	typedef typename C::Tag Tag;
	Aggregator& pkgs;
	PkgIdx& pkgidx;
	Vocabulary& tags;
	OUT out;

public:
	PatchIntToString(Aggregator& pkgs, PkgIdx& pkgidx, Vocabulary& tags, const OUT& out) : pkgs(pkgs), pkgidx(pkgidx), tags(tags), out(out) {}

	PatchIntToString<C, OUT>& operator=(const tagcoll::Patch<int, int>& patch)
	{
		const char* name = pkgidx.name(patch.item);
		if (!name)
			return *this;
		Package pkg = pkgs.index().packageByName(name);
		if (!pkg.valid())
			return *this;

		tagcoll::Patch<std::string, std::string> res(pkg.name());
		for (std::set<int>::const_iterator i = patch.added.begin();
				i != patch.added.end(); ++i)
		{
			Tag tag = tags.tagByID(*i);
			if (tag.valid())
				res.add(tag.fullname());
		}
		for (std::set<int>::const_iterator i = patch.removed.begin();
				i != patch.removed.end(); ++i)
		{
			Tag tag = tags.tagByID(*i);
			if (tag.valid())
				res.remove(tag.fullname());
		}
		*out = res;
		++out;
		return *this;
	}
};

template<typename C, typename OUT>
PatchIntToString<C, OUT> patchIntToString(typename C::Aggregator& pkgs, typename C::PkgIdx& pkgidx, typename C::Vocabulary& tags, const OUT& out)
{
	return PatchIntToString<C, OUT>(pkgs, pkgidx, tags, out);
}

template<typename OUT>
class PatchToString : public wibble::mixin::OutputIterator< PatchToString<OUT> >
{
	OUT out;

public:
	PatchToString(const OUT& out) : out(out) {}

	template<typename PKG, typename TAG>
	PatchToString<OUT>& operator=(const tagcoll::Patch<PKG, TAG>& patch)
	{
		if (!patch.item.valid())
			return *this;

		tagcoll::Patch<std::string, std::string> res(patch.item.name());
		for (typename std::set<TAG>::const_iterator i = patch.added.begin();
				i != patch.added.end(); ++i)
			if (i->valid())
				res.add(i->fullname());
		for (typename std::set<TAG>::const_iterator i = patch.removed.begin();
				i != patch.removed.end(); ++i)
			if (i->valid())
				res.remove(i->fullname());
		*out = res;
		++out;
		return *this;
	}
};

template<typename OUT>
PatchToString<OUT> patchToString(const OUT& out)
{
	return PatchToString<OUT>(out);
}

}
}
}
}

#if 0

namespace tagcoll {
namespace coll {

template<>
struct coll_traits< ept::cache::debtags::DebtagsIndex >
{
	typedef ept::cache::Package<> item_type;
	typedef ept::cache::debtags::Tag tag_type;
	typedef std::set< ept::cache::Package<> > itemset_type;
	typedef std::set<ept::cache::debtags::Tag> tagset_type;
};

}
}

namespace ept {
namespace cache {
namespace debtags {

class DebtagsIndex : public tagcoll::coll::Collection< DebtagsIndex >
{
protected:
	typedef tagcoll::coll::coll_traits< ept::cache::debtags::DebtagsIndex >::item_type Package;
	typedef tagcoll::coll::coll_traits< ept::cache::debtags::DebtagsIndex >::tag_type Tag;
	typedef tagcoll::coll::coll_traits< ept::cache::debtags::DebtagsIndex >::itemset_type PackageSet;
	typedef tagcoll::coll::coll_traits< ept::cache::debtags::DebtagsIndex >::tagset_type TagSet;

	tagcoll::coll::IntDiskIndex index;

//	PkgIdx pkgidx;


	Package itemconv(const int& item) const
	{
//		return this->items.get(item);
	}
	PackageSet itemconv(const std::set<int>& items) const
	{
//		return this->items.fromInt(items);
	}
	int itemconv(const Package& item) const
	{
//		return this->items.get(item);
	}
	std::set<int> itemconv(const PackageSet& items) const
	{
//		return this->items.toInt(items);
	}
	Tag tagconv(const int& tag) const
	{
//		return this->tags.get(tag);
	}
	TagSet tagconv(const std::set<int>& tags) const
	{
//		return this->tags.fromInt(tags);
	}
	int tagconv(const Tag& tag) const
	{
//		return this->tags.get(tag);
	}
	std::set<int> tagconv(const TagSet& tags) const
	{
//		return this->tags.toInt(tags);
	}


public:
	class const_iterator
	{
		const DebtagsIndex& index;
		int idx;
		mutable std::pair< Package, std::set<Tag> >* cached;

	public:
		// Builds an iterator
		const_iterator(const DebtagsIndex& index, int idx)
			: index(index), idx(idx), cached(0) {}
		// Builds the end iterator
		const_iterator(const DebtagsIndex& index)
			: index(index), idx(index.index.itemCount()), cached(0) {}
		~const_iterator() { if (cached) delete cached; }

		std::pair< Package, std::set<Tag> > operator*() const
		{
			return std::make_pair(index.itemconv(idx), index.tagconv(index.index.getTagsOfItem(idx)));
		}
		std::pair< Package, std::set<Tag> >* operator->() const
		{
			if (!cached)
				cached = new std::pair< Package, std::set<Tag> >(operator*());
			return cached;
		}

		const_iterator operator++()
		{
			++idx;
			if (cached) { delete cached; cached = 0; }
			return *this;
		}
		bool operator!=(const const_iterator& iter)
		{
			return idx != iter.idx;
		}
	};
	const_iterator begin() const { return const_iterator(*this, 0); }
	const_iterator end() const { return const_iterator(*this); }

	DebtagsIndex(const tagcoll::diskindex::MasterMMap& master, int pkgindex, int tagindex)
		: index(master, pkgindex, tagindex) {}

	std::set<Package> getItemsHavingTag(const Tag& tag) const
	{
		return itemconv(index.getItemsHavingTag(tagconv(tag)));
	}
	std::set<Package> getItemsHavingTags(const std::set<Tag>& tags) const
	{
		return itemconv(index.getItemsHavingTags(tagconv(tags)));
	}
	std::set<Tag> getTagsOfItem(const Package& item) const
	{
		return tagconv(index.getTagsOfItem(itemconv(item)));
	}
	std::set<Tag> getTagsOfItems(const std::set<Package>& items) const
	{
		return tagconv(index.getTagsOfItems(itemconv(items)));
	}

	bool hasTag(const Tag& tag) const
	{
		return index.hasTag(tagconv(tag));
	}

	std::set<Package> getTaggedItems() const
	{
		return itemconv(index.getTaggedItems());
	}

	std::set<Tag> getAllTags() const
	{
		return tagconv(index.getAllTags());
	}

	int getCardinality(const Tag& tag) const
	{
		return index.getCardinality(tagconv(tag));
	}

	std::set<Tag> getCompanionTags(const std::set<Tag>& tags) const
	{
		return tagconv(index.getCompanionTags(tagconv(tags)));
	}
};

#if 0
template<typename OUT>
class BigMapSerializer : public wibble::mixin::OutputIterator< BigMapSerializer<OUT> >
{
	BigMap& itemMap;
	BigMap& tagMap;
	OUT out;

public:
	BigMapSerializer(BigMap& itemMap, BigMap& tagMap, const OUT& out) :
		itemMap(itemMap), tagMap(tagMap), out(out) {}

	template<typename ITEMS, typename TAGS>
	BigMapSerializer<OUT>& operator=(const std::pair<ITEMS, TAGS>& data)
	{
		if (!data.second.empty())
		{
			*out = make_pair(itemMap.toInt(data.first), tagMap.toInt(data.second));
			++out;
		}
		return *this;
	}
};

template<typename OUT>
BigMapSerializer<OUT> bigMapSerializer(BigMap& itemMap, BigMap& tagMap, const OUT& out)
{
	return BigMapSerializer<OUT>(itemMap, tagMap, out);
}
#endif


#if 0
/**
 * Convert Facets to ints
 */
class FacetIntConverter : public Implementation<FacetIntConverter>,
	public Tagcoll::Converter<aptFront::cache::entity::Facet, int>,
	public Tagcoll::Converter<int, aptFront::cache::entity::Facet>
{
	typedef aptFront::cache::entity::Facet Facet;
	typedef Tagcoll::OpSet<aptFront::cache::entity::Facet> FacetSet;
	typedef Tagcoll::OpSet<int> IntSet;
public:
	virtual int operator()(const aptFront::cache::entity::Facet& item) const;
	virtual aptFront::cache::entity::Facet operator()(const int& item) const;

	virtual IntSet operator()(const FacetSet& item) const
		{ return Tagcoll::Converter<Facet, int>::operator()(item); }
	virtual FacetSet operator()(const IntSet& item) const
		{ return Tagcoll::Converter<int, Facet>::operator()(item); }

	static std::string componentName();
};

/**
 * Convert Facets to strings
 */
class FacetStringConverter : public Implementation<FacetStringConverter>,
	public Tagcoll::Converter<aptFront::cache::entity::Facet, std::string>,
	public Tagcoll::Converter<std::string, aptFront::cache::entity::Facet>
{
	typedef aptFront::cache::entity::Facet Facet;
	typedef Tagcoll::OpSet<aptFront::cache::entity::Facet> FacetSet;
	typedef Tagcoll::OpSet<std::string> StringSet;
public:
	virtual std::string operator()(const aptFront::cache::entity::Facet& item) const;
	virtual aptFront::cache::entity::Facet operator()(const std::string& item) const;

	virtual StringSet operator()(const FacetSet& item) const
		{ return Tagcoll::Converter<Facet, std::string>::operator()(item); }
	virtual FacetSet operator()(const StringSet& item) const
		{ return Tagcoll::Converter<std::string, Facet>::operator()(item); }

	static std::string componentName();
};

/**
 * Convert Vocabulary to ints
 */
class TagIntConverter : public Implementation<TagIntConverter>,
	public Tagcoll::Converter<aptFront::cache::entity::Tag, int>,
	public Tagcoll::Converter<int, aptFront::cache::entity::Tag>
{
	typedef aptFront::cache::entity::Tag Tag;
	typedef Tagcoll::OpSet<aptFront::cache::entity::Tag> TagSet;
	typedef Tagcoll::OpSet<int> IntSet;
public:
	virtual int operator()(const aptFront::cache::entity::Tag& item) const;
	virtual aptFront::cache::entity::Tag operator()(const int& item) const;

	virtual IntSet operator()(const TagSet& item) const
		{ return Tagcoll::Converter<Tag, int>::operator()(item); }
	virtual TagSet operator()(const IntSet& item) const
		{ return Tagcoll::Converter<int, Tag>::operator()(item); }

	static std::string componentName();
};

/**
 * Convert Vocabulary to strings
 */
class TagStringConverter : public Implementation<TagStringConverter>,
	public Tagcoll::Converter<aptFront::cache::entity::Tag, std::string>,
	public Tagcoll::Converter<std::string, aptFront::cache::entity::Tag>
{
	typedef aptFront::cache::entity::Tag Tag;
	typedef Tagcoll::OpSet<aptFront::cache::entity::Tag> TagSet;
	typedef Tagcoll::OpSet<std::string> StringSet;
public:
	virtual std::string operator()(const Tag& item) const;
	virtual Tag operator()(const std::string& item) const;

	virtual StringSet operator()(const TagSet& item) const
		{ return Tagcoll::Converter<Tag, std::string>::operator()(item); }
	virtual TagSet operator()(const StringSet& item) const
		{ return Tagcoll::Converter<std::string, Tag>::operator()(item); }

	TagSet parseTagList(const std::string& str) const;

	static std::string componentName();
};

/**
 * Convert Aggregator to ints
 */
class PackageIntConverter : public Implementation<PackageIntConverter>,
	public Tagcoll::Converter<aptFront::cache::entity::Package, int>,
	public Tagcoll::Converter<int, aptFront::cache::entity::Package>
{
	typedef aptFront::cache::entity::Package Package;
	typedef Tagcoll::OpSet<aptFront::cache::entity::Package> PackageSet;
	typedef Tagcoll::OpSet<int> IntSet;
public:
	virtual int operator()(const Package& item) const;
	virtual Package operator()(const int& item) const;

	virtual IntSet operator()(const PackageSet& item) const
		{ return Tagcoll::Converter<Package, int>::operator()(item); }
	virtual PackageSet operator()(const IntSet& item) const
		{ return Tagcoll::Converter<int, Package>::operator()(item); }

	static std::string componentName();
};

/**
 * Convert Aggregator to strings
 */
class PackageStringConverter : public Implementation<PackageStringConverter>,
	public Tagcoll::Converter<aptFront::cache::entity::Package, std::string>,
	public Tagcoll::Converter<std::string, aptFront::cache::entity::Package>
{
	typedef aptFront::cache::entity::Package Package;
	typedef Tagcoll::OpSet<aptFront::cache::entity::Package> PackageSet;
	typedef Tagcoll::OpSet<std::string> StringSet;
public:
	virtual std::string operator()(const Package& item) const;
	virtual Package operator()(const std::string& item) const;

	virtual StringSet operator()(const PackageSet& item) const
		{ return Tagcoll::Converter<Package, std::string>::operator()(item); }
	virtual PackageSet operator()(const StringSet& item) const
		{ return Tagcoll::Converter<std::string, Package>::operator()(item); }

	static std::string componentName();
};

#endif

}
}
}

#endif

#endif
