/** -*- C++ -*-
 * @file cache/component/state.h
 * @author Peter Rockai <me@mornfall.net>
 */

#include <ept/cache/apt/depcache.h>
#include <ept/cache/apt/algorithms.h>
#include <ept/cache/apt/index.h>
#include <ept/cache/package.h>
#include <ept/cache/version.h>

// needs to go there:w
#include <apt-pkg/pkgcache.h>
#include <apt-pkg/policy.h>

#include <iostream> // XXX

#ifndef EPT_CACHE_APT_STATE_H
#define EPT_CACHE_APT_STATE_H

class pkgPolicy;

namespace ept {
namespace t {
namespace cache {
namespace apt {

template< typename C >
struct PackagePolicy : public pkgPolicy
{
    PackagePolicy( const PackagePolicy &p )
        : pkgPolicy( *static_cast< const pkgPolicy * >( &p ) )
    {
        // std::cerr << "copying PackagePolicy: " << &p << " -> " << this << std::endl;
        unsigned c = Cache->Head().PackageCount;
        unsigned fc = Cache->Head().PackageFileCount;
        Pins = new Pin[c];
        std::copy( p.Pins, p.Pins+c, Pins );
        PFPriority = new signed short[fc];
        std::copy( p.PFPriority, p.PFPriority+fc, PFPriority );
        // i really hate APT API.......
    }

    PackagePolicy( Index< C > &idx )
        : pkgPolicy( &idx.aptCache() )
    {
        // std::cerr << "creating PackagePolicy: " << this << std::endl;
    }
};

template< typename C >
struct State
{
    typedef typename C::VersionPointer VersionPointer;
    typedef typename C::Aggregator Aggregator;
    typedef typename C::Package Package;
    typedef typename C::Version Version;
    typedef typename C::Relation Relation;
    typedef typename C::Index Index;

    struct DepCache : public pkgDepCache {
        DepCache( typename C::Aggregator *pkgs, typename C::State *s, pkgCache *c, pkgPolicy *p )
            : pkgDepCache( c, p ), m_state( s ), m_packages( pkgs ) {}
        typename C::State *m_state;
        typename C::Aggregator *m_packages;

        // reimplement
        virtual void MarkKeep(PkgIterator const &Pkg,bool Soft = false) {
            // std::cerr << "State::DepCache::MarkKeep( " << Pkg.Name() << " ) " << std::endl;
            pkgDepCache::MarkKeep( Pkg, Soft );
            m_state->packageChanged( m_packages->index().packageByName( Pkg.Name() ) );
        }

        virtual void MarkDelete(PkgIterator const &Pkg,bool Purge = false) {
            // std::cerr << "State::DepCache::MarkDelete( " << Pkg.Name() << " ) " << std::endl;
            pkgDepCache::MarkDelete( Pkg, Purge );
            m_state->packageChanged( m_packages->index().packageByName( Pkg.Name() ) );
        }

        virtual void MarkInstall(PkgIterator const &Pkg,bool AutoInst = true,
                                 unsigned long Depth = 0) {
            // std::cerr << "State::DepCache::MarkInstall( " << Pkg.Name() << " ) " << std::endl;
            pkgDepCache::MarkInstall( Pkg, AutoInst, Depth );
            m_state->packageChanged( m_packages->index().packageByName( Pkg.Name() ) );
        }

        virtual void AddStates(const PkgIterator &Pkg,int Add = 1) {
            pkgDepCache::AddStates( Pkg, Add );
            typename C::State::Package p = m_packages->index().packageByName( Pkg.Name() );

            assert( p.id() >= 0 );
            if ( m_state->m_states.size() <= static_cast< unsigned >( p.id() ) )
                m_state->m_states.resize( p.id() + 1, 0 );
            m_state->m_states[ p.id() ] = 0; // clear the state cache
            // any state changes need to be reflected by calling
            // AddStates, so this is the right place to do this
            // however, this method should get a better name...

            if ( p.markedNewInstall() )
                m_state->m_newInstallCount += Add;
            if ( p.markedRemove() )
                m_state->m_removeCount += Add;
            if ( p.markedUpgrade() )
                m_state->m_upgradeCount += Add;
            if ( p.markedReInstall() )
                m_state->m_reInstallCount += Add;
            if ( p.hasVersion() )
                m_state->m_availableCount += Add;
            if ( p.isUpgradable() )
                m_state->m_upgradableCount += Add;
            if ( p.isInstalled() )
                m_state->m_installedCount += Add;
        }
    };

    State( Aggregator & );
    // State( const State & ); // copy ctor - deep copying needed
    virtual ~State();

    enum Action { AInstall, ARemove, AKeep, APurge, AReInstall };
    typedef std::pair< Package, Action > Request;
    typedef wibble::Range< Request > RequestList;

    void revert( wibble::Range< Package > );
    void replay( RequestList );

    void upgrade() {
        // notifyPreChange();
        pkgAllUpgrade( aptState() );
        // notifyPostChange();
    }

    void distUpgrade() {
        // notifyPreChange();
        pkgDistUpgrade( aptState() );
        // notifyPostChange();
    }

    Index &index() const { return packages().index(); }
    Aggregator &packages() const { return m_packages; }

    static std::string sizeString( double );
    std::string downloadSizeString() { return sizeString( downloadSize() ); }
    std::string installSizeString() { return sizeString( installSize() ); }
    double downloadSize() const { return aptState().DebSize(); }
    double installSize() const { return aptState().UsrSize(); }
    int brokenCount() const { return aptState().BrokenCount(); }
    int removeCount() const { return m_removeCount; }
    int newInstallCount() const { return m_newInstallCount; }
    int upgradeCount() const { return m_upgradeCount; }
    int reInstallCount() const { return m_reInstallCount; }

    int installedCount() const { return m_installedCount; }
    int upgradableCount() const { return m_upgradableCount; }
    int availableCount() const { return m_availableCount; }

    bool changed() const;

    void packageChanged( Package p );

// ** UNSAFE METHODS FOLLOW
    typename Package::State packageState( const Package &i ) const;
    bool isInstalled( const Package &i ) const;

    // @todo add a Version variant of the above
    void markInstall( Package p ) {
        action( p, AInstall, true );
    }

    void markRemove( Package p ) {
        action( p, ARemove, true );
    }

    void markKeep( Package p ) {
        action( p, AKeep, true );
    }

    void markPurge( Package p ) {
        action( p, APurge, true );
    }

    void markReInstall( Package p ) {
        action( p, AReInstall, true );
    }

    PackagePolicy< C > &policy() { return *m_policy; }
    DepCache &aptState() const { return *m_aptState; } // XXX evil

    Version candidateVersion( Package p ) const {
        VersionPointer ptr( p.aggregator(),
                            aptState().GetCandidateVer( aptIterator( p ) ) );
        return Version( ptr );
    }

    const State &constThis() { return *this; }

    Version candidateVersion( Package p ) {
        return Version( constThis().candidateVersion( p ), packages() );
    }
    void reload() { close(); open(); }

protected: // this one is evil -- shouldn't be const either
    pkgCache::PkgIterator aptIterator( const Package &p ) const {
        return pkgCache::PkgIterator(
            index().aptCache(),
            p.pointer().package() );
    }

    void open();
    void close();

public:
    void action( Package p, Action a, bool notify );

    // Index &m_index;
    Aggregator &m_packages;
    DepCache *m_aptState;
    PackagePolicy< C > *m_policy;
    int m_removeCount;
    int m_newInstallCount;
    int m_upgradeCount;
    int m_installedCount;
    int m_upgradableCount;
    int m_availableCount;
    int m_reInstallCount;
    // typename Package::State *m_states;
    mutable std::vector< typename Package::State > m_states;
};


}
}
}
}
#endif
