/* -*- c++ -*-
 *
 * fileinfo.h
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@kmldonkey.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifndef __libkmldonkey_fileinfo_h__
#define __libkmldonkey_fileinfo_h__

#include <qstring.h>
#include <qstringlist.h>
#include <qvaluelist.h>
#include <qmap.h>
#include <qcstring.h>

#include <sys/types.h>
#include <time.h>

#include "donkeytypes.h"

class DonkeyMessage;

//! Representation of a file.

class FileInfo
{

    public:

    //! File state.
    enum State {
	Downloading = 0,
	Paused,
	Complete,
	Shared,
	Cancelled,
	New,
	Aborted,
	Queued
    };

    FileInfo(DonkeyMessage* msg, int proto);
    FileInfo(const FileInfo&);
    FileInfo() { num = 0; }
    ~FileInfo();

    void updateFileInfo(DonkeyMessage* msg, int proto);
    void updateFileInfo(const FileInfo* fi);

    //! The numerical ID of the file.
    const int fileNo() const;
    //! The network the file is on.
    const int fileNetwork() const;
    //! The file name.
    const QString& fileName() const;
    //! Alternative file names found on the network.
    const QStringList& fileNames() const;
    //! The file's MD4 hash.
    const QByteArray& fileMD4() const;
    //! The file size.
    const int64 fileSize() const;
    //! Bytes downloaded so far.
    const int64 fileDownloaded() const;
    const int fileNLocations() const;
    const int fileNClients() const;
    //! The file state.
    const State fileState() const;
    //! If state is Aborted, this is the reason why.
    const QString& fileAbortedMsg() const;
    //! String describing file chunks downloaded, one char for each chunk.
    const QByteArray& fileChunks() const;
    //! A map of networks, with each value describing chunk availability, one char for each chunk.
    const QMap<int,QByteArray>& fileAvailability() const;
    //! Current download speed.
    const double fileSpeed() const;
    //! The age of individual chunks.
    const QValueList<time_t>& fileChunksAge() const;
    //! The time the download was started (seconds since Epoch).
    const time_t fileAge() const;
    //! The file format.
    const int fileFormat() const;
    //! A string describing the file format in more detail.
    const QString& fileFormatInfo() const;
    //! Seconds since the file was last seen on the network.
    const int fileLastSeen() const;
    //! The file priority.
    const int filePriority() const;
    //! File availability described by a map of client IDs to strings similar to fileChunks().
    const QMap<int,QString>& fileSources() const;
    //! The file's comment, if any.
    const QString& fileComment() const;

    void setFileName(const QString& newname);

    void addSource(int source);
    void removeSource(int source);
    void updateAvailability(int source, const QString& avail);
    void updateDownloadStatus(DonkeyMessage* msg, int proto);

    //! Converts a QByteArray containing an MD4 hash to a readable string.
    static QString md4ToString(const QByteArray& hash);
    //! Converts a string describing an MD4 hash into a QByteArray.
    static QByteArray stringToMd4(const QString& hash);

private:

    int num, network;
    QString name;
    QStringList names;
    QByteArray md4;
    int64 size, downloaded;
    int nlocations, nclients;
    State state;
    QString abortedmsg;
    QByteArray chunks;
    QMap<int,QByteArray> availability;
    double speed;
    QValueList<time_t> chunks_age;
    time_t age;
    int format;
    QString formatinfo;
    int lastseen, priority;
    QString comment;

    QMap<int,QString> sources;
};

#endif
