#include <gperiodic.h>
#include <errno.h>

struct table_entry header = {
  N_("Name: "),
  N_("Symbol: "),
  N_("Atomic Number: "),
  N_("Atomic Weight (u.m): "),
  N_("Density (g/cm) : "),
  N_("Melting Point (K): "),
  N_("Boiling Point (K): "),
  N_("Atomic Radius (pm): "),
  N_("Covalent Radius (pm): "),
  N_("Ionic Radius (pm): "),
  N_("Atomic Volume (cm/mol): "),
  N_("Specific Heat (@20C J/g mol): "),
  N_("Fusion Heat (kJ/mol): "),
  N_("Evaporation Heat (kJ/mol): "),
  N_("Termal Conductivity (@25C W/m K): "),
  N_("Pauling Negativity Number: "),
  N_("First Ionizing Energy (kJ/mol): "),
  N_("Oxidation states: "),
  N_("Electronic Configuration: "),
  N_("Lattice structure: "),
  N_("Lattice constant (): ")
};

double temperature = 298.15;
GdkColor red_yellow2[3] = {
  {0x7FFFFFFF, 0x0000, 0x3000, 0x0000},
  {0x7FFFFFFF, 0x0000, 0xC000, 0x0000},
  {0x7FFFFFFF, 0x0000, 0x3000, 0xC000}
};

GdkColor yellow_white2[3]= {
  {0x7FFFFFFF, 0x0000, 0x3000, 0x0000},
	{0x7FFFFFFF, 0xC000, 0xC000, 0x0000},
	{0x7FFFFFFF, 0xC000, 0xC000, 0xC000}
};

struct colorscale solid_fluid_cs = { 3, red_yellow2 };

double solid_fluid_crid( struct table_entry *entry ) 
{
  double tm;
	if( entry && entry->info[MELTING] && strcmp(entry->info[MELTING], "n/a") ) {
	  tm = strtod(entry->info[MELTING],0);
		if(!errno) tm = 0;
		if( temperature < tm ) return 0.0;
		else return 1.0;
	}

	return - 1.0;
}

struct colorscale condensed_gas_cs = { 3, yellow_white2 };

double condensed_gas_crit(struct table_entry *entry)
{
  double tb;
  if(entry && entry->info[BOILING] && strcmp(entry->info[BOILING],"n/a")) {
    tb = strtod(entry->info[BOILING],0);
    if (!errno) tb = 0;
    if (temperature < tb) return 0.0;
    else return 1.0;
  }
  else return -1.0;
}


/*
 * NAME  SYMBOL  NUMBER  WEIGHT  DENSITY MELTING  BOILING  ATOMIC_RADIUS COVALENT_RADIUS IONIC_RADIUS ATOMIC_VOLUME SPECIFIC_HEAT FUSION_HEAT EVAPORATION_HEAT TERMAL_CONDUCTIVITY PAULING  ENERGY OXIDATION CONFIGURATION MAX_INFO_NR
 *  X_POS  Y_POS  PIXEL_DEPTH  RED  GREEN  BLUE
 *
 */
struct table_entry table [] = {
 
  {N_("Hydrogen"), "H", "1", "1.00794","0.0708 (@ -253C)", "14.01", "20.28","79","32","154 (-1e)","14.1","14.267 (H-H)","0.117 (H-H)","0.904 (H-H)","0.1815", "2.20","1311.3","1, -1","1s","HEX","3.750", 1, 1, 0x7FFFFFFF, 0xAAAA, 0xFFFF, 0xAAAA},
  
  {N_("Helium"), "He", "2", "4.002602","0.147 (@ -270C)", "0.95", "4.216", "0.0","-","93","31.8","5.188","-","0.08","0.152","-","2361.3","-","1s","HEX","3.570", 18, 1, 0x7FFFFFFF, 0xC000, 0x7000, 0x0},

  {N_("Lithium"), "Li", "3", "6.941","0.534", "553.69", "1118.15","155","163","68 (+1e)","13.1","3.489","2.89","148","84.8", "0.98","519.9","1","[He]2s","BCC","3.490", 1, 2, 0x7FFFFFFF, 0x4444, 0x4444, 0xFFFF},
 
  {N_("Beryllium"), "Be", "4", "9.01218","1.848", "1551", "3243", "112","90","35 (+2e)","5.0","1.824","12.21","309","201","1.57","898.8","2","[He]2s","HEX","2.290", 2, 2, 0x7FFFFFFF, 0xFFFF, 0x4444, 0x4444},
  
  {N_("Boron"), "B", "5", "10.811","2.34", "2573", "3931","98","82","23 (+3e)","4.6","1.025","23.60","504.5","27.4", "2.04","800.2","3","[He]2s2p","TET","8.730", 13, 2, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
  
  {N_("Carbon"), "C", "6", "12.011","2.25 (graphite)", "3820", "5100", "91","77","16 (+4e) 260 (-4e)","5.3","0.711","-","-","1.59","2.55","1085.7","4, 2, -4", "[He]2s2p","DIA","3.570",14, 2, 0x7FFFFFFF, 0xAAAA, 0xFFFF, 0xAAAA},
  
  {N_("Nitrogen"), "N", "7", "14.00674","0.808 (@ -195.8C)", "63.29", "77.4","92","75","13 (+5e) 171 (-3e)","17.3","1.042 (N-N)","-","-","0.026", "3.04","1401.5","5, 4, 3, 2, -3","[He]2s2p","HEX","4.039", 15, 2, 0x7FFFFFFF, 0xAAAA, 0xFFFF, 0xAAAA},
  
  {N_("Oxygen"), "O", "8", "15.9994","1.149 (@ -183C)", "54.8", "90.19","-","73","132 (-2e)","14.0","0.916 (O-O)" ,"-","-","0.027", "3.44","1313.1","-2, -1","[He]2s2p","CUB","6.830", 16, 2, 0x7FFFFFFF, 0xAAAA, 0xFFFF, 0xAAAA},
  
  {N_("Fluorine"), "F", "9", "18.998403","1.108 (@ -189C)", "53.53", "85.01","-","72","133 (-1e)","17.1","0.824 (F-F)","0.51 (F-F)","6.54 (F-F)","0.028", "3.98","1680.0","-1","[He]2s2p","MCL","-", 17, 2, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
  
  {N_("Neon"), "Ne", "10", "20.1797","1.204 (@ -246C)", "48", "27.1","-","71","-","16.8","1.029","-","1.74","(0.0493)", "0.0","2079.4","-","[He]2s2p","FCC","4.430", 18, 2, 0x7FFFFFFF, 0xC000, 0x7000, 0x0},

  {N_("Sodium"), "Na", "11", "22.989768","0.971", "370.96", "1156.1","190","154","97 (+1e)","23.7","1.222","2.64","97.9","142.0", "0.93","495.6","1","[Ne]3s","BCC","4.230", 1, 3, 0x7FFFFFFF, 0x4444, 0x4444, 0xFFFF},
  
  {N_("Magnesium"), "Mg", "12", "24.305","1.738", "922", "1363","160","136","66 (+2e)","14.0","1.025","9.20","131.8","156", "1.31","737.3","2","[Ne]3s","HEX","3.210", 2, 3, 0x7FFFFFFF, 0xFFFF, 0x4444, 0x4444},
  
  {N_("Aluminum"), "Al", "13", "26.981539","2.6989", "933.5", "2740","143","118","51 (+3e)","10.0","0.900","10.75","284.1","237", "1.61","577.2","3","[Ne]3s3p","FCC","4.050", 13, 3, 0x7FFFFFFF, 0x0, 0xC000, 0xFFFF},
  
  {N_("Silicon"), "Si", "14", "28.0855","2.33", "1683", "2628","132","111","42 (+4e) 271  (-4e)","12.1","0.703","50.6","383","149", "1.90","786.0","4, -4","[Ne]3s3p","DIA","5.430", 14, 3, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
  
  {N_("Phosphorus"), "P", "15", "30.973762","1.82 (white phosphorus)", "317.3", "553","128","106","35 (+5e) 212 (-3e)","17.0","0.757","2.51","49.8","(0.236)", "2.19","1011.2","5, 3, -3","[Ne]3s3p","CUB","7.170", 15, 3, 0x7FFFFFFF, 0xAAAA, 0xFFFF, 0xAAAA},
  
  {N_("Sulfur"), "S", "16", "32.066","2.070", "386", "717.824","127","102","30 (+6e) 184 (-2e)","15.5","0.732","1.23","10.5","0.27", "2.58","999.0","6, 4, 2, -2","[Ne]3s3p","ORC","10.470", 16, 3, 0x7FFFFFFF, 0xAAAA, 0xFFFF, 0xAAAA},
 
  {N_("Chlorine"), "Cl", "17", "35.4527","1.56 (@ -33.6C)", "172.2", "238.6","-","99","27 (+7e) 181 (-1e)","18.7","0.477 (Cl-Cl)","6.41 (Cl-Cl)","20.41 (Cl-Cl)","0.009", "3.16","1254.9","7, 5, 3, 1, -1","[Ne]3s3p","ORC","6.240", 17, 3, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
  
  {N_("Argon"), "Ar", "18", "39.948","1.40 (@ -186C)", "83.8", "87.3","2-","98","-","24.2","0.138","-","6.52","0.0177", "0.0","1519.6","-","[Ne]3s3p","FCC","5.260", 18, 3, 0x7FFFFFFF, 0xC000, 0x7000, 0x0},

  {N_("Potassium"), "K", "19", "39.0983","0.856", "336.8", "1047","235","203","133 (+1e)","45.3","0.753","102.5","2.33","79.0", "0.82","418.5","1","[Ar]4s","BCC","5.230", 1, 4, 0x7FFFFFFF, 0x4444, 0x4444, 0xFFFF},
 
  {N_("Calcium"), "Ca", "20", "40.078","1.55", "1112", "1757","197","174","99 (+2e)","29.9","0.653","9.20","153.6","(201)", "1.00","589.4","2","[Ar]4s","FCC","5.580", 2, 4, 0x7FFFFFFF, 0xFFFF, 0x4444, 0x4444},
 
  {N_("Scandium"), "Sc", "21", "44.95591","2.99", "1814", "3104","162","144","72.3 (+3e)","15.0","0.556","15.8","332.7","15.8", "1.36","630.8","3","[Ar]3d4s","HEX","3.310", 3, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Titanium"), "Ti", "22", "47.88","4.54", "1933", "3560","147","132","68 (+4e) 94 (+2e)","10.6","0.523","18.8","422.6","21.9", "1.54","657.8","4, 3","[Ar]3d4s","HEX","2.950", 4, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Vanadium"), "V", "23", "50.9415","6.11", "2160", "3650","134","122","59 (+5e) 74 (+3e)","8.35","0.485","17.5","460","30.7", "1.63","650.1","5, 4, 3, 2, 0","[Ar]3d4s","BCC","3.020", 5, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},

  {N_("Chromium"), "Cr", "24", "51.9961","7.18", "2130", "2945","130","118","52 (+6e) 63 (+3e)","7.23","0.488","21","342","93.9", "1.66","652.4","6, 3, 2, 0","[Ar]3d4s","BCC","2.880", 6, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Manganese"), "Mn", "25", "54.93805","7.21", "1517", "2235","135","117","46 (+7e) 80 (+2e)","7.39","0.477","(13.4)","221","(7.8)", "1.55","716.8","7, 6, 4, 3, 2, 0, -1","[Ar]3d4s","CUB","8.890", 7, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Iron"), "Fe", "26", "55.847","7.874", "1808", "3023","126","117","64 (+3e) 74 (+2e)","7.1","0.443","13.8","~340","80.4", "1.83","759.1", "6, 3, 2, 0, -2","[Ar]3d4s","BCC","2.870",8, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},

  {N_("Cobalt"), "Co", "27", "58.9332","8.9", "1768", "3143","125","116","63 (+3e) 72 (+2e)","6.7","0.456","15.48","389.1","100", "1.88","758.1","3, 2, 0, -1","[Ar]3d4s","HEX","2.510", 9, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Nickel"), "Ni", "28", "58.6934","8.902", "1726", "3005","124","115","69 (+2e)","6.6","0.443","17.61","378.6","90.9", "1.91","736.2","3, 2, 0","[Ar]3d4s","FCC","3.520", 10, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Copper"), "Cu", "29", "63.546","8.96", "1356.6", "2840","128","117","72 (+2e) 96 (+1e)","7.1","0.385","13.01","304.6","401", "1.90","745.0","2, 1","[Ar]3d4s","FCC","3.610", 11, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Zinc"), "Zn", "30", "65.39","7.133", "692.73", "1180","138","125","74 (+2e)","9.2","0.388","7.28","114.8","116", "1.65","905.8","2","[Ar]3d4s","HEX","2.660", 12, 4, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Gallium"), "Ga", "31", "69.723","5.91", "302.93", "2676","141","126","62 (+3e) 81 (+1e)","11.8","0.372","5.59","270.3","28.1", "1.81","578.7","3","[Ar]3d4s4p","ORC","4.510", 13, 4, 0x7FFFFFFF, 0x0, 0xC000, 0xFFFF},
  
  {N_("Germanium"), "Ge", "32", "72.61","5.323", "1210.6", "3103","137","122","53 (+4e) 73 (+2e)","13.6","0.322","36.8","328","60.2", "2.01","760.0","4","[Ar]3d4s4p","DIA","5.660", 14, 4, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
 
  {N_("Arsenic"), "As", "33", "74.92159","5.73 (grey arsenic)", "1090", "876","139","120","46 (+5e) 222 (-3e)","13.1","0.328","-","32.4","(50.2)", "2.18","946.2","5, 3, -2","[Ar]3d4s4p","RHL","4.130", 15, 4, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
 
  {N_("Selenium"), "Se", "34", "78.96","4.79", "490", "958.1","140","116","42 (+6e) 191 (-2e)","16.5","0.321 (Se-Se)","5.23","59.7","0.52", "2.55","940.4","6, 4, -2","[Ar]3d4s4p","HEX","4.360", 16, 4, 0x7FFFFFFF, 0xAAAA, 0xFFFF, 0xAAAA},
 
  {N_("Bromine"), "Br", "35", "79.904","3.12", "265.9", "331.9","-","114","47 (+5e) 196 (-1e)","23.5","0.473 (Br-Br)","10.57 (Br-Br)","29.56 (Br-Br)","0.005", "2.96","1142.0","7, 5, 3, 1, -1","[Ar]3d4s4p","ORC","6.670", 17, 4, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
 
  {N_("Krypton"), "Kr", "36", "83.8","2.155 (@ -153C)", "116.6", "120.85","-","112","-","32.2","0.247","-","9.05","0.0095", "0.0","1350.0","2","[Ar]3d4s4p","FCC","5.720", 18, 4, 0x7FFFFFFF, 0xC000, 0x7000, 0x0},

  {N_("Rubidium"), "Rb", "37", "85.4678","1.532", "312.2", "961","248","216","147 (+1e)","55.9","0.360","2.20","75.8","58.2", "0.82","402.8","1","[Kr]5s","BCC","5.590", 1, 5, 0x7FFFFFFF, 0x4444, 0x4444, 0xFFFF},

  {N_("Strontium"), "Sr", "38", "87.62","2.54", "1042", "1657","215","191","112 (+2e)","33.7","0.301","9.20","144","(35.4)", "0.95","549.0","2","[Kr]5s","FCC","6.080", 2, 5, 0x7FFFFFFF, 0xFFFF, 0x4444, 0x4444},
 
  {N_("Yttrium"), "Y", "39", "88.90585","4.47", "1795", "3611","178","162","89.3 (+3e)","19.8","0.284","11.5","367","(17.2)", "1.22","615.4","3","[Kr]4d5s","HEX","3.650", 3, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},

  {N_("Zirconium"), "Zr", "40", "91.224","6.506", "2125", "4650","160","145","79 (+4e)","14.1","0.281","19.2","567","22.7", "1.33","659.7","4","[Kr]4d5s","HEX","3.230", 4, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Niobium"), "Nb", "41", "92.90638","8.57", "2741", "5015","146","134","69 (+5e)","10.8","0.268","26.8","680","53.7", "1.6","663.6","5, 3","[Kr]4d5s","BCC","3.300", 5, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Molybdenum"), "Mo", "42", "95.94","10.22", "2890", "4885","139","130","62 (+6e) 70 (+4e)","9.4","0.251","28","~590","(138)", "2.16","684.8","6, 5, 4, 3, 2, 0","[Kr]4d5s","BCC","3.150", 6, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Technetium"), "Tc", "43", "97.9072","11.5", "2445", "5150","136","127","56 (+7e)","8.5","0.243","23.8","585","50.6", "1.9","702.2","7","[Kr]4d5s","HEX","2.740", 7, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Ruthenium"), "Ru", "44", "101.07","12.41", "2583", "4173","134","125","67 (+4e)","8.3","0.238","(25.5)","-","117.0", "2.2","710.3","8, 6, 4, 3, 2, 0, -2","[Kr]4d5s","HEX","2.700", 8, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Rhodium"), "Rh", "45", "102.9055","12.41", "2239", "4000","134","125","68 (+3e)","8.3","0.244","21.8","494","150", "2.28","719.5","5, 4, 3,  2, 1, 0","[Kr]4d5s","FCC","3.800", 9, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Palladium"), "Pd", "46", "106.42","12.02", "1825", "3413","137","128","65 (+4e) 80 (+2e)","8.9","0.244","17.24","372.4","71.8", "2.20","803.5","4,  2, 0","[Kr]4d5s","FCC","3.890", 10, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Silver"), "Ag", "47", "107.8682","10.5", "1235.1", "2485","144","134","89 (+2e) 126 (+1e)","10.3"," 0.237","11.95","254.1","429", "1.93","730.5","2, 1","[Kr]4d5s","FCC","4.090", 11, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Cadmium"), "Cd", "48", "112.411","8.65", "594.1", "1038","154","148","97 (+2e)","13.1","0.232","6.11","59.1","96.9", "1.69","867.2","2","[Kr]4d5s","HEX","2.980", 12, 5, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Indium"), "In", "49", "114.818","7.31", "429.32", "2353","166","144","81 (+3e)","15.7","0.234","3.24","225.1","81.8", "1.78","558.0","3","[Kr]4d5s5p","TET","4.590", 13, 5, 0x7FFFFFFF, 0x0, 0xC000, 0xFFFF},
 
  {N_("Tin"), "Sn", "50", "118.71","7.31", "505.1", "2543","162","141","71 (+4e) 93 (+2)","16.3","0.222","7.07","296","66.8", "1.96","708.2", "4, 2","[Kr]4d5s5p","TET","5.820",14, 5, 0x7FFFFFFF, 0x0, 0xC000, 0xFFFF},
 
  {N_("Antimony"), "Sb", "51", "121.760","6.691", "903.9", "1908","159","140","62 (+6e) 245 (-3)","18.4","0.205","20.08","195.2","24.43", "2.05","833.3","5, 3, -2","[Kr]4d5s5p","RHL","4.510", 15, 5, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
 
  {N_("Tellurium"), "Te", "52", "127.6","6.24", "722.7", "1263","160","136","56 (+6e) 211 (-2e)","20.5","0.201","17.91","49.8","14.3", "2.1","869.0","6, 4, 2","[Kr]4d5s5p","HEX","4.450", 16, 5, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},

  {N_("Iodine"), "I", "53", "126.90447","4.93", "386.7", "457.5","-","133","50 (+7e) 220 (-1e)","25.7","0.427 (I-I)","15.52 (I-I)","41.95 (I-I)","(0.45)", "2.66","1008.3","7, 5, 1, -1","[Kr]4d5s5p","ORC","7.720", 17, 5, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},

  {N_("Xenon"), "Xe", "54", "131.29","3.52 (@ -109C)", "161.3", "166.1","-","131","-","42.9","0.158","-","12.65","0.0057", "0.0","1170.0","7","[Kr]4d5s5p","FCC","6.200", 18, 5, 0x7FFFFFFF, 0xC000, 0x7000, 0x0},

  {N_("Cesium"), "Cs", "55", "132.90543","1.873", "301.6", "951.6", "267","235","167 (+1e)","70.0","0.241","2.09","68.3","35.9", "0.79","375.5","1","[Xe]6s","BCC","6.050", 1, 6, 0x7FFFFFFF, 0x4444, 0x4444, 0xFFFF},
  
  {N_("Barium"), "Ba", "56", "137.327","3.5", "1002", "1910","222","198","134 (+2e)","39.0","0.192","7.66","142.0","(18.4)", "0.89","502.5","2","[Xe]6s","BCC","5.020", 2, 6, 0x7FFFFFFF, 0xFFFF, 0x4444, 0x4444},

  {N_("Lanthanum"), "La", "57", "138.9055","6.15", "1194", "3730","187","169","101.6 (+3e)","22.5","0.197","8.5","402","13.4", "1.10","541.1","3","[Xe]6d6s","HEX","3.750", 3, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Cerium"), "Ce", "58", "140.115","6.757", "1072", "3699","181","165","92 (+4e) 103.4 (+3e)","21.0","0.205","5.2","398","11.3", "1.12","540.1","4, 3","[Xe]4f5d6s","FCC","5.160", 4, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Praseodymium"), "Pr", "59", "140.90765","6.773", "1204", "3785","182","165","90 (+4e) 101.3 (+3e)","20.8","0.192","11.3","331","12.5", "1.13","526.6","4, 3","[Xe]4f5d6s","HEX","3.670", 5, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Neodymium"), "Nd", "60", "144.24","7.007", "1294", "3341","182","184","99.5 (+3e)","20.6","0.205","7.1","289","(16.5)", "1.14","531.5","3","[Xe]4f5d6s","HEX","3.660", 6, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Promethium"), "Pm", "61", "144.9127","7.2", "1441", "3000","-","163","97.9 (+3e)","-","0.185","-","-","17.9", "0.0","536","3","[Xe]4f5d6s","n/a","n/a", 7, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Samarium"), "Sm", "62", "150.36","7.520", "1350", "2064","181","162","96.4 (+3e)","19.9","0.180","8.9","165","(13.3)", "1.17","540.1","3, 2","[Xe]4f5d6s","RHL","9.000",8, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Europium"), "Eu", "63", "151.965","5.243", "1095", "1870","199","185","95 (+3e) 109 (+2e)","28.9","0.176","-","176","13.9", "0.0","546.9","3, 2","[Xe]4f5d6s","BCC","4.610", 9, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Gadolinium"), "Gd", "64", "157.25","7.900", "1586", "3539","179","161","93.8 (+3e)","19.9","0.230","-","398","(10.5)", "1.20","594.2","3","[Xe]4f5d6s","HEX","3.640", 10, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Terbium"), "Tb", "65", "158.92534","8.229", "1629", "3296","180","159","84 (+4e) 92.3 (+3e)","19.2","0.183","-","389","11.1", "1.2","569","4, 3","[Xe]4f5d6s","HEX","3.600", 11, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Dysprosium"), "Dy", "66", "162.50","8.55", "1685", "2835","180","159","90.8 (+3e)","19.0","0.173","-","291","10.7", "-","567","3","[Xe]4f5d6s","HEX","3.590", 12, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Holmium"), "Ho", "67", "164.93032","8.795", "1747", "2968","179","158","89.4 (+3e)","18.7","0.164","-","301","(16.2)", "1.23","574","3","[Xe]4f5d6s","HEX","3.580", 13, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Erbium"), "Er", "68", "167.26","9.06", "1802", "3136","178","157","88.1 (+3e)","18.4","0.168","-","317","(14.5)", "1.24","581","3","[Xe]4f5d6s","HEX","3.560", 14, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Thulium"), "Tm", "69", "168.93421","9.321", "1818", "2220","177","156","87 (+3e)","18.1","0.160","-","232","(16.9)", "1.25","589","3, 2","[Xe]4f5d6s","HEX","3.540", 15, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Ytterbium"), "Yb", "70", "173.04","6.9654", "1097", "1466","194","-","85.8 (+3e) 93 (+2e)","24.8","0.145","3.35","159","(34.9)", "1.1","603","3, 2","[Xe]4f5d6s","FCC","5.490", 16, 9, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},

  {N_("Lutetium"), "Lu", "71", "174.967","9.8404", "1936", "3668","175","156","85 (+3e)","17.8","0.155","-","414","(16.4)", "1.27","513","3","[Xe]4f5d6s","HEX","3.510", 3, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Hafnium"), "Hf", "72", "178.49","13.31", "2503", "5470","167","144","78 (+4e)","13.6","0.146","(25.1)","575","23.0", "1.3","575.2","4","[Xe]4f5d6s","HEX","3.200", 4, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Tantalum"), "Ta", "73", "180.9479","16.654", "3269", "5698","149","134","68 (+5e)","10.9","0.140","24.7","758","57.5", "1.5","760.1","5","[Xe]4f5d6s","BCC","3.310", 5, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Tungsten"), "W", "74", "183.84","19.3", "3680", "5930","141","130","62 (+6e) 70 (+4e)","9.53","0.133","(35)","824","173", "1.7","769.7","6, 5, 4, 3, 2, 0","[Xe]4f5d6s","BCC","3.160", 6, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Rhenium"), "Re", "75", "186.207","21.02", "3453", "5900","137","128","53 (+7e) 72 (+4e)","8.85","0.138","34","704","48.0", "1.9","759.1","5, 4, 3, 2, -1","[Xe]4f5d6s","HEX","2.760", 7, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Osmium"), "Os", "76", "190.23","22.57", "3327", "5300","135","126","69 (+6e) 88 (+4e)","8.43","0.131","31.7","738","(87.6)", "2.2","819.8","8, 6, 4, 3, 2, 0, -2","[Xe]4f5d6s","HEX","2.740", 8, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Iridium"), "Ir", "77", "192.22","22.42", "2683", "4403","136","127","68 (+4e)","8.54","0.133","27.61","604","147", "2.20","868.1","6, 4, 3, 2, 1, 0, -1","[Xe]4f5d6s","FCC","3.840", 9, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Platinum"), "Pt", "78", "195.08","21.45", "2045", "4100","139","130","65 (+4e) 80 (+2e)","9.10","0.133","21.76","~470","71.6", "2.28","868.1","4, 2, 0","[Xe]4f5d6s","FCC","3.920", 10, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Gold"), "Au", "79", "196.96654","19.3", "1337.58", "3080","146","134","85 (+3e) 137 (+1e)","10.2","0.129","12.68","~340","318", "2.54","889.3","3, 1","[Xe]4f5d6s","FCC","4.080", 11, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Mercury"), "Hg", "80", "200.59","13.546 (@ +20C)", "234.28", "629.73","157","149","110 (+2e) 127 (+1e)","14.8","0.138","2.295","58.5","8.3", "2.00","1006.0","2, 1", "[Xe]4f5d6s","RHL","2.990",12, 6, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Thallium"), "Tl", "81", "204.3833","11.85", "576.6", "1730","171","148","95 (+3e) 147 (+1e)","17.2","0.128","4.31","162.4","46.1", "1.62","588.9","3, 1","[Xe]4f5d6s6p","HEX","3.460", 13, 6, 0x7FFFFFFF, 0x0, 0xC000, 0xFFFF},
 
  {N_("Lead"), "Pb", "82", "207.2","11.35", "600.65", "2013","175","147","84 (+4e) 120 (+2e)","18.3","0.159","4.77","177.8","35.3", "1.8","715.2","4, 2","[Xe]4f5d6s6p","FCC","4.950", 14, 6, 0x7FFFFFFF, 0x0, 0xC000, 0xFFFF},
 
  {N_("Bismuth"), "Bi", "83", "208.98037","9.747", "544.5", "1883","170","146","74 (+5e) 96 (+3e)","21.3","0.124","11.00","172.0","7.9", "2.02","702.9","5, 3","[Xe]4f5d6s6p","RHL","4.750", 15, 6, 0x7FFFFFFF, 0x0, 0xC000, 0xFFFF},
 
  {N_("Polonium"), "Po", "84", "208.9824","9.32", "527", "1235","176","146","67 (+6e)","22.7","0.125","(10)","(102.9)","-", "2.0","813.1","6, 4, 2","[Xe]4f5d6s6p","SC","3.350", 16, 6, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
 
  {N_("Astatine"), "At", "85", "209.9871","n/a", "575", "610","-","(145)","62 (+7e)","-","-","-","-","-", "2.2","916.3","7, 5, 3, 1, -1","[Xe]4f5d6s6p","n/a","n/a", 17, 6, 0x7FFFFFFF, 0xFFFF, 0x4444, 0xFFFF},
 
  {N_("Radon"), "Rn", "86", "222.0176","4.4 (@ -62C)", "202", "211.4","-","-","-", "-","0.094","-","18.1","0.0036","-","1036.5","-","[Xe]4f5d6s6p","FCC","n/a", 18, 6, 0x7FFFFFFF, 0xC000, 0x7000, 0x0},

  {N_("Francium"), "Fr", "87", "223.0197","n/a", "300", "950","-","-","180 (+1e)","-","-","15","-","-", "0.7","~375","2","[Rn]7s","BCC","n/a", 1, 7, 0x7FFFFFFF, 0x4444, 0x4444, 0xFFFF},

  {N_("Radium"), "Ra", "88", "226.0254","(5.5)", "973", "1413","-","-","143 (+2e)","45.0","0.120","(9.6)","(113)","(18.6)", "0.9","509.0","2","[Rn]7s","n/a","n/a", 2, 7, 0x7FFFFFFF, 0xFFFF, 0x4444, 0x4444},

  {N_("Actinium"), "Ac", "89", "227.0278","n/a", "1320", "3470","188","-","118 (+3e)","22.54","-","(10.5)","(292.9)","-", "1.1","665.5","3","[Rn]6d7s","FCC","5.310", 3, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},

  {N_("Thorium"), "Th", "90", "232.0381","11.78", "2028", "5060","180","165","102 (+4e)","19.8","0.113","16.11","513.7","(54.0)", "1.3","670.4","4","[Rn]5f6d7s","FCC","5.080", 4, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},

  {N_("Protactinium"), "Pa", "91", "231.03588","15.37", "2113", "4300","161","-","89 (+5e) 113 (+3e)","15.0","0.121","16.7","481.2","-", "1.5","n/a","5, 4","[Rn]5f6d7s","TET","3.920", 5, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},

  {N_("Uranium"), "U", "92", "238.0289","19.05", "1405.5", "4018","138","142","80 (+6e) 97 (+4e)","12.5","0.115","12.6","417","27.5", "1.38","686.4","6, 5, 4, 3","[Rn]5f6d7s","ORC","2.850", 6, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},

  {N_("Neptunium"), "Np", "93", "237.048","20.25", "913", "4175","130","-","95 (+4e) 110 (+3e)","21.1","-","(9.6)","336","(6.3)", "1.36","n/a","6, 5, 4, 3","[Rn]5f6d7s","ORC","4.720", 7, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Plutonium"), "Pu", "94", "244.0642","19.84", "914", "3505","151","-","93 (+4e) 108 (+3e)","-","-","2.8","343.5","(6.7)", "1.28","491.9","6, 5, 4, 3","[Rn]5f6d7s","MCL","n/a", 8, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Americium"), "Am", "95", "243.0614","13.67", "1267", "2880","173","-","92 (+4e) 107 (+3e)","20.8","-","(10.0)","238.5","-", "1.3","n/a","6, 5, 4, 3","[Rn]5f6d7s","n/a","n/a", 9, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Curium"), "Cm", "96", "247.0703","13.51", "1340", "n/a","299","n/a","n/a","18.28","n/a","n/a","n/a","n/a", "1.3","(580)","4, 3","[Rn]5f6d7s","n/a","n/A", 10, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Berkelium"), "Bk", "97", "247.0703","13.25", "n/a", "n/a","297","n/a","n/a","n/a","n/a","n/a","n/a","n/a", "1.3","(600)","4, 3","[Rn]5f6d7s","n/a","n/a", 11, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Californium"), "Cf", "98", "251.0796","15.1", "900", "n/a","295","n/a","n/a","n/a","n/a","n/a","n/a","n/a", "1.3","(610)","4, 3","[Rn]5f6d7s","n/a","n/a", 12, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Einsteinium"), "Es", "99", "252.083", "n/a","n/a", "1130", "292","n/a","n/a","n/a","n/a","n/a","n/a","n/a", "1.3","(620)","3","[Rn]5f6d7s","n/a","n/a", 13, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Fermium"), "Fm", "100", "257.0951", "n/a", "1800", "n/a","290","n/a","n/a","n/a","n/a","n/a","n/a","n/a", "1.3","(630)","3","[Rn]5f6d7s","n/a","n/a", 14, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Mendelevium"), "Md", "101", "258.1", "n/a", "1100", "n/a","287","n/a","n/a","n/a","n/a","n/a","n/a","n/a", "1.3","(635)","3","[Rn]5f6d7s","n/a","n/a", 15, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Nobelium"), "No", "102", "259.1009", "n/a", "1100", "n/a","285","n/a","n/a","n/a","n/a","n/a","n/a","n/a", "1.3","(640)","3, 2","[Rn]5f6d7s","n/a","n/a", 16, 10, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},

  {N_("Lawrencium"), "Lr", "103", "262.11", "n/a", "n/a", "n/a","282","n/a","n/a","n/a","n/a","n/a","n/a","n/a","n/a","n/a","3","[Rn]5f6d7s","n/a","n/a", 3, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Rutherfordium"), "Rf", "104", "[261]", "n/a", "n/a", "n/a","n/a","n/a","n/a","n/a","n/a","n/a","n/a","n/a","n/a","n/a","-","[Rn]5f6d7s","n/a","n/a", 4, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Dubnium"), "Db", "105", "[262]", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a","n/a","n/a","n/a","n/a","n/a","n/a","-","[Rn]5f6d6s","n/a","n/a", 5, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Seaborgium"), "Sg", "106", "[266]", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a","-","[Rn]5f6d7s","n/a","n/a", 6, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Bohrium"), "Bh", "107", "[264]", "n/a", "n/a", "n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a","n/a","n/a", "n/a","n/a", "n/a","n/a","n/a", 7, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Hassium"), "Hs", "108", "[269]", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a","n/a","n/a","n/a", 8, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Meitnerium"), "Mt", "109", "[268]", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a","n/a", "n/a","n/a","n/a","n/a","n/a", "n/a", "n/a","n/a","n/a", 9, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
  
  {N_("Ununnilium"), "Uun", "110", "[269]", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a", "n/a", "n/a","n/a","n/a", 10, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Unununium"), "Uuu", "111", "[272]", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a", "n/a", "n/a","n/a","n/a", 11, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Ununbium"), "Uub", "112", "[277]", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a", "n/a", "n/a","n/a","n/a", 12, 7, 0x7FFFFFFF, 0xFFFF, 0xFFFF, 0xAAAA},
 
  {N_("Ununtrium"), "Uut", "113", "n/a", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a","n/a","n/a","n/a","n/a","n/a", "n/a","n/a", "n/a","n/a","n/a", 13, 7, 0x7FFFFFFF, 0xC000, 0xC000, 0xC000},
 
  {N_("Ununquadium"), "Uuq", "114", "[289]", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a","n/a","n/a","n/a","n/a", "n/a", "n/a","n/a", "n/a","n/a","n/a", 14, 7, 0x7FFFFFFF, 0xC000, 0xC000, 0xC000},
 
  {N_("Ununpentium"), "Uup", "115", "n/a", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a", "n/a","n/a","n/a", 15, 7, 0x7FFFFFFF, 0xC000, 0xC000, 0xC000},
 
  {N_("Ununhexium"), "Uuh", "116", "n/a", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a", "n/a","n/a","n/a", 16, 7, 0x7FFFFFFF, 0xC000, 0xC000, 0xC000},
  
  {N_("Ununseptium"), "Uus", "117", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a", "n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a", "n/a","n/a","n/a", 17, 7, 0x7FFFFFFF, 0xC000, 0xC000, 0xC000},
  
  {N_("Ununoctium"), "Uuo", "118", "n/a", "n/a", "n/a","n/a","n/a","n/a", "n/a", "n/a", "n/a","n/a","n/a","n/a","n/a", "n/a","n/a", "n/a","n/a","n/a", 18, 7, 0x7FFFFFFF, 0xC000, 0xC000, 0xC000},
  
 {NULL}
};

void dump_table(void)
{
  struct table_entry *p = table;
  int i;
  while(p->info[NAME]) {
    /*      1  2  3  4  5  6  7  8  9  10 11 12 13 14 15 16 17 18 19*/
    printf("%s,%s,%s,%s,%s,%s,%s,%d,%d,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,0x%08X,0x%08X,0x%08X,0x%08X\n ",
      p->info[NAME],                 /*  1 */
      p->info[SYMBOL],               /*  2 */
      p->info[NUMBER],               /*  3 */
      p->info[WEIGHT],               /*  4 */
      p->info[DENSITY],              /*  5 */
      p->info[MELTING],              /*  6 */
      p->info[BOILING],              /*  7 */
      p->info[ATOMIC_RADIUS],        /*  8 */
      p->info[COVALENT_RADIUS],      /*  9 */
      p->info[IONIC_RADIUS],         /* 10 */
      p->info[ATOMIC_VOLUME],        /* 11 */
      p->info[SPECIFIC_HEAT],        /* 12 */
      p->info[FUSION_HEAT],          /* 13 */
      p->info[EVAPORATION_HEAT],     /* 14 */
      p->info[TERMAL_CONDUCTIVITY],  /* 15 */
      p->info[PAULING],              /* 16 */
      p->info[ENERGY],               /* 17 */
      p->info[OXIDATION],            /* 18 */
      p->info[CONFIGURATION],        /* 19 */
      p->x,                          /* 20 */
      p->y,                          /* 21 */
      p->color.pixel,                /* 22 */
      p->color.red,                  /* 23 */
      p->color.green,                /* 24 */
      p->color.blue                  /* 25 */
    );
    ++p;
  }
}

