#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>

#include <curses.h>

#include "mfsk.h"
#include "complex.h"
#include "fft.h"
#include "viterbi.h"
#include "macro.h"
#include "prefs.h"
#include "varicode.h"
#include "interleave.h"

static int grayenctab[16] = {
	0,  1,  3,  2,
	7,  6,  4,  5,
	15, 14, 12, 13,
	8,  9,  11, 10
};

static complex mixer(struct mfsktx *m, complex in)
{
	complex z;
	float f;

	f = m->freq - (float) BaseTone * SampleRate / SymbolLen;
	f += prefs.txoffset;

	z.re = cos(m->phaseacc);
	z.im = sin(m->phaseacc);

	z = cmul(z, in);

	m->phaseacc -= 2.0 * M_PI * f / SampleRate;

	if (m->phaseacc > M_PI)
		m->phaseacc -= 2.0 * M_PI;
	if (m->phaseacc < M_PI)
		m->phaseacc += 2.0 * M_PI;

	return z;
}

static void sendsymbol(struct mfsktx *m, int sym)
{
	complex in[SymbolLen], out[SymbolLen], z;
	float *buf;
	int i;

	if (m->queuelen > TxQueueSize - 1) {
		fprintf(stderr, "txqueuelen=%d\n", m->queuelen);
		return;
	}

	memset(in , 0, sizeof(in));

	sym = BaseTone + grayenctab[sym & 15];
	in[sym].re = 1.0;

	fft(m->fft, in, out);

	buf = m->queue[m->queuelen++];

	for (i = 0; i < SymbolLen; i++) {
		z = mixer(m, out[i]);
		buf[i] = z.re;
	}
}

static void sendbit(struct mfsktx *m, int bit)
{
	int symbol;

	m->bitshreg = (m->bitshreg << 1) | !!bit;
	m->bitstate++;

	if (m->bitstate == 4) {
		symbol = interleave(m->bitshreg);
		sendsymbol(m, symbol);
		m->bitstate = 0;
		m->bitshreg = 0;
	}
}

static void sendchar(struct mfsktx *m, unsigned char c)
{
	char *code;
	int data;

	code = varienc(c);

	while (*code) {
		data = encode27(&m->encstate, (*code - '0'));
		sendbit(m, data & 2);
		sendbit(m, data & 1);
		code++;
	}

	show_tx_char(c);
}

static void sendidle(struct mfsktx *m)
{
	int i, data;

	sendchar(m, 0);	/* <NUL> */

	data = encode27(&m->encstate, 1);
	sendbit(m, data & 2);
	sendbit(m, data & 1);

	/* extended zero bit stream */
	for (i = 0; i < 16; i++) {
		data = encode27(&m->encstate, 0);
		sendbit(m, data & 2);
		sendbit(m, data & 1);
	}
}

static void flushtx(struct mfsktx *m)
{
	int i, data;

	/* send one superfluous bit to flush the varicode decoder */
	data = encode27(&m->encstate, 1);
	sendbit(m, data & 2);
	sendbit(m, data & 1);

	/* generate encoder tail */
	for (i = 0; i < 8; i++) {
		data = encode27(&m->encstate, 0);
		sendbit(m, data & 2);
		sendbit(m, data & 1);
	}

	/* flush the interleaver */
	for (i = 0; i < 4 + 120 + 16; i++)
		sendbit(m, 0);
}

struct mfsktx *inittx(void)
{
	struct mfsktx *m;

	if ((m = calloc(1, sizeof(struct mfsktx))) == NULL)
		return NULL;

	if ((m->fft = init_fft(SymbolLen)) == NULL)
		return NULL;

	init_interleaver();

	return m;
}

void txstop(struct mfsktx *m)
{
	m->stopflag = 1;
}

float *txprocess(struct mfsktx *m)
{
	float *p = NULL;
	int c, i;

	if (m->queueptr < m->queuelen) {
		p = m->queue[m->queueptr++];
		return p;
	}

	m->queuelen = 0;
	m->queueptr = 0;

	if (m->tune)
		m->state = STATE_TUNE;

	switch (m->state) {
	case STATE_TUNE:
		sendsymbol(m, 0);
		m->state = STATE_FINISH;
		break;

	case STATE_PREAMBLE:
		for (i = 0; i < 8; i++)
			sendsymbol(m, 0);
		m->state = STATE_START;
		break;

	case STATE_START:
		sendchar(m, '\r');
		sendchar(m, 2);			/* STX */
		sendchar(m, '\r');
		m->state = STATE_DATA;
		break;

	case STATE_DATA:
		c = typeahead_get_char();
		if (c == -1)
			sendidle(m);
		else
			sendchar(m, c);
		if (m->stopflag)
			m->state = STATE_END;
		break;

	case STATE_END:
		c = typeahead_get_char();
		if (c == -1) {
			sendchar(m, '\r');
			sendchar(m, 4);		/* EOT */
			sendchar(m, '\r');
			m->state = STATE_FLUSH;
		} else
			sendchar(m, c);
		break;

	case STATE_FLUSH:
		flushtx(m);
		m->state = STATE_FINISH;
		break;

	case STATE_FINISH:
		m->state = STATE_PREAMBLE;
		break;
	}

	if (m->queueptr < m->queuelen) {
		p = m->queue[m->queueptr++];
		return p;
	}

	return NULL;
}
