#include "polymakefile.h"
#include "printer.h"

#include <assert.h>
#include <sstream>

PolymakeProperty::PolymakeProperty(const string &name_, const string &value_):
  name(name_),
  value(value_)
{
}


static string readUntil(FILE *f, int c)
{
  stringstream ret;
  int c2;
  c2=fgetc(f);
  while(c2!=c && c2!=EOF)
    {
      ret<<char(c2);
      c2=fgetc(f);
    }
  return ret.str();
}


void PolymakeFile::open(const char *fileName_)
{
  fileName=string(fileName_);

  FILE *f=fopen(fileName.c_str(),"r");
  assert(f);

  int c=fgetc(f);
  while(c!=EOF)
    {
      if(c=='_')
	{
	  readUntil(f,'\n');
	}
      else if(c!='\n')
	{
	  ungetc(c,f);
	  string name=readUntil(f,'\n');
	  
	  fprintf(Stderr,"Reading:\"%s\"\n",name.c_str());
	  stringstream value;
	  while(1)
	    {
	      string l=readUntil(f,'\n');
	      if(l.size()==0)break;
	      value << l <<endl;
	    }
	  properties.push_back(PolymakeProperty(name.c_str(),value.str().c_str()));
	}
      c=fgetc(f);
    }
}


void PolymakeFile::create(const char *fileName_, const char *application_, const char *type_)
{
  fileName=string(fileName_);
  application=string(application_);
  type=string(type_);
}


void PolymakeFile::close()
{
  FILE *f=fopen(fileName.c_str(),"w");	
  assert(f);

  fprintf(f,"_application %s\n",application.c_str());
  fprintf(f,"_version 2.2\n");
  fprintf(f,"_type %s\n",type.c_str());

  for(list<PolymakeProperty>::const_iterator i=properties.begin();i!=properties.end();i++)
    {
      fprintf(f,"\n%s\n",i->name.c_str());
      fprintf(f,"%s",i->value.c_str());      
    }

  fclose(f);
}


void PolymakeFile::writeStream(ostream &file)
{
  file << "_application " << application << endl;
  file << "_version 2.2\n";
  file << "_type " << type << endl;

  for(list<PolymakeProperty>::const_iterator i=properties.begin();i!=properties.end();i++)
    {
      file << endl << i->name.c_str() << endl;
      file << i->value;
    }
}


list<PolymakeProperty>::iterator PolymakeFile::findProperty(const char *p)
{
  string s(p);
  for(list<PolymakeProperty>::iterator i=properties.begin();i!=properties.end();i++)
    {
      if(s==i->name)return i;
    }

  return properties.end();
}


void PolymakeFile::writeProperty(const char *p, const string &data)
{
  if(hasProperty(p))
    {
      assert(0);
    }
  properties.push_back(PolymakeProperty(string(p),data));
}


bool PolymakeFile::hasProperty(const char *p)
{
  return findProperty(p)!=properties.end();
}


int PolymakeFile::readCardinalProperty(const char *p)
{
  assert(hasProperty(p));
  list<PolymakeProperty>::iterator prop=findProperty(p);
  stringstream s(prop->value);

  int ret;
  s>>ret;

  return ret;
}

 
void PolymakeFile::writeCardinalProperty(const char *p, int n)
{
  stringstream t;
  t<<n<<endl;
  writeProperty(p,t.str());
}


bool PolymakeFile::readBooleanProperty(const char *p)
{
  return false;
}


void PolymakeFile::writeBooleanProperty(const char *p, bool n)
{
}


IntegerMatrix PolymakeFile::readMatrixProperty(const char *p, int height, int width)
{
  IntegerMatrix ret(height,width);

  assert(hasProperty(p));
  list<PolymakeProperty>::iterator prop=findProperty(p);
  stringstream s(prop->value);
  for(int i=0;i<height;i++)
    for(int j=0;j<width;j++)
      {
	int v;
	s>>v;
	ret[i][j]=v;
      }

  return ret;
}


void PolymakeFile::writeMatrixProperty(const char *p, const IntegerMatrix &m, bool indexed)
{
  stringstream t;
  
  for(int i=0;i<m.getHeight();i++)
    {
      for(int j=0;j<m.getWidth();j++)
	{
	  if(j!=0)t<<" ";
	  t<<m[i][j];
	}
      if(indexed)t<<"\t# "<<i;
      t<<endl;
    }
  writeProperty(p,t.str());
}


static list<int> readIntList(istream &s)
{
  list<int> ret;
  int c=s.peek();
  while((c>='0') && (c<='9')|| (c==' '))
    {
      fprintf(Stderr,"?\n");
      int r;
      s >> r;
      ret.push_back(r);
      c=s.peek();
    }
  return ret;
}

vector<list<int> > PolymakeFile::readMatrixIncidenceProperty(const char *p)
{
  vector<list<int> > ret;
  assert(hasProperty(p));
  list<PolymakeProperty>::iterator prop=findProperty(p);
  stringstream s(prop->value);
  
  while((s.peek()!=-1)&&(s.peek()!='\n')&&(s.peek()!=0))
    {
      fprintf(Stderr,"!\n");
      int c=s.get();
      fprintf(Stderr,"%i",c);
      assert(c=='{');
      ret.push_back(readIntList(s));
      c=s.get();
      assert(c=='}');
      c=s.get();
      assert(c=='\n');
    }
  return ret;
}


void PolymakeFile::writeIncidenceMatrixProperty(const char *p, const vector<list<int> > &m)
{
  stringstream t;
  for(int i=0;i<m.size();i++)
    {
      t<<'{';
      list<int> temp=m[i];
      temp.sort();
      for(list<int>::const_iterator j=temp.begin();j!=temp.end();j++)
	{
	  if(j!=temp.begin())t<<' ';
	  t<< *j;
	}
      t<<'}'<<endl;
    }
  writeProperty(p,t.str());
}


IntegerVector PolymakeFile::readCardinalVectorProperty(const char *p)
{
  IntegerVector ret;

  return ret;
}


void PolymakeFile::writeCardinalVectorProperty(const char *p, IntegerVector const &v)
{
  stringstream t;
  
  for(int i=0;i<v.size();i++)
    {
      if(i!=0)t<<" ";
      t<<v[i];
    }
  t<<endl;
  writeProperty(p,t.str());
}


void PolymakeFile::writeStringProperty(const char *p, const string &s)
{
  writeProperty(p,s);
}
