/***** BEGIN LICENSE BLOCK *****

    FlashGot - a Firefox extension for external download managers integration
    Copyright (C) 2004-2009 Giorgio Maone - g.maone@informaction.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
                             
***** END LICENSE BLOCK *****/

const ASK_NEVER = [false, false, false];

// *** Base/Windows DMS ********************************************************
function FlashGotDM(name) {
  if (arguments.length > 0) {
    this._init(name);
  }
}

FlashGotDM.init = function() {
  FlashGotDM.dms = [];
  FlashGotDM.dmtests = {};
  FlashGotDM.executables = {};
  FlashGotDM.deleteOnExit = [];
  FlashGotDM.deleteOnUninstall = [];
  FlashGotDM.initDMS();
};

FlashGotDM.cleanup = function(uninstalling) {
  var trash = [].concat(FlashGotDM.deleteOnExit);
  if (uninstalling) trash = trash.concat(FlashGotDM.deleteOnUninstall);
  for each (var f in trash) {
    if (f instanceof CI.nsIFile) {
      try { f.remove(true); } catch(ex) {}
    }
  }
};

FlashGotDM.prototype = {
  _init: function(name) {
    this.name = name;
    const dms = FlashGotDM.dms;
    var pos = dms.length;
    if (name in dms) {
      var other = dms[name];
      for (var j = pos; j-- > 0;) {
        if (dms[j] == other) {
          pos = j;
          break;
        }
      }
    }
    dms[name] = dms[pos] = this;
  }
,

  _cookieManager: null,
  _exeFile: false,
  _supported: null,
  custom: false,
  disabledLink: false,
  disabledSel: false,
  disabledAll: false,
  exeName: "FlashGot.exe",
  askPath: ASK_NEVER,
  cookieSupport: true,
  postSupport: false,
  priority: "",
  autoselect: true,

  _codeName: null,
  get codeName() {
    return this._codeName || (this._codeName = this.name.replace(/\W/g,"_"));
  },
  
  getPref: function(name, def) {
    return fg.getPref("dmsopts." + this.codeName + "." + name, def);
  },
  setPref: function(name, value) {
    fg.setPref("dmsopts." + this.codeName + "." + name, value);
  },
  
  get asciiFilter() {
    return this.getPref("asciiFilter", false);
  },
  
  get shownInContextMenu() {
    return this.getPref("shownInContextMenu", false);
  },
  set shownInContextMenu(b) {
    this.setPref("shownInContextMenu", b);
    return b;
  }
,
  get service() {
    return fg;
  }
,
  get cookieManager() {
    return this._cookieManager ? this._cookieManager : this._cookieManager =
      CC["@mozilla.org/cookiemanager;1"].getService(CI.nsICookieManager);
  }
,
  get exeFile() {
    if (typeof(this._exeFile) == "object") return this._exeFile;
    const exeName = this.exeName;
    if (!exeName) return this._exeFile = null;
    if (typeof(FlashGotDM.executables[exeName]) == "object") {
      return this._exeFile = FlashGotDM.executables[exeName];
    }
    try {
      var exeFile = fg.profDir.clone();
      exeFile.append(exeName);
      this._exeFile = this.checkExePlatform(exeFile);
      if(this._exeFile) {
        FlashGotDM.deleteOnUninstall.push(this._exeFile);
        if (this.createExecutable()) {
          this.log(this._exeFile.path + " created");
        }
      }
    } catch(ex) {
      this._exeFile=null;
      this.log("Can't init " + exeName + ":\n" + ex.message);
    }
    return FlashGotDM.executables[exeName] = this._exeFile;
  }
,
  checkExePlatform: function(exeFile) {
    
    var path = exeFile.path;
    if (/\/.*\.exe/.test(path)) {
      if (!(fg.getPref("useWine", true) || this.name == fg.defaultDM))
        return null;
      
      if(!FlashGotDM.wine) {
        // check for wine
        var wine = CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
        var winePaths = fg.getPref("wine.paths", "/usr/bin/wine:/usr/local/bin/wine:/opt/local/bin/wine:/Applications/Darwine/Wine.bundle/Contents/bin/wine");
        if (!winePaths) return null;
        
        for each(var winePath in winePaths.split(/[;:,]+/)) {
          try {
            wine.initWithPath(winePath);
            if(wine.exists()) {
              FlashGotDM.wine = wine;
              break;
            }
          } catch(e) {}
        }
        if(!FlashGotDM.wine) return null;
        FlashGotDM.wineExecutables = [];
      }
      FlashGotDM.wineExecutables.push(exeFile);
      return exeFile;
    }
    if (FlashGotDMMac.isMac) return null;
    return /\\.*\.sh$/i.test(path) ? null : exeFile;
  }
,
  get supported() {
    if (typeof(this._supported) == "boolean") return this._supported;
    if (this.customSupportCheck) {
      return this._supported = this.customSupportCheck();
    }
    return this.baseSupportCheck();
  },
  
  baseSupportCheck: function() {
    if (!this.exeName) return true;
    if (!this.exeFile) return false;
    
    var dmtest;
    if (typeof(FlashGotDM.dmtests[this.exeName]) != "string") {
      const dmtestFile = fg.tmpDir.clone();
      dmtestFile.append(this.exeName + ".test");
      try {
        if (dmtestFile.exists()) {
          try { dmtestFile.remove(false); } catch(rex) {}
        }
        this.launchSupportTest(dmtestFile);
        this.log(dmtest = IO.readFile(dmtestFile)); 
      } catch(ex) {
        this.log(ex.message);
        dmtest = "";
      }
      FlashGotDM.dmtests[this.exeName] = dmtest;
    } else dmtest = FlashGotDM.dmtests[this.exeName];
    return this._supported = dmtest.indexOf(this.name+"|OK")>-1;
  }
,
  readWinRegString: function(hkroot, hkpath, hk) {
    if (!hk) hk = "";
    var key, ret = null;
    if ("@mozilla.org/windows-registry-key;1" in CC) {  // Firefox 1.5 or newer
      key = CC["@mozilla.org/windows-registry-key;1"].createInstance(CI.nsIWindowsRegKey);
      key.open(key["ROOT_KEY_" + hkroot], hkpath, key.ACCESS_READ);
      ret = key.readStringValue(hk);
      key.close();
    } else {
      hkroot = hkroot.replace(/^([A-Z])*_(A_Z).*$/, "HK$1$2"); // CURRENT_USER -> HKCU
      if ("@mozilla.org/winhooks;1" in CC) {	// SeaMonkey or other older non-toolkit application
        key = CC["@mozilla.org/winhooks;1"].getService(CI.nsIWindowsRegistry);
      } else if ("@mozilla.org/browser/shell-service;1" in CC) {
        key = CC["@mozilla.org/browser/shell-service;1"].getService(CI.nsIWindowsShellService)
          && ("getRegistryEntry" in key);
      }
      if (key) key.getRegistryEntry(key[hkroot], hkpath, hk);
    }
    return ret;
  }
,
  launchSupportTest: function (testFile) {
    this.runNative(["-o", testFile.path], true);
  },
  
  shouldList: function() {
    return this.supported;
  }
,
  log: function(msg) {
    fg.log(msg);
  }
,
  updateProgress: function(links, idx, len) {
    if (!links.progress) return;
    if ((idx % 100) == 0) {
      if (!len) {
        links.progress.update(100);
        return;
      }
      links.progress.update(70 + 29 * idx / len);
    }
  }
,
  isValidLink: null
,
  get quiet() {
    return this.getPref("quiet") ||
      fg.getPref(this.codeName + ".quiet", false);
  },
  quietOp: function(opType) {
    return this.getPref("quiet." + opType, false) ||
      fg.getPref(this.codeName + ".quiet." + opType, false);
  }
,
  createJobHeader: function(links, opType) {
    return links.length + ";" + this.name + ";" +
      (this.quietOp(opType) ? fg.OP_QET : opType)
      + ";" + links.folder + ";\n"
  }
,
  createJobBody: function(links) {
    var jobLines = [];
    var postData = links.postData || "";
 
    for (var j = 0, len = links.length, l; j < len; j++) {
      jobLines.push((l = links[j]).href,
           l.description,
           this.getCookie(l, links),
           postData);
      this.updateProgress(links, j, len);
    }
    return jobLines.join("\n");
  }
,
  createJob: function(links, opType, extras) {
    var job = this.createJobHeader(links, opType) 
      + this.getReferrer(links) + "\n"
      + this.createJobBody(links);
      
    if (typeof(links.document) == "object") {
      job += "\n" + links.document.referrer + "\n" + links.document.cookie + "\n";
    } else {
      job += "\n\n\n";
    }
    if(!extras) job += "\n\n";
    else {
      while(extras.length < 3) extras.push('');
      job += extras.join("\n");
    }
    var cph = this.getPref("cookiePersistence", null);
    if(cph != null) job += cph;
    return job;
  }
,
  download: function(links, opType) {
    try {
      links.folder = links.folder || (links.length > 0 ? this.selectFolder(links, opType) : "");
      this.checkCookieSupport();
      this.performDownload(links, opType);
    } catch(ex) {
      this.log(ex + "\n" + ex.stack);
    } finally {
      this.updateProgress(links, 0); // 100%
    }
  }
,
  // best override point
  performDownload: function(links, opType) {
    this.performJob(this.createJob(links, opType));
  }
,
  getReferrer: function(links) {
    if (links.redirProcessedBy) {
      for (p in links.redirProcessedBy) {
        if (fg.getPref("redir.anonymous." + p, false)) return "";
      }
    }
    if (!fg.getPref("autoReferrer", true))
      return fg.getPref("fakeReferrer", "");
    
    var ret = links.referrer || links.document && links.document.URL || links[0] && links[0].href;
    
    return (ret && /^https?:*/.test(ret)) ? ret : ""; 
  }
,
  checkCookieSupport: function() {
    this.getCookie = this.cookieSupport && !fg.getPref("omitCookies")
    ? this._getCookie
    :function() { return ""; }
    ;
  }
,
  getCookie: function() { return ""; }
,
  _getCookie: function(link, links) {
    if (!this.cookieSupport) return (this.getCookie = function() { return ""; })();
    var host, cookies;
    if ((cookies = links.cookies)) {
      host = link.host;
      return host && cookies[host] || "";
    }
    this.initCookies(links);
    return this._getCookie(link, links);
  },
  
  initCookies: function(links) {
    var host, cookies, j, objCookie;
    const hostCookies = {};
    
    var l, parts;
    for (j = links.length; j-- > 0;) {
      l = links[j];
      parts = l.href.match(/http[s]{0,1}:\/\/([^\/]+\.[^\/]+)/i); // host?
      if (parts) {
        host = parts[1];
        var hpos = host.indexOf("@");
        if (hpos > -1) host = host.substring(hpos + 1);
        hostCookies[l.host = host] = "";
      } else {
        l.host = null;
      }
    }
    
    var cookieHost, cookieTable, tmpCookie;
    const domainCookies={};

    for (var iter = this.cookieManager.enumerator; iter.hasMoreElements();) {
      if ((objCookie = iter.getNext()) instanceof CI.nsICookie) {
        cookieHost = objCookie.host;
        if (cookieHost.charAt(0) == ".") {
          cookieHost = cookieHost.substring(1);
          cookieTable = domainCookies;
          if (typeof(tmpCookie=domainCookies[cookieHost]) != "string") {
            tmpCookie = "";
          }
        } else {
          if (typeof(tmpCookie=hostCookies[cookieHost])!="string") continue;
          cookieTable = hostCookies;
        }
        cookieTable[cookieHost] = tmpCookie.concat(objCookie.name + "=" + objCookie.value + "; ");
      }
    }

    for (cookieHost in hostCookies) {
      var dotPos;
      for (host = cookieHost; (dotPos=host.indexOf('.'))>=0; ) { 
        if ((tmpCookie = domainCookies[host])) {
          hostCookies[cookieHost] += tmpCookie;
        }
        host = host.substring(dotPos+1);
      }
    }
    
    links.cookies = hostCookies;
  },

  // see http://www.cookiecentral.com/faq/#3.5 and http://www.xulplanet.com/references/xpcomref/ifaces/nsICookie.html
  formatNSCookie: function(cookie) {
    return [
      cookie.host,
      cookie.isDomain ? "TRUE" : "FALSE",
      cookie.path,
      cookie.isSecure? "TRUE" : "FALSE",
      cookie.expires || this.cookieExpires,
      cookie.name,
      cookie.value
    ].join("\t");
  },
  cookieExpires: 0, // to be set once in getCookies()
  createCookieFile: function() {
    if (fg.getPref("omitCookies")) return null;
    
    const cookies = [];
    this.cookieExpires = new Date().getTime() + 24 * 3600 * 3650; // ten years for session cookies
    
    for (var cookie, iter = this.cookieManager.enumerator; iter.hasMoreElements();) {
      if ((cookie = iter.getNext()) instanceof CI.nsICookie) {
        cookies.push(this.formatNSCookie(cookie));
      }
    }
    
    const f = fg.tmpDir.clone();
    f.append("cookies");
    f.createUnique(0, 0600);
    IO.writeFile(f, cookies.join("\n"));
    return f.path;
  }
,
  createJobFile: function(job) {
    const jobFile = fg.tmpDir.clone();
    jobFile.append("flashgot.fgt");
    jobFile.createUnique(0, 0700);
    IO.writeFile(jobFile, job);
    return jobFile;
  }
, 
  
  performJob: function(job) {
    const jobFile = this.createJobFile(job);
    this.runNative([jobFile.path], false);
  }
,
  createExecutable: function() {
    const exeFile = this.exeFile;
    if (!exeFile) return false;
    
    var channel;
    
    const ios = CC['@mozilla.org/network/io-service;1'].getService(CI.nsIIOService);
    var bis = CC['@mozilla.org/binaryinputstream;1'].createInstance(CI.nsIBinaryInputStream);
    
    bis.setInputStream((
      channel = ios.newChannel("chrome://flashgot/content/" + this.exeName, null, null)
    ).open());

    const bytesCount = channel.contentLength;
    const templateImage = bis.readBytes(bytesCount);
    bis.close();
 
    if (exeFile.exists()) {
      try {
        bis.setInputStream((
          channel = ios.newChannelFromURI(ios.newFileURI(exeFile))
        ).open());
        if (channel.contentLength == bytesCount) {
          try {
            if (bis.readBytes(bytesCount) == templateImage) {
              return false;
            }
          } finally {
            bis.close();
          }
        }
      } catch(ioex) {
        this.log(ioex);
      }
    }
    
    var bos = null;
    try {
      const fos = CC["@mozilla.org/network/file-output-stream;1"].createInstance(CI.nsIFileOutputStream);
      fos.init(exeFile, 0x02 | 0x08, 0700, 0);
      bos = CC['@mozilla.org/binaryoutputstream;1'].createInstance(CI.nsIBinaryOutputStream);
      bos.setOutputStream(fos);
      bos.writeBytes(templateImage, bytesCount);
      bos.close();
      return true;
    } catch(ioex) {
      this.log("Error writing " + exeFile.path + ": " + ioex);
    } finally {
      if (bos) try { bos.close(); } catch(e) {}
    }
    return false;
  }
,
  runNative: function(args, blocking, exeFile) {
    try {
      if (typeof(exeFile) == "object"
        || (exeFile = this.exeFile).exists()
        || this.createExecutable()) {
        const proc = CC['@mozilla.org/process/util;1'].createInstance(
          CI.nsIProcess);
        if (FlashGotDM.wine && FlashGotDM.wineExecutables.indexOf(exeFile) > -1) {
          args.unshift(exeFile.path);
          exeFile = FlashGotDM.wine;
        }
        proc.init(exeFile);
        this.log("Running " + exeFile.path + " " + args.join(" ") + " -- " +(blocking ? "blocking" : "async") );
        proc.run(blocking, args, args.length, {});
        if (blocking && proc.exitValue != 0) {
          this.log("Warning: native invocation of\n"
            + exeFile.path
            + "\nwith arguments <"
            + args.join(" ")
            + ">\nreturned " + proc.exitValue);
        }
        return proc.exitValue;
      } else {
        this.log("Bad executable " + exeFile);
      }
    } catch(err) {
      this.log("Error running native executable:\n" + exeFile.path + 
        " " + args.join(" ") + "\n" + err.message);
    }
    return 0xffffffff;
  }
,
  getWindow: function() {
    return fg.getWindow();
  }
,
  selectFolder: function(links, opType) { 
    if (this.quiet || this.quietOp(opType)) return "";

    const autoPref_FF = "browser.download.useDownloadDir";
    const autoPref_Moz = "browser.download.autoDownload";
    
    var initialDir = null;
    var downloadDir = null;
    links.quickDownload = false;
    
    const pref = CC["@mozilla.org/preferences-service;1"].getService(CI.nsIPrefBranch);
    
    function findDownloadDir(prefName) {
      try {
        downloadDir = initialDir = pref.getComplexValue(prefName, CI.nsILocalFile);
        return prefName;
      } catch(ex) {
        return "";
      }
    }
    const isMulti = opType != fg.OP_ONE;
    const multiDirPref = "flashgot.multiDir";
    var downloadDirPref = 
                    (isMulti && findDownloadDir(multiDirPref)) ||
                    findDownloadDir("browser.download.dir") ||
                    findDownloadDir("browser.download.downloadDir") || 
                    findDownloadDir("browser.download.defaultFolder") ||
                    "browser.download.dir"; 
    
    if (isMulti) downloadDirPref = multiDirPref;
    
    try {
      links.quickDownload = pref.getBoolPref(autoPref_FF);
    } catch(noFFEx) {
      try {
        links.quickDownload = pref.getBoolPref(autoPref_Moz);
      } catch(noMozEx) {}
    }
   
    if (!this.askPath[opType]) return "";
    
    if (((!isMulti) || fg.getPref("multiQuiet", false)) && 
        downloadDir && downloadDir.exists() && downloadDir.isDirectory()  && 
        links.quickDownload) {
      return downloadDir.path;
    }
    
    var params = {
      title: "FlashGot (" + this.name.replace(/[\(\)]/g, "") + ")",
      initialDir: initialDir,
      choosenDir: null
    }
    
    this.getWindow().openDialog(
              "chrome://flashgot/content/chooser.xul",
              "flashgotChooser",
              "chrome, dialog, modal, dependent, centerscreen",
              params);
    
    if (params.choosenDir) {
      pref.setComplexValue(downloadDirPref, CI.nsILocalFile, params.choosenDir);
      var path = new String(params.choosenDir.path);
      path._fgSelected = true;
      return path;
    }
  
    throw new Error("Download cancelled by user");

  },
  sanitizeWinArg: function(a) {
    return a.replace(/([\|\(\) &\^])/g, "^$1"); 
  },
  supportURLList: function(links, argsTemplate) {
    if (/\[[^\]]*UFILE[^\]]*\]/.test(argsTemplate) && links.length) {
      // we must create a file list
      var sep = fg.isWindows ? "\r\n" : "\n";
      var urlList = "";
      for (j = 0; j < links.length; j++) {
        urlList += links[j].href + sep;
      }
      links.length = 1;
      return this.createJobFile(urlList).path
    }
    return null;
  },
  nativeUI: null,
  hideNativeUI: function(document) {
    if (!(this.nativeUI && this.getPref("hideNativeUI", true))) return;
    fg.hideNativeUI(document, this.nativeUI);
  }
}




// *** Unix-like DMS ***********************************************************
function FlashGotDMX(name,cmd,argsTemplate) {
  if (arguments.length != 0) {
    this._init(name);
    const cmds = FlashGotDMX.prototype.unixCmds;
    cmds[name] = cmd;
    this.unixCmd = cmd;
    if (argsTemplate) this.argsTemplate = argsTemplate;
    this.cookieSupport =  /\[.*?(?:CFILE|COOKIE).*?\]/.test(this.argsTemplate);
  }
  if (FlashGotDMMac.isMac) {
    this.createJobFile = FlashGotDMMac.prototype.createJobFile;
  }
}
FlashGotDMX.prototype = new FlashGotDM();
FlashGotDMX.constructor = FlashGotDMX;
FlashGotDMX.prototype.exeName = "flashgot.sh";
FlashGotDMX.prototype.askPath = [true, true, true];
FlashGotDMX.prototype.unixCmds = {};
FlashGotDMX.prototype.__defineGetter__("unixShell", function() {
  var f = CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
  try {
    f.initWithPath("/bin/sh");
    if (!f.exists()) {
      this.log(f.path + " not found");
      f = null;
    }
  } catch(ex) {
    f = null;
    this.log("No *X shell: " + ex.message);
  }
  delete FlashGotDMX.prototype.unixShell;
  return FlashGotDMX.prototype.unixShell = f;
});

FlashGotDMX.prototype.argsTemplate = "[URL]";
FlashGotDMX.prototype.launchSupportTest = function(testFile) {
  const cmds = this.unixCmds;
  var script="(\n";

  for (var name in cmds) {
    cmd = cmds[name];
    script +=" [ -x \"`which '" + cmd + "'`\" ] && echo '"
      + name + "|OK' || echo '" + name+"|KO'\n"; 
  }
  script += ") > '" + testFile.path + "'\n"; 
  this.performJob(script, true);
};

FlashGotDMX.prototype.createCmdLine = function(parms) {
  return this.unixCmd + " " +
    this.argsTemplate.replace(/\[(.*?)(URL|FNAME|REFERER|COOKIE|FOLDER|POST|UFILE|CFILE)(.*?)\]/g,
      function(all, before, parm, after) {
          v = parms[parm]; 
          return typeof(v) != "undefined" && v != null
            ? before + v + after
            : "";
      }
   ) +" &\n";
};
FlashGotDMX.prototype.shellEsc = function(s) {
  return s ? s.replace(/([\\\*\?\[\]\$&<>\|\(\)\{\};"'`])/g,"\\$1").replace(/\s/g,"\\ ") : null;
};
FlashGotDMX.prototype.createJob = function(links, opType) {
  const shellEsc = this.shellEsc;
  // basic implementation

  const folder = shellEsc(links.folder);
  const referrer = shellEsc(this.getReferrer(links));
  const postData = shellEsc(links.postData);
  var job = "";
  var l, url;
  
  var cookieFile = this.createCookieFile();
  var urlListFile = this.supportURLList(links, this.argsTemplate);
  for (var j = 0, len = links.length; j < len; j++) {
    l = links[j];
    url = l.href;
    job += this.createCmdLine({
      URL: shellEsc(url),
      FNAME: l.fname && shellEsc(l.fname) || null,
      REFERER: referrer, 
      COOKIE: shellEsc(this.getCookie(l, links)),
      CFILE: cookieFile,
      FOLDER: folder, 
      POST: postData,
      UFILE: shellEsc(urlListFile)
    });
    this.updateProgress(links, j, len);
  }
  return job;
};
FlashGotDMX.prototype.performJob = function(job, blocking) {
  const jobFile = this.createJobFile("#!" + this.unixShell.path + "\n" + job);
  jobFile.permissions = 0700;
  var exeFile = FlashGotDMMac.isMac ? FlashGotDMMac.exeFile : jobFile;
  this.runNative([], blocking, exeFile);
};
FlashGotDMX.prototype.checkExePlatform = function(exeFile) {
  return this.unixShell && exeFile;
};
FlashGotDMX.prototype.createExecutable = function() {
  return false;
};



// *** Mac OS X DMS ************************************************************
function FlashGotDMMac(name, creatorId, macAppName) {
  if (arguments.length != 0) {
    this._initMac(name, creatorId, macAppName);
  }
}
FlashGotDMMac.exeFile = null;
FlashGotDMMac.appleScriptFile = null;
FlashGotDMMac.appleScriptName = "flashgot-mac-script";
FlashGotDMMac.OSASCRIPT = "/usr/bin/osascript";
FlashGotDMMac.isMac = (function() {
  const f = CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
  try {
    f.initWithPath(FlashGotDMMac.OSASCRIPT);
    return f.exists();
  } catch(ex) {
  }
  return false;
})();
FlashGotDMMac.prototype = new FlashGotDM();
FlashGotDMMac.constructor = FlashGotDMMac;
FlashGotDMMac.prototype.exeName = "FlashGot";
FlashGotDMMac.prototype.cookieSupport = false;
FlashGotDMMac.prototype.macCreators = [];
FlashGotDMMac.prototype._initMac = function(name, creatorId, macAppName) {
  this._init(name);
 
  if (creatorId) {
    const creators=FlashGotDMMac.prototype.macCreators;
    creators[creators.length] = {name: name, id: creatorId};
  }
  this.macAppName = macAppName ? macAppName : name;
  this.initAppleScriptBridge();
  FlashGotDMMac.exeFile = this.exeFile;
};

FlashGotDMMac.prototype.initAppleScriptBridge = function() {
  if (FlashGotDMMac.appletScriptFile) return;
  
  var home = fg.home && fg.home.parent;
  if (home) home.append("mac-flashgot");

  if (!(home && home.exists() && home.isWritable())) {
    (FlashGotDMMac.appleScriptFile = fg.tmpDir.clone())
      .append(FlashGotDMMac.appleScriptName);
    return;
  }
  (FlashGotDMMac.appleScriptFile = home.clone())
    .append(FlashGotDMMac.appleScriptName);
  (FlashGotDMMac.prototype._exeFile = home.clone()).append(this.exeName);
  
  this.log("Setting executable permissions on " + this._exeFile.path);
  this._exeFile.permissions = 0700;
  FlashGotDMMac.prototype.createExecutable = function() { return false; }
}
FlashGotDMMac.prototype.shellEsc = function(s) {
  return s ? "'" + s.replace(/'/g, '"\'"') + "'" : null; 
}
FlashGotDMMac.prototype.createScriptLauncher = function() {
  return "#!/bin/sh\n" +
    "SCRIPT=" + this.shellEsc(FlashGotDMMac.appleScriptFile.path) + "\n" +
    "USCRIPT=\"$SCRIPT.$$\"\n" + 
    "mv \"$SCRIPT\" \"$USCRIPT\" || exit 1\n" +
    "head -n 1 \"$USCRIPT\" | grep '#!' >/dev/null &&  \"$USCRIPT\" || " +
    FlashGotDMMac.OSASCRIPT + " \"$USCRIPT\"";
};
FlashGotDMMac.prototype.checkExePlatform = function(exeFile) {
  return FlashGotDMMac.isMac && exeFile || null;
};
FlashGotDMMac.prototype.createExecutable = function() {
  
  
  var exeFile = this._exeFile;
  if (!exeFile) return false;

  try {
   var scriptLauncher = this.createScriptLauncher();
   var mustCreate = true;
   if (exeFile.exists()) {
     if (IO.readFile(exeFile) == scriptLauncher) {
       exists = true;
       if (exeFile.isExecutable()) return false;
       mustCreate = false;
     } else {
       this.log(exeFile.path + " is corrupted or obsolete, replacing it...");
       try { exeFile.remove(true); } catch(rex) {} 
     }
   } else {
     this.log(exeFile.path + " not found, creating it...");
   }
   if (mustCreate) {
      this.log("Creating Mac executable");
      exeFile.create(0, 0700);
      IO.writeFile(exeFile, scriptLauncher);
     
      try {
        this.log("Trying to reset Leopard's quarantine attribute...");
        var xattr =  CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
        xattr.initWithPath("/usr/bin/xattr");
        this.runNative(["-d", "com.apple.quarantine", exeFile.path], true, xattr);
      } catch(e) {
        this.log("Couldn't clear quarantine attribute " + e); 
      }
   }
   this.log("Setting executable permissions on " + exeFile.path);
   exeFile.permissions = 0700;
   
  
   
   
   return mustCreate;
  } catch(ex) {
    this.log("Cannot create Mac executable: " + ex.message);
  }
  return false;
};
FlashGotDMMac.prototype.launchSupportTest = function(testFile) {
  const creators = FlashGotDMMac.prototype.macCreators;
  
  var s = [
    'global gRes',
    'set gRes to ""',
    'on theTest(theName, theId)',
    '  set gRes to gRes & theName',
    '  try',
    '    tell app "Finder" to get application file id theId',
    '    set gRes to gRes & "|OK\n"',
    '  on error',
    '    set gRes to gRes & "|KO\n"',
    '  end try',
    'end theTest'
  ];
  for (var j = creators.length; j-- > 0; ) {
    s.push('theTest("' + creators[j].name + '","' +creators[j].id + '")'); 
  }
  s.push(
    'set theFile to POSIX file "' + testFile.path + '"',
    'try',
    '  set fh to open for access theFile with write permission',
    '  write (gRes) to theFile',
    '  close access fh',
    'on error',
    '  try',
    '    close access fh',
    '  end try',
    'end try'
  );
  this.performJob(s.join("\n"), true);
};
FlashGotDMMac.prototype.createJobFile = function(job) {
  const jobFile = FlashGotDMMac.appleScriptFile;
  try {
    jobFile.remove(true);
  } catch(ex) {}
  try {
    jobFile.create(0, 0600);
    IO.writeFile(jobFile, job, /^#/.test(job) ? null : fg.getPref("appleScriptEncoding"));
    return jobFile;
  } catch(ex) {
    this.log("Cannot write " + (jobFile && jobFile.path) + ex.message);
  }
  return null;
}
FlashGotDMMac.prototype.performJob = function(job, blocking) {
  if (this.createJobFile(job)) 
    this.runNative([], blocking, this.exeFile);
};

FlashGotDMMac.prototype.createJob = function(links,opType) {
  const referrer = this.getReferrer(links);
  var job = "tell application \""+ this.macAppName+ "\"\n";
  for (var j = 0, len = links.length; j < len; j++) {
    job += 'GetURL "' + links[j].href + '" from "' + referrer  + "\"\n";
    this.updateProgress(links, j, len);
  }
  job += "end tell\n";
  return job;
};



// *** Custom DMS **************************************************************
function FlashGotDMCust(name) {
  if (arguments.length == 0 || (!name) || (!name.length)) return;
  name = name.replace(/,/g, " ");
  this._init(name);
  this.prefsBase = "custom." + this.codeName + ".";
}

FlashGotDMCust.init = function() {
  const names = fg.getPref("custom", "").split(/\s*,\s*/);
  for (var j = names.length; j-->0;) {
    new FlashGotDMCust(names[j]);
  }
}

FlashGotDMCust.persist = function() {
  const dms = FlashGotDM.dms;
  const cdms = [];
  for (var j = dms.length; j-->0;) {
    if (dms[j].custom) cdms.push(dms[j].name);
  }
  fg.setPref("custom", cdms.join(","));
}

FlashGotDMCust.prototype = new FlashGotDM();
FlashGotDMCust.constructor = FlashGotDM;

delete FlashGotDMCust.prototype.launchSupportTest;
delete FlashGotDMCust.prototype.exeFile;
FlashGotDMCust.prototype.PLACEHOLDERS = ["URL", "COMMENT", "REFERER", "COOKIE", "FOLDER", "POST", "UFILE", "CFILE"];

FlashGotDMCust.prototype.custom = true;
FlashGotDMCust.prototype. _supported = true;
FlashGotDMCust.prototype.cookieSupport = false;
FlashGotDMCust.prototype.postSupport = true;
FlashGotDMCust.prototype.askPath = [true, true, true];

FlashGotDMCust.prototype.__defineGetter__("exeFile",function() {
  try {
    return fg.prefs.getComplexValue(this.prefsBase + "exe", 
      CI.nsILocalFile);
  } catch(ex) {
    return null;
  }
});
FlashGotDMCust.prototype.__defineSetter__("exeFile",function(v) {
  try {
    if (v) {
      fg.prefs.setComplexValue(this.prefsBase + "exe", 
          CI.nsILocalFile,v);
      return v;
    }
  } catch(ex) {
  }
  return null;
});

FlashGotDMCust.prototype.__defineGetter__("argsTemplate", function() {
  if (this.forcedTemplate) return this.forcedTemplate;
  var t = fg.getPref(this.prefsBase+"args", "[URL]");
  return /['"`]/.test(t) ? this.argsTemplate = t : t;
});
FlashGotDMCust.prototype.__defineSetter__("argsTemplate",function(v) {
  if (!v) {
    v = "";
  } else {
    v = v.replace(/['"`]/g,"");
  }
  fg.setPref(this.prefsBase + "args", v);
  this.askPath = [];
  return v;
});


FlashGotDMCust.prototype.download = function(links, opType) {
  const t = this.argsTemplate;
  this.cookieSupport = /\[.*?(?:CFILE|COOKIE).*?\]/.test(t);
  this.askPath[opType] = /\[.*?FOLDER.*?\]/.test(t);
  var exeFile = this.exeFile;
  // portable hacks
  if (exeFile && !exeFile.exists()) {
    // try changing the first part of path
    var path = exeFile.path;
    var profPath = fg.profDir.path;
    var pos1, pos2;
    if (path[1] == ":" && profPath[1] == ":") { 
      // easy, it's Windows, swap drive letter
      path = profPath[0] + path.substring(1);
    } else if(path.indexOf("/mount/") == 0 && profPath.indexOf("/mount/") == 0) {
      pos1 = path.indexOf("/", 7);
      pos2 = profPath.indexOf("/", 7);
      path = "/mount/" + profPath.substring(7, pos2) + path.substring(pos1); 
    } else if((pos1 = path.indexOf("/",1)) > 0 && (pos2 = profPath.indexOf("/", 1)) > 0) {
      path = profPath.substring(0, pos2) + path.substring(pos1);
    } else exeFile = null;
    if (exeFile) {
      exeFile = exeFile.clone().QueryInterface(CI.nsILocalFile).initWithPath(path);
      if (!exeFile.exists()) exeFile = null;
    }
  }
  links.exeFile= (exeFile || 
    (exeFile = this.exeFile = this.locateExeFile())) ? exeFile : null;
  FlashGotDM.prototype.download.call(this, links, opType);
};

FlashGotDMCust.prototype.locateExeFile = function(name) {


  if (!name) name = this.name;
  var title = fg.getString("custom.exeFile");
  title = 'FlashGot (' + name + ') - ' + title;
  
  const fp = CC["@mozilla.org/filepicker;1"].createInstance(CI.nsIFilePicker);
  const win = this.getWindow();
  fp.init(win, title, CI.nsIFilePicker.modeOpen);
  fp.appendFilters(CI.nsIFilePicker.filterApps);
  fp.appendFilters(CI.nsIFilePicker.filterAll);

  if (fp.show() == CI.nsIFilePicker.returnOK) {
    var file = fp.file.QueryInterface(CI.nsILocalFile);
    if (file.exists()) {
      return file;
    }
  }
  return null;
};

FlashGotDMCust.prototype._addParts=function(a, s) {
  var parts=s.split(/\s+/);
  var k, p;
  for (k in parts) {
    if ((p = parts[k])) {
      a[a.length] = p;
    }
  }
};

FlashGotDMCust.prototype.makeArgs = function(parms) {
  const args = [];
  var t = this.argsTemplate;
  var j, v, len, s;
  
  var idx;
  
  for (var m; 
      m = t.match( /\[([\s\S]*?)(\S*)\b(URL|COMMENT|FNAME|REFERER|COOKIE|FOLDER|POST|CFILE|UFILE)\b(\S*?)([\s\S]*?)\]/); 
      t = t.substring(idx + m[0].length) 
     ) {

    if ((idx = m.index) > 0) {
      this._addParts(args, t.substring(0, idx));
    }
    
    v = parms[m[3]];
    if (!v) continue;
    
    this._addParts(args, m[1]);
    args[args.length] = m[2] + v + m[4];
    this._addParts(args, m[5]);
  }
  
  if (t.length) {
    this._addParts(args, t);
  }
  return args;
};

FlashGotDMCust.prototype.createJob = function(links, opType) {
  return { links: links, opType: opType };
};

FlashGotDMCust.prototype.shellEsc = function(s) {
  return s ? '"' + s.replace(/"/g, '""') + '"' : null;
}

FlashGotDMCust.prototype.winEscHack = function(s) {
  // hack for bug at http://mxr.mozilla.org/seamonkey/source/xpcom/threads/nsProcessCommon.cpp#149 
  return (/[;&=]/.test(s) && !/\s/.test(s)) // "=" and ";" are command line separators on win!!!
    ? s + " " : s; // we add a space to force escaping
}

FlashGotDMCust.prototype.performJob = function(job) {
  const links = job.links;
  const exeFile = links.exeFile;
  if (links.length < 1 || !exeFile) return;
  
  var esc = (fg.isWindows && this.getPref("winEscHack", true))
    ? this.winEscHack : function(s) { return s; }
  
  const folder = links.folder;
  const referrer = esc(this.getReferrer(links));
  const postData = esc(links.postData);
  var cookieFile = this.createCookieFile();
  var urlListFile = this.supportURLList(links, this.argsTemplate);
  var maxLinks = fg.getPref(this.prefsBase + "maxLinks", 0);
  if (maxLinks > 0 && links.length > maxLinks) {
    this.log("Too many links (" + links.length + "), cutting to " 
        + this.prefsBase + "maxLinks (" + maxLinks + ")");
    links.length = maxLinks;
  }
  var l;
 
  
  for (var j = 0, len = links.length; j < len; j++) {
    l = links[j];
    this.runNative(
      this.makeArgs({
        URL: esc(l.href),
        COMMENT: esc(l.description),
        FNAME: l.fname && esc(l.fname) || null,
        REFERER: referrer, 
        COOKIE: esc(this.getCookie(l, links)), 
        FOLDER: folder, 
        POST: postData,
        CFILE: cookieFile,
        UFILE: urlListFile
       }),
       false, exeFile);
    this.updateProgress(links, j, len);
  }
};
FlashGotDMCust.prototype.checkExePlatform = function(exeFile) {
  return exeFile;
};
FlashGotDMCust.prototype.createExecutable = function() {
  return false;
};
// End FlashGotDMCust.prototype

// *****************************************************************************
// END DMS CLASSES
// *****************************************************************************

// DMS initialization

FlashGotDM.initDMS = function() {
  const isWin = fg.isWindows;
  var dm;

  new FlashGotDM("BitComet");

  dm = new FlashGotDM("Download Accelerator Plus");
  dm.nativeUI = "#dapctxmenu1, #dapctxmenu2";
  
  dm.performDownload =  function(links, opType) {
    if (!("IDAPComponent" in CI)) {
      this.log("DAP extension not found, falling back to IE integration");
      FlashGotDM.prototype.performDownload.apply(this, arguments);
      return;
    }
    
    function getDAP() {
      return CC["@speedbit.com/dapfirefox/dapcomponent;1"].createInstance(CI.IDAPComponent);
    }
    function downloadDoc(doc) {
      getDAP().RigthClickMenuDownloadAll(doc, "Firefox");
      doc.defaultView.location.href = "about:blank";
    }
    
    try {
    
      if (opType == fg.OP_ONE) {
        var l = links[0];
        getDAP().RigthClickMenuDownload(
          l.href, this.getReferrer(links), this.getCookie(l, links), l.description, "Firefox");
        return;
      }
      
      if (!links.document) {
        this.log("DAP: No document found");
        return;
      }
      
      var doc = links.document;
      var url = doc.URL;
      if (opType == fg.OP_SEL) {
        var bwin = links.browserWindow || fg.getBrowserWindow(doc);
        if (!bwin) {
          this.log("DAP: no browser window found");
          return;
        }
        var f = bwin.document.getElementById("_flashgot_iframe");
        
        f.docShell.QueryInterface(CI.nsIWebPageDescriptor).loadPage(
          DOM.getDocShellFromWindow(doc.defaultView).QueryInterface(CI.nsIWebPageDescriptor).currentDescriptor,
          2
        );
        
        var dm = this;
        f.addEventListener("DOMContentLoaded", function(ev) {
          try {
            var f = ev.currentTarget;
            f.removeEventListener(ev.type, arguments.callee, false);
            var doc = f.contentDocument;
            if (doc.URL != url) return;
            var root = doc.documentElement;
            while(root.firstChild) root.removeChild(root.firstChild);
            root.appendChild(doc.createElement("head"));
            root.appendChild(doc.createElement("body"));
            var frag = doc.createDocumentFragment();
            var a, l;
            for(var j = 0; j < links.length; j++) {
              a = doc.createElement("a");
              l = links[j];
              a.href = l.href;
              a.appendChild(doc.createTextNode(l.description));
              frag.appendChild(a);
            }
            doc.body.appendChild(frag);
            downloadDoc(doc, "Firefox");
          } catch(e) {
            dm.log("DAP Selection: " + e.message + " - " + e.stack);
          }
        }, false);
        
      } else {
        downloadDoc(doc);
      }
    } catch(e) {
      this.log("DAP: " + e.message + " - " + e.stack);
      return;
    }
  };
  
  
  
  
  new FlashGotDM("Download Master");
  
  for each (dm in [new FlashGotDM("DTA"), new FlashGotDM("DTA (Turbo)")]) {
    dm.__defineGetter__("supported", 
      function() { 
        return  "dtaIFilterManager" in CI || "@downthemall.net/privacycontrol;1" in CC 
    });
    dm.turboDTA = /Turbo/.test(dm.name);
    dm.nativeUI = dm.turboDTA
      ? "#context-dta-savelinkt, #context-tdta, #dtaCtxTDTA, #dtaCtxSaveT, #dtaCtxTDTA-direct, #dtaCtxSaveT-direct"
      : "#context-dta-savelink, #context-dta, #dtaCtxDTA, #dtaCtxSave, #dtaCtxDTA-direct, #dtaCtxSave-direct";
      
    dm.performDownload = function(links, opType) {
      if(!links.document) {
        this.log("No document found in " + links);
        return;
      }
      var w = links.browserWindow || fg.getBrowserWindow(links.document);
      if(!(w && w.DTA_AddingFunctions && w.DTA_AddingFunctions.saveLinkArray)) {
        this.log("DTA Support problem: " + w + ", " + (w && w.DTA_AddingFunctions) + ", tb:" +
          w.gBrowser + ", wl:" + w.location + ", wo:" + w.wrappedJSObject + ", " +
            (w && w.DTA_AddingFunctions && w.DTA_AddingFunctions.saveLinkArray));
        return;
      }
      var mlSupport = w.DTA_getLinkPrintMetalink;
      var turbo = this.turboDTA;
      var cs = links.document && links.document.characterSet || "UTF-8";
      var anchors = [], images = [], l, arr;
      var hash, ml;
      var referrer = this.getReferrer(links);
      var tag;
      var single = opType == fg.OP_ONE;
      var ios = CC["@mozilla.org/network/io-service;1"].getService(CI.nsIIOService);
      var wrapURL = w.DTA_URL
        ? w.DTA_URL.toSource().indexOf("DTA_URL(url, preference)") > -1
          ? function(url, cs) { return new w.DTA_URL(ios.newURI(url, cs, null)); }
          : function(url, cs) { return new w.DTA_URL(url, cs); }
        : function(url) { return url };
      var lastItem = null;
      for (var j = 0, len = links.length; j < len; j++) {
        l = links[j];
        arr = single || !(tag = l.tagName) || tag.toLowerCase() == "a" ? anchors : images;
        try {
          arr.push(lastItem = { 
              url: wrapURL(l.href, cs),
              description: l.description,
              ultDescription: '',
              referrer: referrer
          });
          
          if (arr == anchors && mlSupport && l.href.indexOf("#") > 0) {
            hash = l.href.match(/.*#(.*)/)[1];
            ml = mlSupport(l.href);
            if (ml) {
              arr.push(lastItem = {
                url: wrapURL(ml, cs),
                description: '[metalink] http://www.metalinker.org/',
                ultDescription: '',
                referrer: referrer,
                metalink: true
              });
            }
          } 
        } catch(e) {
          this.log("DTA: " + e.message);
        }
        this.updateProgress(links, j, len);
      }
      if (!lastItem) {
        this.log("DTA: no link found in " + links.length);
      } else if (single && w.DTA_AddingFunctions.saveSingleLink) {
        w.DTA_AddingFunctions.saveSingleLink(turbo, lastItem.url, lastItem.referrer, lastItem.description, links.postData);
      } else {
        w.DTA_AddingFunctions.saveLinkArray(turbo, anchors, images);
      }
    }
  }
  
  new FlashGotDM("FlashGet");
  
  dm = new FlashGotDM("FlashGet 2");
  dm = new FlashGotDM("FlashGet 2.x");
  dm.nativeUI = "#flashgetSingle, #flashgetAll, #flashgetSep";
  
  dm = new FlashGotDM("Free Download Manager");
  
  new FlashGotDM("FreshDownload");
  
  
  dm = new FlashGotDM("GetRight");
  dm.metalinkSupport = true;
  dm.download=function(links, opType) {
    if (opType == fg.OP_ONE && !fg.getPref("GetRight.quick")) {
      opType = fg.OP_SEL;
    }
    FlashGotDM.prototype.download.call(this, links, opType);
  };
  
  dm.createJob = function(links, opType) {
    var folder = links.folder;
    if (!(folder && folder._fgSelected)) folder = false;
    
    var referrer = this.getReferrer(links);
    
    switch (opType) {
      case fg.OP_ONE:
        var job = FlashGotDM.prototype.createJob.call(this, links, opType,
          fg.getPref("GetRight.old") ? ["old"] : null
          ).replace(/; /g, ";");
        return job;
      case fg.OP_SEL:
      case fg.OP_ALL:
        var urlList = "";
        var referrerLine = (referrer && referrer.length > 0) ? "\r\nReferer: " + referrer + "\r\n" : "\r\n";
        var replacer = fg.getPref("GetRight.replaceSpecialChars", true) ? /[^\w\.-]/g : /[\x00-\x1f\\]+/g;
        var l, k, len, decodedURL, urlParts, fileSpec, cookie;
        
        for (var j = 0; j < links.length; j++) {
          l=links[j];
          
          if (l.fname) fileSpec = l.fname;
          else if (folder) {
            fileSpec = '';
            decodedURL = unescape(l.href);
            urlParts = decodedURL.match(/\/\/.+[=\/]([^\/]+\.\w+)/);
            if (!urlParts) urlParts=l.href.match(/.*\/(.*\w+.*)/);
            if (urlParts && (fileSpec = urlParts[1])
              // && (links.length==1 ||  !/\.(php|[\w]?htm[l]?|asp|jsp|do|xml|rdf|\d+)$/i.test(fileSpec))
             ) {  
              urlParts = fileSpec.match(/(.*\.\w+).*/);
              if (urlParts) fileSpec = urlParts[1];
              fileSpec = fileSpec.replace(replacer, '_');
            } else continue;
          } else fileSpec = '';
          
          if (fileSpec) {
            if (folder) fileSpec = folder + "\\" + fileSpec;
            fileSpec = "File: " + fileSpec + "\r\n";
          }
          
          urlList+="URL: "+l.href
            +"\r\nDesc: "+l.description + "\r\n" + fileSpec;
          
            if (l.md5) {
            urlList += "MD5: " + l.md5 + "\r\n";
          }
          if (l.sha1) {
            urlList += "SHA1: " + l.sha1+ "\r\n";
          }
          if (l.metalinks) {
            for (k = 0, len = Math.min(16, l.metalinks.length); k < len; k++) {
              urlList += "Alt: " + l.metalinks[k] + "\r\n";
            }
          } else {
            urlList += referrerLine;
            if ((cookie = this.getCookie(l, links))) {
              urlList += "Cookie: " + cookie + "\r\n";
            }
          }
          this.updateProgress(links, j, len);
        }
        var file = fg.tmpDir.clone();
        file.append("flashgot.grx");
        file.createUnique(0,0600);
        var charset=null;
        try {
          charset=fg.getPref("GetRight.charset",
            fg.prefService.QueryInterface(CI.nsIPrefBranch
            ).getComplexValue("intl.charset.default",
              CI.nsIPrefLocalizedString).data);
        } catch(ex) {}
        IO.writeFile(file, urlList, charset);
        referrer = file.path;
        break;
    }
    var cmdOpts="/Q";
    if (fg.getPref("GetRight.autostart",false)) { // CHECK ME!!!
      cmdOpts+="\n /AUTO";
    }
    return this.createJobHeader({ length: 0, folder: "" }, opType) +
      referrer + "\n" + cmdOpts;
  };
  dm.askPath=[false,true,true];
  
  new FlashGotDM("GigaGet");
  
  new FlashGotDM("HiDownload");
  new FlashGotDM("InstantGet");
  
  dm = new FlashGotDM("iGetter Win");
  dm.nativeUI = "#all-igetter, #igetter-link";
  dm.__defineGetter__("supported", function() {
    if (typeof(this._supported) == "boolean") return this._supported;
    if (FlashGotDMMac.isMac) return this._supported = false;
    
    this._supported = ("nsIGetterMoz" in CI);
    this.cookieSupport = false;
    if (this._supported) return true;
    this.cookieSupport = true;
    return this._supported = !!this.createExecutable();
  });
  dm.createExecutable = function() {
    var exeFile, path, key;
    
    exeFile = CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
    try {
      path = this.readWinRegString("CURRENT_USER", "Software\\iGetter")
      exeFile.initWithPath(path);
    } catch(e) {
      path = null;
    }
    if (!(path && exeFile.exists())) {
      try {
        exeFile = CC["@mozilla.org/file/directory_service;1"].getService(CI.nsIProperties)
                    .get("ProgF", CI.nsIFile);
        exeFile.append("iGetter");
        exeFile.append("iGetter.exe");
      } catch(e) {
        path = "C:\\Program Files\\iGetter\\iGetter.exe";
        try {
          exeFile.initWithPath(path);
        } catch(e2) {
          return null;
        }
      }
    }
    
    this.browser = 3;
    if ("@mozilla.org/xre/app-info;1" in Components.classes) {
      var info = Components.classes["@mozilla.org/xre/app-info;1"].getService(Components.interfaces.nsIXULAppInfo);
      if(info.name.indexOf("Firefox") > -1) this.browser = 4;
    }	
    
    return exeFile.exists() ? this._exeFile = exeFile : null;
  }
  dm.createJob = function(links, opType) {
    const cs = this.cookieSupport;
    var l;
    var job = [this.getReferrer(links)];
    for (var j=0; j < links.length; j++) {
      l = links[j];
      job.push(l.href,
        cs ? l.description + "~%iget^=" + this.getCookie(l, links)
           : l.description
      );
    }
    return job.join("\r\n") + "\r\n";
  };
  dm.performJob = function(job) {
    const file = this.createJobFile(job);
    if (this.exeFile) {
      this.runNative(['-f', file.path, '-b', this.browser])
    } else {
      CC["@presenta/iGetter"]
              .getService(CI.nsIGetterMoz)
              .NewURL(file.path);
      if (file.exists()) file.remove(0);
    }
  };
  
  new FlashGotDM("Internet Download Accelerator");
  (new FlashGotDM("Internet Download Manager")).postSupport = true;

  var lg2002 = new FlashGotDM("LeechGet 2002");
  var lg2004 = new FlashGotDM("LeechGet");
  
  lg2004.createJob = lg2002.createJob = function(links, opType) {
    var referrer;
    switch (opType) {
      case fg.OP_ONE:
        return FlashGotDM.prototype.createJob.call(this, links, 
            links.quickDownload ? fg.OP_ONE : fg.OP_SEL);
        
      case fg.OP_SEL:
        var htmlDoc="<html><head><title>FlashGot selection</title></head><body>";
        var l;
        for (var j=0, len=links.length; j<len; j++) {
          l = links[j];
          var des = l.description;
          var tag = l.tagName ? l.tagName.toLowerCase() : "";
          htmlDoc = htmlDoc.concat(tag == "img"
            ? '<img src="' + l.href + '" alt="' + des
              + '" width="' + l.width + '" height="' + l.height +
              "\" />\n"
            : "<a href=\"" + l.href + "\">" + des + "</a>\n");
          this.updateProgress(links, j, len);
        }
        referrer = fg.httpServer.addDoc(
          htmlDoc.concat("</body></html>")
        );
        break;
       default:
        referrer = links.document && links.document.URL || "";
        if (referrer.match(/^\s*file:/i)) { // fix for local URLs
          // we serve local URLs through built-in HTTP server...
          return this.createJob(links,fg.OP_SEL);
        }
    }
    return this.createJobHeader({ length: 0, folder: "" },opType) + referrer + "\n";
  };
 
  new FlashGotDM("Net Transport");
  new FlashGotDM("Net Transport 2");
  new FlashGotDM("NetAnts");
  new FlashGotDM("Mass Downloader");
  
  dm = new FlashGotDM("Orbit");
  dm.nativeUI = "#OrbitDownloadUp, #OrbitDownload, #OrbitDownloadAll";
  
  dm = new FlashGotDM("ReGet");
  dm.postSupport = true;
  if("@reget.com/regetextension;1" in CC) {
    try {
      dm.reGetService = CC["@reget.com/regetextension;1"].createInstance(CI.IRegetDownloadFFExtension);
      if (dm.reGetService.isExtensionEnabled()) {
        dm._supported = true;
        dm.performJob = function() {};
        dm.createJob = function(links, opType) {
          const rg = this.reGetService;
          var l;
          var len = links.length;
          var ref = links.referrer;
          if (len == 1) {
            l = links[0];
            rg.setUrl(l.href);
            rg.setInfo(l.description);
            rg.setCookie(this.getCookie(l, links));
            rg.setReferer(ref);
            rg.setPostData(links.postData);
            rg.setConfirmation(true);
            rg.addDownload();
            return;
          }
          for (var j = 0; j < len; j++) {
            l = links[j];
            rg.addToMassDownloadList(
              l.href,
              ref,
              this.getCookie(l, links),
              l.description,
              "");
            this.updateProgress(links, j, len);
          }
          rg.runMassDownloadList();
        }
      }
    } catch(rgEx) {}
  }
  
  if (isWin) {
    dm = new FlashGotDMCust("Retriever");
    dm.cookieSupport = true;
    dm.askPath = ASK_NEVER;
    dm.custom = false;
    dm._supported = null;
    
    if (fg.getPref(dm.prefsBase + "maxLinks", -1000) == -1000) {
      fg.setPref(dm.prefsBase + "maxLinks", 10);
    }
    dm.customSupportCheck = function() {
      var wrk;
      try {
        cmd = this.readWinRegString("CLASSES_ROOT", "Retriever.Retriever.jar.HalogenWare\\shell\\Open\\command");
        this.jarPath = cmd.replace(/.*-jar "?(.*?\.jar).*/, "$1");
        this.argsTemplate = "[URL] [Referer:REFERER] [Cookie:COOKIE] [post:POST]";
        
        var exeFile = CC["@mozilla.org/file/directory_service;1"].getService(CI.nsIProperties)
          .get("WinD", CI.nsIFile);
        exeFile.append("System32");
        exeFile.append("javaw.exe");
        this.exeFile = exeFile;
        
        return true;
      } catch(e) {
        return false;
      } 
    };
    
    dm.makeArgs = function(parms) {
      return ["-jar", this.jarPath].concat(
        FlashGotDMCust.prototype.makeArgs.apply(this, arguments)
      );
    };
    
    dm = new FlashGotDMCust("DownloadStudio");
    dm.cookieSupport = true;
    dm.askPath = ASK_NEVER;
    dm.custom = false;
    dm._supported = null;
    
    dm.customSupportCheck = function() {
      var wrk;
      try {
        var path = this.readWinRegString("LOCAL_MACHINE", "SOFTWARE\\Conceiva\\DownloadStudio", "Path");
        if (!path) return false;
        var exeFile = CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
        exeFile.initWithPath(path);
        exeFile.append("DownloadStudio.exe");
        if (exeFile.exists() && exeFile.isExecutable()){
          this.exeFile = exeFile;
          this.argsTemplate = "<downloadstudio><originator>firefox</originator> " +
            "<script><add_jobs display_dialog=yes><joblist><job> [<url>URL</url>] " +
            "[<url_list_file>UFILE</url_list_file>] [<referer>REFERER</referer>] " +
            "[<post_data>POST</post_data>] [<cookie>COOKIE</cookie>] " +
            "</job></joblist></add_jobs></script></downloadstudio>";
          return true;
        }
      } catch(e) {}
      return false;
    };
    
  }
  
  const httpFtpValidator = function(url) {
    return /^(http:|ftp:)/.test(url);
  };
  dm = new FlashGotDM("Star Downloader");
  dm.cookieSupport = false;
  dm.isValidLink = httpFtpValidator;
  
  dm = new FlashGotDM("TrueDownloader");
  dm.isValidLink = httpFtpValidator;
  

  dm = new FlashGotDM("Thunder");
  dm.nativeUI = "#ThunderDownloadUp, #ThunderDownload, #ThunderDownloadAll";
  dm.thunderDelay = 3000;
  dm.performDownload = function(links, opType) {
    var self = this;
    var postCall = function() {
      self.__proto__.performDownload.call(self, links, opType);
    }
    try {
      var path = this.readWinRegString("LOCAL_MACHINE","Software\\Thunder Network\\ThunderOem\\thunder_backwnd", "Path");
      var exeFile = CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
      exeFile.initWithPath(path);
      this.runNative([], false, exeFile);
      fg.delayExec(postCall, this.thunderDelay); // give thunder one second to popup (more if cold start)
      this.thunderDelay = 1000;
      return;
    } catch(e) {}
    postCall;
  }
  new FlashGotDM("Thunder (Old)");

  
  if (isWin) {
    dm = new FlashGotDM("WellGet");
    dm.autoselect = false;
    dm.getRelativeExe = function() {
      try {
        return fg.prefs.getComplexValue("WellGet.path", CI.nsILocalFile);
      } catch(ex) {}
      return null;
    };
    dm.customSupportCheck = function() {
      var wellGetExe = this.getRelativeExe();
      try {
         var currentPath = wellGetExe.path;
         if(wellGetExe.exists() && wellGetExe.isExecutable()) return true;
         
         wellGetExe.initWithPath(fg.profDir.path.substring(0,2) +
           currentPath.substring(2));
         if (wellGetExe.exists() && wellGetExe.isExecutable()) {
           if(wellGetExe.path != currentPath) {
              fg.prefs.setComplexValue("WellGet.path",  CI.nsILocalFile, wellGetExe);
           }
           return true;
         }
         return false;
      } catch(ex) {
      }
      
      return !wellGetExe && this.baseSupportCheck();
    };
    dm.createJob = function(links, opType) {
      var wellGetExe = this.getRelativeExe();
      return FlashGotDM.prototype.createJob.call(this, links, opType, 
        wellGetExe ? [wellGetExe.path] : null);
    };
    dm.shouldList = function() { return true; }
  }

  dm = new FlashGotDM("JDownloader");
  dm.askPath = [true, true, true];
  dm.customSupportCheck = function() {
    try {
      if (this._checkPath()) return true;
      
      var self = this;
      fg.delayExec(function() {
        var r = self._createRequest("GET", function(r) {
          self._handleResponse(r.responseText.split(/[\r\n]+/));
        });
        r.send(null);
      }, 0);
    } catch(e) {
      fg.log(e);
    }
    return false;
  };
  dm._handleResponse = function(res) {
    if (res.length < 2) return;
    this.setPref("path",fg.isWindows ? res[0].replace(/\//g, '\\') : res[0]);
    if (this._checkPath()) {
      this.setPref("args", res[1].substring(0, res[1].lastIndexOf(res[0])).replace(/^java\s*/, ''));
      this._supported = true;
    }
  };
  dm._checkPath = function() {
    try {
    var p = this.getPref("path", "");
      if (p) {
        var f = CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
        f.initWithPath(p);
        if (f.exists()) {
          this.jdFile = f;
          return true;
        }
      }
    } catch(e) {}
    this.setPref("path", "");
    this.jdFile = null;
    return false;
  };
  dm._createRequest = function(method, callback) {
    var r = CC["@mozilla.org/xmlextras/xmlhttprequest;1"].createInstance(CI.nsIXMLHttpRequest);
    var url = this.getPref("url");
    r.open(method, url, true);
    try {
      var chan = r.channel;
      if (chan instanceof CI.nsIHttpChannel) {
        chan.referrer = chan.URI;
      }
    } catch(e) {
      fg.log(e);
    }
    try {
      r.setRequestHeader("referer", url);
    } catch(e) {
      fg.log(e);
    }
    if (callback) {
      r.onreadystatechange = function() {
        if (r.readyState == 4) {
          try {
            fg.log("JDownloader response:\n" + r.status + "\n" + r.responseText);
          } catch (e) {}
          callback(r);
        }
      };
    }
    return r;
  }
  dm.MAX_RETRIES = 3;
  dm.performDownload = function(links, opType) {
    var pp = { urls: [], descriptions: [], cookies: [] };
    var l, j, len;
    for (j = 0, len = links.length; j < len; j++) {
      l = links[j];
      pp.urls.push(l.href);
      pp.descriptions.push(l.description);
      pp.cookies.push(this.getCookie(links, l));
    }

    var data = [
      "referer=" + encodeURIComponent(this.getReferrer(links)),
      "autostart=" + (this.getPref("autostart", true) ? "1" : "0")
    ];
    if (links.folder && links.folder._fgSelected) data.push("dir=" + encodeURIComponent(links.folder));              
    if (links.postData) data.push("postData=" + encodeURIComponent(links.postData));
    for (j in pp) {
      data.push(j + "=" + encodeURIComponent(pp[j].join("\n")));
    }
    this._post(data.join("&"), this.MAX_RETRIES);
  };
  dm._post = function(data, retries) {
    var self = this;
    var r = this._createRequest("POST", function(r) {
      if (r.status != 200) self._handleRetry(data, retries);
    });
    
    r.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
    r.send(data);
  };
  
  dm._handleRetry = function(data, retries) {
    var msg, fail = true;
    if (retries-- == this.MAX_RETRIES) {
      try {
        if (!this.java) {
          var java = CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
          java.initWithPath(DOM.mostRecentBrowserWindow.java.lang.System.getProperty("java.home"));
          java.append("bin");
          java.append(fg.isWindows ? "java.exe" : "java");
          this.java = java;
        }
        var args = this.getPref("args").split(/\s+/);
        args.push(this.jdFile.path);
        fail = !this.runNative(args, false, this.java);
      } catch(e) {
        fg.log(e);
        this.java = null;
      } 
      if (fail) msg = "Cannot launch JDownloader. Please launch it manually or ensure Java is enabled in your browser.";
    } else if (retries == 0) {
      msg = "JDownloader not responding on " + this.getPref("url") + "!\nPlease check your firewall settings.";
    } else {
      fail = false;
    }
    if (fail) {
      DOM.mostRecentBrowserWindow.alert(msg);
    } else {
      var self = this;
      fg.delayExec(function() { self._post(data, retries); }, this.getPref("delay", 8) * 1000);
    }
  };

  dm = new FlashGotDMX("Aria", "aria", "[-r REFERER] [-d FOLDER] -g [URL]");
  dm.createJob = function(links,opType) {
    return FlashGotDMX.prototype.createJob.call(this,links,opType) + "\nsleep 4\n" + this.unixCmd+" -s &\n";
  };
  
  dm = new FlashGotDMX("Downloader 4 X (nt)", "nt");
  dm.createJob = function(links,opType) {
    return this.unixCmd + "&\nsleep 1\n" +
      (links.folder && links.folder._fgSelected
      ? this.unixCmd + " -d '" + links.folder + "'\n"
      :"") + 
      FlashGotDMX.prototype.createJob.call(this,links,opType);
  };
  
  dm = new FlashGotDMX("Downloader 4 X", "d4x", "[--referer REFERER] [--directory FOLDER] [-a URL] [--al POST] [COOKIE]");
  dm.askPath = [false, true, true];
  dm.cookieSupport = true;
  dm.createJob = function(links, opType) {
    const shellEsc = this.shellEsc;
    const referrer = shellEsc(this.getReferrer(links));
    const folder = links.folder._fgSelected && links.folder || null;
    const quiet = this.quiet;
    const len = links.length;
    var job;
    
    if (len > 0) {
        
      var urls = [];
      for (var j = 0; j < len; j++) {
        urls.push(shellEsc(links[j].href));
        this.updateProgress(links, j, len);
      }
      urls = urls.join(" ");
      
      var promptURLs_fakePost = null;
      var quietURLs_fakeCookie = null;
      
      if (quiet) {
        quietURLs_fakeCookie = urls;
        urls = null;
      } else if(len > 1) {
        promptURLs_fakePost = urls;
        urls = null;
      }
      
      var cookieFile = this.createCookieFile();
      
      job = cookieFile
        ? "mkdir -p $HOME/.netscape && ln -fs " + 
          shellEsc(this.createCookieFile().path) + 
          " $HOME/.netscape/cookies\n"
        : "";
       
      job += this.createCmdLine({
         URL: urls, 
         REFERER: referrer,
         COOKIE: quietURLs_fakeCookie || null,
         FOLDER: folder,
         POST: promptURLs_fakePost
      }); 
    } else job = "";
    
    return job;
  };
  
  dm = new FlashGotDMX("GNOME Gwget","gwget");
  dm.askPath = ASK_NEVER;
  dm.createJob=function(links, opType) {
    if (opType == fg.OP_ALL) {
      links.length = 1;
      links[0].href = links.document ? links.document.URL : this.getReferrer(links);
      opType = fg.OP_ONE;
    }
    return FlashGotDMX.prototype.createJob.call(this, links, opType)
  } 
  
  dm = new FlashGotDMX("KDE KGet","kget");
  dm.askPath = ASK_NEVER;
  
  if (isWin) {
    new FlashGotDM("wxDownload Fast");
  } else {
    dm=new FlashGotDMX("wxDownload Fast", "wxdfast", "[-reference REFERER] [-destination FOLDER] [-list UFILE]");
    dm.askPath = ASK_NEVER;
  }

  dm = new FlashGotDMX("cURL","curl", '-L -O [-o FNAME] [--referer REFERER] [-b COOKIE] [-d POST] [URL]');
  dm.postSupport = true;
  if (FlashGotDMMac.isMac) dm.autoselect = false;
  dm.createJob = function(links,opType) {
    var job="[ -x \"`which 'xterm'`\" ] &&  DOWN_CMD='xterm -e curl' || DOWN_CMD='curl'\n";
    if (links.folder) job += "cd '" + links.folder + "'\n";
    this.unixCmd = "$DOWN_CMD";
    return job + FlashGotDMX.prototype.createJob.call(this,links,opType);
  };
  
  dm = new FlashGotDMX("Wget", "wget", '-c [-O FNAME] [--directory-prefix=FOLDER] [--referer=REFERER] [--post-data=POST] [--load-cookies=CFILE] [--header=Cookie:COOKIE] [--input-file=UFILE]');
  dm.postSupport = true;
  dm.createJob=function(links,opType) {
    var job="[ -x \"`which 'xterm'`\" ] &&  DOWN_CMD='xterm -e wget' || DOWN_CMD='wget'\n";
    this.unixCmd = "$DOWN_CMD";
    return job + FlashGotDMX.prototype.createJob.call(this,links,opType);
  };
  
  dm = new FlashGotDMX("Aria 2", "aria2c", '--continue [-d FOLDER] [-o FNAME] [--referer=REFERER] [--load-cookies=CFILE] [--input-file=UFILE]');
  dm.postSupport = false;
  dm.createJob=function(links,opType) {
    var job="[ -x \"`which 'xterm'`\" ] &&  DOWN_CMD='xterm -e aria2c' || DOWN_CMD='aria2c'\n";
    this.unixCmd = "$DOWN_CMD";
    return job + FlashGotDMX.prototype.createJob.call(this,links,opType);
  };

  function FlashGotDMSD(version) {
    this._initMac(typeof(version) == "number" && version > 3 ? "Speed Download" : ("Speed Download " + version), "Spee");
    this.version = version;
    if (version > 2 || version == "Lite") {
      this.cookieSupport = true;
      this.postSupport = true;
    }
  };
  
  FlashGotDMSD.prototype=new FlashGotDMMac();
  FlashGotDMSD.prototype.createJob = function(links,opType) {
    var urlList = [];
    var cookieList = [];
    var l;
    for (var j=0, len = links.length; j < len; j++) {
      l = links[j];
      urlList.push(l.href);
      if (this.cookieSupport) {
        cookieList.push(this.getCookie(l, links));
      }
      this.updateProgress(links, j, len);
    }
    var job = 'tell app "' + this.macAppName + 
      '" to AddURL {"' + urlList.join('","') + '"}';
    
    if (this.postSupport) {

      if (links.postData) { 
        job +=' with form data "' + links.postData + '"';
      }

      const referer = this.getReferrer(links);
      if (referer && referer.length) {
        job += ' from "' + referer + '"';
      }

      if (cookieList.length) {
        job += ' with cookies {"' + cookieList.join('","') + '"}';
      }
    }
    
    return job;
  };
  
  if (fg.getPref("oldSD", false)) {
    new FlashGotDMSD(2);
    new FlashGotDMSD(3);
  }
  new FlashGotDMSD(3.5);
  new FlashGotDMSD("Lite");
  
  dm = new FlashGotDMMac("Leech", "com.manytricks.Leech");
  dm.askPath = [true, true, true];
  dm.cookieSupport = dm.postSupport = true;
  dm.createJob = function(links, opType) {
    var urlList = [];
    var cookieList = [];
    var l;
    for (var j = 0, len = links.length; j < len; j++) {
      l = links[j];
      urlList.push(l.href);
      if (this.cookieSupport) {
        cookieList.push(this.getCookie(l, links).replace(/;\s*$/, ''));
      }
      this.updateProgress(links, j, len);
    }
    
    var job = 'tell app "' + this.macAppName + '" to download URLs {"'
      + urlList.join('", "') + '"}';
    if (links.postData) {
      job += ' by posting data "' + links.postData + '"';
    }
    job += ' to POSIX path "' + links.folder + '"';
    if (cookieList.length) {
      job += ' using cookies "' + cookieList.join('; ') + '; "'; 
    }
    const referer = this.getReferrer(links);
    if (referer && referer.length) {
      job += ' with referrer "' + referer + '"';
    }
    
    return job;
  }
  
  dm = new FlashGotDMMac("iGetter", "iGET");
  dm.cookieSupport = true;
  dm.createJob = function(links, opType) {
    const referrer = this.getReferrer(links);
    var l, params = [];
    for (var j = 0, len = links.length; j < len; j++) {
      l = links[j];
      params.push('{\u00ABclass ----\u00BB:"' + l.href + 
        '", \u00ABclass refe\u00BB:"' + referrer  + 
        '", \u00ABclass cook\u00BB:"' + this.getCookie(l, links) +
        '"}');
      this.updateProgress(links, j, len);
    }
    var job = "tell application \""+ this.macAppName+ 
      "\"\n\u00ABevent iGETGURL\u00BB {" +
       params.join(",") +
      "} given \u00ABclass brsg\u00BB:\"MOZB\"\n" +
      "end tell\n";
    return job;
  };
  

  
  
  if ("nsIDownloadManager" in CI) {
    dm = new FlashGotDM(fg.getString("dm.builtIn"));
    dm._codeName = "_Built_In_";
    dm._supported = true;
    dm.priority = "zzz"; // put on the bottom of the list
    
    dm.askPath = [true, true, true];
    dm.postSupport = true;
    dm.performDownload = function(links, opType) {
      var ios = CC["@mozilla.org/network/io-service;1"].getService(CI.nsIIOService);
      const persistFlags = CI.nsIWebBrowserPersist.PERSIST_FLAGS_AUTODETECT_APPLY_CONVERSION;
      const dType = CI.nsIDownloadManager.DOWNLOAD_TYPE_DOWNLOAD;
      var postData = links.postStream || null;
      var cs = links.document && links.document.characterSet || "UTF-8";
      var ref = this.getReferrer(links);
      var refURI = ref && ios.newURI(ref, cs, null) || null;
      var uri, folder, file, m;
      var persist, args;
      var now = Date.now() * 1000;
      var dm = CC["@mozilla.org/download-manager;1"].getService(CI.nsIDownloadManager);
      folder = CC["@mozilla.org/file/local;1"].createInstance(CI.nsILocalFile);
      folder.initWithPath(links.folder);
      var mozAddDownload;
      if(dm.startBatchUpdate) {
        mozAddDownload = typeof(dType) == "undefined" 
          ? function(src, dest, des, persist) { return dm.addDownload(src, dest, des, null, now, null, persist); }
          : function(src, dest, des, persist) { return dm.addDownload(dType, src, dest, des, null, null, now, null, persist); }
          ;
        dm.startBatchUpdate();
      } else {
        mozAddDownload = function(src, dest, des, persist) { return dm.addDownload(dType, src, dest, des, null, now, null, persist); };
      }
      var dl;
      for(var j = 0, len = links.length, l; j < len; j++) {
        l = links[j];
        try {
          uri = ios.newURI(l.href, cs, null);
          if(!(uri instanceof CI.nsIURL)) continue;
          file = folder.clone();
          file.append(
            l.fname ||
            unescape((uri.fileName || uri.filePath.replace(/.*?([^\/]*)\/?$/, '$1') || uri.host))
            .replace(/[\x00-\x1f\\\/\&\|\^;:]+/g, '_')
            );
          if(!this.getPref("overwrite", false)) {
            for (;;) {
              if(!file.exists()) {
                file.create(0, 0644);
                break;
              } else { // rename
                m = file.leafName.match(/(.*?)(?:\((\d+)\))?(\.[^\.]+$|$)/);
                file.leafName = m[1] + "(" + ((m[2] && parseInt(m[2]) || 0) + 1) + ")" + m[3]; 
              }
            }
          }
          persist = CC["@mozilla.org/embedding/browser/nsWebBrowserPersist;1"].createInstance(CI.nsIWebBrowserPersist);
          persist.persistFlags = persistFlags;
          
          fg.log("Saving " + l.href + " to " + file.path);
          
          persist.progressListener = dl = 
            mozAddDownload(uri, ios.newFileURI(file), file.leafName, persist)
              .QueryInterface(CI.nsIWebProgressListener);
          persist.saveURI(uri, null, // cachekey
                  refURI, postData, null, file);
          this.updateProgress(links, j, len);
        } catch (e) {
          fg.log("Skipping link " + l.href + ": " + e);
        }
      }
      if(dm.endBatchUpdate) dm.endBatchUpdate();
      if(dm.flush) dm.flush();
      
      if(this.getPref("showDM", true)) {
        try { // SeaMonkey
          dm.open(links.browserWindow, dl);
        } catch(notSeamonkey) {
          
          const DMBRANCH = "browser.download.manager.";
          var prefs = fg.prefService.getBranch(DMBRANCH);
          try {
            if (!(prefs.getBoolPref("showWhenStarting") && prefs.getBoolPref("useWindow")))
              return;
          } catch(noPref) {
            return;
          }
          
          
          
          try { // 1.9 (Toolkit)
             // http://mxr.mozilla.org/seamonkey/source/toolkit/components/downloads/src/nsDownloadProxy.h#94
             var dmui = CC["@mozilla.org/download-manager-ui;1"].getService(CI.nsIDownloadManagerUI);
             var focus = false;
             try {
               focus = prefs.getBoolPref("focusWhenStarting");
             } catch(noPref) {}
             if (dmui.visible && !focus) {
               dmui.getAttention();
               return;
             }
             dmui.show(null, dl, CI.nsIDownloadManagerUI.REASON_NEW_DOWNLOAD);
          } catch(e1) {
            try { // 1.8 (Firefox 2)
              links.browserWindow.document.getElementById("Tools:Downloads").doCommand();
            } catch(e2) {
            }
          }
        }
      }
    };
  }
  
  FlashGotDMCust.init();
  
  fg.sortDMS();
  
  dm = null;
};