/*
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "build_config.h"
#include "compiler.h"

/* ==================== RUNTIME_RM ==================== */
#ifdef RUNTIME_RM

CODE16;

#include "assert.h"
#include "debug.h"
#include "stdio.h"
#include "ptrace.h"

void
bios_19_xxxx(struct regs *regs)
{
}

__attribute__((__noreturn__)) void
bios_18_xxxx(struct regs *regs)
{
	putcstr("\n");
	putcstr("No operating system found.\n");

	asm volatile (
		"cli\n"
		"hlt\n"
	);

	/*NOTREACHED*/
	assert(0);
}

#endif /* RUNTIME_RM */
/* ==================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

CODE16;

#include "var.h"
#include "disk.h"
#include "debug.h"
#include "stdio.h"
#include "ptrace.h"
#include "entry.h"
#include "cmos.h"
#include "el_torito.h"
#include "assert.h"

static int
boot_floppy(void)
{
	unsigned char drive_type;
	uint16_t ax, bx, cx, dx;

	/*
	 * Lookup first floppy
	 */
	drive_type = cmos_get(fd_config) >> 4;

	if (drive_type == 0) {
		/* No floppy A present */
		return -1;
	}

	/* Read boot record. */
	asm volatile (
		"\tmovw %%di, %%es\n"
		"\tint $0x13\n"
		"\tpushfw\n"
		"\tpopw %%ax\n"
		: "=a" (ax), "=b" (bx), "=c" (cx), "=d" (dx)
		: "0" ((uint16_t) ((0x02 << 8) | (1 << 0))), /* Read, # Secs */
		  "1" ((uint16_t) 0x7c00),                   /* Offset */
		  "2" ((uint16_t) ((0 << 8) | (1 << 0))),    /* Cyl, Sec */
		  "3" ((uint16_t) ((0 << 8) | (0x00 << 0))), /* Head, Drive */
		  "D" ((uint16_t) 0x0000)                    /* Segment */
	);
	if (ax & (1 << 0)) {
		/* Read error. */
		return -1;
	}

#if 0
	/*
	 * Dump boot sector
	 */
	int i;
	bprintf("\n");
	for (i = 0; i < 512; i += 2) {
		bprintf("%04x ", get_word(0x0000, (unsigned short *) (0x7c00 + i)));
	}
#endif
	
	/* Check for boot signature. */
	if (get_word(0x0000, 0x7c00 + 0x1fe) != 0xaa55) {
		/* No boot block on floppy */
		return -1;
	}

	return 0x00;
}

static int
boot_cdrom(void)
{
	unsigned char device;
	struct boot_record_volume *brv;
	struct validation *validation;
	struct initial *initial;
	unsigned char ret;
	unsigned char remapemul;

	/*
	 * Lookup first CDROM.
	 */
	device = ebda_get(cdidmap[0xe0 - 0xe0]);
	if (MAX_BIOS_HDS <= device) {
		/* No cdrom found. */
		return -1;
	}

	/*
	 * Read boot record volume descriptor.
	 */
	ret = read_sector_lba_atapi(device, 1, BRV_SECTOR,
			0x0400, 0x0000, 2048);
	if (ret != 0) {
		return -1;
	}
	assert(ret == 0);

	brv = (struct boot_record_volume *) 0x4000;
	if (brv->indicator != 0) {
		/* No bootable CDROM. */
		ebda_put(cdemu.active, 0);
		return -1;
	}

	/*
	 * Read boot catalog.
	 */
	ret = read_sector_lba_atapi(device, 1, brv->bootcat_p,
			0x0400, 0x0000, 2048);
	if (ret != 0) {
		return -1;
	}
	assert(ret == 0);

	validation = (struct validation *) 0x4000;
	assert(validation->header_id == 0x01);
	assert(validation->platform_id == 0x0);

	initial = (struct initial *) ((char *) 0x4000 + sizeof(struct validation));
	if (initial->boot_indicator != 0x88) {
		/* No bootable CDROM. */
		ebda_put(cdemu.active, 0);
		return -1;
	}

	/*
	 * Save values.
	 */
	ebda_put(cdemu.media, initial->boot_media);
	if (initial->boot_media == 0x00) {
		/*
		 * FIXME ElTorito Hardcoded. cdrom is hardcoded as device 0xE0.
		 * Win2000 cd boot needs to know it booted from cd
		 */
		ebda_put(cdemu.emulated_drive, 0xe0);
	} else if (initial->boot_media < 4) {
		ebda_put(cdemu.emulated_drive, 0x00);
	} else {
		ebda_put(cdemu.emulated_drive, 0x80);
	}

	/* FIXME ElTorito Harddisk. current code can only emulate a floppy */

	ebda_put(cdemu.controller_index, device / 2);
	ebda_put(cdemu.device_spec, device % 2);

	ebda_put(cdemu.load_segment, initial->load_segment);
	ebda_put(cdemu.buffer_segment, 0x0000);

	ebda_put(cdemu.sector_count, initial->sector_count);

	ebda_put(cdemu.ilba, initial->load_rba);

	if (ebda_get(cdemu.media) == 0x01) {
		/* 1.2M floppy */
		ebda_put(cdemu.vdevice.spt, 15);
		ebda_put(cdemu.vdevice.cylinders, 80);
		ebda_put(cdemu.vdevice.heads, 2);
	} else if (ebda_get(cdemu.media) == 0x02) {
		/* 1.44M floppy */
		ebda_put(cdemu.vdevice.spt, 18);
		ebda_put(cdemu.vdevice.cylinders, 80);
		ebda_put(cdemu.vdevice.heads, 2);
	} else if (ebda_get(cdemu.media) == 0x03) {
		/* 2.88M floppy */
		ebda_put(cdemu.vdevice.spt, 36);
		ebda_put(cdemu.vdevice.cylinders, 80);
		ebda_put(cdemu.vdevice.heads, 2);
	} else if (ebda_get(cdemu.media) == 0x04) {
		/* Harddisk */
		assert(0);      /* FIXME VOSSI */
	}

	/* Everything is ok, so from now on, the emulation is active. */
	if (ebda_get(cdemu.media) != 0x00) {
		ebda_put(cdemu.active, 1);
	}

	if (ebda_get(cdemu.media) == 0x00) {
		/* No Emulation */
		/* Nothing to do... */

	} else if (0x01 <= ebda_get(cdemu.media)
		&& ebda_get(cdemu.media) <= 0x03) {
		/* Floppy Emulation */
		/* Increment number of floppies. */
		uint16_t sys_conf;
		uint8_t num;

		sys_conf = var_get(sys_conf);

		if ((sys_conf >> 0) & 1) {
			num = ((sys_conf >> 6) & 3) + 1;
		} else {
			num = 0;
		}
		assert(num <= 2);
		num++;

		sys_conf &= ~(1 << 0);
		sys_conf &= ~(3 << 6);
		sys_conf |= 1 << 0;
		sys_conf |= (num - 1) << 6;

		var_put(sys_conf, sys_conf);

	} else if (ebda_get(cdemu.media) == 0x04) {
		/* Harddisk Emulation */
		/* Increment number of harddisks. */
		uint8_t hd_num;

		hd_num = var_get(hd_num);

		hd_num++;

		var_put(hd_num, hd_num);

	} else {
		assert(0);
	}

	remapemul = ebda_get(cdemu.media);

	bprintf("Emulation: ");

	if (remapemul == 0x00) {
		/* No emulation, read load_rba directly. */
		bprintf("None - ");

	} else if (remapemul == 0x01) {
		bprintf("1.2 MB disk - ");

	} else if (remapemul == 0x02) {
		bprintf("1.44 MB disk - ");

	} else if (remapemul == 0x03) {
		bprintf("2.88 MB disk - ");

	} else if (remapemul == 0x04) {
		bprintf("Hard Disk - ");

	} else {
		bprintf("Illegal emulation.\n");
		for (;;);

	}

	/* We should read 512 byte blocks! FIXME VOSSI */
	ret = read_sector_lba_atapi(device,
			1 + (ebda_get(cdemu.sector_count) - 1) / 4,
			ebda_get(cdemu.ilba),
			0x0000, 0x7c00, 2048);
	if (ret != 0) {
		/* Read error. */
		return -1;
	}

	return ebda_get(cdemu.emulated_drive);
}

static int
boot_harddisk(void)
{
	unsigned char device;
	uint16_t ax, bx, cx, dx;

	/* Lookup first harddisk. */
	device = ebda_get(hdidmap[0x80 - 0x80]);
	if (MAX_BIOS_HDS <= device) {
		/* No harddisk found. */
		return -1;
	}

	/* Read boot record. */
	asm volatile (
		"\tmovw %%di, %%es\n"
		"\tint $0x13\n"
		"\tmovw %%ds, %%di\n"
		"\tmovw %%di, %%es\n"
		"\tpushfw\n"
		"\tpopw %%ax\n"
		: "=a" (ax), "=b" (bx), "=c" (cx), "=d" (dx)
		: "0" ((uint16_t) ((0x02 << 8) | (1 << 0))), /* Read, # Secs */
		  "1" ((uint16_t) 0x7c00),                   /* Offset */
		  "2" ((uint16_t) ((0 << 8) | (1 << 0))),    /* Cyl, Sec */
		  "3" ((uint16_t) ((0 << 8) | (0x80 << 0))), /* Head, Drive */
		  "D" ((uint16_t) 0x0000)                    /* Segment */
	);
	if (ax & (1 << 0)) {
		/* Read error. */
		return -1;
	}

	/* Check for boot signature. */
	if (get_word(0x0000, 0x7c00 + 0x1fe) != 0xaa55) {
		/* No boot block on hard disk. */
		return -1;
	}

	return 0x80;
}

static int
boot_int18h(void)
{
	asm volatile (
		"\tint $0x18\n"
	);
	/* never returns */
	return -1;
}

static int
boot_int19h(void)
{
	asm volatile (
		"\tint $0x19\n"
	);
	/* only returns if no boot device available */
	return -1;
}

uint8_t
boot(void)
{
	enum {
		BOOT_NONE,
		BOOT_A,
		BOOT_CDROM,
		BOOT_C,
		BOOT_D,
		BOOT_E,
		BOOT_F,
		BOOT_LS_ZIP,
		BOOT_SCSI,
	};
	static const uint8_t seq[][3] = {
		{ BOOT_A, BOOT_C, BOOT_SCSI },
		{ BOOT_C, BOOT_A, BOOT_SCSI },
		{ BOOT_C, BOOT_CDROM, BOOT_A },
		{ BOOT_CDROM, BOOT_C, BOOT_A },
		{ BOOT_D, BOOT_A, BOOT_SCSI },
		{ BOOT_E, BOOT_A, BOOT_SCSI },
		{ BOOT_F, BOOT_A, BOOT_SCSI },
		{ BOOT_SCSI, BOOT_A, BOOT_C },
		{ BOOT_SCSI, BOOT_C, BOOT_A },
		{ BOOT_C, BOOT_NONE, BOOT_NONE },
		{ BOOT_LS_ZIP, BOOT_C, BOOT_NONE },
	};
	uint8_t cmos_seq;
	int bootdev;
	int i;

	bprintf("\n\nChecking boot devices ...\n");

	cmos_seq = cmos_get(x3c) & 0xf;
	if (sizeof(seq) / sizeof(seq[0]) <= cmos_seq) {
		cmos_seq = 0;
	}

	bootdev = -1;
	for (i = 0; i < 3; i++) {
		uint8_t dev;

		dev = seq[cmos_seq][i];

		if (dev == BOOT_NONE) {
			/* Nothing... */

		} else if (dev == BOOT_A) {
			/* Try to boot from floppy. */
			DEBUGPRINT(3, "Trying to boot from floppy... ");
			bprintf(" - Floppy:   ");
			bootdev = boot_floppy();

		} else if (dev == BOOT_CDROM) {
			/* Try to boot from cdrom. */
			DEBUGPRINT(3, "Trying to boot from cdrom... ");
			bprintf(" - CDROM:    ");
			bootdev = boot_cdrom();

		} else if (dev == BOOT_C) {
			/* Try to boot from harddisk. */
			DEBUGPRINT(3, "Trying to boot from harddisk... ");
			bprintf(" - Harddisk: ");
			bootdev = boot_harddisk();

		} else if (dev == BOOT_SCSI) {
			/* Try to boot from external device. */
			DEBUGPRINT(3, "Trying to boot from external device... ");
			bprintf(" - Int 19h:  ");
			bootdev = boot_int19h();

		} else {
			/* FIXME */
			bprintf("Unsupported boot entry in CMOS, "
				"please run setup.\n");
			continue;
		}

		if (bootdev != -1) {
			DEBUGPRINT(3, "OK\n");
			bprintf("OK\n");
			break;
		} else {
			DEBUGPRINT(3, "Failed\n");
			bprintf("Failed\n");
		}
	}

#if 0
	/* Switch off A20 gate. */
	outb(0, 0x92);
#endif

	if (bootdev == -1) {
		/* last chance - never returns */
		DEBUGPRINT(3, "Trying int18h... \n");
		boot_int18h();
	}

	return bootdev;
}

#endif /* INIT_RM */
