/*
 * Copyright (C) 2012 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue.h"

#include "ferranti_abtaster.h"

#define COMP_(x) ferranti_abtaster_ ## x

struct cpssp {
	/* Config */
	char name[1024];

	/* Ports */
	struct sig_boolean *port_bit0;
	struct sig_boolean *port_bit1;
	struct sig_boolean *port_bit2;
	struct sig_boolean *port_bit3;
	struct sig_boolean *port_bit4;
	struct sig_boolean *port_ready;

	struct sig_boolean *port_hole0;
	struct sig_boolean *port_hole1;
	struct sig_boolean *port_hole2;
	struct sig_boolean *port_hole3;
	struct sig_boolean *port_hole4;
	struct sig_boolean *port_hole5;

	struct sig_magneto_optical *port_media;
	unsigned long long pos;
};

static void
COMP_(next)(struct cpssp *cpssp)
{
	int ret;
	char c[7];
	char dummy;

	/*
	 * Read next code.
	 */
	ret = sig_magneto_optical_read(cpssp->port_media, cpssp,
			cpssp->pos++, c, &dummy, &dummy);
	if (ret == 0) {
		c[0] = '*';
		c[1] = '*';
		c[2] = '*';
		c[3] = '.';
		c[4] = '*';
		c[5] = '*';
	}
#if 0
	fprintf(stderr, "Read %c%c%c%c%c%c\n",
			c[0], c[1], c[2], c[3], c[4], c[5]);
#endif

	sig_boolean_set(cpssp->port_bit0, cpssp, c[5] == '*');
	sig_boolean_set(cpssp->port_bit1, cpssp, c[4] == '*');
	sig_boolean_set(cpssp->port_bit2, cpssp, c[2] == '*');
	sig_boolean_set(cpssp->port_bit3, cpssp, c[1] == '*');
	sig_boolean_set(cpssp->port_bit4, cpssp, c[0] == '*');
	sig_boolean_set(cpssp->port_ready, cpssp, ret != 0);

	sig_boolean_set(cpssp->port_hole0, cpssp, c[5] == '*');
	sig_boolean_set(cpssp->port_hole1, cpssp, c[4] == '*');
	sig_boolean_set(cpssp->port_hole2, cpssp, c[3] == '.');
	sig_boolean_set(cpssp->port_hole3, cpssp, c[2] == '*');
	sig_boolean_set(cpssp->port_hole4, cpssp, c[1] == '*');
	sig_boolean_set(cpssp->port_hole5, cpssp, c[0] == '*');
}

static void
COMP_(start_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	if (val) {
		/*
		 * Read next code.
		 */
		COMP_(next)(cpssp);
	}
}

static void
COMP_(power_set)(void *_cpssp, unsigned int val)
{
	/* ... */
}

static void
COMP_(change_set)(void *_cpssp, const char *path)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	char name[1024];

	/*
	 * Remove old Lochstreifen (if any).
	 */
	system_port_disconnect(cpssp->name, "media");

	if (*path == '\0') {
		return;
	}

	/*
	 * Insert new Lochstreifen (if any).
	 */
	name[0] = ':';
	strcpy(&name[1], path);
	system_port_connect(cpssp->name, "media", name, "connect");

	cpssp->pos = 0;

	/*
	 * Read first code.
	 */
	COMP_(next)(cpssp);
}

void *
COMP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_start,
	struct sig_boolean *port_bit0,
	struct sig_boolean *port_bit1,
	struct sig_boolean *port_bit2,
	struct sig_boolean *port_bit3,
	struct sig_boolean *port_bit4,
	struct sig_boolean *port_ready,
	struct sig_magneto_optical *port_media,
	struct sig_string *port_change,
	struct sig_boolean *port_hole0,
	struct sig_boolean *port_hole1,
	struct sig_boolean *port_hole2,
	struct sig_boolean *port_hole3,
	struct sig_boolean *port_hole4,
	struct sig_boolean *port_hole5
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = COMP_(power_set),
	};
	static const struct sig_boolean_funcs start_funcs = {
		.set = COMP_(start_set),
	};
	static const struct sig_string_funcs change_funcs = {
		.set = COMP_(change_set),
	};
	struct cpssp *cpssp;

	system_name_push(name);

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	assert(strlen(system_path()) < sizeof(cpssp->name));
	strcpy(cpssp->name, system_path());

	/* Call */
	/* Out */
	cpssp->port_bit0 = port_bit0;
	sig_boolean_connect_out(port_bit0, cpssp, 0);
	cpssp->port_bit1 = port_bit1;
	sig_boolean_connect_out(port_bit1, cpssp, 0);
	cpssp->port_bit2 = port_bit2;
	sig_boolean_connect_out(port_bit2, cpssp, 0);
	cpssp->port_bit3 = port_bit3;
	sig_boolean_connect_out(port_bit3, cpssp, 0);
	cpssp->port_bit4 = port_bit4;
	sig_boolean_connect_out(port_bit4, cpssp, 0);
	cpssp->port_ready = port_ready;
	sig_boolean_connect_out(port_ready, cpssp, 0);

	cpssp->port_hole0 = port_hole0;
	sig_boolean_connect_out(port_hole0, cpssp, 0);
	cpssp->port_hole1 = port_hole1;
	sig_boolean_connect_out(port_hole1, cpssp, 0);
	cpssp->port_hole2 = port_hole2;
	sig_boolean_connect_out(port_hole2, cpssp, 0);
	cpssp->port_hole3 = port_hole3;
	sig_boolean_connect_out(port_hole3, cpssp, 0);
	cpssp->port_hole4 = port_hole4;
	sig_boolean_connect_out(port_hole4, cpssp, 0);
	cpssp->port_hole5 = port_hole5;
	sig_boolean_connect_out(port_hole5, cpssp, 0);

	cpssp->port_media = port_media;

	/* In */
	sig_boolean_connect_in(port_power, cpssp, &power_funcs);
	sig_boolean_connect_in(port_start, cpssp, &start_funcs);
	sig_string_connect(port_change, cpssp, &change_funcs);

	system_name_pop();

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* FIXME */

	free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fp)
{
}

void
COMP_(resume)(void *_cpssp, FILE *fp)
{
}
