/*
 * Copyright (c) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * Copyright (c) 2004 Fabrice Bellard.
 *
 * Derived from QEMU (hw/cirrus_vga_rop.h)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/*
 * This generates all possible blitter funtions for exactly one raster
 * operation specified as code ROP_CODE(d, s) and name ROP_NAME in all
 * possible color depths, for example
 *
 * #define ROP_NAME src_or_dst
 * #define ROP_CODE(d, s) d = (s) | (d)
 *
 * This file get's included 15 times from chip_cirrus_gd5446.c, once for each
 * of the 16 possible logical operation on source and destination minus
 * the "destination identity".
 *
 * Special thanks to qemu's cirrus vga for this idea!
 */

#ifndef ROP_NAME
#error chip_cirrus_gd5446_rop.c needs ROP_NAME defined
#endif
#ifndef ROP_CODE
#error chip_cirrus_gd5446_rop.c needs ROP_CODE defined
#endif

/*
 * This is the most simple raster operation. It just replaces all bytes in
 * the destination area with the result of the logical operation ROP_CODE
 * of themselves and their corresponding bytes in the source area.
 */
static void
paste(_cirrus_bitblt_forward_, ROP_NAME)(struct cpssp *cpssp)
{
        int x, y;
        unsigned long dest, src;
        int dest_jump, src_jump;
	uint8_t d;

#ifdef CIRRUS_DEBUG_BITBLITTER
        faum_log(FAUM_LOG_DEBUG, __FUNCTION__, "", "\n");
#endif

        dest = cpssp->bitblt.destination_pointer;
        src  = cpssp->bitblt.source_pointer;

        /* What to jump after each line? */
        dest_jump = cpssp->bitblt.dest_pitch - cpssp->bitblt.width;
        src_jump = cpssp->bitblt.src_pitch - cpssp->bitblt.width;

        for (y = 0; y < cpssp->bitblt.height; y++) {
                for (x = 0; x < cpssp->bitblt.width; x++) {
			d = video_readb(cpssp, dest);
                        ROP_CODE(d, video_readb(cpssp, src));
			video_writeb(cpssp, dest, d);
                        dest++;
                        src++;
                }
                dest += dest_jump;
                src  += src_jump;
        }
}

/*
 * The same as _cirrus_bitblt_forward_bla(), but supporting transparency.
 * The pixel to be written is compared to a stored key color,
 * and if it matches, it will not be written.
 * Works only for 8bpp and 16bpp graphics modes.
 */
static void
paste(_cirrus_bitblt_transparent_forward_, ROP_NAME)(struct cpssp *cpssp)
{
        int x, y;
        unsigned long dest, src;
        int dest_jump, src_jump;
        int i;
	uint8_t d;

#ifdef CIRRUS_DEBUG_BITBLITTER
        faum_log(FAUM_LOG_DEBUG, __FUNCTION__, "", "\n");
#endif

        if (CIRRUS_IS_8BPP) {
                if (cpssp->bitblt.key_color[0]
                    != cpssp->bitblt.key_color[1]) {
#ifdef CIRRUS_DEBUG_BITBLITTER
                        faum_log(FAUM_LOG_DEBUG, __FUNCTION__, "",
                                 "Key colors should have been programmed "
                                 "identically. Aborting...\n");
#endif
                        _cirrus_bitblt_reset(cpssp);
                        return;
                }
        }

        dest = cpssp->bitblt.destination_pointer;
        src  = cpssp->bitblt.source_pointer;

        /* What to jump after each line? */
        dest_jump = cpssp->bitblt.dest_pitch - cpssp->bitblt.width;
        src_jump = cpssp->bitblt.src_pitch - cpssp->bitblt.width;

        for (y = 0; y < cpssp->bitblt.height; y++) {
                i = 1;          /* 16bpp mode starts with high byte */
                for (x = 0; x < cpssp->bitblt.width; x++) {
			d = video_readb(cpssp, dest);
                        ROP_CODE(d, video_readb(cpssp, src));
                        if (d != cpssp->bitblt.key_color[i]) {
				video_writeb(cpssp, dest, d);
                        }
                        if (i) {
                                i = 0;
                        } else {
                                i = 1;
                        }
                        dest++;
                        src++;
                }
                dest += dest_jump;
                src  += src_jump;
        }
}

/*
 * This is the second most simple raster operation. It's
 * _cirrus_bitblt_forward_BLA in reverse (right to left and bottom to top).
 * This is necessary if both source and destination area overlap and the 
 * source is higher on the y-axis than the destination, e.g.
 *   +--------------+
 *   | src     +--------------+
 *   |         |dest          |
 *   +-------- |              |
 *             +--------------+
 * The source and destination address is the highest value of each area.
 */
static void
paste(_cirrus_bitblt_backward_, ROP_NAME)(struct cpssp *cpssp)
{
        int x, y;
        unsigned long dest, src;
        int dest_jump, src_jump;
	uint8_t d;

#ifdef CIRRUS_DEBUG_BITBLITTER
        faum_log(FAUM_LOG_DEBUG, __FUNCTION__, "", "\n");
#endif

        dest = cpssp->bitblt.destination_pointer;
        src  = cpssp->bitblt.source_pointer;

        /* What to jump after each line? */
        /* Remember, both pitches here are negative. */
        dest_jump = cpssp->bitblt.width + cpssp->bitblt.dest_pitch;
        src_jump = cpssp->bitblt.width + cpssp->bitblt.src_pitch;

        for (y = 0; y < cpssp->bitblt.height; y++) {
                for (x = 0; x < cpssp->bitblt.width; x++) {
			d = video_readb(cpssp, dest);
                        ROP_CODE(d, video_readb(cpssp, src));
			video_writeb(cpssp, dest, d);
                        dest--;
                        src--;
                }
                dest += dest_jump;
                src  += src_jump;
        }
}

/*
 * The same as _cirrus_bitblt_backward_bla(), but supporting transparency.
 * The pixel to be written is compared to a stored key color,
 * and if it matches, it will not be written.
 * Works only for 8bpp and 16bpp graphics modes.
 */
static void
paste(_cirrus_bitblt_transparent_backward_, ROP_NAME)(struct cpssp *cpssp)
{
        int x, y;
        unsigned long dest, src;
        int dest_jump, src_jump;
        int i;
	uint8_t d;

#ifdef CIRRUS_DEBUG_BITBLITTER
        faum_log(FAUM_LOG_DEBUG, __FUNCTION__, "", "\n");
#endif

        if (CIRRUS_IS_8BPP) {
                if (cpssp->bitblt.key_color[0]
                    != cpssp->bitblt.key_color[1]) {
#ifdef CIRRUS_DEBUG_BITBLITTER
                        faum_log(FAUM_LOG_DEBUG, __FUNCTION__, "",
                                 "Key colors should have been programmed "
                                 "identically. Aborting...\n");
#endif
                        _cirrus_bitblt_reset(cpssp);
                        return;
                }
        }

        dest = cpssp->bitblt.destination_pointer;
        src  = cpssp->bitblt.source_pointer;

        /* What to jump after each line? */
        /* Remember, both pitches here are negative. */
        dest_jump = cpssp->bitblt.width + cpssp->bitblt.dest_pitch;
        src_jump = cpssp->bitblt.width + cpssp->bitblt.src_pitch;

        for (y = 0; y < cpssp->bitblt.height; y++) {
                i = 0;          /* 16bpp mode starts lines with low byte */
                for (x = 0; x < cpssp->bitblt.width; x++) {
			d = video_readb(cpssp, dest);
                        ROP_CODE(d, video_readb(cpssp, src));
                        if (d != cpssp->bitblt.key_color[i]) {
				video_writeb(cpssp, dest, d);
                        }
                        if (i) {
                                i = 0;
                        } else {
                                i = 1;
                        }
                        dest--;
                        src--;
                }
                dest += dest_jump;
                src  += src_jump;
        }
}

/*
 * Only these functions
 * don't care about the color depth set in cpssp->blt_mode & 0x30,
 * all others do.
 */
#define ROP_DEPTH 8
#include "chip_cirrus_gd5446_rop_depth.c"
#undef ROP_DEPTH

#define ROP_DEPTH 16
#include "chip_cirrus_gd5446_rop_depth.c"
#undef ROP_DEPTH

#define ROP_DEPTH 24
#include "chip_cirrus_gd5446_rop_depth.c"
#undef ROP_DEPTH

#define ROP_DEPTH 32
#include "chip_cirrus_gd5446_rop_depth.c"
#undef ROP_DEPTH
