/*
 * Copyright (C) 2003-2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef STATE
#if defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT

struct {
	/* 9-24 */
	uint8_t e;	/* MTRR Enable */
	uint8_t fe;	/* Fixed Range Registers Enable */
	uint8_t type;	/* Default Memory Type */

	/* 9-25 */
	uint8_t fix64k[8];	/* Memory Type 0x00000-0x7ffff */
	uint8_t fix16k[16];	/* Memory Type 0x80000-0xbffff */
	uint8_t fix4k[64];	/* Memory Type 0xc0000-0xfffff */

	/* 9-26 */
	struct {
		uint8_t valid;
		uint8_t type;
		uint64_t base;
		uint64_t mask;
	} var[8];
} NAME;

#endif /* defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT */
#endif /* STATE */
#ifdef EXPORT
#if defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT

/*forward*/ static uint8_t
NAME_(type)(struct cpssp *cpssp, paddr_t paddr);
/*forward*/ static int
NAME_(wrmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t val);
/*forward*/ static int
NAME_(rdmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t *valp);
/*forward*/ static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int n_val);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT */
#endif /* EXPORT */
#ifdef BEHAVIOR
#if defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT

enum NAME_(mem_type) {
	NAME_(UC) = 0,
	NAME_(WC) = 1,
	NAME_(WT) = 4,
	NAME_(WP) = 5,
	NAME_(WB) = 6,
};

static uint8_t
NAME_(type)(struct cpssp *cpssp, paddr_t paddr)
{
	/* What about overlapping regions? FIXME */
	int i;

	if (! cpssp->NAME.e) {
		return NAME_(UC);
	}

	if (cpssp->NAME.fe) {
		if (paddr < 0x80000) {
			return cpssp->NAME.fix64k[(paddr - 0x00000) / 0x10000];
		} else if (paddr < 0xc0000) {
			return cpssp->NAME.fix16k[(paddr - 0x80000) / 0x4000];
		} else if (paddr < 0x100000) {
			return cpssp->NAME.fix4k[(paddr - 0xc0000) / 0x1000];
		}
	}

	for (i = 0; i < 8; i++) {
		if (cpssp->NAME.var[i].valid
		 && cpssp->NAME.var[i].base
				== (paddr & cpssp->NAME.var[i].mask)) {
			return cpssp->NAME.var[i].type;
		}
	}

	/* Default memory type. */
	return cpssp->NAME.type;
}

static int
NAME_(wrmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t val)
{
	int o;
	int i;

	switch (ecx) {
	case 0xfe: /* MSR_IA32_MTRRCAP */
		/* 9-23 */
		/* Read-only Register */
		return 1;

	case 0x2ff: /* MSR_IA32_MTRR_DEF_TYPE */
		/* 9-24 */
		if ((val & 0xfffffffffffff300ULL) != 0) {
			/* Reserved bits set. */
			return 0;
		}
		switch ((int) (val >> 0) & 0xff) {
		case NAME_(UC):
		case NAME_(WC):
		case NAME_(WT):
		case NAME_(WP):
		case NAME_(WB):
			break;
		default:
			/* Bad default memory type. */
			return 0;
		}
		cpssp->NAME.e = (val >> 11) & 1;
		cpssp->NAME.fe = (val >> 10) & 1;
		cpssp->NAME.type = (val >> 0) & 0xff;
		return 1;

	case 0x250: /* MSR_IA32_MTRR_FIX64K_00000 */
		/* 9-25 */
		o = 0;
		/* Check types. */
		for (i = 0; i < 8; i++) {
			switch ((val >> (i*8)) & 0xff) {
			case NAME_(UC):
			case NAME_(WC):
			case NAME_(WT):
			case NAME_(WP):
			case NAME_(WB):
				break;
			default:
				/* Bad memory type. */
				return 0;
			}
		}
		for (i = 0; i < 8; i++) {
			cpssp->NAME.fix64k[o + i] = (val >> (i*8)) & 0xff;
		}
		return 1;

	case 0x258: /* MSR_IA32_MTRR_FIX16K_80000 */
	case 0x259: /* MSR_IA32_MTRR_FIX16K_A0000 */
		/* 9-25 */
		o = (ecx & 1) * 8;
		/* Check types. */
		for (i = 0; i < 8; i++) {
			switch ((val >> (i*8)) & 0xff) {
			case NAME_(UC):
			case NAME_(WC):
			case NAME_(WT):
			case NAME_(WP):
			case NAME_(WB):
				break;
			default:
				/* Bad memory type. */
				return 0;
			}
		}
		for (i = 0; i < 8; i++) {
			cpssp->NAME.fix16k[o + i] = (val >> (i*8)) & 0xff;
		}
		return 1;

	case 0x268: /* MSR_IA32_MTRR_FIX4K_C0000 */
	case 0x269: /* MSR_IA32_MTRR_FIX4K_C8000 */
	case 0x26a: /* MSR_IA32_MTRR_FIX4K_D0000 */
	case 0x26b: /* MSR_IA32_MTRR_FIX4K_D8000 */
	case 0x26c: /* MSR_IA32_MTRR_FIX4K_E0000 */
	case 0x26d: /* MSR_IA32_MTRR_FIX4K_E8000 */
	case 0x26e: /* MSR_IA32_MTRR_FIX4K_F0000 */
	case 0x26f: /* MSR_IA32_MTRR_FIX4K_F8000 */
		/* 9-25 */
		o = (ecx & 3) * 8;
		/* Check types. */
		for (i = 0; i < 8; i++) {
			switch ((val >> (i*8)) & 0xff) {
			case NAME_(UC):
			case NAME_(WC):
			case NAME_(WT):
			case NAME_(WP):
			case NAME_(WB):
				break;
			default:
				/* Bad memory type. */
				return 0;
			}
		}
		for (i = 0; i < 8; i++) {
			cpssp->NAME.fix4k[o + i] = (val >> (i*8)) & 0xff;
		}
		return 1;

	case 0x200: /* MSR_IA32_MTRR_PHYSBASE0 */
	case 0x202: /* MSR_IA32_MTRR_PHYSBASE1 */
	case 0x204: /* MSR_IA32_MTRR_PHYSBASE2 */
	case 0x206: /* MSR_IA32_MTRR_PHYSBASE3 */
	case 0x208: /* MSR_IA32_MTRR_PHYSBASE4 */
	case 0x20a: /* MSR_IA32_MTRR_PHYSBASE5 */
	case 0x20c: /* MSR_IA32_MTRR_PHYSBASE6 */
	case 0x20e: /* MSR_IA32_MTRR_PHYSBASE7 */
		/* 9-26 */
		o = (ecx >> 1) & 3;
		/* Check reserved bits. */
		if ((val >> CONFIG_CPU_PHYS_BITS)
		 || (val & 0xf00)) {
			/* Reserved bits set. */
			return 0;
		}
		/* Check memory type. */
		switch ((int) (val >> 0) & 0xff) {
		case NAME_(UC):
		case NAME_(WC):
		case NAME_(WT):
		case NAME_(WP):
		case NAME_(WB):
			break;
		default:
			/* Bad memory type. */
			return 0;
		}
		cpssp->NAME.var[o].base = val & 0xffffff000ULL;
		cpssp->NAME.var[o].type = (val >> 0) & 0xff;
		return 1;

	case 0x201: /* MSR_IA32_MTRR_PHYSMASK0 */
	case 0x203: /* MSR_IA32_MTRR_PHYSMASK1 */
	case 0x205: /* MSR_IA32_MTRR_PHYSMASK2 */
	case 0x207: /* MSR_IA32_MTRR_PHYSMASK3 */
	case 0x209: /* MSR_IA32_MTRR_PHYSMASK4 */
	case 0x20b: /* MSR_IA32_MTRR_PHYSMASK5 */
	case 0x20d: /* MSR_IA32_MTRR_PHYSMASK6 */
	case 0x20f: /* MSR_IA32_MTRR_PHYSMASK7 */
		/* 9-26 */
		o = (ecx >> 1) & 3;
		/* Check reserved bits. */
		if ((val >> CONFIG_CPU_PHYS_BITS)
		 || (val & 0x7ff)) {
			/* Reserved bits set. */
			return 0;
		}
		cpssp->NAME.var[o].mask = val & 0xffffff000ULL;
		cpssp->NAME.var[o].valid = (val >> 11) & 1;
		return 1;
	}
	return 0;
}

static int
NAME_(rdmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t *valp)
{
	int o;
	int i;

	switch (ecx) {
	case 0xfe: /* MSR_IA32_MTRRCAP */
		/* 9-23 */
		*valp = (1 << 10) /* Write combining supported. */
		      | (1 << 8)  /* Fixed range regs supported. */
		      | (8 << 0); /* 8 Variable range regs supported. */
		return 1;

	case 0x2ff: /* MSR_IA32_MTRR_DEF_TYPE */
		/* 9-24 */
		*valp = (cpssp->NAME.e << 11)
			| (cpssp->NAME.fe << 10)
			| (cpssp->NAME.type << 0);
		return 1;

	case 0x250: /* MSR_IA32_MTRR_FIX64K_00000 */
		/* 9-25 */
		o = 0;
		*valp = 0;
		for (i = 0; i < 8; i++) {
			*valp |= (uint64_t) cpssp->NAME.fix64k[o + i] << (i*8);
		}
		return 1;

	case 0x258: /* MSR_IA32_MTRR_FIX16K_80000 */
	case 0x259: /* MSR_IA32_MTRR_FIX16K_A0000 */
		/* 9-25 */
		o = (ecx & 1) * 8;
		*valp = 0;
		for (i = 0; i < 8; i++) {
			*valp |= (uint64_t) cpssp->NAME.fix16k[o + i] << (i*8);
		}
		return 1;

	case 0x268: /* MSR_IA32_MTRR_FIX4K_C0000 */
	case 0x269: /* MSR_IA32_MTRR_FIX4K_C8000 */
	case 0x26a: /* MSR_IA32_MTRR_FIX4K_D0000 */
	case 0x26b: /* MSR_IA32_MTRR_FIX4K_D8000 */
	case 0x26c: /* MSR_IA32_MTRR_FIX4K_E0000 */
	case 0x26d: /* MSR_IA32_MTRR_FIX4K_E8000 */
	case 0x26e: /* MSR_IA32_MTRR_FIX4K_F0000 */
	case 0x26f: /* MSR_IA32_MTRR_FIX4K_F8000 */
		/* 9-25 */
		o = (ecx & 3) * 8;
		*valp = 0;
		for (i = 0; i < 8; i++) {
			*valp |= (uint64_t) cpssp->NAME.fix4k[o + i] << (i*8);
		}
		return 1;

	case 0x200: /* MSR_IA32_MTRR_PHYSBASE0 */
	case 0x202: /* MSR_IA32_MTRR_PHYSBASE1 */
	case 0x204: /* MSR_IA32_MTRR_PHYSBASE2 */
	case 0x206: /* MSR_IA32_MTRR_PHYSBASE3 */
	case 0x208: /* MSR_IA32_MTRR_PHYSBASE4 */
	case 0x20a: /* MSR_IA32_MTRR_PHYSBASE5 */
	case 0x20c: /* MSR_IA32_MTRR_PHYSBASE6 */
	case 0x20e: /* MSR_IA32_MTRR_PHYSBASE7 */
		/* 9-26 */
		o = (ecx >> 1) & 3;
		*valp = (cpssp->NAME.var[o].base << (12-12))
			| (cpssp->NAME.var[o].type << 0);
		return 1;

	case 0x201: /* MSR_IA32_MTRR_PHYSMASK0 */
	case 0x203: /* MSR_IA32_MTRR_PHYSMASK1 */
	case 0x205: /* MSR_IA32_MTRR_PHYSMASK2 */
	case 0x207: /* MSR_IA32_MTRR_PHYSMASK3 */
	case 0x209: /* MSR_IA32_MTRR_PHYSMASK4 */
	case 0x20b: /* MSR_IA32_MTRR_PHYSMASK5 */
	case 0x20d: /* MSR_IA32_MTRR_PHYSMASK6 */
	case 0x20f: /* MSR_IA32_MTRR_PHYSMASK7 */
		/* 9-26 */
		o = (ecx >> 1) & 3;
		*valp = (cpssp->NAME.var[o].mask << (12-12))
			| (cpssp->NAME.var[o].valid << 11);
		return 1;
	}
	return 0;
}

static void
NAME_(n_reset_set)(struct cpssp *cpssp, unsigned int n_val)
{
	cpssp->NAME.e = 0;
	cpssp->NAME.fe = 0;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* defined CONFIG_CPU_MTRR_SUPPORT && CONFIG_CPU_MTRR_SUPPORT */
#endif /* BEHAVIOR */
