/*
 * Parts derived from QEMU sources.
 * Modified for FAUmachine by Volkmar Sieh.
 *
 *  Copyright (c) 2002-2017 FAUmachine Team.
 *  Copyright (c) 2003-2005 Fabrice Bellard.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

/* Set this to '1' to trace execution. */
#define DEBUG_CONTROL_BIOS		0
#define DEBUG_CONTROL_FLOW		0
#define DEBUG_CONTROL_FLOW_FLAGS	0
#define DEBUG_CONTROL_FLOW_REGS		0
#define DEBUG_CONTROL_FLOW_CREGS	0
#define DEBUG_CONTROL_FLOW_SREGS	0

#include "arch_gen_cpu_x86_state.h"

#ifdef STATE


// struct {
        /* temporaries if we cannot store them in host registers */
#ifndef AREG1
        target_ulong t0;
#endif
#ifndef AREG2
        target_ulong t1;
#endif
#ifndef AREG3
        target_ulong t2;
#endif

        /* standard registers */
#define R_EAX 0
#define R_ECX 1
#define R_EDX 2
#define R_EBX 3
#define R_ESP 4
#define R_EBP 5
#define R_ESI 6
#define R_EDI 7
#define R_AL 0
#define R_CL 1
#define R_DL 2
#define R_BL 3
#define R_AH 4
#define R_CH 5
#define R_DH 6
#define R_BH 7
#if 80486 <= CONFIG_CPU && CONFIG_CPU_LM_SUPPORT
#define CPU_NB_REGS 16
#else
#define CPU_NB_REGS 8
#endif
        target_ulong regs[CPU_NB_REGS];
        target_ulong eip;
#define EIP(cpssp)      (cpssp->eip)

/* Don't change these definitions! Hardcoded in real CPU! */
#define CPU_ID_SHIFT    (21)
#if 0 /* Not used */
#define CPU_VIP_SHIFT   (20)
#define CPU_VIF_SHIFT   (19)
#endif
#define CPU_AC_SHIFT    (18)
#define CPU_VM_SHIFT    (17)
#define CPU_RF_SHIFT    (16)
	/* Bit 15 not used (0). */
#define CPU_NT_SHIFT    (14)
#define CPU_IOPL_SHIFT  (12)    /* Two bits! */
#define CPU_OF_SHIFT    (11)
#define CPU_DF_SHIFT    (10)
#define CPU_IF_SHIFT    (9)
#define CPU_TF_SHIFT    (8)
#define CPU_SF_SHIFT    (7)
#define CPU_ZF_SHIFT    (6)
	/* Bit 5 not used (0). */
#define CPU_AF_SHIFT    (4)
	/* Bit 3 not used (0). */
#define CPU_PF_SHIFT    (2)
	/* Bit 1 not used (1). */
#define CPU_CF_SHIFT    (0)

#define CPU_ID_MASK     (1 << CPU_ID_SHIFT)
#if 0 /* Not used */
#define CPU_VIP_MASK    (1 << CPU_VIP_SHIFT)
#define CPU_VIF_MASK    (1 << CPU_VIF_SHIFT)
#endif
#define CPU_AC_MASK     (1 << CPU_AC_SHIFT)
#define CPU_VM_MASK     (1 << CPU_VM_SHIFT)
#define CPU_RF_MASK     (1 << CPU_RF_SHIFT)
#define CPU_NT_MASK     (1 << CPU_NT_SHIFT)
#define CPU_IOPL_MASK   (3 << CPU_IOPL_SHIFT)
#define CPU_OF_MASK     (1 << CPU_OF_SHIFT)
#define CPU_DF_MASK     (1 << CPU_DF_SHIFT)
#define CPU_IF_MASK     (1 << CPU_IF_SHIFT)
#define CPU_TF_MASK     (1 << CPU_TF_SHIFT)
#define CPU_SF_MASK     (1 << CPU_SF_SHIFT)
#define CPU_ZF_MASK     (1 << CPU_ZF_SHIFT)
#define CPU_AF_MASK     (1 << CPU_AF_SHIFT)
#define CPU_PF_MASK     (1 << CPU_PF_SHIFT)
#define CPU_CF_MASK     (1 << CPU_CF_SHIFT)

#define CC_C    CPU_CF_MASK
#define CC_P    CPU_PF_MASK
#define CC_A    CPU_AF_MASK
#define CC_Z    CPU_ZF_MASK
#define CC_S    CPU_SF_MASK
#define CC_O    CPU_OF_MASK
	target_ulong eflags; /* eflags register. During CPU emulation, CC
                                flags and DF are set to zero because they are
                                stored elsewhere */

        /* emulator internal eflags handling */
        target_ulong cc_src;
        target_ulong cc_dst;
	enum {
		CC_OP_DYNAMIC, /* must use dynamic code to get cc_op */
		CC_OP_EFLAGS,  /* all cc are explicitely computed, CC_SRC = flags */

		CC_OP_MULB, /* modify all flags, C, O = (CC_SRC != 0) */
		CC_OP_MULW,
		CC_OP_MULL,
		CC_OP_MULQ,

		CC_OP_DIVB, /* modify all ZF flag */
		CC_OP_DIVW,
		CC_OP_DIVL,
		CC_OP_DIVQ,

		CC_OP_ADDB, /* modify all flags, CC_DST = res, CC_SRC = src1 */
		CC_OP_ADDW,
		CC_OP_ADDL,
		CC_OP_ADDQ,

		CC_OP_ADCB, /* modify all flags, CC_DST = res, CC_SRC = src1 */
		CC_OP_ADCW,
		CC_OP_ADCL,
		CC_OP_ADCQ,

		CC_OP_SUBB, /* modify all flags, CC_DST = res, CC_SRC = src1 */
		CC_OP_SUBW,
		CC_OP_SUBL,
		CC_OP_SUBQ,

		CC_OP_SBBB, /* modify all flags, CC_DST = res, CC_SRC = src1 */
		CC_OP_SBBW,
		CC_OP_SBBL,
		CC_OP_SBBQ,

		CC_OP_LOGICB, /* modify all flags, CC_DST = res */
		CC_OP_LOGICW,
		CC_OP_LOGICL,
		CC_OP_LOGICQ,

		CC_OP_INCB, /* modify all flags except, CC_DST = res, CC_SRC = C */
		CC_OP_INCW,
		CC_OP_INCL,
		CC_OP_INCQ,

		CC_OP_DECB, /* modify all flags except, CC_DST = res, CC_SRC = C  */
		CC_OP_DECW,
		CC_OP_DECL,
		CC_OP_DECQ,

		CC_OP_SHLB, /* modify all flags, CC_DST = res, CC_SRC.msb = C */
		CC_OP_SHLW,
		CC_OP_SHLL,
		CC_OP_SHLQ,

		CC_OP_SARB, /* modify all flags, CC_DST = res, CC_SRC.lsb = C */
		CC_OP_SARW,
		CC_OP_SARL,
		CC_OP_SARQ,

		CC_OP_NB,
	} cc_op;
#define CC_SRC(cpssp)   (cpssp->cc_src)
#define CC_DST(cpssp)   (cpssp->cc_dst)
#define CC_OP(cpssp)    (cpssp->cc_op)

        int32_t df; /* D flag : 1 if D = 0, -1 if D = 1 */
#define DF(cpssp)       (cpssp->df)

/* hidden flags - used internally by qemu to represent additionnal cpu
   states. Only the CPL and INHIBIT_IRQ are not redundant. We avoid
   using the IOPL_MASK, TF_MASK and VM_MASK bit position to ease oring
   with eflags. */
/* current cpl */
#define HF_CPL_SHIFT         0
/* true if hardware interrupts must be disabled for next instruction */
#define HF_INHIBIT_IRQ_SHIFT 3
/* 16 or 32 segments */
#define HF_CS32_SHIFT        4
#define HF_SS32_SHIFT        5
/* zero base for DS, ES and SS : can be '0' only in 32 bit CS segment */
#define HF_ADDSEG_SHIFT      6
/* copy of CR0.PE (protected mode) */
#define HF_PE_SHIFT          7
#define HF_TF_SHIFT          8 /* must be same as eflags */
#define HF_MP_SHIFT          9 /* the order must be MP, EM, TS */
#define HF_EM_SHIFT         10
#define HF_TS_SHIFT         11
#define HF_IOPL_SHIFT       12 /* must be same as eflags */
#define HF_LMA_SHIFT        14 /* only used on x86_64: long mode active */
#define HF_CS64_SHIFT       15 /* only used on x86_64: 64 bit code segment  */
#define HF_OSFXSR_SHIFT     16 /* CR4.OSFXSR */
#define HF_VM_SHIFT         17 /* must be same as eflags */
#define HF_HALTED_SHIFT     18 /* CPU halted */
#define HF_NE_SHIFT         19 /* copy of CR0.NE */
#define HF_WAITING_FOR_STARTUP_SHIFT 20 /* Appl. Proc. waiting for Startup */

#define HF_CPL_MASK          (3 << HF_CPL_SHIFT)
#define HF_INHIBIT_IRQ_MASK  (1 << HF_INHIBIT_IRQ_SHIFT)
#define HF_CS32_MASK         (1 << HF_CS32_SHIFT)
#define HF_SS32_MASK         (1 << HF_SS32_SHIFT)
#define HF_ADDSEG_MASK       (1 << HF_ADDSEG_SHIFT)
#define HF_PE_MASK           (1 << HF_PE_SHIFT)
#define HF_TF_MASK           (1 << HF_TF_SHIFT)
#define HF_MP_MASK           (1 << HF_MP_SHIFT)
#define HF_EM_MASK           (1 << HF_EM_SHIFT)
#define HF_TS_MASK           (1 << HF_TS_SHIFT)
#define HF_LMA_MASK          (1 << HF_LMA_SHIFT)
#define HF_CS64_MASK         (1 << HF_CS64_SHIFT)
#define HF_OSFXSR_MASK       (1 << HF_OSFXSR_SHIFT)
#define HF_HALTED_MASK       (1 << HF_HALTED_SHIFT)
#define HF_NE_MASK           (1 << HF_NE_SHIFT)
#define HF_WAITING_FOR_STARTUP_MASK (1 << HF_WAITING_FOR_STARTUP_SHIFT)
        uint32_t hflags; /* hidden flags, see HF_xxx constants */

        /* Hardware Configuration */
        unsigned int apic_cluster_id;
        unsigned int apic_arbitration_id;
        /* ... */

        /* segments */
	struct SegmentCache {
		uint32_t selector;
		target_ulong base;
		uint32_t limit;
#define DESC_G_MASK     (1 << 23)
#define DESC_B_SHIFT    22
#define DESC_B_MASK     (1 << DESC_B_SHIFT)
#define DESC_L_SHIFT    21 /* x86_64 only : 64 bit code segment */
#define DESC_L_MASK     (1 << DESC_L_SHIFT)
#define DESC_AVL_MASK   (1 << 20)
#define DESC_P_MASK     (1 << 15)
#define DESC_DPL_SHIFT  13
#define DESC_S_MASK     (1 << 12)
#define DESC_TYPE_SHIFT 8
#define DESC_A_MASK     (1 << 8)

#define DESC_CS_MASK    (1 << 11) /* 1=code segment 0=data segment */
#define DESC_C_MASK     (1 << 10) /* code: conforming */
#define DESC_R_MASK     (1 << 9)  /* code: readable */

#define DESC_E_MASK     (1 << 10) /* data: expansion direction */
#define DESC_W_MASK     (1 << 9)  /* data: writable */

#define DESC_TSS_BUSY_MASK (1 << 9)
		uint32_t flags;
#define R_ES 0
#define R_CS 1
#define R_SS 2
#define R_DS 3
#define R_FS 4
#define R_GS 5
#define CPU_NB_SEGS 6
	} segs[CPU_NB_SEGS], /* selector values */
		ldt,
		tr,
		gdt, /* only base and limit are used */
		idt; /* only base and limit are used */

        /* msr registers */
#if 80686 <= CONFIG_CPU
        uint32_t update_signature;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_SEP_SUPPORT
        uint32_t sysenter_cs; /* sysenter registers */
        uint32_t sysenter_esp;
        uint32_t sysenter_eip;
#endif

#define MSR_EFER_SCE   (1 << 0)
#define MSR_EFER_LME   (1 << 8)
#define MSR_EFER_LMA   (1 << 10)
#define MSR_EFER_NXE   (1 << 11)
#define MSR_EFER_FFXSR (1 << 14)
        uint64_t efer;
        uint64_t star;
#if 80486 <= CONFIG_CPU && CONFIG_CPU_LM_SUPPORT
#define MSR_LSTAR               0xc0000082
#define MSR_CSTAR               0xc0000083
#define MSR_FMASK               0xc0000084
#define MSR_FSBASE              0xc0000100
#define MSR_GSBASE              0xc0000101
#define MSR_KERNELGSBASE        0xc0000102
        target_ulong lstar;
        target_ulong cstar;
        target_ulong fmask;
        target_ulong kernelgsbase;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_PAT_SUPPORT
        uint8_t pat[8];
#endif
        /* All "normal" MSRs are kept in an array */
#define CPU_N_MSR	0x1000 /* Caution: must be highest! FIXME */
        struct {
                uint32_t low;
                uint32_t high;
        } msr[CPU_N_MSR];

#define STATE_POWER             0x01
#define STATE_RESET             0x10
#define STATE_N_RESET_ACK       0x20
#if 80386 <= CONFIG_CPU
#define STATE_INIT              0x40
#define STATE_N_INIT_ACK        0x80
#endif
        unsigned int state_ext;
#if 80386 <= CONFIG_CPU
        int state_startup;
        uint8_t state_startup_vec;
#endif

        /* exception/interrupt handling */
        jmp_buf jmp_env;
#define CPU_FAULT_DE            (0)     /* Divide Error */
#define CPU_FAULT_DB            (1)     /* Debug */
#define CPU_FAULT_NMI           (2)     /* NMI */
#define CPU_FAULT_BP            (3)     /* Breakpoint */
#define CPU_FAULT_OF            (4)     /* Overflow */
#define CPU_FAULT_BR            (5)     /* BOUND Range Exceeded */
#define CPU_FAULT_UD            (6)     /* Invalid Opcode */
#define CPU_FAULT_NM            (7)     /* Device Not Available */
#define CPU_FAULT_DF            (8)     /* Double fault */
/* #define CPU_FAULT_??         (9) */  /* Coprocessor Segment Overrun */
#define CPU_FAULT_TS            (10)    /* Invalid TSS */
#define CPU_FAULT_NP            (11)    /* Segment Not Present */
#define CPU_FAULT_SS            (12)    /* Stack-Segment Fault */
#define CPU_FAULT_GP            (13)    /* General Protection */
#define CPU_FAULT_PF            (14)    /* Page Fault */
/* #define CPU_FAULT_??         (15) */ /* Intel reserved */
#define CPU_FAULT_MF            (16)    /* Floating-Point Error (Math Fault) */
#define CPU_FAULT_AC            (17)    /* Alignment Check */
#define CPU_FAULT_MC            (18)    /* Machine Check */
#define CPU_FAULT_XF            (19)    /* Streaming SIMD Extensions */
        int exception_index;
        int error_code;
        int exception_is_int;
        target_ulong exception_next_eip;

/*
 * %cr0 Bits
 */
#define CPU_CR0_PE_SHIFT        (0)     /* Protection Enable */
#define CPU_CR0_MP_SHIFT        (1)     /* Monitor Coprocessor */
#define CPU_CR0_EM_SHIFT        (2)     /* Emulation */
#define CPU_CR0_TS_SHIFT        (3)     /* Task Switched */
#define CPU_CR0_ET_SHIFT        (4)     /* Extension Type */
#define CPU_CR0_NE_SHIFT        (5)     /* Numeric Error */
#define CPU_CR0_WP_SHIFT        (16)    /* Write Protect */
#define CPU_CR0_AM_SHIFT        (18)    /* Alignment Mask */
#define CPU_CR0_NW_SHIFT        (29)    /* Not Write-through */
#define CPU_CR0_CD_SHIFT        (30)    /* Cache Disable */
#define CPU_CR0_PG_SHIFT        (31)    /* Paging */

#define CPU_CR0_PE_MASK         (1 << CPU_CR0_PE_SHIFT)
#define CPU_CR0_MP_MASK         (1 << CPU_CR0_MP_SHIFT)
#define CPU_CR0_EM_MASK         (1 << CPU_CR0_EM_SHIFT)
#define CPU_CR0_TS_MASK         (1 << CPU_CR0_TS_SHIFT)
#define CPU_CR0_ET_MASK         (1 << CPU_CR0_ET_SHIFT)
#define CPU_CR0_NE_MASK         (1 << CPU_CR0_NE_SHIFT)
#define CPU_CR0_WP_MASK         (1 << CPU_CR0_WP_SHIFT)
#define CPU_CR0_AM_MASK         (1 << CPU_CR0_AM_SHIFT)
#define CPU_CR0_NW_MASK         (1 << CPU_CR0_NW_SHIFT)
#define CPU_CR0_CD_MASK         (1 << CPU_CR0_CD_SHIFT)
#define CPU_CR0_PG_MASK         (1 << CPU_CR0_PG_SHIFT)

/*
 * %cr3 Bits
 */
#define CPU_PWT_SHIFT   (3)     /* Page-level Writes Transparent */
#define CPU_PCD_SHIFT   (4)     /* Page-level Cache Disable */

#define CPU_PWT_MASK    (1 << CPU_PWT_SHIFT)
#define CPU_PCD_MASK    (1 << CPU_PCD_SHIFT)

/*
 * %cr4 Bits
 */
#define CPU_CR4_VME_SHIFT       (0)     /* Virtual-8086 Mode Extensions */
#define CPU_CR4_PVI_SHIFT       (1) /* Protected-Mode Virtual Interrupts */
#define CPU_CR4_TSD_SHIFT       (2) /* Time Stamp Disable */
#define CPU_CR4_DE_SHIFT        (3) /* Debugging Extensions */
#define CPU_CR4_PSE_SHIFT       (4) /* Page Size Extensions */
#define CPU_CR4_PAE_SHIFT       (5) /* Physical Address Extension */
#define CPU_CR4_MCE_SHIFT       (6) /* Machine Check Enable */
#define CPU_CR4_PGE_SHIFT       (7) /* Page Global Enable */
#define CPU_CR4_PCE_SHIFT       (8) /* Performance-Monitoring Counter Enable */
#define CPU_CR4_OSFXSR_SHIFT    (9) /* OS Support for FXSAVE and FXRSTOR */
#define CPU_CR4_OSXMMEXCPT_SHIFT (10) /* OS Support for Unmasked SIMD Exceptions */

#define CPU_CR4_VME_MASK        (1 << CPU_CR4_VME_SHIFT)
#define CPU_CR4_PVI_MASK        (1 << CPU_CR4_PVI_SHIFT)
#define CPU_CR4_TSD_MASK        (1 << CPU_CR4_TSD_SHIFT)
#define CPU_CR4_DE_MASK         (1 << CPU_CR4_DE_SHIFT)
#define CPU_CR4_PSE_MASK        (1 << CPU_CR4_PSE_SHIFT)
#define CPU_CR4_PAE_MASK        (1 << CPU_CR4_PAE_SHIFT)
#define CPU_CR4_MCE_MASK        (1 << CPU_CR4_MCE_SHIFT)
#define CPU_CR4_PGE_MASK        (1 << CPU_CR4_PGE_SHIFT)
#define CPU_CR4_PCE_MASK        (1 << CPU_CR4_PCE_SHIFT)
#define CPU_CR4_OSFXSR_MASK     (1 << CPU_CR4_OSFXSR_SHIFT)
#define CPU_CR4_OSXMMEXCPT_MASK (1 << CPU_CR4_OSXMMEXCPT_SHIFT)
        target_ulong cr[5]; /* NOTE: cr1 is unused */

        target_ulong dr[8]; /* debug registers */
#define CPU_INTERRUPT_IRQ       0x1 /* normal interrupt pending */
#define CPU_INTERRUPT_NMI       0x2 /* non-maskable interrupt pending */
#define CPU_INTERRUPT_SMI       0x4 /* system management interrupt pending */
        int interrupt_request;

        /* soft mmu support */
        /* in order to avoid passing too many arguments to the memory
           write helpers, we store some rarely used information in the CPU
           context) */
        unsigned long mem_write_pc; /* host pc at which the memory was
                                        written */
        /*
         * Code Generation
         */
        /* XXX: make safe guess about sizes */
#define MAX_OP_PER_INSTR 32
#define OPC_BUF_SIZE 512
#define OPC_MAX_SIZE (OPC_BUF_SIZE - MAX_OP_PER_INSTR)
#define OPPARAM_BUF_SIZE (OPC_BUF_SIZE * 3)

        struct TranslationBlock *current_tb; /* currently executing TB */

        uint16_t *gen_opc_ptr;
        uint32_t *gen_opparam_ptr;

        uint16_t gen_opc_buf[OPC_BUF_SIZE];
        uint32_t gen_opparam_buf[OPPARAM_BUF_SIZE];
        target_ulong gen_opc_pc[OPC_BUF_SIZE];
        uint8_t gen_opc_instr_start[OPC_BUF_SIZE];
        uint8_t gen_opc_cc_op[OPC_BUF_SIZE];

        long gen_labels[OPC_BUF_SIZE];
        int nb_gen_labels;

        struct TranslationBlock *tb_jmp_cache[TB_JMP_CACHE_SIZE];

/*
 * maximum total translate dcode allocated:
 * NOTE: the translated code area cannot be too big because on some
 * archs the range of "fast" function calls is limited. Here is a
 * summary of the ranges:
 *
 * i386  : signed 32 bits
 * arm   : signed 26 bits
 * ppc   : signed 24 bits
* sparc : signed 32 bits
* alpha : signed 23 bits
*/
#if defined(__alpha__)
#define CODE_GEN_BUFFER_SIZE    (2 * 1024 * 1024)
#elif defined(__ia64)
#define CODE_GEN_BUFFER_SIZE    (4 * 1024 * 1024)       /* range of addl */
#elif defined(__powerpc__)
#define CODE_GEN_BUFFER_SIZE    (6 * 1024 * 1024)
#else
#define CODE_GEN_BUFFER_SIZE    (8 * 1024 * 1024) /* FIXME? fox - current QEMU h
as 16 */
#endif
        uint8_t code_gen_buffer[CODE_GEN_BUFFER_SIZE]
                        __attribute__((aligned(32)));
        uint8_t *code_gen_ptr;

#define L2_BITS 10
#define L1_BITS (32 - L2_BITS - TARGET_PAGE_BITS)

#define L1_SIZE (1 << L1_BITS)
#define L2_SIZE (1 << L2_BITS)
        PageDesc *l1_map[L1_SIZE];

        TranslationBlock tbs[CODE_GEN_MAX_BLOCKS];
        int nb_tbs;

        int tb_invalidated_flag;

        TranslationBlock *tb_phys_hash[CODE_GEN_PHYS_HASH_SIZE];

        unsigned long link_tb;          /* address and number (1..0) of
                                           block to link with */

#define MAX_CODE_BITMAPS        128
#define CODE_BITMAP_SIZE        (TARGET_PAGE_SIZE / 8)

        /* Faults */
        struct {
                int used;
                unsigned int regno;
                unsigned int bitno;
                void *sig;
        } fault[16];

	int id;

	struct {
		/* System Management Mode Base Address (SMBASE) */
		uint32_t smbase;
		int smm; /* whether CPU is in system management mode (SMM) */
	} core;
// } NAME;

#endif /* STATE */
#ifdef EXPORT

#if 80486 <= CONFIG_CPU
/*forward*/ static int
NAME_(core_cr0_wp_get)(struct cpssp *cpssp);
#endif
/*forward*/ static int
NAME_(core_cr0_pg_get)(struct cpssp *cpssp);
/*forward*/ static paddr_t
NAME_(core_cr3_get)(struct cpssp *cpssp);
#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
/*forward*/ static int
NAME_(core_cr4_pae_get)(struct cpssp *cpssp);
#endif
#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
/*forward*/ static int
NAME_(core_cr4_pse_get)(struct cpssp *cpssp);
#endif
#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
/*forward*/ static int
NAME_(core_cr4_pge_get)(struct cpssp *cpssp);
#endif
#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
/*forward*/ static int
NAME_(core_efer_lme_get)(struct cpssp *cpssp);
#endif
#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
/*forward*/ static int
NAME_(core_efer_nxe_get)(struct cpssp *cpssp);
#endif
/*forward*/ static void __attribute__((__noreturn__))
NAME_(step)(void *_cpssp);
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
/*forward*/ static void
NAME_(core_startup)(struct cpssp *cpssp, uint8_t vector);
/*forward*/ static void
NAME_(core_init_by_apic)(struct cpssp *cpssp);
#endif
#if 80486 <= CONFIG_CPU
/*forward*/ static void
NAME_(core_n_init_set)(struct cpssp *cpssp, unsigned int n_val);
#endif /* 80386 <= CONFIG_CPU */
/*forward*/ static void
NAME_(core_nmi_set)(struct cpssp *cpssp, unsigned int val);
#if 0 /* Not used. */
/*forward*/ static void
NAME_(core_nmi_ack)(struct cpssp *cpssp);
#endif
/*forward*/ static void
NAME_(core_irq_set)(struct cpssp *cpssp, unsigned int val);
#if 80386 <= CONFIG_CPU
/*forward*/ static void
NAME_(core_smi_set)(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
NAME_(core_smi_ack)(struct cpssp *cpssp);
#endif
/*forward*/ static void
NAME_(core_n_ignne_set)(struct cpssp *cpssp, unsigned int n_val);
/*forward*/ static void
NAME_(core_n_reset_set)(struct cpssp *cpssp, unsigned int n_val);
/*forward*/ static void
NAME_(core_power_set)(struct cpssp *cpssp, unsigned int val);

#endif /* EXPORT */
#ifdef BEHAVIOR

/*
 * Macros
 */
/* XXX: add a is_user flag to have proper security support */
#define PUSHW(cpssp, ssp, sp, sp_mask, val)\
{\
	sp -= 2;\
	stw_kernel(cpssp, (ssp) + (sp & (sp_mask)), (val));\
}

#define PUSHL(cpssp, ssp, sp, sp_mask, val)\
{\
	sp -= 4;\
	stl_kernel(cpssp, (ssp) + (sp & (sp_mask)), (val));\
}

#define POPW(cpssp, ssp, sp, sp_mask, val)\
{\
	val = lduw_kernel(cpssp, (ssp) + (sp & (sp_mask)));\
	sp += 2;\
}

#define POPL(cpssp, ssp, sp, sp_mask, val)\
{\
	val = (uint32_t)ldl_kernel(cpssp, (ssp) + (sp & (sp_mask)));\
	sp += 4;\
}

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT

#define PUSHQ(cpssp, sp, val)\
{\
	sp -= 8;\
	stq_kernel(cpssp, sp, (val));\
}

#define POPQ(cpssp, sp, val)\
{\
	val = ldq_kernel(cpssp, sp);\
	sp += 8;\
}

#endif /* CONFIG_CPU_LM_SUPPORT */

#if DEBUG_CONTROL_BIOS || DEBUG_CONTROL_FLOW
static void
NAME_(dump)(struct cpssp *cpssp)
{
#if DEBUG_CONTROL_BIOS
	fprintf(stderr, " AX=%04lx BX=%04lx CX=%04lx DX=%04lx BP=%04lx SP=%04lx DI=%04lx SI=%04lx",
			(unsigned long) cpssp->regs[R_EAX] & 0xffff,
			(unsigned long) cpssp->regs[R_EBX] & 0xffff,
			(unsigned long) cpssp->regs[R_ECX] & 0xffff,
			(unsigned long) cpssp->regs[R_EDX] & 0xffff,
			(unsigned long) cpssp->regs[R_EBP] & 0xffff,
			(unsigned long) cpssp->regs[R_ESP] & 0xffff,
			(unsigned long) cpssp->regs[R_EDI] & 0xffff,
			(unsigned long) cpssp->regs[R_ESI] & 0xffff);
	fprintf(stderr, " CS=%04x SS=%04x DS=%04x ES=%04x",
			(unsigned int) cpssp->segs[R_CS].selector,
			(unsigned int) cpssp->segs[R_SS].selector,
			(unsigned int) cpssp->segs[R_DS].selector,
			(unsigned int) cpssp->segs[R_ES].selector);
#endif
#if DEBUG_CONTROL_FLOW_REGS
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
		fprintf(stderr, " %016lx %016lx %016lx %016lx %016lx %016lx %016lx %016lx %016lx %016lx %016lx %016lx %016lx %016lx %016lx %016lx",
				(unsigned long) cpssp->regs[R_EAX], (unsigned long) cpssp->regs[R_EBX],
				(unsigned long) cpssp->regs[R_ECX], (unsigned long) cpssp->regs[R_EDX],
				(unsigned long) cpssp->regs[R_EBP], (unsigned long) cpssp->regs[R_ESP],
				(unsigned long) cpssp->regs[R_EDI], (unsigned long) cpssp->regs[R_ESI],
				(unsigned long) cpssp->regs[8], (unsigned long) cpssp->regs[9],
				(unsigned long) cpssp->regs[10], (unsigned long) cpssp->regs[11],
				(unsigned long) cpssp->regs[12], (unsigned long) cpssp->regs[13],
				(unsigned long) cpssp->regs[14], (unsigned long) cpssp->regs[15]);
	} else
#endif
	{
		fprintf(stderr, " %08lx %08lx %08lx %08lx %08lx %08lx %08lx %08lx",
				(unsigned long) cpssp->regs[R_EAX], (unsigned long) cpssp->regs[R_EBX],
				(unsigned long) cpssp->regs[R_ECX], (unsigned long) cpssp->regs[R_EDX],
				(unsigned long) cpssp->regs[R_EBP], (unsigned long) cpssp->regs[R_ESP],
				(unsigned long) cpssp->regs[R_EDI], (unsigned long) cpssp->regs[R_ESI]);
	}
#endif
#if DEBUG_CONTROL_FLOW_CREGS
	fprintf(stderr, " %08lx %08lx %08lx %08lx",
			(unsigned int) cpssp->cr[0],
			(unsigned int) cpssp->cr[2],
			(unsigned int) cpssp->cr[3],
			(unsigned int) cpssp->cr[4]);
#if 0
	fprintf(stderr, " %08lx",
			(unsigned int) cpssp->cr[8]);
#endif
#endif
#if DEBUG_CONTROL_FLOW_SREGS
	fprintf(stderr, " %04x %04x %04x %04x %04x %04x",
			(unsigned int) cpssp->segs[R_CS].selector,
			(unsigned int) cpssp->segs[R_SS].selector,
			(unsigned int) cpssp->segs[R_DS].selector,
			(unsigned int) cpssp->segs[R_ES].selector,
			(unsigned int) cpssp->segs[R_FS].selector,
			(unsigned int) cpssp->segs[R_GS].selector);
	fprintf(stderr, " %08x %08x %08x %08x %08x %08x",
			(unsigned int) cpssp->segs[R_CS].base,
			(unsigned int) cpssp->segs[R_SS].base,
			(unsigned int) cpssp->segs[R_DS].base,
			(unsigned int) cpssp->segs[R_ES].base,
			(unsigned int) cpssp->segs[R_FS].base,
			(unsigned int) cpssp->segs[R_GS].base);
#endif
#if DEBUG_CONTROL_FLOW_FLAGS
	fprintf(stderr, " %08lx", compute_eflags(cpssp));
#endif
	fprintf(stderr, "\n");
}
#endif /* DEBUG_CONTROL_BIOS || DEBUG_CONTROL_FLOW */

static void
NAME_(update_cr0)(struct cpssp *cpssp, uint32_t new_cr0)
{
	int pe_state;

	if ((new_cr0 & (CPU_CR0_PG_MASK | CPU_CR0_WP_MASK | CPU_CR0_PE_MASK))
	 != (cpssp->cr[0] & (CPU_CR0_PG_MASK | CPU_CR0_WP_MASK | CPU_CR0_PE_MASK))) {
		NAME_(core_mmu_tlb_flush)(cpssp, 1);
	}
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	if (!(cpssp->cr[0] & CPU_CR0_PG_MASK) && (new_cr0 & CPU_CR0_PG_MASK)
	 && (cpssp->efer & MSR_EFER_LME)) {
		/* enter in long mode */
		/* XXX: generate an exception */
		if (!(cpssp->cr[4] & CPU_CR4_PAE_MASK))
			return;
		cpssp->efer |= MSR_EFER_LMA;
		cpssp->hflags |= HF_LMA_MASK;
	} else if ((cpssp->cr[0] & CPU_CR0_PG_MASK) && !(new_cr0 & CPU_CR0_PG_MASK)
		&& (cpssp->efer & MSR_EFER_LMA)) {
		/* exit long mode */
		cpssp->efer &= ~MSR_EFER_LMA;
		cpssp->hflags &= ~(HF_LMA_MASK | HF_CS64_MASK);
		cpssp->eip &= 0xffffffff;
	}
#endif
	cpssp->cr[0] = new_cr0 | CPU_CR0_ET_MASK;

	/* update PE flag in hidden flags */
	pe_state = (cpssp->cr[0] & CPU_CR0_PE_MASK);
	cpssp->hflags = (cpssp->hflags & ~HF_PE_MASK)
		| (pe_state << HF_PE_SHIFT);
	/* ensure that ADDSEG is always set in real mode */
	cpssp->hflags |= ((pe_state ^ 1) << HF_ADDSEG_SHIFT);
	/* update FPU flags */
	cpssp->hflags = (cpssp->hflags & ~(HF_MP_MASK | HF_EM_MASK | HF_TS_MASK))
		| ((new_cr0 << (HF_MP_SHIFT - 1)) & (HF_MP_MASK | HF_EM_MASK | HF_TS_MASK));
	if(new_cr0 & CPU_CR0_NE_MASK) {
		cpssp->hflags |= HF_NE_MASK;
	}
}

/* XXX: in legacy PAE mode, generate a GPF if reserved bits are set in
   the PDPT */
static void
NAME_(update_cr3)(struct cpssp *cpssp, target_ulong new_cr3)
{
	cpssp->cr[3] = new_cr3;
	if (cpssp->cr[0] & CPU_CR0_PG_MASK) {
		NAME_(core_mmu_tlb_flush)(cpssp, 1);
	}
}

#if 80486 <= CONFIG_CPU
static int
NAME_(core_cr0_wp_get)(struct cpssp *cpssp)
{
	return (cpssp->cr[0] >> CPU_CR0_WP_SHIFT) & 1;
}
#endif /* 80486 <= CONFIG_CPU */

static int
NAME_(core_cr0_pg_get)(struct cpssp *cpssp)
{
	return (cpssp->cr[0] >> CPU_CR0_PG_SHIFT) & 1;
}

static paddr_t
NAME_(core_cr3_get)(struct cpssp *cpssp)
{
	return cpssp->cr[3];
}

#if defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT
static int
NAME_(core_cr4_pae_get)(struct cpssp *cpssp)
{
	return (cpssp->cr[4] >> CPU_CR4_PAE_SHIFT) & 1;
}
#endif /* defined CONFIG_CPU_PAE_SUPPORT && CONFIG_CPU_PAE_SUPPORT */

#if defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT
static int
NAME_(core_cr4_pse_get)(struct cpssp *cpssp)
{
	return (cpssp->cr[4] >> CPU_CR4_PSE_SHIFT) & 1;
}
#endif /* defined CONFIG_CPU_PSE_SUPPORT && CONFIG_CPU_PSE_SUPPORT */

#if defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT
static int
NAME_(core_cr4_pge_get)(struct cpssp *cpssp)
{
	return (cpssp->cr[4] >> CPU_CR4_PGE_SHIFT) & 1;
}
#endif /* defined CONFIG_CPU_PGE_SUPPORT && CONFIG_CPU_PGE_SUPPORT */

#if defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORT
static int
NAME_(core_efer_lme_get)(struct cpssp *cpssp)
{
	return (cpssp->hflags >> HF_LMA_SHIFT) & 1;

}
#endif /* defined CONFIG_CPU_LM_SUPPORT && CONFIG_CPU_LM_SUPPORTS */

#if defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT
static int
NAME_(core_efer_nxe_get)(struct cpssp *cpssp)
{
	return (cpssp->efer & MSR_EFER_NXE) ? 1 : 0;
}
#endif /* defined CONFIG_CPU_NX_SUPPORT && CONFIG_CPU_NX_SUPPORT */

static void
NAME_(update_cr4)(struct cpssp *cpssp, uint32_t new_cr4)
{
	if ((new_cr4 & (CPU_CR4_PGE_MASK | CPU_CR4_PAE_MASK | CPU_CR4_PSE_MASK))
	 != (cpssp->cr[4] & (CPU_CR4_PGE_MASK | CPU_CR4_PAE_MASK | CPU_CR4_PSE_MASK))) {
		NAME_(core_mmu_tlb_flush)(cpssp, 1);
	}
	/* SSE handling */
	if (
#ifdef CONFIG_CPU_SSE_SUPPORT
	    ! CONFIG_CPU_SSE_SUPPORT
#else
	    1
#endif
	) {
		new_cr4 &= ~CPU_CR4_OSFXSR_MASK;
	} else if (new_cr4 & CPU_CR4_OSFXSR_MASK) {
		cpssp->hflags |= HF_OSFXSR_MASK;
	} else {
		cpssp->hflags &= ~HF_OSFXSR_MASK;
	}
	cpssp->cr[4] = new_cr4;
}

void
NAME_(helper_movl_crN_T0)(struct cpssp *cpssp, int reg, target_ulong t0)
{
	switch(reg) {
	case 0:
		NAME_(update_cr0)(cpssp, t0);
		break;
	case 3:
		NAME_(update_cr3)(cpssp, t0);
		break;
	case 4:
		NAME_(update_cr4)(cpssp, t0);
		break;
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	case 8:
		NAME_(core_apic_tpr_set)(cpssp, t0);
		break;
#endif
	default:
		cpssp->cr[reg] = t0;
		break;
	}
}

/* XXX: do more */
void
NAME_(helper_movl_drN_T0)(struct cpssp *cpssp, int reg, target_ulong t0)
{
	cpssp->dr[reg] = t0;
}

#if CONFIG_CPU <= 80386
#undef CPU_ID_MASK
#define CPU_ID_MASK	(0 << CPU_ID_SHIFT)
#undef CPU_AC_MASK
#define CPU_AC_MASK	(0 << CPU_AC_SHIFT)
#endif

const uint8_t NAME_(parity_table)[256] = {
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	CC_P, 0, 0, CC_P, 0, CC_P, CC_P, 0,
	0, CC_P, CC_P, 0, CC_P, 0, 0, CC_P,
};

void
NAME_(lock)(struct cpssp *cpssp)
{
}

void
NAME_(unlock)(struct cpssp *cpssp)
{
}

/*
 * This function must always be used to load data in the segment
 * cache: it synchronizes the hflags with the segment cache values.
 */
static void
cpu_x86_load_seg_cache(
	struct cpssp *cpssp,
	int seg_reg,
	unsigned int selector,
	target_ulong base,
	unsigned int limit,
	unsigned int flags
)
{
	struct SegmentCache *sc;
	unsigned int new_hflags;

	sc = &cpssp->segs[seg_reg];
	sc->selector = selector;
	sc->base = base;
	sc->limit = limit;
	sc->flags = flags;

	/* Update the hidden flags. */
	if (seg_reg == R_CS) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
		if ((cpssp->hflags & HF_LMA_MASK) && (flags & DESC_L_MASK)) {
			/* long mode */
			cpssp->hflags |= HF_CS32_MASK | HF_SS32_MASK | HF_CS64_MASK;
			cpssp->hflags &= ~(HF_ADDSEG_MASK);
		} else 
#endif
		{
			/* legacy / compatibility case */
			new_hflags = (cpssp->segs[R_CS].flags & DESC_B_MASK)
			             >> (DESC_B_SHIFT - HF_CS32_SHIFT);
			cpssp->hflags = (cpssp->hflags & ~(HF_CS32_MASK | HF_CS64_MASK))
			              | new_hflags;
		}
	}
	new_hflags = (cpssp->segs[R_SS].flags & DESC_B_MASK)
		     >> (DESC_B_SHIFT - HF_SS32_SHIFT);
	if (cpssp->hflags & HF_CS64_MASK) {
		/* zero base assumed for DS, ES and SS in long mode */
	} else if (! (cpssp->cr[0] & CPU_CR0_PE_MASK)
	        || (cpssp->eflags & CPU_VM_MASK)
	        || ! (new_hflags & HF_CS32_MASK)) {
		/*
		 * FIXME: try to avoid this test. The problem comes from the
		 * fact that is real mode or vm86 mode we only modify the
		 * 'base' and 'selector' fields of the segment cache to go
		 * faster. A solution may be to force addseg to one in
		 * translate-i386.c.
		 */
		new_hflags |= HF_ADDSEG_MASK;
	} else {
		new_hflags |= ((cpssp->segs[R_DS].base
			       | cpssp->segs[R_ES].base
			       | cpssp->segs[R_SS].base) != 0)
			      << HF_ADDSEG_SHIFT;
	}
	cpssp->hflags = (cpssp->hflags & ~(HF_SS32_MASK | HF_ADDSEG_MASK))
		      | new_hflags;
}

/*
 * Wrapper, just in case memory mappings must be changed.
 */
static inline void
cpu_x86_set_cpl(struct cpssp *cpssp, int cpl)
{
#if HF_CPL_MASK == 3
	cpssp->hflags = (cpssp->hflags & ~HF_CPL_MASK) | cpl;
#else
#error HF_CPL_MASK is hardcoded
#endif
}

/* return non zero if error */
static inline int
load_segment(struct cpssp *cpssp, uint32_t *e1_ptr, uint32_t *e2_ptr, int selector)
{
	struct SegmentCache *dt;
	int index_;
	target_ulong ptr;

	if (selector & 0x4)
		dt = &cpssp->ldt;
	else
		dt = &cpssp->gdt;
	index_ = selector & ~7;
	if ((index_ + 7) > dt->limit)
		return -1;
	ptr = dt->base + index_;
	*e1_ptr = ldl_kernel(cpssp, ptr);
	*e2_ptr = ldl_kernel(cpssp, ptr + 4);
	return 0;
}

static inline unsigned int
get_seg_limit(uint32_t e1, uint32_t e2)
{
	unsigned int limit;
	limit = (e1 & 0xffff) | (e2 & 0x000f0000);
	if (e2 & DESC_G_MASK)
		limit = (limit << 12) | 0xfff;
	return limit;
}

static inline uint32_t
get_seg_base(uint32_t e1, uint32_t e2)
{
	return ((e1 >> 16) | ((e2 & 0xff) << 16) | (e2 & 0xff000000));
}

static inline void
load_seg_cache_raw_dt(struct SegmentCache *sc, uint32_t e1, uint32_t e2)
{
    sc->base = get_seg_base(e1, e2);
    sc->limit = get_seg_limit(e1, e2);
    sc->flags = e2;
}

/* init the segment cache in vm86 mode. */
static inline void
load_seg_vm(struct cpssp *cpssp, int seg, int selector)
{
	selector &= 0xffff;
	cpu_x86_load_seg_cache(cpssp, seg, selector, (selector << 4), 0xffff, 0);
}

/* XXX: merge with load_seg() */
static void
tss_load_seg(struct cpssp *cpssp, int seg_reg, int selector)
{
    uint32_t e1, e2;
    int rpl, dpl, cpl;

    if ((selector & 0xfffc) != 0) {
	if (load_segment(cpssp, &e1, &e2, selector) != 0)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
	if (!(e2 & DESC_S_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
	rpl = selector & 3;
	dpl = (e2 >> DESC_DPL_SHIFT) & 3;
	cpl = cpssp->hflags & HF_CPL_MASK;
	if (seg_reg == R_CS) {
	    if (!(e2 & DESC_CS_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
	    /* XXX: is it correct ? */
	    if (dpl != rpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
	    if ((e2 & DESC_C_MASK) && dpl > rpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
	} else if (seg_reg == R_SS) {
	    /* SS must be writable data */
	    if ((e2 & DESC_CS_MASK) || !(e2 & DESC_W_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
	    if (dpl != cpl || dpl != rpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
	} else {
	    /* not readable code */
	    if ((e2 & DESC_CS_MASK) && !(e2 & DESC_R_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
	    /* if data or non conforming code, checks the rights */
	    if (((e2 >> DESC_TYPE_SHIFT) & 0xf) < 12) {
		if (dpl < cpl || dpl < rpl)
		    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
	    }
	}
	if (!(e2 & DESC_P_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, selector & 0xfffc);
	cpu_x86_load_seg_cache(cpssp, seg_reg, selector,
		       get_seg_base(e1, e2),
		       get_seg_limit(e1, e2),
		       e2);
    } else {
	if (seg_reg == R_SS || seg_reg == R_CS)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, selector & 0xfffc);
    }
}

#define SWITCH_TSS_JMP  0
#define SWITCH_TSS_IRET 1
#define SWITCH_TSS_CALL 2

/* XXX: restore CPU state in registers (PowerPC case) */
void
NAME_(switch_tss)(
	struct cpssp *cpssp,
	int tss_selector,
	uint32_t e1,
	uint32_t e2,
	int source,
	uint32_t next_eip
)
{
    int tss_limit, tss_limit_max, type, old_tss_limit_max, old_type, v1, v2, i;
    target_ulong tss_base;
    uint32_t new_regs[8], new_segs[6];
    uint32_t new_eflags, new_eip, new_cr3, new_ldt, new_trap;
    uint32_t old_eflags, eflags_mask;
    struct SegmentCache *dt;
    int index_;
    target_ulong ptr;

    type = (e2 >> DESC_TYPE_SHIFT) & 0xf;

    /* if task gate, we read the TSS segment and we load it */
    if (type == 5) {
	if (!(e2 & DESC_P_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, tss_selector & 0xfffc);
	tss_selector = e1 >> 16;
	if (tss_selector & 4)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, tss_selector & 0xfffc);
	if (load_segment(cpssp, &e1, &e2, tss_selector) != 0)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, tss_selector & 0xfffc);
	if (e2 & DESC_S_MASK)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, tss_selector & 0xfffc);
	type = (e2 >> DESC_TYPE_SHIFT) & 0xf;
	if ((type & 7) != 1)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, tss_selector & 0xfffc);
    }

    if (!(e2 & DESC_P_MASK))
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, tss_selector & 0xfffc);

    if (type & 8)
	tss_limit_max = 103;
    else
	tss_limit_max = 43;
    tss_limit = get_seg_limit(e1, e2);
    tss_base = get_seg_base(e1, e2);
    if ((tss_selector & 4) != 0 ||
	tss_limit < tss_limit_max)
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, tss_selector & 0xfffc);
    old_type = (cpssp->tr.flags >> DESC_TYPE_SHIFT) & 0xf;
    if (old_type & 8)
	old_tss_limit_max = 103;
    else
	old_tss_limit_max = 43;

    /* read all the registers from the new TSS */
    if (type & 8) {
	/* 32 bit */
	new_cr3 = ldl_kernel(cpssp, tss_base + 0x1c);
	new_eip = ldl_kernel(cpssp, tss_base + 0x20);
	new_eflags = ldl_kernel(cpssp, tss_base + 0x24);
	for(i = 0; i < 8; i++)
	    new_regs[i] = ldl_kernel(cpssp, tss_base + (0x28 + i * 4));
	for(i = 0; i < 6; i++)
	    new_segs[i] = lduw_kernel(cpssp, tss_base + (0x48 + i * 4));
	new_ldt = lduw_kernel(cpssp, tss_base + 0x60);
	new_trap = ldl_kernel(cpssp, tss_base + 0x64);
    } else {
	/* 16 bit */
	new_cr3 = 0;
	new_eip = lduw_kernel(cpssp, tss_base + 0x0e);
	new_eflags = lduw_kernel(cpssp, tss_base + 0x10);
	for(i = 0; i < 8; i++)
	    new_regs[i] = lduw_kernel(cpssp, tss_base + (0x12 + i * 2)) | 0xffff0000;
	for(i = 0; i < 4; i++)
	    new_segs[i] = lduw_kernel(cpssp, tss_base + (0x22 + i * 2));
	new_ldt = lduw_kernel(cpssp, tss_base + 0x2a);
	new_segs[R_FS] = 0;
	new_segs[R_GS] = 0;
	new_trap = 0;
    }

    /* NOTE: we must avoid memory exceptions during the task switch,
       so we make dummy accesses before */
    /* XXX: it can still fail in some cases, so a bigger hack is
       necessary to valid the TLB after having done the accesses */

    v1 = ldub_kernel(cpssp, cpssp->tr.base);
    v2 = ldub_kernel(cpssp, cpssp->tr.base + old_tss_limit_max);
    stb_kernel(cpssp, cpssp->tr.base, v1);
    stb_kernel(cpssp, cpssp->tr.base + old_tss_limit_max, v2);

    /* clear busy bit (it is restartable) */
    if (source == SWITCH_TSS_JMP || source == SWITCH_TSS_IRET) {
	target_ulong ptr_;
	uint32_t e2_;
	ptr_ = cpssp->gdt.base + (cpssp->tr.selector & ~7);
	e2_ = ldl_kernel(cpssp, ptr_ + 4);
	e2_ &= ~DESC_TSS_BUSY_MASK;
	stl_kernel(cpssp, ptr_ + 4, e2_);
    }
    old_eflags = compute_eflags(cpssp);
    if (source == SWITCH_TSS_IRET)
	old_eflags &= ~CPU_NT_MASK;

    /* save the current state in the old TSS */
    if (type & 8) {
	/* 32 bit */
	stl_kernel(cpssp, cpssp->tr.base + 0x20, next_eip);
	stl_kernel(cpssp, cpssp->tr.base + 0x24, old_eflags);
	stl_kernel(cpssp, cpssp->tr.base + (0x28 + 0 * 4), cpssp->regs[R_EAX]);
	stl_kernel(cpssp, cpssp->tr.base + (0x28 + 1 * 4), cpssp->regs[R_ECX]);
	stl_kernel(cpssp, cpssp->tr.base + (0x28 + 2 * 4), cpssp->regs[R_EDX]);
	stl_kernel(cpssp, cpssp->tr.base + (0x28 + 3 * 4), cpssp->regs[R_EBX]);
	stl_kernel(cpssp, cpssp->tr.base + (0x28 + 4 * 4), cpssp->regs[R_ESP]);
	stl_kernel(cpssp, cpssp->tr.base + (0x28 + 5 * 4), cpssp->regs[R_EBP]);
	stl_kernel(cpssp, cpssp->tr.base + (0x28 + 6 * 4), cpssp->regs[R_ESI]);
	stl_kernel(cpssp, cpssp->tr.base + (0x28 + 7 * 4), cpssp->regs[R_EDI]);
	for(i = 0; i < 6; i++)
	    stw_kernel(cpssp, cpssp->tr.base + (0x48 + i * 4), cpssp->segs[i].selector);
    } else {
	/* 16 bit */
	stw_kernel(cpssp, cpssp->tr.base + 0x0e, next_eip);
	stw_kernel(cpssp, cpssp->tr.base + 0x10, old_eflags);
	stw_kernel(cpssp, cpssp->tr.base + (0x12 + 0 * 2), cpssp->regs[R_EAX]);
	stw_kernel(cpssp, cpssp->tr.base + (0x12 + 1 * 2), cpssp->regs[R_ECX]);
	stw_kernel(cpssp, cpssp->tr.base + (0x12 + 2 * 2), cpssp->regs[R_EDX]);
	stw_kernel(cpssp, cpssp->tr.base + (0x12 + 3 * 2), cpssp->regs[R_EBX]);
	stw_kernel(cpssp, cpssp->tr.base + (0x12 + 4 * 2), cpssp->regs[R_ESP]);
	stw_kernel(cpssp, cpssp->tr.base + (0x12 + 5 * 2), cpssp->regs[R_EBP]);
	stw_kernel(cpssp, cpssp->tr.base + (0x12 + 6 * 2), cpssp->regs[R_ESI]);
	stw_kernel(cpssp, cpssp->tr.base + (0x12 + 7 * 2), cpssp->regs[R_EDI]);
	for(i = 0; i < 4; i++)
	    stw_kernel(cpssp, cpssp->tr.base + (0x22 + i * 4), cpssp->segs[i].selector);
    }

    /* now if an exception occurs, it will occurs in the next task
       context */

    if (source == SWITCH_TSS_CALL) {
	stw_kernel(cpssp, tss_base, cpssp->tr.selector);
	new_eflags |= CPU_NT_MASK;
    }

    /* set busy bit */
    if (source == SWITCH_TSS_JMP || source == SWITCH_TSS_CALL) {
	target_ulong ptr_;
	uint32_t e2_;
	ptr_ = cpssp->gdt.base + (tss_selector & ~7);
	e2_ = ldl_kernel(cpssp, ptr_ + 4);
	e2_ |= DESC_TSS_BUSY_MASK;
	stl_kernel(cpssp, ptr_ + 4, e2_);
    }

    /* set the new CPU state */
    /* from this point, any exception which occurs can give problems */
    cpssp->cr[0] |= CPU_CR0_TS_MASK;
    cpssp->hflags |= HF_TS_MASK;
    cpssp->tr.selector = tss_selector;
    cpssp->tr.base = tss_base;
    cpssp->tr.limit = tss_limit;
    cpssp->tr.flags = e2 & ~DESC_TSS_BUSY_MASK;

    if ((type & 8) && (cpssp->cr[0] & CPU_CR0_PG_MASK)) {
	NAME_(update_cr3)(cpssp, new_cr3);
    }

    /* load all registers without an exception, then reload them with
       possible exception */
    cpssp->eip = new_eip;
    eflags_mask = CPU_TF_MASK | CPU_AC_MASK | CPU_ID_MASK |
	CPU_IF_MASK | CPU_IOPL_MASK | CPU_VM_MASK | CPU_RF_MASK | CPU_NT_MASK;
    if (!(type & 8))
	eflags_mask &= 0xffff;
    load_eflags(cpssp, new_eflags, eflags_mask);
    /* XXX: what to do in 16 bit case ? */
    cpssp->regs[R_EAX] = new_regs[0];
    cpssp->regs[R_ECX] = new_regs[1];
    cpssp->regs[R_EDX] = new_regs[2];
    cpssp->regs[R_EBX] = new_regs[3];
    cpssp->regs[R_ESP] = new_regs[4];
    cpssp->regs[R_EBP] = new_regs[5];
    cpssp->regs[R_ESI] = new_regs[6];
    cpssp->regs[R_EDI] = new_regs[7];
    if (new_eflags & CPU_VM_MASK) {
	for(i = 0; i < 6; i++)
	    load_seg_vm(cpssp, i, new_segs[i]);
	/* in vm86, CPL is always 3 */
	cpu_x86_set_cpl(cpssp, 3);
    } else {
	/* CPL is set the RPL of CS */
	cpu_x86_set_cpl(cpssp, new_segs[R_CS] & 3);
	/* first just selectors as the rest may trigger exceptions */
	for(i = 0; i < 6; i++)
	    cpu_x86_load_seg_cache(cpssp, i, new_segs[i], 0, 0, 0);
    }

    cpssp->ldt.selector = new_ldt & ~4;
    cpssp->ldt.base = 0;
    cpssp->ldt.limit = 0;
    cpssp->ldt.flags = 0;

    /* load the LDT */
    if (new_ldt & 4)
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, new_ldt & 0xfffc);

    if ((new_ldt & 0xfffc) != 0) {
	dt = &cpssp->gdt;
	index_ = new_ldt & ~7;
	if ((index_ + 7) > dt->limit)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, new_ldt & 0xfffc);
	ptr = dt->base + index_;
	e1 = ldl_kernel(cpssp, ptr);
	e2 = ldl_kernel(cpssp, ptr + 4);
	if ((e2 & DESC_S_MASK) || ((e2 >> DESC_TYPE_SHIFT) & 0xf) != 2)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, new_ldt & 0xfffc);
	if (!(e2 & DESC_P_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, new_ldt & 0xfffc);
	load_seg_cache_raw_dt(&cpssp->ldt, e1, e2);
    }

    /* load the segments */
    if (!(new_eflags & CPU_VM_MASK)) {
	tss_load_seg(cpssp, R_CS, new_segs[R_CS]);
	tss_load_seg(cpssp, R_SS, new_segs[R_SS]);
	tss_load_seg(cpssp, R_ES, new_segs[R_ES]);
	tss_load_seg(cpssp, R_DS, new_segs[R_DS]);
	tss_load_seg(cpssp, R_FS, new_segs[R_FS]);
	tss_load_seg(cpssp, R_GS, new_segs[R_GS]);
    }

    /* check that EIP is in the CS segment limits */
    if (new_eip > cpssp->segs[R_CS].limit) {
	/* XXX: different exception if CALL ? */
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
    }
}

/* check if Port I/O is allowed in TSS */
static inline void
check_io(struct cpssp *cpssp, int addr, int size)
{
    int io_offset, val, mask;

    /* ,,open'' our backdoor for bios debugging output */
    if (addr == 0xffff && size == 1) {
	    return;
    }

    /* TSS must be a valid 32 bit one */
    if (!(cpssp->tr.flags & DESC_P_MASK) ||
	((cpssp->tr.flags >> DESC_TYPE_SHIFT) & 0xf) != 9 ||
	cpssp->tr.limit < 103)
	goto fail;
    io_offset = lduw_kernel(cpssp, cpssp->tr.base + 0x66);
    io_offset += (addr >> 3);
    /* Note: the check needs two bytes */
    if ((io_offset + 1) > cpssp->tr.limit)
	goto fail;
    val = lduw_kernel(cpssp, cpssp->tr.base + io_offset);
    val >>= (addr & 7);
    mask = (1 << size) - 1;
    /* all bits must be zero to allow the I/O */
    if ((val & mask) != 0) {
    fail:
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
    }
}

void
NAME_(check_iob_T0)(struct cpssp *cpssp, uint16_t t0)
{
	check_io(cpssp, t0, 1);
}

void
NAME_(check_iow_T0)(struct cpssp *cpssp, uint16_t t0)
{
	check_io(cpssp, t0, 2);
}

void
NAME_(check_iol_T0)(struct cpssp *cpssp, uint16_t t0)
{
	check_io(cpssp, t0, 4);
}

void
NAME_(check_iob_DX)(struct cpssp *cpssp)
{
	check_io(cpssp, cpssp->regs[R_EDX] & 0xffff, 1);
}

void
NAME_(check_iow_DX)(struct cpssp *cpssp)
{
	check_io(cpssp, cpssp->regs[R_EDX] & 0xffff, 2);
}

void
NAME_(check_iol_DX)(struct cpssp *cpssp)
{
	check_io(cpssp, cpssp->regs[R_EDX] & 0xffff, 4);
}

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT

static void add128(uint64_t *plow, uint64_t *phigh, uint64_t a, uint64_t b)
{
    *plow += a;
    /* carry test */
    if (*plow < a)
	(*phigh)++;
    *phigh += b;
}

static void neg128(uint64_t *plow, uint64_t *phigh)
{
    *plow = ~ *plow;
    *phigh = ~ *phigh;
    add128(plow, phigh, 1, 0);
}

static void mul64(uint64_t *plow, uint64_t *phigh, uint64_t a, uint64_t b)
{
    uint32_t a0, a1, b0, b1;
    uint64_t v;

    a0 = a;
    a1 = a >> 32;

    b0 = b;
    b1 = b >> 32;

    v = (uint64_t)a0 * (uint64_t)b0;
    *plow = v;
    *phigh = 0;

    v = (uint64_t)a0 * (uint64_t)b1;
    add128(plow, phigh, v << 32, v >> 32);

    v = (uint64_t)a1 * (uint64_t)b0;
    add128(plow, phigh, v << 32, v >> 32);

    v = (uint64_t)a1 * (uint64_t)b1;
    *phigh += v;
}

static void imul64(uint64_t *plow, uint64_t *phigh, int64_t a, int64_t b)
{
    int sa, sb;
    sa = (a < 0);
    if (sa)
	a = -a;
    sb = (b < 0);
    if (sb)
	b = -b;
    mul64(plow, phigh, a, b);
    if (sa ^ sb) {
	neg128(plow, phigh);
    }
}

/* return TRUE if overflow */
static int div64(uint64_t *plow, uint64_t *phigh, uint64_t b)
{
    uint64_t q, r, a1, a0;
    int i, qb, ab;

    a0 = *plow;
    a1 = *phigh;
    if (a1 == 0) {
	q = a0 / b;
	r = a0 % b;
	*plow = q;
	*phigh = r;
    } else {
	if (a1 >= b)
	    return 1;
	/* XXX: use a better algorithm */
	for(i = 0; i < 64; i++) {
	    ab = a1 >> 63;
	    a1 = (a1 << 1) | (a0 >> 63);
	    if (ab || a1 >= b) {
		a1 -= b;
		qb = 1;
	    } else {
		qb = 0;
	    }
	    a0 = (a0 << 1) | qb;
	}
	*plow = a0;
	*phigh = a1;
    }
    return 0;
}

/* return TRUE if overflow */
static int idiv64(uint64_t *plow, uint64_t *phigh, int64_t b)
{
    int sa, sb;
    sa = ((int64_t)*phigh < 0);
    if (sa)
	neg128(plow, phigh);
    sb = (b < 0);
    if (sb)
	b = -b;
    if (div64(plow, phigh, b) != 0)
	return 1;
    if (sa ^ sb) {
	if (*plow > (1ULL << 63))
	    return 1;
	*plow = - *plow;
    } else {
	if (*plow >= (1ULL << 63))
	    return 1;
    }
    if (sa)
	*phigh = - *phigh;
    return 0;
}

void
NAME_(helper_mulq_EAX_T0)(struct cpssp *cpssp, uint64_t t0)
{
    uint64_t r0, r1;

    mul64(&r0, &r1, cpssp->regs[R_EAX], t0);
    cpssp->regs[R_EAX] = r0;
    cpssp->regs[R_EDX] = r1;
    cpssp->cc_dst = r0;
    cpssp->cc_src = r1;
}

void NAME_(helper_imulq_EAX_T0)(struct cpssp *cpssp, uint64_t t0)
{
    uint64_t r0, r1;

    imul64(&r0, &r1, cpssp->regs[R_EAX], t0);
    cpssp->regs[R_EAX] = r0;
    cpssp->regs[R_EDX] = r1;
    cpssp->cc_dst = r0;
    cpssp->cc_src = ((int64_t)r1 != ((int64_t)r0 >> 63));
}

uint64_t
NAME_(helper_imulq_T0_T1)(struct cpssp *cpssp, uint64_t t0, uint64_t t1)
{
	uint64_t r0, r1;

	imul64(&r0, &r1, t0, t1);
	cpssp->cc_dst = r0;
	cpssp->cc_src = ((int64_t)r1 != ((int64_t)r0 >> 63));

	return r0;
}

void
NAME_(helper_divq_EAX_T0)(struct cpssp *cpssp, uint64_t t0)
{
    uint64_t r0, r1;

    if (t0 == 0) {
	NAME_(raise_exception)(cpssp, CPU_FAULT_DE);
    }
    r0 = cpssp->regs[R_EAX];
    r1 = cpssp->regs[R_EDX];
    if (div64(&r0, &r1, t0))
	NAME_(raise_exception)(cpssp, CPU_FAULT_DE);
    cpssp->regs[R_EAX] = r0;
    cpssp->regs[R_EDX] = r1;
}

void
NAME_(helper_idivq_EAX_T0)(struct cpssp *cpssp, uint64_t t0)
{
    uint64_t r0, r1;

    if (t0 == 0) {
	NAME_(raise_exception)(cpssp, CPU_FAULT_DE);
    }
    r0 = cpssp->regs[R_EAX];
    r1 = cpssp->regs[R_EDX];
    if (idiv64(&r0, &r1, t0))
	NAME_(raise_exception)(cpssp, CPU_FAULT_DE);
    cpssp->regs[R_EAX] = r0;
    cpssp->regs[R_EDX] = r1;
}

uint64_t
NAME_(helper_bswapq_T0)(struct cpssp *cpssp, uint64_t t0)
{
	return bswap64(t0);
}

#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */

void
NAME_(helper_cmpxchg8b)(struct cpssp *cpssp, target_ulong a0)
{
    uint64_t d;
    int eflags;

    eflags = compute_all(cpssp);
    d = ldq(cpssp, a0);
    if (d == (((uint64_t)cpssp->regs[R_EDX] << 32) | (uint32_t)cpssp->regs[R_EAX])) {
	stq(cpssp, a0, ((uint64_t)cpssp->regs[R_ECX] << 32) | (uint32_t)cpssp->regs[R_EBX]);
	eflags |= CC_Z;
    } else {
	cpssp->regs[R_EDX] = (uint32_t) (d >> 32);
	cpssp->regs[R_EAX] = (uint32_t) d;
	eflags &= ~CC_Z;
    }
    cpssp->cc_src = eflags;
}

static inline unsigned int
get_sp_mask(unsigned int e2)
{
	if (e2 & DESC_B_MASK)
		return 0xffffffff;
	else
		return 0xffff;
}

void
NAME_(helper_enter_level)(struct cpssp *cpssp, int level, int data32, target_ulong t1)
{
    target_ulong ssp;
    uint32_t esp_mask, esp, ebp;

    esp_mask = get_sp_mask(cpssp->segs[R_SS].flags);
    ssp = cpssp->segs[R_SS].base;
    ebp = cpssp->regs[R_EBP];
    esp = cpssp->regs[R_ESP];
    if (data32) {
	/* 32 bit */
	esp -= 4;
	while (--level) {
	    esp -= 4;
	    ebp -= 4;
	    stl(cpssp, ssp + (esp & esp_mask), ldl(cpssp, ssp + (ebp & esp_mask)));
	}
	esp -= 4;
	stl(cpssp, ssp + (esp & esp_mask), t1);
    } else {
	/* 16 bit */
	esp -= 2;
	while (--level) {
	    esp -= 2;
	    ebp -= 2;
	    stw(cpssp, ssp + (esp & esp_mask), lduw(cpssp, ssp + (ebp & esp_mask)));
	}
	esp -= 2;
	stw(cpssp, ssp + (esp & esp_mask), t1);
    }
}

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
void
NAME_(helper_enter64_level)(struct cpssp *cpssp, int level, int data64, target_ulong t1)
{
    target_ulong esp, ebp;
    ebp = cpssp->regs[R_EBP];
    esp = cpssp->regs[R_ESP];

    if (data64) {
	/* 64 bit */
	esp -= 8;
	while (--level) {
	    esp -= 8;
	    ebp -= 8;
	    stq(cpssp, esp, ldq(cpssp, ebp));
	}
	esp -= 8;
	stq(cpssp, esp, t1);
    } else {
	/* 16 bit */
	esp -= 2;
	while (--level) {
	    esp -= 2;
	    ebp -= 2;
	    stw(cpssp, esp, lduw(cpssp, ebp));
	}
	esp -= 2;
	stw(cpssp, esp, t1);
    }
}
#endif

void
NAME_(helper_lldt_T0)(struct cpssp *cpssp, uint16_t t0)
{
    int selector;
    struct SegmentCache *dt;
    uint32_t e1, e2;
    int index_, entry_limit;
    target_ulong ptr;

    selector = t0 & 0xffff;
    if ((selector & 0xfffc) == 0) {
	/* XXX: NULL selector case: invalid LDT */
	cpssp->ldt.base = 0;
	cpssp->ldt.limit = 0;
    } else {
	if (selector & 0x4)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	dt = &cpssp->gdt;
	index_ = selector & ~7;
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	if (cpssp->hflags & HF_LMA_MASK)
	    entry_limit = 15;
	else
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */
	    entry_limit = 7;
	if ((index_ + entry_limit) > dt->limit)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	ptr = dt->base + index_;
	e1 = ldl_kernel(cpssp, ptr);
	e2 = ldl_kernel(cpssp, ptr + 4);
	if ((e2 & DESC_S_MASK) || ((e2 >> DESC_TYPE_SHIFT) & 0xf) != 2)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	if (!(e2 & DESC_P_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, selector & 0xfffc);
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	if (cpssp->hflags & HF_LMA_MASK) {
	    uint32_t e3;
	    e3 = ldl_kernel(cpssp, ptr + 8);
	    load_seg_cache_raw_dt(&cpssp->ldt, e1, e2);
	    cpssp->ldt.base |= (target_ulong)e3 << 32;
	} else
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */
	{
	    load_seg_cache_raw_dt(&cpssp->ldt, e1, e2);
	}
    }
    cpssp->ldt.selector = selector;
}

void
NAME_(helper_ltr_T0)(struct cpssp *cpssp, uint16_t t0)
{
    int selector;
    struct SegmentCache *dt;
    uint32_t e1, e2;
    int index_, type, entry_limit;
    target_ulong ptr;

    selector = t0 & 0xffff;
    if ((selector & 0xfffc) == 0) {
	/* NULL selector case: invalid TR */
	cpssp->tr.base = 0;
	cpssp->tr.limit = 0;
	cpssp->tr.flags = 0;
    } else {
	if (selector & 0x4)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	dt = &cpssp->gdt;
	index_ = selector & ~7;
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	if (cpssp->hflags & HF_LMA_MASK)
	    entry_limit = 15;
	else
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */
	    entry_limit = 7;
	if ((index_ + entry_limit) > dt->limit)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	ptr = dt->base + index_;
	e1 = ldl_kernel(cpssp, ptr);
	e2 = ldl_kernel(cpssp, ptr + 4);
	type = (e2 >> DESC_TYPE_SHIFT) & 0xf;
	if ((e2 & DESC_S_MASK) ||
	    (type != 1 && type != 9))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	if (!(e2 & DESC_P_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, selector & 0xfffc);
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	if (cpssp->hflags & HF_LMA_MASK) {
	    uint32_t e3, e4;
	    e3 = ldl_kernel(cpssp, ptr + 8);
	    e4 = ldl_kernel(cpssp, ptr + 12);
	    if ((e4 >> DESC_TYPE_SHIFT) & 0xf)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	    load_seg_cache_raw_dt(&cpssp->tr, e1, e2);
	    cpssp->tr.base |= (target_ulong)e3 << 32;
	} else
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */
	{
	    load_seg_cache_raw_dt(&cpssp->tr, e1, e2);
	}
	e2 |= DESC_TSS_BUSY_MASK;
	stl_kernel(cpssp, ptr + 4, e2);
    }
    cpssp->tr.selector = selector;
}

/* only works if protected mode and not VM86. seg_reg must be != R_CS */
void
NAME_(load_seg)(struct cpssp *cpssp, int seg_reg, int selector)
{
    uint32_t e1, e2;
    int cpl, dpl, rpl;
    struct SegmentCache *dt;
    int index_;
    target_ulong ptr;

    selector &= 0xffff;
    cpl = cpssp->hflags & HF_CPL_MASK;
    if ((selector & 0xfffc) == 0) {
	/* null selector case */
	if (seg_reg == R_SS
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	    && (!(cpssp->hflags & HF_CS64_MASK) || cpl == 3)
#endif
	    )
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
	cpu_x86_load_seg_cache(cpssp, seg_reg, selector, 0, 0, 0);
    } else {

	if (selector & 0x4)
	    dt = &cpssp->ldt;
	else
	    dt = &cpssp->gdt;
	index_ = selector & ~7;
	if ((index_ + 7) > dt->limit)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	ptr = dt->base + index_;
	e1 = ldl_kernel(cpssp, ptr);
	e2 = ldl_kernel(cpssp, ptr + 4);

	if (!(e2 & DESC_S_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	rpl = selector & 3;
	dpl = (e2 >> DESC_DPL_SHIFT) & 3;
	if (seg_reg == R_SS) {
	    /* must be writable segment */
	    if ((e2 & DESC_CS_MASK) || !(e2 & DESC_W_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	    if (rpl != cpl || dpl != cpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	} else {
	    /* must be readable segment */
	    if ((e2 & (DESC_CS_MASK | DESC_R_MASK)) == DESC_CS_MASK)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);

	    if (!(e2 & DESC_CS_MASK) || !(e2 & DESC_C_MASK)) {
		/* if not conforming code, test rights */
		if (dpl < cpl || dpl < rpl)
		    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	    }
	}

	if (!(e2 & DESC_P_MASK)) {
	    if (seg_reg == R_SS)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_SS, selector & 0xfffc);
	    else
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, selector & 0xfffc);
	}

	/* set the access bit if not already set */
	if (!(e2 & DESC_A_MASK)) {
	    e2 |= DESC_A_MASK;
	    stl_kernel(cpssp, ptr + 4, e2);
	}

	cpu_x86_load_seg_cache(cpssp, seg_reg, selector,
		       get_seg_base(e1, e2),
		       get_seg_limit(e1, e2),
		       e2);
#if 0
	fprintf(logfile, "load_seg: sel=0x%04x base=0x%08lx limit=0x%08lx flags=%08x\n",
		selector, (unsigned long)sc->base, sc->limit, sc->flags);
#endif
    }
}

/* protected mode jump */
void
NAME_(helper_ljmp_protected_T0_T1)(struct cpssp *cpssp, int next_eip_addend, uint16_t t0, target_ulong t1)
{
    int new_cs, gate_cs, type;
    uint32_t e1, e2, cpl, dpl, rpl, limit;
    target_ulong new_eip, next_eip;

    new_cs = t0;
    new_eip = t1;
    if ((new_cs & 0xfffc) == 0)
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
    if (load_segment(cpssp, &e1, &e2, new_cs) != 0)
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
    cpl = cpssp->hflags & HF_CPL_MASK;
    if (e2 & DESC_S_MASK) {
	if (!(e2 & DESC_CS_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	dpl = (e2 >> DESC_DPL_SHIFT) & 3;
	if (e2 & DESC_C_MASK) {
	    /* conforming code segment */
	    if (dpl > cpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	} else {
	    /* non conforming code segment */
	    rpl = new_cs & 3;
	    if (rpl > cpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	    if (dpl != cpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	}
	if (!(e2 & DESC_P_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, new_cs & 0xfffc);
	limit = get_seg_limit(e1, e2);
	if (new_eip > limit &&
	    !(cpssp->hflags & HF_LMA_MASK) && !(e2 & DESC_L_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	cpu_x86_load_seg_cache(cpssp, R_CS, (new_cs & 0xfffc) | cpl,
		       get_seg_base(e1, e2), limit, e2);
	EIP(cpssp) = new_eip;
    } else {
	/* jump to call or task gate */
	dpl = (e2 >> DESC_DPL_SHIFT) & 3;
	rpl = new_cs & 3;
	cpl = cpssp->hflags & HF_CPL_MASK;
	type = (e2 >> DESC_TYPE_SHIFT) & 0xf;
	switch(type) {
	case 1: /* 286 TSS */
	case 9: /* 386 TSS */
	case 5: /* task gate */
	    if (dpl < cpl || dpl < rpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	    next_eip = cpssp->eip + next_eip_addend;
	    NAME_(switch_tss)(cpssp, new_cs, e1, e2, SWITCH_TSS_JMP, next_eip);
	    cpssp->cc_op = CC_OP_EFLAGS;
	    break;
	case 4: /* 286 call gate */
	case 12: /* 386 call gate */
	    if ((dpl < cpl) || (dpl < rpl))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	    if (!(e2 & DESC_P_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, new_cs & 0xfffc);
	    gate_cs = e1 >> 16;
	    new_eip = (e1 & 0xffff);
	    if (type == 12)
		new_eip |= (e2 & 0xffff0000);
	    if (load_segment(cpssp, &e1, &e2, gate_cs) != 0)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, gate_cs & 0xfffc);
	    dpl = (e2 >> DESC_DPL_SHIFT) & 3;
	    /* must be code segment */
	    if (((e2 & (DESC_S_MASK | DESC_CS_MASK)) !=
		 (DESC_S_MASK | DESC_CS_MASK)))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, gate_cs & 0xfffc);
	    if (((e2 & DESC_C_MASK) && (dpl > cpl)) ||
		(!(e2 & DESC_C_MASK) && (dpl != cpl)))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, gate_cs & 0xfffc);
	    if (!(e2 & DESC_P_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, gate_cs & 0xfffc);
	    limit = get_seg_limit(e1, e2);
	    if (new_eip > limit)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
	    cpu_x86_load_seg_cache(cpssp, R_CS, (gate_cs & 0xfffc) | cpl,
				   get_seg_base(e1, e2), limit, e2);
	    EIP(cpssp) = new_eip;
	    break;
	default:
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	    break;
	}
    }
}

/* real mode call */
void
NAME_(helper_lcall_real_T0_T1)(struct cpssp *cpssp, int shift, int next_eip, uint16_t t0, target_ulong t1)
{
    int new_cs, new_eip;
    uint32_t esp, esp_mask;
    target_ulong ssp;

    new_cs = t0;
    new_eip = t1;
    esp = cpssp->regs[R_ESP];
    esp_mask = get_sp_mask(cpssp->segs[R_SS].flags);
    ssp = cpssp->segs[R_SS].base;
    if (shift) {
	PUSHL(cpssp, ssp, esp, esp_mask, cpssp->segs[R_CS].selector);
	PUSHL(cpssp, ssp, esp, esp_mask, next_eip);
    } else {
	PUSHW(cpssp, ssp, esp, esp_mask, cpssp->segs[R_CS].selector);
	PUSHW(cpssp, ssp, esp, esp_mask, next_eip);
    }

    cpssp->regs[R_ESP] = (cpssp->regs[R_ESP] & ~esp_mask) | (esp & esp_mask);
    cpssp->eip = new_eip;
    cpssp->segs[R_CS].selector = new_cs;
    cpssp->segs[R_CS].base = (new_cs << 4);
}

static inline __attribute__((__always_inline__)) void
get_ss_esp_from_tss(struct cpssp *cpssp, uint32_t *ss_ptr, uint32_t *esp_ptr, int dpl)
{
	int type, index_, shift;

	assert(cpssp->tr.flags & DESC_P_MASK);
	type = (cpssp->tr.flags >> DESC_TYPE_SHIFT) & 0xf;
	assert((type & 7) == 1);
	shift = type >> 3;
	index_ = (dpl * 4 + 2) << shift;
	if (index_ + (4 << shift) - 1 > cpssp->tr.limit)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, cpssp->tr.selector & 0xfffc);
	if (shift == 0) {
		*esp_ptr = lduw_kernel(cpssp, cpssp->tr.base + index_);
		*ss_ptr = lduw_kernel(cpssp, cpssp->tr.base + index_ + 2);
	} else {
		*esp_ptr = ldl_kernel(cpssp, cpssp->tr.base + index_);
		*ss_ptr = lduw_kernel(cpssp, cpssp->tr.base + index_ + 4);
	}
}

/* protected mode call */
void
NAME_(helper_lcall_protected_T0_T1)(struct cpssp *cpssp, int shift, int next_eip_addend, uint16_t t0, target_ulong t1)
{
    int new_cs, new_stack, i;
    uint32_t e1, e2, cpl, dpl, rpl, selector, offset, param_count;
    uint32_t ss, ss_e1, ss_e2, sp, type, ss_dpl, sp_mask;
    uint32_t val, limit, old_sp_mask;
    target_ulong ssp, old_ssp, next_eip, new_eip;

    new_cs = t0;
    new_eip = t1;
    next_eip = cpssp->eip + next_eip_addend;
    if ((new_cs & 0xfffc) == 0)
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
    if (load_segment(cpssp, &e1, &e2, new_cs) != 0)
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
    cpl = cpssp->hflags & HF_CPL_MASK;
    if (e2 & DESC_S_MASK) {
	if (!(e2 & DESC_CS_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	dpl = (e2 >> DESC_DPL_SHIFT) & 3;
	if (e2 & DESC_C_MASK) {
	    /* conforming code segment */
	    if (dpl > cpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	} else {
	    /* non conforming code segment */
	    rpl = new_cs & 3;
	    if (rpl > cpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	    if (dpl != cpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	}
	if (!(e2 & DESC_P_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, new_cs & 0xfffc);

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	/* XXX: check 16/32 bit cases in long mode */
	if (shift == 2) {
	    target_ulong rsp;
	    /* 64 bit case */
	    rsp = cpssp->regs[R_ESP];
	    PUSHQ(cpssp, rsp, cpssp->segs[R_CS].selector);
	    PUSHQ(cpssp, rsp, next_eip);
	    /* from this point, not restartable */
	    cpssp->regs[R_ESP] = rsp;
	    cpu_x86_load_seg_cache(cpssp, R_CS, (new_cs & 0xfffc) | cpl,
				   get_seg_base(e1, e2),
				   get_seg_limit(e1, e2), e2);
	    EIP(cpssp) = new_eip;
	} else
#endif
	{
	    sp = cpssp->regs[R_ESP];
	    sp_mask = get_sp_mask(cpssp->segs[R_SS].flags);
	    ssp = cpssp->segs[R_SS].base;
	    if (shift) {
		PUSHL(cpssp, ssp, sp, sp_mask, cpssp->segs[R_CS].selector);
		PUSHL(cpssp, ssp, sp, sp_mask, next_eip);
	    } else {
		PUSHW(cpssp, ssp, sp, sp_mask, cpssp->segs[R_CS].selector);
		PUSHW(cpssp, ssp, sp, sp_mask, next_eip);
	    }

	    limit = get_seg_limit(e1, e2);
	    if (new_eip > limit)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	    /* from this point, not restartable */
	    cpssp->regs[R_ESP] = (cpssp->regs[R_ESP] & ~sp_mask) | (sp & sp_mask);
	    cpu_x86_load_seg_cache(cpssp, R_CS, (new_cs & 0xfffc) | cpl,
				   get_seg_base(e1, e2), limit, e2);
	    EIP(cpssp) = new_eip;
	}
    } else {
	/* check gate type */
	type = (e2 >> DESC_TYPE_SHIFT) & 0x1f;
	dpl = (e2 >> DESC_DPL_SHIFT) & 3;
	rpl = new_cs & 3;
	switch(type) {
	case 1: /* available 286 TSS */
	case 9: /* available 386 TSS */
	case 5: /* task gate */
	    if (dpl < cpl || dpl < rpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	    NAME_(switch_tss)(cpssp, new_cs, e1, e2, SWITCH_TSS_CALL, next_eip);
	    cpssp->cc_op = CC_OP_EFLAGS;
	    return;
	case 4: /* 286 call gate */
	case 12: /* 386 call gate */
	    break;
	default:
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	    break;
	}
	shift = type >> 3;

	if (dpl < cpl || dpl < rpl)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
	/* check valid bit */
	if (!(e2 & DESC_P_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP,  new_cs & 0xfffc);
	selector = e1 >> 16;
	offset = (e2 & 0xffff0000) | (e1 & 0x0000ffff);
	param_count = e2 & 0x1f;
	if ((selector & 0xfffc) == 0)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);

	if (load_segment(cpssp, &e1, &e2, selector) != 0)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	if (!(e2 & DESC_S_MASK) || !(e2 & (DESC_CS_MASK)))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	dpl = (e2 >> DESC_DPL_SHIFT) & 3;
	if (dpl > cpl)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
	if (!(e2 & DESC_P_MASK))
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, selector & 0xfffc);

	if (!(e2 & DESC_C_MASK) && dpl < cpl) {
	    /* to inner privilege */
	    get_ss_esp_from_tss(cpssp, &ss, &sp, dpl);
	    if ((ss & 0xfffc) == 0)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
	    if ((ss & 3) != dpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
	    if (load_segment(cpssp, &ss_e1, &ss_e2, ss) != 0)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
	    ss_dpl = (ss_e2 >> DESC_DPL_SHIFT) & 3;
	    if (ss_dpl != dpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
	    if (!(ss_e2 & DESC_S_MASK) ||
		(ss_e2 & DESC_CS_MASK) ||
		!(ss_e2 & DESC_W_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
	    if (!(ss_e2 & DESC_P_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);

	    //	    push_size = ((param_count * 2) + 8) << shift;

	    old_sp_mask = get_sp_mask(cpssp->segs[R_SS].flags);
	    old_ssp = cpssp->segs[R_SS].base;

	    sp_mask = get_sp_mask(ss_e2);
	    ssp = get_seg_base(ss_e1, ss_e2);
	    if (shift) {
		PUSHL(cpssp, ssp, sp, sp_mask, cpssp->segs[R_SS].selector);
		PUSHL(cpssp, ssp, sp, sp_mask, cpssp->regs[R_ESP]);
		for(i = param_count - 1; i >= 0; i--) {
		    val = ldl_kernel(cpssp, old_ssp + ((cpssp->regs[R_ESP] + i * 4) & old_sp_mask));
		    PUSHL(cpssp, ssp, sp, sp_mask, val);
		}
	    } else {
		PUSHW(cpssp, ssp, sp, sp_mask, cpssp->segs[R_SS].selector);
		PUSHW(cpssp, ssp, sp, sp_mask, cpssp->regs[R_ESP]);
		for(i = param_count - 1; i >= 0; i--) {
		    val = lduw_kernel(cpssp, old_ssp + ((cpssp->regs[R_ESP] + i * 2) & old_sp_mask));
		    PUSHW(cpssp, ssp, sp, sp_mask, val);
		}
	    }
	    new_stack = 1;
	} else {
	    /* to same privilege */
	    sp = cpssp->regs[R_ESP];
	    sp_mask = get_sp_mask(cpssp->segs[R_SS].flags);
	    ssp = cpssp->segs[R_SS].base;
	    //	    push_size = (4 << shift);
	    new_stack = 0;
	    ss_e1 = 0; /* avoid warning */
	    ss_e2 = 0; /* avoid warning */
	    ss = 0; /* avoid warning */
	}

	if (shift) {
	    PUSHL(cpssp, ssp, sp, sp_mask, cpssp->segs[R_CS].selector);
	    PUSHL(cpssp, ssp, sp, sp_mask, next_eip);
	} else {
	    PUSHW(cpssp, ssp, sp, sp_mask, cpssp->segs[R_CS].selector);
	    PUSHW(cpssp, ssp, sp, sp_mask, next_eip);
	}

	/* from this point, not restartable */

	if (new_stack) {
	    ss = (ss & ~3) | dpl;
	    cpu_x86_load_seg_cache(cpssp, R_SS, ss,
				   ssp,
				   get_seg_limit(ss_e1, ss_e2),
				   ss_e2);
	}

	selector = (selector & ~3) | dpl;
	cpu_x86_load_seg_cache(cpssp, R_CS, selector,
		       get_seg_base(e1, e2),
		       get_seg_limit(e1, e2),
		       e2);
	cpu_x86_set_cpl(cpssp, dpl);
	cpssp->regs[R_ESP] = (cpssp->regs[R_ESP] & ~sp_mask) | (sp & sp_mask);
	EIP(cpssp) = offset;
    }
}

/* real and vm86 mode iret */
void
NAME_(helper_iret_real)(struct cpssp *cpssp, int shift)
{
    uint32_t sp, new_cs, new_eip, new_eflags, sp_mask;
    target_ulong ssp;
    int eflags_mask;

#if DEBUG_CONTROL_BIOS || DEBUG_CONTROL_FLOW
    if (2 <= DEBUG_CONTROL_BIOS
     || 2 <= DEBUG_CONTROL_FLOW
     || loglevel) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	    if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
		    fprintf(stderr, "%d: iret at %016llx",
				    cpssp->id,
				    (unsigned long long) cpssp->eip);
	    } else
#endif
	    {
		    fprintf(stderr, "%d: iret at %08llx (%08llx:%08llx)",
				    cpssp->id,
				    (unsigned long long) cpssp->eip
				    + (unsigned long long) cpssp->segs[R_CS].base,
				    (unsigned long long) cpssp->segs[R_CS].base,
				    (unsigned long long) cpssp->eip);
	    }
	    NAME_(dump)(cpssp);
    }
#endif

    sp_mask = 0xffff; /* XXXX: use SS segment size ? */
    sp = cpssp->regs[R_ESP];
    ssp = cpssp->segs[R_SS].base;
    if (shift == 1) {
	/* 32 bits */
	POPL(cpssp, ssp, sp, sp_mask, new_eip);
	POPL(cpssp, ssp, sp, sp_mask, new_cs);
	new_cs &= 0xffff;
	POPL(cpssp, ssp, sp, sp_mask, new_eflags);
    } else {
	/* 16 bits */
	POPW(cpssp, ssp, sp, sp_mask, new_eip);
	POPW(cpssp, ssp, sp, sp_mask, new_cs);
	POPW(cpssp, ssp, sp, sp_mask, new_eflags);
    }
    cpssp->regs[R_ESP] = (cpssp->regs[R_ESP] & ~sp_mask) | (sp & sp_mask);
    load_seg_vm(cpssp, R_CS, new_cs);
    cpssp->eip = new_eip;
    if (cpssp->eflags & CPU_VM_MASK)
	eflags_mask = CPU_TF_MASK | CPU_AC_MASK | CPU_ID_MASK | CPU_IF_MASK | CPU_RF_MASK | CPU_NT_MASK;
    else
	eflags_mask = CPU_TF_MASK | CPU_AC_MASK | CPU_ID_MASK | CPU_IF_MASK | CPU_IOPL_MASK | CPU_RF_MASK | CPU_NT_MASK;
    if (shift == 0)
	eflags_mask &= 0xffff;
    load_eflags(cpssp, new_eflags, eflags_mask);
}

static inline void
NAME_(validate_seg)(struct cpssp *cpssp, int seg_reg, int cpl)
{
    int dpl;
    uint32_t e2;

    /* XXX: on x86_64, we do not want to nullify FS and GS because
       they may still contain a valid base. I would be interested to
       know how a real x86_64 CPU behaves */
    if ((seg_reg == R_FS || seg_reg == R_GS) &&
	(cpssp->segs[seg_reg].selector & 0xfffc) == 0)
	return;

    e2 = cpssp->segs[seg_reg].flags;
    dpl = (e2 >> DESC_DPL_SHIFT) & 3;
    if (!(e2 & DESC_CS_MASK) || !(e2 & DESC_C_MASK)) {
	/* data or non conforming code segment */
	if (dpl < cpl) {
	    cpu_x86_load_seg_cache(cpssp, seg_reg, 0, 0, 0, 0);
	}
    }
}

/* protected mode iret */
static void
NAME_(helper_ret_protected)(struct cpssp *cpssp, int shift, int is_iret, int addend)
{
    uint32_t new_cs, new_eflags, new_ss;
    uint32_t new_es, new_ds, new_fs, new_gs;
    uint32_t e1, e2, ss_e1, ss_e2;
    int cpl, dpl, rpl, eflags_mask, iopl;
    target_ulong ssp, sp, new_eip, new_esp, sp_mask;

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
    if (shift == 2)
	sp_mask = -1;
    else
#endif
	sp_mask = get_sp_mask(cpssp->segs[R_SS].flags);
    sp = cpssp->regs[R_ESP];
    ssp = cpssp->segs[R_SS].base;
    new_eflags = 0; /* avoid warning */
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
    if (shift == 2) {
	POPQ(cpssp, sp, new_eip);
	POPQ(cpssp, sp, new_cs);
	new_cs &= 0xffff;
	if (is_iret) {
	    POPQ(cpssp, sp, new_eflags);
	}
    } else
#endif
    if (shift == 1) {
	/* 32 bits */
	POPL(cpssp, ssp, sp, sp_mask, new_eip);
	POPL(cpssp, ssp, sp, sp_mask, new_cs);
	new_cs &= 0xffff;
	if (is_iret) {
	    POPL(cpssp, ssp, sp, sp_mask, new_eflags);
	    if (new_eflags & CPU_VM_MASK)
		goto return_to_vm86;
	}
    } else {
	/* 16 bits */
	POPW(cpssp, ssp, sp, sp_mask, new_eip);
	POPW(cpssp, ssp, sp, sp_mask, new_cs);
	if (is_iret)
	    POPW(cpssp, ssp, sp, sp_mask, new_eflags);
    }
    /* QEMU debug stuff removed */
    if ((new_cs & 0xfffc) == 0)
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
    if (load_segment(cpssp, &e1, &e2, new_cs) != 0)
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
    if (!(e2 & DESC_S_MASK) ||
	!(e2 & DESC_CS_MASK))
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
    cpl = cpssp->hflags & HF_CPL_MASK;
    rpl = new_cs & 3;
    if (rpl < cpl)
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
    dpl = (e2 >> DESC_DPL_SHIFT) & 3;
    if (e2 & DESC_C_MASK) {
	if (dpl > rpl)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
    } else {
	if (dpl != rpl)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_cs & 0xfffc);
    }
    if (!(e2 & DESC_P_MASK))
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, new_cs & 0xfffc);

    sp += addend;
    if (rpl == cpl && (!(cpssp->hflags & HF_CS64_MASK) ||
		       ((cpssp->hflags & HF_CS64_MASK) && !is_iret))) {
	/* return to same priledge level */
	cpu_x86_load_seg_cache(cpssp, R_CS, new_cs,
		       get_seg_base(e1, e2),
		       get_seg_limit(e1, e2),
		       e2);
    } else {
	/* return to different privilege level */
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	if (shift == 2) {
	    POPQ(cpssp, sp, new_esp);
	    POPQ(cpssp, sp, new_ss);
	    new_ss &= 0xffff;
	} else
#endif
	if (shift == 1) {
	    /* 32 bits */
	    POPL(cpssp, ssp, sp, sp_mask, new_esp);
	    POPL(cpssp, ssp, sp, sp_mask, new_ss);
	    new_ss &= 0xffff;
	} else {
	    /* 16 bits */
	    POPW(cpssp, ssp, sp, sp_mask, new_esp);
	    POPW(cpssp, ssp, sp, sp_mask, new_ss);
	}
	if ((new_ss & 0xfffc) == 0) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	    /* NULL ss is allowed in long mode if cpl != 3*/
	    /* XXX: test CS64 ? */
	    if ((cpssp->hflags & HF_LMA_MASK) && rpl != 3) {
		cpu_x86_load_seg_cache(cpssp, R_SS, new_ss,
				       0, 0xffffffff,
				       DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
				       DESC_S_MASK | (rpl << DESC_DPL_SHIFT) |
				       DESC_W_MASK | DESC_A_MASK);
		ss_e2 = DESC_B_MASK; /* XXX: should not be needed ? */
	    } else
#endif
	    {
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
	    }
	} else {
	    if ((new_ss & 3) != rpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_ss & 0xfffc);
	    if (load_segment(cpssp, &ss_e1, &ss_e2, new_ss) != 0)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_ss & 0xfffc);
	    if (!(ss_e2 & DESC_S_MASK) ||
		(ss_e2 & DESC_CS_MASK) ||
		!(ss_e2 & DESC_W_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_ss & 0xfffc);
	    dpl = (ss_e2 >> DESC_DPL_SHIFT) & 3;
	    if (dpl != rpl)
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, new_ss & 0xfffc);
	    if (!(ss_e2 & DESC_P_MASK))
		NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, new_ss & 0xfffc);
	    cpu_x86_load_seg_cache(cpssp, R_SS, new_ss,
			   get_seg_base(ss_e1, ss_e2),
			   get_seg_limit(ss_e1, ss_e2),
			   ss_e2);
	}

	cpu_x86_load_seg_cache(cpssp, R_CS, new_cs,
		       get_seg_base(e1, e2),
		       get_seg_limit(e1, e2),
		       e2);
	cpu_x86_set_cpl(cpssp, rpl);
	sp = new_esp;
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	if (cpssp->hflags & HF_CS64_MASK)
	    sp_mask = -1;
	else
#endif
	    sp_mask = get_sp_mask(ss_e2);

	/* validate data segments */
	NAME_(validate_seg)(cpssp, R_ES, rpl);
	NAME_(validate_seg)(cpssp, R_DS, rpl);
	NAME_(validate_seg)(cpssp, R_FS, rpl);
	NAME_(validate_seg)(cpssp, R_GS, rpl);

	sp += addend;
    }
    cpssp->regs[R_ESP] = (cpssp->regs[R_ESP] & ~sp_mask) | (sp & sp_mask);
    cpssp->eip = new_eip;
    if (is_iret) {
	/* NOTE: 'cpl' is the _old_ CPL */
	eflags_mask = CPU_TF_MASK | CPU_AC_MASK | CPU_ID_MASK | CPU_RF_MASK | CPU_NT_MASK;
	if (cpl == 0)
	    eflags_mask |= CPU_IOPL_MASK;
	iopl = (cpssp->eflags >> CPU_IOPL_SHIFT) & 3;
	if (cpl <= iopl)
	    eflags_mask |= CPU_IF_MASK;
	if (shift == 0)
	    eflags_mask &= 0xffff;
	load_eflags(cpssp, new_eflags, eflags_mask);
    }
    return;

 return_to_vm86:
    POPL(cpssp, ssp, sp, sp_mask, new_esp);
    POPL(cpssp, ssp, sp, sp_mask, new_ss);
    POPL(cpssp, ssp, sp, sp_mask, new_es);
    POPL(cpssp, ssp, sp, sp_mask, new_ds);
    POPL(cpssp, ssp, sp, sp_mask, new_fs);
    POPL(cpssp, ssp, sp, sp_mask, new_gs);

    /* modify processor state */
    load_eflags(cpssp, new_eflags, CPU_TF_MASK | CPU_AC_MASK | CPU_ID_MASK
		| CPU_IF_MASK | CPU_IOPL_MASK | CPU_VM_MASK | CPU_NT_MASK);
    load_seg_vm(cpssp, R_CS, new_cs & 0xffff);
    cpu_x86_set_cpl(cpssp, 3);
    load_seg_vm(cpssp, R_SS, new_ss & 0xffff);
    load_seg_vm(cpssp, R_ES, new_es & 0xffff);
    load_seg_vm(cpssp, R_DS, new_ds & 0xffff);
    load_seg_vm(cpssp, R_FS, new_fs & 0xffff);
    load_seg_vm(cpssp, R_GS, new_gs & 0xffff);

    cpssp->eip = new_eip & 0xffff;
    cpssp->regs[R_ESP] = new_esp;
}

void
NAME_(helper_iret_protected)(struct cpssp *cpssp, int shift, int next_eip)
{
    int tss_selector, type;
    uint32_t e1, e2;

#if DEBUG_CONTROL_FLOW
    if (2 <= DEBUG_CONTROL_FLOW
		    || loglevel) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	    if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
		    fprintf(stderr, "%d: iret at %016llx",
				    cpssp->id,
				    (unsigned long long) cpssp->eip);
	    } else
#endif
	    {
		    fprintf(stderr, "%d: iret at %08llx (%08llx:%08llx)",
				    cpssp->id,
				    (unsigned long long) cpssp->eip
				    + (unsigned long long) cpssp->segs[R_CS].base,
				    (unsigned long long) cpssp->segs[R_CS].base,
				    (unsigned long long) cpssp->eip);
	    }
	    NAME_(dump)(cpssp);
    }
#endif

    /* specific case for TSS */
    if (cpssp->eflags & CPU_NT_MASK) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	if (cpssp->hflags & HF_LMA_MASK)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
#endif
	tss_selector = lduw_kernel(cpssp, cpssp->tr.base + 0);
	if (tss_selector & 4)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, tss_selector & 0xfffc);
	if (load_segment(cpssp, &e1, &e2, tss_selector) != 0)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, tss_selector & 0xfffc);
	type = (e2 >> DESC_TYPE_SHIFT) & 0x17;
	/* NOTE: we check both segment and busy TSS */
	if (type != 3)
	    NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, tss_selector & 0xfffc);
	NAME_(switch_tss)(cpssp, tss_selector, e1, e2, SWITCH_TSS_IRET, next_eip);
    } else {
	NAME_(helper_ret_protected)(cpssp, shift, 1, 0);
    }
}

void
NAME_(helper_lret_protected)(struct cpssp *cpssp, int shift, int addend)
{
	NAME_(helper_ret_protected)(cpssp, shift, 0, addend);
}

#if 80486 <= CONFIG_CPU && CONFIG_CPU_SEP_SUPPORT
void
NAME_(helper_sysenter)(struct cpssp *cpssp)
{
    if (cpssp->sysenter_cs == 0) {
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
    }
    cpssp->eflags &= ~(CPU_VM_MASK | CPU_IF_MASK | CPU_RF_MASK);
    cpu_x86_set_cpl(cpssp, 0);
    cpu_x86_load_seg_cache(cpssp, R_CS, cpssp->sysenter_cs & 0xfffc,
			   0, 0xffffffff,
			   DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			   DESC_S_MASK |
			   DESC_CS_MASK | DESC_R_MASK | DESC_A_MASK);
    cpu_x86_load_seg_cache(cpssp, R_SS, (cpssp->sysenter_cs + 8) & 0xfffc,
			   0, 0xffffffff,
			   DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			   DESC_S_MASK |
			   DESC_W_MASK | DESC_A_MASK);
    cpssp->regs[R_ESP] = cpssp->sysenter_esp;
    EIP(cpssp) = cpssp->sysenter_eip;
}

void
NAME_(helper_sysexit)(struct cpssp *cpssp)
{
    int cpl;

    cpl = cpssp->hflags & HF_CPL_MASK;
    if (cpssp->sysenter_cs == 0 || cpl != 0) {
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
    }
    cpu_x86_set_cpl(cpssp, 3);
    cpu_x86_load_seg_cache(cpssp, R_CS, ((cpssp->sysenter_cs + 16) & 0xfffc) | 3,
			   0, 0xffffffff,
			   DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			   DESC_S_MASK | (3 << DESC_DPL_SHIFT) |
			   DESC_CS_MASK | DESC_R_MASK | DESC_A_MASK);
    cpu_x86_load_seg_cache(cpssp, R_SS, ((cpssp->sysenter_cs + 24) & 0xfffc) | 3,
			   0, 0xffffffff,
			   DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			   DESC_S_MASK | (3 << DESC_DPL_SHIFT) |
			   DESC_W_MASK | DESC_A_MASK);
    cpssp->regs[R_ESP] = cpssp->regs[R_ECX];
    EIP(cpssp) = cpssp->regs[R_EDX];
}
#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_SEP_SUPPORT */

void
NAME_(helper_syscall)(struct cpssp *cpssp, int next_eip_addend)
{
    int selector;

    if (!(cpssp->efer & MSR_EFER_SCE)) {
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_UD, 0);
    }
    selector = (cpssp->star >> 32) & 0xffff;
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
    if (cpssp->hflags & HF_LMA_MASK) {
	int code64;

	cpssp->regs[R_ECX] = cpssp->eip + next_eip_addend;

	cpssp->regs[11] = compute_eflags(cpssp);

	code64 = cpssp->hflags & HF_CS64_MASK;

	cpu_x86_set_cpl(cpssp, 0);
	cpu_x86_load_seg_cache(cpssp, R_CS, selector & 0xfffc,
			   0, 0xffffffff,
			       DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			       DESC_S_MASK |
			       DESC_CS_MASK | DESC_R_MASK | DESC_A_MASK | DESC_L_MASK);
	cpu_x86_load_seg_cache(cpssp, R_SS, (selector + 8) & 0xfffc,
			       0, 0xffffffff,
			       DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			       DESC_S_MASK |
			       DESC_W_MASK | DESC_A_MASK);
	cpssp->eflags &= ~cpssp->fmask;
	if (code64)
	    cpssp->eip = cpssp->lstar;
	else
	    cpssp->eip = cpssp->cstar;
    } else
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */
    {
	cpssp->regs[R_ECX] = (uint32_t)(cpssp->eip + next_eip_addend);

	cpu_x86_set_cpl(cpssp, 0);
	cpu_x86_load_seg_cache(cpssp, R_CS, selector & 0xfffc,
			   0, 0xffffffff,
			       DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			       DESC_S_MASK |
			       DESC_CS_MASK | DESC_R_MASK | DESC_A_MASK);
	cpu_x86_load_seg_cache(cpssp, R_SS, (selector + 8) & 0xfffc,
			       0, 0xffffffff,
			       DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			       DESC_S_MASK |
			       DESC_W_MASK | DESC_A_MASK);
	cpssp->eflags &= ~(CPU_IF_MASK | CPU_RF_MASK | CPU_VM_MASK);
	cpssp->eip = (uint32_t)cpssp->star;
    }
}

void
NAME_(helper_sysret)(struct cpssp *cpssp, int dflag)
{
    int cpl, selector;

    if (!(cpssp->efer & MSR_EFER_SCE)) {
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_UD, 0);
    }
    cpl = cpssp->hflags & HF_CPL_MASK;
    if (!(cpssp->cr[0] & CPU_CR0_PE_MASK) || cpl != 0) {
	NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
    }
    selector = (cpssp->star >> 48) & 0xffff;
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
    if (cpssp->hflags & HF_LMA_MASK) {
	if (dflag == 2) {
	    cpu_x86_load_seg_cache(cpssp, R_CS, (selector + 16) | 3,
				   0, 0xffffffff,
				   DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
				   DESC_S_MASK | (3 << DESC_DPL_SHIFT) |
				   DESC_CS_MASK | DESC_R_MASK | DESC_A_MASK |
				   DESC_L_MASK);
	    cpssp->eip = cpssp->regs[R_ECX];
	} else {
	    cpu_x86_load_seg_cache(cpssp, R_CS, selector | 3,
				   0, 0xffffffff,
				   DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
				   DESC_S_MASK | (3 << DESC_DPL_SHIFT) |
				   DESC_CS_MASK | DESC_R_MASK | DESC_A_MASK);
	    cpssp->eip = (uint32_t)cpssp->regs[R_ECX];
	}
	cpu_x86_load_seg_cache(cpssp, R_SS, selector + 8,
			       0, 0xffffffff,
			       DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			       DESC_S_MASK | (3 << DESC_DPL_SHIFT) |
			       DESC_W_MASK | DESC_A_MASK);
	load_eflags(cpssp, (uint32_t)(cpssp->regs[11]), CPU_TF_MASK | CPU_AC_MASK | CPU_ID_MASK |
		    CPU_IF_MASK | CPU_IOPL_MASK | CPU_VM_MASK | CPU_RF_MASK | CPU_NT_MASK);
	cpu_x86_set_cpl(cpssp, 3);
    } else
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */
    {
	cpu_x86_load_seg_cache(cpssp, R_CS, selector | 3,
			       0, 0xffffffff,
			       DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			       DESC_S_MASK | (3 << DESC_DPL_SHIFT) |
			       DESC_CS_MASK | DESC_R_MASK | DESC_A_MASK);
	cpssp->eip = (uint32_t)cpssp->regs[R_ECX];
	cpu_x86_load_seg_cache(cpssp, R_SS, selector + 8,
			       0, 0xffffffff,
			       DESC_G_MASK | DESC_B_MASK | DESC_P_MASK |
			       DESC_S_MASK | (3 << DESC_DPL_SHIFT) |
			       DESC_W_MASK | DESC_A_MASK);
	cpssp->eflags |= CPU_IF_MASK;
	cpu_x86_set_cpl(cpssp, 3);
    }
}

#if 80486 <= CONFIG_CPU
void
NAME_(helper_invlpg)(struct cpssp *cpssp, target_ulong addr)
{
	NAME_(core_mmu_invlpg)(cpssp, addr);
}
#endif /* 80486 <= CONFIG_CPU */

#if 80586 <= CONFIG_CPU
void
NAME_(helper_rdtsc)(struct cpssp *cpssp)
{
	uint64_t val;

	if ((cpssp->cr[4] & CPU_CR4_TSD_MASK)
	 && ((cpssp->hflags & HF_CPL_MASK) != 0)) {
		NAME_(raise_exception)(cpssp, CPU_FAULT_GP);
	}
#if 0
	val = cpssp->process.inst_cnt;
#else
	/* Beware of overflows! */
	{	uint64_t t;
		uint64_t x;
		uint64_t y;

		/* Obey offset (see below)! FIXME */
		t = time_virt();
		x = t / TIME_HZ;
		y = t % TIME_HZ;
		val = (x * CONFIG_CPU_FREQ) + (y * CONFIG_CPU_FREQ) / TIME_HZ;
	}
#endif
	cpssp->regs[R_EAX] = (uint32_t) (val >>  0);
	cpssp->regs[R_EDX] = (uint32_t) (val >> 32);
}
#endif /* 80586 <= CONFIG_CPU */

#if 80486 <= CONFIG_CPU && CONFIG_CPU_MSR_SUPPORT
void
NAME_(helper_wrmsr)(struct cpssp *cpssp)
{
	uint64_t val;

	val = ((uint32_t) cpssp->regs[R_EAX]) | ((uint64_t)((uint32_t) cpssp->regs[R_EDX]) << 32);

	switch ((uint32_t) cpssp->regs[R_ECX]) {
#if 80686 <= CONFIG_CPU
	case 0x8b: /* MSR_IA32_BIOS_SIGN_ID */
		cpssp->update_signature = cpssp->regs[R_EDX];
		break;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_TSC_SUPPORT
	case 0x10: /* MSR_IA32_TSC */
		/* Setting TSC value. */
		/* FIXME */
		break;
#endif

	case 0x2a: /* MSR_EBL_CR_POWERON */
		/* Processor Hard Poweron Configuration */
		faum_log(FAUM_LOG_WARNING, "CPU", "",
			"Writing %08x:%08x to EBL_CR_POWERON MSR 0x%x.\n",
			(uint32_t)cpssp->regs[R_EDX], (uint32_t)cpssp->regs[R_EAX], (uint32_t)cpssp->regs[R_ECX]);

		cpssp->msr[cpssp->regs[R_ECX]].low = (uint32_t) cpssp->regs[R_EAX];
		cpssp->msr[cpssp->regs[R_ECX]].high = (uint32_t) cpssp->regs[R_EDX];
		break;

	case 0x11e: /* MSR_BBL_CR_CTL3 */
		/* L2 Cache Control Register */
		faum_log(FAUM_LOG_WARNING, "CPU", "",
			"Writing %08x:%08x to BBL_CR_CTL3 MSR 0x%x.\n",
			(uint32_t)cpssp->regs[R_EDX], (uint32_t)cpssp->regs[R_EAX], (uint32_t)cpssp->regs[R_ECX]);

		cpssp->msr[cpssp->regs[R_ECX]].low = (uint32_t) cpssp->regs[R_EAX];
		cpssp->msr[cpssp->regs[R_ECX]].high = (uint32_t) cpssp->regs[R_EDX];
		break;

#if 80486 <= CONFIG_CPU && CONFIG_CPU_SEP_SUPPORT
	case 0x174: /* MSR_IA32_SYSENTER_CS */
		cpssp->sysenter_cs = val & 0xffff;
		break;

	case 0x175: /* MSR_IA32_SYSENTER_ESP */
		cpssp->sysenter_esp = val;
		break;

	case 0x176: /* MSR_IA32_SYSENTER_EIP */
		cpssp->sysenter_eip = val;
		break;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MCA_SUPPORT
	/*
	 * Machine-Check Architecture
	 */
	case 0x179: /* MSR_IA32_MCG_CAP */
		/* Read-only. */
		assert(0); /* FIXME */
		break;

	case 0x17a: /* MSR_IA32_MCG_STATUS */
		/* Read-only. */
		assert(cpssp->regs[R_EAX] == 0
		    && cpssp->regs[R_EDX] == 0); /* FIXME */
		break;

	case 0x17b: /* MSR_IA32_MCG_CTL */
		/* Just remember setting. */
		cpssp->msr[cpssp->regs[R_ECX]].low = (uint32_t) cpssp->regs[R_EAX];
		cpssp->msr[cpssp->regs[R_ECX]].high = (uint32_t) cpssp->regs[R_EDX];
		break;

	case 0x400: /* MSR_IA32_MC0_CTL */
	case 0x404: /* MSR_IA32_MC1_CTL */
	case 0x408: /* MSR_IA32_MC2_CTL */
	case 0x40c: /* MSR_IA32_MC3_CTL */
	case 0x410: /* MSR_IA32_MC4_CTL */
		/* Just remember setting. */
		cpssp->msr[cpssp->regs[R_ECX]].low = (uint32_t) cpssp->regs[R_EAX];
		cpssp->msr[cpssp->regs[R_ECX]].high = (uint32_t) cpssp->regs[R_EDX];
		break;

	case 0x401: /* MSR_IA32_MC0_STATUS */
	case 0x405: /* MSR_IA32_MC1_STATUS */
	case 0x409: /* MSR_IA32_MC2_STATUS */
	case 0x40d: /* MSR_IA32_MC3_STATUS */
	case 0x411: /* MSR_IA32_MC4_STATUS */
		/* Software might write '0' to register to clear. */
		/* Writing 1s will cause a general protection fault. */
		if (cpssp->regs[R_EAX] != 0
		 || cpssp->regs[R_EDX] != 0) {
			NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
		}
		/* FIXME */
		break;

	case 0x402: /* MSR_IA32_MC0_ADDR */
	case 0x406: /* MSR_IA32_MC1_ADDR */
	case 0x40a: /* MSR_IA32_MC2_ADDR */
	case 0x40e: /* MSR_IA32_MC3_ADDR */
	case 0x412: /* MSR_IA32_MC4_ADDR */
		/* Read-only. */
		assert(0); /* FIXME */

	case 0x403: /* MSR_IA32_MC0_MISC */
	case 0x407: /* MSR_IA32_MC1_MISC */
	case 0x40b: /* MSR_IA32_MC2_MISC */
	case 0x40f: /* MSR_IA32_MC3_MISC */
	case 0x413: /* MSR_IA32_MC4_MISC */
		/* Not implemented. */
		assert(0); /* FIXME */
		break;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_PAT_SUPPORT
	case 0x277: /* MSR_IA32_PAT */ {
		uint32_t val;
		unsigned int i;

		/* 3-628. */
		for (i = 0; i < 8; i++) {
			if (i < 4) {
				val = cpssp->regs[R_EAX];
			} else {
				val = cpssp->regs[R_EDX];
			}
			val >>= i * 8;
			val &= 0xff;

			if (val == 0x02
			 || val == 0x03
			 || 0x08 <= val) {
				NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);
			}

			cpssp->pat[i] = val;
		}
		break;
	    }
#endif
#if CONFIG_CPU >= 80486 && (CONFIG_CPU_NX_SUPPORT || CONFIG_CPU_LM_SUPPORT)
	case 0xc0000080: /* MSR_EFER */
		{
			uint64_t update_mask;
			update_mask = 0;
			if (CONFIG_CPU_SYSCALL_SUPPORT) {
				update_mask |= MSR_EFER_SCE;
			}
			if (CONFIG_CPU_FFXSR_SUPPORT) {
				update_mask |= MSR_EFER_FFXSR;
			}
			if (CONFIG_CPU_NX_SUPPORT) {
				update_mask |= MSR_EFER_NXE;
			}
			if (CONFIG_CPU_LM_SUPPORT) {
				update_mask |= MSR_EFER_LME;
			}
			cpssp->efer = (cpssp->efer & ~update_mask)
				  | (val & update_mask);
		}
		break;
#endif
	case 0xc0000081: /* MSR_STAR */
		cpssp->star = val;
		break;

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	case MSR_LSTAR: /* 0xc0000082 */
		cpssp->lstar = val;
		break;
	case MSR_CSTAR: /* 0xc0000083 */
		cpssp->cstar = val;
		break;
	case MSR_FMASK: /* 0xc0000084 */
		cpssp->fmask = val;
		break;
	case MSR_FSBASE: /* 0xc0000100 */
		cpssp->segs[R_FS].base = val;
		break;
	case MSR_GSBASE: /* 0xc0000101 */
		cpssp->segs[R_GS].base = val;
		break;
	case MSR_KERNELGSBASE: /* 0xc0000102 */
		cpssp->kernelgsbase = val;
		break;
#endif
	default:
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
		if (NAME_(core_apic_wrmsr)(cpssp, cpssp->regs[R_ECX], ((uint64_t) cpssp->regs[R_EDX] << 32) | cpssp->regs[R_EAX])) {
			break;
		}
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT
		if (NAME_(core_mtrr_wrmsr)(cpssp, cpssp->regs[R_ECX], ((uint64_t) cpssp->regs[R_EDX] << 32) | cpssp->regs[R_EAX])) {
			break;
		}
#endif
		faum_log(FAUM_LOG_WARNING, "CPU", "",
			"Writing %08x:%08x to   MSR 0x%x.\n",
			(uint32_t) cpssp->regs[R_EDX], (uint32_t) cpssp->regs[R_EAX], (uint32_t) cpssp->regs[R_ECX]);
		/* should generate a GPF according to em64t manual -
		 * but QEMU doesn't do it either... - FIXME? */
		break;
	}
}

void
NAME_(helper_rdmsr)(struct cpssp *cpssp)
{
	switch ((uint32_t) cpssp->regs[R_ECX]) {
#if 80686 <= CONFIG_CPU
	case 0x8b: /* MSR_IA32_BIOS_SIGN_ID */
		cpssp->regs[R_EAX] = 0;
		cpssp->regs[R_EDX] = cpssp->update_signature;
		break;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_TSC_SUPPORT
	case 0x10: /* MSR_IA32_TSC */
		/* Read TSC (that's the same as "rdtsc"!). */
		NAME_(helper_rdtsc)(cpssp);
		break;
#endif

	case 0x2a: /* MSR_EBL_CR_POWERON */
		/* Processor Hard Poweron Configuration */
		cpssp->regs[R_EAX] = 0 /* FIXME */
			| cpssp->apic_cluster_id << 16
			| cpssp->apic_arbitration_id << 20
			/* FIXME */
			;
		cpssp->regs[R_EDX] = 0;

		faum_log(FAUM_LOG_WARNING, "CPU", "",
			"Reading %08x:%08x from EBL_CR_POWERON.\n",
				(uint32_t) cpssp->regs[R_EDX], (uint32_t) cpssp->regs[R_EAX]);
		break;

	case 0x11e: /* MSR_BBL_CR_CTL3 */
		/* L2 Cache Control Register */
		cpssp->regs[R_EAX] = cpssp->msr[cpssp->regs[R_ECX]].low;
		cpssp->regs[R_EDX] = cpssp->msr[cpssp->regs[R_ECX]].high;

		faum_log(FAUM_LOG_WARNING, "CPU", "",
			"Reading %08x:%08x from BBL_CR_CTL3 MSR 0x%x.\n",
				(uint32_t)cpssp->regs[R_EDX], (uint32_t)cpssp->regs[R_EAX], (uint32_t)cpssp->regs[R_ECX]);
		break;

#if 80486 <= CONFIG_CPU && CONFIG_CPU_SEP_SUPPORT
	case 0x174: /* MSR_IA32_SYSENTER_CS */
		cpssp->regs[R_EAX] = cpssp->sysenter_cs;
		cpssp->regs[R_EDX] = 0;
		break;

	case 0x175: /* MSR_IA32_SYSENTER_ESP */
		cpssp->regs[R_EAX] = cpssp->sysenter_esp;
		cpssp->regs[R_EDX] = 0;
		break;

	case 0x176: /* MSR_IA32_SYSENTER_EIP */
		cpssp->regs[R_EAX] = cpssp->sysenter_eip;
		cpssp->regs[R_EDX] = 0;
		break;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MCA_SUPPORT
	/*
	 * Machine-Check Architecture
	 */
	case 0x179: /* MSR_IA32_MCG_CAP */
		cpssp->regs[R_EAX] = (1 << 8)	/* MCG_CTL available */
		    | (5 << 0);	/* 5 error-reporting banks */
		cpssp->regs[R_EDX] = 0;
		break;

	case 0x17a: /* MSR_IA32_MCG_STATUS */
		cpssp->regs[R_EAX] = (0 << 2)	/* No machine check in progress */
		    | (0 << 1)	/* No error IP valid */
		    | (0 << 0);	/* No restart IP valid */
		cpssp->regs[R_EDX] = 0;
		break;

	case 0x17b: /* MSR_IA32_MCG_CTL */
		/* Just report setting. */
		cpssp->regs[R_EAX] = cpssp->msr[cpssp->regs[R_ECX]].low;
		cpssp->regs[R_EDX] = cpssp->msr[cpssp->regs[R_ECX]].high;
		break;

	case 0x400: /* MSR_IA32_MC0_CTL */
	case 0x404: /* MSR_IA32_MC1_CTL */
	case 0x408: /* MSR_IA32_MC2_CTL */
	case 0x40c: /* MSR_IA32_MC3_CTL */
	case 0x410: /* MSR_IA32_MC4_CTL */
		/* Just report setting. */
		cpssp->regs[R_EAX] = cpssp->msr[cpssp->regs[R_ECX]].low;
		cpssp->regs[R_EDX] = cpssp->msr[cpssp->regs[R_ECX]].high;
		break;

	case 0x401: /* MSR_IA32_MC0_STATUS */
	case 0x405: /* MSR_IA32_MC1_STATUS */
	case 0x409: /* MSR_IA32_MC2_STATUS */
	case 0x40d: /* MSR_IA32_MC3_STATUS */
	case 0x411: /* MSR_IA32_MC4_STATUS */
		cpssp->regs[R_EAX] = 0;	/* No error pending. */
		cpssp->regs[R_EDX] = 0;
		break;

	case 0x402: /* MSR_IA32_MC0_ADDR */
	case 0x406: /* MSR_IA32_MC1_ADDR */
	case 0x40a: /* MSR_IA32_MC2_ADDR */
	case 0x40e: /* MSR_IA32_MC3_ADDR */
	case 0x412: /* MSR_IA32_MC4_ADDR */
		cpssp->regs[R_EAX] = 0;	/* No error pending. */
		cpssp->regs[R_EDX] = 0;
		break;

	case 0x403: /* MSR_IA32_MC0_MISC */
	case 0x407: /* MSR_IA32_MC1_MISC */
	case 0x40b: /* MSR_IA32_MC2_MISC */
	case 0x40f: /* MSR_IA32_MC3_MISC */
	case 0x413: /* MSR_IA32_MC4_MISC */
		cpssp->regs[R_EAX] = 0;	/* Register not implemented. */
		cpssp->regs[R_EDX] = 0;
		break;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_PAT_SUPPORT
	case 0x277: /* MSR_IA32_PAT */
		/* 3-628. */
		cpssp->regs[R_EAX] = (cpssp->pat[0] <<  0)
		    | (cpssp->pat[1] <<  8)
		    | (cpssp->pat[2] << 16)
		    | (cpssp->pat[3] << 24);
		cpssp->regs[R_EDX] = (cpssp->pat[4] <<  0)
		    | (cpssp->pat[5] <<  8)
		    | (cpssp->pat[6] << 16)
		    | (cpssp->pat[7] << 24);
		break;
#endif

	case 0xc0000080: /* MSR_EFER */
		cpssp->regs[R_EAX] = (uint32_t)(cpssp->efer);
		cpssp->regs[R_EDX] = (uint32_t)(cpssp->efer >> 32);
		break;
	case 0xc0000081: /* MSR_STAR */
		cpssp->regs[R_EAX] = (uint32_t)(cpssp->star);
		cpssp->regs[R_EDX] = (uint32_t)(cpssp->star >> 32);
		break;

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	case MSR_LSTAR: /* 0xc0000082 */
		cpssp->regs[R_EAX] = (uint32_t)(cpssp->lstar);
		cpssp->regs[R_EDX] = (uint32_t)(cpssp->lstar >> 32);
		break;
	case MSR_CSTAR: /* 0xc0000083 */
		cpssp->regs[R_EAX] = (uint32_t)(cpssp->cstar);
		cpssp->regs[R_EDX] = (uint32_t)(cpssp->cstar >> 32);
		break;
	case MSR_FMASK: /* 0xc0000084 */
		cpssp->regs[R_EAX] = (uint32_t)(cpssp->fmask);
		cpssp->regs[R_EDX] = (uint32_t)(cpssp->fmask >> 32);
		break;
	case MSR_FSBASE: /* 0xc0000100 */
		cpssp->regs[R_EAX] = (uint32_t)(cpssp->segs[R_FS].base);
		cpssp->regs[R_EDX] = (uint32_t)(cpssp->segs[R_FS].base >> 32);
		break;
	case MSR_GSBASE: /* 0xc0000101 */
		cpssp->regs[R_EAX] = (uint32_t)(cpssp->segs[R_GS].base);
		cpssp->regs[R_EDX] = (uint32_t)(cpssp->segs[R_GS].base >> 32);
		break;
	case MSR_KERNELGSBASE: /* 0xc0000102 */
		cpssp->regs[R_EAX] = (uint32_t)(cpssp->kernelgsbase);
		cpssp->regs[R_EDX] = (uint32_t)(cpssp->kernelgsbase >> 32);
		break;
#endif

	default:
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	    {
		uint64_t val;

		if (NAME_(core_apic_rdmsr)(cpssp, cpssp->regs[R_ECX], &val)) {
			cpssp->regs[R_EAX] = (uint32_t) (val >>  0);
			cpssp->regs[R_EDX] = (uint32_t) (val >> 32);
			break;
		}
	    }
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT
	    {
		uint64_t val;

		if (NAME_(core_mtrr_rdmsr)(cpssp, cpssp->regs[R_ECX], &val)) {
			cpssp->regs[R_EAX] = (uint32_t) (val >>  0);
			cpssp->regs[R_EDX] = (uint32_t) (val >> 32);
			break;
		}
	    }
#endif
		/* should generate a GPF according to em64t manual -
		 * but QEMU doesn't do it either... - FIXME? */
		cpssp->regs[R_EAX] = 0;
		cpssp->regs[R_EDX] = 0;
		faum_log(FAUM_LOG_WARNING, "CPU", "",
			"Reading %08x:%08x from MSR 0x%x.\n",
			(uint32_t) cpssp->regs[R_EDX], (uint32_t) cpssp->regs[R_EAX], (uint32_t) cpssp->regs[R_ECX]);
		break;
	}
}
#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_MSR_SUPPORT */

uint32_t
NAME_(helper_lsl)(struct cpssp *cpssp, uint16_t t0)
{
    unsigned int selector, limit;
    uint32_t e1, e2, eflags;
    int rpl, dpl, cpl, type;

    eflags = NAME_(cc_table[cpssp->cc_op]).compute_all(cpssp);
    selector = t0 & 0xffff;
    if (load_segment(cpssp, &e1, &e2, selector) != 0)
	goto fail;
    rpl = selector & 3;
    dpl = (e2 >> DESC_DPL_SHIFT) & 3;
    cpl = cpssp->hflags & HF_CPL_MASK;
    if (e2 & DESC_S_MASK) {
	if ((e2 & DESC_CS_MASK) && (e2 & DESC_C_MASK)) {
	    /* conforming */
	} else {
	    if (dpl < cpl || dpl < rpl)
		goto fail;
	}
    } else {
	type = (e2 >> DESC_TYPE_SHIFT) & 0xf;
	switch(type) {
	case 1:
	case 2:
	case 3:
	case 9:
	case 11:
	    break;
	default:
	    goto fail;
	}
	if (dpl < cpl || dpl < rpl) {
	fail:
	    cpssp->cc_src = eflags & ~CC_Z;
	    return 0; /* Correct? FIXME */
	}
    }
    limit = get_seg_limit(e1, e2);
    cpssp->cc_src = eflags | CC_Z;
    return limit;
}

uint32_t
NAME_(helper_lar)(struct cpssp *cpssp, uint16_t t0)
{
    unsigned int selector;
    uint32_t e1, e2, eflags;
    int rpl, dpl, cpl, type;

    eflags = NAME_(cc_table[cpssp->cc_op]).compute_all(cpssp);
    selector = t0 & 0xffff;
    if ((selector & 0xfffc) == 0)
	goto fail;
    if (load_segment(cpssp, &e1, &e2, selector) != 0)
	goto fail;
    rpl = selector & 3;
    dpl = (e2 >> DESC_DPL_SHIFT) & 3;
    cpl = cpssp->hflags & HF_CPL_MASK;
    if (e2 & DESC_S_MASK) {
	if ((e2 & DESC_CS_MASK) && (e2 & DESC_C_MASK)) {
	    /* conforming */
	} else {
	    if (dpl < cpl || dpl < rpl)
		goto fail;
	}
    } else {
	type = (e2 >> DESC_TYPE_SHIFT) & 0xf;
	switch(type) {
	case 1:
	case 2:
	case 3:
	case 4:
	case 5:
	case 9:
	case 11:
	case 12:
	    break;
	default:
	    goto fail;
	}
	if (dpl < cpl || dpl < rpl) {
	fail:
	    cpssp->cc_src = eflags & ~CC_Z;
	    return 0; /* Correct? FIXME */
	}
    }
    cpssp->cc_src = eflags | CC_Z;
    return e2 & 0x00f0ff00;
}

void
NAME_(helper_verr)(struct cpssp *cpssp, uint16_t t0)
{
    unsigned int selector;
    uint32_t e1, e2, eflags;
    int rpl, dpl, cpl;

    eflags = NAME_(cc_table[cpssp->cc_op]).compute_all(cpssp);
    selector = t0 & 0xffff;
    if ((selector & 0xfffc) == 0)
	goto fail;
    if (load_segment(cpssp, &e1, &e2, selector) != 0)
	goto fail;
    if (!(e2 & DESC_S_MASK))
	goto fail;
    rpl = selector & 3;
    dpl = (e2 >> DESC_DPL_SHIFT) & 3;
    cpl = cpssp->hflags & HF_CPL_MASK;
    if (e2 & DESC_CS_MASK) {
	if (!(e2 & DESC_R_MASK))
	    goto fail;
	if (!(e2 & DESC_C_MASK)) {
	    if (dpl < cpl || dpl < rpl)
		goto fail;
	}
    } else {
	if (dpl < cpl || dpl < rpl) {
	fail:
	    cpssp->cc_src = eflags & ~CC_Z;
	    return;
	}
    }
    cpssp->cc_src = eflags | CC_Z;
}

void
NAME_(helper_verw)(struct cpssp *cpssp, uint16_t t0)
{
    unsigned int selector;
    uint32_t e1, e2, eflags;
    int rpl, dpl, cpl;

    eflags = NAME_(cc_table[cpssp->cc_op]).compute_all(cpssp);
    selector = t0 & 0xffff;
    if ((selector & 0xfffc) == 0)
	goto fail;
    if (load_segment(cpssp, &e1, &e2, selector) != 0)
	goto fail;
    if (!(e2 & DESC_S_MASK))
	goto fail;
    rpl = selector & 3;
    dpl = (e2 >> DESC_DPL_SHIFT) & 3;
    cpl = cpssp->hflags & HF_CPL_MASK;
    if (e2 & DESC_CS_MASK) {
	goto fail;
    } else {
	if (dpl < cpl || dpl < rpl)
	    goto fail;
	if (!(e2 & DESC_W_MASK)) {
	fail:
	    cpssp->cc_src = eflags & ~CC_Z;
	    return;
	}
    }
    cpssp->cc_src = eflags | CC_Z;
}

#if 80386 <= CONFIG_CPU
void
NAME_(helper_rsm)(struct cpssp *cpssp)
{
	uint32_t cr0;
	uint32_t cr3;
	uint32_t base;
	uint32_t lval;
	int i;

	/*
	 * read back CPU status
	 */
	base = cpssp->core.smbase + 0x8000;
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	/* Offset   Register */
	/* 7DCFH - 7C00H Reserved  - we'll store other internals here: */
	/*
	 * restore hidden info from reserved space
	 * FIXME: perhaps there is even more than this :)
	 */
	cpssp->tr.flags = NAME_(mr_data_q)(cpssp, base+0x7d20);
	cpssp->tr.limit = NAME_(mr_data_q)(cpssp, base+0x7d28);
	cpssp->tr.base = NAME_(mr_data_q)(cpssp, base+0x7d30);
	cpssp->segs[R_SS].flags = NAME_(mr_data_q)(cpssp, base+0x7d38);
	cpssp->segs[R_GS].flags = NAME_(mr_data_q)(cpssp, base+0x7d40);
	cpssp->segs[R_FS].flags = NAME_(mr_data_q)(cpssp, base+0x7d48);
	cpssp->segs[R_ES].flags = NAME_(mr_data_q)(cpssp, base+0x7d50);
	cpssp->segs[R_DS].flags = NAME_(mr_data_q)(cpssp, base+0x7d58);
	cpssp->segs[R_CS].flags = NAME_(mr_data_q)(cpssp, base+0x7d60);
	cpssp->segs[R_SS].limit = NAME_(mr_data_q)(cpssp, base+0x7d68);
	cpssp->segs[R_GS].limit = NAME_(mr_data_q)(cpssp, base+0x7d70);
	cpssp->segs[R_FS].limit = NAME_(mr_data_q)(cpssp, base+0x7d78);
	cpssp->segs[R_ES].limit = NAME_(mr_data_q)(cpssp, base+0x7d80);
	cpssp->segs[R_DS].limit = NAME_(mr_data_q)(cpssp, base+0x7d88);
	cpssp->segs[R_CS].limit = NAME_(mr_data_q)(cpssp, base+0x7d90);
	cpssp->segs[R_SS].base = NAME_(mr_data_q)(cpssp, base+0x7d98);
	cpssp->segs[R_GS].base = NAME_(mr_data_q)(cpssp, base+0x7da0);
	cpssp->segs[R_FS].base = NAME_(mr_data_q)(cpssp, base+0x7da8);
	cpssp->segs[R_ES].base = NAME_(mr_data_q)(cpssp, base+0x7db0);
	cpssp->segs[R_DS].base = NAME_(mr_data_q)(cpssp, base+0x7db8);
	cpssp->segs[R_CS].base = NAME_(mr_data_q)(cpssp, base+0x7dc0);
	NAME_(update_cr4)(cpssp, NAME_(mr_data_q)(cpssp, base+0x7dc8));
	/* 7EF8H    SMBASE Field (Doubleword) */
	cpssp->core.smbase = NAME_(mr_data_l)(cpssp, base+0x7ef8);
	/* 7EFCH    SMM Revision Identifier Field (Doubleword) */

	lval = NAME_(mr_data_l)(cpssp, base+0x7f00);
	/* 7F00H    I/O Instruction Restart Field (Word) */
		/* FIXME */
	/* 7F02H    Auto HALT Restart Field (Word) */
	cpssp->hflags |= ((lval >> 16) & 1) << HF_HALTED_SHIFT;

	/* 7F04H    IEDBASE */
	/* 7F08H - 7F1BH  Reserved */
	/* 7F1CH    R15 */
	cpssp->regs[15] = NAME_(mr_data_q)(cpssp, base+0x7f1c);
	/* 7F24H    R14 */
	cpssp->regs[14] = NAME_(mr_data_q)(cpssp, base+0x7f24);
	/* 7F2CH    R13 */
	cpssp->regs[13] = NAME_(mr_data_q)(cpssp, base+0x7f2c);
	/* 7F34H    R12 */
	cpssp->regs[12] = NAME_(mr_data_q)(cpssp, base+0x7f34);
	/* 7F3CH    R11 */
	cpssp->regs[11] = NAME_(mr_data_q)(cpssp, base+0x7f3c);
	/* 7F44H    R10 */
	cpssp->regs[10] = NAME_(mr_data_q)(cpssp, base+0x7f44);
	/* 7F4CH    R9 */
	cpssp->regs[9] = NAME_(mr_data_q)(cpssp, base+0x7f4c);
	/* 7F54H    R8 */
	cpssp->regs[8] = NAME_(mr_data_q)(cpssp, base+0x7f54);
	/* 7F5CH    RAX */
	cpssp->regs[R_EAX] = NAME_(mr_data_q)(cpssp, base+0x7f5c);
	/* 7F64H    RCX */
	cpssp->regs[R_ECX] = NAME_(mr_data_q)(cpssp, base+0x7f64);
	/* 7F6CH    RDX */
	cpssp->regs[R_EDX] = NAME_(mr_data_q)(cpssp, base+0x7f6c);
	/* 7F74H    RBX */
	cpssp->regs[R_EBX] = NAME_(mr_data_q)(cpssp, base+0x7f74);
	/* 7F7CH    RSP */
	cpssp->regs[R_ESP] = NAME_(mr_data_q)(cpssp, base+0x7f7c);
	/* 7F84H    RBP */
	cpssp->regs[R_EBP] = NAME_(mr_data_q)(cpssp, base+0x7f84);
	/* 7F8CH    RSI */
	cpssp->regs[R_ESI] = NAME_(mr_data_q)(cpssp, base+0x7f8c);
	/* 7F94H    RDI */
	cpssp->regs[R_EDI] = NAME_(mr_data_q)(cpssp, base+0x7f94);
	/* 7F9CH    IO_MEM_ADDR */
	/* 7FA4H    IO_MISC */
	/* 7FA8H    ES SEL */
	cpssp->segs[R_ES].selector = NAME_(mr_data_l)(cpssp, base+0x7fa8);
	/* 7FACH    CS SEL */
	cpssp->segs[R_CS].selector = NAME_(mr_data_l)(cpssp, base+0x7fac);
	/* 7FB0H    SS SEL */
	cpssp->segs[R_SS].selector = NAME_(mr_data_l)(cpssp, base+0x7fb0);
	/* 7FB4H    DS SEL */
	cpssp->segs[R_DS].selector = NAME_(mr_data_l)(cpssp, base+0x7fb4);
	/* 7FB8H    FS SEL */
	cpssp->segs[R_FS].selector = NAME_(mr_data_l)(cpssp, base+0x7fb8);
	/* 7FBCH    GS SEL */
	cpssp->segs[R_GS].selector = NAME_(mr_data_l)(cpssp, base+0x7fbc);
	/* 7FC0H    LDTR SEL */
	/* 7FC4H    TR SEL */
	cpssp->tr.selector = NAME_(mr_data_l)(cpssp, base+0x7fc4);
	/* 7FE8H    RFLAGS */
	load_eflags(cpssp, NAME_(mr_data_q)(cpssp, base+0x7fe8), ~(CC_O | CC_S | CC_Z | CC_A | CC_P | CC_C));
	/* 7FE0H    IA32_EFER */
	cpssp->efer = NAME_(mr_data_q)(cpssp, base+0x7fe0);
	/* 7FD8H    RIP */
	cpssp->eip = NAME_(mr_data_q)(cpssp, base+0x7fd8);
	/* 7FD0H    DR6 */
	cpssp->dr[6] = NAME_(mr_data_q)(cpssp, base+0x7fd0);
	/* 7FC8H    DR7 */
	cpssp->dr[7] = NAME_(mr_data_q)(cpssp, base+0x7fc8);
	/* 7FF8H    CR0 */
	cr0 = NAME_(mr_data_q)(cpssp, base+0x7ff8);
	/* 7FF0H    CR3 */
	cr3 = NAME_(mr_data_q)(cpssp, base+0x7ff0);
	NAME_(update_cr0)(cpssp, cr0);
	NAME_(update_cr3)(cpssp, cr3);
#else /* not (CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT) */
	/* Offset   Register */
	/* 7EF7H-7E00H Reserved */
	/*
	 * restore hidden info from reserved space
	 * FIXME: perhaps there is even more than this :)
	 */
	cpssp->tr.flags = NAME_(mr_data_l)(cpssp, base+0x7ea0);
	cpssp->tr.limit = NAME_(mr_data_l)(cpssp, base+0x7ea4);
	cpssp->tr.base = NAME_(mr_data_l)(cpssp, base+0x7ea8);
	cpssp->segs[R_SS].flags = NAME_(mr_data_l)(cpssp, base+0x7eac);
	cpssp->segs[R_GS].flags = NAME_(mr_data_l)(cpssp, base+0x7eb0);
	cpssp->segs[R_FS].flags = NAME_(mr_data_l)(cpssp, base+0x7eb4);
	cpssp->segs[R_ES].flags = NAME_(mr_data_l)(cpssp, base+0x7eb8);
	cpssp->segs[R_DS].flags = NAME_(mr_data_l)(cpssp, base+0x7ebc);
	cpssp->segs[R_CS].flags = NAME_(mr_data_l)(cpssp, base+0x7ec0);
	cpssp->segs[R_SS].limit = NAME_(mr_data_l)(cpssp, base+0x7ec4);
	cpssp->segs[R_GS].limit = NAME_(mr_data_l)(cpssp, base+0x7ec8);
	cpssp->segs[R_FS].limit = NAME_(mr_data_l)(cpssp, base+0x7ecc);
	cpssp->segs[R_ES].limit = NAME_(mr_data_l)(cpssp, base+0x7ed0);
	cpssp->segs[R_DS].limit = NAME_(mr_data_l)(cpssp, base+0x7ed4);
	cpssp->segs[R_CS].limit = NAME_(mr_data_l)(cpssp, base+0x7ed8);
	cpssp->segs[R_SS].base = NAME_(mr_data_l)(cpssp, base+0x7edc);
	cpssp->segs[R_GS].base = NAME_(mr_data_l)(cpssp, base+0x7ee0);
	cpssp->segs[R_FS].base = NAME_(mr_data_l)(cpssp, base+0x7ee4);
	cpssp->segs[R_ES].base = NAME_(mr_data_l)(cpssp, base+0x7ee8);
	cpssp->segs[R_DS].base = NAME_(mr_data_l)(cpssp, base+0x7eec);
	cpssp->segs[R_CS].base = NAME_(mr_data_l)(cpssp, base+0x7ef0);
	NAME_(update_cr4)(cpssp, NAME_(mr_data_l)(cpssp, base+0x7ef4));
	/* 7EF8H    SMBASE Field (Doubleword) */
	cpssp->core.smbase = NAME_(mr_data_l)(cpssp, base+0x7ef8);

	lval = NAME_(mr_data_l)(cpssp, base+0x7f00);
	/* 7F00H    I/O Instruction Restart Field (Word) */
		/* FIXME */
	/* 7F02H    Auto HALT Restart Field (Word) */
	cpssp->hflags |= ((lval >> 16) & 1) << HF_HALTED_SHIFT;

	/* 7FA0H    I/O Memory Address Field, see Section 13.7 */
	/* 7FA4H    I/O State Field, see Section 13.7 */
	/* I/O Support, I/O restart and auto HALT unsupported */
	/* 7FA8H    ES* */
	cpssp->segs[R_ES].selector = NAME_(mr_data_l)(cpssp, base+0x7fa8);
	/* 7FACH    CS* */
	cpssp->segs[R_CS].selector = NAME_(mr_data_l)(cpssp, base+0x7fac);
	/* 7FB0H    SS* */
	cpssp->segs[R_SS].selector = NAME_(mr_data_l)(cpssp, base+0x7fb0);
	/* 7FB4H    DS* */
	cpssp->segs[R_DS].selector = NAME_(mr_data_l)(cpssp, base+0x7fb4);
	/* 7FB8H    FS* */
	cpssp->segs[R_FS].selector = NAME_(mr_data_l)(cpssp, base+0x7fb8);
	/* 7FBCH    GS* */
	cpssp->segs[R_GS].selector = NAME_(mr_data_l)(cpssp, base+0x7fbc);
	/* 7FC0H    Reserved */
	/* 7FC4H    TR* */
	cpssp->tr.selector = NAME_(mr_data_l)(cpssp, base+0x7fc4);
	/* 7FC8H    DR7 */
	cpssp->dr[7] = NAME_(mr_data_l)(cpssp, base+0x7fc8);
	/* 7FCCH    DR6 */
	cpssp->dr[6] = NAME_(mr_data_l)(cpssp, base+0x7fcc);
	/* 7FD0H    EAX */
	cpssp->regs[R_EAX] = NAME_(mr_data_l)(cpssp, base+0x7fd0);
	/* 7FD4H    ECX */
	cpssp->regs[R_ECX] = NAME_(mr_data_l)(cpssp, base+0x7fd4);
	/* 7FD8H    EDX */
	cpssp->regs[R_EDX] = NAME_(mr_data_l)(cpssp, base+0x7fd8);
	/* 7FDCH    EBX */
	cpssp->regs[R_EBX] = NAME_(mr_data_l)(cpssp, base+0x7fdc);
	/* 7FE0H    ESP */
	cpssp->regs[R_ESP] = NAME_(mr_data_l)(cpssp, base+0x7fe0);
	/* 7FE4H    EBP */
	cpssp->regs[R_EBP] = NAME_(mr_data_l)(cpssp, base+0x7fe4);
	/* 7FE8H    ESI */
	cpssp->regs[R_ESI] = NAME_(mr_data_l)(cpssp, base+0x7fe8);
	/* 7FECH    EDI */
	cpssp->regs[R_EDI] = NAME_(mr_data_l)(cpssp, base+0x7fec);
	/* 7FF0H    EIP */
	cpssp->eip = NAME_(mr_data_l)(cpssp, base+0x7ff0);
	/* 7FF4H    EFLAGS */
	load_eflags(cpssp, NAME_(mr_data_l)(cpssp, base+0x7ff4), ~(CC_O | CC_S | CC_Z | CC_A | CC_P | CC_C));
	/* 7FF8H    CR3 */
	/* 7FFCH    CR0 */
	cr0 = NAME_(mr_data_l)(cpssp, base+0x7ffc);
	cr3 = NAME_(mr_data_l)(cpssp, base+0x7ff8);
	NAME_(update_cr0)(cpssp, cr0);
	/* cr3 update must be *after* cr0 update */
	NAME_(update_cr3)(cpssp, cr3);
#endif /* not (CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT) */

#if 0 /* no debugging messages in "production" version :) */
	faum_log(FAUM_LOG_DEBUG, "CPU", "",
			"RSM called... SMBASE now %08x\n", cpssp->core.smbase);
#endif
	/*
	 * restore qemu special flags
	 */
	/*
	 * FIXME: maybe there's a better way than this
	 *	or even more to do here
	 */
	for (i=0; i < CPU_NB_SEGS; i++) {
		cpu_x86_load_seg_cache(cpssp, i, cpssp->segs[i].selector,
				cpssp->segs[i].base, cpssp->segs[i].limit,
				cpssp->segs[i].flags);
	}
	/*
	 * ...and now we're back from SMM
	 */
	cpssp->core.smm = 0;

	/* Host bridge changes memory on non-smm/smm changes. */
	NAME_(core_mmu_tlb_flush)(cpssp, 1);
}
#endif /* 80386 <= CONFIG_CPU */

#define SHIFT 0
#include "arch_gen_cpu_x86_core_fast_template.h"
#undef SHIFT

#define SHIFT 1
#include "arch_gen_cpu_x86_core_fast_template.h"
#undef SHIFT

#if CONFIG_CPU >= 80386
#define SHIFT 2
#include "arch_gen_cpu_x86_core_fast_template.h"
#undef SHIFT
#endif

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
#define SHIFT 3
#include "arch_gen_cpu_x86_core_fast_template.h"
#undef SHIFT
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */

static int compute_all_eflags(struct cpssp *cpssp)
{
    return cpssp->cc_src;
}

static int compute_c_eflags(struct cpssp *cpssp)
{
    return cpssp->cc_src & CC_C;
}

const CCTable NAME_(cc_table)[CC_OP_NB] = {
    [CC_OP_DYNAMIC] = { /* should never happen */ },

    [CC_OP_EFLAGS] = { compute_all_eflags, compute_c_eflags },

    [CC_OP_MULB] = { compute_all_mulb, compute_c_mulb },
    [CC_OP_MULW] = { compute_all_mulw, compute_c_mulw },
#if CONFIG_CPU >= 80386
    [CC_OP_MULL] = { compute_all_mull, compute_c_mull },
#endif
    [CC_OP_ADDB] = { compute_all_addb, compute_c_addb },
    [CC_OP_ADDW] = { compute_all_addw, compute_c_addw  },
#if CONFIG_CPU >= 80386
    [CC_OP_ADDL] = { compute_all_addl, compute_c_addl  },
#endif

    [CC_OP_ADCB] = { compute_all_adcb, compute_c_adcb },
    [CC_OP_ADCW] = { compute_all_adcw, compute_c_adcw  },
#if CONFIG_CPU >= 80386
    [CC_OP_ADCL] = { compute_all_adcl, compute_c_adcl  },
#endif
    [CC_OP_SUBB] = { compute_all_subb, compute_c_subb  },
    [CC_OP_SUBW] = { compute_all_subw, compute_c_subw  },
#if CONFIG_CPU >= 80386
    [CC_OP_SUBL] = { compute_all_subl, compute_c_subl  },
#endif
    [CC_OP_SBBB] = { compute_all_sbbb, compute_c_sbbb  },
    [CC_OP_SBBW] = { compute_all_sbbw, compute_c_sbbw  },
#if CONFIG_CPU >= 80386
    [CC_OP_SBBL] = { compute_all_sbbl, compute_c_sbbl  },
#endif
    [CC_OP_LOGICB] = { compute_all_logicb, compute_c_logicb },
    [CC_OP_LOGICW] = { compute_all_logicw, compute_c_logicw },
#if CONFIG_CPU >= 80386
    [CC_OP_LOGICL] = { compute_all_logicl, compute_c_logicl },
#endif
    [CC_OP_INCB] = { compute_all_incb, compute_c_incb },
    [CC_OP_INCW] = { compute_all_incw, compute_c_incw },
#if CONFIG_CPU >= 80386
    [CC_OP_INCL] = { compute_all_incl, compute_c_incl },
#endif
    [CC_OP_DECB] = { compute_all_decb, compute_c_incb },
    [CC_OP_DECW] = { compute_all_decw, compute_c_incw },
#if CONFIG_CPU >= 80386
    [CC_OP_DECL] = { compute_all_decl, compute_c_incl },
#endif
    [CC_OP_SHLB] = { compute_all_shlb, compute_c_shlb },
    [CC_OP_SHLW] = { compute_all_shlw, compute_c_shlw },
#if CONFIG_CPU >= 80386
    [CC_OP_SHLL] = { compute_all_shll, compute_c_shll },
#endif
    [CC_OP_SARB] = { compute_all_sarb, compute_c_sarb },
    [CC_OP_SARW] = { compute_all_sarw, compute_c_sarw },
#if CONFIG_CPU >= 80386
    [CC_OP_SARL] = { compute_all_sarl, compute_c_sarl },
#endif

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
    [CC_OP_MULQ] = { compute_all_mulq, compute_c_mulq },

    [CC_OP_ADDQ] = { compute_all_addq, compute_c_addq  },

    [CC_OP_ADCQ] = { compute_all_adcq, compute_c_adcq  },

    [CC_OP_SUBQ] = { compute_all_subq, compute_c_subq  },

    [CC_OP_SBBQ] = { compute_all_sbbq, compute_c_sbbq  },

    [CC_OP_LOGICQ] = { compute_all_logicq, compute_c_logicq },

    [CC_OP_INCQ] = { compute_all_incq, compute_c_incq },

    [CC_OP_DECQ] = { compute_all_decq, compute_c_incq },

    [CC_OP_SHLQ] = { compute_all_shlq, compute_c_shlq },

    [CC_OP_SARQ] = { compute_all_sarq, compute_c_sarq },

    [CC_OP_DIVQ] = { compute_all_divq, compute_c_divq  },
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */

    [CC_OP_DIVB] = { compute_all_divb, compute_c_divb  },
    [CC_OP_DIVW] = { compute_all_divw, compute_c_divw  },
#if CONFIG_CPU >= 80386
    [CC_OP_DIVL] = { compute_all_divl, compute_c_divl  },
#endif
};

/*
 * 8-3
 */
static void
NAME_(core_do_reset)(struct cpssp *cpssp, int reset_flag)
{
	int i;

	if (reset_flag) {
		NAME_(apic_n_reset_set)(cpssp, 0);
		NAME_(cache1i_n_reset_set)(cpssp, 0);
		NAME_(cache1d_n_reset_set)(cpssp, 0);
		NAME_(mmui_n_reset_set)(cpssp, 0);
		NAME_(mmud_n_reset_set)(cpssp, 0);
		NAME_(a20gate_n_reset_set)(cpssp, 0);
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT
		NAME_(mtrr_n_reset_set)(cpssp, 0);
#endif
		NAME_(cache2_n_reset_set)(cpssp, 0);
	}

	cpssp->hflags = 0;
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if (! cpssp->apic.bsp) { /* FIXME */
		cpssp->hflags |= HF_WAITING_FOR_STARTUP_MASK;
	}
#endif

	/*
	 * must do it in this sequence:
	 * - first initialize real-mode (cr0)
	 * - then initialize segment registers
	 */
	cpssp->eflags = 0x00000002; /* reserved bit */
	cpssp->cc_src = cpssp->eflags
		& (CC_O | CC_S | CC_Z | CC_A | CC_P | CC_C);
	cpssp->df = 1 - (2 * ((cpssp->eflags >> 10) & 1));
	cpssp->cc_op = CC_OP_EFLAGS;
	cpssp->eflags
		&= ~(CPU_DF_MASK | CC_O | CC_S | CC_Z | CC_A | CC_P | CC_C);

	NAME_(update_cr0)(cpssp, 0x60000010);	/* write through */
						/* caching disabled */
						/* reserved bit */
	/* NAME_(update_cr2)(cpssp, 0x00000000); */
	NAME_(update_cr3)(cpssp, 0x00000000);
	NAME_(update_cr4)(cpssp, 0x00000000);

	cpssp->exception_index = -1;
	if (reset_flag) {
		cpssp->interrupt_request = 0;
	}

	/* code segment register */
#if CONFIG_CPU <= 80286
	cpu_x86_load_seg_cache(cpssp, R_CS, 0xf000, 0x000f0000, 0xffff, 0);
#else
	cpu_x86_load_seg_cache(cpssp, R_CS, 0xf000, 0xffff0000, 0xffff, 0);
#endif

	/* instruction pointer */
	cpssp->eip = 0x0000fff0;

	/* other segment registers */
	cpu_x86_load_seg_cache(cpssp, R_DS, 0, 0, 0xffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_ES, 0, 0, 0xffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_SS, 0, 0, 0xffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_FS, 0, 0, 0xffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_GS, 0, 0, 0xffff, 0);

	/* edx - depends on processor version - FIXME VOSSI */
	cpssp->regs[R_EDX] = 0x00000600; /* indicate P6 processor */

	/* eax - should contain BIST code - FIXME VOSSI */
	cpssp->regs[R_EAX] = 0x00000000;

	/* other registers */
	cpssp->regs[R_EBX] = 0x00000000;
	cpssp->regs[R_ECX] = 0x00000000;
	cpssp->regs[R_ESI] = 0x00000000;
	cpssp->regs[R_EDI] = 0x00000000;
	cpssp->regs[R_EBP] = 0x00000000;
	cpssp->regs[R_ESP] = 0x00000000;

	if (reset_flag) {
		/* FPU init */
		cpssp->fpstt = 0;
		cpssp->fpus = 0;
		cpssp->fpuc = 0x37f;
		for (i = 0; i < 8; i++) {
			cpssp->fptags[i] = 1;
		}
		cpssp->mxcsr = 0;
		memset(cpssp->xmm_regs, 0, sizeof(cpssp->xmm_regs));
		memset(&cpssp->xmm_t0, 0, sizeof(cpssp->xmm_t0));
		memset(&cpssp->mmx_t0, 0, sizeof(cpssp->mmx_t0));
	}

	/* gdtr, idtr */
	cpssp->gdt.base = 0x00000000;
	cpssp->gdt.limit = 0xffff;
	cpssp->idt.base = 0x00000000;
	cpssp->idt.limit = 0xffff;

	/* ldtr, task register */
	cpssp->ldt.base = 0x00000000;
	cpssp->ldt.limit = 0xffff;
	cpssp->ldt.flags = DESC_P_MASK;

	cpssp->tr.base = 0x00000000;
	cpssp->tr.limit = 0xffff;
	cpssp->tr.flags = DESC_P_MASK;

	/* dr0 - dr3 */
	cpssp->dr[0] = 0;
	cpssp->dr[1] = 0;
	cpssp->dr[2] = 0;
	cpssp->dr[3] = 0;

	/* dr6 */
	cpssp->dr[6] = 0;

	/* dr7 */
	cpssp->dr[7] = 0;

	/* time-stamp counter */
	/* FIXME */

	/* perf. counters and event select */
	/* FIXME VOSSI */

	/* all other MSRs */
#if CONFIG_CPU >= 80486 && CONFIG_CPU_SEP_SUPPORT
	cpssp->sysenter_cs = 0;
	cpssp->sysenter_esp = 0;
	cpssp->sysenter_eip = 0;
#endif
	cpssp->efer = 0;
	cpssp->star = 0;
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	cpssp->lstar = 0;
	cpssp->cstar = 0;
	cpssp->fmask = 0;
	cpssp->kernelgsbase = 0;
#endif
	memset(cpssp->msr, 0, sizeof(cpssp->msr));

#if 80486 <= CONFIG_CPU && CONFIG_CPU_PAT_SUPPORT
	/* PAT Table */
	cpssp->pat[0] = 6; /* Write back */
	cpssp->pat[1] = 4; /* Write through */
	cpssp->pat[2] = 7; /* Uncached */
	cpssp->pat[3] = 0; /* Uncacheable */
	cpssp->pat[4] = 6; /* Write back */
	cpssp->pat[5] = 4; /* Write through */
	cpssp->pat[6] = 7; /* Uncached */
	cpssp->pat[7] = 0; /* Uncacheable */
#endif

	/* machine-check architecture */
	/* FIXME VOSSI */

	/* System Management */
	if (reset_flag) {
		cpssp->core.smbase = 0x30000;
	}
	cpssp->core.smm = 0;

	/* APIC */
	/* FIXME VOSSI */
}

static void
NAME_(core_reset)(struct cpssp *cpssp)
{
	NAME_(core_do_reset)(cpssp, 1);
}

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
static void
NAME_(core_init)(struct cpssp *cpssp)
{
	NAME_(core_do_reset)(cpssp, 0);
}
#endif

#if 80386 <= CONFIG_CPU
/*
 * handle SMI by writing CPU state to SMRAM and setting up SMI environment
 */
void
NAME_(do_smm)(struct cpssp *cpssp)
{
	uint32_t base; /* This is correct, it's 32 Bit even on AMD64! */
	uint32_t cr0;

	cpssp->core.smm = 1;

	/* Host bridge changes memory on non-smm/smm changes. */
	NAME_(core_mmu_tlb_flush)(cpssp, 1);

	/*
	 * Must switch of paging first.
	 */
	cr0 = cpssp->cr[0];
	cpssp->cr[0] &= ~CPU_CR0_PE_MASK;
	cpssp->cr[0] &= ~CPU_CR0_EM_MASK;
	cpssp->cr[0] &= ~CPU_CR0_TS_MASK;
	cpssp->cr[0] &= ~CPU_CR0_PG_MASK;
	NAME_(update_cr0)(cpssp, cpssp->cr[0]);

	/*
	 * write current cpu status to SMBASE+0x8000+Offset:
	 */
	base = cpssp->core.smbase + 0x8000;

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	/* Source: EM64T Software Developers Guide Vol. 2, Appendix A and E
	 * (they really put the exact same table into two different
	 * appendixes :)  ) */
	/* Offset   Register */
	/* 7FF8H    CR0 */
	NAME_(mw_data_q)(cpssp, base+0x7ff8, cr0);
	/* 7FF0H    CR3 */
	NAME_(mw_data_q)(cpssp, base+0x7ff0, cpssp->cr[3]);
	/* 7FE8H    RFLAGS */
	NAME_(mw_data_q)(cpssp, base+0x7fe8, compute_eflags(cpssp));
	/* 7FE0H    IA32_EFER */
	NAME_(mw_data_q)(cpssp, base+0x7fe0, cpssp->efer);
	/* 7FD8H    RIP */
	NAME_(mw_data_q)(cpssp, base+0x7fd8, cpssp->eip);
	/* 7FD0H    DR6 */
	NAME_(mw_data_q)(cpssp, base+0x7fd0, cpssp->dr[6]);
	/* 7FC8H    DR7 */
	NAME_(mw_data_q)(cpssp, base+0x7fc8, cpssp->dr[7]);
	/* 7FC4H    TR SEL */
	NAME_(mw_data_l)(cpssp, base+0x7fc4, cpssp->tr.selector);
	/* 7FC0H    LDTR SEL */
	/* 7FBCH    GS SEL */
	NAME_(mw_data_l)(cpssp, base+0x7fbc, cpssp->segs[R_GS].selector);
	/* 7FB8H    FS SEL */
	NAME_(mw_data_l)(cpssp, base+0x7fb8, cpssp->segs[R_FS].selector);
	/* 7FB4H    DS SEL */
	NAME_(mw_data_l)(cpssp, base+0x7fb4, cpssp->segs[R_DS].selector);
	/* 7FB0H    SS SEL */
	NAME_(mw_data_l)(cpssp, base+0x7fb0, cpssp->segs[R_SS].selector);
	/* 7FACH    CS SEL */
	NAME_(mw_data_l)(cpssp, base+0x7fac, cpssp->segs[R_CS].selector);
	/* 7FA8H    ES SEL */
	NAME_(mw_data_l)(cpssp, base+0x7fa8, cpssp->segs[R_ES].selector);
	/* 7FA4H    IO_MISC */
	/* 7F9CH    IO_MEM_ADDR */
	/* 7F94H    RDI */
	NAME_(mw_data_q)(cpssp, base+0x7f94, cpssp->regs[R_EDI]);
	/* 7F8CH    RSI */
	NAME_(mw_data_q)(cpssp, base+0x7f8c, cpssp->regs[R_ESI]);
	/* 7F84H    RBP */
	NAME_(mw_data_q)(cpssp, base+0x7f84, cpssp->regs[R_EBP]);
	/* 7F7CH    RSP */
	NAME_(mw_data_q)(cpssp, base+0x7f7c, cpssp->regs[R_ESP]);
	/* 7F74H    RBX */
	NAME_(mw_data_q)(cpssp, base+0x7f74, cpssp->regs[R_EBX]);
	/* 7F6CH    RDX */
	NAME_(mw_data_q)(cpssp, base+0x7f6c, cpssp->regs[R_EDX]);
	/* 7F64H    RCX */
	NAME_(mw_data_q)(cpssp, base+0x7f64, cpssp->regs[R_ECX]);
	/* 7F5CH    RAX */
	NAME_(mw_data_q)(cpssp, base+0x7f5c, cpssp->regs[R_EAX]);
	/* 7F54H    R8 */
	NAME_(mw_data_q)(cpssp, base+0x7f54, cpssp->regs[8]);
	/* 7F4CH    R9 */
	NAME_(mw_data_q)(cpssp, base+0x7f4c, cpssp->regs[9]);
	/* 7F44H    R10 */
	NAME_(mw_data_q)(cpssp, base+0x7f44, cpssp->regs[10]);
	/* 7F3CH    R11 */
	NAME_(mw_data_q)(cpssp, base+0x7f3c, cpssp->regs[11]);
	/* 7F34H    R12 */
	NAME_(mw_data_q)(cpssp, base+0x7f34, cpssp->regs[12]);
	/* 7F2CH    R13 */
	NAME_(mw_data_q)(cpssp, base+0x7f2c, cpssp->regs[13]);
	/* 7F24H    R14 */
	NAME_(mw_data_q)(cpssp, base+0x7f24, cpssp->regs[14]);
	/* 7F1CH    R15 */
	NAME_(mw_data_q)(cpssp, base+0x7f1c, cpssp->regs[15]);
	/* 7F08H - 7F1BH  Reserved */
	/* 7F04H    IEDBASE */
	/* 7F02H    Auto HALT Restart Field (Word) */
	NAME_(mw_data_w)(cpssp, base+0x7f02, ((cpssp->hflags >> HF_HALTED_SHIFT) & 1) << 0);
	/* 7F00H    I/O Instruction Restart Field (Word) */
	NAME_(mw_data_w)(cpssp, base+0x7f00, 0x0000);
	/* 7EFCH    SMM Revision Identifier Field (Doubleword) */
	/* see i386 implementation below for details */
	NAME_(mw_data_l)(cpssp, base+0x7efc, 0x00020001);
	/* 7EF8H    SMBASE Field (Doubleword) */
	NAME_(mw_data_l)(cpssp, base+0x7ef8, cpssp->core.smbase);
	/* 7EF7H - 7EA8H Reserved */
	/* 7EA4H    LDT Info */
	/* 7EA0H    LDT Limit */
	/* 7E9CH    LDT Base (Lower 32 Bits) */
	/* 7E98H    IDT Limit */
	/* 7E94H    IDT Base (Lower 32 Bits) */
	/* 7E90H    GDT Limit */
	/* 7E8CH    GDT Base (Lower 32 Bits) */
	/* 7E8BH - 7E44H Reserved */
	/* 7E40H    CR4 */
	/* 7E3FH - 7DF0H Reserved */
	/* 7DE8H    IO_EIP */
	/* 7DE7H - 7DDCH Reserved */
	/* 7DD8H    IDT Base (Upper 32 Bits) */
	/* 7DD4H    LDT Base (Upper 32 Bits) */
	/* 7DD0H    GDT Base (Upper 32 Bits) */
	/* 7DCFH - 7C00H Reserved  - we'll store other internals here: */
	/* The following registers are saved (but not readable) and restored upon exiting SMM:
	 * -   Control register CR4. (This register is cleared to all 0s while in SMM).
	 * -   The hidden segment descriptor information stored in segment registers CS, DS, ES, FS,
	 *     GS, and SS.
	 */
	NAME_(mw_data_q)(cpssp, base+0x7dc8, cpssp->cr[4]);
	NAME_(mw_data_q)(cpssp, base+0x7dc0, cpssp->segs[R_CS].base);
	NAME_(mw_data_q)(cpssp, base+0x7db8, cpssp->segs[R_DS].base);
	NAME_(mw_data_q)(cpssp, base+0x7db0, cpssp->segs[R_ES].base);
	NAME_(mw_data_q)(cpssp, base+0x7da8, cpssp->segs[R_FS].base);
	NAME_(mw_data_q)(cpssp, base+0x7da0, cpssp->segs[R_GS].base);
	NAME_(mw_data_q)(cpssp, base+0x7d98, cpssp->segs[R_SS].base);
	NAME_(mw_data_q)(cpssp, base+0x7d90, cpssp->segs[R_CS].limit);
	NAME_(mw_data_q)(cpssp, base+0x7d88, cpssp->segs[R_DS].limit);
	NAME_(mw_data_q)(cpssp, base+0x7d80, cpssp->segs[R_ES].limit);
	NAME_(mw_data_q)(cpssp, base+0x7d78, cpssp->segs[R_FS].limit);
	NAME_(mw_data_q)(cpssp, base+0x7d70, cpssp->segs[R_GS].limit);
	NAME_(mw_data_q)(cpssp, base+0x7d68, cpssp->segs[R_SS].limit);
	NAME_(mw_data_q)(cpssp, base+0x7d60, cpssp->segs[R_CS].flags);
	NAME_(mw_data_q)(cpssp, base+0x7d58, cpssp->segs[R_DS].flags);
	NAME_(mw_data_q)(cpssp, base+0x7d50, cpssp->segs[R_ES].flags);
	NAME_(mw_data_q)(cpssp, base+0x7d48, cpssp->segs[R_FS].flags);
	NAME_(mw_data_q)(cpssp, base+0x7d40, cpssp->segs[R_GS].flags);
	NAME_(mw_data_q)(cpssp, base+0x7d38, cpssp->segs[R_SS].flags);
	NAME_(mw_data_q)(cpssp, base+0x7d30, cpssp->tr.base);
	NAME_(mw_data_q)(cpssp, base+0x7d28, cpssp->tr.limit);
	NAME_(mw_data_q)(cpssp, base+0x7d20, cpssp->tr.flags);
	/* maybe we've got even more to store here... FIXME? */
#else /* not (CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT) */
	/* Offset   Register */
	/* 7FFCH    CR0 */
	NAME_(mw_data_l)(cpssp, base+0x7ffc, cr0);
	/* 7FF8H    CR3 */
	NAME_(mw_data_l)(cpssp, base+0x7ff8, cpssp->cr[3]);
	/* 7FF4H    EFLAGS */
	NAME_(mw_data_l)(cpssp, base+0x7ff4, compute_eflags(cpssp));
	/* 7FF0H    EIP */
	NAME_(mw_data_l)(cpssp, base+0x7ff0, cpssp->eip);
	/* 7FECH    EDI */
	NAME_(mw_data_l)(cpssp, base+0x7fec, cpssp->regs[R_EDI]);
	/* 7FE8H    ESI */
	NAME_(mw_data_l)(cpssp, base+0x7fe8, cpssp->regs[R_ESI]);
	/* 7FE4H    EBP */
	NAME_(mw_data_l)(cpssp, base+0x7fe4, cpssp->regs[R_EBP]);
	/* 7FE0H    ESP */
	NAME_(mw_data_l)(cpssp, base+0x7fe0, cpssp->regs[R_ESP]);
	/* 7FDCH    EBX */
	NAME_(mw_data_l)(cpssp, base+0x7fdc, cpssp->regs[R_EBX]);
	/* 7FD8H    EDX */
	NAME_(mw_data_l)(cpssp, base+0x7fd8, cpssp->regs[R_EDX]);
	/* 7FD4H    ECX */
	NAME_(mw_data_l)(cpssp, base+0x7fd4, cpssp->regs[R_ECX]);
	/* 7FD0H    EAX */
	NAME_(mw_data_l)(cpssp, base+0x7fd0, cpssp->regs[R_EAX]);
	/* 7FCCH    DR6 */
	NAME_(mw_data_l)(cpssp, base+0x7fcc, cpssp->dr[6]);
	/* 7FC8H    DR7 */
	NAME_(mw_data_l)(cpssp, base+0x7fc8, cpssp->dr[7]);
	/* 7FC4H    TR* */
	NAME_(mw_data_l)(cpssp, base+0x7fc4, cpssp->tr.selector);
	/* 7FC0H    Reserved */
	/* 7FBCH    GS* */
	NAME_(mw_data_l)(cpssp, base+0x7fbc, cpssp->segs[R_GS].selector);
	/* 7FB8H    FS* */
	NAME_(mw_data_l)(cpssp, base+0x7fb8, cpssp->segs[R_FS].selector);
	/* 7FB4H    DS* */
	NAME_(mw_data_l)(cpssp, base+0x7fb4, cpssp->segs[R_DS].selector);
	/* 7FB0H    SS* */
	NAME_(mw_data_l)(cpssp, base+0x7fb0, cpssp->segs[R_SS].selector);
	/* 7FACH    CS* */
	NAME_(mw_data_l)(cpssp, base+0x7fac, cpssp->segs[R_CS].selector);
	/* 7FA8H    ES* */
	NAME_(mw_data_l)(cpssp, base+0x7fa8, cpssp->segs[R_ES].selector);
	/* 7FA4H    I/O State Field, see Section 13.7 */
	NAME_(mw_data_l)(cpssp, base+0x7fa4, 0x00000000); /* see SMM revision ID below */
	/* 7FA0H    I/O Memory Address Field, see Section 13.7 */
	NAME_(mw_data_l)(cpssp, base+0x7fa0, 0x00000000); /* see SMM revision ID below */
	/* 7F9FH-7F03H Reserved */
	/* 7F02H    Auto HALT Restart Field (Word) */
	/* difficult to implement in the current CPU simulation... */
	NAME_(mw_data_w)(cpssp, base+0x7f02, 0x0000);
	/* 7F00H    I/O Instruction Restart Field (Word) */
	NAME_(mw_data_w)(cpssp, base+0x7f00, 0x0000);
	/* 7EFCH    SMM Revision Identifier Field (Doubleword) */
	/* Processors that have an SMM revision ID of 30004H or higher have
	 * I/O Instruction Restart. */
	/* The upper word of the SMM revision identifier refers to the
	 * extensions available. If the I/O instruction restart flag (bit 16)
	 * is set, the processor supports I/O instruction restart. If the
	 * SMBASE relocation flag (bit 17) is set, SMRAM base address
	 * relocation is supported. */
	/* we use "no i/o restart, but smbase relocation, revision 1" */
	NAME_(mw_data_l)(cpssp, base+0x7efc, 0x00020001);
	/* 7EF8H    SMBASE Field (Doubleword) */
	NAME_(mw_data_l)(cpssp, base+0x7ef8, cpssp->core.smbase);
	/* 7EF7H-7E00H Reserved - we'll store other internals here: */
	/* The following registers are saved (but not readable) and restored upon exiting SMM:
	 * -   Control register CR4. (This register is cleared to all 0s while in SMM).
	 * -   The hidden segment descriptor information stored in segment registers CS, DS, ES, FS,
	 *     GS, and SS.
	 */
	NAME_(mw_data_l)(cpssp, base+0x7ef4, cpssp->cr[4]);
	NAME_(mw_data_l)(cpssp, base+0x7ef0, cpssp->segs[R_CS].base);
	NAME_(mw_data_l)(cpssp, base+0x7eec, cpssp->segs[R_DS].base);
	NAME_(mw_data_l)(cpssp, base+0x7ee8, cpssp->segs[R_ES].base);
	NAME_(mw_data_l)(cpssp, base+0x7ee4, cpssp->segs[R_FS].base);
	NAME_(mw_data_l)(cpssp, base+0x7ee0, cpssp->segs[R_GS].base);
	NAME_(mw_data_l)(cpssp, base+0x7edc, cpssp->segs[R_SS].base);
	NAME_(mw_data_l)(cpssp, base+0x7ed8, cpssp->segs[R_CS].limit);
	NAME_(mw_data_l)(cpssp, base+0x7ed4, cpssp->segs[R_DS].limit);
	NAME_(mw_data_l)(cpssp, base+0x7ed0, cpssp->segs[R_ES].limit);
	NAME_(mw_data_l)(cpssp, base+0x7ecc, cpssp->segs[R_FS].limit);
	NAME_(mw_data_l)(cpssp, base+0x7ec8, cpssp->segs[R_GS].limit);
	NAME_(mw_data_l)(cpssp, base+0x7ec4, cpssp->segs[R_SS].limit);
	NAME_(mw_data_l)(cpssp, base+0x7ec0, cpssp->segs[R_CS].flags);
	NAME_(mw_data_l)(cpssp, base+0x7ebc, cpssp->segs[R_DS].flags);
	NAME_(mw_data_l)(cpssp, base+0x7eb8, cpssp->segs[R_ES].flags);
	NAME_(mw_data_l)(cpssp, base+0x7eb4, cpssp->segs[R_FS].flags);
	NAME_(mw_data_l)(cpssp, base+0x7eb0, cpssp->segs[R_GS].flags);
	NAME_(mw_data_l)(cpssp, base+0x7eac, cpssp->segs[R_SS].flags);
	NAME_(mw_data_l)(cpssp, base+0x7ea8, cpssp->tr.base);
	NAME_(mw_data_l)(cpssp, base+0x7ea4, cpssp->tr.limit);
	NAME_(mw_data_l)(cpssp, base+0x7ea0, cpssp->tr.flags);
	/* maybe we've got even more to store here... FIXME? */
#endif /* not CONFIG_CPU_LM_SUPPORT */
	/*
	 * set up SMM environment:
	 */
	/* General-purpose registers    Undefined */
	/* EFLAGS                      00000002H */
	load_eflags(cpssp, 0x00000002, ~(CC_O | CC_S | CC_Z | CC_A | CC_P | CC_C));
	cpssp->cc_op = CC_OP_EFLAGS;

	/* CS selector            SMM Base shifted right 4 bits */
	/* CS base                    SMM Base */
	cpu_x86_load_seg_cache(cpssp, R_CS, (cpssp->core.smbase >> 4), cpssp->core.smbase,
			0xffffffff, 0);
	/* DS, ES, FS, GS, SS Selectors 0000H */
	/* DS, ES, FS, GS, SS Bases     000000000H */
	/* DS, ES, FS, GS, SS Limits    0FFFFFFFFH */
	cpu_x86_load_seg_cache(cpssp, R_DS, 0x0000, 0x00000000, 0xffffffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_ES, 0x0000, 0x00000000, 0xffffffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_FS, 0x0000, 0x00000000, 0xffffffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_GS, 0x0000, 0x00000000, 0xffffffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_SS, 0x0000, 0x00000000, 0xffffffff, 0);

	/* CR0                    PE, EM, TS, and PG flags set to 0;
	 *                            others unmodified */
	cpssp->cr[0] &= ~CPU_CR0_PE_MASK;
	cpssp->cr[0] &= ~CPU_CR0_EM_MASK;
	cpssp->cr[0] &= ~CPU_CR0_TS_MASK;
	cpssp->cr[0] &= ~CPU_CR0_PG_MASK;
	NAME_(update_cr0)(cpssp, cpssp->cr[0]);
	/* CR4                    Cleared to zero */
	NAME_(update_cr4)(cpssp, 0x00000000);
	/* DR6                    Undefined */
	/* DR7                    00000400 */
	cpssp->dr[7] = 0x00000400;
	/* EIP                    00008000H */
	cpssp->eip = 0x00008000;
}
#endif /* 80386 <= CONFIG_CPU */

/* protected mode interrupt */
static void
NAME_(do_interrupt_protected)(struct cpssp *cpssp, int intno, int is_int, int error_code, unsigned int next_eip, int is_hw)
{
        struct SegmentCache *dt;
        target_ulong ptr, ssp;
        int type, dpl, selector, ss_dpl, cpl, sp_mask;
        int has_error_code, new_stack, shift;
        uint32_t e1, e2, offset, ss, esp, ss_e1, ss_e2;
        uint32_t old_eip;

        has_error_code = 0;
        if (! is_int && ! is_hw) {
                switch(intno) {
                case 8:
                case 10:
                case 11:
                case 12:
                case 13:
                case 14:
                case 17:
                        has_error_code = 1;
                        break;
                }
        }
        if (is_int)
                old_eip = next_eip;
        else
                old_eip = cpssp->eip;

        dt = &cpssp->idt;
        if (intno * 8 + 7 > dt->limit)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, intno * 8 + 2);
        ptr = dt->base + intno * 8;
        e1 = ldl_kernel(cpssp, ptr);
        e2 = ldl_kernel(cpssp, ptr + 4);
        /* check gate type */
        type = (e2 >> DESC_TYPE_SHIFT) & 0x1f;
        switch(type) {
        case 5: /* task gate */
                /* must do that check here to return the correct error code */
                if (!(e2 & DESC_P_MASK))
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, intno * 8 + 2);
                NAME_(switch_tss)(cpssp, intno * 8, e1, e2, SWITCH_TSS_CALL, old_eip);
                if (has_error_code) {
                        int mask, type_;
                        /* push the error code */
                        type_ = (cpssp->tr.flags >> DESC_TYPE_SHIFT) & 0xf;
                        shift = type_ >> 3;
                        if (cpssp->segs[R_SS].flags & DESC_B_MASK)
                                mask = 0xffffffff;
                        else
                                mask = 0xffff;
                        esp = (cpssp->regs[R_ESP] - (2 << shift)) & mask;
                        ssp = cpssp->segs[R_SS].base + esp;
                        if (shift)
                                stl_kernel(cpssp, ssp, error_code);
                        else
                                stw_kernel(cpssp, ssp, error_code);
                        cpssp->regs[R_ESP] = (esp & mask) | (cpssp->regs[R_ESP] & ~mask);
                }
                return;
        case 6: /* 286 interrupt gate */
        case 7: /* 286 trap gate */
        case 14: /* 386 interrupt gate */
        case 15: /* 386 trap gate */
                break;
        default:
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, intno * 8 + 2);
                break;
        }
        dpl = (e2 >> DESC_DPL_SHIFT) & 3;
        cpl = cpssp->hflags & HF_CPL_MASK;
        /* check privledge if software int */
        if (is_int && dpl < cpl)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, intno * 8 + 2);
        /* check valid bit */
        if (!(e2 & DESC_P_MASK))
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, intno * 8 + 2);
        selector = e1 >> 16;
        offset = (e2 & 0xffff0000) | (e1 & 0x0000ffff);
        if ((selector & 0xfffc) == 0)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);

        if (load_segment(cpssp, &e1, &e2, selector) != 0)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
        if (!(e2 & DESC_S_MASK) || !(e2 & (DESC_CS_MASK)))
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
        dpl = (e2 >> DESC_DPL_SHIFT) & 3;
        if (dpl > cpl)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
        if (!(e2 & DESC_P_MASK))
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, selector & 0xfffc);
        if (!(e2 & DESC_C_MASK) && dpl < cpl) {
                /* to inner priviledge */
                get_ss_esp_from_tss(cpssp, &ss, &esp, dpl);
                if ((ss & 0xfffc) == 0)
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
                if ((ss & 3) != dpl)
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
                if (load_segment(cpssp, &ss_e1, &ss_e2, ss) != 0)
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
                ss_dpl = (ss_e2 >> DESC_DPL_SHIFT) & 3;
                if (ss_dpl != dpl)
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
                if (!(ss_e2 & DESC_S_MASK)
                 || (ss_e2 & DESC_CS_MASK)
                 || !(ss_e2 & DESC_W_MASK))
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
                if (!(ss_e2 & DESC_P_MASK))
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, ss & 0xfffc);
                new_stack = 1;
                sp_mask = get_sp_mask(ss_e2);
                ssp = get_seg_base(ss_e1, ss_e2);
        } else if ((e2 & DESC_C_MASK) || dpl == cpl) {
                /* to same priviledge */
                if (cpssp->eflags & CPU_VM_MASK)
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
                new_stack = 0;
                sp_mask = get_sp_mask(cpssp->segs[R_SS].flags);
                ssp = cpssp->segs[R_SS].base;
                esp = cpssp->regs[R_ESP];
                dpl = cpl;
                ss_e1 = 0; /* avoid warning */
                ss_e2 = 0; /* avoid warning */
                ss = 0; /* avoid warning */
        } else {
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
        }

        shift = type >> 3;

#if 0
        /* XXX: check that enough room is available */
        push_size = 6 + (new_stack << 2) + (has_error_code << 1);
        if (cpssp->eflags & CPU_VM_MASK)
                push_size += 8;
        push_size <<= shift;
#endif

        if (shift == 1) {
                if (new_stack) {
                        if (cpssp->eflags & CPU_VM_MASK) {
                                PUSHL(cpssp, ssp, esp, sp_mask, cpssp->segs[R_GS].selector);
                                PUSHL(cpssp, ssp, esp, sp_mask, cpssp->segs[R_FS].selector);
                                PUSHL(cpssp, ssp, esp, sp_mask, cpssp->segs[R_DS].selector);
                                PUSHL(cpssp, ssp, esp, sp_mask, cpssp->segs[R_ES].selector);
                        }
                        PUSHL(cpssp, ssp, esp, sp_mask, cpssp->segs[R_SS].selector);
                        PUSHL(cpssp, ssp, esp, sp_mask, cpssp->regs[R_ESP]);
                }
                PUSHL(cpssp, ssp, esp, sp_mask, compute_eflags(cpssp));
                PUSHL(cpssp, ssp, esp, sp_mask, cpssp->segs[R_CS].selector);
                PUSHL(cpssp, ssp, esp, sp_mask, old_eip);
                if (has_error_code) {
                        PUSHL(cpssp, ssp, esp, sp_mask, error_code);
                }
        } else {
                if (new_stack) {
                        if (cpssp->eflags & CPU_VM_MASK) {
                                PUSHW(cpssp, ssp, esp, sp_mask, cpssp->segs[R_GS].selector);
                                PUSHW(cpssp, ssp, esp, sp_mask, cpssp->segs[R_FS].selector);
                                PUSHW(cpssp, ssp, esp, sp_mask, cpssp->segs[R_DS].selector);
                                PUSHW(cpssp, ssp, esp, sp_mask, cpssp->segs[R_ES].selector);
                        }
                        PUSHW(cpssp, ssp, esp, sp_mask, cpssp->segs[R_SS].selector);
                        PUSHW(cpssp, ssp, esp, sp_mask, cpssp->regs[R_ESP]);
                }
                PUSHW(cpssp, ssp, esp, sp_mask, compute_eflags(cpssp));
                PUSHW(cpssp, ssp, esp, sp_mask, cpssp->segs[R_CS].selector);
                PUSHW(cpssp, ssp, esp, sp_mask, old_eip);
                if (has_error_code) {
                        PUSHW(cpssp, ssp, esp, sp_mask, error_code);
                }
        }
        if (new_stack) {
                if (cpssp->eflags & CPU_VM_MASK) {
                        cpu_x86_load_seg_cache(cpssp, R_ES, 0, 0, 0, 0);
                        cpu_x86_load_seg_cache(cpssp, R_DS, 0, 0, 0, 0);
                        cpu_x86_load_seg_cache(cpssp, R_FS, 0, 0, 0, 0);
                        cpu_x86_load_seg_cache(cpssp, R_GS, 0, 0, 0, 0);
                }
                ss = (ss & ~3) | dpl;
                cpu_x86_load_seg_cache(cpssp, R_SS, ss,
                                ssp, get_seg_limit(ss_e1, ss_e2), ss_e2);
        }
        cpssp->regs[R_ESP] = (cpssp->regs[R_ESP] & ~sp_mask) | (esp & sp_mask);

        selector = (selector & ~3) | dpl;
        cpu_x86_load_seg_cache(cpssp, R_CS, selector,
                        get_seg_base(e1, e2), get_seg_limit(e1, e2), e2);
        cpu_x86_set_cpl(cpssp, dpl);
        cpssp->eip = offset;

        /* interrupt gate clear IF mask */
        if ((type & 1) == 0) {
                cpssp->eflags &= ~CPU_IF_MASK;
        }
        cpssp->eflags &= ~(CPU_TF_MASK | CPU_VM_MASK | CPU_RF_MASK | CPU_NT_MASK);
}

/* real mode interrupt */
static void
NAME_(do_interrupt_real)(struct cpssp *cpssp, int intno, int is_int, int error_code, unsigned int next_eip)
{
        struct SegmentCache *dt;
        target_ulong ptr, ssp;
        int selector;
        uint32_t offset, esp;
        uint32_t old_cs, old_eip;

        /* real mode (simpler !) */
        dt = &cpssp->idt;
        if (intno * 4 + 3 > dt->limit) {
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, intno * 8 + 2);
        }
        ptr = dt->base + intno * 4;
        offset = lduw_kernel(cpssp, ptr);
        selector = lduw_kernel(cpssp, ptr + 2);
        esp = cpssp->regs[R_ESP];
        ssp = cpssp->segs[R_SS].base;
        if (is_int) {
                old_eip = next_eip;
        } else {
                old_eip = cpssp->eip;
        }
        old_cs = cpssp->segs[R_CS].selector;

        /* XXX: use SS segment size ? */
        PUSHW(cpssp, ssp, esp, 0xffff, compute_eflags(cpssp));
        PUSHW(cpssp, ssp, esp, 0xffff, old_cs);
        PUSHW(cpssp, ssp, esp, 0xffff, old_eip);

        /* update processor state */
        cpssp->regs[R_ESP] = (cpssp->regs[R_ESP] & ~0xffff) | (esp & 0xffff);
        cpssp->eip = offset;
        cpssp->segs[R_CS].selector = selector;
        cpssp->segs[R_CS].base = (selector << 4);
        cpssp->eflags &= ~(CPU_IF_MASK | CPU_TF_MASK | CPU_AC_MASK | CPU_RF_MASK);
}

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
static target_ulong
get_rsp_from_tss(struct cpssp *cpssp, int level)
{
    int index_;

#if 0
    printf("TR: base=" TARGET_FMT_lx " limit=%x\n",
           cpssp->tr.base, cpssp->tr.limit);
#endif

    if (!(cpssp->tr.flags & DESC_P_MASK)) {
      assert(0);
    }
    index_ = 8 * level + 4;
    if ((index_ + 7) > cpssp->tr.limit)
        NAME_(raise_exception_err)(cpssp, CPU_FAULT_TS, cpssp->tr.selector & 0xfffc);
    return ldq_kernel(cpssp, cpssp->tr.base + index_);
}

/* 64 bit interrupt */
static void
NAME_(do_interrupt64)(struct cpssp *cpssp, int intno, int is_int, int error_code, target_ulong next_eip, int is_hw)
{
        struct SegmentCache *dt;
        target_ulong ptr;
        int type, dpl, selector, cpl, ist;
        int has_error_code, new_stack;
        uint32_t e1, e2, e3, ss;
        target_ulong old_eip, esp, offset;

        has_error_code = 0;
        if (!is_int && !is_hw) {
                switch(intno) {
                        case 8:
                        case 10:
                        case 11:
                        case 12:
                        case 13:
                        case 14:
                        case 17:
                                has_error_code = 1;
                                break;
                }
        }
        if (is_int)
                old_eip = next_eip;
        else
                old_eip = cpssp->eip;

        dt = &cpssp->idt;
        if (intno * 16 + 15 > dt->limit)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, intno * 16 + 2);
        ptr = dt->base + intno * 16;
        e1 = ldl_kernel(cpssp, ptr);
        e2 = ldl_kernel(cpssp, ptr + 4);
        e3 = ldl_kernel(cpssp, ptr + 8);
        /* check gate type */
        type = (e2 >> DESC_TYPE_SHIFT) & 0x1f;
        switch(type) {
                case 14: /* 386 interrupt gate */
                case 15: /* 386 trap gate */
                        break;
                default:
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, intno * 16 + 2);
                        break;
        }
        dpl = (e2 >> DESC_DPL_SHIFT) & 3;
        cpl = cpssp->hflags & HF_CPL_MASK;
        /* check privledge if software int */
        if (is_int && dpl < cpl)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, intno * 16 + 2);
        /* check valid bit */
        if (!(e2 & DESC_P_MASK))
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, intno * 16 + 2);
        selector = e1 >> 16;
        offset = ((target_ulong)e3 << 32) | (e2 & 0xffff0000) | (e1 & 0x0000ffff);

        ist = e2 & 7;
        if ((selector & 0xfffc) == 0)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, 0);

        if (load_segment(cpssp, &e1, &e2, selector) != 0)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
        if (!(e2 & DESC_S_MASK) || !(e2 & (DESC_CS_MASK)))
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
        dpl = (e2 >> DESC_DPL_SHIFT) & 3;
        if (dpl > cpl)
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
        if (!(e2 & DESC_P_MASK))
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_NP, selector & 0xfffc);
        if (!(e2 & DESC_L_MASK) || (e2 & DESC_B_MASK))
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
        if ((!(e2 & DESC_C_MASK) && dpl < cpl) || ist != 0) {
                /* to inner priviledge */
                if (ist != 0) {
                        esp = get_rsp_from_tss(cpssp, ist + 3);
                } else {
                        esp = get_rsp_from_tss(cpssp, dpl);
                }
                esp &= ~0xfLL; /* align stack */
                ss = 0;
                new_stack = 1;
        } else if ((e2 & DESC_C_MASK) || dpl == cpl) {
                /* to same priviledge */
                if (cpssp->eflags & CPU_VM_MASK)
                        NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
                new_stack = 0;
                if (ist != 0) {
                        esp = get_rsp_from_tss(cpssp, ist + 3);
                } else {
                        esp = cpssp->regs[R_ESP];
                }
                esp &= ~0xfLL; /* align stack */
                dpl = cpl;
        } else {
                NAME_(raise_exception_err)(cpssp, CPU_FAULT_GP, selector & 0xfffc);
                new_stack = 0; /* avoid warning */
                esp = 0; /* avoid warning */
        }

        PUSHQ(cpssp, esp, cpssp->segs[R_SS].selector);
        PUSHQ(cpssp, esp, cpssp->regs[R_ESP]);
        PUSHQ(cpssp, esp, compute_eflags(cpssp));
        PUSHQ(cpssp, esp, cpssp->segs[R_CS].selector);
        PUSHQ(cpssp, esp, old_eip);
        if (has_error_code) {
                PUSHQ(cpssp, esp, error_code);
        }

        if (new_stack) {
                ss = 0 | dpl;
                cpu_x86_load_seg_cache(cpssp, R_SS, ss, 0, 0, 0);
        }
        cpssp->regs[R_ESP] = esp;

        selector = (selector & ~3) | dpl;
        cpu_x86_load_seg_cache(cpssp, R_CS, selector,
                        get_seg_base(e1, e2),
                        get_seg_limit(e1, e2),
                        e2);
        cpu_x86_set_cpl(cpssp, dpl);
        cpssp->eip = offset;

        /* interrupt gate clear IF mask */
        if ((type & 1) == 0) {
                cpssp->eflags &= ~CPU_IF_MASK;
        }
        cpssp->eflags &= ~(CPU_TF_MASK | CPU_VM_MASK | CPU_RF_MASK | CPU_NT_MASK);
}
#endif /* CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT */

void
NAME_(do_interrupt)(struct cpssp *cpssp)
{
        if (cpssp->cr[0] & CPU_CR0_PE_MASK) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
                if (cpssp->hflags & HF_LMA_MASK) {
                        NAME_(do_interrupt64)(cpssp, cpssp->exception_index, 0, 0, 0, 1);
                } else
#endif
                {
                        NAME_(do_interrupt_protected)(cpssp, cpssp->exception_index, 0, 0, 0, 1);
                }
        } else {
                NAME_(do_interrupt_real)(cpssp, cpssp->exception_index, 0, 0, 0);
        }
        cpssp->exception_index = -1;
}

void
NAME_(do_exception)(struct cpssp *cpssp)
{
        if (cpssp->cr[0] & CPU_CR0_PE_MASK) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
                if (cpssp->hflags & HF_LMA_MASK) {
                        NAME_(do_interrupt64)(cpssp, cpssp->exception_index,
                                       cpssp->exception_is_int,
                                       cpssp->error_code,
                                       cpssp->exception_next_eip,
                                       0);
                } else
#endif
                {
                        NAME_(do_interrupt_protected)(cpssp, cpssp->exception_index,
                                        cpssp->exception_is_int,
                                        cpssp->error_code,
                                        cpssp->exception_next_eip,
                                        0);
                }
        } else {
                NAME_(do_interrupt_real)(cpssp, cpssp->exception_index,
                                cpssp->exception_is_int,
                                cpssp->error_code,
                                cpssp->exception_next_eip);
        }
        cpssp->exception_index = -1;
}

static void __attribute__((__noreturn__))
NAME_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:	;
	/* prepare setjmp context for exception handling */
	setjmp(cpssp->jmp_env);
	cpssp->link_tb = 0;

	while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
		cpssp->current_tb = (TranslationBlock *) 0;

		if (unlikely(cpssp->state_ext != STATE_POWER)) {
			if (! (cpssp->state_ext & STATE_POWER)) {
#if DEBUG_CONTROL_FLOW
				if (2 <= DEBUG_CONTROL_FLOW
				 || loglevel) {
					fprintf(stderr, "%lld: Power-off\n",
						cpssp->process.inst_cnt);
				}
#endif
				cpssp->process.inst_cnt = cpssp->process.inst_limit;

			} else if ((cpssp->state_ext & STATE_N_RESET_ACK)
#if 80386 <= CONFIG_CPU
					|| (cpssp->state_ext & STATE_N_INIT_ACK)
#endif
					){
				if (cpssp->state_ext & STATE_N_RESET_ACK) {
					/*
					 * Reset
					 */
#if DEBUG_CONTROL_FLOW
					if (2 <= DEBUG_CONTROL_FLOW
					 || loglevel) {
						fprintf(stderr, "%lld: Reset\n",
							cpssp->process.inst_cnt);
					}
#endif
					cpssp->state_ext &= ~STATE_N_RESET_ACK;
					NAME_(core_reset)(cpssp);
				}
#if 80386 < CONFIG_CPU
				if (cpssp->state_ext & STATE_N_INIT_ACK) {
					/*
					 * Init
					 */
#if DEBUG_CONTROL_FLOW
					if (2 <= DEBUG_CONTROL_FLOW
					 || loglevel) {
						fprintf(stderr, "%lld: Init\n",
							cpssp->process.inst_cnt);
					}
#endif
					cpssp->state_ext &= ~STATE_N_INIT_ACK;
					NAME_(core_init)(cpssp);
				}
#endif /* 80386 < CONFIG_CPU */
				/* No time increment here! */

			} else if ((cpssp->state_ext & STATE_RESET)
#if 80386 < CONFIG_CPU
					|| (cpssp->state_ext & STATE_INIT)
#endif
					) {
				/*
				 * Waiting while reset/init active...
				 */
#if DEBUG_CONTROL_FLOW
				if (2 <= DEBUG_CONTROL_FLOW
				 || loglevel) {
					fprintf(stderr, "%lld: Resetting\n",
						cpssp->process.inst_cnt);
				}
#endif
				cpssp->process.inst_cnt = cpssp->process.inst_limit;
			}

#if 80386 < CONFIG_CPU
		} else if (unlikely(cpssp->hflags & HF_WAITING_FOR_STARTUP_MASK)) {
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
				if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
					fprintf(stderr, "%d: Waiting for Startup-IPI at %016llx",
							cpssp->id,
							(unsigned long long) cpssp->eip);
				} else
#endif
				{
					fprintf(stderr, "%d: Waiting for Startup-IPI at %08llx (%08llx:%08llx)",
							cpssp->id,
							(unsigned long long) cpssp->eip
							+ (unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->eip);
				}
				NAME_(dump)(cpssp);
			}
#endif
			if (cpssp->state_startup) {
				/* Leave WAITING_FOR_STARTUP state. */
				cpssp->state_startup = 0;
				cpssp->hflags &= ~HF_WAITING_FOR_STARTUP_MASK;

				/* Load code segment register. */
				cpu_x86_load_seg_cache(cpssp, R_CS,
					(uint16_t) cpssp->state_startup_vec << 8, /* Selector */
					(uint32_t) cpssp->state_startup_vec << 12,/* Base */
					0xffff,                 /* Limit */
					0);                     /* Flags */

				/* Load instruction pointer. */
				cpssp->eip = 0x00000000;
			}

			cpssp->process.inst_cnt = cpssp->process.inst_limit;
#endif /* 80386 < CONFIG_CPU */

		} else if (unlikely(0 <= cpssp->exception_index)) {
			/*
			 * If an exception is pending, we execute it here.
			 * Simulate a real cpu exception. On i386, it can
			 * trigger new exceptions, but we do not handle
			 * double or triple faults yet.
			 */
#if DEBUG_CONTROL_BIOS
			if (2 <= DEBUG_CONTROL_BIOS
			 || loglevel) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
				if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
					fprintf(stderr, "int 0x%02x at %016llx",
							cpssp->exception_index,
							(unsigned long long) EIP(cpssp));
				} else
#endif
				{
					fprintf(stderr, "int 0x%02x at %08llx (%08llx:%08llx)",
							cpssp->exception_index,
							(unsigned long long) EIP(cpssp)
							+ (unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) EIP(cpssp));
				}
				NAME_(dump)(cpssp);
			}
#endif
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
				if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
					fprintf(stderr, "%lld: Exception 0x%02x at %016llx",
							cpssp->process.inst_cnt,
							cpssp->exception_index,
							(unsigned long long) EIP(cpssp));
				} else
#endif
				{
					fprintf(stderr, "%lld: Exception 0x%02x at %08llx (%08llx:%08llx)",
							cpssp->process.inst_cnt,
							cpssp->exception_index,
							(unsigned long long) EIP(cpssp)
							+ (unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) EIP(cpssp));
				}
				NAME_(dump)(cpssp);
			}
#endif
			NAME_(do_exception)(cpssp);
			cpssp->link_tb = 0;

			cpssp->process.inst_cnt++;

#if 80386 <= CONFIG_CPU
		} else if (unlikely((cpssp->interrupt_request & CPU_INTERRUPT_SMI)
			&& ! cpssp->core.smm
			&& ! (cpssp->hflags & HF_INHIBIT_IRQ_MASK))) {
			cpssp->hflags &= ~HF_HALTED_MASK;

			/* signal "acknowledge" */
			NAME_(core_smi_ack)(cpssp);

#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
				if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
					fprintf(stderr, "%lld: SMM-Interrupt at %016llx",
							cpssp->process.inst_cnt,
							(unsigned long long) cpssp->eip);
				} else
#endif
				{
					fprintf(stderr, "%lld: SMM-Interrupt at %08llx (%08llx:%08llx)",
							cpssp->process.inst_cnt,
							(unsigned long long) cpssp->eip
							+ (unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->eip);
				}
				NAME_(dump)(cpssp);
			}
#endif

			NAME_(do_smm)(cpssp);
			cpssp->link_tb = 0;

			cpssp->process.inst_cnt++;

#endif /* 80386 <= CONFIG_CPU */

		} else if (unlikely((cpssp->interrupt_request & CPU_INTERRUPT_IRQ)
				&& (cpssp->eflags & CPU_IF_MASK)
				&& ! (cpssp->hflags & HF_INHIBIT_IRQ_MASK))) {
			/*
			 * If hardware interrupt pending, we execute it.
			 */
			cpssp->hflags &= ~HF_HALTED_MASK;

			cpssp->exception_index = NAME_(core_ack)(cpssp);
			cpssp->exception_is_int = 0;
			cpssp->error_code = 0;
			cpssp->exception_next_eip = 0;

#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
				if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
					fprintf(stderr, "%lld: Interrupt 0x%02x at %016llx",
							cpssp->process.inst_cnt,
							cpssp->exception_index,
							(unsigned long long) cpssp->eip);
				} else
#endif
				{
					fprintf(stderr, "%lld: Interrupt 0x%02x at %08llx (%08llx:%08llx)",
							cpssp->process.inst_cnt,
							cpssp->exception_index,
							(unsigned long long) cpssp->eip
							+ (unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->eip);
				}
				NAME_(dump)(cpssp);
			}
#endif
			NAME_(do_interrupt)(cpssp);
			cpssp->link_tb = 0;

			cpssp->process.inst_cnt++;

		} else if (unlikely(cpssp->hflags & HF_HALTED_MASK)) {
			/*
			 * Do nothing...
			 */
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
				if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
					fprintf(stderr, "%lld: Halting at %016llx",
							cpssp->process.inst_cnt,
							(unsigned long long) cpssp->eip);
				} else
#endif
				{
					fprintf(stderr, "%lld: Halting at %08llx (%08llx:%08llx)",
							cpssp->process.inst_cnt,
							(unsigned long long) cpssp->eip
							+ (unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->eip);
				}
				NAME_(dump)(cpssp);
			}
#endif

			cpssp->process.inst_cnt = cpssp->process.inst_limit;

		} else {
			/*
			 * Execute the generated code.
			 */
			TranslationBlock *tb;
			target_ulong cs_base;
			target_ulong pc;
			unsigned int flags;

#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
				if ((cpssp->hflags >> HF_LMA_SHIFT) & 1) {
					fprintf(stderr, "%lld: Executing at %016llx",
							cpssp->process.inst_cnt,
							(unsigned long long) cpssp->eip);
				} else
#endif
				{
					fprintf(stderr, "%lld: Executing at %08llx (%08llx:%08llx)",
							cpssp->process.inst_cnt,
							(unsigned long long) cpssp->eip
							+ (unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->segs[R_CS].base,
							(unsigned long long) cpssp->eip);
				}
				NAME_(dump)(cpssp);
			}
#endif

			/*
			 * We record a subset of the CPU state. It will
			 * always be the same before a given translated block
			 * is executed.
			 */
			flags = cpssp->hflags;
			flags |= (cpssp->eflags & (CPU_IOPL_MASK | CPU_TF_MASK | CPU_VM_MASK));
			cs_base = cpssp->segs[R_CS].base;
			pc = cs_base + cpssp->eip;

			tb = NAME_(tb_get)(cpssp, pc, cs_base, flags, cpssp->link_tb);

			cpssp->link_tb = 0;
			cpssp->current_tb = tb;

#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT && defined(__i386__)
			asm volatile ("subl $0x04,%%esp\n"
					"push %%ebx\n"
					"push %%esi\n"
					"push %%edi\n"
					"call *%0\n"
					"pop %%edi\n"
					"pop %%esi\n"
					"pop %%ebx\n"
					"addl $0x04,%%esp\n"
					: : "r" ((void (*)(void)) tb->tc_ptr) : "ebx", "esi", "edi");
#else
			asm volatile (
				"movq %1, %%rbp\n"
				"call *%0\n"
				: /* No Output */
				: "r" (tb->tc_ptr), "r" (cpssp)
				: "rbp", /* Set here. */
			          "rbx", "r12", "r13", /* T0, T1, A0 used without save. */
				  "rax", "rcx", "rdx", "rdi", "rsi", "r8", "r9", "r10", "r11", /* Caller saved. */
				  "memory"
			);
#endif
		}
	}

	sched_to_scheduler();

	goto again;
}

/*
 * Signal an interruption. It is executed in the main CPU loop.
 * is_int is TRUE if coming from the int instruction. next_eip is the
 * EIP value AFTER the interrupt instruction. It is only relevant if
 * is_int is TRUE.
 */
void __attribute__((__noreturn__))
NAME_(raise_interrupt)(struct cpssp *cpssp, int intno, int is_int, int error_code, int next_eip_addend)
{
        cpssp->exception_index = intno;
        cpssp->error_code = error_code;
        cpssp->exception_is_int = is_int;
        cpssp->exception_next_eip = cpssp->eip + next_eip_addend;
        cpssp->process.inst_cnt++;
        longjmp(cpssp->jmp_env, 1);
}

/* shortcuts to generate exceptions */

void __attribute__((__noreturn__))
NAME_(raise_exception_err)(struct cpssp *cpssp, int exception_index, int error_code)
{
        NAME_(raise_interrupt)(cpssp, exception_index, 0, error_code, 0);
}

void __attribute__((__noreturn__))
NAME_(raise_exception)(struct cpssp *cpssp, int exception_index)
{
        NAME_(raise_interrupt)(cpssp, exception_index, 0, 0, 0);
}

static int
NAME_(core_port)(
	struct cpssp *cpssp,
	const char *port,
	unsigned int *regnop,
	unsigned int *bitnop
)
{
	char *port2;

	/* Get Fault Type */
	if (strncmp(port, "bitflip", strlen("bitflip")) == 0) {
		port += strlen("bitflip");
	} else {
		return 1;
	}

	/* Skip '/' */
	if (*port == '/') {
		port++;
	} else {
		return 1;
	}

	/* Get register number. */
	*regnop = strtoul(port, &port2, 0);
	if (CPU_NB_REGS <= *regnop) {
		return 1;
	}
	port = port2;

	/* Skip '/' */
	if (*port == '/') {
		port++;
	} else {
		return 1;
	}

	/* Get bit number. */
	*bitnop = strtoul(port, &port2, 0);
#if 80486 <= CONFIG_CPU && CONFIG_CPU_LM_SUPPORT
	if (64 <= *bitnop) {
		return 1;
	}
#elif 80386 <= CONFIG_CPU
	if (32 <= *bitnop) {
		return 1;
	}
#else
	if (16 <= *bitnop) {
		return 1;
	}
#endif
	port = port2;

	if (*port != '\0') {
		return 1;
	}

	return 0;
}

static void
NAME_(core_fault_set)(void *_cpssp, unsigned int i, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;
	unsigned int regno;
	unsigned int bitno;

	if (! val) {
		return;
	}

	assert(cpssp->fault[i].used);

	regno = cpssp->fault[i].regno;
	bitno = cpssp->fault[i].bitno;

	fprintf(stderr, "Flip bit %d in reg %d\n", bitno, regno);
}

static void
NAME_(core_fault_set0)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 0, val);
}

static void
NAME_(core_fault_set1)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 1, val);
}

static void
NAME_(core_fault_set2)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 2, val);
}

static void
NAME_(core_fault_set3)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 3, val);
}

static void
NAME_(core_fault_set4)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 4, val);
}

static void
NAME_(core_fault_set5)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 5, val);
}

static void
NAME_(core_fault_set6)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 6, val);
}

static void
NAME_(core_fault_set7)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 7, val);
}


static void
NAME_(core_fault_set8)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 8, val);
}

static void
NAME_(core_fault_set9)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 9, val);
}

static void
NAME_(core_fault_set10)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 10, val);
}

static void
NAME_(core_fault_set11)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 11, val);
}

static void
NAME_(core_fault_set12)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 12, val);
}

static void
NAME_(core_fault_set13)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 13, val);
}

static void
NAME_(core_fault_set14)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 14, val);
}

static void
NAME_(core_fault_set15)(void *_cpssp, unsigned int val)
{
	NAME_(core_fault_set)(_cpssp, 15, val);
}

static void
NAME_(connect)(void *_cpssp, const char *port, void *_sig)
{
	static const struct sig_boolean_funcs fault_funcs[] = {
		{ .set = NAME_(core_fault_set0), },
		{ .set = NAME_(core_fault_set1), },
		{ .set = NAME_(core_fault_set2), },
		{ .set = NAME_(core_fault_set3), },
		{ .set = NAME_(core_fault_set4), },
		{ .set = NAME_(core_fault_set5), },
		{ .set = NAME_(core_fault_set6), },
		{ .set = NAME_(core_fault_set7), },
		{ .set = NAME_(core_fault_set8), },
		{ .set = NAME_(core_fault_set9), },
		{ .set = NAME_(core_fault_set10), },
		{ .set = NAME_(core_fault_set11), },
		{ .set = NAME_(core_fault_set12), },
		{ .set = NAME_(core_fault_set13), },
		{ .set = NAME_(core_fault_set14), },
		{ .set = NAME_(core_fault_set15), },
	};
	struct cpssp *cpssp = _cpssp;
	struct sig_boolean *sig = (struct sig_boolean *) _sig;
	unsigned int regno;
	unsigned int bitno;
	unsigned int i;

	assert(sizeof(fault_funcs) / sizeof(fault_funcs[0])
			== sizeof(cpssp->fault) / sizeof(cpssp->fault[0]));

	if (NAME_(core_port)(cpssp, port, &regno, &bitno)) {
		return;
	}
fprintf(stderr, "%s %s\n", __FUNCTION__, port);

	/* Lookup Unused Entry */
	for (i = 0; ; i++) {
		assert(i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]));
		if (! cpssp->fault[i].used) {
			break;
		}
	}

	/* Add Entry */
	cpssp->fault[i].used = 1;
	cpssp->fault[i].regno = regno;
	cpssp->fault[i].bitno = bitno;

	cpssp->fault[i].sig = sig;

	sig_boolean_connect_in(cpssp->fault[i].sig, cpssp, &fault_funcs[i]);
}

static void
NAME_(disconnect)(void *_cpssp, const char *port)
{
	struct cpssp *cpssp = _cpssp;
	unsigned int regno;
	unsigned int bitno;
	unsigned int i;

	if (NAME_(core_port)(_cpssp, port, &regno, &bitno)) {
		return;
	}

	/* Lookup Entry */
	for (i = 0; ; i++) {
		assert(i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]));
		if (cpssp->fault[i].used
		 && cpssp->fault[i].regno == regno
		 && cpssp->fault[i].bitno == bitno) {
			/* Entry found. */
			break;
		}
	}

	// sig_boolean_disconnect_in(cpssp->fault[i].sig, cpssp);

	/* Disable Entry */
	cpssp->fault[i].used = 0;
}

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
void
NAME_(core_startup)(struct cpssp *cpssp, uint8_t vector)
{
	cpssp->state_startup_vec = vector;
	cpssp->state_startup = 1;
}

void
NAME_(core_init_by_apic)(struct cpssp *cpssp)
{
	cpssp->state_ext |= STATE_N_INIT_ACK;
}
#endif

#if 80486 <= CONFIG_CPU
static void
NAME_(core_n_init_set)(struct cpssp *cpssp, unsigned int n_val)
{
	if (n_val) {
		/* Init gone. */
		cpssp->state_ext &= ~STATE_INIT;
	} else {
		/* New init. */
		cpssp->state_ext |= STATE_INIT | STATE_N_INIT_ACK;
	}
}
#endif /* 80386 <= CONFIG_CPU */

static void
NAME_(core_nmi_set)(struct cpssp *cpssp, unsigned int val)
{
	if (val) {
		cpssp->interrupt_request |= CPU_INTERRUPT_NMI;
	}
}

#if 0 /* Not used. */
static void
NAME_(core_nmi_ack)(struct cpssp *cpssp)
{
	cpssp->interrupt_request &= ~CPU_INTERRUPT_NMI;
}
#endif

static void
NAME_(core_irq_set)(struct cpssp *cpssp, unsigned int val)
{
	if (val) {
		cpssp->interrupt_request |= CPU_INTERRUPT_IRQ;
	} else {
		cpssp->interrupt_request &= ~CPU_INTERRUPT_IRQ;
	}
}

#if 80386 <= CONFIG_CPU
static void
NAME_(core_smi_set)(struct cpssp *cpssp, unsigned int val)
{
	if (val) {
		cpssp->interrupt_request |= CPU_INTERRUPT_SMI;
	}
}

void
NAME_(core_smi_ack)(struct cpssp *cpssp)
{
	cpssp->interrupt_request &= ~CPU_INTERRUPT_SMI;
}
#endif

static void
NAME_(core_n_ignne_set)(struct cpssp *cpssp, unsigned int n_val)
{
	cpssp->state_n_ignne = n_val;
}

static void
NAME_(core_n_reset_set)(struct cpssp *cpssp, unsigned int n_val)
{
	if (n_val) {
		/* Reset gone. */
		cpssp->state_ext &= ~STATE_RESET;
	} else {
		/* New reset. */
		cpssp->state_ext |= STATE_RESET | STATE_N_RESET_ACK;
	}
}

static void
NAME_(core_power_set)(struct cpssp *cpssp, unsigned int val)
{
	if (val) {
		cpssp->state_ext |= STATE_POWER;
	} else {
		cpssp->state_ext &= ~STATE_POWER;
	}
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_BIOS
#undef DEBUG_CONTROL_FLOW
#undef DEBUG_CONTROL_FLOW_FLAGS
#undef DEBUG_CONTROL_FLOW_REGS
#undef DEBUG_CONTROL_FLOW_CREGS
#undef DEBUG_CONTROL_FLOW_SREGS
