/*
 * Copyright (C) 2014-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * Port Control and Interrupts (PORT)
 */

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */
#ifdef STATE

struct {
	unsigned int state_ext[32];
	unsigned int state_int[32][CONFIG_NALTS];

	/* 11.5.1 */
	uint8_t isf[32];
	uint8_t irqc[32];
	uint8_t mux[32];
	uint8_t dse[32];
	uint8_t pfe[32];
	uint8_t sre[32];
	uint8_t pe[32];
	uint8_t ps[32];

	uint8_t state[32];
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
NAME_(internalNM_in_set)(struct cpssp *cpssp, int n, int m, unsigned int);
/*forward*/ static void
NAME_(reset)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

static void
NAME_(update_mux)(struct cpssp *cpssp, int n)
{
	unsigned int alt;
	unsigned int val;

	/* Set internal outputs. */
	for (alt = 0; alt < CONFIG_NALTS; alt++) {
		if (alt == cpssp->NAME.mux[n]) {
			val = cpssp->NAME.state_ext[n];
		} else {
			val = SIG_STD_LOGIC_Z;
		}
		NAME_(internalNM_out_set)(cpssp, alt, n, val);
	}

	/* Set external output. */
	val = cpssp->NAME.state_int[n][cpssp->NAME.mux[n]];
	NAME_(pinN_out_set)(cpssp, n, val);
}

static void
NAME_(update_irq)(struct cpssp *cpssp)
{
	int irq;
	int n;

	irq = 0;
	for (n = 0; n < 32; n++) {
		irq |= cpssp->NAME.isf[n];
	}

	NAME_(irq_set)(cpssp, irq);
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	int n;

	assert(bs == 0b1111);

	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000 ... 0x07c:
		/* Pin Control Register n */
		/* 11.5.1 */
		n = addr >> 2;
		if ((bs >> 3) & 1) {
			*valp |= cpssp->NAME.isf[n] << 24;
		}
		if ((bs >> 2) & 1) {
			*valp |= cpssp->NAME.irqc[n] << 16;
		}
		if ((bs >> 1) & 1) {
			*valp |= cpssp->NAME.mux[n] << 8;
		}
		if ((bs >> 0) & 1) {
			*valp |= cpssp->NAME.dse[n] << 6;
			*valp |= cpssp->NAME.pfe[n] << 4;
			*valp |= cpssp->NAME.sre[n] << 2;
			*valp |= cpssp->NAME.pe[n] << 1;
			*valp |= cpssp->NAME.ps[n] << 0;
		}
		break;

	case 0x080:
		/* Global Pin Control Low Register */
		/* 11.5.2 */
		*valp = 0x00000000; /* Write only register */
		break;

	case 0x084:
		/* Global Pin Control High Register */
		/* 11.5.3 */
		*valp = 0x00000000; /* Write only register */
		break;

	case 0x0a0:
		/* Interrupt Status Flag Register */
		for (n = 0; n < 32; n++) {
			*valp |= cpssp->NAME.isf[n] << n;
		}
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		*valp = 0;
		break;
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	int n;

	assert(bs == 0b1111);

	addr &= 0x1000 - 1;

	switch (addr) {
	case 0x000 ... 0x07c:
		/* Pin Control Register n */
		/* 11.5.1 */
		n = addr >> 2;
		if ((bs >> 3) & 1) {
			/* 31-25: reserved */
			cpssp->NAME.isf[n] &= ~((val >> 24) & 1);
		}
		if ((bs >> 2) & 1) {
			/* 23-20: reserved */
			cpssp->NAME.irqc[n] = (val >> 16) & 0xf;
		}
		if ((bs >> 1) & 1) {
			/* 15-11: reserved */
			cpssp->NAME.mux[n] = (val >> 8) & 0x7;
		}
		if ((bs >> 0) & 1) {
			/* 7: reserved */
			cpssp->NAME.dse[n] = (val >> 6) & 1;
			/* 5: reserved */
			cpssp->NAME.pfe[n] = (val >> 4) & 1;
			/* 3: reserved */
			cpssp->NAME.sre[n] = (val >> 2) & 1;
			cpssp->NAME.pe[n] = (val >> 1) & 1;
			cpssp->NAME.ps[n] = (val >> 0) & 1;
		}
		NAME_(update_mux)(cpssp, n);
		NAME_(update_irq)(cpssp);
		break;

	case 0x080:
		/* Global Pin Control Low Register */
		/* 11.5.2 */
		for (n = 0; n < 16; n++) {
			if ((val >> (n+16)) & 1) {
				/* Update register 'n'. */
				/* 15-11: reserved */
				cpssp->NAME.mux[n] = (val >> 8) & 0x7;
				/* 7: reserved */
				cpssp->NAME.dse[n] = (val >> 6) & 1;
				/* 5: reserved */
				cpssp->NAME.pfe[n] = (val >> 4) & 1;
				/* 3: reserved */
				cpssp->NAME.sre[n] = (val >> 2) & 1;
				cpssp->NAME.pe[n] = (val >> 1) & 1;
				cpssp->NAME.ps[n] = (val >> 0) & 1;

				NAME_(update_mux)(cpssp, n);
			}
		}
		break;

	case 0x084:
		/* Global Pin Control High Register */
		/* 11.5.3 */
		for (n = 16; n < 32; n++) {
			if ((val >> n) & 1) {
				/* Update register 'n'. */
				/* 15-11: reserved */
				cpssp->NAME.mux[n] = (val >> 8) & 0x7;
				/* 7: reserved */
				cpssp->NAME.dse[n] = (val >> 6) & 1;
				/* 5: reserved */
				cpssp->NAME.pfe[n] = (val >> 4) & 1;
				/* 3: reserved */
				cpssp->NAME.sre[n] = (val >> 2) & 1;
				cpssp->NAME.pe[n] = (val >> 1) & 1;

				NAME_(update_mux)(cpssp, n);
			}
		}
		break;

	case 0x0a0:
		/* Interrupt Status Flag Register */
		/* 11.5.4 */
		for (n = 0; n < 32; n++) {
			cpssp->NAME.isf[n] &= ~((val >> n) & 1);
		}
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
	}
}

#if 0 /* Not used. FIXME */
static void
NAME_(_state_N_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	switch (val) {
	case SIG_STD_LOGIC_0:
	case SIG_STD_LOGIC_L:
		val = 0;
		break;
	case SIG_STD_LOGIC_1:
	case SIG_STD_LOGIC_H:
		val = 0;
		break;
	default:
		val = 0; /* FIXME */
		break;
	}

	switch (cpssp->NAME.irqc[n]) {
	case 0x0: /* irq disabled */
		break;
	case 0x8: /* irq when logic zero */
		cpssp->NAME.isf[n] = val == 0;
		break;
	case 0x9: /* irq on rising edge */
		cpssp->NAME.isf[n] = (! cpssp->NAME.state[n]) & val;
		break;
	case 0xa: /* irq on falling edge */
		cpssp->NAME.isf[n] = cpssp->NAME.state[n] & (! val);
		break;
	case 0xb: /* irq on either edge */
		cpssp->NAME.isf[n] = cpssp->NAME.state[n] ^ val;
		break;
	case 0xc: /* irq when logic one */
		cpssp->NAME.isf[n] = val == 1;
		break;
	default:
		assert(0); /* FIXME */
	}

	cpssp->NAME.state[n] = val;

	NAME_(update_irq)(cpssp);
}
#endif

static void
NAME_(internalNM_in_set)(struct cpssp *cpssp, int n, int m, unsigned int val)
{
	assert(0 <= n && n < CONFIG_NALTS);
	assert(0 <= m && m < 32);

	cpssp->NAME.state_int[m][n] = val;

	if (cpssp->NAME.mux[m] == n) {
		NAME_(pinN_out_set)(cpssp, m, val);
	}
}

static void
NAME_(pinN_in_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(0 <= n && n < 32);

	cpssp->NAME.state_ext[n] = val;

	NAME_(internalNM_out_set)(cpssp, cpssp->NAME.mux[n], n, val);
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	int n;

	/* See 10.2.1 */
	for (n = 0; n < 32; n++) {
		/* Some exceptions - FIXME */
		cpssp->NAME.ps[n] = 1;
		cpssp->NAME.pe[n] = 0;
		cpssp->NAME.dse[n] = 0;
		cpssp->NAME.sre[n] = 1;
		cpssp->NAME.mux[n] = 0b000;
		cpssp->NAME.pfe[n] = 0;
		cpssp->NAME.irqc[n] = 0b000;
		cpssp->NAME.isf[n] = 0;
	}
	NAME_(update_irq)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	unsigned int pin;
	unsigned int alt;

	for (pin = 0; pin < 32; pin++) {
		for (alt = 0; alt < CONFIG_NALTS; alt++) {
			cpssp->NAME.state_int[pin][alt] = SIG_STD_LOGIC_Z;
		}
		cpssp->NAME.state_ext[pin] = SIG_STD_LOGIC_Z;
	}
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
