# Copyright (C) 2010-2012 Marie E. Rognes
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Johan Hake 2011
# Modified by Anders Logg 2011
#
# Last changed: 2012-01-10

import ufl
import dolfin.cpp as cpp

from ufl.algorithms import extract_arguments
from dolfin import FunctionSpace, MixedFunctionSpace, VectorFunctionSpace
from dolfin import Function, TrialFunction, Argument


__all__ = ["derivative", "adjoint", "increase_order", "tear"]

def adjoint(form, reordered_arguments=None):

    # Call UFL directly if new arguments are provided directly
    if reordered_arguments is not None:
        return ufl.adjoint(form, reordered_arguments=reordered_arguments)

    # Extract form arguments
    arguments = extract_arguments(form)
    if any(arg.part() != None for arg in arguments):
        cpp.dolfin_error("formmanipulation.py",
                         "compute adjoint of form",
                         "parts not supported")
    if not (len(arguments) == 2):
        cpp.dolfin_error("formmanipulation.py",
                         "compute adjoint of form",
                         "Form is not bilinear")

    # Define new Argument(s) in the same spaces
    # (NB: Order does not matter anymore here because number is absolute)
    v_1 = Argument(arguments[1].function_space(), arguments[0].number(), arguments[0].part())
    v_0 = Argument(arguments[0].function_space(), arguments[1].number(), arguments[1].part())

    # Call ufl.adjoint with swapped arguments as new arguments
    return ufl.adjoint(form, reordered_arguments=(v_1, v_0))

adjoint.__doc__ = ufl.adjoint.__doc__

def derivative(form, u, du=None, coefficient_derivatives=None):
    if du is None:
        # Get existing arguments from form and position the new one with the next argument number
        from ufl.algorithms import extract_arguments
        form_arguments = extract_arguments(form)

        number = max([-1] + [arg.number() for arg in form_arguments]) + 1

        if any(arg.part() is not None for arg in form_arguments):
            cpp.dolfin_error("formmanipulation.py",
                             "compute derivative of form",
                             "Cannot automatically create third argument using parts, please supply one")
        part = None

        if isinstance(u, Function):
            V = u.function_space()
            du = Argument(V, number, part)
        elif isinstance(u, (list,tuple)) and all(isinstance(w, Function) for w in u):
            V = MixedFunctionSpace([w.function_space() for w in u])
            du = ufl.split(Argument(V, number, part))
        else:
            cpp.dolfin_error("formmanipulation.py",
                             "compute derivative of form",
                             "Cannot automatically create third argument, please supply one")
    return ufl.derivative(form, u, du, coefficient_derivatives)

derivative.__doc__ = ufl.derivative.__doc__

def increase_order(V):
    """
    For a given function space, return the same space, but with a
    higher polynomial degree
    """

    n = V.num_sub_spaces()
    if n > 0:
        spaces = []
        for i in range(n):
            V_i = V.sub(i)
            element = V_i.ufl_element()
            # Handle VectorFunctionSpaces specially
            if isinstance(element, ufl.VectorElement):
                spaces += [VectorFunctionSpace(V_i.mesh(),
                                               element.family(),
                                               element.degree() + 1,
                                               dim=element.num_sub_elements())]
            # Handle all else as MixedFunctionSpaces
            else:
                spaces += [increase_order(V_i)]

        return MixedFunctionSpace(spaces)

    if V.ufl_element().family() == "Real":
        return FunctionSpace(V.mesh(), "Real", 0)

    return FunctionSpace(V.mesh(), V.ufl_element().family(),
                         V.ufl_element().degree() + 1)

def change_regularity(V, family):
    """
    For a given function space, return the corresponding space with
    the finite elements specified by 'family'. Possible families
    are the families supported by the form compiler
    """

    n = V.num_sub_spaces()
    if n > 0:
        return MixedFunctionSpace([change_regularity(V.sub(i), family)
                                   for i in range(n)])

    element = V.ufl_element()
    shape = element.value_shape()
    if not shape:
        return FunctionSpace(V.mesh(), family, element.degree())

    return MixedFunctionSpace([FunctionSpace(V.mesh(), family, element.degree())
                               for i in range(shape[0])])

def tear(V_h):
    """
    For a given function space, return the corresponding discontinuous
    space
    """
    W = change_regularity(V_h, "DG")
    return W

