/* term.cxx
     $Id: term.cxx,v 1.4 2001/11/27 23:57:36 elf Exp $

   written by Marc Singer
   27 July 1997

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1997 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   with your Debian GNU/Linux system, in
   /usr/share/common-licenses/GPL, or with the Debian GNU/Linux hello
   source package as the file COPYING. If not, write to the Free
   Software Foundation, Inc., 59 Temple Place -Suite 330, MA
   02111-1307, USA.

   -----------
   DESCRIPTION
   -----------

   Console or terminal I/O routines.  Upon this foundation will be
   built windowing.  Beneath it is the terminal capabilities.

*/

#include "std.h"

#include "termcap.h"
#include "term.h"

#define CB_BUFFER_MAX		(4*1024)

Termcap LTerm::g_termcap;


/* LTerm::set_attribute

   is one of the more complex functions in LTerm because of the
   unpredictability of the termcap features.  For example, some
   terminals implement sequences to enable attributes individually,
   but can only disable them as a group.  We do our best here to make
   sure that we send an optimal sequence to the terminal to change the
   settings as requested.

   The important thing to notice is that the current attribute value
   may change during this update.  Resetting bold may require
   reasserting other attributes that were cleared with bold.

*/

void LTerm::set_attribute (unsigned16 attribute, unsigned16 mask)
{
  attribute = ((m_attribute & ~mask) | (attribute & mask));

				// Map bright colors to emboldend ones
  if (mask & attrFgBright)
    attribute = (attribute & ~attrBold)
		 | ((attribute & attrFgBright) ? attrBold : 0);

				// Disable attributes first
  if (!(attribute & attrBold) && (m_attribute & attrBold)) {
    m_cb += g_termcap.compose (termcapEndMode, m_pb + m_cb);
    m_attribute &= ~(attrAllMode & ~attrBold);
  }
  if (!(attribute & attrUnderline) && (m_attribute & attrUnderline)) {
    if (g_termcap.has (termcapEndUnderline))
      m_cb += g_termcap.compose (termcapEndUnderline, m_pb + m_cb);
    else {
      m_cb += g_termcap.compose (termcapEndMode, m_pb + m_cb);
      m_attribute &= ~(attrAllMode & ~attrUnderline);
    }
  }
  if (!(attribute & attrInverse) && (m_attribute & attrInverse)) {
    m_cb += g_termcap.compose (termcapEndMode, m_pb + m_cb);
    m_attribute &= ~(attrAllMode & ~attrInverse);
  }
  if (!(attribute & attrStandout) && (m_attribute & attrStandout)) {
    if (g_termcap.has (termcapEndStandout))
      m_cb += g_termcap.compose (termcapEndStandout, m_pb + m_cb);
    else {
      m_cb += g_termcap.compose (termcapEndMode, m_pb + m_cb);
      m_attribute &= ~(attrAllMode & ~attrStandout);
    }
  }
  if (!(attribute & attrBlink) && (m_attribute & attrBlink)) {
    m_cb += g_termcap.compose (termcapEndMode, m_pb + m_cb);
    m_attribute &= ~(attrAllMode & ~attrBlink);
  }

				// Enable missing attributes
  if ((attribute & attrBold) && !(m_attribute & attrBold))
    m_cb += g_termcap.compose (termcapStartBold, m_pb + m_cb);
  if ((attribute & attrUnderline) && !(m_attribute & attrUnderline))
    m_cb += g_termcap.compose (termcapStartUnderline, m_pb + m_cb);
  if ((attribute & attrInverse) && !(m_attribute & attrInverse))
    m_cb += g_termcap.compose (termcapStartInverse, m_pb + m_cb);
  if ((attribute & attrStandout) && !(m_attribute & attrStandout))
    m_cb += g_termcap.compose (termcapStartStandout, m_pb + m_cb);
  if ((attribute & attrBlink) && !(m_attribute & attrBlink))
    m_cb += g_termcap.compose (termcapStartBlink, m_pb + m_cb);

				// Set color attribute
  if ((attribute & attrFgMask) != (m_attribute & attrFgMask)) {
    unsigned16 color = (attribute & attrFgMask);
    if (color == attrFgDefault)
      m_cb += g_termcap.compose (termcapSetAnsiForeground, m_pb + m_cb, 9);
    else
      m_cb += g_termcap.compose (termcapSetAnsiForeground, m_pb + m_cb, 
				 (color - attrFgFirst) >> attrFgShift);
  }
  if ((attribute & attrBgMask) != (m_attribute & attrBgMask)) {
    unsigned16 color = (attribute & attrBgMask);
    if (color == attrBgDefault)
      m_cb += g_termcap.compose (termcapSetAnsiBackground, m_pb + m_cb, 9);
    else
      m_cb += g_termcap.compose (termcapSetAnsiBackground, m_pb + m_cb, 
				 (color - attrBgFirst) >> attrBgShift);
  }

  m_attribute = attribute;
}

void LTerm::flush (void)
{
  if (m_cb) {
    write (m_fh, m_pb, m_cb);
    memset (m_pb, 0, m_cb);	// *** FIXME debug code
    m_cb = 0;
  }
}


void LTerm::init (void)
{
  g_termcap.init ();

  if ((m_pb = (char*) malloc (CB_BUFFER_MAX)))
    m_cbMax = CB_BUFFER_MAX;
  m_fh = 1;			// Standard out
}


void LTerm::release_this (void)
{
  if (m_pb) {
    free (m_pb);
    m_pb = NULL;
    m_cbMax = 0;
  }
}

int LTerm::sprintf (char* pb, ...)
{
  va_list ap;
  va_start (ap, pb);

  int result = vsprintf (m_pb + m_cb, pb, ap);
  va_end (ap);

  m_cb += result;
  return result;
}
