/* getstr.cxx
     $Id: getstr.cxx,v 1.12 2001/11/27 23:57:36 elf Exp $

   written by Marc Singer
   3 October 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   with your Debian GNU/Linux system, in
   /usr/share/common-licenses/GPL, or with the Debian GNU/Linux hello
   source package as the file COPYING. If not, write to the Free
   Software Foundation, Inc., 59 Temple Place -Suite 330, MA
   02111-1307, USA.

   -----------
   DESCRIPTION
   -----------

   User input function(s) for well protected interactions with the user.  

*/


#include "std.h"
#include "window.h"
#include "termcap.h"


/* LWindow::getstr

   fetches a string from the user in a nice, emacs sort of way.  The
   return valuse is zero if the user input is accepted.  It is !0 if
   the user pressed escape.

*/

int LWindow::getstr (int x, int y, char* szPrompt, 
		     char* szResponse, int cchResponseMax, 
		     const char* szDefault)
{
  if (szDefault) {
    if (szResponse != szDefault)
      strcpy (szResponse, szDefault);
  }
  else
    *szResponse = 0;

  off (A_REVERSE);
  printfxy (x, y, szPrompt);
  x += strlen (szPrompt);

  int ibSelect = 0, cbSelect = strlen (szResponse);
  on (A_REVERSE);
  printfxy (x, y, szResponse);
  off (A_REVERSE);

  int point = 0;
  int cch = strlen (szResponse);
  int ch;
  int scroll = 0;
  int scrollDelta = (dx () - x)/2;
  int scrollZone = 10;
  while ((ch = getch (point + x - scroll, y)) != '\n') {
    switch (ch) {
    case -1:			// Getstr happening off screen
      assert_ (ch != -1);
      break;
    case 27:			// ESC
    case 7:			// ^G -- abort
      return -1;
    case 1:			// ^A -- beginning of line
      point = 0;
      cbSelect = 0;
      break;
    case 5:			// ^E -- end of line
      point = cch;
      cbSelect = 0; 
      break;
    case 4:			// ^D -- delete
    case KEYCODE (termcapKeyDelete):
      if (point < cch) {
	memmove (szResponse + point, szResponse + point + 1, 
		 cch - point);
	--cch;
      }
      break;

    case 8:			// ^H -- backspace
    case KEYCODE (termcapKeyBackspace):
      if (cbSelect) {
	*szResponse = 0;
	cbSelect = 0;
	point = 0;
	cch = 0;
      }
      else if (point) {
	memmove (szResponse + point - 1, szResponse + point,
		 cch - point + 1);
	--point;
	--cch;
      }
      break;

    case 11:			// ^K -- delete to end of line
      if (cbSelect) {
	memmove (szResponse + ibSelect,
		 szResponse + ibSelect + cbSelect,
		 cch - ibSelect - cbSelect + 1);
	cch -= cbSelect;
	point = ibSelect;
	cbSelect = 0;
      }
      else {
	szResponse[point] = 0;
	cch = point;
      }
      break;

    case 2:			// ^B -- backward character
    case KEYCODE (termcapKeyLeft):
      if (point)
	--point;
      cbSelect = 0;
      break;

    case 6:			// ^F -- forward character
    case KEYCODE (termcapKeyRight):
      if (point < cch)
	++point;
      cbSelect = 0;
      break;

    default:
      if (isprint (ch)) {
	if (cbSelect) {		// Replace selection
	  memmove (szResponse + ibSelect + 1, 
		   szResponse + ibSelect + cbSelect,
		   cch - ibSelect - cbSelect + 1);
	  cch -= cbSelect;
	  point = ibSelect;
	  cbSelect = 0;
	}
	else {
	  if (cch >= cchResponseMax - 1) // Leave room for nul, right?
	    continue;		// FIXME: how about a beep here
	  if (point != cch)
	    memmove (szResponse + point + 1, szResponse + point,
		     cch - point + 1);
	  else {
	    assert_ (point < cchResponseMax - 1);
	    szResponse[point + 1] = 0;
	  }
	}
	szResponse[point++] = ch;
	++cch;
      }
      break;			// FIXME: how about a beep on bad input
    }
    // FIXME: We need to break-up the display so we show the
    //        highlighted portion AND the unhighlighted portion.
    if (cbSelect)
      on (A_REVERSE);
    else
      off (A_REVERSE);
				// Check for scroll changes
    while (point - scroll >= dx () - x)
      scroll += scrollDelta;
    while (scroll && point < scroll + scrollZone)
      scroll -= scrollDelta;

    printfxy (x, y, "%-*.*s", dx () - x, dx () - x, szResponse + scroll);
    if (cbSelect)
      off (A_REVERSE);
  }
  return 0;
}
		     

