/* cvs.cxx
     $Id: cvs.cxx,v 1.26 2001/11/27 23:57:36 elf Exp $

   written by Marc Singer
   20 September 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   with your Debian GNU/Linux system, in
   /usr/share/common-licenses/GPL, or with the Debian GNU/Linux hello
   source package as the file COPYING. If not, write to the Free
   Software Foundation, Inc., 59 Temple Place -Suite 330, MA
   02111-1307, USA.

   -----------
   DESCRIPTION
   -----------

   CVS specific control functions.

   --------------------
   IMPLEMENTATION NOTES
   --------------------

   m_rgFlags:			File and directory flags
   
   This array maps one to one to directory listing in
   m_pDir->m_rgInfo.  Each item there is examined and the flags are
   set appropriately.

   *** This has been superceeded by a user field in the directory
   structures.  We need this because the maintainence of parallel data
   structures prohibits sorting.

*/


#include "std.h"
#include "preferences.h"	// Everyone MUST get into the pref scene

#include <sys/stat.h>
#include <fnmatch.h>
#include <time.h>
#include "cvs.h"
#include "path.h"
#include "file.h"
#include "mman.h"

static char g_pchIgnoreDefault[] = "core RCSLOG tags TAGS RCS SCCS .make.state"
	" .nse_depinfo #* .#* cvslog.* ,* CVS* .del-* *.a *.o *.so *.Z"
	" *~ *.old *.elc *.ln *.bak *.BAK *.orig *.rej";
char* LCVS::g_pchIgnore;


void LCVS::analyze (void)
{
  if (!m_fControlled)
    return;

  assert_ (m_pDir);
  char sz[256];
  strcpy (sz, m_pDir->path ());
  int cchPath = strlen (sz);
  strcat (sz, "/");
  strcat (sz, "CVS/Entries");
  struct stat stat;
  if (::stat (sz, &stat) || !S_ISREG (stat.st_mode))
    return;

				// --- Check for changes to CVS/Entries
  int fh = -1;			// Done for the compiler and it's warnings
  char* pbEntries = NULL;
  if (stat.st_mtime != m_timeEntries) {
    m_timeEntries = stat.st_mtime;
  
    int cbEntries = stat.st_size;
    fh = open (sz, O_RDONLY);
    pbEntries = (char*) malloc (cbEntries + 1);
    pbEntries[cbEntries] = 0;
    assert_ (fh != -1);
    read (fh, pbEntries, cbEntries);
    close (fh);
  }

				// --- Read and parse ignore data
  TRACE((T_CVS_TRACE, "parsing .cvsignore"));
  strcpy (sz + cchPath, "/.cvsignore");
  if (!::stat (sz, &stat) && S_ISREG (stat.st_mode)
      && (fh = open (sz, O_RDONLY))) {
    char* pch = (char*) malloc (stat.st_size + 1);
    read (fh, pch, stat.st_size);
    pch[stat.st_size] = 0;
    close (fh);
    m_pchIgnore = _parse_ignore (pch);
    free (pch);
  }
  if (!g_pchIgnore) 
    g_pchIgnore = _parse_ignore (g_pchIgnoreDefault);

				// -- Parse the CVS/Entries file
  if (pbEntries) {
    TRACE((T_CVS_TRACE, "parsing CVS/Entries"));
    char* pb = pbEntries;
    bool fDirectory;
    while (*pb) {
      fDirectory = (*pb == 'D');
      assert_ (fDirectory || *pb == '/');
      if (!fDirectory) {
	++pb;
	int cch = strcspn (pb, "/");
	int iFile = m_pDir->find (pb, cch);
	if (iFile >= 0)
	  set_flag (iFile, flagControlled);
	pb += cch + 1;
	cch = strcspn (pb, "/");
	bool fVersion = set_version (pb, cch);
	pb += cch + 1;
	cch = strcspn (pb, "/");
	time_t time = fVersion ? recover_time (pb, cch) : 0;
//  const char* szNewTime = asctime (gmtime (&time));
//  fprintf (stderr, "From %*.*s\n  to %s\n", cch, cch, pb, szNewTime);
	if (m_pDir->time (iFile) != time)
	  set_flag (iFile, flagEdited);
      }
      pb += strcspn (pb, "\n");
      if (*pb)
	++pb;
    }
    free (pbEntries);
  }

  bool fSenseBinary = g_preferences.as_bool ("SenseBinaryFiles");

  TRACE((T_CVS_TRACE, "stat'ing files"));
  for (int iFile = 0; iFile < m_pDir->count_files (); ++iFile) {
    if (!m_pDir->is_file (iFile))
      continue;
    if (flag (iFile) & flagControlled)
      continue;
    LPath path (m_pDir->path ());
    path += m_pDir->file (iFile);
    if (fSenseBinary && file_guess_binary (path))
      set_flag (iFile, flagBinary);
    if (!_is_ignored (m_pDir->file (iFile)))
      continue;
    set_flag (iFile, flagIgnored);
  }

  TRACE((T_CVS_TRACE, "parsing status data"));
  parse_status ();
  TRACE((T_CVS_TRACE, "cvs.analyze complete"));
}

/* LCVS::fetch_status

   requests file status information from CVS.  If the directory is not
   controlled, it will not call cvs.  If the szFiles parameter is not
   NULL, it will be appended to the end of the status request command
   thus allowing the status of specific files to be requested.

   FIXME: presently, szFiles can be only one file.

   If the UseStatusHash preference is true, we will try to use the -H
   switch for cvs status which can improve the performance of the call
   dramatically.

*/

void LCVS::fetch_status (const char* szFiles)
{
  release_filestatus ();
  if (!m_fControlled)
    return;
  
     // *** Replaced tmpnam with mkstemp, but did not smooth over the
     // code that uses it.  This isn't really cool since mkstemp is
     // designed to open the file along with making a name for it.
     // I've done this to eliminate a compiler warning.
     // Unfortunately, I'm doing an end-run around the real problem.
  char szFileTemp[sizeof (P_tmpdir) + 20];
  strcpy (szFileTemp, P_tmpdir);
  strcat (szFileTemp, "/curvesXXXXXX");
  int fhTemp = mkstemp (szFileTemp);
  if (fhTemp != -1)
    close (fhTemp);
  //  char* szFileTemp = tmpnam (NULL);

  m_szFileStatus = (char*) malloc (strlen (szFileTemp) + 1);
  strcpy (m_szFileStatus, szFileTemp);

  char* sz = (char*) malloc ((szFiles ? strlen (szFiles) : 0) + CB_COMMAND*2
			     + strlen (m_pDir->path ()));
  bool fOK = false;
  if (g_preferences.as_bool ("UseStatusHash")) {
    sprintf (sz, "cd %s ; %s status -l -H %s > %s 2>&1", 
	     m_pDir->path (), g_preferences.fetch ("cvs"), 
	     szFiles ? szFiles : "", m_szFileStatus);
    TRACE((T_CVS_TRACE, "Executing: '%s'\n", sz));
    
    fOK = (system (sz) == 0);
  }
  if (!fOK) {
    sprintf (sz, "cd %s ; %s status -l %s > %s 2>&1", 
	     m_pDir->path (), g_preferences.fetch ("cvs"),
	     szFiles ? szFiles : "", m_szFileStatus);
    TRACE((T_CVS_TRACE, "Executing: '%s'\n", sz));
    system (sz);
  }
  free (sz);
}

void LCVS::init (LDirectory* pDir)
{
  release_this ();

  m_pDir = pDir;

  m_fControlled = false;	// Default state

  assert_ (m_pDir);
  char sz[256];
  strcpy (sz, m_pDir->path ());
  strcat (sz, "/");
  strcat (sz, "CVS");
  struct stat stat;
  if (::stat (sz, &stat) || !S_ISDIR (stat.st_mode))
    return;
  m_fControlled = true;
}


/* LCurves::_is_ignored

   attempts to match the given filename with one of the patterns in
   the list of ignored files.

*/

bool LCVS::_is_ignored (const char* szFile)
{
  if (m_pchIgnore) {
    char* pch = m_pchIgnore;
    while (pch && *((short*) pch)) {
      if (fnmatch (pch + sizeof (short), szFile, 
		   FNM_NOESCAPE | FNM_PERIOD) == 0)
	return true;
      pch += sizeof (short) + *((short*) pch);
    }
  }

  if (g_pchIgnore) {
    char* pch = g_pchIgnore;
    while (pch && *((short*) pch)) {
      if (fnmatch (pch + sizeof (short), szFile, 
		   FNM_NOESCAPE | FNM_PERIOD) == 0)
	return true;
      pch += sizeof (short) + *((short*) pch);
    }
  }

  return false;
}


/* LCVS::parse_status

   parses the fetched filestatus information.  Note this this
   information may be a subset of all of the files in the controlled
   directory.

   FIXME: This parse routine may be prone to failure because of the
   inability of many string manipulation routines to accept byte
   limits.  We memory map the status file, but we do not prevent
   reading past the end of file.  However, more to the point, we
   should avoid scanning the file more than once.  The memory mapping
   makes it easy to do the kind of parsing we're used to, but we want
   to be able to parse in one pass.  For speed.

*/

void LCVS::parse_status (void) 
{
  int fh = -1;
  char* pbMap = NULL;
  struct stat stat;
  if (!::stat (m_szFileStatus, &stat) && stat.st_size
      && ((fh = open (m_szFileStatus, O_RDONLY)) != -1)
      && (pbMap = (char*) mmap (NULL, stat.st_size + 1, PROT_READ, 
				MAP_FILE | MAP_PRIVATE, fh, 0))) {
    int cchEnd;
    for (char* pb = pbMap; pb < pbMap + stat.st_size; pb += cchEnd +
	   ((pb[cchEnd] == '\n') ? 1 : 0)) {
      cchEnd = strcspn (pb, "\n");
      if (cchEnd < 5)
	continue;
      if (strncasecmp (pb, "File: ", 6) != 0)
	continue;

      char* pbFile = pb + 6;
      if (strncasecmp (pbFile, "no file ", 8) == 0)
	pbFile += 8;
      int iFile = m_pDir->find (pbFile, strcspn (pbFile, "\n \t"));
      if (iFile == -1)
	iFile = m_pDir->add_file (pbFile, strcspn (pbFile, "\n \t"));
      assert_ (iFile != -1);
      char* pbStatus = strstr (pbFile, "Status: ");
      assert_ (pbStatus < pb + cchEnd);
      pbStatus += 8;
      clear_flag (iFile, flagTransient);
      set_flag (iFile, flagControlled);
      if (     strncasecmp (pbStatus, "Up-to-date", 10) == 0)
	continue;
      else if (strncasecmp (pbStatus, "Needs Checkout", 14) == 0)
	set_flag (iFile, flagStatusCheckout);
      else if (strncasecmp (pbStatus, "Needs Patch", 11) == 0)
	set_flag (iFile, flagStatusPatch);
      else if (strncasecmp (pbStatus, "Needs Merge", 11) == 0)
	set_flag (iFile, flagStatusMerge);
      else if (strncasecmp (pbStatus, "File had conflicts on merge", 27) == 0)
	set_flag (iFile, flagStatusConflict);
      else if (strncasecmp (pbStatus, "Locally Modified", 16) == 0)
	set_flag (iFile, flagEdited);
      else if (strncasecmp (pbStatus, "Locally Added", 13) == 0)
	set_flag (iFile, flagAdded);
      else if (strncasecmp (pbStatus, "Locally Removed", 15) == 0)
	set_flag (iFile, flagRemoved);
      else if (strncasecmp (pbStatus, "Unknown", 7) == 0)
	continue;
      else if (strncasecmp (pbStatus, "Entry Invalid", 13) == 0)
	continue;
      else if (strncasecmp (pbStatus, "Unresolved Conflict", 19) == 0)
	continue;
      else
	assert_ (0);
    }
  }
  if (pbMap)
    munmap (pbMap, stat.st_size);
  if (fh != -1)
    close (fh);
}


/* LCVS::_parse_ignore

   processes the data from either a .cvsignore file or from the
   default list of ignored patterns.  The malloc'd buffer returned can
   be used to efficiently scan the patterns for matches.

   The output is a buffer of data structured with short (2 byte)
   counts of characters that offset to the next pattern.  If the count
   is zero, the end of the list has been achieved.

*/

char* LCVS::_parse_ignore (char* pch)
{
  int cbIgnore = 0;
  int cIgnore = 0;
  int ich = 0;
  do {
    int cch = strcspn (pch + ich, "\n ");
    if (cch) {
      cbIgnore += cch + 1;
      if (!(cch & 1))
	++cbIgnore;		// Permit aligned references to shorts
      ++cIgnore;
    }
    ich += cch;
    if (pch[ich])
      ++ich;
  } while (pch[ich]);
  char* pchIgnoreOrg
    = (char*) malloc (cbIgnore + sizeof (short)*(cIgnore + 1));
  ich = 0;
  char* pchIgnore = pchIgnoreOrg;
  do {
    int cch = strcspn (pch + ich, "\n ");
    if (cch) {
      *((short*) pchIgnore) = (short) cch + 1 + ((cch & 1) ? 0 : 1);
      pchIgnore += sizeof (short);
      strncpy (pchIgnore, pch + ich, cch);
      pchIgnore[cch] = 0;
      pchIgnore += cch + 1;
      if (!(cch & 1))
	++pchIgnore;
    }
    ich += cch;
    if (pch[ich])
      ++ich;
  } while (pch[ich]);
  *((short*) pchIgnore) = 0;	// Terminate list

  return pchIgnoreOrg;
}


/* LCVS::recover_time
   
   recovers the time_t value from a string describing the time in the
   form: Wed Sep 27 01:57:06 1995.

*/

time_t LCVS::recover_time (char* pch, int cch)
{
  //  static const char* m_rgszDay[] = { "...
  struct tm tm;
  memset (&tm, 0, sizeof (tm));
  char sz[80];

  if (cch > (int) sizeof (sz) - 1)
    return 0;
  strncpy (sz, pch, cch);
  sz[cch] = 0;
  pch = sz;
  int ich;
				// Month
  ich = strcspn (pch, " ");
  if (!*pch)
    return 0;
  pch += ich + 1;
  pch[3] = 0;
  static const char* m_szMonths = "JanFebMarAprMayJunJulAugSepOctNovDec";
  tm.tm_mon = ((char*) strstr (m_szMonths, pch) - m_szMonths)/3;

				// Date
  pch += 4;
  while (*pch == ' ')
    ++pch;
  tm.tm_mday = atoi (pch);

				// Hour
  pch += strcspn (pch, " ") + 1;
  tm.tm_hour = atoi (pch);

				// Minute
  pch += strcspn (pch, ":") + 1;
  tm.tm_min = atoi (pch);

				// Second
  pch += strcspn (pch, ":") + 1;
  tm.tm_sec = atoi (pch);

				// Year
  pch += strcspn (pch, " ") + 1;
  tm.tm_year = atoi (pch) - 1900;

  return mktime (&tm);
}


void LCVS::release_filestatus (void)
{
  if (m_szFileStatus) {
    unlink (m_szFileStatus);
    free (m_szFileStatus);
    m_szFileStatus = NULL;
  }
}

void LCVS::release_this (void)
{
  release_filestatus ();

  if (m_pchIgnore) {
    free (m_pchIgnore);
    m_pchIgnore = NULL;
  }
}

/* LCVS::set_version

   stores the file version with the rest of the file status
   information.  FIXME: this is not yet implemented.  We need to
   attach a structure to the directory entry instead of using the
   void* as a bitmask.  Ho hum.  At least we check for new entries so
   we can avoid the time-consuming search for an invalid date.

*/

int LCVS::set_version (char* pch, int cch)
{
  if (cch == 1 && *pch == '0')
    return 0;
  return 1;
}
