;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: server.lisp,v 1.50 2002/03/29 22:55:40 craig Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :imho)

(defun alive-procp (proc)
  (when (processp proc)
    (process-active-p proc)))

;; ------------------------------------------------------------
;; Listener socket control

#+cmu
(defun init/server-socket (state)
  "Control the state of the listening socket."
  (ecase state
    (:start
     (unless *lisp-listener*
       (setf *lisp-listener* (ext:create-inet-listener *lisp-server-port*))))
    (:stop
     (when *lisp-listener*
;;       (unix::unix-close *lisp-listener*)
       (ext:close-socket *lisp-listener*)
       (setq *lisp-listener* nil)))
    (:restart
     (init/server-socket :stop)
     (init/server-socket :start))
    (:report
     (cmsg "~&;; IMHO listener is ~a~%"
           (if *lisp-listener* "running" "not running"))))
  (values))

;; ------------------------------------------------------------
;; Reaping of expired sessions

(defun session-timeout-monitor ()
  (loop
   (ignore-errors (clean-sessions :exec t))
   (sleep *session-timeout-monitor-seconds*)))

(defun clean-sessions (&key exec debug limit)
  (let ((time (get-universal-time)))
    (labels ((check-session (session-key session)
               (and session
                    (with-lock-held (*session-timeout-lock*)
                      (let ((session-life (- time (session-timestamp session))))
                        (when debug
                          (cmsg "session with ID ~a is ~d seconds old" session-key session-life))
                        (when (> session-life (or limit (session-timeout session)))
                          (when debug
                            (cmsg "session ~s is stale" session))
                          (when exec
                            (end-session session)))))))
             (check-application (application-name application)
               (when debug
                 (cmsg "checking application ~a" application-name))
               (maphash #'check-session (application-sessions application))))
      (maphash #'check-application *imho-active-apps*))))

(defun init/timeout-monitor (state)
  "Control the state of the timeout monitor. Argument can be one of
:start, :stop, and :restart."
  (ecase state
    (:start
     (unless (alive-procp *timeout-monitor*)
       (setf *timeout-monitor*
             (make-process #'session-timeout-monitor
                           :name "Session Timeout Monitor"))))
    (:stop
     (when *timeout-monitor*
       (destroy-process *timeout-monitor*)
       (setq *timeout-monitor* nil)))
    (:restart
     (init/timeout-monitor :stop)
     (init/timeout-monitor :start))
    (:report
     (cmsg "session timeout monitor is ~a"
           (if (alive-procp *timeout-monitor*) "running" "not running"))))
  (values))

;; ------------------------------------------------------------
;; Request processing control

(defun cgi-server-failsafe (stream &key (connector-function 'warp-server))
  (handler-case
      (funcall connector-function stream)
    (error ()
      (dformat :errors "CGI - Ejecting error"))))

#+cmu
(defun cgi-process (fd &key (connector-function 'warp-server))
  (lambda ()
    (let ((stream (sys:make-fd-stream fd :input t :output t :element-type 'unsigned-byte)))
      (unwind-protect
           (if *production*
               (cgi-server-failsafe stream
				    :connector-function connector-function)
               (funcall connector-function stream))
        (close stream)))))

#+cmu
(defun cgi-listener (&key (connector-function 'warp-server))
  "This function runs in a thread for the lifetime of the server
process, listening on the IMHO port and spawning client socket
connections."
  (let (socket host retries)
    (loop
     (process-wait-until-fd-usable *lisp-listener* :input)
     (dformat :connections "Listen socket received request")
     (setq retries 0)
     (handler-case
         (multiple-value-setq
             (socket host)
           (ext:accept-tcp-connection *lisp-listener*))
       (error ()
         (dformat :errors "accept-tcp-connection failed, waiting 5 seconds...")
         (setq socket nil)
         (if (= retries *accept-retry-limit*)
             (init/server-socket :restart)
             (progn
               (incf retries)
               (sleep 5)))))
     (when socket
       (make-process (cgi-process socket
				  :connector-function connector-function)
                     :name (ip-address-string host))
       (setq retries 0)))))

(defun lookup-connector (c)
  (cdr (assoc c *imho-connectors*)))

(defun init/imho (state &key (connector :warp))
  "Control the state of the IMHO server. Argument can be one of
:start, :stop, and :restart."
  (ecase state
    (:start
     (unless (alive-procp *lisp-server*)
       #+cmu
       (sys:ignore-interrupt unix:sigpipe)
       (init/server-socket :start)
       (setf *lisp-server*
             (make-process (lambda ()
			     (cgi-listener :connector-function (lookup-connector connector)))
                           :name (format nil "IMHO listener on port ~d" *lisp-server-port*)))
       (init/timeout-monitor :start)
       ))
    (:stop
     (when *lisp-server*
       (init/timeout-monitor :stop)
       (destroy-process *lisp-server*)
       (init/server-socket :stop)
       (setq *lisp-server* nil)))
    (:restart
     (init/imho :stop)
     (init/imho :start))
    (:report
     (cmsg "IMHO server process is ~a"
             (if (alive-procp *lisp-server*) "running" "not running"))
     (init/timeout-monitor :report)
     (init/server-socket :report)))
  (values))

;; ------------------------------------------------------------
;; function passed to export-url

(defun init/application (app-class state &key (connector :warp))
  (ecase state
    (:start
     (maphash (lambda (k v)
                (declare (ignore k))
                (when (equal app-class (type-of v))
                  (return-from init/application)))
              *imho-active-apps*)
     (let* ((app (make-instance app-class))
            (url (base-url app)))
       (setf (gethash url *imho-active-apps*) app)
       (application-startup app)
       (init/imho :start :connector connector)))
    (:stop
     (maphash (lambda (k v)
                (when (equal app-class (type-of v))
                  (application-shutdown v)
                  (remhash k *imho-active-apps*)))
              *imho-active-apps*)
     (when (= 0 (hash-table-count *imho-active-apps*))
       (init/imho :stop :connector connector)))
    (:restart
     (init/application app-class :stop :connector connector)
     (init/application app-class :start :connector connector)))
  (values))
