;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: modal-editor.lisp,v 1.12 2001/11/27 17:30:53 jesse Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.
;;;
;;; A modal editor proxy-renders another class depending on its state.
;;; The editor can operate in a number of modes, each mode displaying
;;; a screen for editing various properties of an object.  Modes are
;;; selected with the mode-selector-popup class (see below).

(in-package :imho)

(defclass modal-editor (html-element)
  ((edit-mode
    :initform nil)
   (propagate
    :initform t
    :initarg :propagate)))

;; sets the first screen when the editor is initialized.
(defmethod default-mode ((self modal-editor))
  'modal-editor-empty)

;; init hook for modal-editor
(defmethod modal-init ((self t))
  )

;; cancel out of a mode, back to the default mode
(defun cancel-editing (class)
  (let ((editor (page-for-session class)))
    (setf (slot-value editor 'edit-mode) nil)))

;; put the editor into a mode
(defmethod start-editing ((self modal-editor) mode)
  (setf (slot-value self 'edit-mode) mode)
  (modal-init (get-modal-editor self)))

;; return a session-page for a mode
(defmethod get-modal-editor ((self modal-editor))
  (let ((mode (or (slot-value self 'edit-mode)
                  (default-mode self))))
    (with-slots (propagate)
        self
      (if propagate
          (page-for-session mode :value (element-value self))
          (page-for-session mode)))))

(defmethod element-active-children ((self modal-editor))
  (list (get-modal-editor self)))
   
(defmethod preawake :before ((self modal-editor))
  (let ((editor (get-modal-editor self)))
    (setf (element-parent editor) self)
    (preawake editor)))
  
(defmethod awake :before ((self modal-editor))
  (let ((editor (get-modal-editor self)))
    (setf (element-parent editor) self)
    (awake editor)))

;; render the mode screen
(defmethod render-html ((self modal-editor) stream)
  (let ((editor (get-modal-editor self)))
    ;; (setf (slot-value editor 'window) nil)
    (setf (element-parent editor) self)
    (render-html editor stream)))

;; ------------------------------------------------------------
;; placeholder class to indicate that a modal editor has not got some
;; default contents defined

(defclass modal-editor-empty (html-element)
  ())

(defmethod render-html ((self modal-editor-empty) stream)
  (html-stream
   stream
   (:p
    (:princ "Empty modal editor"))))

;; ------------------------------------------------------------
;; A popup list tailored to smack a modal-editor into a particular
;; state.

(defclass mode-selector-popup (popup-list)
  ())
  
(defun change-action (url action)
  (let ((sl (position #\/ url :test #'char= :from-end t)))
    (concatenate 'string (subseq url 0 (1+ sl)) action)))


(defmethod generate-js-case ((element mode-selector-popup) alist)
  (let ((fn (make-string-output-stream)))
    (do ((case alist (cdr case)))
        ((null case))
      (if (cdar case)
          (format fn "if (popup_value(this) == '~a') {~%  form.target='';~%form.action='~a';~%form.submit();~%} "
                  (symbol-name (caar case))
                  (change-action (element-url
				  (find-parent element #'(lambda (elem) (typep elem 'modal-editor))))
                                 (concatenate 'string "set-mode?"
                                              (package-name (symbol-package (cdar case)))
                                              "+"
                                              (symbol-name (cdar case)))))
          (format fn "if (false) {} "))
      (when (cdr case)
        (write-string "else " fn)))
    (get-output-stream-string fn)))

(defmethod get-editing-modes ((self mode-selector-popup))
  '("No editing modes"))

(defun generate-cases (modes)
  (mapcar (lambda (mode)
            (if (listp mode)
                (list (gensym) (car mode) (cadr mode))
                (list :null mode nil))) modes))

(defmethod awake :before ((self mode-selector-popup))
  (let* ((modes (get-editing-modes self))
         (cases (generate-cases modes)))
    (setf (items self) cases)
    (setf (slot-value self 'onchange)
          (generate-js-case self
                            (mapcar (lambda (case) (cons (car case) (caddr case))) cases)))))

(define-wm set-mode ((self t) (class string))
  (when class
    (destructuring-bind (package classname)
        (split class #\+)
      (start-editing self (intern classname package))
      nil)))