;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;;
;;; Copyright 2002 Matthew Danish <mrd@debian.org>
;;; Redistributable under MIT/X style license, see IMHO 'COPYING' file
;;;
;;; Some code is from mod-lisp example code.  Marc Battyani has made these
;;; available to the public domain.
;;;
;;; -------------------------------------------------------------------
;;;
;;; Connector architecture currently works as follows:
;;;
;;; IMHO creates a socket fd stream listening on the servlet port
;;; and when a connection from the Apache-module comes in, it calls
;;; a given function with the stream argument.  In this case it is
;;; MOD-LISP-SERVER which gets called.  The job of that function is
;;; to read in the header and data from the Apache-module and setup
;;; a 'request' data structure for PARSE-CONTENT and HANDLE-REQUEST
;;; to deal with.  It must also provide a callback function, in this
;;; case MOD-LISP-RESPONSE, which gets called by IMHO when it is ready
;;; to send back data.
;;;
;;; TODO:
;;; - Clean-up request data structure
;;; - Possibly implement connector class and generics?
;;; - Clean up request handling and put various bits in better places;
;;;   such as PARSE-URI and other request-parsing matters...
;;; - optimize


;;; -------------------------------------------------------------------
;;; mod_lisp connector for IMHO
;;; -------------------------------------------------------------------

(in-package #:imho)

(defvar *keep-mod-lisp-socket* nil)

;; from mod-lisp example
;; write a header back to mod_lisp
(defun write-header-line (stream key value)
  (write-string key stream)
  (write-char #\NewLine stream)
  (write-string value stream)
  (write-char #\NewLine stream))

;; for the express purpose of reading in a header-line from mod_lisp
;; over a non-character stream
;; not guaranteed to be a good way of reading in lines from
;; non-character streams in general
(defun read-byte-line (stream &optional (eof-error-p t) (eof-value nil))
  (let ((line (make-array 0
			  :element-type 'character
			  :fill-pointer t)))
    (loop for b = (read-byte stream eof-error-p eof-value)
	  until (or (null b)
                    (eql (char-code #\newline) b)
		    (eql (char-code #\return) b))

	  do (vector-push-extend (code-char b) line)
	  finally (progn (when (and b (eql (char-code #\return) b))
			   (read-byte stream eof-error-p eof-value))
			 (return line)))))


;; from mod-lisp example
;; parses the incoming mod_lisp request into header and content values
(defun read-mod-lisp-request (stream)
  (let* ((header (loop for key = (read-byte-line stream t nil)
		       while (and key (string-not-equal key "end"))
		       for value = (read-byte-line stream t nil)
		       collect (cons (intern (string-upcase key)
					     :keyword)
				     (if (string-equal key
						       "content-type")
					 (parse-content-type-header value)
					 value))))
	 (content-length (cdr (assoc :content-length header
				     :test #'string-equal)))
	 (content (when content-length
		    (make-array (parse-integer content-length
					       :junk-allowed t)))))
    (when content
      (read-sequence content stream)
      (push (cons :posted-content content) header))
    header))

  
;; gets a value given a key, from the mod_lisp header
(defun get-mod-lisp-key-value (header key)
  (cdr (assoc key header :test #'string-equal)))

;; dispatches a mod_lisp request to IMHO, after setting up the
;; necessary data structures
(defun exec-mod-lisp-request (header active-request stream)
  (setf (request-stream active-request) stream)
  (parse-uri active-request (get-mod-lisp-key-value header "url"))
  (setf (request-http-method active-request)
	(get-mod-lisp-key-value header "method"))
  (setf (request-protocol active-request)
	(get-mod-lisp-key-value header "server-protocol"))
  (setf (request-headers-in active-request)
	header)
  (setf (request-client-content active-request)
	(get-mod-lisp-key-value header "posted-content"))

  (parse-content active-request)
  (handle-request active-request))
  
;; a callback function which writes the output of IMHO to mod_lisp
;; with the necessary headers
(defun mod-lisp-response (request)
  (let ((stream (request-stream request)))
    (write-header-line stream "Status" "200 OK")
    (write-header-line stream "Content-Type" (request-response-type request))
    
    (let ((body (request-response-body request)))
      (when (and (consp body) (equal (car body) :data))
        (setq body (cdr body)))
      
      (if (request-response-length request)
          (write-header-line stream "Content-Length"
                             (princ-to-string (request-response-length request)))
          (write-header-line stream "Content-Length"
                             (princ-to-string (length body))))

      (when *keep-mod-lisp-socket*
        (write-header-line stream "Keep-Socket" "1"))
      
      (write-string "end" stream)
      (write-char #\NewLine stream)
      (write-sequence body stream))))


;; the "main" function which reads and executes a request    
(defun mod-lisp-server (stream)
  (let ((active-request (make-request :response-callback #'mod-lisp-response)))
    (handler-case
        (loop
         (progn (exec-mod-lisp-request (read-mod-lisp-request stream)
                                       active-request
                                       stream)
                (force-output stream)))
      (error (e)
        (warn "Error in mod-lisp-server ~A" e)))))
         
