(in-package :burp)

(defclass burp-login (html-form)
  ((destination
    :initarg :destination
    :documentation
    "Where to proceed when the login completes."))
  )

(defmethod shared-initialize ((form burp-login) slots &rest)
  (call-next-method)
  (setf (slot-value form 'method) "burp-process-login")
  (instantiate-children
   form
   `(("userid"   text-field)
     ("password" password-field)
     ("newuser" hyperlink
      :reference ,(refer-wm burp-newuser)
      :value "But I don't have a user id!")
     ("submit" submit-button))))

(define-wm burp-process-login ((element t))
  (with-slots (destination)
    element
    (let ((user (element-value (child-element element "userid")))
          (pass (element-value (child-element element "password"))))
      (format t ";; Logging in ~A/~A => ~s~%" user pass destination)
      (funcall #'imho::restore-session-instance destination))))

(define-wm burp-newuser ((element t))
  (session-instance 'burp-newuser :destination (slot-value element 'destination)))

(defclass burp-newuser (html-form)
  ((destination
    :initarg :destination
    :documentation
    "Where to proceed when the form completes.")
   (message
    :initarg :message
    :initform ""
    :documentation
    "In case something goes wrong with form processing")))

(defmethod shared-initialize ((form burp-newuser) slots &rest)
  (call-next-method)
  (setf (slot-value form 'method) "burp-process-newuser")
  (instantiate-children
   form
   `(("userid"   text-field)
     ("name" text-field)
     ("password" password-field)
     ("password-confirmation" password-field)
     ("email" text-field)
     ("submit" submit-button)
     ("message"        static-string
      :value ,(lambda () (slot-value form 'message))))))

(defun verify-and-get-new-user (element)
  (flet ((userid-exists (user-id)
           (car (select 'burp-user :where [= [user_id] user-id]))))
    (let ((user (element-value (child-element element "userid")))
          (name (element-value (child-element element "name")))
          (pass (element-value (child-element element "password")))
          (confirm-pass (element-value
                         (child-element element "password-confirmation")))
          (email (element-value (child-element element "email"))))
      (if (or (string= user "") (userid-exists user))
          (values nil nil "User ID already exists or is invalid; please try again")
          (if (string= pass confirm-pass)
              (if (< (length pass) 5)
                  (values nil nil "Password must be at least 5 characters long")
                  (let ((new-user (make-instance 'burp-user :user user :name name
                                                 :password pass :email email)))
                    (values t new-user nil)))
              (values nil nil "Passwords don't match; please try again"))))))
  
(define-wm burp-process-newuser ((element t))
  (multiple-value-bind (success new-user error-message)
      (verify-and-get-new-user element)
    (if success
        (with-slots (destination)
          element
          (update-records-from-instance new-user)
          (setf (slot-value *active-session* 'user) new-user)
          (funcall #'imho::restore-session-instance destination))
        (progn
          (setf (slot-value element 'message) error-message)
          nil))))
           
(def-view-class burp-user ()
  ((user-id
    :initarg :user
    :db-kind :key
    :type (string 30))
   (name
    :reader user-name
    :initarg :name
    :db-kind :base
    :type (string 50))
   (password
    :initarg :password
    :db-kind :base
    :type (string 30))
   (email
    :initarg :email
    :db-kind :base
    :type (string 200)))
  )
