;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: elements.lisp,v 1.86 2002/02/19 16:19:29 jesse Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :imho)

;; ------------------------------------------------------------
;; Hashtable of children

(defmethod element-children ((element html-element))
  (slot-value element 'children))

;; ------------------------------------------------------------
;; Children which should have values taken

(defmethod element-active-children ((element html-element))
  (let (children)
    (maphash (lambda (k v)
	       (declare (ignore k))
	       (setq children (cons v children)))
	     (element-children element))
    children))

(defun html-element-all-children (element)
  "Given ELEMENT, returns a list of all elements composing it."
  (let ((children (list element)))
    (maphash (lambda (k v)
	       (declare (ignore k))
	       (setq children (append (html-element-all-children v) children)))
	     (element-children element))
    children))

;; ------------------------------------------------------------
;; clos-method: initialize-instance
;;
;; initialize the internal and external names for an element

(defmethod shared-initialize :after ((element html-element) slots &rest initargs)
  (declare (ignore slots initargs))
  (when (and (slot-boundp element 'element-external-name)
             (not (slot-boundp element 'element-internal-name)))
    (setf (slot-value element 'element-internal-name)
          (slot-value element 'element-external-name)))
  (when *active-session*
    (set-session-element *active-session* element))
  (create element))

(defvar *check-elements* t)
(defvar *check-elements-table* (make-hash-table))

;; Called to establish display bindings between conjugate element
;; value, and display values of constituents

(defmethod preawake ((element html-element))
  (maphash (lambda (k v)
             (declare (ignore k))
             (preawake v))
           (element-children element)))

;; called on the element being returned to the client

(defmethod awake ((element html-element))
  (maphash (lambda (k v)
             (declare (ignore k))
             (awake v))
           (element-children element)))

;; ------------------------------------------------------------
;; Create a child html-element and insert into parent.

(defun make-child (parent child-class internal-name &optional &rest initargs)
  "create a child html-element and insert it into its parent"
  (let ((instance-args (append (list :parent parent) initargs)))
    (let ((child (apply #'make-instance child-class instance-args)))
      (set-child-element parent internal-name child))))


;; Called at element creation time, using the contents of the
;; defelements form for the element being created

(defun create-child (element elt)
  "helper for creation of children defined by defelements"
  (apply #'make-child element (cadr elt) (ensure-keyword (car elt)) (cddr elt)))

(defmethod create ((element html-element))
  (mapc (lambda (elt)
          (create-child element elt))
        (element-parts element)))

;; means of defining the subelements of an html element

(defmethod element-parts ((element html-element))
  nil)

(defmacro defelements (element part-list)
  (let* ((initializers (cadr part-list)))
    `(progn
      (defmethod element-parts ((element ,element))
        (append (call-next-method) ',initializers)))))

(defmacro defbindings (element part-list)
  "Creates child-elements of an IMHO element and assigns values
to those elements as defined in part-list.

The part-list is a list of lists, each list specifies a child-element,
its element-type and a way for that child element to receive its value.
Values are assigned by :binding, :sbinding or :initargs.

:binding - arg: a lambda function to populate the element-value of the
child-element.  Takes the element-value of the parent element as its
single argument.

:sbinding - arg: a lambda function to populate the element-value of the
child-element.  Takes the parent-element as its single argument.

:initargs - arg: a list of initarg keywords and values to initialize
the child-element.  The child-element's value can be set later using
the awake or preawake method for the parent element.

Format:
 (defbindings element
   ((child-element
     :type element-type
     :binding (lambda ([element-value])
	        [some code])
    :sbinding (lambda ([element])
		 [some code])
    :initargs (:initarg value :initarg value etc...))
    (child-element
     :type ...
     :binding ...)))"
 
  (flet ((get-binding (subelement)
	   (let ((pos (position :binding subelement)))
             `(setf (element-value (child-element element ',(car subelement)))
                    (funcall ,(nth (1+ pos) subelement)
                      (element-value element)))))
         (get-sbinding (subelement)
           (let ((pos (position :sbinding subelement)))
             `(setf (element-value (child-element element ',(car subelement)))
                    (funcall ,(nth (1+ pos) subelement)
                      element))))
         (get-initializer (subelement)
           (let ((pos (position :initargs subelement))
                 (type (position :type subelement)))
             (if pos
                 (cons (car subelement) (cons (nth (1+ type) subelement)
                                              (nth (1+ pos) subelement)))
                 (list (car subelement) (nth (1+ type) subelement))))))
    (let* ((bound (filter-list2 part-list
                                :test (lambda (x)
                                        (member :binding (cdr x)))))
           (binders (mapcar #'get-binding bound))
           (sbound (filter-list2 part-list
                                :test (lambda (x)
                                        (member :sbinding (cdr x)))))
           (sbinders (mapcar #'get-sbinding sbound))
           (initializers (mapcar #'get-initializer part-list)))
      `(progn
        (defmethod preawake :before ((element ,element))
          ,@sbinders
          ,@binders)
        (defmethod element-parts ((element ,element))
          (append (call-next-method)
                  ',initializers))))))


;; ------------------------------------------------------------
;; framework-method: render-html
;;
;; This is the default renderer for html-elements. If this method has not
;; been specialized to a subclass, we look around for an html template
;; for this html-element.
;;
;; The parsed template contains a list of strings and elements: render
;; them sequentially, just dumping strings onto the output stream, and
;; calling render-html on other children.

(defmethod render-html ((element html-element) stream)
  (let ((template (html-template *active-application* (type-of element))))
    (dolist (item (html-template-content template))
      (ecase (car item)
        (:string
         (render-html (cadr item) stream))
        (:method
         (apply (second item) `(,element ,stream ,@(if (third item) (list (third item))))))
        (:child
         (render-html (child-element element (cadr item)) stream))))))

(defvar *scriptable-root* (find-class 'html-element))

;; FIXME: this should be recursive *and* memoized

(defun scriptable-types-1 (type)
  "given class TYPE, return the list of superclasses which derive from
html-element"
  (let* ((class (typecase type (symbol (find-class type)) (t type)))
         (supers (class-direct-superclasses class)))
    (cons class
          (and (not (member *scriptable-root* supers))
               (mapcar #'scriptable-types-1 supers)))))

(defun scriptable-types (type)
  "given class TYPE, return the list of superclasses which derive from
html-element"
  (cons 'html-element (mapcar #'class-name (flatten (scriptable-types-1 type)))))

(defmethod render-html :before ((element html-element) stream)
  (declare (ignore stream))
  (let ((type (type-of element)))
    (when (not (gethash type *active-components*))
      (mapc (lambda (type)
              (setf (gethash type *active-components*) t))
            (scriptable-types type))
      )))
  
(defmethod render-html :around ((element html-element) stream)
  (with-slots (element-external-name element-internal-name)
    element
    (let ((sname (symbol-name (type-of element))))
      (if *add-element-comments*
          (html-stream
           stream
           (:comment
            (format stream "start html-element: ~a, ~a, ~a"
                    sname element-internal-name element-external-name))))
      (call-next-method)
      (if *add-element-comments*
          (html-stream
           stream
           (:comment
            (format stream "end html-element: ~a, ~a, ~a"
                    sname element-internal-name element-external-name)))))))


;; Just for orthogonality's sake

(defmethod render-html ((obj t) stream)
  (if (functionp obj)
      (princ (funcall obj) stream)
      (princ obj stream)))

(defmethod render-html ((string string) stream)
  (write-string string stream))

;; ------------------------------------------------------------
;; accessors for 'html-element value'

;; FIX: differentiate between display value and internal value? - JLB

(defmethod element-value ((element html-element))
  (slot-value element 'value))

(defmethod set-element-value (element value)
  (cmsg "IMHO WARNING: cannot assign value ~s to element ~s."
        value element))

(defmethod set-element-value ((element html-element) value)
  (setf (slot-value element 'value) value))

(defsetf element-value set-element-value)

;; ------------------------------------------------------------
;; request - response loop

(defmethod take-values-form-request (element request)
  (declare (ignore element request)))

(defmethod take-values-from-request ((element html-element) request)
  (mapc (lambda (child)
	  (take-values-from-request child request))
	(element-active-children element)))

;; ------------------------------------------------------------------
;; targets and method invocation

(defmethod element-caller ((element html-element))
  element)

(defmethod element-target ((element html-element))
  (if (not (element-parent element))
      nil
      ;; return the closest parent target
      (element-target (element-parent element))))

(defmethod target-name ((element html-element))
  (element-external-name (element-target element)))

(defmethod element-args ((element html-element))
  nil)

(defmethod element-frame-args ((element t))
  nil)

;; ------------------------------------------------------------
;; Build an URL for a html-element.

(defun element-url (element &key target)
  "return the url which refers to ELEMENT"
  (let* ((caller (element-caller element))
         (target (or target (element-external-name (or (element-target element) element))))
         (args (element-args element))
         (path (concatenate 'string
                           *active-url*
                           ;; caller
                           (if caller
                               (element-external-name caller)
                               "no-caller")
                           "/"
                           ;; target
                           target
                           "/"
                           ;; method
                           (find-element-method element))))
    (when args
      ;; FIXME: do inverse url argument encoding
      (setq path (concatenate 'string path "?"
                              (extern-ref (if (listp args) (car args) args)))))
    path))

(defun find-element-method (element)
  "return the method associated with an ELEMENT"
  (cond ((element-method element)
         (string (maybecar (element-method element))))
        ((element-parent element)
         (string (find-element-method (element-parent element))))
        (t
         "")))

(defun find-element-target (element)
  "return the frame target associated with ELEMENT"
  (cond ((listp (element-method element))
         (cadr (element-method element)))
        (t
         "_self")))

(defun element-target-frame (element)
  "return the frame target associated with ELEMENT"
  (cond ((listp (element-method element))
         (cadr (element-method element)))
        (t
         "_self")))

;; needed by with-action
;; [2000/07/09 :lh]
(defmethod build-element-url ((element html-element)
                              &key (caller nil) (reference nil) (arg nil))
  (let ((path (concatenate 'string *active-url*
                           (if caller (slot-value caller 'element-external-name)
                               "no-caller") "/"
                               (slot-value element 'element-external-name) "/")))
    (if reference
	(progn
	  (setq path (concatenate 'string path reference))
	  (if arg
	      ;; FIXME: do inverse url argument encoding
	      (setq path (concatenate 'string path "?" arg)))))
    path))
 

;; is this used?  [2000/07/08 :lh]
(defmethod build-element-url (no-parent &key (reference nil) (arg nil))
  (declare (ignore no-parent))
  (let ((path (concatenate 'string *active-url* "parentless" "/")))
    (if reference
	(progn
	  (setq path (concatenate 'string path reference))
	  (if arg
	      ;; FIXME: do inverse url argument encoding
	      (setq path (concatenate 'string path "?" arg)))))
    path))

;; ------------------------------------------------------------
;; tree traversal

(defmethod find-above ((element html-element) (child-name string)
                       &key search-self)
  "Search up the parent tree, checking each parent for a child
named child-name."
  (let ((parent (element-parent element)))
    (if search-self
        (let ((child (child-element element child-name)))
          (if child
              child
              (find-above parent child-name :search-self t)))
        (when parent
          (find-above parent child-name) :search-self t))))

(defun find-parent (element test)
  "find the first parent that satisfies test"
  (when element
    (let ((parent (element-parent element)))
      (if (funcall test parent)
          parent
          (find-parent parent test)))))
                 
(defun element-root (element)
  "Return the topmost parent of the given element's rendering tree."
  (and element
       (do ((next element (element-parent root))
            (root element))
           ((not next) root)
         (setf root next))))

;; ------------------------------------------------------------
;; get/set/find a child element

(defmethod child-element ((element html-element) element-internal-name
                          &key (if-does-not-exist :marker))
  (let* ((name (ensure-keyword element-internal-name))
         (child (gethash name (element-children element))))
    (or child
        (case if-does-not-exist
          (:marker
           (let ((new-se (assoc name (element-parts element)
                                :test #'eq)))
             (if new-se
                 (create-child element new-se)
                 (format nil "[Missing Child: ~a]" name))))
          (:error
           (error 'imho-error :text (format nil "Element ~s is missing child ~s" element element-internal-name)))
          (t
           nil)))))

(defmethod set-child-element ((element html-element) element-internal-name child)
  (let ((name (ensure-keyword element-internal-name)))
    (setf (element-parent child) element)
    (setf (slot-value child 'element-internal-name) name)
    (setf (gethash name (element-children element)) child))
  child)

(defsetf child-element (element element-internal-name) (child)
  `(set-child-element ,element ,element-internal-name ,child))

(defmacro child-value (ele child)
  `(element-value (child-element ,ele ,child)))


;; ------------------------------------------------------------
;; Bind multiple child values from a html-element instance

(defmacro with-element-values (slots instance &body body)
  (let ((in (gensym)))
    `(let ((,in ,instance))
      (symbol-macrolet
          ,(mapcar #'(lambda (slot-entry)
                       (let ((variable-name 
                              (if (symbolp slot-entry)
                                  slot-entry
                                  (car slot-entry)))
                             (slot-name
                              (if (symbolp slot-entry)
                                  slot-entry
                                  (cadr slot-entry))))
                         `(,variable-name
                           (element-value (child-element ,in ,(ensure-keyword slot-name) :if-does-not-exist :error)))))
                   slots)
        ,@body))))

(defmacro with-elements (slots instance &body body)
  (let ((in (gensym)))
    `(let ((,in ,instance))
      (symbol-macrolet
          ,(mapcar #'(lambda (slot-entry)
                       (let ((variable-name 
                              (if (symbolp slot-entry)
                                  slot-entry
                                  (car slot-entry)))
                             (slot-name
                              (if (symbolp slot-entry)
                                  slot-entry
                                  (cadr slot-entry))))
                         `(,variable-name
                           (child-element ,in ,(ensure-keyword slot-name)))))
                   slots)
        ,@body))))

