/*
 * Copyright 2011 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.google.ipc.invalidation.util;

import com.google.common.base.Receiver;


/**
 * Utilities to enable creation of lazy strings, where the instantiation of the string is delayed
 * so that, e.g., log messages that aren't printed have reduced overhead.
 */
public class LazyString {

  /**
   * Given an {@code element} to be logged lazily, returns null if the object is null. Otherwise,
   * return an object that would convert it to a string using {@code builderFunction}. I.e., this
   * method will call {@code builderFunction} with a new {@link TextBuilder} return the string
   * created with it.
   */
  public static <T> Object toLazyCompactString(final T element,
      final Receiver<TextBuilder> builderFunction) {
    if (element == null) {
      return null;
    }
    return new Object() {
      @Override
      public String toString() {
        TextBuilder builder = new TextBuilder();
        builderFunction.accept(builder);
        return builder.toString();
      }
    };
  }

  private LazyString() {  // To prevent instantiation.
  }
}
