// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/accelerators/nested_accelerator_delegate.h"

#include "ash/accelerators/accelerator_controller.h"
#include "ash/shell.h"
#include "ui/aura/window_event_dispatcher.h"
#include "ui/base/accelerators/accelerator.h"
#include "ui/events/event.h"
#include "ui/events/event_constants.h"
#include "ui/events/event_utils.h"
#include "ui/views/controls/menu/menu_controller.h"

namespace ash {
namespace {

bool IsPossibleAcceleratorNotForMenu(const ui::Accelerator& accelerator) {
  // For shortcuts generated by Ctrl or Alt plus a letter, number or
  // the tab key, we want to exit the context menu first and then
  // repost the event. That allows for the shortcut execution after
  // the context menu has exited.
  if (accelerator.type() == ui::ET_KEY_PRESSED &&
      (accelerator.modifiers() & (ui::EF_CONTROL_DOWN | ui::EF_ALT_DOWN))) {
    const ui::KeyboardCode key_code = accelerator.key_code();
    if ((key_code >= ui::VKEY_A && key_code <= ui::VKEY_Z) ||
        (key_code >= ui::VKEY_0 && key_code <= ui::VKEY_9) ||
        (key_code == ui::VKEY_TAB)) {
      return true;
    }
  }
  return false;
}

bool ShouldProcessAcceleratorNow(const ui::Accelerator& accelerator) {
  if (!IsPossibleAcceleratorNotForMenu(accelerator))
    return true;

  if (views::MenuController* menu_controller =
          views::MenuController::GetActiveInstance()) {
    menu_controller->CancelAll();
    return false;
  }
  return true;
}

}  // namespace

NestedAcceleratorDelegate::NestedAcceleratorDelegate() {
}

NestedAcceleratorDelegate::~NestedAcceleratorDelegate() {
}

NestedAcceleratorDelegate::Result NestedAcceleratorDelegate::ProcessAccelerator(
    const ui::Accelerator& accelerator) {
  if (!ShouldProcessAcceleratorNow(accelerator))
    return RESULT_PROCESS_LATER;

  ash::AcceleratorController* accelerator_controller =
      ash::Shell::GetInstance()->accelerator_controller();
  if (!accelerator_controller)
    return RESULT_NOT_PROCESSED;
  return accelerator_controller->Process(accelerator) ? RESULT_PROCESSED
                                                      : RESULT_NOT_PROCESSED;
}

}  // namespace ash
