;;; @file track.s
;;; Commodore program for reading or writing 1541 disk tracks
;;; @author Marko Mkel (msmakela@nic.funet.fi)

;;; The protocol for reading a 1541 disk track is as follows.
;;; C=			other computer
;;;			<- 1..35 (track number)
;;; <last sector> ->
;;; <sector (0..20)><data (256 bytes) ->
;;; <sector (0..20)><data (256 bytes) ->
;;; ...

;;; The protocol for writing a 1541 disk sector is as follows.
;;; C=			other computer
;;;			<- <track (128+1..128+35)>
;;; <last sector number> ->
;;;			<- <sector ($ff=done)><data (256 bytes)>
;;; <sector> ->

;;; Termination
;;; C=			other computer
;;;			<- 00

#include "ext.s"

second	= $ff93		; Send secondary address after LISTEN
ciout	= $ffa8		; Output byte to serial port
listen	= $ffb1		; Command serial bus to listen
unlsn	= $ffae		; Command serial bus to unlisten

fa	= $ba		; current device number

temp	= $fb		; number of sectors
stemp	= $fc		; temporary variable for dsend
stemp2	= $fd		; temporary variable for dsend
buffer	= $1e00		; buffer for a disk sector

; Drive program commands
CMD_READ	= 0
CMD_WRITE	= 1

#if host = 20		; VIC-20
iecport1	=	$912c
dato		=	32
clko		=	2
iecport2	=	$911f
atno		=	128
clki		=	1
dati		=	2
#define blank
#define unblank
#endif
#if host = 16			; C16/C116/plus4
iecport1	=	1
iecport2	=	iecport1
dato		=	1
clko		=	2
atno		=	4
clki		=	64
dati		=	128
;; switch to 1 MHz mode, blank the screen and wait for vertical blank
#define blank lda #2:sta $ff13:lda $ff06:and #$ef:sta $ff06:lda#1:\
	.(:bad:bit $ff1c:beq bad:.):
;; unblank the screen
#define unblank lda $ff06:ora #$10:sta $ff06
#endif
#if host = 64			; 64/128
iecport1	=	$dd00
iecport2	=	iecport1
atno		=	8
clko		=	16
dato		=	32
clki		=	64
dati		=	128
;; switch to 1 MHz mode, disable sprites, blank the screen and wait for vsync
#define blank lda #0:sta $d015:sta $d030:lda $d011:and #$ef:sta $d011:\
	.(:bad:lda $d011:bpl bad:.):
;; unblank the screen
#define unblank lda $d011:and #$7f:ora #$10:sta $d011
#endif

	jsr prepare		; prepare for using the link
	blank			; switch off the screen, go to 1 MHz
	;; start the program downloaded to the disk drive
	lda fa
	jsr listen
	lda #$6f
	jsr second
	lda #"M"
	jsr ciout
	lda #"-"
	jsr ciout
	lda #"W"
	jsr ciout
	lda #$00
	jsr ciout
	lda #$00
	jsr ciout
	lda #$01
	jsr ciout
	lda #$d0
	jsr ciout
	jsr unlsn		; UNLISTEN leaves DATA=CLK=0
	sei

	jsr drecv		; wait for the "drive ready" signal
	lda iecport1		; prepare to send to the serial bus
	ora #<(dato+clko)
	sta iecport1
loop0	jsr receive
	jmp loop1
loop	jsr receive_switch	; get the command
loop1	bne noexit
	lda #$ff		; 0 => terminate
	jsr dsend		; tell drive to terminate
	unblank
	jmp exit
noexit	bpl rdtrack
	and #$7f

	;; write track A
wrtrack	jsr dsend		; send track number
	.(
wrnext	jsr drecv_switch	; get sector number
	lda iecport1		; prepare to send to the serial bus
	ora #<(dato+clko)
	sta iecport1
	jsr send_switch		; relay it back
	jsr receive_switch	; get number of sector to write
	jsr dsend		; pass it to the drive
	cmp #$ff		; quit?
	beq loop0
	ldx #0			; get 256 bytes of data
wrdata	jsr receive
	jsr dsend
	dex
	bne wrdata
	beq wrnext
	.)

	;; read track A
rdtrack	jsr dsend		; send track number
	.(
	jsr drecv_switch	; get number of last sector
	sta temp
	jsr send_switch
rdnext	jsr drecv		; get sector number
	jsr send
	ldx #0
rddata1	jsr drecv		; get 256 bytes of data
	sta buffer,x
	inx
	bne rddata1
rddata2	lda buffer,x
	jsr send		; relay the data
	inx
	bne rddata2
	dec temp
	bpl rdnext
	bmi loop
	.)

	;; switch to input from serial bus and receive a byte in A
drecv_switch
	lda iecport1
	and #<(255-dato-clko)
	sta iecport1
	;; receive a byte in A
drecv
#if dati = 128
	lda iecport2
	bpl drecv		; wait for DATA=1
#else
	lda #dati
poll	bit iecport2
	beq poll		; wait for DATA=1
#endif
	ldy #4
	jsr delay12
	jsr delay12
	lda #0
drloop	sta stemp
	lda iecport2
	and #clki|dati
#if dati = 128 && clki = 64
	ora stemp
	ror
	ror
	sta stemp
#else
#if dati = 2 && clki = 1
	lsr
	ror
	ror
	ror
	ror
	ora stemp
#else
#err "unsupported system"
#endif
#endif
	dey
	bne drloop
	rts

	;; send a byte in A
dsend	sta stemp
	lda iecport1
	and #255-clko-dato
	sta stemp2
	tay
	;; send b7..b6
	lda stemp
	sty iecport1		; CLK=DATA=0
#if dato = 32 && clko = 16	; C64/C128
	ror
	ror
	and #<(dato+clko)
	ora stemp2
	jsr delay12
#else
	asl
	rol
	rol
	nop
	nop
	and #3
	tay
	lda clkdat,y
	ora stemp2
#endif
	sta iecport1
	;; send b5..b4
	lda stemp
#if dato = 32 && clko = 16	; C64/C128
	and #<(dato+clko)
	ora stemp2
	nop
	asl stemp
	asl stemp
#else
	lsr
	lsr
	lsr
	lsr
	nop
	and #3
	tay
	lda clkdat,y
	ora stemp2
#endif
	sta iecport1
	;; send b3..b2
	lda stemp
#if dato = 32 && clko = 16	; C64/C128
	and #<(dato+clko)
	ora stemp2
	nop
	asl stemp
	asl stemp
#else
	lsr
	lsr
	nop
	nop
	nop
	and #3
	tay
	lda clkdat,y
	ora stemp2
#endif
	sta iecport1
	;; send b1..b0
	lda stemp
#if dato = 32 && clko = 16	; C64/C128
	and #<(dato+clko)
	ora stemp2
	nop
	jsr delay12
#else
	nop
	nop
	nop
	nop
	nop
	and #3
	tay
	lda clkdat,y
	ora stemp2
#endif
	sta iecport1
	ora #<(dato+clko)
	jsr delay12
	jsr delay12
	sta iecport1
delay12	rts

#if dato = 32 && clko = 16	; C64/C128
#else
clkdat	.byte 0, clko, dato, <(dato+clko)
#endif
