/**
 * @file prlink.c
 * IBM PC transfer routines for the 4-bit/8-bit PET/C64/C128/Vic-20
 * to Centronics cable designed by Marko Mkel
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/*
 * Copyright  1994-1996 Marko Mkel and Olaf Seibert
 * Copyright  2001 Marko Mkel
 * Original Linux and Commodore 64/128/Vic-20 version by Marko Mkel
 * Ported to the PET and the Amiga series by Olaf Seibert
 * Restructured by Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef COMM_PC
# include "info.h"
# include "prlink.h"
# include "pcpar.h"

/** current state of the acknowledgement line (BUSY) */
static unsigned ack;

/** Open the communication channel
 * @param dev		name of the communication device
 * @param hostinfo	(output) the computer model information
 * @return		zero on success, nonzero on failure
 */
int
prlink_init (const char* dev, struct hostinfo* hostinfo)
{
  unsigned char detectbuf[5];
  if (!open_port (dev))
    return 1;

  out_data (8); /* set the -FLAG signal to high */
  ack = 0x80 & in_stat (); /* get the current state of the BUSY line */

  if (prlink_write ("", 1))
    return 1;
  if (prlink_read (detectbuf, sizeof detectbuf))
    return 2;
  hostinfo->host = detectbuf[0];
  hostinfo->driver = (unsigned) detectbuf[1] | (unsigned) detectbuf[2] << 8;
  hostinfo->basic = (unsigned) detectbuf[3] | (unsigned) detectbuf[4] << 8;
  return 0;
}
/** Close the communication channel */
void
prlink_close (void)
{
  close_port ();
}

#if !defined __BCC__ && (!defined __GNUC__ || defined USE_PPDEV)
/** Send data (unoptimised)
 * @param buf		the data to be sent
 * @param len		length of the data in bytes
 */
static void
prlink_write_slow (const void* buf, unsigned len)
{
  register const unsigned char* buffer = buf;

  while (len--) {
    /* invert bits 0, 1 and 3 */
    register unsigned char byte = *buffer++ ^ 0x0b;
    out_ctrl (byte);
    out_data (byte & ~8); /* drop the -FLAG to low */
    while (ack == (0x80 & in_stat ())); /* wait for BUSY to change */
    out_data (8); /* raise -FLAG again */
    ack ^= 0x80; /* store the new state of -BUSY */
  }
}

/** Receive data (unoptimised)
 * @param buf		the data to be received
 * @param len		length of the data in bytes
 */
static void
prlink_read_slow (void* buf, unsigned len)
{
  register unsigned char data;
  register unsigned char* buffer = buf;

  while (len--) {
    out_ctrl (4); /* set the bidirectional signals to input */
    out_data (0); /* drop -FLAG */
    while (ack == (0x80 & in_stat ())); /* wait for -BUSY to change */
    out_data (8); /* raise -FLAG */

    data = in_ctrl () & 0x0f; /* read the low nybble */

    out_data (0); /* drop -FLAG */
    while (ack != (0x80 & in_stat ())); /* wait for -BUSY to change */

    data |= in_ctrl () << 4; /* read the high nybble */

    out_data (8); /* raise -FLAG */

    *buffer++ = data ^ 0xbb;
  }
}
#endif /* !__BCC__ && (!__GNUC__ || USE_PPDEV) */

/** Send data
 * @param buf		the data to be sent
 * @param len		length of the data in bytes
 * @return		zero on success, nonzero on failure
 */
int
prlink_write (const void* buf, unsigned len)
{
# ifdef __GNUC__
#  ifdef USE_PPDEV
  if (baseaddr == (unsigned) -1) {
    prlink_write_slow (buf, len);
    return 0;
  }
#  endif /* USE_PPDEV */
  __asm__ volatile ("movb %%al,%%ah\n\t"
		    "0:\t"
		    "movb (%%ebx),%%al\n\t"
		    "xorb $0xb,%%al\n\t"
		    "addw $2,%%dx\n\t"
		    "outb %%al,%%dx\n\t"
		    "subw $2,%%dx\n\t"
		    "andb $0xf7,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "incw %%dx\n"
		    "1:\t"
		    "inb %%dx,%%al\n\t"
		    "andb $0x80,%%al\n\t"
		    "cmpb %%al,%%ah\n\t"
		    "je 1b\n\t"
		    "decw %%dx\n\t"
		    "xorb $0x80,%%ah\n\t"
		    "movb $8,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "inc %%ebx\n\t"
		    "loop 0b\n\t"
		    "xorl %%ebx,%%ebx\n\t"
		    "movb %%ah,%%bl\n\t"
		    : "=b" (ack)
		    : "ax" (ack), "b" (buf), "c" (len), "d" (baseaddr));
  return 0;
# else /* BCC does not recognize #elif */
# ifdef __BCC__
#  asm
    push bp
    mov bp,sp
    push es
    mov dx,_baseaddr
    mov cx,len_
    mov bx,buf_
    mov ax,I 0x40
    mov es,ax
    mov ah,_ack

.sloop:
    mov al,[bx]
    xor al,I 0xb
    add dx,I 2
    out dx,al
    sub dx,I 2
    and al,I 0xf7
    out dx,al
    inc dx

.owait:
    in al,dx
    and al,I 0x80
    cmp al,ah
    jne .nowait
    eseg
    mov al,byte [0x71]	; is the keyboard interrupt flag set?
    test al,I 0x80
    jz .owait
    jmp near _abreak

.nowait:
    dec dx
    mov ah,al
    mov al,I 8
    out dx,al

    inc bx
    loop .sloop
    mov _ack,ah
    pop es
    pop bp
    xor ax,ax
#  endasm
# else
  prlink_write_slow (buf, len);
  return 0;
# endif
# endif /* BCC does not recognize #elif */
}

/** Receive data
 * @param buf		the data to be received
 * @param len		length of the data in bytes
 * @return		zero on success, nonzero on failure
 */
int
prlink_read (void* buf, unsigned len)
{
# ifdef __GNUC__
#  ifdef USE_PPDEV
  if (baseaddr == (unsigned) -1) {
    prlink_read_slow (buf, len);
    return 0;
  }
#  endif /* USE_PPDEV */
  __asm__ volatile ("movb %%al,%%ah\n"
		    "0:\t"
		    "xorb %%al,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "incw %%dx\n"
		    "1:\t"
		    "inb %%dx,%%al\n\t"
		    "andb $0x80,%%al\n\t"
		    "cmpb %%ah,%%al\n\t"
		    "je 1b\n\t"
		    "decw %%dx\n\t"
		    "movb $8,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "addw $2,%%dx\n\t"
		    "movb $4,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "inb %%dx,%%al\n\t"
		    "andb $0xf,%%al\n\t"
		    "movb %%al,(%%ebx)\n\t"
		    "subw $2,%%dx\n\t"
		    "xorb %%al,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "incw %%dx\n"
		    "2:\t"
		    "inb %%dx,%%al\n\t"
		    "andb $0x80,%%al\n\t"
		    "cmpb %%ah,%%al\n\t"
		    "jne 2b\n\t"
		    "decw %%dx\n\t"
		    "movb $8,%%al\n\t"
		    "outb %%al,%%dx\n\t"
		    "addw $2,%%dx\n\t"
		    "inb %%dx,%%al\n\t"
		    "subw $2,%%dx\n\t"
		    "salb $4,%%al\n\t"
		    "orb (%%ebx),%%al\n\t"
		    "xorb $0xbb,%%al\n\t"
		    "movb %%al,(%%ebx)\n\t"
		    "inc %%ebx\n\t"
		    "loop 0b"
		    : /* no outputs */
		    : "ax" (ack), "b" (buf), "c" (len), "d" (baseaddr));
  return 0;
# else /* BCC does not recognize #elif */
# ifdef __BCC__
#  asm
    push bp
    mov bp,sp
    push es
    mov dx,_baseaddr
    mov cx,len_
    mov bx,buf_
    mov ax,I 0x40
    mov es,ax
    mov ah,_ack

.rloop:
    xor al,al
    out dx,al
    inc dx

.iwait1:
    in al,dx
    and al,I 0x80
    cmp al,ah
    jne .niwait1
    eseg
    mov al,byte [0x71]	; is the keyboard interrupt flag set?
    test al,I 0x80
    jz .iwait1
    jmp near _abreak

.niwait1:
    dec dx
    mov al,I 8
    out dx,al
    add dx,I 2
    mov al,I 4
    out dx,al
    in al,dx
    and al,I 0xf
    mov [bx],al

    sub dx,I 2
    xor al,al
    out dx,al
    inc dx

.iwait2:
    in al,dx
    and al,I 0x80
    cmp al,ah
    je .niwait2
    eseg
    mov al,byte [0x71]	; is the keyboard interrupt flag set?
    test al,I 0x80
    jz .owait
    jmp near _abreak

.niwait2:
    dec dx
    mov al,I 8
    out dx,al

    add dx,I 2
    in al,dx
    sub dx,I 2
        
    sal al,I 1
    sal al,I 1
    sal al,I 1
    sal al,I 1
    or al,[bx]
    xor al,I 0xbb
    mov [bx],al

    inc bx
    loop .rloop
    pop es
    pop bp
    xor ax,ax
#  endasm
# else
  prlink_read_slow (buf, len);
  return 0;
# endif
# endif /* BCC does not recognize #elif */
}
#endif /* COMM_PC */
