/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * The Original Code is Copyright (C) 2009 Blender Foundation.
 * All rights reserved.
 */

/** \file
 * \ingroup spfile
 */

#include "BLI_blenlib.h"
#include "BLI_utildefines.h"

#include "BKE_context.h"
#include "BKE_screen.h"

#include "BLT_translation.h"

#include "DNA_screen_types.h"
#include "DNA_space_types.h"
#include "DNA_userdef_types.h"

#include "MEM_guardedalloc.h"

#include "RNA_access.h"
#include "RNA_define.h"

#include "ED_fileselect.h"

#include "UI_interface.h"
#include "UI_resources.h"

#include "WM_api.h"
#include "WM_types.h"

#include "file_intern.h"
#include "fsmenu.h"

#include <string.h>

static bool file_panel_operator_poll(const bContext *C, PanelType *UNUSED(pt))
{
  SpaceFile *sfile = CTX_wm_space_file(C);
  return (sfile && sfile->op);
}

static void file_panel_operator_header(const bContext *C, Panel *panel)
{
  SpaceFile *sfile = CTX_wm_space_file(C);
  wmOperator *op = sfile->op;

  BLI_strncpy(panel->drawname, WM_operatortype_name(op->type, op->ptr), sizeof(panel->drawname));
}

static void file_panel_operator(const bContext *C, Panel *panel)
{
  SpaceFile *sfile = CTX_wm_space_file(C);
  wmOperator *op = sfile->op;

  UI_block_func_set(uiLayoutGetBlock(panel->layout), file_draw_check_cb, NULL, NULL);

  /* Hack: temporary hide.*/
  const char *hide[] = {"filepath", "files", "directory", "filename"};
  for (int i = 0; i < ARRAY_SIZE(hide); i++) {
    PropertyRNA *prop = RNA_struct_find_property(op->ptr, hide[i]);
    if (prop) {
      RNA_def_property_flag(prop, PROP_HIDDEN);
    }
  }

  uiTemplateOperatorPropertyButs(
      C, panel->layout, op, UI_BUT_LABEL_ALIGN_NONE, UI_TEMPLATE_OP_PROPS_SHOW_EMPTY);

  /* Hack: temporary hide.*/
  for (int i = 0; i < ARRAY_SIZE(hide); i++) {
    PropertyRNA *prop = RNA_struct_find_property(op->ptr, hide[i]);
    if (prop) {
      RNA_def_property_clear_flag(prop, PROP_HIDDEN);
    }
  }

  UI_block_func_set(uiLayoutGetBlock(panel->layout), NULL, NULL, NULL);
}

void file_tool_props_region_panels_register(ARegionType *art)
{
  PanelType *pt;

  pt = MEM_callocN(sizeof(PanelType), "spacetype file operator properties");
  strcpy(pt->idname, "FILE_PT_operator");
  strcpy(pt->label, N_("Operator"));
  strcpy(pt->translation_context, BLT_I18NCONTEXT_DEFAULT_BPYRNA);
  pt->flag = PNL_NO_HEADER;
  pt->poll = file_panel_operator_poll;
  pt->draw_header = file_panel_operator_header;
  pt->draw = file_panel_operator;
  BLI_addtail(&art->paneltypes, pt);
}

static void file_panel_execution_cancel_button(uiLayout *layout)
{
  uiLayout *row = uiLayoutRow(layout, false);
  uiLayoutSetScaleX(row, 0.8f);
  uiLayoutSetFixedSize(row, true);
  uiItemO(row, IFACE_("Cancel"), ICON_NONE, "FILE_OT_cancel");
}

static void file_panel_execution_execute_button(uiLayout *layout, const char *title)
{
  uiLayout *row = uiLayoutRow(layout, false);
  uiLayoutSetScaleX(row, 0.8f);
  uiLayoutSetFixedSize(row, true);
  /* Just a display hint. */
  uiLayoutSetActiveDefault(row, true);
  uiItemO(row, title, ICON_NONE, "FILE_OT_execute");
}

static void file_panel_execution_buttons_draw(const bContext *C, Panel *panel)
{
  bScreen *screen = CTX_wm_screen(C);
  SpaceFile *sfile = CTX_wm_space_file(C);
  FileSelectParams *params = ED_fileselect_get_params(sfile);
  uiBlock *block = uiLayoutGetBlock(panel->layout);
  uiBut *but;
  uiLayout *row;
  PointerRNA params_rna_ptr, *but_extra_rna_ptr;

  const bool overwrite_alert = file_draw_check_exists(sfile);
  const bool windows_layout =
#ifdef _WIN32
      true;
#else
      false;
#endif

  RNA_pointer_create(&screen->id, &RNA_FileSelectParams, params, &params_rna_ptr);

  row = uiLayoutRow(panel->layout, false);
  uiLayoutSetScaleY(row, 1.3f);

  /* callbacks for operator check functions */
  UI_block_func_set(block, file_draw_check_cb, NULL, NULL);

  but = uiDefButR(block,
                  UI_BTYPE_TEXT,
                  -1,
                  "",
                  0,
                  0,
                  UI_UNIT_X * 5,
                  UI_UNIT_Y,
                  &params_rna_ptr,
                  "filename",
                  0,
                  0.0f,
                  (float)FILE_MAXFILE,
                  0,
                  0,
                  TIP_(overwrite_alert ? N_("File name, overwrite existing") : N_("File name")));

  BLI_assert(!UI_but_flag_is_set(but, UI_BUT_UNDO));
  BLI_assert(!UI_but_is_utf8(but));

  UI_but_func_complete_set(but, autocomplete_file, NULL);
  /* silly workaround calling NFunc to ensure this does not get called
   * immediate ui_apply_but_func but only after button deactivates */
  UI_but_funcN_set(but, file_filename_enter_handle, NULL, but);

  if (params->flag & FILE_CHECK_EXISTING) {
    but_extra_rna_ptr = UI_but_extra_operator_icon_add(
        but, "FILE_OT_filenum", WM_OP_EXEC_REGION_WIN, ICON_ADD);
    RNA_int_set(but_extra_rna_ptr, "increment", 1);
    but_extra_rna_ptr = UI_but_extra_operator_icon_add(
        but, "FILE_OT_filenum", WM_OP_EXEC_REGION_WIN, ICON_REMOVE);
    RNA_int_set(but_extra_rna_ptr, "increment", -1);
  }

  /* check if this overrides a file and if the operator option is used */
  if (overwrite_alert) {
    UI_but_flag_enable(but, UI_BUT_REDALERT);
  }
  UI_block_func_set(block, NULL, NULL, NULL);

  {
    uiLayout *sub = uiLayoutRow(row, false);
    uiLayoutSetOperatorContext(sub, WM_OP_EXEC_REGION_WIN);

    if (windows_layout) {
      file_panel_execution_execute_button(sub, params->title);
      file_panel_execution_cancel_button(sub);
    }
    else {
      file_panel_execution_cancel_button(sub);
      file_panel_execution_execute_button(sub, params->title);
    }
  }
}

void file_execute_region_panels_register(ARegionType *art)
{
  PanelType *pt;

  pt = MEM_callocN(sizeof(PanelType), "spacetype file execution buttons");
  strcpy(pt->idname, "FILE_PT_execution_buttons");
  strcpy(pt->label, N_("Execute Buttons"));
  strcpy(pt->translation_context, BLT_I18NCONTEXT_DEFAULT_BPYRNA);
  pt->flag = PNL_NO_HEADER;
  pt->poll = file_panel_operator_poll;
  pt->draw = file_panel_execution_buttons_draw;
  BLI_addtail(&art->paneltypes, pt);
}
