/* Circumference.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * computes the circumference of the graph.
 */
public class Circumference extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        int[][] adjlist = bundle.adjacencyList();
        int n = adjlist.length;
        if(n <= 2)
            //can't contain a cycle: circumference = undefined
            return IntegerValue.undefinedValue(this);
        boolean[] currentPath = new boolean[n];
        int circumference=0;
        for(int i = 0; i < n; i++)
            circumference = circumference(adjlist, i, i, 0, currentPath, circumference);
        if(circumference==2)
            //acyclic graph: circumference = undefined
            return IntegerValue.undefinedValue(this);
        return new IntegerValue(circumference, this);
    }
    
    private int circumference(int[][] adjlist, int startVertex, int currentVertex, int currentLength, boolean[] currentPath, int maximum){
        if(maximum == adjlist.length || maximum >= adjlist.length - startVertex)
            //added check to see if maximum >= adjlist.length - startVertex. Any cycle found from this point on can only contain
            //at most adjlist.length - startVertex vertices so no need to keep on looking.
            return maximum;
        currentPath[currentVertex] = true;
        for(int i=0; i < adjlist[currentVertex].length; i++)
            if(adjlist[currentVertex][i] == startVertex) {
                if (maximum < currentLength + 1)
                    maximum = currentLength + 1;
            } else if(adjlist[currentVertex][i] > startVertex && !currentPath[adjlist[currentVertex][i]])
                //added check to see if adjlist[currentVertex][i] > startVertex. This should avoid redundancy. Not yet tested!!!
                maximum = circumference(adjlist, startVertex, adjlist[currentVertex][i], currentLength+1, currentPath, maximum);
        currentPath[currentVertex] = false;
        return maximum;
    }
       
    public String getInvariantId() {
        return "org.grinvin.invariants.Circumference";
    }
}
