/*
 *  PGOption.m
 *  graphviz
 *
 *  Created by Glen Low on Fri Dec 12 2003.
 *  Copyright (c) 2003, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


#import "PGOption.h"
#import "PGAttribute.h"
#import "PGSchema.h"

static NSNumber* falseValue_;
static NSNumber* trueValue_;

@implementation PGOption

+ (void) initialize
	{
		falseValue_ = [[NSNumber alloc] initWithBool: NO];
		trueValue_ = [[NSNumber alloc] initWithBool: YES];
	}
	
- (id) initWithSchema: (id <PGSchema>) schema prefix: (NSString*) prefix attribute: (PGAttribute*) attribute
	{
		if ((self = [super init]))
			{
				prefix_ = [prefix copy];
				attribute_ = [attribute retain];
				cell_ = [[schema cellForType: [attribute_ type]] retain];

				// read in any default value from the user defaults system
				NSString* defaultValue =
					[[NSUserDefaults standardUserDefaults] stringForKey: [NSString stringWithFormat: @"%@%@", prefix, [attribute name]]];
				if (defaultValue)
					[self setStringValue: defaultValue];
			}
		return self;
	}

- (NSSet*) usedBy
	{
		return [attribute_ usedBy];
	}
	
- (NSString*) name
	{
		return [attribute_ name];
	}

- (NSCell*) objectValueCell
	{
		return cell_;
	}

- (NSComparisonResult) compare: (PGOption*) other
	{
		return [attribute_ compare: other->attribute_];
	}
	
- (NSString*) string
	{
		if (!value_ || [value_ isEqual: @""])
			// if empty, output nothing
			// NOTE: this should be the most common case
			return nil;
		else if ([value_ respondsToSelector: @selector (isEqualToValue:)])
			{
				// if it's a simple flag, either output nothing or output the prefix
				if ([value_ performSelector: @selector (isEqualToValue:) withObject: falseValue_])
					return nil;
				else if ([value_ performSelector: @selector (isEqualToValue:) withObject: trueValue_])
					return [NSString stringWithFormat: @"-%@", prefix_];
			}
		
		// get the name or description of the value
		NSString* stringValue = [self stringValue];
		
		if ([stringValue isEqualToString: @""])
			return nil;
		else
			{
				// if the attribute has a name, output the name=value pair, else just output the value
				NSString* attributeName = [attribute_ name];
				return attributeName && ![attributeName isEqualToString: @""] ?
					[NSString stringWithFormat: @"-%@%@=%@", prefix_, attributeName, stringValue] :
					[NSString stringWithFormat: @"-%@%@", prefix_, stringValue];
			}
	}

- (NSString*) tip
	{
		return [attribute_ tip];
	}

- (NSString*) description
	{
		return [attribute_ description];
	}
	
- (id) objectValue
	{
		return value_;
	}

- (void) setObjectValue: (id) value
	{
		if (value_ != value)
			{
				[value_ release];
				value_ = [value copy];
				
				[[NSNotificationCenter defaultCenter] postNotificationName: OptionDidChange object: self];
			}
	}
	
- (NSString*) stringValue
	{
		id objectValue = [self objectValue];
		if (!objectValue)
			return @"";
		else if ([objectValue respondsToSelector: @selector (name)])
			return [objectValue name];
		else
			{
				NSFormatter* formatter = [cell_ formatter];
				if (formatter)
					return [formatter stringForObjectValue: objectValue];
				else
					return [objectValue description];
			}
	}
	
- (void) setStringValue: (NSString*) name
	{
		id value = [PGDescriptor descriptorForScope: [attribute_ type] name: name];
		if (!value)
			{
				NSFormatter* formatter = [cell_ formatter];
				if (formatter)
					[formatter getObjectValue: &value forString: name errorDescription: nil];
				else
					value = name;
			}
			
		[self setObjectValue: value];
	}
	
- (void) bindToPopUpButton: (NSPopUpButton*) popUp
	{
		NSMenu* menu = [[cell_ menu] copy];
		[popUp setMenu: menu];
		[menu release];
		
		[popUp selectItemAtIndex: [popUp indexOfItemWithRepresentedObject: [self objectValue]]];
		[popUp setTarget: self];
		[popUp setAction: @selector (takeRepresentedObject:)];
	}

- (void) bindToTextField: (NSTextField*) text
	{
		[text setFormatter: [cell_ formatter]];
		[self bindToControl: text];
	}
	
- (void) bindToControl: (NSControl*) control
	{
		[control setObjectValue: [self objectValue]];
		[control setTarget: self];
		[control setAction: @selector (takeObjectValue:)];
	}
	
- (void) takeObjectValue: (id) target
	{
		[self setObjectValue: (NSObject*) [target objectValue]];
	}

- (void) takeRepresentedObject: (id) target
	{
		[self setObjectValue: [[target cell] representedObject]];
	}

- (void) remember
	{
		NSString* key = [NSString stringWithFormat: @"%@%@", prefix_, [attribute_ name]];
		NSString* stringValue = [self stringValue];
		NSUserDefaults* defaults = [NSUserDefaults standardUserDefaults];
		NSString* registeredStringValue = [[defaults volatileDomainForName: NSRegistrationDomain] objectForKey: key];
		
		if (stringValue && ![stringValue isEqualToString: (registeredStringValue ? registeredStringValue : @"")])
			[defaults setObject: stringValue forKey: key];
		else
			[defaults removeObjectForKey: key];
	}
	
- (void) dealloc
	{
		[prefix_ release];
		[attribute_ release];
		[value_ release];
		[cell_ release];
		
		[super dealloc];
	}
	
@end

NSString* OptionDidChange = @"OptionDidChange";

