/*
 *  PGFontTextFieldCell.m
 *  graphviz
 *
 *  Created by Glen Low on Wed May 05 2004.
 *  Copyright (c) 2004, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "PGFontTextFieldCell.h"

static NSMutableArray* fonts_ = nil;
static NSFont* nilFont_ = nil;
static NSMutableParagraphStyle* style_ = nil;

@interface PGFontTextFormatter: NSFormatter
	{
	}
@end

@implementation PGFontTextFieldCell

+ (void) initialize
	{
		// get all available fonts and sort them
		NSArray* fonts = [[NSFontManager sharedFontManager] availableFonts];
		fonts_ = [[NSMutableArray alloc] initWithArray: fonts];
		[fonts_ sortUsingSelector: @selector (compare:)];
		
		// remove all the fonts starting with ".", which are all in a chunk
		BOOL inDot = NO;
		int index = 0;
		while (index < [fonts_ count])
			if ([[fonts_ objectAtIndex: index] hasPrefix: @"."])
				{
					inDot = YES;
					[fonts_ removeObjectAtIndex: index];
				}
			else if (inDot)
				// just past the chunk of fonts with "."; since we are sorted, there can't be any more, so just bail
				break;
			else
				// not past the chunk of fonts with "."
				++index;
		
		// we use the Last Resort font as the nil font, since it's not selectable by the user
		nilFont_ = [[NSFont fontWithName: @"LastResort" size: 0.0] retain];
		
		style_ = [[NSMutableParagraphStyle alloc] init];
		[style_ setAlignment: NSCenterTextAlignment];
		[style_ setLineBreakMode: NSLineBreakByClipping];
	}
	
+ (NSFont*) nilFont
	{
		return nilFont_;
	}

- (id) initTextCell: (NSString*) string
	{
		if ((self = [super initTextCell: string]))
			[self setFormatter: [[[PGFontTextFormatter alloc] init] autorelease]];
			
		return self;
	}
					
- (void) drawWithFrame: (NSRect) cellFrame inView: (NSView*) controlView
	{
		NSRect textFrame = NSMakeRect (
			cellFrame.origin.x,
			cellFrame.origin.y,
			cellFrame.size.width - cellFrame.size.height,
			cellFrame.size.height);
		[super drawWithFrame: textFrame inView: controlView];
		
		NSRect swatchFrame = NSMakeRect (
			cellFrame.origin.x + cellFrame.size.width - cellFrame.size.height + 4.0,
			cellFrame.origin.y,
			cellFrame.size.height,
			cellFrame.size.height);
			
		id objectValue = [self objectValue];
		if (!NSIsEmptyRect (swatchFrame))
			{
				float lineHeight = NSHeight (swatchFrame) - 2.0;
				[style_ setMinimumLineHeight: lineHeight];
				[style_ setMaximumLineHeight: lineHeight];
				
				NSString* representation;
				NSFont* font;
				float size = [[self font] pointSize];
				if ([objectValue isKindOfClass: [NSFont class]] && ![objectValue isEqual: nilFont_])
					{
						representation = [[objectValue fontName] substringToIndex: 1];
						font = [[NSFontManager sharedFontManager] convertFont: objectValue toSize: size];
					}
				else
					{
						representation = @"?";
						font = [NSFont userFontOfSize: size];
					}
				
				[representation
					drawInRect: NSInsetRect (swatchFrame, 1.0, 1.0)
					withAttributes: [NSDictionary dictionaryWithObjectsAndKeys:
						font, NSFontAttributeName,
						[NSColor controlDarkShadowColor], NSForegroundColorAttributeName,
						style_, NSParagraphStyleAttributeName,
						nil]];
			}
	}

- (BOOL) trackMouse: (NSEvent*) event inRect: (NSRect) cellFrame ofView: (NSView*) controlView untilMouseUp: (BOOL) untilMouseUp
	{
		if ([controlView convertPoint: [event locationInWindow] fromView: nil].x < cellFrame.origin.x + cellFrame.size.width - cellFrame.size.height)
			return [super trackMouse: event inRect: cellFrame ofView: controlView untilMouseUp: untilMouseUp];
		else
			{
				event = [[controlView window] nextEventMatchingMask: NSLeftMouseUpMask];
				if ([controlView
					mouse: [controlView convertPoint: [event locationInWindow] fromView: nil]
					inRect: cellFrame])
					{
						[NSApp orderFrontFontPanel: self];
						return YES;
					}
				else
					return NO;				
			}
	}

- (void) editWithFrame: (NSRect) rect inView: (NSView*) controlView editor: (NSText*) text delegate: (id) object event: (NSEvent*) event
	{
		rect.size.width -= rect.size.height + 3.0;
		[super editWithFrame: rect inView: controlView editor: text delegate: object event: event];
	}

- (void) selectWithFrame: (NSRect) rect inView: (NSView*) controlView editor: (NSText*) text delegate: (id) object start: (int) selStart length: (int) selLength
	{
		rect.size.width -= rect.size.height + 3.0;
		[super selectWithFrame: rect inView: controlView editor: text delegate: object start: selStart length: selLength];
	}

@end

@implementation PGFontTextFormatter

- (NSString*) stringForObjectValue: (id) object
	{
		if (!object || [object isEqual: nilFont_])
			return @"";
		else if ([object respondsToSelector: @selector (fontName)])
			return [object fontName];
		else
			return [object description];
	}

- (BOOL) getObjectValue: (id*) object forString: (NSString*) string errorDescription: (NSString**) error
	{
		if (!string || [string isEqualToString: @""])
			*object = nilFont_;
		else
			{
				NSFont* selectedFont = [NSFont fontWithName: string size: 0.0];
				if (selectedFont)
					*object = selectedFont;
				else
					*object = string;
			}
		return YES;
	}
	
- (BOOL) isPartialStringValid: (NSString**) partialStringPtr proposedSelectedRange: (NSRangePointer) proposedSelRangePtr originalString: (NSString*) origString originalSelectedRange: (NSRange) origSelRange errorDescription: (NSString**) error
	{
		int partialStringLength = [*partialStringPtr length];
		if (origSelRange.location + origSelRange.length == [origString length]
			&& proposedSelRangePtr->length == 0
			&& proposedSelRangePtr->location == partialStringLength
			&& partialStringLength > origSelRange.location)
			{
				int select = 0;
				int start = 0;
				int finish = [fonts_ count];
				
				while (start < finish)
					{
						select = (start + finish - 1) / 2;
						
						switch ([*partialStringPtr compare: [fonts_ objectAtIndex: select]])
							{
								case NSOrderedAscending:
									finish = select;
									break;
								case NSOrderedDescending:
									start = select + 1;
									break;
								case NSOrderedSame:
									return YES;
									break;
							}
					}
					
				if (start < [fonts_ count])
					{
						NSString* selectedFont = [fonts_ objectAtIndex: start];
						if ([selectedFont hasPrefix: *partialStringPtr])
							{
								*partialStringPtr = selectedFont;
								*proposedSelRangePtr = NSMakeRange (partialStringLength, [selectedFont length] - partialStringLength);
								return NO;
							}
					}
			}
		
		return YES;
	}


@end