#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>
#include <sys/stat.h>
#if defined(HAVE_CLAMAV)
# include <clamav.h>
#else
#warning No antivirus library defined, this module will not operate with any virus database
#endif
#include <gtk/gtk.h>
#ifdef HAVE_EDV2
# include <endeavour2.h>
#endif

#include "cfg.h"
#include "win.h"
#include "wincb.h"
#include "windb.h"
#include "core.h"
#include "cfglist.h"
#include "config.h"


/* Utilities */
static gchar *COPY_SHORTEN_STRING(const gchar *s, gint max);
static const gchar *TIME_DURATION_STRING(
	core_struct *core, gulong dt, gulong t
);

#if defined(HAVE_CLAMAV)
static void WinDBUpdateIterate(
	core_struct *core, win_struct *win,
	GtkCTree *ctree, GtkCTreeNode *pnode, gchar **strv,
	struct cl_node *clam_pnode,
	const gboolean verbose, gint *patterns_loaded
);
#endif
void WinDBUpdate(
	win_struct *win, const gchar *db_location,
	const gboolean verbose
);
void WinDBQueueUpdate(win_struct *win);
void WinDBClear(win_struct *win);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#define INSERT_NODE_FOLDER(_s_,_pnode_,_snode_) {       \
 if(columns > 0)                                        \
  strv[0] = (_s_);                                      \
 node = gtk_ctree_insert_node(                          \
  ctree, (_pnode_), (_snode_),                          \
  strv, WIN_LIST_PIXMAP_TEXT_SPACING,                   \
  core->folder_closed_pixmap,                           \
  core->folder_closed_mask,                             \
  core->folder_opened_pixmap,				\
  core->folder_opened_mask,				\
  FALSE, TRUE						\
 );							\
}
#define INSERT_NODE_ITEM(_s_,_pnode_,_snode_)   {       \
 if(columns > 0)                                        \
  strv[0] = (_s_);                                      \
 node = gtk_ctree_insert_node(                          \
  ctree, (_pnode_), (_snode_),                          \
  strv, WIN_LIST_PIXMAP_TEXT_SPACING,                   \
  core->db_item_pixmap, core->db_item_mask,             \
  core->db_item_pixmap, core->db_item_mask,             \
  TRUE, FALSE                                           \
 );                                                     \
}


/*
 *      Returns a copy of the string that is no longer than max
 *      characters with appropriate shortened notation where
 *      appropriate.
 */
static gchar *COPY_SHORTEN_STRING(const gchar *s, gint max)
{
	gint len;

	if(s == NULL)
	    return(NULL);

	len = STRLEN(s);
	if((len > max) && (max > 3))
	{
	    /* Need to shorten string */
	    gint i = len - max + 3;

	    return(g_strdup_printf(
		"...%s", (const gchar *)(s + i)
	    ));
	}
	else
	{
	    return(STRDUP(s));
	}
}

#ifndef HAVE_EDV2
static const gchar *EDVSizeStrDelim(gulong v)
{
	static gchar s[80];
	g_snprintf(
	    s, sizeof(s),
	    "%ld", v
	);
	return(s);
}
#endif

/*
 *      Returns a statically allocated string describing the time
 *      lapsed specified by dt.
 */
static const gchar *TIME_DURATION_STRING(
	core_struct *core, gulong dt, gulong t
)
{
	gulong ct;
	static gchar buf[80];
			     
	/* Less than one second? */
	if(dt < 1)
	{
	    g_snprintf(
		buf, sizeof(buf),
		"less than a second ago"
	    );
	}
	/* Less than one minute? */
	else if(dt < (1 * 60))  
	{
	    ct = MAX(dt / 1, 1);
	    g_snprintf( 
		buf, sizeof(buf),   
		"%ld second%s ago",
		ct, (ct == 1) ? "" : "s"
	    );
	}     
	/* Less than one hour? */  
	else if(dt < (60 * 60))
	{
	    ct = MAX(dt / 60, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld minute%s ago",
		ct, (ct == 1) ? "" : "s"
	    );
	}     
	/* Less than one day? */
	else if(dt < (24 * 60 * 60))
	{
	    ct = MAX(dt / 60 / 60, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld hour%s ago",
		ct, (ct == 1) ? "" : "s"
	    );               
	}
	/* Less than one week? */
	else if(dt < (7 * 24 * 60 * 60))
	{
	    ct = MAX(dt / 60 / 60 / 24, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld day%s ago",
		ct, (ct == 1) ? "" : "s"
	    );
	}     
	/* Less than one month (30 days)? */
	else if(dt < (30 * 24 * 60 * 60))
	{
	    ct = MAX(dt / 60 / 60 / 24 / 7, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld week%s ago",
		ct, (ct == 1) ? "" : "s"
	    );
	}     
	/* Less than 6 months ago? */
	else if(dt < (6 * 30 * 24 * 60 * 60))
	{
	    ct = MAX(dt / 60 / 60 / 24 / 30, 1);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld month%s",
		ct, (ct == 1) ? "" : "s"
	    );
	}     
	else
	{
#ifdef HAVE_EDV2
	    strncpy(buf, EDVDateString(core->edv_ctx, t), sizeof(buf));
#else
	    gchar *s;
	    time_t tt = (time_t)t;
	    strncpy(buf, ctime(&tt), sizeof(buf));
	    s = strchr(buf, '\n');
	    if(s != NULL)
		*s = '\0';
#endif
	}

	buf[sizeof(buf) - 1] = '\0';

	return(buf);
}


#if defined(HAVE_CLAMAV)
/*
 *	Loads Clam AntiVirus child nodes and patterns.
 */
static void WinDBUpdateIterate(
	core_struct *core, win_struct *win,
	GtkCTree *ctree, GtkCTreeNode *pnode, gchar **strv,
	struct cl_node *clam_pnode,
	const gboolean verbose, gint *patterns_loaded
)
{
	gint i;
	struct cli_ac_node *clam_ac_node;
	GtkCList *clist = GTK_CLIST(ctree);
	const gint columns = clist->columns;

#define UPDATE_PATTERNS_LOADED_STATUS	{		\
 /* Count this pattern as loaded */			\
 *patterns_loaded = (*patterns_loaded) + 1;		\
							\
 /* Update progress every 50 patterns loaded */		\
 if(verbose && (((*patterns_loaded) % 50) == 0))	\
  WinStatusProgress(win, -1.0f, TRUE);			\
							\
 /* Update status message every 100 patterns loaded */	\
 if(verbose && (((*patterns_loaded) % 100) == 0)) {	\
  gchar *s2 = STRDUP(EDVSizeStrDelim(			\
    (gulong)(*patterns_loaded)				\
  ));							\
  gchar *s = g_strdup_printf(				\
   "%s patterns loaded...",				\
   s2							\
  );							\
  g_free(s2);						\
  WinStatusMessage(win, s, TRUE);			\
  g_free(s);						\
 }							\
							\
 /* Thaw and refreeze list every 500 patterns loaded */	\
 if(((*patterns_loaded) % 500) == 0) {			\
  gtk_clist_thaw(clist);				\
  gtk_clist_freeze(clist);				\
 }							\
}

	/* Iterate through Extended Boyer-Moore list */
	if(clam_pnode->bm_suffix != NULL)
	{
	    gchar *virname;
	    struct cli_bm_patt *pattern;
	    GtkCTreeNode *node, *pnode2;

	    for(i = 0; clam_pnode->bm_suffix[i] != NULL; i++)
	    {
		if(win->stop_count > 0)
		    break;

		/* Get list of patterns */
	        pattern = clam_pnode->bm_suffix[i];

		INSERT_NODE_FOLDER(
		    "Extended Boyer-Moore", pnode, NULL
		);
		if(node == NULL)
		    break;
		pnode2 = node;

		/* Iterate through list of patterns */
		while(pattern != NULL)
		{
		    if(win->stop_count > 0)
			break;

		    if(!STRISEMPTY(pattern->virname) &&
		       !STRISEMPTY(pattern->viralias)
		    )
		    {
			virname = g_strdup_printf(
			    "%s (%s)",
			    pattern->virname,
			    pattern->viralias
			);
		    }     
		    else if(!STRISEMPTY(pattern->virname))
		    {
			virname = STRDUP(pattern->virname);
		    }
		    else if(!STRISEMPTY(pattern->viralias))
		    {
			virname = STRDUP(pattern->viralias);
		    }
		    else
		    {
			pattern = pattern->next;
			continue;
		    }

		    /* Add this pattern to the list */
		    INSERT_NODE_ITEM(virname, pnode2, NULL);
		    g_free(virname);

		    UPDATE_PATTERNS_LOADED_STATUS

		    /* Get next pattern */
		    pattern = pattern->next;
		}
	    }
	}


	/* Iterate through Extended Aho-Corasick list */
	for(i = 0; i < clam_pnode->ac_nodes; i++)
	{
	    if(win->stop_count > 0)
		break;

	    clam_ac_node = clam_pnode->ac_nodetable[i];
	    if(clam_ac_node != NULL)
	    {
		gchar *virname;
		GtkCTreeNode *node, *pnode2;

		/* Get list of patterns */
		struct cli_ac_patt *pattern = clam_ac_node->list;

		/* Add folder node if pattern list is not empty */
		if(pattern != NULL)
		{
		    INSERT_NODE_FOLDER(
			"Extended Aho-Corasick", pnode, NULL
		    );
		    if(node == NULL)
			break;
		    pnode2 = node;
		}

		/* Iterate through list of patterns */
		while(pattern != NULL)
		{
		    if(win->stop_count > 0)
			break;

		    if(!STRISEMPTY(pattern->virname) && 
		       !STRISEMPTY(pattern->viralias) 
		    )
		    {
			virname = g_strdup_printf(
			    "%s (%s)",
			    pattern->virname,
			    pattern->viralias
			);
		    }
		    else if(!STRISEMPTY(pattern->virname))
		    {
			virname = STRDUP(pattern->virname);
		    }
		    else if(!STRISEMPTY(pattern->viralias))
		    {
			virname = STRDUP(pattern->viralias);
		    }
		    else
		    {   
			pattern = pattern->next;
			continue;
		    }   

		    /* Add this pattern to the list */
		    INSERT_NODE_ITEM(virname, pnode2, NULL);
		    g_free(virname);

		    UPDATE_PATTERNS_LOADED_STATUS

		    /* Get next pattern */
		    pattern = pattern->next;
		}
	    }
	}


	/* Iterate through MD5 list */
	if(clam_pnode->md5_hlist != NULL)
	{
	    gchar *virname;
	    struct cli_md5_node *pattern;
	    GtkCTreeNode *node, *pnode2;

	    for(i = 0; clam_pnode->md5_hlist[i] != NULL; i++)
	    {
		if(win->stop_count > 0)
		    break;

		/* Get list of patterns */
		pattern = clam_pnode->md5_hlist[i];

		INSERT_NODE_FOLDER(
		    "MD5", pnode, NULL
		);
		if(node == NULL)
		    break;
		pnode2 = node;

		/* Iterate through list of patterns */
		while(pattern != NULL)
		{
		    if(win->stop_count > 0)
			break;

		    if(!STRISEMPTY(pattern->virname) &&
		       !STRISEMPTY(pattern->viralias) 
		    )
		    {
			virname = g_strdup_printf(
			    "%s (%s)",
			    pattern->virname,
			    pattern->viralias
			);
		    }
		    else if(!STRISEMPTY(pattern->virname))
		    {
			virname = STRDUP(pattern->virname);
		    }
		    else if(!STRISEMPTY(pattern->viralias))
		    {
			virname = STRDUP(pattern->viralias);
		    }
		    else
		    {
			pattern = pattern->next;
			continue;
		    }

		    /* Add this pattern to the list */
		    INSERT_NODE_ITEM(virname, pnode2, NULL);
		    g_free(virname);

		    UPDATE_PATTERNS_LOADED_STATUS

		    /* Get next pattern */
		    pattern = pattern->next;
		}
	    }
	}

#undef UPDATE_PATTERNS_LOADED_STATUS

#endif
}

/*
 *	Updates the Win's Virus Database Tree with the information
 *	from the specified virus database location.
 */
void WinDBUpdate(
	win_struct *win, const gchar *db_location,
	const gboolean verbose
)
{
#if defined(HAVE_CLAMAV)
	struct stat stat_buf;
	struct cl_node *database = NULL;
	gint status, database_entries = 0, patterns_loaded = 0;
#endif
	GtkWidget *w, *toplevel;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	core_struct *core;
	GtkCTree *ctree = (win != NULL) ? (GtkCTree *)win->db_ctree : NULL;
	if(ctree == NULL)
	    return;     

	toplevel = win->toplevel;
	clist = GTK_CLIST(ctree);
	core = CORE(win->core);
	cfg_list = core->cfg_list;


	/* Clear */
	WinDBClear(win);


	/* Begin updating */

	/* Reset stop count and update so the stop button can be used */
	win->stop_count = 0;
	WinSetPassiveBusy(win, TRUE);
	WinUpdate(win);

	if(verbose)
	    WinStatusProgress(win, -1.0f, TRUE);

	/* If the virus database location was not specified then use
	 * the one specified in the configuration
	 */
	if(db_location == NULL)
	    db_location = CFGItemListGetValueS(
		cfg_list, CFG_PARM_DB_LOCATION
	    );
#if defined(HAVE_CLAMAV)
	if(db_location == NULL)
	    db_location = cl_retdbdir();
#endif
	if(db_location == NULL)
	{
	    if(verbose)
	    {
		WinStatusMessage(
		    win,
"No virus database location specified",
		    FALSE
		);
		WinStatusProgress(win, 0.0f, FALSE);
	    }
	    WinSetPassiveBusy(win, FALSE);
	    return;
	}


#if defined(HAVE_CLAMAV)
	if(stat(db_location, &stat_buf))
	{
	    const gint error_code = (gint)errno;
	    if(verbose)
	    {
		gchar *s, *error_msg = STRDUP(g_strerror(error_code));
		if(error_msg != NULL)
		{
		    *error_msg = (gchar)toupper((int)*error_msg);
		    s = g_strdup_printf(
			"%s: %s",
			db_location,
			error_msg
		    );
		    WinStatusMessage(win, s, FALSE);
		    g_free(s);
		    g_free(error_msg);
		}
		WinStatusProgress(win, 0.0f, FALSE);
	    }
	    WinSetPassiveBusy(win, FALSE);
	    return;
	}

	/* Load virus database */
	if(verbose)
	{
	    gchar *buf = g_strdup_printf(
		"Loading virus database \"%s\"...",
		db_location
	    );
	    WinStatusMessage(win, buf, TRUE);
	    g_free(buf);
	}
	if(S_ISDIR(stat_buf.st_mode))
	    status = cl_loaddbdir(db_location, &database, &database_entries);
	else
	    status = cl_loaddb(db_location, &database, &database_entries);
	if(!status && (database != NULL) && (database_entries > 0))
	{
	    gint i;
	    const gint columns = clist->columns;
	    gchar **strv = (gchar **)g_malloc(columns * sizeof(gchar *));
	    GtkCTreeNode *node;

	    if(verbose)
		WinStatusProgress(win, -1.0f, TRUE);

	    for(i = 0; i < columns; i++)
		strv[i] = "";

	    /* Build the virus database */
	    if(cl_build(database))
	    {
		if(verbose)
		{
		    WinStatusMessage(
			win,
"Unable to build the virus database",
			FALSE
		    );
		    WinStatusProgress(win, 0.0f, FALSE);
		}
		g_free(strv);
		WinSetPassiveBusy(win, FALSE);
		return;
	    }


	    /* Begin adding the nodes and patterns to the Virus
	     * Database Tree
	     */
	    if(verbose)
		WinStatusProgress(win, -1.0f, TRUE);
	    gtk_clist_freeze(clist);

	    /* Add root node */
	    INSERT_NODE_FOLDER("", NULL, NULL);
	    if(node == NULL)
	    {
		gtk_clist_thaw(clist);
		g_free(strv);
		WinSetPassiveBusy(win, FALSE);
		return;
	    }
	    win->db_root_node = node;

	    /* Add all child nodes and items to this node */
	    WinDBUpdateIterate(
		core, win, ctree, node, strv, database, verbose,
		&patterns_loaded
	    );

	    gtk_clist_thaw(clist);
	    if(verbose)
		WinStatusProgress(win, -1.0f, TRUE);


	    /* Delete the virus database */
	    cl_free(database);

	    g_free(strv);
	}
	else
	{
	    WinStatusMessage(
		win,
"Unable to load the virus database",
		FALSE
	    );
	}
#endif


	/* Update Virus Database Stats Label */
	w = win->db_stats_label;
	if(w != NULL)
	{
	    gulong	cur_t = (gulong)time(NULL),
			last_updated_t = CFGItemListGetValueL(
			    cfg_list, CFG_PARM_DB_LAST_UPDATE
			);
	    gchar	*db_location_s = COPY_SHORTEN_STRING(
			    db_location, 35
			),
			*last_updated_s = STRDUP((last_updated_t > 0) ?
		TIME_DURATION_STRING(core, cur_t - last_updated_t, last_updated_t) : "Unknown"
			),
			*patterns_loaded_s = STRDUP(EDVSizeStrDelim(
		(gulong)patterns_loaded
			));
	    gchar *buf = g_strdup_printf(
		"%s\n%s\n%s%s",
		db_location_s,
		last_updated_s,
		patterns_loaded_s,
		(win->stop_count > 0) ? " (Interrupted)" : ""
	    );
	    gtk_label_set_text(GTK_LABEL(w), buf);
	    g_free(buf);
	    g_free(db_location_s);
	    g_free(last_updated_s);
	    g_free(patterns_loaded_s);
	}

	/* Print results */
	if(verbose)
	{
	    gchar *patterns_loaded_s = STRDUP(EDVSizeStrDelim(
		(gulong)patterns_loaded
	    ));
	    gchar *buf = g_strdup_printf(
		"Virus database loaded with %s patterns",
		patterns_loaded_s
	    );

	    /* Raise Win */
	    WinMap(win);

	    WinStatusMessage(win, buf, FALSE);
	    g_free(buf);
	    g_free(patterns_loaded_s);

	    WinStatusProgress(win, 0.0f, FALSE);
	}

	WinSetPassiveBusy(win, FALSE);
}

/*
 *	Queues a call to WinDBUpdate().
 */
void WinDBQueueUpdate(win_struct *win)
{
	if((win != NULL) ? (win->db_ctree_load_idleid == 0) : FALSE)
	    win->db_ctree_load_idleid = gtk_idle_add_priority(
		G_PRIORITY_LOW,
		WinDBTreeLoadStartIdleCB, win
	    );
}

/*
 *	Deletes all nodes in the Win's Virus Database Tree & Stats
 *	label.
 */
void WinDBClear(win_struct *win)
{
	GtkCList *clist;
	core_struct *core;
	GtkCTree *ctree = (win != NULL) ? (GtkCTree *)win->db_ctree : NULL;
	if(ctree == NULL)
	    return;

	clist = GTK_CLIST(ctree);
	core = CORE(win->core);

	/* Begin clearing the tree */
	gtk_clist_freeze(clist);

	/* Remove the root node */
	if(win->db_root_node != NULL)
	{
	    gtk_ctree_remove_node(ctree, win->db_root_node);
	    win->db_root_node = NULL;
	}

	gtk_clist_thaw(clist);


	/* Clear stats label */
	gtk_label_set_text(GTK_LABEL(win->db_stats_label), "");
}
