/* $TOG: Region.c /main/31 1998/02/06 17:50:22 kaleb $ */
/************************************************************************

Copyright 1987, 1988, 1998  The Open Group

All Rights Reserved.

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of The Open Group shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from The Open Group.


Copyright 1987, 1988 by Digital Equipment Corporation, Maynard, Massachusetts.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of Digital not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  

DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.

************************************************************************/
/* $XFree86: xc/lib/X11/Region.c,v 1.5 1999/05/09 10:50:01 dawes Exp $ */
/*
 * The functions in this file implement the Region abstraction, similar to one
 * used in the X11 sample server. A Region is simply an area, as the name
 * implies, and is implemented as a "y-x-banded" array of rectangles. To
 * explain: Each Region is made up of a certain number of rectangles sorted
 * by y coordinate first, and then by x coordinate.
 *
 * Furthermore, the rectangles are banded such that every rectangle with a
 * given upper-left y coordinate (y1) will have the same lower-right y
 * coordinate (y2) and vice versa. If a rectangle has scanlines in a band, it
 * will span the entire vertical distance of the band. This means that some
 * areas that could be merged into a taller rectangle will be represented as
 * several shorter rectangles to account for shorter rectangles to its left
 * or right but within its "vertical scope".
 *
 * An added constraint on the rectangles is that they must cover as much
 * horizontal area as possible. E.g. no two rectangles in a band are allowed
 * to touch.
 *
 * Whenever possible, bands will be merged together to cover a greater vertical
 * distance (and thus reduce the number of rectangles). Two bands can be merged
 * only if the bottom of one touches the top of the other and they have
 * rectangles in the same places (of the same width, of course). This maintains
 * the y-x-banding that's so nice to have...
 */

// Stolen from Gtk+ and C++-ized by Ron Steinke, January 2003

#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "region.h"
#include "region-generic.h"

/*
#ifdef DEBUG
#include <stdio.h>
#define assert(expr) {if (!(expr)) fprintf(stderr,\
"Assertion failed file %s, line %d: expr\n", __FILE__, __LINE__); }
#else
#define assert(expr)
#endif
*/

#define MAX(x, y) ((x) > (y) ? (x) : (y))
#define MIN(x, y) ((x) < (y) ? (x) : (y))

using namespace wftk;

/*	Create a new empty region	*/

Region::Region ()
{
  rects = new RegionBox[1];

  numRects = 0;
  extents.x1 = 0;
  extents.y1 = 0;
  extents.x2 = 0;
  extents.y2 = 0;
  size = 1;
}

/**
 * gdk_region_rectangle:
 * @rectangle: a #GdkRectangle
 * 
 * Creates a new region containing the area @rectangle.
 * 
 * Return value: a new region
 **/
Region::Region (const SDL_Rect& rectangle)
{
  rects = new RegionBox[1];

  if (rectangle.w <= 0 || rectangle.h <= 0)
    {
      numRects = 0;
      extents.x1 = 0;
      extents.y1 = 0;
      extents.x2 = 0;
      extents.y2 = 0;
    }
  else
    {
      numRects = 1;
      extents.x1 = rects[0].x1 = rectangle.x;
      extents.y1 = rects[0].y1 = rectangle.y;
      extents.x2 = rects[0].x2 = rectangle.x + rectangle.w;
      extents.y2 = rects[0].y2 = rectangle.y + rectangle.h;
    }
  size = 1;
}

Region::Region (const Point& p)
{
  rects = new RegionBox[1];
  numRects = 1;
  extents.x1 = rects[0].x1 = p.x;
  extents.y1 = rects[0].y1 = p.y;
  extents.x2 = rects[0].x2 = p.x + 1;
  extents.y2 = rects[0].y2 = p.y + 1;
  size = 1;
}

/**
 * gdk_region_copy:
 * @region: a #GdkRegion
 * 
 * Copies @region, creating an identical new region.
 * 
 * Return value: a new region identical to @region
 **/
Region::Region (const Region& region)
{
  rects = new RegionBox[region.numRects];

  numRects = region.numRects;
  extents = region.extents;
  size = region.numRects;
  
  memcpy (rects, region.rects, region.numRects * sizeof (RegionBox));
}

Region::RectList& Region::RectList::operator=(const RectList& rl)
{
  if(rl.nrects_ != nrects_) {
    delete [] rects_;
    rects_ = new SDL_Rect[rl.nrects_];
    nrects_ = rl.nrects_;
  }

  memcpy(rects_, rl.rects_, nrects_ * sizeof(SDL_Rect));

  return *this;
}

/**
 * gdk_region_get_rectangles:
 * @region: a #GdkRegion
 * @rectangles: return location for an array of rectangles
 * @n_rectangles: length of returned array
 *
 * Obtains the area covered by the region as a list of rectangles.
 * The array returned in @rectangles must be freed with g_free().
 * 
 **/
Region::RectList
Region::getRectangles () const
{
  RectList rectangles(numRects);

  for (int i = 0; i < numRects; i++)
    rectangles[i] = rects[i];

  return rectangles;
}

/*-
 *-----------------------------------------------------------------------
 * miSetExtents --
 *	Reset the extents of a region to what they should be. Called by
 *	miSubtract and miIntersect b/c they can't figure it out along the
 *	way or do so easily, as miUnion can.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The region's 'extents' structure is overwritten.
 *
 *-----------------------------------------------------------------------
 */
void
Region::miSetExtents()
{
  RegionBox *pBox, *pBoxEnd, *pExtents;

  if (numRects == 0)
    {
      extents.x1 = 0;
      extents.y1 = 0;
      extents.x2 = 0;
      extents.y2 = 0;
      return;
    }

  pExtents = &extents;
  pBox = rects;
  pBoxEnd = &pBox[numRects - 1];

    /*
     * Since pBox is the first rectangle in the region, it must have the
     * smallest y1 and since pBoxEnd is the last rectangle in the region,
     * it must have the largest y2, because of banding. Initialize x1 and
     * x2 from  pBox and pBoxEnd, resp., as good things to initialize them
     * to...
     */
  pExtents->x1 = pBox->x1;
  pExtents->y1 = pBox->y1;
  pExtents->x2 = pBoxEnd->x2;
  pExtents->y2 = pBoxEnd->y2;

  assert(pExtents->y1 < pExtents->y2);
  while (pBox <= pBoxEnd)
    {
      if (pBox->x1 < pExtents->x1)
	{
	  pExtents->x1 = pBox->x1;
	}
      if (pBox->x2 > pExtents->x2)
	{
	  pExtents->x2 = pBox->x2;
	}
      pBox++;
    }
  assert(pExtents->x1 < pExtents->x2);
}

/* TranslateRegion(pRegion, x, y)
   translates in place
   added by raymond
*/

void
Region::offset (int x, int y)
{
  int nbox;
  RegionBox *pbox;

  pbox = rects;
  nbox = numRects;

  while(nbox--)
    {
      pbox->x1 += x;
      pbox->x2 += x;
      pbox->y1 += y;
      pbox->y2 += y;
      pbox++;
    }
  extents.x1 += x;
  extents.x2 += x;
  extents.y1 += y;
  extents.y2 += y;
}

/* 
   Utility procedure Compress:
   Replace r by the region r', where 
     p in r' iff (Quantifer m <= dx) (p + m in r), and
     Quantifier is Exists if grow is TRUE, For all if grow is FALSE, and
     (x,y) + m = (x+m,y) if xdir is TRUE; (x,y+m) if xdir is FALSE.

   Thus, if xdir is TRUE and grow is FALSE, r is replaced by the region
   of all points p such that p and the next dx points on the same
   horizontal scan line are all in r.  We do this using by noting
   that p is the head of a run of length 2^i + k iff p is the head
   of a run of length 2^i and p+2^i is the head of a run of length
   k. Thus, the loop invariant: s contains the region corresponding
   to the runs of length shift.  r contains the region corresponding
   to the runs of length 1 + dxo & (shift-1), where dxo is the original
   value of dx.  dx = dxo & ~(shift-1).  As parameters, s and t are
   scratch regions, so that we don't have to allocate them on every
   call.
*/

#define ZOpRegion(a,b) if (grow) *a |= *b; \
			 else *a &= *b
#define ZShiftRegion(a,b) if (xdir) a->offset (b,0); \
			  else a->offset (0,b)

void
Region::Compress(
	 Region *s,
	 Region *t,
	 unsigned int      dx,
	 int        xdir,
	 int        grow)
{
  unsigned int shift = 1;

  *s = *this;
  while (dx)
    {
      if (dx & shift)
	{
	  ZShiftRegion(this, -(int)shift);
	  ZOpRegion(this, s);
	  dx -= shift;
	  if (!dx) break;
        }
      *t = *s;
      ZShiftRegion(s, -(int)shift);
      ZOpRegion(s, t);
      shift <<= 1;
    }
}

#undef ZOpRegion
#undef ZShiftRegion
#undef ZCopyRegion

void
Region::shrink(int dx, int dy)
{
  if (!dx && !dy)
    return;

  Region s, t;

  int grow = (dx < 0);
  if (grow)
    dx = -dx;
  if (dx)
     Compress(&s, &t, (unsigned) 2*dx, true, grow);
     
  grow = (dy < 0);
  if (grow)
    dy = -dy;
  if (dy)
     Compress(&s, &t, (unsigned) 2*dy, false, grow);
  
  offset (dx, dy);
}


/*======================================================================
 *	    Region Intersection
 *====================================================================*/
/*-
 *-----------------------------------------------------------------------
 * miIntersectO --
 *	Handle an overlapping band for miIntersect.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Rectangles may be added to the region.
 *
 *-----------------------------------------------------------------------
 */
/* static void*/
void
Region::miIntersectO (
	      RegionBox *r1,
	      RegionBox *r1End,
	      RegionBox *r2,
	      RegionBox *r2End,
	      int          y1,
	      int          y2)
{
  int  	x1;
  int  	x2;
  RegionBox *pNextRect;

  pNextRect = &rects[numRects];

  while ((r1 != r1End) && (r2 != r2End))
    {
      x1 = MAX (r1->x1,r2->x1);
      x2 = MIN (r1->x2,r2->x2);

      /*
       * If there's any overlap between the two rectangles, add that
       * overlap to the new region.
       * There's no need to check for subsumption because the only way
       * such a need could arise is if some region has two rectangles
       * right next to each other. Since that should never happen...
       */
      if (x1 < x2)
	{
	  assert (y1<y2);

	  MEMCHECK (this, pNextRect, rects);
	  pNextRect->x1 = x1;
	  pNextRect->y1 = y1;
	  pNextRect->x2 = x2;
	  pNextRect->y2 = y2;
	  numRects += 1;
	  pNextRect++;
	  assert (numRects <= size);
	}

      /*
       * Need to advance the pointers. Shift the one that extends
       * to the right the least, since the other still has a chance to
       * overlap with that region's next rectangle, if you see what I mean.
       */
      if (r1->x2 < r2->x2)
	{
	  r1++;
	}
      else if (r2->x2 < r1->x2)
	{
	  r2++;
	}
      else
	{
	  r1++;
	  r2++;
	}
    }
}

/**
 * gdk_region_intersect:
 * @source1: a #GdkRegion
 * @source2: another #GdkRegion
 *
 * Sets the area of @source1 to the intersection of the areas of @source1
 * and @source2. The resulting area is the set of pixels contained in
 * both @source1 and @source2.
 **/
Region&
Region::operator&=(const Region& other)
{
  /* check for trivial reject */
  if ((!(numRects)) || (!(other.numRects))  ||
      (!EXTENTCHECK(&extents, &other.extents)))
    numRects = 0;
  else
    miRegionOp (this, &other, 
    		&Region::miIntersectO, (nonOverlapFunc) NULL, (nonOverlapFunc) NULL);
    
  /*
   * Can't alter region's extents before miRegionOp depends on the
   * extents of the regions being unchanged. Besides, this way there's
   * no checking against rectangles that will be nuked due to
   * coalescing, so we have to examine fewer rectangles.
   */
  miSetExtents();

  return *this;
}

Region& Region::operator=(const Region &rgn)
{
  if (this != &rgn) /*  don't want to copy to itself */
    {  
      if (size < rgn.numRects)
        {
	  delete [] rects;
	  rects = new RegionBox[rgn.numRects];
	  size = rgn.numRects;
	}
      numRects = rgn.numRects;
      extents.x1 = rgn.extents.x1;
      extents.y1 = rgn.extents.y1;
      extents.x2 = rgn.extents.x2;
      extents.y2 = rgn.extents.y2;

      memcpy (rects, rgn.rects, rgn.numRects * sizeof (RegionBox));
    }

  return *this;
}

void Region::clear()
{
  numRects = 0;
  extents.x1 = 0;
  extents.y1 = 0;
  extents.x2 = 0;
  extents.y2 = 0;
}


/*======================================================================
 *	    Generic Region Operator
 *====================================================================*/

/*-
 *-----------------------------------------------------------------------
 * miCoalesce --
 *	Attempt to merge the boxes in the current band with those in the
 *	previous one. Used only by miRegionOp.
 *
 * Results:
 *	The new index for the previous band.
 *
 * Side Effects:
 *	If coalescing takes place:
 *	    - rectangles in the previous band will have their y2 fields
 *	      altered.
 *	    - pReg->numRects will be decreased.
 *
 *-----------------------------------------------------------------------
 */
/* static int*/
int
Region::miCoalesce (
	    int       prevStart,    /* Index of start of previous band */
	    int       curStart)     /* Index of start of current band */
{
  RegionBox *pPrevBox;   	/* Current box in previous band */
  RegionBox *pCurBox;    	/* Current box in current band */
  RegionBox *pRegEnd;    	/* End of region */
  int	    	curNumRects;	/* Number of rectangles in current
				 * band */
  int	    	prevNumRects;	/* Number of rectangles in previous
				 * band */
  int	    	bandY1;	    	/* Y1 coordinate for current band */

  pRegEnd = &rects[numRects];

  pPrevBox = &rects[prevStart];
  prevNumRects = curStart - prevStart;

    /*
     * Figure out how many rectangles are in the current band. Have to do
     * this because multiple bands could have been added in miRegionOp
     * at the end when one region has been exhausted.
     */
  pCurBox = &rects[curStart];
  bandY1 = pCurBox->y1;
  for (curNumRects = 0;
       (pCurBox != pRegEnd) && (pCurBox->y1 == bandY1);
       curNumRects++)
    {
      pCurBox++;
    }
    
  if (pCurBox != pRegEnd)
    {
      /*
       * If more than one band was added, we have to find the start
       * of the last band added so the next coalescing job can start
       * at the right place... (given when multiple bands are added,
       * this may be pointless -- see above).
       */
      pRegEnd--;
      while (pRegEnd[-1].y1 == pRegEnd->y1)
	{
	  pRegEnd--;
	}
      curStart = pRegEnd - rects;
      pRegEnd = rects + numRects;
    }
	
  if ((curNumRects == prevNumRects) && (curNumRects != 0)) {
    pCurBox -= curNumRects;
    /*
     * The bands may only be coalesced if the bottom of the previous
     * matches the top scanline of the current.
     */
    if (pPrevBox->y2 == pCurBox->y1)
      {
	/*
	 * Make sure the bands have boxes in the same places. This
	 * assumes that boxes have been added in such a way that they
	 * cover the most area possible. I.e. two boxes in a band must
	 * have some horizontal space between them.
	 */
	do
	  {
	    if ((pPrevBox->x1 != pCurBox->x1) ||
		(pPrevBox->x2 != pCurBox->x2))
	      {
		/*
		 * The bands don't line up so they can't be coalesced.
		 */
		return (curStart);
	      }
	    pPrevBox++;
	    pCurBox++;
	    prevNumRects -= 1;
	  } while (prevNumRects != 0);

	numRects -= curNumRects;
	pCurBox -= curNumRects;
	pPrevBox -= curNumRects;

	/*
	 * The bands may be merged, so set the bottom y of each box
	 * in the previous band to that of the corresponding box in
	 * the current band.
	 */
	do
	  {
	    pPrevBox->y2 = pCurBox->y2;
	    pPrevBox++;
	    pCurBox++;
	    curNumRects -= 1;
	  }
	while (curNumRects != 0);

	/*
	 * If only one band was added to the region, we have to backup
	 * curStart to the start of the previous band.
	 *
	 * If more than one band was added to the region, copy the
	 * other bands down. The assumption here is that the other bands
	 * came from the same region as the current one and no further
	 * coalescing can be done on them since it's all been done
	 * already... curStart is already in the right place.
	 */
	if (pCurBox == pRegEnd)
	  {
	    curStart = prevStart;
	  }
	else
	  {
	    do
	      {
		*pPrevBox++ = *pCurBox++;
	      }
	    while (pCurBox != pRegEnd);
	  }
	    
      }
  }
  return curStart;
}

/*-
 *-----------------------------------------------------------------------
 * miRegionOp --
 *	Apply an operation to two regions. Called by miUnion, miInverse,
 *	miSubtract, miIntersect...
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The new region is overwritten.
 *
 * Notes:
 *	The idea behind this function is to view the two regions as sets.
 *	Together they cover a rectangle of area that this function divides
 *	into horizontal bands where points are covered only by one region
 *	or by both. For the first case, the nonOverlapFunc is called with
 *	each the band and the band's upper and lower extents. For the
 *	second, the overlapFunc is called to process the entire band. It
 *	is responsible for clipping the rectangles in the band, though
 *	this function provides the boundaries.
 *	At the end of each band, the new region is coalesced, if possible,
 *	to reduce the number of rectangles in the region.
 *
 *-----------------------------------------------------------------------
 */
/* static void*/
void
Region::miRegionOp(
	   const Region *reg1,
	   const Region *reg2,
	   overlapFunc    overlapFn,   	        /* Function to call for over-
						 * lapping bands */
	   nonOverlapFunc nonOverlap1Fn,	/* Function to call for non-
						 * overlapping bands in region
						 * 1 */
	   nonOverlapFunc nonOverlap2Fn)	/* Function to call for non-
						 * overlapping bands in region
						 * 2 */
{
    RegionBox *r1; 	    	    	/* Pointer into first region */
    RegionBox *r2; 	    	    	/* Pointer into 2d region */
    RegionBox *r1End;    	    	/* End of 1st region */
    RegionBox *r2End;    	    	/* End of 2d region */
    int    	  ybot;	    	    	/* Bottom of intersection */
    int  	  ytop;	    	    	/* Top of intersection */
    RegionBox *oldRects;   	    	/* Old rects for newReg */
    int	    	  prevBand;   	    	/* Index of start of
					 * previous band in newReg */
    int	  	  curBand;    	    	/* Index of start of current
					 * band in newReg */
    RegionBox *r1BandEnd;  	    	/* End of current band in r1 */
    RegionBox *r2BandEnd;  	    	/* End of current band in r2 */
    int     	  top;	    	    	/* Top of non-overlapping
					 * band */
    int     	  bot;	    	    	/* Bottom of non-overlapping
					 * band */
    
    /*
     * Initialization:
     *	set r1, r2, r1End and r2End appropriately, preserve the important
     * parts of the destination region until the end in case it's one of
     * the two source regions, then mark the "new" region empty, allocating
     * another array of rectangles for it to use.
     */
    r1 = reg1->rects;
    r2 = reg2->rects;
    r1End = r1 + reg1->numRects;
    r2End = r2 + reg2->numRects;
    
    oldRects = rects;
    
    EMPTY_REGION(this);

    /*
     * Allocate a reasonable number of rectangles for the new region. The idea
     * is to allocate enough so the individual functions don't need to
     * reallocate and copy the array, which is time consuming, yet we don't
     * have to worry about using too much memory. I hope to be able to
     * nuke the Xrealloc() at the end of this function eventually.
     */
    size = MAX (reg1->numRects, reg2->numRects) * 2;
    rects = new RegionBox[size];
    
    /*
     * Initialize ybot and ytop.
     * In the upcoming loop, ybot and ytop serve different functions depending
     * on whether the band being handled is an overlapping or non-overlapping
     * band.
     * 	In the case of a non-overlapping band (only one of the regions
     * has points in the band), ybot is the bottom of the most recent
     * intersection and thus clips the top of the rectangles in that band.
     * ytop is the top of the next intersection between the two regions and
     * serves to clip the bottom of the rectangles in the current band.
     *	For an overlapping band (where the two regions intersect), ytop clips
     * the top of the rectangles of both regions and ybot clips the bottoms.
     */
    if (reg1->extents.y1 < reg2->extents.y1)
      ybot = reg1->extents.y1;
    else
      ybot = reg2->extents.y1;
    
    /*
     * prevBand serves to mark the start of the previous band so rectangles
     * can be coalesced into larger rectangles. qv. miCoalesce, above.
     * In the beginning, there is no previous band, so prevBand == curBand
     * (curBand is set later on, of course, but the first band will always
     * start at index 0). prevBand and curBand must be indices because of
     * the possible expansion, and resultant moving, of the new region's
     * array of rectangles.
     */
    prevBand = 0;
    
    do
      {
	curBand = numRects;

	/*
	 * This algorithm proceeds one source-band (as opposed to a
	 * destination band, which is determined by where the two regions
	 * intersect) at a time. r1BandEnd and r2BandEnd serve to mark the
	 * rectangle after the last one in the current band for their
	 * respective regions.
	 */
	r1BandEnd = r1;
	while ((r1BandEnd != r1End) && (r1BandEnd->y1 == r1->y1))
	  {
	    r1BandEnd++;
	  }
	
	r2BandEnd = r2;
	while ((r2BandEnd != r2End) && (r2BandEnd->y1 == r2->y1))
	  {
	    r2BandEnd++;
	  }
	
	/*
	 * First handle the band that doesn't intersect, if any.
	 *
	 * Note that attention is restricted to one band in the
	 * non-intersecting region at once, so if a region has n
	 * bands between the current position and the next place it overlaps
	 * the other, this entire loop will be passed through n times.
	 */
	if (r1->y1 < r2->y1)
	  {
	    top = MAX (r1->y1,ybot);
	    bot = MIN (r1->y2,r2->y1);

	    if ((top != bot) && (nonOverlap1Fn != 0))
	      {
		(this->*nonOverlap1Fn) (r1, r1BandEnd, top, bot);
	      }

	    ytop = r2->y1;
	  }
	else if (r2->y1 < r1->y1)
	  {
	    top = MAX (r2->y1,ybot);
	    bot = MIN (r2->y2,r1->y1);

	    if ((top != bot) && (nonOverlap2Fn != 0))
	      {
		(this->*nonOverlap2Fn) (r2, r2BandEnd, top, bot);
	      }

	    ytop = r1->y1;
	  }
	else
	  {
	    ytop = r1->y1;
	  }

	/*
	 * If any rectangles got added to the region, try and coalesce them
	 * with rectangles from the previous band. Note we could just do
	 * this test in miCoalesce, but some machines incur a not
	 * inconsiderable cost for function calls, so...
	 */
	if (numRects != curBand)
	  {
	    prevBand = miCoalesce (prevBand, curBand);
	  }

	/*
	 * Now see if we've hit an intersecting band. The two bands only
	 * intersect if ybot > ytop
	 */
	ybot = MIN (r1->y2, r2->y2);
	curBand = numRects;
	if (ybot > ytop)
	  {
	    (this->*overlapFn) (r1, r1BandEnd, r2, r2BandEnd, ytop, ybot);

	  }
	
	if (numRects != curBand)
	  {
	    prevBand = miCoalesce (prevBand, curBand);
	  }

	/*
	 * If we've finished with a band (y2 == ybot) we skip forward
	 * in the region to the next band.
	 */
	if (r1->y2 == ybot)
	  {
	    r1 = r1BandEnd;
	  }
	if (r2->y2 == ybot)
	  {
	    r2 = r2BandEnd;
	  }
      } while ((r1 != r1End) && (r2 != r2End));

    /*
     * Deal with whichever region still has rectangles left.
     */
    curBand = numRects;
    if (r1 != r1End)
      {
	if (nonOverlap1Fn != (nonOverlapFunc )NULL)
	  {
	    do
	      {
		r1BandEnd = r1;
		while ((r1BandEnd < r1End) && (r1BandEnd->y1 == r1->y1))
		  {
		    r1BandEnd++;
		  }
		(this->*nonOverlap1Fn) (r1, r1BandEnd,
				     MAX (r1->y1,ybot), r1->y2);
		r1 = r1BandEnd;
	      } while (r1 != r1End);
	  }
      }
    else if ((r2 != r2End) && (nonOverlap2Fn != (nonOverlapFunc) NULL))
      {
	do
	  {
	    r2BandEnd = r2;
	    while ((r2BandEnd < r2End) && (r2BandEnd->y1 == r2->y1))
	      {
		r2BandEnd++;
	      }
	    (this->*nonOverlap2Fn) (r2, r2BandEnd,
			       MAX (r2->y1,ybot), r2->y2);
	    r2 = r2BandEnd;
	  } while (r2 != r2End);
      }

    if (numRects != curBand)
    {
      (void) miCoalesce (prevBand, curBand);
    }

    /*
     * A bit of cleanup. To keep regions from growing without bound,
     * we shrink the array of rectangles to match the new number of
     * rectangles in the region. This never goes to 0, however...
     *
     * Only do this stuff if the number of rectangles allocated is more than
     * twice the number of rectangles in the region (a simple optimization...).
     */
    if (numRects < (size >> 1))
      {
	if (REGION_NOT_EMPTY (this))
	  {
	    size = numRects;
	    RegionBox *tmp = new RegionBox[size];
	    memcpy(tmp, rects, size * sizeof(RegionBox));
	    delete [] rects;
	    rects = tmp;
	  }
	else
	  {
	    /*
	     * No point in doing the extra work involved in an Xrealloc if
	     * the region is empty
	     */
	    size = 1;
	    delete [] rects;
	    rects = new RegionBox[1];
	  }
      }
    delete [] oldRects;
}


/*======================================================================
 *	    Region Union
 *====================================================================*/

/*-
 *-----------------------------------------------------------------------
 * miUnionNonO --
 *	Handle a non-overlapping band for the union operation. Just
 *	Adds the rectangles into the region. Doesn't have to check for
 *	subsumption or anything.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	pReg->numRects is incremented and the final rectangles overwritten
 *	with the rectangles we're passed.
 *
 *-----------------------------------------------------------------------
 */
void
Region::miUnionNonO (
	     RegionBox *r,
	     RegionBox *rEnd,
	     int          y1,
	     int          y2)
{
  RegionBox *pNextRect;

  pNextRect = &rects[numRects];

  assert(y1 < y2);

  while (r != rEnd)
    {
      assert(r->x1 < r->x2);
      MEMCHECK(this, pNextRect, rects);
      pNextRect->x1 = r->x1;
      pNextRect->y1 = y1;
      pNextRect->x2 = r->x2;
      pNextRect->y2 = y2;
      numRects += 1;
      pNextRect++;

      assert(numRects<=size);
      r++;
    }
}


/*-
 *-----------------------------------------------------------------------
 * miUnionO --
 *	Handle an overlapping band for the union operation. Picks the
 *	left-most rectangle each time and merges it into the region.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Rectangles are overwritten in pReg->rects and pReg->numRects will
 *	be changed.
 *
 *-----------------------------------------------------------------------
 */

/* static void*/
void
Region::miUnionO (
	  RegionBox *r1,
	  RegionBox *r1End,
	  RegionBox *r2,
	  RegionBox *r2End,
	  int          y1,
	  int          y2)
{
  RegionBox *	pNextRect;
    
  pNextRect = &rects[numRects];

#define MERGERECT(r) 					\
    if ((numRects != 0) &&  			\
	(pNextRect[-1].y1 == y1) &&  			\
	(pNextRect[-1].y2 == y2) &&  			\
	(pNextRect[-1].x2 >= r->x1))  			\
      {  						\
	if (pNextRect[-1].x2 < r->x2)  			\
	  {  						\
	    pNextRect[-1].x2 = r->x2;  			\
	    assert(pNextRect[-1].x1<pNextRect[-1].x2); 	\
	  }  						\
      }  						\
    else  						\
      {  						\
	MEMCHECK(this, pNextRect, rects); 	\
	pNextRect->y1 = y1;  				\
	pNextRect->y2 = y2;  				\
	pNextRect->x1 = r->x1;  			\
	pNextRect->x2 = r->x2;  			\
	numRects += 1;  				\
        pNextRect += 1;  				\
      }  						\
    assert(numRects<=size);			\
    r++;
    
    assert (y1<y2);
    while ((r1 != r1End) && (r2 != r2End))
    {
	if (r1->x1 < r2->x1)
	{
	    MERGERECT(r1);
	}
	else
	{
	    MERGERECT(r2);
	}
    }
    
    if (r1 != r1End)
    {
	do
	{
	    MERGERECT(r1);
	} while (r1 != r1End);
    }
    else while (r2 != r2End)
    {
	MERGERECT(r2);
    }
}

/**
 * gdk_region_union:
 * @source1:  a #GdkRegion
 * @source2: a #GdkRegion 
 * 
 * Sets the area of @source1 to the union of the areas of @source1 and
 * @source2. The resulting area is the set of pixels contained in
 * either @source1 or @source2.
 **/
Region&
Region::operator|=(const Region& other)
{
  /*  checks all the simple cases */

    /*
     * region and other are the same or other is empty
     */
  if ((this == &other) || (!(other.numRects)))
    return *this;

    /* 
     * region is empty
     */
  if (!numRects)
    {
      operator=(other);
      return *this;
    }

  /*
     * region completely subsumes other
     */
  if ((numRects == 1) && 
      (extents.x1 <= other.extents.x1) &&
      (extents.y1 <= other.extents.y1) &&
      (extents.x2 >= other.extents.x2) &&
      (extents.y2 >= other.extents.y2))
    return *this;

  /*
     * other completely subsumes region
     */
  if ((other.numRects == 1) && 
      (other.extents.x1 <= extents.x1) &&
      (other.extents.y1 <= extents.y1) &&
      (other.extents.x2 >= extents.x2) &&
      (other.extents.y2 >= extents.y2))
    {
      operator=(other);
      return *this;
    }

  miRegionOp (this, &other, &Region::miUnionO, 
	      &Region::miUnionNonO, &Region::miUnionNonO);

  extents.x1 = MIN (extents.x1, other.extents.x1);
  extents.y1 = MIN (extents.y1, other.extents.y1);
  extents.x2 = MAX (extents.x2, other.extents.x2);
  extents.y2 = MAX (extents.y2, other.extents.y2);

  return *this;
}


/*======================================================================
 * 	    	  Region Subtraction
 *====================================================================*/

/*-
 *-----------------------------------------------------------------------
 * miSubtractNonO --
 *	Deal with non-overlapping band for subtraction. Any parts from
 *	region 2 we discard. Anything from region 1 we add to the region.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	pReg may be affected.
 *
 *-----------------------------------------------------------------------
 */
/* static void*/
void
Region::miSubtractNonO1 (
		 RegionBox *r,
		 RegionBox *rEnd,
		 int          y1,
		 int          y2)
{
  RegionBox *	pNextRect;
	
  pNextRect = &rects[numRects];
	
  assert(y1<y2);

  while (r != rEnd)
    {
      assert (r->x1<r->x2);
      MEMCHECK (this, pNextRect, rects);
      pNextRect->x1 = r->x1;
      pNextRect->y1 = y1;
      pNextRect->x2 = r->x2;
      pNextRect->y2 = y2;
      numRects += 1;
      pNextRect++;

      assert (numRects <= size);

      r++;
    }
}

/*-
 *-----------------------------------------------------------------------
 * miSubtractO --
 *	Overlapping band subtraction. x1 is the left-most point not yet
 *	checked.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	pReg may have rectangles added to it.
 *
 *-----------------------------------------------------------------------
 */
/* static void*/
void
Region::miSubtractO(
	     RegionBox *r1,
	     RegionBox *r1End,
	     RegionBox *r2,
	     RegionBox *r2End,
	     int          y1,
	     int          y2)
{
  RegionBox *	pNextRect;
  int  	x1;
    
  x1 = r1->x1;
    
  assert(y1<y2);
  pNextRect = &rects[numRects];

  while ((r1 != r1End) && (r2 != r2End))
    {
      if (r2->x2 <= x1)
	{
	  /*
	   * Subtrahend missed the boat: go to next subtrahend.
	   */
	  r2++;
	}
      else if (r2->x1 <= x1)
	{
	  /*
	   * Subtrahend preceeds minuend: nuke left edge of minuend.
	   */
	  x1 = r2->x2;
	  if (x1 >= r1->x2)
	    {
	      /*
	       * Minuend completely covered: advance to next minuend and
	       * reset left fence to edge of new minuend.
	       */
	      r1++;
	      if (r1 != r1End)
		x1 = r1->x1;
	    }
	  else
	    {
	      /*
	       * Subtrahend now used up since it doesn't extend beyond
	       * minuend
	       */
	      r2++;
	    }
	}
      else if (r2->x1 < r1->x2)
	{
	  /*
	   * Left part of subtrahend covers part of minuend: add uncovered
	   * part of minuend to region and skip to next subtrahend.
	   */
	  assert(x1<r2->x1);
	  MEMCHECK(this, pNextRect, rects);
	  pNextRect->x1 = x1;
	  pNextRect->y1 = y1;
	  pNextRect->x2 = r2->x1;
	  pNextRect->y2 = y2;
	  numRects += 1;
	  pNextRect++;

	  assert(numRects<=size);

	  x1 = r2->x2;
	  if (x1 >= r1->x2)
	    {
	      /*
	       * Minuend used up: advance to new...
	       */
	      r1++;
	      if (r1 != r1End)
		x1 = r1->x1;
	    }
	  else
	    {
	      /*
	       * Subtrahend used up
	       */
	      r2++;
	    }
	}
      else
	{
	  /*
	   * Minuend used up: add any remaining piece before advancing.
	   */
	  if (r1->x2 > x1)
	    {
	      MEMCHECK(this, pNextRect, rects);
	      pNextRect->x1 = x1;
	      pNextRect->y1 = y1;
	      pNextRect->x2 = r1->x2;
	      pNextRect->y2 = y2;
	      numRects += 1;
	      pNextRect++;
	      assert(numRects<=size);
	    }
	  r1++;
	  if (r1 != r1End)
	    x1 = r1->x1;
	}
    }

  /*
     * Add remaining minuend rectangles to region.
     */
  while (r1 != r1End)
    {
      assert(x1<r1->x2);
      MEMCHECK(this, pNextRect, rects);
      pNextRect->x1 = x1;
      pNextRect->y1 = y1;
      pNextRect->x2 = r1->x2;
      pNextRect->y2 = y2;
      numRects += 1;
      pNextRect++;

      assert(numRects<=size);

      r1++;
      if (r1 != r1End)
	{
	  x1 = r1->x1;
	}
    }
}

/**
 * gdk_region_subtract:
 * @source1: a #GdkRegion
 * @source2: another #GdkRegion
 *
 * Subtracts the area of @source2 from the area @source1. The resulting
 * area is the set of pixels contained in @source1 but not in @source2.
 **/
Region&
Region::operator-=(const Region& other)
{
  /* check for trivial reject */
  if ((!(numRects)) || (!(other.numRects)) ||
      (!EXTENTCHECK(&extents, &other.extents)))
    return *this;
 
  miRegionOp (this, &other, &Region::miSubtractO,
	      &Region::miSubtractNonO1, (nonOverlapFunc) NULL);

  /*
   * Can't alter region's extents before we call miRegionOp because miRegionOp
   * depends on the extents of those regions being the unaltered. Besides, this
   * way there's no checking against rectangles that will be nuked
   * due to coalescing, so we have to examine fewer rectangles.
   */
  miSetExtents ();

  return *this;
}

/**
 * gdk_region_xor:
 * @source1: a #GdkRegion
 * @source2: another #GdkRegion
 *
 * Sets the area of @source1 to the exclusive-OR of the areas of @source1
 * and @source2. The resulting area is the set of pixels contained in one
 * or the other of the two sources but not in both.
 **/
Region&
Region::operator^=(const Region& other)
{
  Region temp(other - *this);

  *this -= other;

  return *this |= temp;
}

/*
 *	Check to see if two regions are equal	
 */
bool
Region::operator==(const Region& r2) const
{
  if (numRects != r2.numRects) return false;
  else if (numRects == 0) return true;
  else if (extents.x1 != r2.extents.x1) return false;
  else if (extents.x2 != r2.extents.x2) return false;
  else if (extents.y1 != r2.extents.y1) return false;
  else if (extents.y2 != r2.extents.y2) return false;
  else
    for(int i=0; i < numRects; i++ )
      {
	if (rects[i].x1 != r2.rects[i].x1) return false;
	else if (rects[i].x2 != r2.rects[i].x2) return false;
	else if (rects[i].y1 != r2.rects[i].y1) return false;
	else if (rects[i].y2 != r2.rects[i].y2) return false;
      }
  return true;
}

// sort function for std::map

bool
Region::operator<(const Region& r2) const
{
  if (numRects != r2.numRects) return numRects < r2.numRects;
  else if (numRects == 0) return false;
  else if (extents.x1 != r2.extents.x1) return extents.x1 < r2.extents.x1;
  else if (extents.x2 != r2.extents.x2) return extents.x2 < r2.extents.x2;
  else if (extents.y1 != r2.extents.y1) return extents.y1 < r2.extents.y1;
  else if (extents.y2 != r2.extents.y2) return extents.y2 < r2.extents.y2;
  else
    for(int i=0; i < numRects; i++ )
      {
	if (rects[i].x1 != r2.rects[i].x1) return rects[i].x1 < r2.rects[i].x1;
	else if (rects[i].x2 != r2.rects[i].x2) return rects[i].x2 < r2.rects[i].x2;
	else if (rects[i].y1 != r2.rects[i].y1) return rects[i].y1 < r2.rects[i].y1;
	else if (rects[i].y2 != r2.rects[i].y2) return rects[i].y2 < r2.rects[i].y2;
      }
  return false;
}

bool
Region::contains(const Point& p) const
{
  int i;

  if (numRects == 0)
    return false;
  if (!INBOX(extents, p.x, p.y))
    return false;
  for (i=0; i<numRects; i++)
    {
      if (INBOX (rects[i], p.x, p.y))
	return true;
    }
  return false;
}

Region::OverlapType
Region::overlap (const SDL_Rect& rectangle) const
{
  RegionBox *pbox;
  RegionBox *pboxEnd;
  RegionBox  rect;
  RegionBox *prect = &rect;
  bool      partIn, partOut;
  int rx, ry;

  rx = rectangle.x;
  ry = rectangle.y;
  
  prect->x1 = rx;
  prect->y1 = ry;
  prect->x2 = rx + rectangle.w;
  prect->y2 = ry + rectangle.h;
    
    /* this is (just) a useful optimization */
  if ((numRects == 0) || !EXTENTCHECK (&extents, prect))
    return OVERLAP_OUT;

  partOut = false;
  partIn = false;

    /* can stop when both partOut and partIn are TRUE, or we reach prect->y2 */
  for (pbox = rects, pboxEnd = pbox + numRects;
       pbox < pboxEnd;
       pbox++)
    {

      if (pbox->y2 <= ry)
	continue;	/* getting up to speed or skipping remainder of band */

      if (pbox->y1 > ry)
	{
	  partOut = true;	/* missed part of rectangle above */
	  if (partIn || (pbox->y1 >= prect->y2))
	    break;
	  ry = pbox->y1;	/* x guaranteed to be == prect->x1 */
	}

      if (pbox->x2 <= rx)
	continue;		/* not far enough over yet */

      if (pbox->x1 > rx)
	{
	  partOut = true;	/* missed part of rectangle to left */
	  if (partIn)
	    break;
	}

      if (pbox->x1 < prect->x2)
	{
	  partIn = true;	/* definitely overlap */
	  if (partOut)
	    break;
	}

      if (pbox->x2 >= prect->x2)
	{
	  ry = pbox->y2;	/* finished with this band */
	  if (ry >= prect->y2)
	    break;
	  rx = prect->x1;	/* reset x out to left again */
	}
      else
	{
	  /*
	   * Because boxes in a band are maximal width, if the first box
	   * to overlap the rectangle doesn't completely cover it in that
	   * band, the rectangle must be partially out, since some of it
	   * will be uncovered in that band. partIn will have been set true
	   * by now...
	   */
	  break;
	}

    }

  return (partIn ? ((ry < prect->y2) ? OVERLAP_PART : OVERLAP_IN) : OVERLAP_OUT);
}

Region
Region::boundary(bool inside) const
{
  if(inside) {
    // the region lies wholy in the interior of 'container'
    Region container(wftk::Rect(extents.x1 - 1, extents.y1 - 1,
	2 + extents.x2 - extents.x1, 2 + extents.y2 - extents.y1));
    return (container - *this).boundary(false) & container;
  }

  Region out;

  // We'll use a 'tmp' region to add rectangles, instead of allocating a new
  // region every time, with the corresponding memory allocation overhead.
  // We play some games with 'tmp', because we're passing it in const, to
  // avoid calling 'new' and 'delete'. There will still be memory allocation
  // internally in 'out', but the region code is already set up to make
  // that efficient.

  // tmp_space is properly stack aligned to hold a Region,
  // this is a trick to avoid calling the Region constructor and
  // destructor
  union AlignedType {
    long l;
    RegionBox b;
    RegionBox* p;
  };
  const unsigned altype_size = sizeof(AlignedType);
  AlignedType tmp_space[(sizeof(Region) + altype_size - 1)/altype_size];
  Region& tmp = *((Region*) &tmp_space);
  tmp.size = tmp.numRects = 1;
  tmp.rects = &tmp.extents;

  for(RegionBox* rect = rects; rect < rects + numRects; ++rect) {
    // top and bottom
    tmp.extents.x1 = rect->x1 - 1;
    tmp.extents.x2 = rect->x2 + 1;
    tmp.extents.y1 = rect->y1 - 1;
    tmp.extents.y2 = rect->y1;
    out |= tmp;
    tmp.extents.y1 = rect->y2;
    tmp.extents.y2 = rect->y2 + 1;
    out |= tmp;

    // left and right
    tmp.extents.y1 = rect->y1 - 1;
    tmp.extents.y2 = rect->y2 + 1;
    tmp.extents.x1 = rect->x1 - 1;
    tmp.extents.x2 = rect->x1;
    out |= tmp;
    tmp.extents.x1 = rect->x2;
    tmp.extents.x2 = rect->x2 + 1;
    out |= tmp;
  }

  out -= *this;
  return out;
}

Region::const_iterator::const_iterator(const RegionBox* begin, const RegionBox* end) :
	box_(begin), final_(end)
{
  if(box_ != final_) {
    p_.x = box_->x1;
    p_.y = box_->y1;
  }
}

Region::const_iterator&
Region::const_iterator::operator++()
{
  if(box_ == final_)
    return *this;

  if(++p_.x == box_->x2) {
    if(++p_.y == box_->y2) {
      if(++box_ != final_) {
        p_.x = box_->x1;
        p_.y = box_->y1;
      }
    }
    else
      p_.x = box_->x1;
  }

  return *this;
}
