/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "rect.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "point.h"

#include <iostream>
#include <assert.h>

namespace wftk {

Rect Rect::invalid = Rect(0,0,-1,-1);
Rect Rect::empty = Rect(0,0,0,0);

Rect::Rect( int x_, int y_, int w_, int h_) :
  valid_(w_ >= 0 && h_ >= 0)
{
  if(valid_) {
    x = x_;
    y = y_;
    w = w_;
    h = h_;
  }
}

/*
Rect::Rect(const Point &upperLeft, const Point &lowerRight) :
  upperLeft_(upperLeft),
  lowerRight_(lowerRight)
{
}
*/

bool
Rect::contains(const Point &p) const
{
  return p.y >= y && p.y < y + h && p.x >= x && p.x < x + w;

}

bool
Rect::contains(const Rect &r) const 
{
  return x <= r.x && x + w >= r.x + r.w && y <= r.y && y + h >= r.y + r.h;
}

void
Rect::translate(int dx, int dy)
{
  x += dx;
  y += dy;
}

void 
Rect::warp(const Point &newOrigin)
{
  x = newOrigin.x;
  y = newOrigin.y;
}

void
Rect::resize(int w_, int h_)
{
  valid_ = (w_ >= 0 && h_ >= 0);

  if(valid_) {
    w = w_;
    h = h_;
  }
}

Rect
Rect::unite(const Rect &r) const
{
  if(!valid_ || !r.valid_)
    return invalid;

  int min_x = (x < r.x) ? x : r.x;
  int min_y = (y < r.y) ? y : r.y;
  int max_x = (x + w > r.x + r.w) ? x + w : r.x + r.w;
  int max_y = (y + h > r.y + r.h) ? y + h : r.y + r.h;

  assert(min_x <= max_x);
  assert(min_y <= max_y);

  return Rect(min_x, min_y, max_x - min_x, max_y - min_y);
}

Rect 
Rect::intersect(const Rect &r) const
{
  if(!valid_ || !r.valid_)
    return invalid;

  int min_x = (x > r.x) ? x : r.x;
  int min_y = (y > r.y) ? y : r.y;
  int max_x = (x + w < r.x + r.w) ? x + w : r.x + r.w;
  int max_y = (y + h < r.y + r.h) ? y + h : r.y + r.h;

  if(min_x > max_x || min_y > max_y)
    return invalid;

  return Rect(min_x, min_y, max_x - min_x, max_y - min_y);
}

std::ostream&
operator<<(std::ostream& s, const Point& p)
{
  s << "(" << p.x << ":" << p.y << ")";
  return s;
}  

std::ostream&
operator<<(std::ostream& s, const Rect& p)
{
    s << "(" << p.x << "," << p.y << "," << p.w << "," << p.h << ")";
    return s;
}

} // namespace wftk
