
/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// originally written by Karsten Laux, June 1999  

#include "painter.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "sge_blib.h"

//disable debugging infos for this module
#include "debug.h"

#include <math.h>
#include <cassert>

namespace wftk {

#ifndef M_PI
  #define M_PI 3.14159265359
#endif

#ifndef SWAP
#define SWAP(a,b,h) { h = a; a = b; b = h; }
#endif

Painter::Painter(Surface* surf):
  surface_(surf), 
  color_("white"), 
  fill_(false),
  mode_(OP_REPLACE)
{
}

void 
Surface::setPixel(const Point& p, const Color& col)
{
  if(!sdlSurface_ || !rect().contains(p))
    return;
  
  Uint32 pixel = pixelformat().mapToPixel(col);

  Uint32 pixeladdr = pixelformat().bpp() * p.x + pitch() * p.y;
  
  lock();
  
  writePixel(pixeladdr, pixel);

  unlock();
}
  
 
Color
Surface::getPixel(const Point& p) const
{
  if(!sdlSurface_ || !rect().contains(p))
    return Color();

  Uint32 pixeladdr = pixelformat().bpp() * p.x + pitch() * p.y;

  //lock surface
  lock();

  Uint32 pixel = readPixel(pixeladdr);

  unlock();

  return pixelformat().mapToColor(pixel);
}

void 
Painter::hLine(const Point& start_, const Point& end_ , const Color& col)
{
  if(!surface_ || surface_->empty())
    return;
  
  if(start_.y < 0 || (unsigned) start_.y >= surface_->height())
    return;

  Point start,end;

  start.y = start_.y;
  
  start.x = (start_.x > 0) ? start_.x : 0;
  start.x = ((unsigned) start.x < surface_->width()) ? start.x : surface_->width()-1;

  end.x = (end_.x > 0) ? end_.x : 0;
  end.x = ((unsigned) end.x < surface_->width()) ? end.x : surface_->width()-1;
 
 
  int x1, x2;
  int y;

  Uint32 pixel = surface_->pixelformat().mapToPixel(col);

  Debug::channel(Debug::DRAWING) <<"Painter: " <<col<< " using bitpattern "
	<< (void*) pixel << Debug::endl; // so it prints in hex

  if(start.x > end.x)
    {
      x1 = end.x;
      x2 = start.x;
    }
  else
    {
      x1 = start.x;
      x2 = end.x;
    }

  y = start.y;

  
  surface_->lock();

  Uint32 pixeladdr =
    surface_->pixelformat().bpp() * x1 + surface_->pitch() * y;
  Uint16 skip = surface_->pixelformat().bpp();

   
  while(x1<=x2)
    {     
      writePixel(pixeladdr, pixel);

      x1++;
      pixeladdr += skip;
    }

  surface_->unlock();

  
  
}
   
void 
Painter::vLine(const Point& start_ , const Point& end_ , const Color& col)
{
  
  if(!surface_ || surface_->empty())
    return;

  if(start_.x < 0 || (unsigned) start_.x >= surface_->width())
    return;

  Point start, end;

  start.x = start_.x;

  start.y = (start_.y > 0) ? start_.y : 0;
  start.y = ((unsigned) start.y < surface_->height()) ? start.y : surface_->height()-1;

  end.y = (end_.y > 0) ? end_.y : 0;
  end.y = ((unsigned) end.y < surface_->height()) ? end.y : surface_->height()-1;
   
 
  int x;
  int y1, y2;

  Uint32 pixel = surface_->pixelformat().mapToPixel(col);

  Debug::channel(Debug::DRAWING) <<"Painter: " <<col<< " using bitpattern "
	<< (void*) pixel << Debug::endl; // so it prints in hex

  if(start.y > end.y)
    {
      y1 = end.y;
      y2 = start.y;
    }
  else
    {
      y1 = start.y;
      y2 = end.y;
    }

  x = start.x;

  Uint32 pixeladdr =
    surface_->pixelformat().bpp() * x + surface_->pitch() * y1;

  Uint16 skip = surface_->pitch();

  surface_->lock();
  while(y1<=y2)
    {
      writePixel(pixeladdr, pixel);
      
      y1++;
      pixeladdr += skip;
    }

  surface_->unlock();


}
  
void 
Painter::hLine(const Point& p1, const Point& p2)
{
  hLine(p1,p2,color_);
}

  
void 
Painter::vLine(const Point& p1, const Point& p2)
{
  vLine(p1,p2,color_);
}
  

void 
Painter::circle(const Point& c, unsigned int r, unsigned int t)
{
  if(fill_)
    ellipseFill(c,r,r);
  else
    ellipse(c,r,r,t);
}

void 
Painter::ellipse(const Point& c, 
		  unsigned int rx, unsigned int ry, unsigned int t)
{
  if(!surface_ || surface_->empty())
    return;
  if(rx == 0 || ry == 0)
    return;

  if(fill_)
    {
      ellipseFill(c,rx,ry);
      return;
    }
  
   int steps = 3 * (rx > ry) ? rx : ry;  
  
  float alpha = 0;
  float delta = M_PI/2/steps;

  Point p;
  Point v;
  Uint32 pixeladdr;

  Uint32 pixel = surface_->pixelformat().mapToPixel(color_);

  surface_->lock();
 
  while(steps > 0)
    {
      v.x = (int)(rx * cos(alpha));
      v.y = (int)(ry * sin(alpha));
      
      p.x = c.x + v.x;
      p.y = c.y + v.y;

      if(Rect(0,0, surface_->width(), surface_->height()).contains(p))
        {
          pixeladdr=surface_->pixelformat().bpp()*p.x +surface_->pitch()*p.y;
          for(unsigned int i=0; i < t; i++)
            {  
              writePixel(pixeladdr, pixel);
              pixeladdr++;
            }
        }

      v.x = -v.x;
      p.x = c.x + v.x;

      if(Rect(0,0, surface_->width(), surface_->height()).contains(p))
        {
          pixeladdr=surface_->pixelformat().bpp()*p.x +surface_->pitch()*p.y;
          for(unsigned int i=0; i < t; i++)
            {  
              writePixel(pixeladdr, pixel);
              pixeladdr++;
            }
        }

      v.y = -v.y;
      p.y = c.y + v.y;
      if(Rect(0,0, surface_->width(), surface_->height()).contains(p))
        {
          pixeladdr=surface_->pixelformat().bpp()*p.x +surface_->pitch()*p.y;
          for(unsigned int i=0; i < t; i++)
            {  
              writePixel(pixeladdr, pixel);
              pixeladdr++;
            }
        }

      v.x = -v.x;
      p.x = c.x + v.x;
      if(Rect(0,0, surface_->width(), surface_->height()).contains(p))
        {
          pixeladdr=surface_->pixelformat().bpp()*p.x +surface_->pitch()*p.y;
          for(unsigned int i=0; i < t; i++)
            {  
              writePixel(pixeladdr, pixel);
              pixeladdr++;
            }
        }

      alpha = alpha + delta;
      steps--;
    }


  surface_->unlock();
}


void 
Painter::ellipseFill(const Point& c, 
		      unsigned int rx, unsigned int ry)
{
  if(!surface_ || surface_->empty())
    return;
  if(rx == 0 || ry == 0)
    return;
  unsigned int steps = 0;  
 
  Point p1;
  Point p2;
  Point v;
 
  while(steps < ry)
    {
      
      v.x = (int)(rx * cos(asin((double)steps/ry)));
     
      p1.x = c.x - v.x;
      p1.y = c.y - steps;

      p2.x = c.x + v.x;
      p2.y = c.y - steps;

      hLine(p1,p2,color_);

 
      p1.x = c.x - v.x;
      p1.y = c.y + steps;

      p2.x = c.x + v.x;
      p2.y = c.y + steps;

      hLine(p1,p2,color_);

      steps++;
    }

}


void 
Painter::line(const Point& start, const Point& end)
{
  line(start, end, color_);
}

void 
Painter::line(const Point& start , const Point& end , const Color& col)
{
  
  if(!surface_ || surface_->empty())
    return;
  
  if(!Rect(0,0,surface_->width(), surface_->height()).contains(start))
    return;
  if(!Rect(0,0,surface_->width(), surface_->height()).contains(end))
    return;

  Sint32 dx, dy, sdx, sdy, x, y, px, py;

  dx = end.x - start.x;
  dy = end.y - start.y;

  //check if we may optimize to either vLine or hLine ...
  if(dx == 0)
    {
      //this is a vertical line, call vLine and return
      vLine(start, end, col);
      return;
    }
  if(dy == 0)
    {
      //this is a horizontal line, call hLine and return
      hLine(start, end, col);
      return;
    }

  //here comes the hard job ... bresenham:

  sdx = (dx < 0) ? -1 : 1;
  sdy = (dy < 0) ? -1 : 1;

  dx = sdx * dx + 1;
  dy = sdy * dy + 1;

  x = y = 0;

  px = start.x;
  py = start.y;

  //map the color to the appropriate bit pattern to write ...
  Uint32 pixel = surface_->pixelformat().mapToPixel(col);
  Uint32 pixeladdr; 
  
  surface_->lock();

  //FIXME the calculation of the pixeladdr could be reduced to addings ...
  //mo multiplies are needed in fact

  if (dx >= dy)
    {
      for (x = 0; x < dx; x++)
	{
	  pixeladdr = surface_->pixelformat().bpp()*px + surface_->pitch()*py;
	  writePixel(pixeladdr, pixel);

	  y += dy;
	  if (y >= dx)
	    {
	      y -= dx;
	      py += sdy;
	    }
	  px += sdx;
	}
    }
  else
    {
      for (y = 0; y < dy; y++)
	{
	  pixeladdr = surface_->pixelformat().bpp()*px + surface_->pitch()*py;
	  writePixel(pixeladdr, pixel);

	  x += dx;
	  if (x >= dy)
	    {
	      x -= dy;
	      px += sdx;
	    }
	  py += sdy;
	}
    }


  surface_->unlock();


}

void 
Painter::box(const Point& upperleft , const Point& lowerright)
{
  
  if(!surface_ || surface_->empty())
    return;
  
  Point p1, p2, p3, p4;

  if(fill_)
    {
      //call the fast SDL filler
      surface_->fill(Rect(), color_);
    }
  else
    {
      //sort the points (turns even weird boxes into real boxes ...
      
      //first get upperleft and lowerrigt edge right
      if(upperleft.x < lowerright.x)
	{ 
	  p1.x = upperleft.x; 
	  p3.x = lowerright.x;
	}
      else
	{ 
	  p3.x = upperleft.x; 
	  p1.x = lowerright.x;
	} 
      if(upperleft.y < lowerright.y)
	{
	  p1.y = upperleft.y; 
	  p3.y = lowerright.y;
	}
      else
	{
	  p3.y = upperleft.y; 
	  p1.y = lowerright.y;
	}
      
      // the other edges
      p2 = Point(p3.x, p1.y);
      p4 = Point(p1.x, p3.y);

      //and now ! ... draw it.
      hLine(p1, p2);
      vLine(p2, p3);
      hLine(p3, p4);
      vLine(p4, p1);
      
	
    }

}

void
Painter::writePixel(Uint32 pixeladdr, Uint32 pixel)
{
  assert(surface_ && !surface_->empty());

  switch(mode_)
    {
    case OP_REPLACE:
      surface_->writePixel(pixeladdr, pixel);
      break;
    case OP_AND:
      pixel = pixel & surface_->readPixel(pixeladdr);
      surface_->writePixel(pixeladdr, pixel);
      break;
    case OP_OR:
      pixel = pixel | surface_->readPixel(pixeladdr);
      surface_->writePixel(pixeladdr, pixel);
      break;   
    case OP_XOR:
      pixel = pixel ^ surface_->readPixel(pixeladdr);
      surface_->writePixel(pixeladdr, pixel);
      break;
    }
}

 
  
void 
Painter::trigon(const Point& p1, const Point& p2, const Point& p3, const Color& c)
{
  if(!surface_ || surface_->empty())
    return;

  if(!fill_)
    {
      line(p1, p2, c);
      line(p2, p3, c);
      line(p3, p1, c);
    }
  else
    {
      Sint16 xl,xs,y,t;
      int y1 = p1.y;
      int y2 = p2.y;
      int y3 = p3.y;
      int x1 = p1.x;
      int x2 = p2.x;
      int x3 = p3.x;

      if ( y1 > y2 ) {
	SWAP(y1,y2,y);
	SWAP(x1,x2,y);
      }
      
      if ( y2 > y3 ) {
	SWAP(y2,y3,y);
	SWAP(x2,x3,y);
      }
      
      if ( y1 > y2 ) {
	SWAP(y1,y2,y);
	SWAP(x1,x2,y);
      }
      
      for ( y = y1; y <= y3; y++) {
	if ( y > y2 ) {
	  t = (y2-y3);
	  t = t?t:1;
                        xs = ((x2-x3)*(y-y2))/t+x2;
	}
	else {
	  t = (y1-y2);
	  t = t?t:1;
	  xs = ((x1-x2)*(y-y1))/t+x1;
	}
	t = (y1-y3);
	xl = ((x1-x3)*(y-y1))/((t==0)?1:t)+x1;
	
	hLine(Point(xl,y),Point(xs,y),c);
      }    
      
    }
}
  
  
void 
Painter::trigon(const Point& p1, const Point& p2, const Point& p3)
{
  trigon(p1, p2, p3, color_);
}

  
  
void 
Painter::shadedTrigon(const Point& d1, const Point& d2, const Point& d3,
		      const Color& c1, const Color& c2, const Color& c3)
{ 
  if(!surface_ || surface_->empty())
    return;

  Uint32 pixel1 = surface_->pixelformat().mapToPixel(c1);
  Uint32 pixel2 = surface_->pixelformat().mapToPixel(c2);
  Uint32 pixel3 = surface_->pixelformat().mapToPixel(c3);
  sge_FadedTrigon(surface_->sdlSurface_, d1.x, d1.y, d2.x, d2.y, d3.x, d3.y, 
		  pixel1, pixel2, pixel3);
}
  
void 
Painter::texturedTrigon(const Point& d1, const Point& d2, const Point& d3,
			const Point& s1, const Point& s2, const Point& s3, Surface* texture)
{
  if(!surface_ || surface_->empty())
    return;

  sge_TexturedTrigon(surface_->sdlSurface_, d1.x, d1.y, d2.x, d2.y, d3.x, d3.y, 
		     texture->sdlSurface_, s1.x, s1.y, s2.x, s2.y, s3.x, s3.y); 
}
   
  
} //namespace wftk
