namespace eval chmod {
    variable chmod

    # Define the chmod array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set chmod {
	list.reset ""
	list.clear ""
	entry.dir ""
	entry.regexp ""
	dialog ""
	dir ""
	make_parent ""
    }
}

# chmod::create --
#
#   Method to create the dialog box for the chmod command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chmod::create { } {
    global tkWorld
    variable chmod

    # Put the focus on the chmod dialog if it is already open.
    if [winfo exists $chmod(dialog)] {
	switch -- [wm state $chmod(dialog)] {
	    normal {
		raise $chmod(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $chmod(dialog)
	    }
	}
	focus $chmod(dialog)
	return
    } else {
	set chmod(dialog) [dialog::create .chmod Chmod]
    }

    # The first tab has the file information, along with the make
    # parent command line option.
    set tab1 [tabnotebook::page [dialog::interior \
	    $chmod(dialog)] "Options"]

    # Use a frame to encapsulate the dir options so that
    # the frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    button $f1.file_button \
	    -text "File" \
	    -width 5 \
	    -command chmod::open
    set chmod(entry.file) [entry $f1.file_entry \
	    -width 35 \
	    -textvariable chmod::chmod(file)]
    label $f1.dir_label \
	    -text "Directory" \
	    -width 8
    set chmod(entry.dir) [entry $f1.dir_entry \
	    -width 35 \
	    -textvariable chmod::chmod(dir)]
    checkbutton $f1.recursive \
	    -text "Recursive" \
	    -variable chmod::chmod(recursive) \
	    -onvalue "R" \
	    -offvalue ""
    grid $f1.file_button $f1.file_entry \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f1.dir_label $f1.dir_entry $f1.recursive \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Now build the individual checkbutton options available to
    # the user.
    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]

    set col 0
    foreach t "All User Group Other" {
	set x [string tolower $t]
	set cb($x) [checkbox::create $f2.$x "$t"]
	grid $f2.$x \
		-row 0 \
		-column $col \
		-sticky w \
		-padx 5 \
		-pady 0
	incr col
    }
    foreach {a b} {Read 4 Write 2 Execute 1} {
	set t [string tolower $a]
	checkbox::add $cb(all) $a $b "0" chmod::chmod(all_perm_$t)
	checkbox::add $cb(user) $a $b "0" chmod::chmod(user_perm_$t)
	checkbox::add $cb(group) $a $b "0" chmod::chmod(group_perm_$t)
	checkbox::add $cb(other) $a $b "0" chmod::chmod(other_perm_$t)
    }
    unset a b col cb x t

    # Build the first tab.
    pack $f1 $f2 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $chmod(entry.file)

    # Define the lists for the reset and clear methods
    set chmod(list.reset) "recursive all_perm_read all_perm_write \
	    all_perm_execute user_perm_read user_perm_write \
	    user_perm_execute group_perm_read group_perm_write \
	    group_perm_execute other_perm_read other_perm_write \
	    other_perm_execute"
    set chmod(list.clear) "file dir"
}

# chmod::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chmod::ok { } {
    global tkWorld
    variable chmod

    # Build the command line for the recursive argument.
    set cmd_arg ""
    if [string length $chmod(recursive)] {
	set cmd_arg "-$chmod(recursive) "
    }

    # Now define the permission stuff if they are turned on.
    foreach x "all user group other" {
	set total($x) [expr $chmod($x\_perm_read) + \
		$chmod($x\_perm_write) + \
		$chmod($x\_perm_execute)]
    }
    if {$total(all) > 0} {
	append cmd_arg "\"$total(all)$total(all)$total(all)\" "
    } elseif {$total(user) > 0 || $total(group) > 0 || $total(other) > 0} {
	append cmd_arg "\"$total(user)$total(group)$total(other)\" "
    } else {
	append cmd_arg {"000"}
    }
    unset total

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    $tkWorld(cmd_center) insert insert "chmod $cmd_arg "
    if [string length $chmod(dir)] {
	$tkWorld(cmd_center) insert insert "$chmod(dir) "
    }
    if [string length $chmod(file)] {
	$tkWorld(cmd_center) insert insert "$chmod(file) "
    }


    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# chmod::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chmod::reset { } {
    variable chmod

    # Allow for the permission elements to maintain an integer value.
    foreach x $chmod(list.reset) {
	switch -glob $x {
	    all_perm* -
	    user_perm* -
	    group_perm* -
	    other_perm* {
		set chmod($x) 0
	    }
	    default {
		set chmod($x) ""
	    }
	}
    }
}

# chmod::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chmod::clear { } {
    variable chmod

    # Reset the data structure elements and bg/fg.
    foreach x $chmod(list.clear) {
	set chmod($x) ""
	$chmod(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $chmod(entry.dir)
}

# chmod::help --
#
#   Method to invoke the Chmod Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chmod::help { } {
    global tkWorld

    help::create "help/chmod.html" "Chmod Command Help"
}

# chmod::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chmod::close { } {
    variable chmod
    
    balloonhelp::cancel
    destroy $chmod(dialog)
}

# chmod::open --
#
#   Method to add a file to the file entry and move the cursor
#   index of the entry to the end.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chmod::open { } {
    variable chmod

    # Insert the file list and move the cursor.
    $chmod(entry.file) insert insert [file::select]
    $chmod(entry.file) icursor end

    # Set the focus on the entry with the file list in it.
    focus $chmod(entry.file)
}
