#include <iostream>
#include <sstream>

#include "Tools.h"
#include "StringTools.h"
#include "Configuration.h"
#include "XMLTools.h"
#include "XMLParser.h"
#include "XMLException.h"
#include "SDLFrameRate.h"
#include "PlayerStatus.h"

#include "SoundInterface.h"

#include "ObjectRepository.h"
#include "AllObjects.h"

#include "ScoreTable.h"
#include "PlayerConfiguration.h"
#include "PlayGround.h"
#include "GameRecorder.h"
#include "GameControlBase.h"


//----------------------------------------------------------------------------
GameControlBase::Running GameControlBase::Running::sm_instance;
GameControlBase::UnloadingCrate GameControlBase::UnloadingCrate::sm_instance;
GameControlBase::UnloadingDelay GameControlBase::UnloadingDelay::sm_instance;
GameControlBase::OutOfFuelDelay GameControlBase::OutOfFuelDelay::sm_instance;
GameControlBase::ExplosionDelay GameControlBase::ExplosionDelay::sm_instance;
GameControlBase::AddFuelToScore GameControlBase::AddFuelToScore::sm_instance;
GameControlBase::AddBonusToScore GameControlBase::AddBonusToScore::sm_instance;
GameControlBase::Final GameControlBase::Final::sm_instance;


//----------------------------------------------------------------------------
void GameControlBase::GameStateBase::updateShipControl(ShipControl control,
                                                       GameControlBase *game)
{
    game->do_updateShipControl(control);
}

//----------------------------------------------------------------------------
void GameControlBase::GameStateBase::updateBonus(GameControlBase *game)
{
    game->do_updateBonus();
}

//----------------------------------------------------------------------------
void GameControlBase::GameStateBase::onShipOutOfFuel(const Ship *ship,
                                                     GameControlBase *game)
{
    assert(ship != NULL);

    if (ship == game->getPlayerShip())
    {
        SoundInterface::getInstance()->onShipOutOfFuel();

        game->initOutOfFuel();
        game->setState(OutOfFuelDelay::getInstance());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::GameStateBase::onObjectDestroyed(
    const ObjectBase *object, GameControlBase *game)
{
    assert(object != NULL);

    game->removeDestroyObjective(object->getId());
    game->removeReachedObjectives();
}

//----------------------------------------------------------------------------
void GameControlBase::GameStateBase::onPlayerShipDestroyed(
    Ship *ship, GameControlBase *game)
{
    while (ship->hasCrates())
    {
        ObjectRepository::getInstance()->markObjectToRemove(ship->popCrate());
    }
}


//----------------------------------------------------------------------------
void GameControlBase::Running::onPreUpdate(GameControlBase *game)
{
    Ship *ship = game->getPlayerShip();
    const Platform *landedPlatform = ship->getLandedPlatform();

    if (landedPlatform != NULL && landedPlatform->isFuelPlatform())
    {
        ship->incFuel(5, game->getPlayerFuel());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::Running::onShipLanded(
    Ship *ship, const Platform *platform, GameControlBase *game)
{
    assert(ship != NULL);
    assert(platform != NULL);

    // Keep sure, that the ship's thrusters are turned off,
    // before changing to a game state with disabled ship steering.
    ship->setThrust(false);

    XMLNode *objective = game->m_gameControlNode->getFirstNode();
    assert(objective != NULL);

    const std::string &objName = objective->getName();

    // Check, if there are any crates to pick up.
    // Note, that bonus and fuel crates may be picked up anytime.

    Crate *crateToPickup = PlayGround::getInstance()->hasCrateFor(ship);
    if (crateToPickup)
    {
        bool autoDelete = true;
        bool pickup = false;

        switch (crateToPickup->getType())
        {
        case CrateSurfaces::T_SMALL:
        case CrateSurfaces::T_MEDIUM:
        case CrateSurfaces::T_BIG:
            if (objName == "pickup")
            {
                if (XML_TOOLS::popUnsignedFromPropertyTokenString(
                        objective->getMandatoryProperty("id"),
                        crateToPickup->getId()))
                {
                    autoDelete = false;
                    pickup = true;
                    ship->pushCrate(crateToPickup);

                    XMLNode *onpickup = objective->getNode("onpickup");
                    if (onpickup)
                    {
                        game->handleEvents(onpickup);
                        objective->removeNode(onpickup);
                        delete onpickup;
                    }
                }
            }
            break;

        case CrateSurfaces::T_BONUS:
            pickup = true;
            break;

        case CrateSurfaces::T_FUEL:
            pickup = true;
            if (crateToPickup->getValue() == 0)
            {
                ship->setFuel(game->getPlayerFuel());
            }
            else
            {
                unsigned amount = ship->getFuel() +
                    crateToPickup->getValue() * SDLFrameRate::getFrameRate();
                ship->setFuel(MIN(amount, game->getPlayerFuel()));
            }
            break;

        default:
            break;
        }

        if (pickup)
        {
            PlayerStatus::getInstance()->addToScore(
                ScoreTable::getInstance()->getCratePickUp(crateToPickup));
            SoundInterface::getInstance()->onCratePickUp();

            ObjectRepository::getInstance()->markObjectToRemove(
                crateToPickup, autoDelete);
        }
    }

    if (objName == "pickup")
    {
        // Check, if there are crates to unload.

        const XMLProperty *objPlatform =
            objective->getMandatoryProperty("platform");

        if (objPlatform->getAsUnsigned() == platform->getId())
        {
            assert(game->m_platformToUnload == NULL);
            game->m_platformToUnload = platform;
            game->unloadCrate();
        }
    }
    else if (objName == "land")
    {
        // Check, if landing on 'platform' is the current objective.
        (void)XML_TOOLS::popFirstUnsignedFromPropertyTokenString(
            objective->getMandatoryProperty("platform"),
            platform->getId());

        game->handleGameContinuation();
    }
}


//----------------------------------------------------------------------------
void GameControlBase::UnloadingCrate::onPreUpdate(GameControlBase *game)
{
    if (game->hasFrameCounterReachedLimit())
    {
        assert(game->m_crateToUnload != NULL);
        assert(game->m_platformToUnload != NULL);

        ObjectRepository::getInstance()->markObjectToRemove(
            game->m_crateToUnload);
        game->m_crateToUnload = NULL;

        if (game->getPlayerShip()->hasCrates())
        {
            game->initUnloadingDelay();
            game->setState(UnloadingDelay::getInstance());
        }
        else
        {
            game->m_platformToUnload = NULL;
            game->handleGameContinuation();
        }
    }
}

//----------------------------------------------------------------------------
void GameControlBase::UnloadingCrate::onPlayerShipDestroyed(
    Ship *ship, GameControlBase *game)
{
    GameStateBase::onPlayerShipDestroyed(ship, game);

    assert(game->m_crateToUnload != NULL);
    assert(game->m_platformToUnload != NULL);

    ObjectRepository::getInstance()->markObjectToRemove(
        game->m_crateToUnload);
    game->m_crateToUnload = NULL;
    game->m_platformToUnload = NULL;
}


//----------------------------------------------------------------------------
void GameControlBase::UnloadingDelay::onPreUpdate(GameControlBase *game)
{
    if (game->hasFrameCounterReachedLimit())
    {
        game->unloadCrate();
    }
}


//----------------------------------------------------------------------------
void GameControlBase::OutOfFuelDelay::onPreUpdate(GameControlBase *game)
{
    if (game->hasFrameCounterReachedLimit())
    {
        game->getPlayerShip()->setExplode();
    }
}


//----------------------------------------------------------------------------
void GameControlBase::ExplosionDelay::onPreUpdate(GameControlBase *game)
{
    if (game->hasFrameCounterReachedLimit())
    {
        PlayerStatus::getInstance()->decLifes();

        game->removeReachedObjectives();

        // If the only remaining objective is landing on the home platform,
        // it has to be removed. The level thus will end immediately.
        game->removeLandOnHomePlatformObjective();

        if (PlayerStatus::getInstance()->getLifes() > 0 &&
            game->hasObjectivesLeft())
        {
            game->resetPlayerShip();
            game->setState(Running::getInstance());
        }
        else
        {
            game->setState(Final::getInstance());
        }
    }
}


//----------------------------------------------------------------------------
void GameControlBase::AddFuelToScore::onPreUpdate(GameControlBase *game)
{
   if (!game->getPlayerShip()->decFuel(game->getPlayerFuelDecrement()))
   {
       if (game->hasFrameCounterReachedLimit())
       {
           game->resetFrameCounter();

           PlayerStatus::getInstance()->addToScore(
               ScoreTable::getInstance()->getSpareFuelStep());
       }
   }
   else if (PlayerStatus::getInstance()->getBonus() > 0)
   {
       game->initAddBonusToScore();
       game->setState(AddBonusToScore::getInstance());
   }
   else
   {
       game->setState(Final::getInstance());
   }
}


//----------------------------------------------------------------------------
void GameControlBase::AddBonusToScore::onPreUpdate(GameControlBase *game)
{
    if (game->hasFrameCounterReachedLimit())
    {
        game->resetFrameCounter();

        PlayerStatus *ps = PlayerStatus::getInstance();
        if (ps->getBonus() > 0)
        {
            ps->addToScore(1);
            ps->decBonus(1);
        }
        else
        {
            game->setState(Final::getInstance());
        }
    }
}



//----------------------------------------------------------------------------
void GameControlBase::ConvertNumberRangesVisitor::do_visit(XMLNode *n)
{
    if (n->getName() == "land")
    {
        convertProperty(n, "platform");
    }
    else if (n->getName() == "pickup" ||
             n->getName() == "destroy")
    {
        convertProperty(n, "id");
    }
    else if (n->getName() == "onpickup" ||
             n->getName() == "onreached")
    {
        convertProperty(n, "show");
        convertProperty(n, "deactivate");
        convertProperty(n, "activate");
        convertProperty(n, "toggle");
    }
}

//----------------------------------------------------------------------------
void GameControlBase::ConvertNumberRangesVisitor::convertProperty(XMLNode *n, const char *name)
{
    XMLProperty *p = n->getMandatoryProperty(name);
    std::string value = p->getValue();
    p->setValue(STRING_TOOLS::convertNumberRanges(value));
}


//----------------------------------------------------------------------------
void GameControlBase::RemoveDestroyObjectiveVisitor::do_visit(XMLNode *n)
{
    if (n->getName() == "destroy")
    {
        (void)XML_TOOLS::popUnsignedFromPropertyTokenString(
            n->getMandatoryProperty("id"), m_id);
    }
}


//----------------------------------------------------------------------------
void GameControlBase::RemoveReachedObjectivesVisitor::postVisit()
{
    for (XMLNodeList::const_iterator
             iter = m_list.begin(),
             end = m_list.end();
         iter != end; ++iter)
    {
        m_gc->handleEvents((*iter)->getNode("onreached"));
        m_gc->m_gameControlNode->removeNode(*iter);
        delete *iter;
    }
}

//----------------------------------------------------------------------------
void GameControlBase::RemoveReachedObjectivesVisitor::do_visit(XMLNode *n)
{
    const std::string &name = n->getName();

    if ((name == "pickup" && n->getStringProperty("id").empty()) ||
        (name == "land" && n->getStringProperty("platform").empty()) ||
        (name == "destroy" && n->getStringProperty("id").empty()))
    {
        m_list.push_back(n);
    }
}



//----------------------------------------------------------------------------
GameControlBase *GameControlBase::sm_instance = NULL;


//----------------------------------------------------------------------------
GameControlBase::GameControlBase()
{
    m_player = NULL;
    m_playerId = 0;
    m_playerFuel = 0;
    m_playerFuelDecrement = 0;

    m_initialXPosition = 0;
    m_initialYPosition = 0;
    m_homePlatformId = 0;

    m_descriptionNode = NULL;
    m_gameControlNode = NULL;

    m_crateToUnload = NULL;
    m_platformToUnload = NULL;

    m_bonusFrameCounter = 0;
    m_bonusFrameCounterLimit = 0;
    m_bonusDecrement = 0;

    m_frameCounter = 0;
    m_frameCounterLimit = 0;

    m_state = Running::getInstance();
}

//----------------------------------------------------------------------------
GameControlBase::~GameControlBase()
{
    m_player = NULL;

    ZAP_POINTER(m_descriptionNode);
    ZAP_POINTER(m_gameControlNode);

    m_crateToUnload = NULL;
    m_platformToUnload = NULL;

    m_state = NULL;
}


//----------------------------------------------------------------------------
void GameControlBase::init(const char *mission, const char *level)
{
    destroy();

    std::string full;
    full.append(Configuration::getInstance()->getDataDir())
        .append("/levels/").append(mission).append("/").append(level);

    XMLParser p;
    XMLNode root;
    p.parse(full.c_str(), root);

    sm_instance = new GameControlBase();

    XMLNode *levelNode = root.getMandatoryNode("level");

    PlayGround::init(mission, levelNode->getMandatoryNode("playground"));
    sm_instance->initBonus(levelNode->getNode("bonus"));
    sm_instance->initStartPosition(
        levelNode->getMandatoryNode("startposition"));

    sm_instance->m_descriptionNode = levelNode->popNode("description");

    sm_instance->m_gameControlNode = levelNode->popNode("gamecontrol");
    if (sm_instance->m_gameControlNode)
    {
        ConvertNumberRangesVisitor v;
        sm_instance->m_gameControlNode->acceptAllNodes(v);
    }

    PlayerStatus::getInstance()->clearMessage();
}

//----------------------------------------------------------------------------
void GameControlBase::destroy()
{
    ObjectRepository::getInstance()->clear();
    PlayGround::destroy();
    ZAP_POINTER(sm_instance);
}

//----------------------------------------------------------------------------
void GameControlBase::initBonus(const XMLNode *bonusNode)
{
    if (bonusNode)
    {
        PlayerStatus::getInstance()->setInitialBonus(
            bonusNode->getUnsignedProperty("initial"));
        m_bonusDecrement = bonusNode->getUnsignedProperty("decrement");
        m_bonusFrameCounterLimit =
            bonusNode->getUnsignedProperty("delay", 10) *
            SDLFrameRate::getFrameRate() / 10;
    }
    else
    {
        PlayerStatus::getInstance()->setInitialBonus(0);
        m_bonusDecrement = 0;
        m_bonusFrameCounterLimit = 0;
    }
}

//----------------------------------------------------------------------------
void GameControlBase::initStartPosition(const XMLNode *startPositionNode)
{
    // Levels for the test suite may not have any properties.
    if (!startPositionNode->hasProperties())  return;

    m_playerFuel = SDLFrameRate::getFrameRate() *
        startPositionNode->getUnsignedProperty("fuel", 100);

    createPlayerShip();

    m_homePlatformId = startPositionNode->getIntProperty("platform", 0);
    if (m_homePlatformId > 0)
    {
        // Start on a platform.

        Platform *homePlatform = dynamic_cast<Platform*>(
            ObjectRepository::getInstance()->getObject(m_homePlatformId));
        if (homePlatform == NULL)
        {
            std::ostringstream s;
            s << "The platform with the id '" << m_homePlatformId
              << "' cannot be declared as home platform "
              << "because it was not defined within the <decorations> node"
              << std::ends;
            throw XMLException(s.str());
        }

        const SDL_Rect &landingZone = homePlatform->getLandingZone();

        m_initialXPosition =
            landingZone.x + landingZone.w/2 - m_player->getPosition().w/2;
        m_initialYPosition =
            landingZone.y - m_player->getBottomPixels().y;
    }
    else
    {
        // We have a "flying" start here.

        m_initialXPosition =
            startPositionNode->getIntProperty("x") -
            m_player->getPosition().w/2;
        m_initialYPosition =
            startPositionNode->getIntProperty("y") -
            m_player->getPosition().h/2;
    }

    resetPlayerShip();
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship)
{
    do_explode(ship);

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, Grenade *grenade)
{
    do_explode(ship);
    do_explode(grenade);

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, Missile *missile)
{
    do_explode(ship);
    do_explode(missile);

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, MortarBase *mortar)
{
    do_explode(ship);
    do_explode(mortar);

    PlayerStatus::getInstance()->addToScore(
        ScoreTable::getInstance()->getMortarShot());

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, ProjectileBase *projectile)
{
    do_explode(ship);
    do_explode(projectile);

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, SAMBatteryBase *sam)
{
    do_explode(ship);
    do_explode(sam);

    PlayerStatus::getInstance()->addToScore(
        ScoreTable::getInstance()->getSAMBatteryShot());

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, Tank *tank)
{
    do_explode(ship);
    do_explode(tank);

    PlayerStatus::getInstance()->addToScore(
        ScoreTable::getInstance()->getTankShot());

    initExplosion();
    setState(ExplosionDelay::getInstance());
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Ship *ship, TurretBase *turret)
{
    do_explode(ship);
    do_explode(turret);

    PlayerStatus::getInstance()->addToScore(
        ScoreTable::getInstance()->getTurretShot());

    initExplosion();
    setState(ExplosionDelay::getInstance());
}


//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile)
{
    do_explode(missile);
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, Grenade *grenade)
{
    do_explode(missile);
    do_explode(grenade);

    if (grenade->getCreatorId() == getPlayerShipId())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getMissileShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, MortarBase *mortar)
{
    do_explode(missile);
    do_explode(mortar);

    if (missile->getCreatorId() == getPlayerShipId())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getMortarShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, ProjectileBase *projectile)
{
    do_explode(missile);
    do_explode(projectile);

    if (projectile->getCreatorId() == getPlayerShipId())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getMissileShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, SAMBatteryBase *sam)
{
    do_explode(missile);
    do_explode(sam);

    if (missile->getCreatorId() == getPlayerShipId())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getSAMBatteryShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, SwitchBase *s)
{
    do_explode(missile);
    s->toggle();
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, Tank *tank)
{
    do_explode(missile);
    do_explode(tank);

    if (missile->getCreatorId() == getPlayerShipId())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getTankShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Missile *missile, TurretBase *turret)
{
    do_explode(missile);
    do_explode(turret);

    if (missile->getCreatorId() == getPlayerShipId())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getTurretShot());
    }
}


//----------------------------------------------------------------------------
void GameControlBase::onCollision(Grenade *grenade)
{
    do_explode(grenade);
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Grenade *grenade, MortarBase *mortar)
{
    do_explode(grenade);
    do_explode(mortar);

    if (grenade->getCreatorId() == getPlayerShipId())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getMortarShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Grenade *grenade, SwitchBase *s)
{
    do_explode(grenade);
    s->toggle();
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Grenade *grenade, Tank *tank)
{
    do_explode(grenade);
    do_explode(tank);

    if (grenade->getCreatorId() == getPlayerShipId())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getTankShot());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(Grenade *grenade, TurretBase *turret)
{
    do_explode(grenade);
    do_explode(turret);

    if (grenade->getCreatorId() == getPlayerShipId())
    {
        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getTurretShot());
    }
}


//----------------------------------------------------------------------------
void GameControlBase::onCollision(ProjectileBase *projectile)
{
    do_explode(projectile);
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(ProjectileBase *projectile,
                                  MortarBase *mortar)
{
    do_explode(projectile);
    if (mortar->decAndTestHitPoints())
    {
        do_explode(mortar);

        if (projectile->getCreatorId() == getPlayerShipId())
        {
            PlayerStatus::getInstance()->addToScore(
                ScoreTable::getInstance()->getMortarShot());
        }
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(ProjectileBase *projectile, SwitchBase *s)
{
    do_explode(projectile);
    s->toggle();
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(ProjectileBase *projectile, Tank *tank)
{
    do_explode(projectile);
    if (tank->decAndTestHitPoints())
    {
        do_explode(tank);

        if (projectile->getCreatorId() == getPlayerShipId())
        {
            PlayerStatus::getInstance()->addToScore(
                ScoreTable::getInstance()->getTankShot());
        }
    }
}

//----------------------------------------------------------------------------
void GameControlBase::onCollision(ProjectileBase *projectile,
                                  TurretBase *turret)
{
    do_explode(projectile);
    if (turret->decAndTestHitPoints())
    {
        do_explode(turret);

        if (projectile->getCreatorId() == getPlayerShipId())
        {
            PlayerStatus::getInstance()->addToScore(
                ScoreTable::getInstance()->getTurretShot());
        }
    }
}


//----------------------------------------------------------------------------
void GameControlBase::handleEvents(const XMLNode *node)
{
    if (node == NULL)  return;

    const XMLProperty *prop = NULL;
    std::vector<unsigned> ids;

    // Activate laser barriers?
    prop = node->getProperty("activate");
    if (prop)
    {
        STRING_TOOLS::toUnsignedVector(prop->getValue(), ids);
        for (size_t i=0; i<ids.size(); i++)
        {
            Barrier *barrier = dynamic_cast<Barrier*>(
                ObjectRepository::getInstance()->getObject(ids[i]));
            if (barrier)  barrier->activate();
        }
    }

    // Deactivate laser barriers?
    prop = node->getProperty("deactivate");
    if (prop)
    {
        STRING_TOOLS::toUnsignedVector(prop->getValue(), ids);
        for (size_t i=0; i<ids.size(); i++)
        {
            Barrier *barrier = dynamic_cast<Barrier*>(
                ObjectRepository::getInstance()->getObject(ids[i]));
            if (barrier)  barrier->deactivate();
        }
    }

    // Toggle laser barriers?
    prop = node->getProperty("toggle");
    if (prop)
    {
        STRING_TOOLS::toUnsignedVector(prop->getValue(), ids);
        for (size_t i=0; i<ids.size(); i++)
        {
            Barrier *barrier = dynamic_cast<Barrier*>(
                ObjectRepository::getInstance()->getObject(ids[i]));
            if (barrier)  barrier->toggle();
        }
    }

    // Show previously hidden objects?
    prop = node->getProperty("show");
    if (prop)
    {
        STRING_TOOLS::toUnsignedVector(prop->getValue(), ids);
        for (size_t i=0; i<ids.size(); i++)
        {
            ObjectRepository::getInstance()->showObjectId(ids[i]);
        }
    }

    // Change the text of the status bar?
    prop = node->getProperty("status");
    if (prop)
    {
        PlayerStatus::getInstance()->setMessage(prop->getValue());
    }
}


//----------------------------------------------------------------------------
void GameControlBase::do_explode(Grenade *grenade)
{
    CreateExplosionParticleConstVisitor v;
    grenade->accept(v);

    SoundInterface::getInstance()->onGrenadeExplosion();

    ObjectRepository::getInstance()->markObjectToRemove(grenade);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(Missile *missile)
{
    CreateExplosionParticleConstVisitor v;
    missile->accept(v);

    SoundInterface::getInstance()->onMissileExplosion();

    ObjectRepository::getInstance()->markObjectToRemove(missile);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(MortarBase *mortar)
{
    CreateExplosionParticleConstVisitor v;
    mortar->accept(v);

    SoundInterface::getInstance()->onMortarExplosion();

    ObjectRepository::getInstance()->markObjectToRemove(mortar);

    onObjectDestroyed(mortar);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(ProjectileBase *projectile)
{
    CreateExplosionParticleConstVisitor v;
    projectile->accept(v);

    SoundInterface::getInstance()->onProjectileExplosion();

    ObjectRepository::getInstance()->markObjectToRemove(projectile);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(SAMBatteryBase *sam)
{
    CreateExplosionParticleConstVisitor v;
    sam->accept(v);

    SoundInterface::getInstance()->onSAMBatteryExplosion();

    ObjectRepository::getInstance()->markObjectToRemove(sam);

    onObjectDestroyed(sam);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(Ship *ship)
{
    SoundInterface::getInstance()->onShipThrustOff();

    CreateExplosionParticleConstVisitor v;
    ship->accept(v);

    SoundInterface::getInstance()->onShipExplosion();

    ObjectRepository::getInstance()->markObjectToRemove(ship, false);

    if (ship == m_player)
    {
        onPlayerShipDestroyed();

        // Setting m_player to NULL lets GameBridge::getPlayerShip()
        // return NULL. This assures that if the ship explodes, all objects
        // (turrets, missiles, etc.) behave, as if no ship is present.
        m_player = NULL;
    }
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(Tank *tank)
{
    CreateExplosionParticleConstVisitor v;
    tank->accept(v);

    SoundInterface::getInstance()->onTankExplosion();

    ObjectRepository::getInstance()->markObjectToRemove(tank);

    onObjectDestroyed(tank);
}

//----------------------------------------------------------------------------
void GameControlBase::do_explode(TurretBase *turret)
{
    CreateExplosionParticleConstVisitor v;
    turret->accept(v);

    SoundInterface::getInstance()->onTurretExplosion();

    ObjectRepository::getInstance()->markObjectToRemove(turret);

    onObjectDestroyed(turret);
}


//----------------------------------------------------------------------------
void GameControlBase::createPlayerShip()
{
    assert(m_player == NULL);
    assert(m_playerId == 0);

    m_player = new Ship(
        (ShipSurfaces::Ship)
        PlayerConfiguration::getInstance()->getPlayer()->getShipType());
    m_playerId = m_player->getId();

    ObjectRepository::getInstance()->addObject(m_player);
}

//----------------------------------------------------------------------------
void GameControlBase::resetPlayerShip()
{
    if (!m_player)
    {
        m_player = dynamic_cast<Ship*>(
            ObjectRepository::getInstance()->getObject(m_playerId));
        assert(m_player != NULL);

        ObjectRepository::getInstance()->showObject(m_player);
    }

    m_player->setExplode(false);
    m_player->resetPosition(
        m_initialXPosition, m_initialYPosition, m_homePlatformId > 0);
    m_player->setFuel(getPlayerFuel());
}

//----------------------------------------------------------------------------
void GameControlBase::initUnloadingDelay()
{
    resetFrameCounter();
    setFrameCounterLimit(SDLFrameRate::getFrameRate() / 2);
}

//----------------------------------------------------------------------------
void GameControlBase::initOutOfFuel()
{
    resetFrameCounter();
    setFrameCounterLimit(SDLFrameRate::getFrameRate());
}

//----------------------------------------------------------------------------
void GameControlBase::initExplosion()
{
    resetFrameCounter();
    setFrameCounterLimit(SDLFrameRate::getFrameRate());
}

//----------------------------------------------------------------------------
void GameControlBase::initAddFuelToScore()
{
    // A fuel of 100 seconds means, that in every frame
    // 50 fuel units are decremented from the ship
    // and the score is incremented once.
    //
    // A fuel of 50 seconds thus has to decrement 25 fuel units in every frame
    // and the score shall only be incremented every second frame.
    //
    // This way, the speed of the fuel-to-score adding
    // is not depending on the value of the ship's initial fuel.

    unsigned playerFuelReference = 100 * SDLFrameRate::getFrameRate();
    unsigned ratio = playerFuelReference / getPlayerFuel();
    if (ratio == 0)
    {
        ratio = 1;
    }

    // @todo: Was 50 @96fps and should be 37.5 @72fps.
    //        But since the friction effect is lower, we use 40 here
    //        to keep the fuel bonus nearly constant.
    //        A clean solution depends on a framerate-independend
    //        implementation of the friction model.
    m_playerFuelDecrement = 40 / ratio;
    resetFrameCounter();
    setFrameCounterLimit(ratio);

    SoundInterface::getInstance()->onAddFuelToScore();
}

//----------------------------------------------------------------------------
void GameControlBase::initAddBonusToScore()
{
    resetFrameCounter();
    setFrameCounterLimit(2);

    SoundInterface::getInstance()->onAddBonusToScore();
}

//----------------------------------------------------------------------------
void GameControlBase::removeDestroyObjective(unsigned id)
{
    RemoveDestroyObjectiveVisitor v(id);
    m_gameControlNode->acceptAllNodes(v);
}

//----------------------------------------------------------------------------
void GameControlBase::removeLandOnHomePlatformObjective()
{
    XMLNode *objective = m_gameControlNode->getFirstNode();
    if (objective == NULL || objective->getName() != "land")  return;

    XMLProperty *platform = objective->getMandatoryProperty("platform");
    XML_TOOLS::popUnsignedFromPropertyTokenString(platform, m_homePlatformId);
    if (platform->getValue().empty())
    {
        m_gameControlNode->removeNode(objective);
    }
}

//----------------------------------------------------------------------------
void GameControlBase::removeReachedObjectives()
{
    RemoveReachedObjectivesVisitor v(this);
    m_gameControlNode->acceptAllNodes(v);
    v.postVisit();

    //std::cout << "-----" << std::endl;
    //XMLWriteToStreamVisitor v2(std::cout);
    //m_gameControlNode->accept(v2);
}

//----------------------------------------------------------------------------
void GameControlBase::handleGameContinuation()
{
    removeReachedObjectives();
    if (hasObjectivesLeft())
    {
        setState(Running::getInstance());
    }
    else
    {
        initAddFuelToScore();
        setState(AddFuelToScore::getInstance());
    }
}

//----------------------------------------------------------------------------
void GameControlBase::unloadCrate()
{
    assert(m_platformToUnload != NULL);

    if (m_player->hasCrates())
    {
        assert(m_crateToUnload == NULL);
        m_crateToUnload = m_player->popCrate();

        PlayerStatus::getInstance()->addToScore(
            ScoreTable::getInstance()->getCrateRescued(m_crateToUnload));
        SoundInterface::getInstance()->onCrateUnload();

        // Move the crate to a position beside the ship
        // and add it back to the playground.

        const SDL_Rect &landingZone = m_platformToUnload->getLandingZone();
        Sint16 x = m_player->getPosition().x + m_player->getBottomPixels().x
            + m_player->getBottomPixels().w/2;
        if (x - landingZone.x < landingZone.x + landingZone.w - x)
        {
            // Draw the crate at the ship's right side.
            x += m_player->getBottomPixels().w/2;
            x += m_crateToUnload->getPosition().w/4;
        }
        else
        {
            // Draw the crate at the ship's left side.
            x -= m_player->getBottomPixels().w/2;
            x -= m_crateToUnload->getPosition().w/4;
        }
        x -= m_crateToUnload->getPosition().w / 2;

        Sint16 y = m_player->getPosition().y + m_player->getBottomPixels().y
            - m_crateToUnload->getPosition().h + 1;
        // "+ 1", because m_player->getBottomPixels().y
        // lies one pixel above landingZone.y

        m_crateToUnload->setPosition(x, y);

        ObjectRepository::getInstance()->showObject(m_crateToUnload);

        initUnloadingDelay();
        setState(UnloadingCrate::getInstance());
    }
    else
    {
        m_platformToUnload = NULL;
        handleGameContinuation();
    }
}

//----------------------------------------------------------------------------
void GameControlBase::update()
{
    updateBonus();

    onPreUpdate();
    PlayGround::getInstance()->update();
    onPostUpdate();

    //GameRecorder::getInstance()->writeFrame();
}


//----------------------------------------------------------------------------
void GameControlBase::do_updateShipControl(ShipControl control)
{
    assert(m_player != NULL);

    switch (control)
    {
    case ROT_LEFT:
        m_player->setRotation(Ship::ROT_LEFT);
        break;
    case ROT_RIGHT:
        m_player->setRotation(Ship::ROT_RIGHT);
        break;
    case ROT_OFF:
        m_player->setRotation(Ship::ROT_NONE);
        break;
    case THRUST_ON:
        m_player->setThrust(true);
        break;
    case THRUST_OFF:
        m_player->setThrust(false);
        break;
    case ALIGN_ON:
        m_player->setAlign(true);
        break;
    case ALIGN_OFF:
        m_player->setAlign(false);
        break;
    case FIRE:
        m_player->setFire();
        break;
    }
}

//----------------------------------------------------------------------------
void GameControlBase::do_updateBonus()
{
    if (m_bonusFrameCounter++ == m_bonusFrameCounterLimit)
    {
        m_bonusFrameCounter = 0;
        PlayerStatus::getInstance()->decBonus(m_bonusDecrement);
    }
}
