/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.remote.resolver.soap;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.security.PrivilegedActionException;

import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.MBeanServerFactory;
import javax.management.ObjectName;
import javax.management.remote.JMXServiceURL;

import mx4j.log.Logger;
import mx4j.remote.ConnectionResolver;
import mx4j.tools.remote.soap.ClientSOAPConnection;
import mx4j.tools.remote.soap.SOAPConnector;
import mx4j.tools.remote.soap.SOAPConnectorServer;
import org.apache.axis.client.AdminClient;
import org.apache.axis.client.Service;
import org.apache.axis.utils.Options;
import org.apache.axis.configuration.FileProvider;

/**
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.5 $
 */
public class SOAPResolver extends ConnectionResolver
{
   private static final String SERVICE_JMX_SOAP = "service:jmx:soap";
   private static final String ID_CONTEXT = "/id/";
   private static final String SERVER_DEPLOY_WSDD = "server-deploy.wsdd";
   private static final String SERVER_UNDEPLOY_WSDD = "server-undeploy.wsdd";
   private static final String CLIENT_WSDD = "client.wsdd";
   private static int connectorID;

   private final Map mbeanServerIds = new HashMap();

   public Object lookupClient(JMXServiceURL address, Map environment) throws IOException
   {
      String endpoint = getEndpoint(address, environment);

      InputStream wsdd = getClass().getResourceAsStream(CLIENT_WSDD);
      if (wsdd == null) throw new IOException("Could not find AXIS deployment descriptor");
      Service service = new Service(new FileProvider(wsdd));
      service.setMaintainSession(true);

      return new ClientSOAPConnection(endpoint, service);
   }

   public Object bindClient(Object client, Map environment) throws IOException
   {
      return client;
   }

   private String getEndpoint(JMXServiceURL address, Map environment)
   {
      String transport = getTransportProtocol(environment);
      return transport + getEndpointPath(address);
   }

   private String getTransportProtocol(Map environment)
   {
      String transport = null;
      if (environment != null) transport = (String)environment.get(SOAPConnector.SOAP_TRANSPORT_PROTOCOL);
      if (transport == null || transport.length() == 0) transport = "http";
      transport = transport.toLowerCase();
      return transport;
   }

   private String getEndpointPath(JMXServiceURL url)
   {
      String address = url.toString();
      return address.substring(SERVICE_JMX_SOAP.length());
   }

   public Object createServer(JMXServiceURL url, Map environment) throws IOException
   {
      String connectorID = findConnectorID(url);
      if (connectorID == null) return null;

      String mbeanServerId = findMBeanServerId(connectorID);
      if (mbeanServerId == null) return null;

      List servers = MBeanServerFactory.findMBeanServer(mbeanServerId);
      if (servers.size() == 1) return servers.get(0);
      return null;
   }

   private String findConnectorID(JMXServiceURL url)
   {
      String path = url.getURLPath();
      if (path == null) return null;
      int index = path.indexOf(ID_CONTEXT);
      if (index < 0) return null;
      return path.substring(index + ID_CONTEXT.length());
   }

   private String findMBeanServerId(String connectorID)
   {
      synchronized (mbeanServerIds)
      {
         return (String)mbeanServerIds.get(connectorID);
      }
   }

   public JMXServiceURL bindServer(Object server, JMXServiceURL url, Map environment) throws IOException
   {
      String existingID = findConnectorID(url);
      String connectorID = existingID;
      if (connectorID == null) connectorID = generateConnectorID();

      final MBeanServer mbeanServer = (MBeanServer)server;
      try
      {
         try
         {
            String mbeanServerId = (String)AccessController.doPrivileged(new PrivilegedExceptionAction()
            {
               public Object run() throws JMException
               {
                  return mbeanServer.getAttribute(new ObjectName("JMImplementation:type=MBeanServerDelegate"), "MBeanServerId");
               }
            });
            synchronized (mbeanServerIds)
            {
               String existing = findMBeanServerId(connectorID);
               if (existing != null && !existing.equals(mbeanServerId)) throw new IOException("SOAPConnectorServer with ID " + connectorID + " is already attached to MBeanServer with ID " + existing);
               mbeanServerIds.put(connectorID, mbeanServerId);
            }
         }
         catch (PrivilegedActionException x)
         {
            throw (JMException)x.getException();
         }
      }
      catch (JMException x)
      {
         throw new IOException("Cannot retrieve MBeanServer ID " + x.toString());
      }

      deploy(url, environment);

      JMXServiceURL address = null;
      if (existingID != null)
      {
         address = url;
      }
      else
      {
         String path = url.getURLPath();
         if (path == null) path = "";
         if (!path.startsWith("/")) path = "/" + path;
         if (path.endsWith("/")) path = path.substring(0, path.length() - 1);
         address = new JMXServiceURL(url.getProtocol(), url.getHost(), url.getPort(), path + ID_CONTEXT + connectorID);
      }
      return address;
   }

   private void deploy(JMXServiceURL address, Map environment) throws IOException
   {
      String deployPath = environment == null ? null : (String)environment.get(SOAPConnectorServer.AXIS_DEPLOY_URL);
      if (deployPath == null || deployPath.length() == 0) deployPath = SOAPConnectorServer.DEFAULT_AXIS_DEPLOY_URL;

      JMXServiceURL temp = new JMXServiceURL(address.getProtocol(), address.getHost(), address.getPort(), deployPath);
      String deployEndpoint = getEndpoint(temp, environment);

      try
      {
         AdminClient deployer = new AdminClient();
         Options options = new Options(null);
         options.setDefaultURL(deployEndpoint);
         InputStream wsdd = getClass().getResourceAsStream(SERVER_DEPLOY_WSDD);
         if (wsdd == null) throw new IOException("Could not find AXIS deployment descriptor");
         deployer.process(options, wsdd);
      }
      catch (RuntimeException x)
      {
         throw x;
      }
      catch (Exception x)
      {
         Logger logger = getLogger();
         if (logger.isEnabledFor(Logger.INFO)) logger.info("Exception while deploying AXIS service", x);
         throw new IOException("Could not deploy SOAPConnectorServer to AXIS " + x.toString());
      }
   }

   private String generateConnectorID()
   {
      synchronized (SOAPResolver.class)
      {
         return String.valueOf(++connectorID);
      }
   }

   public void unbindServer(Object server, JMXServiceURL address, Map environment) throws IOException
   {
      String connectorID = findConnectorID(address);
      if (connectorID == null) throw new IOException("Unknown SOAPConnectorServer ID: " + address);
      synchronized (mbeanServerIds)
      {
         mbeanServerIds.remove(connectorID);
      }

      undeploy(address, environment);
   }

   private void undeploy(JMXServiceURL address, Map environment) throws IOException
   {
      String deployPath = environment == null ? null : (String)environment.get(SOAPConnectorServer.AXIS_DEPLOY_URL);
      if (deployPath == null || deployPath.length() == 0) deployPath = SOAPConnectorServer.DEFAULT_AXIS_DEPLOY_URL;

      JMXServiceURL temp = new JMXServiceURL(address.getProtocol(), address.getHost(), address.getPort(), deployPath);
      String undeployEndpoint = getEndpoint(temp, environment);

      try
      {
         AdminClient deployer = new AdminClient();
         Options options = new Options(null);
         options.setDefaultURL(undeployEndpoint);
         InputStream wsdd = getClass().getResourceAsStream(SERVER_UNDEPLOY_WSDD);
         if (wsdd == null) throw new IOException("Could not find AXIS deployment descriptor");
         deployer.process(options, wsdd);
      }
      catch (Exception x)
      {
         Logger logger = getLogger();
         if (logger.isEnabledFor(Logger.INFO)) logger.info("Exception while deploying AXIS service", x);
         throw new IOException("Could not deploy SOAPConnectorServer to AXIS " + x.toString());
      }
   }
}
