
/***************************************************************************
 *                                                                         *
 *   KCPULoad is copyright (c) 1999-2000, Markus Gustavsson                *
 *                         (c) 2002, Ben Burton                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "icontoggleaction.h"
#include "kcpudock.h"
#include "kcpuload.h"
#include "kcpuproc.h"

#include <kaction.h>
#include <kconfig.h>
#include <klocale.h>
#include <kpopupmenu.h>

KCPULoad::KCPULoad(QWidget *parent, const char *name) :
        StatPopup(true, parent, name) {
    // Create the /proc reading class and check for SMP.
    proc = new KCPUProc();
    supportSMP = proc->hasSMP();

    // Set up actions and read the config file.
    setupActions();

    // Create system tray windows.
    dock[0] = new KCPUDock(0, this);

    if (supportSMP && actSMP->isChecked()) {
        dock[1] = new KCPUDock(1, this);

        ((KCPUDock*)dock[0])->setCPULabel(1);
        ((KCPUDock*)dock[1])->setCPULabel(2);
    } else {
        ((KCPUDock*)dock[0])->setCPULabel(0);
    }

    // Initialise the pop-up window.
    readPopupState();

    // Off we go!
    requestResize();
    if (isActive())
        startUpdates();
}

KCPULoad::~KCPULoad() {
    delete proc;
}

void KCPULoad::setSMP(bool set) {
    if (! supportSMP)
        return;

    if (set && ! dock[1]) {
        // Enable SMP.
        dock[1] = new KCPUDock(1, this);

        ((KCPUDock*)dock[0])->setCPULabel(1);
        ((KCPUDock*)dock[1])->setCPULabel(2);
    } else if ((! set) && dock[1]) {
        // Disable SMP.
        ((KCPUDock*)dock[0])->setCPULabel(0);

        cleanUpDock(1);
        delete dock[1];
        dock[1] = 0;
    }

    requestResize();
    if (isActive())
        takeReading();

    config->setGroup("General Options");
    config->writeEntry("SMP", set);
    config->sync();
}

QString KCPULoad::dockName(int which) const {
    return (which == 0 ? i18n("CPU 0") : i18n("CPU 1"));
}

QColor KCPULoad::defaultDockColor(int which) const {
    return (which == 0 ? QColor(0, 255, 0) : QColor(255, 0, 0));
}

void KCPULoad::setupCustomActions() {
    if (supportSMP) {
        bool bVal = config->readBoolEntry("SMP", false);
        actSMP = new IconToggleAction(i18n("Enable S&MP"), "smp",
            i18n("S&MP Enabled"), "smpon", 0, coll, "smp");
        actSMP->setChecked(bVal);
        connect(actSMP, SIGNAL(toggled(bool)), this, SLOT(setSMP(bool)));
    }
}

void KCPULoad::insertCustomItems(KPopupMenu* menu) {
    if (supportSMP) {
        actSMP->plug(menu);
        menu->insertSeparator();
    }
}

void KCPULoad::takeReadingInternal() {
    proc->readLoad();

    if (dock[1]) {
        if (isSplit()) {
            upper[0] = proc->userPercent0();
            lower[0] = proc->systemPercent0();
            upper[1] = proc->userPercent1();
            lower[1] = proc->systemPercent1();
        } else {
            upper[0] = proc->totalPercent0();
            upper[1] = proc->totalPercent1();
        }
    } else {
        if (isSplit()) {
            upper[0] = proc->userPercent();
            lower[0] = proc->systemPercent();
        } else {
            upper[0] = proc->totalPercent();
        }
    }

    if (isVisible()) {
        if (dock[1]) {
            if (isSplit()) {
                fullReading.sprintf(i18n(
                    "Current CPU User: %d%%, C0: %d%%, C1: %d%%.\n"
                    "Current CPU System: %d%%, C0: %d%%, C1: %d%%.").latin1(),
                    proc->userPercent(), upper[0], upper[1],
                    proc->systemPercent(), lower[0], lower[1]);
            } else {
                fullReading.sprintf(i18n(
                    "Current CPU usage: %d%%, C0: %d%%, C1: %d%%.").latin1(),
                    proc->totalPercent(), upper[0], upper[1]);
            }
        } else {
            if (isSplit()) {
                fullReading.sprintf(i18n(
                    "Current CPU User: %d%%.\n"
                    "Current CPU System: %d%%.").latin1(),
                    upper[0], lower[0]);
            } else {
                fullReading.sprintf(i18n(
                    "Current CPU usage: %d%%.").latin1(), upper[0]);
            }
        }
    }
}

#include "kcpuload.moc"
