/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_auth_basic.pike,v 1.50.2.3 2004/01/24 00:10:43 vida Exp $
 */

//
//! module: CAMAS: Basic Auth
//!  Basic Auth Module for CAMAS.<br />
//!  This module will give the interface between standard
//!  flat file (eg. /etc/passwd) to get names, login etc...
//! inherits: module
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_auth_basic.pike,v 1.50.2.3 2004/01/24 00:10:43 vida Exp $
//
// Faster Debugging
#include <module.h>
#include <camas/maindefvars.h>
#include <camas/globals.h>
inherit "module";

constant cvs_version="$Id: camas_auth_basic.pike,v 1.50.2.3 2004/01/24 00:10:43 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Auth Basic";
constant module_doc  = "Basic Auth Module for CAMAS.<br>"
                       "This module will give the interface between standard "
                       "flat file (eg. /etc/passwd) to get names, login, etc..<br />"
                       "Additionnaly this module will try to find a suitable IMAP server " 
                       "given the login name. The priority of the method allow you to choose "
                       "the order we try each method. The lowest priority is tried first."
                       "<br />You can have several Auth modules, the internal priority of these "
                       "modules can be used to specify the order in which the user are "
                       "checked against each module.";
constant module_unique = 1;
constant thread_safe=1;

// Global variables pour this module
mapping userdatabase; // The contents of namedb in array
// in the same form than the file
// format
object filewatch;     // Filewatcher for autoreloading of file...

// used to store the 10 priorities
array priorities;

// Functions to hide parameters when they are not need
int hide_file() {
  return (QUERY (usernamemethod) != "file");
}

// Variables
void create()
{
#ifdef CAMAS_DEBUG
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
#endif
  defvar("usernamemethod","file",
         "User names method", TYPE_STRING_LIST, "Chooses the method used "
         "to set the user names.", ({ "file", "none" }));

  defvar("namedb","/etc/passwd",
         "User names file", TYPE_FILE, "If method is set to file, this is "
         "the file that will be searched for the users real name."
         "<br />Format is composed of lines like: "
         "'login:surname:name:domain:email'.", 0, hide_file);

  defvar("dnsname","","The sitewide DNS domain name",TYPE_STRING,
         "The DNS domain name used for CAMAS webmail");

  defvar("emaillogin",0,"Login using email address",TYPE_FLAG,
         "Use the email address to login instead of the imap login");

  defvar("basic_priority",9,"IMAP Basic:Priority",TYPE_MULTIPLE_INT,
         "The priority of the basic method. The basic method allow you to give only one "
	       "IMAP server to use for anything", ({ 1, 2, 3, 4, 5, 6, 7, 8, 9 }));
  defvar("basic_imapserver", "localhost", "IMAP Basic:IMAP server", TYPE_STRING,
  	     "The IMAP server");
  defvar("basic_imapport", 143, "IMAP Basic:IMAP server port", TYPE_INT,
         "The port of the IMAP server we connect to");
  
  defvar("imapserverfromemail_priority", 2, "IMAP server from email:Priority",
         TYPE_MULTIPLE_INT,
         "The priority of this method. Using this method, CAMAS will guess IMAP server"
	       "from the domain in the email address and the server listed here",
         ({ 1, 2, 3, 4, 5, 6, 7, 8, 9 }));
  defvar("imapserverfromemail_imapserver", DEFAULT_IMAPSERVERS , 
         "IMAP server from email:Servers", TYPE_TEXT_FIELD,
         "List of servers to use. "
         "<i><b>Format:</b> Domain Server[:port]</i><br />"
         "Example:\n"
         "<pre>acc.umu.se imap.acc.umu.se<br />"
         "some.where  mail.some.where</pre>"
         "<b>Note</b>: To use this feature your users must loggin with an email address");
}

array|int readfile(string file)
{
  // Should we replace this by Stdio.read_bytes ?
  object o=Stdio.File();
  if(!o->open(file,"r"))
    return 0;
  array result=( (o->read()-"\r") / "\n");
  o->close();

  return result - ({ "" });
}

int passwd_load(string passwdfile)
{
  if (QUERY(usernamemethod)=="file")    // We use flat file database
  {
    CDEBUG("(re)loading password file");
    array|int tempdb=readfile(passwdfile);
    if (intp(tempdb)) {
      CDEBUG(" Cannot read "+passwdfile+" file !\n");
    }
    else {
      CDEBUG("Data base :" + sprintf("%O", tempdb) +"\n");
      sort(tempdb);      // sort it
      CDEBUG("Sorted Data base :" + sprintf("%O",tempdb) +"\n");
      userdatabase = ([ ]);    // initialize it :)
      foreach( tempdb, string foo) {
        array entry = foo / ":";
        if (sizeof(entry) > 6) { // Test if entry[0...6] exists
          userdatabase += ([ entry[0] : ({  entry[0], // Login
                                            entry[4], // Gecos
                                            entry[5], // Home directory
                                            entry[6]  // Shell
                                         }) ]);
        }
      }
      // userdatabase += ({ foo / ":" }); // populate
    }
    CDEBUG(sprintf("UserDatabase:%O\n",mkmapping(indices(userdatabase),values(userdatabase))));
    return 0;
  }
  else {
    userdatabase = 0; // Garbage collector :)
    return -1;
  }
}

void fillpriorities()
{
  priorities = allocate(10);
  priorities[QUERY(basic_priority)] = "Basic";
  priorities[QUERY(imapserverfromemail_priority)] = "ImapServerFromEmail";
}

void start(int cnt, object conf)
{
  if (QUERY(usernamemethod)=="file") {
    passwd_load(QUERY(namedb));
    filewatch = FileWatch.Callout(QUERY(namedb), 60, passwd_load);
  }
  else {
    userdatabase = 0;
  }
  fillpriorities();
}

string status()
{
  string out;
  out = "Basic Authentication module.";
  if (QUERY(usernamemethod)=="file")
    out += sprintf("<br />Status :<br />"
                   "<ul><li>Method : file</li>"
                   "<li>Entries in %s : %d</li>"
                   "</ul>",QUERY(namedb), sizeof(userdatabase));
  return out;
}

array(string) query_provides()
{
  return(({"camas_basic_auth", "camas_auth"}));
}

/*
 * What we provide here
 */

//
//! method: int version(void)
//!  Give the CAMAS_AUTH api version
//!  supported by the module
//! returns:
//!  the version of the API
//! note:
//!  The base API is 1. But if we provide v2, we
//!  *MUST* be backward compatible with v1.
//
int version()
{
  CDEBUG("version()");
  return 2;
}

//
//! method: string|int getlogin(string login)
//!  Return the imap login to be used to connect into IMAP
//!  server. Or a int to throw with an error
//! arg: string login
//!  The login typed/entered on the CAMAS login string.
//!  This can be an email, a part of an email or an imap id.
//! returns:
//!  a string : the imap login to be used.<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standard code or for unknown error type)<br />
//!  1 : Bad password<br />
//!  2 : Access denied<br />
//!  3 : Account is locked<br />
//!  4 : Change your password<br />
//
string|int getlogin(string login)
{
  string|int out;
  if (QUERY(usernamemethod)=="file") {
    if( userdatabase[login] ) out = userdatabase[login][0];
    else out = 0;
    CDEBUG("getlogin("+login+") = " + out );
    return out;
  }
  return login;
}

//
//! method: string|int getdnsname(void)
//!  Gets the sitewide email domain name that can be used
//!  for email login or to complete unqualified emails
//! returns:
//!  a string : the dns domain name<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standard code or unknown error type)<br />
//!  1 : There is no domain name set, use whole email or imap login
//
string|int getdnsname()
{
  CDEBUG("getdnsname()");
  if(QUERY(dnsname) !="")
    return QUERY(dnsname);
  return 1;
}

//
//! method: int getemaillogin(void)
//!  Do the email is used instead of the imap login as
//!  login method ?
//! returns:
//!  a int : 1 = Yes, 0 = No
//
int getemaillogin()
{
  CDEBUG("getemaillogin()");
  return (int)QUERY(emaillogin);
}

//
//! method: array|int getfullnames(string login)
//!  Get the name and the surname of the specifier imap login
//! arg: string login
//!  The imap login
//! returns:
//!  a array : the name and surname in the form :
//!  ({ "surname", "name", "email@domain.com", "imaplogin" })<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standard error code or unknown error type)<br />
//!  1 : Empty set
//
array|int getfullnames(string login)
{
  CDEBUG("getfullnames("+login+")");
  array|int out;
  if (QUERY(usernamemethod)=="file") {
    CDEBUG("usernamemethod is file");
    if( userdatabase[login] ) {
      CDEBUG("login " + login + " found in database");
      out = (userdatabase[login][1]/",")[0]/" ";
      if (sizeof(out)==1)
        out += ({ "" });
      if (getemaillogin()) {
        out += ({ login });
	CDEBUG("User login using email address");
      } else {
        if (stringp(getdnsname()) && sizeof(getdnsname()))
	{
          out += ({ login + "@" + getdnsname() });
	  CDEBUG("Guessed email address from login and DNS domain from "
	    "the Auth module");
	}
        else
        {
	  string main_domain = my_configuration()->get_provider("camas_main")->QUERY (smtpmaildomain);
          if(stringp(main_domain) && sizeof(main_domain))
	  {
            out += ({ login + "@" + my_configuration()->get_provider("camas_main")->QUERY (smtpmaildomain) });
	    CDEBUG("Guessed email address from login and DNS domain from "
	      "Camas main module");
	  }
          else
	  {
            out += ({ login + "@unknown.domain" });
	    CDEBUG("Fallback: set email address to login@unknown.domain");
	  }
        }
      }
      out += ({ login });
    }
    else
    {
      out = 0;
      CDEBUG("usernamemethod is not file :(");
    }
    CDEBUG("getfullnames("+login+") = " + sprintf("%O",out));
    return out;
  }
  return 1;
}

//
//! class: Basic
//!  This class provides basic IMAP server that is it only gives one IMAP server
//!  from the CIF.
//
class Basic {

  mapping(string:string|int)|int getimapserver(string login)
  {
    CDEBUG("getimapserver from Basic class\n");
    mapping res = ([ 
                    "imapserver":   QUERY(basic_imapserver),
                    "imapport": QUERY(basic_imapport),
                  ]);
    array loginsplit = login / "@";
    if(sizeof(loginsplit) == 2 && loginsplit[1])
      res += ([ "maildomain": loginsplit[1] ]);
    CDEBUG(sprintf("result=%O\n", res));
    return res;
  }
};

//! class: ImapServerFromEmail
//!  This class returns the IMAP server for the given email using the IMAP server list
//!  in the CIF
class ImapServerFromEmail {
 
  array imap_servers() {
    string imapservers = QUERY (imapserverfromemail_imapserver);

    array list = (imapservers-"\r")/"\n";
    array servers=({ });

    int i=0;
    foreach(list, string line) {
      if (strlen(line)>0) {
        string ldomain = "", lserver = "", lcfg = "";
        sscanf(line, "%*[ ]%s%*[\t ]%s%*[\t ]%s", ldomain, lserver, lcfg);
        if (strlen(lserver)==0) {
          sscanf(line, "%*[ ]%s%*[\t ]%s", ldomain, lserver);
        }
        if (sizeof(ldomain) > 0) {
          mapping lcfgmap=([]);
          foreach ((lcfg / ",") , string param) {
            array parsplt = param / ":";
            if(sizeof(parsplt[0]-" "))
              if (sizeof(parsplt)>1) {
                lcfgmap[parsplt[0]] = parsplt[1];
              } else {
                lcfgmap[param] = 1;
              }
          }
          servers += ({ ({ldomain, lserver, lcfgmap}) });
        }
      }
  }
  // Only the non-empty elements returned
  return servers;
  }

  mapping(string:string|int)|int getimapserver(string login)
  {
    CDEBUG("getimapserver from ImapServerFromEmail class\n");
    mapping res = ([ ]);
    array loginsplit = login / "@";
    // we don't try some magic guessing here, it has a domain 
    // and then we try or it doesn't and then go to less priority
    // class
    if(sizeof(loginsplit) != 2)
    {
      CDEBUG("login doesn't contain any domain name, exiting...\n");
      return 0;
    }
    string mxdomain = loginsplit[1];
    foreach(imap_servers(), array arr) {
      array servs = arr[1]/":";
      string server = servs[0];
      int port;
      string mailpath;
      if (arr[2]->mailpath)
        mailpath = arr[2]->mailpath;
      string prefsbox;
      if (arr[2]->prefsbox)
        prefsbox = arr[2]->prefsbox;
      if (sizeof(servs) > 1)
        port = (int) servs[1];
      if(mxdomain == arr[0] || mxdomain == arr[1]) {
        // Exact match, we're done
        if(server)
          res += ([ "imapserver"  : server ]);
        if(port)
          res += ([ "imapport": port ]);
        if(mailpath)
          res += ([ "mailpath": mailpath ]);
        if(prefsbox && mailpath)
          res += ([ "baseprefsbox": prefsbox ]);
        res += ([ "maildomain": mxdomain ]);
      }
    }
    CDEBUG(sprintf("result=%O\n", res));
    return res;
  }
};

//
//! method: mapping(string:string|int) getimapserver(string login)
//!  Return the imap informations to be used to connect into IMAP
//!  server. Or a int to throw with an error
//! arg: string login
//!  The login typed/entered on the CAMAS login string.
//!  This can be an email, a part of an email or an imap id.
//! returns:
//!  a mapping : maps configuration name to their values<br />
//!   usual names include domain, imapport, imapserver.
//!  an int : an error code if there is an error.
//
mapping(string:string|int)|int getimapserver(string login)
{
  mapping imapserver = ([ ]);
  // for priorities see fillpriorities()
  array lpriorities = priorities;
  for(int i = 0; i < sizeof(lpriorities); i++)
  {
    // the priority of each class
    string classname = lpriorities[i];
    if(classname)
    {
      CDEBUG(sprintf("i=%d\n", i));
      CDEBUG(sprintf("getimapserver: calling %s.getimapserver(%s)\n", classname, login));
      // launch getimapserver function for class named classnamed
      object obj = this_object()[classname]();
      mapping|int tmp = obj->getimapserver(login);
      if(tmp)
      {
        // don't overwrite existing value(s), only add new one in imapserver
        foreach(indices(tmp), string indice)
        {
          if(!imapserver[indice] && tmp[indice])
            imapserver += ([ indice: tmp[indice] ]);
        }
      }
    }
  }
  if(imapserver)
    return imapserver;
  return 0;
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the call / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: usernamemethod
//! Chooses the method used to set the user names.
//!  type: TYPE_STRING_LIST
//!  name: User names method
//
//! defvar: namedb
//! If method is set to file, this is the file that will be searched for the users real name.<br />Format is composed of lines like: 'login:surname:name:domain:email'.
//!  type: TYPE_FILE
//!  name: User names file
//
//! defvar: dnsname
//! The DNS domain name used for CAMAS webmail
//!  type: TYPE_STRING
//!  name: The sitewide DNS domain name
//
//! defvar: emaillogin
//! Use the email address to login instead of the imap login
//!  type: TYPE_FLAG
//!  name: Login using email address
//
//! defvar: basic_priority
//! The priority of the basic method. The basic method allow you to give only one IMAP server to use for anything
//!  type: TYPE_MULTIPLE_INT
//!  name: IMAP Basic:Priority
//
//! defvar: basic_imapserver
//! The IMAP server
//!  type: TYPE_STRING
//!  name: IMAP Basic:IMAP server
//
//! defvar: basic_imapport
//! The port of the IMAP server we connect to
//!  type: TYPE_INT
//!  name: IMAP Basic:IMAP server port
//
//! defvar: imapserverfromemail_priority
//! The priority of this method. Using this method, CAMAS will guess IMAP serverfrom the domain in the email address and the server listed here
//!  type: TYPE_MULTIPLE_INT
//!  name: IMAP server from email:Priority
//
//! defvar: imapserverfromemail_imapserver
//! List of servers to use. <i><b>Format:</b> Domain Server[:port]</i><br />Example:
//!<pre>acc.umu.se imap.acc.umu.se<br />some.where  mail.some.where</pre><b>Note</b>: To use this feature your users must loggin with an email address
//!  type: TYPE_TEXT_FIELD
//!  name: IMAP server from email:Servers
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

