#ifdef RCSID
static char RCSid[] =
"$Header$";
#endif

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
Name
  vmintcls.cpp - T3 metaclass - intrinsic class
Function
  
Notes
  
Modified
  03/08/00 MJRoberts  - Creation
*/

#include <stdlib.h>
#include <assert.h>

#include "vmtype.h"
#include "vmobj.h"
#include "vmglob.h"
#include "vmintcls.h"
#include "vmmeta.h"
#include "vmfile.h"
#include "vmlst.h"
#include "vmstack.h"


/* ------------------------------------------------------------------------ */
/*
 *   statics 
 */

/* metaclass registration object */
static CVmMetaclassClass metaclass_reg_obj;
CVmMetaclass *CVmObjClass::metaclass_reg_ = &metaclass_reg_obj;


/* ------------------------------------------------------------------------ */
/* 
 *   create dynamically using stack arguments 
 */
vm_obj_id_t CVmObjClass::create_from_stack(VMG_ const uchar **pc_ptr,
                                           uint argc)
{
    /* it is illegal to create this type of object dynamically */
    err_throw(VMERR_ILLEGAL_NEW);
    return VM_INVALID_OBJ;
}

/* 
 *   create with no initial contents 
 */
vm_obj_id_t CVmObjClass::create(VMG_ int in_root_set)
{
    vm_obj_id_t id = vm_new_id(vmg_ in_root_set, FALSE, FALSE);
    new (vmg_ id) CVmObjClass();
    return id;
}

/*
 *   create with a given dependency index 
 */
vm_obj_id_t CVmObjClass::create_dyn(VMG_ uint meta_idx)
{
    vm_obj_id_t id = vm_new_id(vmg_ FALSE, FALSE, FALSE);
    new (vmg_ id) CVmObjClass(vmg_ FALSE, meta_idx, id);
    return id;
}

/*
 *   create with a given dependency index 
 */
CVmObjClass::CVmObjClass(VMG_ int in_root_set, uint meta_idx,
                         vm_obj_id_t self)
{
    /* calls of this form can't come from the image file */
    assert(!in_root_set);

    /* allocate the extension */
    ext_ = (char *)G_mem->get_var_heap()->alloc_mem(8, this);

    /* set up the extension - write the length and dependency index */
    oswp2(ext_, 8);
    oswp2(ext_ + 2, meta_idx);
    oswp4(ext_ + 4, VM_INVALID_OBJ);

    /* register myself with the metaclass table */
    register_meta(vmg_ self);
}

/* 
 *   notify of deletion 
 */
void CVmObjClass::notify_delete(VMG_ int in_root_set)
{
    /* free our extension */
    if (ext_ != 0 && !in_root_set)
        G_mem->get_var_heap()->free_mem(ext_);
}

/* set a property */
void CVmObjClass::set_prop(VMG_ CVmUndo *undo,
                           vm_obj_id_t self, vm_prop_id_t prop,
                           const vm_val_t *val)
{
    vm_obj_id_t mod_obj;
    
    /* if I have a modifier object, pass the setprop to the modifier */
    if ((mod_obj = get_mod_obj()) != VM_INVALID_OBJ)
    {
        /* set the property in the modifier object */
        vm_objp(vmg_ mod_obj)->set_prop(vmg_ undo, mod_obj, prop, val);
    }
    else
    {
        /* if we don't have a modifier, we can't set the property */
        err_throw(VMERR_INVALID_SETPROP);
    }
}

/*
 *   Get a list of our properties 
 */
void CVmObjClass::build_prop_list(VMG_ vm_obj_id_t self, vm_val_t *retval)
{
    uint meta_idx;
    vm_obj_id_t mod_obj;
    vm_meta_entry_t *entry;
    size_t my_prop_cnt;
    size_t mod_prop_cnt;
    vm_val_t mod_val;
    CVmObjList *lst;
    CVmObjList *mod_lst;

    /* presume we won't find any static properties of our own */
    my_prop_cnt = 0;

    /* presume we won't find a metaclass table entry for our metaclass */
    entry = 0;

    /* look at our metaclass table, if we have one */
    meta_idx = osrp2(ext_ + 2);
    if (meta_idx < G_meta_table->get_count())
    {
        /* get my metaclass table entry */
        entry = G_meta_table->get_entry(meta_idx);

        /* if we have an entry, count the properties */
        if (entry != 0)
            my_prop_cnt = list_static_props(vmg_ self, entry, 0, 0);
    }

    /* if we have a modifier object, get its property list */
    if ((mod_obj = get_mod_obj()) != VM_INVALID_OBJ)
    {
        /* get the modifier's property list - we'll add it to our own */
        vm_objp(vmg_ mod_obj)->build_prop_list(vmg_ self, &mod_val);

        /* get the result as a list object, properly cast */
        mod_lst = (CVmObjList *)vm_objp(vmg_ mod_val.val.obj);

        /*
         *   As an optimization, if we don't have any properties of our own
         *   to add to the modifier list, just return the modifier list
         *   directly (thus avoiding unnecessarily creating another copy of
         *   the list with no changes).
         */
        if (my_prop_cnt == 0)
        {
            /* the modifier list is the entire return value */
            *retval = mod_val;
            return;
        }

        /* get the size of the modifier list */
        mod_prop_cnt = vmb_get_len(mod_lst->get_as_list());
    }
    else
    {
        /* 
         *   we have no modifier object - for the result list, we simply
         *   need our own list, so set the modifier list to nil 
         */
        mod_val.set_nil();
        mod_prop_cnt = 0;
    }

    /* for gc protection, push the modifier's list */
    G_stk->push(&mod_val);

    /*
     *   Allocate a list big enough to hold the modifier's list plus our own
     *   list.  
     */
    retval->set_obj(CVmObjList::
                    create(vmg_ FALSE, my_prop_cnt + mod_prop_cnt));

    /* push the return value list for gc protection */
    G_stk->push(retval);

    /* get it as a list object, properly cast */
    lst = (CVmObjList *)vm_objp(vmg_ retval->val.obj);

    /* start the list with our own properties */
    if (entry != 0)
        list_static_props(vmg_ self, entry, lst, 0);

    /* copy the modifier list into the results, if there is a modifier list */
    if (mod_prop_cnt != 0)
        lst->cons_copy_elements(my_prop_cnt, mod_lst->get_as_list());

    /* done with the gc protection */
    G_stk->discard(2);
}

/*
 *   List my metaclass's static properties.  Returns the number of statics
 *   we found.  We'll add the properties to the given list; if the list is
 *   null, we'll simply return the count.  
 */
size_t CVmObjClass::list_static_props(VMG_ vm_obj_id_t self,
                                      vm_meta_entry_t *entry,
                                      CVmObjList *lst, size_t starting_idx)
{
    size_t i;
    size_t cnt;
    size_t idx;
    
    /* count the valid entries */
    for (i = 1, idx = starting_idx, cnt = 0 ;
         i <= entry->func_xlat_cnt_ ; ++i)
    {
        vm_prop_id_t prop;
        vm_val_t val;
        vm_obj_id_t source_obj;

        /* get this property */
        prop = entry->xlat_func(i);
        
        /* 
         *   If this one's valid, check to see if it's implemented as a
         *   static method on this intrinsic class object.
         *   
         *   To determine if the property is implemented as a static, call
         *   the property on self without an argc pointer - this is the
         *   special form of the call which merely retrieves the value of
         *   the property without evaluating it.  If the property is defined
         *   on the object, and the source of the definition is self,
         *   include this one in the list of directly-defined properties.  
         */
        if (prop != VM_INVALID_PROP
            && get_prop(vmg_ prop, &val, self, &source_obj, 0)
            && source_obj == self)
        {
            /* 
             *   it's defined statically - if we have a list, add this one
             *   to the list 
             */
            if (lst != 0)
            {
                /* set up a value containing the property pointer */
                val.set_propid(prop);

                /* add it to the list at the next free slot */
                lst->cons_set_element(idx, &val);

                /* advance to the next slot */
                ++idx;
            }

            /* count it */
            ++cnt;
        }
    }

    /* return the number of valid static properties we found */
    return cnt;
}


/* 
 *   get a property 
 */
int CVmObjClass::get_prop(VMG_ vm_prop_id_t prop, vm_val_t *val,
                          vm_obj_id_t self, vm_obj_id_t *source_obj,
                          uint *argc)
{
    uint meta_idx;
    vm_obj_id_t mod_obj;

    /* 
     *   pass the property request as a static property of the metaclass
     *   with which we're associated 
     */
    meta_idx = osrp2(ext_ + 2);
    if (meta_idx < G_meta_table->get_count())
    {
        /* call the static property in our metaclass */
        if (G_meta_table->get_entry(meta_idx)->meta_
            ->call_stat_prop(vmg_ val, 0, argc, prop))
        {
            /* the metaclass object handled it, so we are the definer */
            *source_obj = self;
            return TRUE;
        }
    }

    /* try handling it through the modifier object, if we have one */
    if ((mod_obj = get_mod_obj()) != VM_INVALID_OBJ)
    {
        /* let the modifier have a crack at it */
        if (vm_objp(vmg_ mod_obj)->get_prop(vmg_ prop, val, mod_obj,
                                            source_obj, argc))
            return TRUE;
    }

    /* inherit default handling */
    return CVmObject::get_prop(vmg_ prop, val, self, source_obj, argc);
}

/* 
 *   save to a file 
 */
void CVmObjClass::save_to_file(VMG_ class CVmFile *fp)
{
    size_t len;
    
    /* write our data */
    len = osrp2(ext_);
    fp->write_bytes(ext_, len);
}

/* 
 *   restore from a file 
 */
void CVmObjClass::restore_from_file(VMG_ vm_obj_id_t self,
                                    CVmFile *fp, CVmObjFixup *)
{
    size_t len;

    /* read the length */
    len = fp->read_uint2();

    /* free any existing extension */
    if (ext_ != 0)
    {
        G_mem->get_var_heap()->free_mem(ext_);
        ext_ = 0;
    }

    /* allocate the space */
    ext_ = (char *)G_mem->get_var_heap()->alloc_mem(len, this);

    /* store our length */
    vmb_put_len(ext_, len);

    /* 
     *   read the contents (note that we've already read the length prefix,
     *   so subtract it out of the total remaining to be read) 
     */
    fp->read_bytes(ext_ + VMB_LEN, len - VMB_LEN);

    /* register myself with the metaclass table */
    register_meta(vmg_ self);
}

/* load from an image file */
void CVmObjClass::load_from_image(VMG_ vm_obj_id_t self,
                                  const char *ptr, size_t siz)
{
    /* save a pointer to the image file data as our extension */
    ext_ = (char *)ptr;

    /* make sure the length is valid */
    if (siz < 8)
        err_throw(VMERR_INVAL_METACLASS_DATA);

    /* register myself */
    register_meta(vmg_ self);
}

/* 
 *   reset to the initial load state 
 */
void CVmObjClass::reset_to_image(VMG_ vm_obj_id_t self)
{
    /* re-register myself for the re-load */
    register_meta(vmg_ self);
}

/*
 *   Register myself with the metaclass dependency table - this establishes
 *   the association between the metaclass in the dependency table and this
 *   instance, so that the metaclass knows about the instance that
 *   represents it.  
 */
void CVmObjClass::register_meta(VMG_ vm_obj_id_t self)
{
    uint meta_idx;

    /* determine which metaclass we represent */
    meta_idx = osrp2(ext_ + 2);

    /* 
     *   if this is a valid metaclass index, store a reference to myself
     *   in the metaclass table for this index - this will let the
     *   metaclass that we represent find us when asked for its class
     *   object 
     */
    if (meta_idx < G_meta_table->get_count())
        G_meta_table->get_entry(meta_idx)->class_obj_ = self;
}

/*
 *   Get the number of superclasses 
 */
int CVmObjClass::get_superclass_count(VMG_ vm_obj_id_t) const
{
    uint meta_idx;

    /* determine which metaclass we represent */
    meta_idx = osrp2(ext_ + 2);

    /* 
     *   if this is a valid metaclass index, ask the metaclass object to
     *   tell us the superclass count 
     */
    if (meta_idx < G_meta_table->get_count())
        return G_meta_table->get_entry(meta_idx)->meta_
            ->get_supermeta_count(vmg0_);
    else
        return 0;
}

/*
 *   Get a superclass 
 */
vm_obj_id_t CVmObjClass::get_superclass(VMG_ vm_obj_id_t,
                                        int sc_idx) const
{
    uint meta_idx;

    /* determine which metaclass we represent */
    meta_idx = osrp2(ext_ + 2);

    /* 
     *   if this is a valid metaclass index, ask the metaclass object to
     *   retrieve the superclass
     */
    if (meta_idx < G_meta_table->get_count())
        return G_meta_table->get_entry(meta_idx)->meta_
            ->get_supermeta(vmg_ sc_idx);
    else
        return VM_INVALID_OBJ;
}

/*
 *   Determine if I'm an instance of the given object 
 */
int CVmObjClass::is_instance_of(VMG_ vm_obj_id_t obj)
{
    uint meta_idx;

    /* determine which metaclass we represent */
    meta_idx = osrp2(ext_ + 2);

    /* if this is a valid metaclass index, ask the metaclass object */
    if (meta_idx < G_meta_table->get_count())
    {
        /* ask the metaclass if it's an instance of the given object */
        return G_meta_table->get_entry(meta_idx)->meta_
            ->is_meta_instance_of(vmg_ obj);
    }
    else
    {
        /* not a valid metaclass - we can't make sense of this */
        return FALSE;
    }
}

