#charset "us-ascii"

/* Copyright (c) 2000, 2002 by Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - Thing
 *   
 *   This module defines Thing, the base class for physical objects in the
 *   simulation.  We also define some utility classes that Thing uses
 *   internally.  
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   Sense Information entry.  Thing.senseInfoTable() returns a list of
 *   these objects to provide full sensory detail on the objects within
 *   range of a sense.  
 */
class SenseInfo: object
    construct(obj, trans, obstructor, ambient)
    {
        /* set the object being described */
        self.obj = obj;

        /* remember the transparency and obstructor */
        self.trans = trans;
        self.obstructor = obstructor;

        /* 
         *   set the energy level, as seen from the point of view - adjust
         *   the level by the transparency 
         */
        self.ambient = (ambient != nil
                        ? adjustBrightness(ambient, trans)
                        : nil);
    }
    
    /* the object being sensed */
    obj = nil

    /* the transparency from the point of view to this object */
    trans = nil

    /* the obstructor that introduces a non-transparent value of trans */
    obstructor = nil

    /* the ambient sense energy level at this object */
    ambient = nil
;

/*
 *   Can-Touch information.  This object keeps track of whether or not a
 *   given object is able to reach out and touch another object. 
 */
class CanTouchInfo: object
    /* construct, given the touch path */
    construct(path) { touchPath = path; }
    
    /* the full reach-and-touch path from the source to the target */
    touchPath = nil

    /* 
     *   if we have calculated whether or not the source can touch the
     *   target, we'll set the property canTouch to nil or true
     *   accordingly; if this property is left undefined, this information
     *   has never been calculated 
     */
    // canTouch = nil
;

/*
 *   Given a sense information table (a LookupTable returned from
 *   Thing.senseInfoTable()), return a vector of only those objects in the
 *   table that match the given criteria.
 *   
 *   'func' is a function that takes two arguments, func(obj, info), where
 *   'obj' is a simulation object and 'info' is the corresponding
 *   SenseInfo object.  This function is invoked for each object in the
 *   sense info table; if 'func' returns true, then 'obj' is part of the
 *   list that we return.
 *   
 *   The return value is a simple vector of game objects.  (Note that
 *   SenseInfo objects are not returned - just the simulation objects.)  
 */
senseInfoTableSubset(senseTab, func)
{
    local vec;
    
    /* set up a vector for the return list */
    vec = new Vector(32);

    /* scan the table for objects matching criteria given by 'func' */
    senseTab.forEachAssoc(new function(obj, info)
    {
        /* if the function accepts this object, include it in the vector */
        if ((func)(obj, info))
            vec.append(obj);
    });

    /* return the result vector */
    return vec;
}

/* ------------------------------------------------------------------------ */
/*
 *   Command check status object.  This is an abstract object that we use
 *   in to report results from a check of various kinds.
 *   
 *   The purpose of this object is to consolidate the code for certain
 *   kinds of command checks into a single routine that can be used for
 *   different purposes - verification, selection from multiple
 *   possibilities (such as multiple paths), and command action
 *   processing.  This object encapsulates a status - success or failure -
 *   and, when the status is failure, a message giving the reason for the
 *   failure.
 */
class CheckStatus: object
    /* did the check succeed or fail? */
    isSuccess = nil

    /* 
     *   the message property or string, and parameters, for failure -
     *   this is for use with reportFailure or the like 
     */
    msgProp = nil
    msgParams = []
;

/* 
 *   Success status object.  Note that this is a single object, not a
 *   class - there's no distinct information per success indicator, so we
 *   only need this single success indicator for all uses. 
 */
checkStatusSuccess: CheckStatus
    isSuccess = true
;

/* 
 *   Failure status object.  Unlike the success indicator, this is a
 *   class, because we need to keep track of the separate failure message
 *   for each kind of failure.  
 */
class CheckStatusFailure: CheckStatus
    construct(prop, [params])
    {
        isSuccess = nil;
        msgProp = prop;
        msgParams = params;
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Equivalent group state information.  This keeps track of a state and
 *   the number of items in that state when we're listing a group of
 *   equivalent items in different states.  
 */
class EquivalentStateInfo: object
    construct(obj, nameProp)
    {
        /* remember the state object and the name property to display */
        stateObj = obj;
        stateNameProp = nameProp;

        /* this is the first one in this state */
        stateCount = 1;
    }

    /* get the name to use for listing purposes */
    name = (stateObj.(stateNameProp))

    /* the ThingState object describing the state */
    stateObj = nil

    /* the property to evaluate to get the name for listing purposes */
    stateNameProp = nil

    /* the number of items in this state */
    stateCount = 0
;

/* ------------------------------------------------------------------------ */
/*
 *   "State" of a Thing.  This is an object abstractly describing the
 *   state of an object that can assume different states.
 *   
 *   The 'listName', 'inventoryName', and 'wornName' give the names of
 *   state as displayed in room/contents listings, inventory listings, and
 *   listings of items being worn by an actor.  This state name is
 *   displayed along with the item name (usually parenthetically after the
 *   item name, but the exact nature of the display is controlled by the
 *   language-specific part of the library).
 *   
 *   The 'listingOrder' is an integer giving the listing order of this
 *   state relative to other states of the same kind of object.  When we
 *   show a list of equivalent items in different states, we'll order the
 *   state names in ascending order of listingOrder.  
 */
class ThingState: object
    /* 
     *   The name of the state to use in ordinary room/object contents
     *   listings.  If the name is nil, no extra state information is
     *   shown in a listing for an object in this state.  (It's often
     *   desirable to leave the most ordinary state an object can be in
     *   unnamed, to avoid belaboring the obvious.  For example, a match
     *   that isn't burning would probably not want to mention "(not lit)"
     *   every time it's listed.)  
     */
    listName = nil

    /* 
     *   The state name to use in inventory lists.  By default, we just
     *   use the base name. 
     */
    inventoryName = (listName)

    /* 
     *   The state name to use in listings of items being worn.  By
     *   default, we just use the base name.
     */
    wornName = (listName)

    /* the relative listing order */
    listingOrder = 0

    /*
     *   Match the name of an object in this state.  'obj' is the object
     *   to be matched; 'origTokens' and 'adjustedTokens' have the same
     *   meanings they do for Thing.matchName; and 'states' is a list of
     *   all of the possible states the object can assume.
     *   
     *   Implementation of this is always language-specific.  In most
     *   cases, this should do something along the lines of checking for
     *   the presence (in the token list) of words that only apply to
     *   other states, rejecting the match if any such words are found.
     *   For example, the ThingState object representing the unlit state
     *   of a light source might check for the presence of 'lit' as an
     *   adjective, and reject the object if it's found.  
     */
    matchName(obj, origTokens, adjustedTokens, states)
    {
        /* by default, simply match the object */
        return obj;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Object with vocabulary.  This is the base class for any object that
 *   can define vocabulary words.  
 */
class VocabObject: object
    /*
     *   Match a name as used in a noun phrase in a player's command to
     *   this object.  The parser calls this routine to test this object
     *   for a match to a noun phrase when all of the following conditions
     *   are true:
     *   
     *   - this object is in scope;
     *   
     *   - our vocabulary matches the noun phrase, which means that ALL of
     *   the words in the player's noun phrase are associated with this
     *   object with the corresponding parts of speech.  Note the special
     *   wildcard vocabulary words: '#' as an adjective matches any number
     *   used as an adjective; '*' as a noun matches any word used as any
     *   part of speech.
     *   
     *   'origTokens' is the list of the original input words making up
     *   the noun phrase, in canonical tokenizer format.  Each element of
     *   this list is a sublist representing one token.
     *   
     *   'adjustedTokens' is the "adjusted" token list, which provides
     *   more information on how the parser is analyzing the phrase but
     *   may not contain the exact original tokens of the command.  In the
     *   adjusted list, the tokens are represented by pairs of values in
     *   the list: the first value of each pair is a string giving the
     *   adjusted token text, and the second value of the pair is a
     *   property ID giving the part of speech of the parser's
     *   interpretation of the phrase.  For example, if the noun phrase is
     *   "red book", the list might look like ['red', &adjective, 'book',
     *   &noun].
     *   
     *   The adjusted token list in some cases contains different tokens
     *   than the original input.  For example, when the command contains
     *   a spelled-out number, the parser will translate the spelled-out
     *   number to a numeral format and provide the numeral string in the
     *   adjusted token list: 'a hundred and thirty-four' will become
     *   '134' in the adjusted token list.
     *   
     *   If this object does not match the noun phrase, this routine
     *   returns nil.  If the object is a match, the routine returns
     *   'self'.  The routine can also return a different object, or even
     *   a list of objects - in this case, the parser will consider the
     *   noun phrase to have matched the returned object or objects rather
     *   than this original match.
     *   
     *   By default, we simply return 'self'.  Note that, by default, it
     *   is not even necessary to check that the input tokens match our
     *   vocabulary, because the parser will already have done that for us
     *   - this routine is only called after the parser has already
     *   determined that all of the noun phrase's words match ours.  
     */
    matchName(origTokens, adjustedTokens)
    {
        local st;
        
        /* 
         *   if we have a state, ask our state object to check for words
         *   applying only to other states 
         */
        if ((st = getState()) != nil)
            return st.matchName(self, origTokens, adjustedTokens, allStates);

        /* by default, accept the parser's determination that we match */
        return self;
    }

    /*
     *   Match a name in a disambiguation response.  This is similar to
     *   matchName(), but is called for each object in an ambiguous object
     *   list for which a disambiguation response was provided.  As with
     *   matchName(), we only call this routine for objects that match the
     *   dictionary vocabulary.
     *   
     *   This routine is separate from matchName() because a
     *   disambiguation response usually only contains a partial name.
     *   For example, the exchange might go something like this:
     *   
     *   >take box
     *.  Which box do you mean, the cardboard box, or the wood box?
     *   >cardboard
     *   
     *   Note that it is not safe to assume that the disambiguation
     *   response can be prepended to the original noun phrase to make a
     *   complete noun phrase; if this were safe, we'd simply concatenate
     *   the two strings and call matchName().  This would work for the
     *   example above, since we'd get "cardboard box" as the new noun
     *   phrase, but it wouldn't work in general.  Consider these examples:
     *   
     *   >open post office box
     *.  Which post office box do you mean, box 100, box 101, or box 102?
     *   
     *   >take jewel
     *.  Which jewel do you mean, the emerald, or the diamond?
     *   
     *   There's no general way of assembling the disambiguation response
     *   and the original noun phrase together into a new noun phrase, so
     *   rather than trying to use matchName() for both purposes, we
     *   simply use a separate routine to match the disambiguation name.
     *   
     *   Note that, when this routine is called, this object will have
     *   been previously matched with matchName(), so there is no question
     *   that this object matches the original noun phrase.  The only
     *   question is whether or not this object matches the response to
     *   the "which one do you mean" question.
     *   
     *   The return value has the same meaning as for matchName().  
     */
    matchNameDisambig(origTokens, adjustedTokens)
    {
        /* by default, use the same processing as matchName */
        return matchName(origTokens, adjustedTokens);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Thing: the basic class for game objects.  An object of this class
 *   represents a physical object in the simulation.
 */
class Thing: VocabObject
    /*
     *   "Special" description.  This is the generic place to put a
     *   description of the object that should appear in the containing
     *   room's full description.  If the object defines a special
     *   description, the object is NOT listed in the basic contents list
     *   of the room, because listing it with the contents would be
     *   redundant with the special description.
     *   
     *   By default, we have no special description.  If a special
     *   description is desired, define this to a double-quoted string
     *   containing the special description, or to a method that displays
     *   the special description.  
     */
    specialDesc = nil

    /*
     *   The special descriptions to use under obscured and distant
     *   viewing conditions.  By default, these simply display the normal
     *   special description, but these can be overridden if desired to
     *   show different messages under these viewing conditions. 
     */
    distantSpecialDesc = (specialDesc)
    obscuredSpecialDesc = (specialDesc)

    /*
     *   List order for the special description.  Whenever there's more
     *   than one object showing a specialDesc at the same time (in a
     *   single room description, for example), we'll use this to order the
     *   specialDesc displays.  We'll display in ascending order of this
     *   value.  By default, we use the same value for everything, so
     *   listing order is arbitrary; when one specialDesc should appear
     *   before or after another, this property can be used to control the
     *   relative ordering.  
     */
    specialDescOrder = 100

    /*
     *   Show my special description, given a SenseInfo object for the
     *   visual sense path from the point of view of the description.  
     */
    showSpecialDescWithInfo(info)
    {
        /* determine what to show, based on the sense path */
        if (canDetailsBeSensed(sight, info))
        {
            /* 
             *   our details can be sensed, so show our special
             *   description 
             */
            showSpecialDesc();
        }
        else if (info.trans == obscured)
        {
            /* 
             *   our details cannot be sensed due to an obscured sight
             *   path; show our obscured special description 
             */
            showObscuredSpecialDesc();
        }
        else if (info.trans == distant)
        {
            /* 
             *   our details cannot be sensed due to a distant sight path;
             *   show our distant special description 
             */
            showDistantSpecialDesc();
        }
    }

    /*
     *   Show the special description, if we have one.  If we are using
     *   our initial description, we'll show that; otherwise, if we have a
     *   specialDesc property, we'll show that.
     *   
     *   Note that the initial description overrides the specialDesc
     *   property whenever useInitDesc() returns true.  This allows an
     *   object to have both an initial description that is used until the
     *   object is moved, and a separate special description used
     *   thereafter.  
     */
    showSpecialDesc()
    {
        /* 
         *   if we are to use our initial description, show that;
         *   otherwise, show our special description 
         */
        if (useInitDesc())
            initDesc;
        else
            specialDesc;
    }

    /* show the special description under obscured viewing conditions */
    showObscuredSpecialDesc()
    {
        if (useInitDesc())
            obscuredInitDesc;
        else
            obscuredSpecialDesc;
    }

    /* show the special description under distant viewing conditions */
    showDistantSpecialDesc()
    {
        if (useInitDesc())
            distantInitDesc;
        else
            distantSpecialDesc;
    }

    /*
     *   Determine if we should use a special description.  By default, we
     *   have a special description if we have either a non-nil
     *   specialDesc property, or we have an initial description.  
     */
    useSpecialDesc()
    {
        /* 
         *   if we have a non-nil specialDesc, or we are to use an initial
         *   description, we have a special description 
         */
        return propType(&specialDesc) != TypeNil || useInitDesc();
    }
    

    /*
     *   If we define a non-nil initDesc, this property will be called to
     *   describe the object in room listings until the object is first
     *   moved to a new location.  By default, objects don't have initial
     *   descriptions.  
     */
    initDesc = nil

    /*
     *   The initial descriptions to use under obscured and distant
     *   viewing conditions.  By default, these simply show the plain
     *   initDesc; these can be overridden, if desired, to show
     *   alternative messages when viewing conditions are less than ideal.
     *   
     *   Note that in order for one of these alternative initial
     *   descriptions to be shown, the regular initDesc MUST be defined,
     *   even if it's never actually used.  We make the decision to
     *   display these other descriptions based on the existence of a
     *   non-nil initDesc, so always define initDesc whenever these are
     *   defined.  
     */
    obscuredInitDesc = (initDesc)
    distantInitDesc = (initDesc)

    /*
     *   If we define a non-nil initExamineDesc, and the object has never
     *   been moved, this property will be called to describe the object,
     *   rather than the normal "desc" property, to describe the object if
     *   examined.  This can be used to customize the description the
     *   player sees in parallel to initDesc.  
     */
    initExamineDesc = nil

    /*
     *   Determine if I should be described using my initial description.
     *   This returns true if I have an initial description that isn't
     *   nil, and I have never been moved out of my initial location.  If
     *   this returns nil, the object should be described in room
     *   descriptions using the ordinary generated message.  
     */
    useInitDesc()
    {
        return !moved && propType(&initDesc) != TypeNil;
    }

    /*
     *   Flag: I've been moved out of my initial location.  Whenever we
     *   move the object to a new location, we'll set this to true.  
     */
    moved = nil

    /*
     *   Flag: I've been seen by the player character.  This is nil by
     *   default; we set this to true whenever we mention the object's
     *   name in a room description or in the description of the contents
     *   list of another object, as long as the player character is the
     *   point-of-view character for the description.  
     */
    seen = nil

    /*
     *   Flag: I've been desribed.  This is nil by default; we set this to
     *   true whenever the player explicitly examines the object. 
     */
    described = nil

    /*
     *   Determine if a given actor has seen this object.  By default, we
     *   simply return our 'seen' flag, which keeps track of whether or not
     *   the player character has seen the object.  Some games might want
     *   to track which actors have seen which object individually; we make
     *   this a separate method specifically to allow for such
     *   modification.  
     */
    seenBy(actor) { return seen; }

    /*
     *   Set my 'seen' flag for the given actor.  By default, we only track
     *   the 'seen' status for the player character, so this simply sets
     *   the 'seen' property to the given value if the actor is the player
     *   character, and does nothing if the actor is an NPC.  Games that
     *   want to track who has seen which objects individually by actor can
     *   override this (and seenBy) to keep more detailed information.  
     */
    setSeenBy(actor, val)
    {
        /* if the player character is doing the looking, mark me as seen */
        if (actor != nil && actor.isPlayerChar())
            seen = val;
    }

    /*
     *   Hide from 'all'.  If this returns true, this object will never be
     *   included in 'all' lists for the given action class.  This should
     *   generally be set only for objects that serve some sort of
     *   internal purpose and don't represent physical objects in the
     *   model world.  By default, objects are not hidden from 'all' lists.
     *   
     *   Note that returning nil doesn't put the object into an 'all'
     *   list.  Rather, it simply *leaves* it in any 'all' list it should
     *   happen to be in.  Each action controls its own selection criteria
     *   for 'all', and different verbs use different criteria.  No matter
     *   how an action chooses its 'all' list, though, an item will always
     *   be excluded if hideFromAll() returns true for the item.  
     */
    hideFromAll(action) { return nil; }
    
    /*
     *   Determine if I'm to be listed at all in my room's description.
     *   Most objects should be listed normally, but some types of objects
     *   should be suppressed from the normal room listing.  For example,
     *   fixed-in-place scenery objects are generally described in the
     *   custom message for the containing room, so these are normally
     *   omitted from the listing of the room's contents.
     *   
     *   By default, we'll return true, unless we are using our special
     *   description; we return nil if our special description is to be
     *   used because the special description would be redundant with the
     *   default listing.
     *   
     *   Individual objects are free to override this as needed to control
     *   their listing status.  
     */
    isListed { return !useSpecialDesc(); }

    /*
     *   Determine if I'm listed in explicit "examine" and "look in"
     *   descriptions of my container.  By default, we return true as long
     *   as we're not using our special description - examining or looking
     *   in a container will normally show special message for any
     *   contents of the container, so we don't want to list the items
     *   with special descriptions in the ordinary list as well.  
     */
    isListedInContents { return !useSpecialDesc(); }

    /*
     *   Determine if I'm listed in inventory listings.  By default, we
     *   include every item in an inventory list.
     */
    isListedInInventory { return true; }

    /* 
     *   by default, regular objects are not listed when they arrive
     *   aboard vehicles (only actors are normally listed in this fashion) 
     */
    isListedAboardVehicle = nil

    /*
     *   Determine if I'm listed as being located in the given room part.
     *   If we're not listed normally, we won't be listed in a room part.
     *   Otherwise, if our "nominal room part location" is the same as the
     *   given part, we'll return true to be listed as one of the part's
     *   contents; if not, we'll return nil.  
     */
    isListedInRoomPart(part)
    {
        /* if I'm not listed normally, don't list me in a room part */
        if (!isListed)
            return nil;

        /* list me if I'm nominally in the part */
        return isNominallyInRoomPart(part);
    }

    /*
     *   Determine if we're nominally in the given room part (floor,
     *   ceiling, wall, etc).  This returns true if we *appear* to be
     *   located directly in/on the given room part object.
     *   
     *   In most cases, a portable object might start out with a special
     *   initial room part location, but once moved and then dropped
     *   somewhere, ends up nominally in the nominal drop destination of
     *   the location where it was dropped.  For example, a poster might
     *   start out being nominally attached to a wall, or a light bulb
     *   might be nominally hanging from the ceiling; if these objects are
     *   taken and then dropped somewhere, they'll simply end up on the
     *   floor.
     *   
     *   Our default behavior models this.  If we've never been moved,
     *   we'll indicate that we're in our initial room part location,
     *   given by initNominalRoomPartLocation.  Once we've been moved (or
     *   if our initNominalRoomPartLocation is nil), we'll figure out what
     *   the nominal drop destination is for our current container, and
     *   then see if we appear to be in that nominal drop destination.  
     */
    isNominallyInRoomPart(part)
    {
        /* 
         *   If we're not directly in the apparent container of the given
         *   room part, then we certainly are not even nominally in the
         *   room part.  We only appear to be in a room part when we're
         *   actually in the room directly containing the room part. 
         */
        if (location == nil
            || location.getRoomPartLocation(part) != location)
            return nil;

        /* if we've never been moved, use our initial room part location */
        if (!moved && initNominalRoomPartLocation != nil)
            return (part == initNominalRoomPartLocation);

        /* if we have an explicit special room part location, use that */
        if (specialNominalRoomPartLocation != nil)
            return (part == specialNominalRoomPartLocation);

        /*
         *   We don't seem to have an initial or special room part
         *   location, but there's still one more possibility: if the room
         *   part is the nominal drop destination, then we are by default
         *   nominally in that room part, because that's where we
         *   otherwise end up in the absence of any special room part
         *   location setting.  
         */
        if (location.getNominalDropDestination() == part)
            return true;

        /* we aren't nominally in the given room part */
        return nil;
    }

    /*
     *   Determine if I should show my special description with the
     *   description of the given room part (floor, ceiling, wall, etc).
     *   
     *   By default, we'll include our special description with a room
     *   part's description if either (1) we are using our initial
     *   description, and our initNominalRoomPartLocation is the given
     *   part; or (2) we are using our special description, and our
     *   specialNominalRoomPartLocation is the given part.
     *   
     *   Note that, by default, we do NOT use our special description for
     *   the "default" room part location - that is, for the nominal drop
     *   destination for our containing room, which is where we end up by
     *   default, in the absence of an initial or special room part
     *   location setting.  We don't use our special description in this
     *   default location because special descriptions are most frequently
     *   used to describe an object that is specially situated, and hence
     *   we don't want to assume a default situation.  
     */
    useSpecialDescInRoomPart(part)
    {
        /* if we're not nominally in the part, rule it out */
        if (!isNominallyInRoomPart(part))
            return nil;

        /* 
         *   if we're using our initial description, and we are explicitly
         *   in the given room part initially, use the special description
         *   for the room part 
         */
        if (useInitDesc() && initNominalRoomPartLocation == part)
            return true;

        /* 
         *   if we're using our special description, and we are explicitly
         *   in the given room part as our special location, use the
         *   special description 
         */
        if (useSpecialDesc() && specialNominalRoomPartLocation == part)
            return true;

        /* do not use the special description in the room part */
        return nil;
    }

    /* 
     *   Our initial room part location.  By default, we set this to nil,
     *   which means that we'll use the nominal drop destination of our
     *   actual initial location when asked.  If desired, this can be set
     *   to another part; for example, if a poster is initially described
     *   as being "on the north wall," this should set to the default north
     *   wall object.  
     */
    initNominalRoomPartLocation = nil

    /*
     *   Our "special" room part location.  By default, we set this to
     *   nil, which means that we'll use the nominal drop destination of
     *   our actual current location when asked.
     *   
     *   This property has a function similar to
     *   initNominalRoomPartLocation, but is used to describe the nominal
     *   room part container of the object has been moved (and even before
     *   it's been moved, if initNominalRoomPartLocation is nil).
     *   
     *   It's rare for an object to have a special room part location
     *   after it's been moved, because most games simply don't provide
     *   commands for things like re-attaching a poster to a wall or
     *   re-hanging a fan from the ceiling.  When it is possible to move
     *   an object to a new special location, though, this property can be
     *   used to flag its new special location.  
     */
    specialNominalRoomPartLocation = nil

    /*
     *   Determine if my contents are to be listed when I'm shown in a
     *   listing - in a room description, inventory list, or when examining
     *   my own container.  By default, my contents will be listed if and
     *   only if I'm listed.
     *   
     *   Note that this doesn't affect listing my contents when I'm
     *   directly examined, because directly examining an object by default
     *   lists its direct listable contents regardless of this setting.  
     */
    contentsListed { return isListed; }

    /*
     *   Determine if my contents are listed separately from my own list
     *   entry.  If this is true, then my contents will be listed in a
     *   separate sentence from my own listing; for example, if 'self' is a
     *   box, we'll be listed like so:
     *   
     *.    You are carrying a box. The box contains a key and a book.
     *   
     *   By default, this is nil, which means that we list our contents
     *   parenthetically after our name:
     *   
     *.     You are carrying a box (which contains a key and a book).
     *   
     *   Using a separate listing is sometimes desirable for an object that
     *   will routinely contain listable objects that in turn have listable
     *   contents of their own, because it can help break up a long listing
     *   that would otherwise use too many nested parentheticals.
     *   
     *   Note that this only applies to "wide" listings; "tall" listings
     *   will show contents in the indented tree format regardless of this
     *   setting.  
     */
    contentsListedSeparately = nil

    /*
     *   The language-dependent part of the library must provide a couple
     *   of basic descriptor methods that we depend upon.  We provide
     *   several descriptor methods that we base on the basic methods.
     *   The basic methods that must be provided by the language extension
     *   are:
     *   
     *   listName - return a string giving the "list name" of the object;
     *   that is, the name that should appear in inventory and room
     *   contents lists.  This is called with the visual sense info set
     *   according to the point of view.
     *   
     *   countName(count) - return a string giving the "counted name" of
     *   the object; that is, the name as it should appear with the given
     *   quantifier.  In English, this might return something like 'one
     *   box' or 'three books'.  This is called with the visual sense info
     *   set according to the point of view.  
     */

    /*
     *   Show this item as part of a list.  'options' is a combination of
     *   ListXxx flags indicating the type of listing.  'infoTab' is a
     *   lookup table of SenseInfo objects giving the sense information
     *   for the point of view.  
     */
    showListItem(options, pov, infoTab)
    {
        /* show the list, using the 'listName' of the state */
        showListItemGen(options, pov, infoTab, &listName);
    }

    /*
     *   General routine to show the item as part of a list.
     *   'stateNameProp' is the property to use in any listing state
     *   object to obtain the state name.  
     */
    showListItemGen(options, pov, infoTab, stateNameProp)
    {
        local info;
        local st;

        /* get my visual information from the point of view */
        info = infoTab[self];

        /* show the item's list name */
        say(withVisualSenseInfo(pov, info, &listName));

        /* if we have a list state with a name, show it */
        if ((st = getStateWithInfo(info)) != nil
            && st.(stateNameProp) != nil)
            libMessages.showListState(st.(stateNameProp));
    }

    /*
     *   Show this item as part of a list, grouped with a count of
     *   list-equivalent items.
     */
    showListItemCounted(lst, options, pov, infoTab)
    {
        /* show the item using the 'listName' property of the state */
        showListItemCountedGen(lst, options, pov, infoTab, &listName);
    }

    /* 
     *   General routine to show this item as part of a list, grouped with
     *   a count of list-equivalent items.  'stateNameProp' is the
     *   property of any list state object that we should use to obtain
     *   the name of the listing state.  
     */
    showListItemCountedGen(lst, options, pov, infoTab, stateNameProp)
    {
        local info;
        local stateList;
        
        /* get the visual information from the point of view */
        info = infoTab[self];
        
        /* show our counted name */
        say(countListName(lst.length(), pov, info));

        /* check for list states */
        stateList = new Vector(10);
        foreach (local cur in lst)
        {
            local st;

            /* get this item's sense information from the list */
            info = infoTab[cur];
            
            /* 
             *   if this item has a list state with a name, include it in
             *   our list of states to show 
             */
            if ((st = cur.getStateWithInfo(info)) != nil
                && st.(stateNameProp) != nil)
            {
                local stInfo;

                /* 
                 *   if this state is already in our list, simply
                 *   increment the count of items in this state;
                 *   otherwise, add the new state to the list 
                 */
                stInfo = stateList.valWhich({x: x.stateObj == st});
                if (stInfo != nil)
                {
                    /* it's already in the list - count the new item */
                    stInfo.stateCount++;
                }
                else
                {
                    /* it's not in the list - add it */
                    stateList.append(
                        new EquivalentStateInfo(st, stateNameProp));
                }
            }
        }

        /* if the state list is non-empty, show it */
        if (stateList.length() != 0)
        {
            /* put the state list in its desired order */
            stateList.sort(SortAsc, {a, b: (a.stateObj.listingOrder
                                            - b.stateObj.listingOrder)});
                                           
            /*
             *   If there's only one item in the state list, and all of
             *   the objects are in that state, then show the "all in
             *   state" message.  Otherwise, show the list of states with
             *   counts.
             *   
             *   (Note that it's possible to have just one state in the
             *   list without having all of the objects in this state,
             *   because we could have some of the objects in an unnamed
             *   state, in which case they wouldn't contribute to the list
             *   at all.)  
             */
            if (stateList.length() == 1
                && stateList[1].stateCount == lst.length())
            {
                /* everything is in the same state */
                libMessages.allInSameListState(stateList[1].stateCount,
                                               stateList[1].name);
            }
            else
            {
                /* list the state items */
                equivalentStateLister.showListAll(stateList.toList(), 0, 0);
            }
        }
    }

    /*
     *   Single-item counted listing description.  This is used to display
     *   an item with a count of equivalent items ("four gold coins").
     *   'info' is the sense information from the current point of view
     *   for 'self', which we take to be representative of the sense
     *   information for all of the equivalent items.  
     */
    countListName(equivCount, pov, info)
    {
        return withVisualSenseInfo(pov, info, &countName, equivCount);
    }

    /*
     *   Show this item as part of an inventory list.  By default, we'll
     *   show the regular list item name.  
     */
    showInventoryItem(options, pov, infoTab)
    {
        /* show the item, using the inventory state name */
        showListItemGen(options, pov, infoTab, &inventoryName);
    }
    showInventoryItemCounted(lst, options, pov, infoTab)
    {
        /* show the item, using the inventory state name */
        showListItemCountedGen(lst, options, pov, infoTab, &inventoryName);
    }

    /*
     *   Show this item as part of a list of items being worn.
     */
    showWornItem(options, pov, infoTab)
    {
        /* show the item, using the worn-listing state name */
        showListItemGen(options, pov, infoTab, &wornName);
    }
    showWornItemCounted(lst, options, pov, infoTab)
    {
        /* show the item, using the worn-listing state name */
        showListItemCountedGen(lst, options, pov, infoTab, &wornName);
    }

    /*
     *   Get the "listing state" of the object, given the visual sense
     *   information for the object from the point of view for which we're
     *   generating the listing.  This returns a ThingState object
     *   describing the object's state for the purposes of listings.  This
     *   should return nil if the object doesn't have varying states for
     *   listings.
     *   
     *   By default, we return a list state if the visual sense path is
     *   transparent or attenuated, or we have large visual scale.  In
     *   other cases, we assume that the details of the object are not
     *   visible under the current sense conditions; since the list state
     *   is normally a detail of the object, we don't return a list state
     *   when the details of the object are not visible.  
     */
    getStateWithInfo(info)
    {
        /* 
         *   if our details can be sensed, return the list state;
         *   otherwise, return nil, since the list state is a detail 
         */
        if (canDetailsBeSensed(sight, info))
            return getState();
        else
            return nil;
    }

    /* 
     *   Get our state - returns a ThingState object describing the state.
     *   By default, we don't have varying states, so we simply return
     *   nil.  
     */
    getState = nil

    /*
     *   Get a list of all of our possible states.  For an object that can
     *   assume varying states as represented by getState, this should
     *   return the list of all possible states that the object can
     *   assume.  
     */
    allStates = []

    /*
     *   The default long description, which is displayed in response to
     *   an explicit player request to examine the object.  We'll use a
     *   generic library message; most objects should override this to
     *   customize the object's desription.
     *   
     *   Note that we show this as a "default descriptive report," because
     *   this default message indicates merely that there's nothing
     *   special to say about the object.  If we generate any additional
     *   description messages, such as status reports ("it's open" or "it
     *   contains a gold key") or special descriptions for things inside,
     *   we clearly *do* have something special to say about the object,
     *   so we'll want to suppress the nothing-special message.  To
     *   accomplish this suppression, all we have to do is report our
     *   generic description as a default descriptive report, and the
     *   transcript will automatically filter it out if there are any
     *   other reports for this same action.
     *   
     *   Note that any time this is overridden by an object with any sort
     *   of actual description, the override should NOT use
     *   defaultDescReport.  Instead, simply set this to display the
     *   descriptive message directly:
     *   
     *   desc = "It's a big green box. " 
     */
    desc { defaultDescReport(&thingDesc, self); }

    /*
     *   The default "distant" description.  This is displayed when an
     *   actor explicitly examines this object from a point of view where
     *   we have a "distant" sight path to the object, but only if our
     *   sightSize is not 'large'.  If our sightSize is 'large', we'll
     *   always show our regular description ('desc'), even at a distance,
     *   because we are so large that details are visible even at a
     *   distance.  
     */
    distantDesc { libMessages.distantThingDesc(self); }

    /*
     *   The default "obscured" description.  This is displayed when an
     *   actor explicitly examines this object from a point of view where
     *   we have an "obscured" sight path to the object, but only if our
     *   sightSize is not 'large'.  If our sightSize is 'large', we'll
     *   always show our regular description ('desc'), even when obscured,
     *   because we're so large that details are visible even under
     *   obscured conditions. 
     */
    obscuredDesc(obs) { libMessages.obscuredThingDesc(self, obs); }

    /* 
     *   The "sound description," which is the description displayed when
     *   an actor explicitly listens to the object.  This is used when we
     *   have a transparent sense path and no associated "emanation"
     *   object; when we have an associated emanation object, we use its
     *   description instead of this one.  
     */
    soundDesc { libMessages.thingSoundDesc(self); }

    /* distant sound description */
    distantSoundDesc { libMessages.distantThingSoundDesc(self); }

    /* obscured sound description */
    obscuredSoundDesc(obs) { libMessages.obscuredThingSoundDesc(self, obs); }

    /*
     *   The "smell description," which is the description displayed when
     *   an actor explicitly smells the object.  This is used when we have
     *   a transparent sense path to the object, and we have no
     *   "emanation" object; when we have an associated emanation object,
     *   we use its description instead of this one.  
     */
    smellDesc { libMessages.thingSmellDesc(self); }

    /* distant smell description */
    distantSmellDesc { libMessages.distantThingSmellDesc(self); }

    /* obscured smell description */
    obscuredSmellDesc(obs) { libMessages.obscuredThingSmellDesc(self, obs); }

    /*
     *   The "taste description," which is the description displayed when
     *   an actor explicitly tastes the object.  Note that, unlike sound
     *   and smell, we don't distinguish levels of transparency or
     *   distance with taste, because tasting an object requires direct
     *   physical contact with it.  
     */
    tasteDesc { libMessages.thingTasteDesc(self); }

    /* 
     *   The "feel description," which is the description displayed when
     *   an actor explicitly feels the object.  As with taste, we don't
     *   distinguish transparency or distance. 
     */
    feelDesc { libMessages.thingFeelDesc(self); }

    /*
     *   Show the smell/sound description for the object as part of a room
     *   description.  These are displayed when the object is in the room
     *   and it has a presence in the corresponding sense.  By default,
     *   these show nothing.
     *   
     *   In most cases, regular objects don't override these, because most
     *   regular objects have no direct sensory presence of their own.
     *   Instead, a Noise or Odor is created and added to the object's
     *   direct contents, and the Noise or Odor provides the object's
     *   sense presence.  
     */
    smellHereDesc() { }
    soundHereDesc() { }

    /*
     *   "Equivalence" flag.  If this flag is set, then all objects with
     *   the same immediate superclass will be considered interchangeable;
     *   such objects will be listed collectively in messages (so we would
     *   display "five coins" rather than "a coin, a coin, a coin, a coin,
     *   and a coin"), and will be treated as equivalent in resolving noun
     *   phrases to objects in user input.
     *   
     *   By default, this property is nil, since we want most objects to
     *   be treated as unique.  
     */
    isEquivalent = nil

    /*
     *   My Distinguisher list.  This is a list of Distinguisher objects
     *   that can be used to distinguish this object from other objects.
     *   
     *   Distinguishers are listed in order of priority.  The
     *   disambiguation process looks for distinguishers capable of
     *   telling objects apart, starting with the first in the list.  The
     *   BasicDistinguisher is generally first in every object's list,
     *   because any two objects can be told apart if they come from
     *   different classes.
     *   
     *   By default, each object has the "basic" distinguisher, which
     *   tells objects apart on the basis of the "isEquivalent" property
     *   and their superclasses; and the ownership/location distinguisher,
     *   which tells objects apart based on ownership and location.  
     */
    distinguishers = [basicDistinguisher, ownershipAndLocationDistinguisher]

    /*
     *   Determine if I'm equivalent, for the purposes of command
     *   vocabulary, to given object.
     *   
     *   We'll run through our list of distinguishers and check with each
     *   one to see if it can tell us apart from the other object.  If we
     *   can find at least one distinguisher that can tell us apart, we're
     *   not equivalent.  If we have no distinguisher that can tell us
     *   apart from the other object, we're equivalent.  
     */
    isVocabEquivalent(obj)
    {
        /* 
         *   Check each distinguisher - if we can find one that can tell
         *   us apart from the other object, we're not equivalent.  If
         *   there are no distinguishers that can tell us apart, we're
         *   equivalent.  
         */
        return (distinguishers.indexWhich(
            {cur: cur.canDistinguish(self, obj)}) == nil);
    }

    /*
     *   Determine if I'm a "collective" object for the given object.  A
     *   collective object is one that has a plural name that is the same
     *   as that of one or more other items, and can be used as a single
     *   object to represent the other items in a command.  For example, a
     *   "bag of marbles" could serve as a collective for the marbles
     *   within: any time a command like "take marbles" is entered, the
     *   bag could be used as a stand-in for all of the contained marbles.
     *   
     *   In order to be a collective for some objects, an object must have
     *   vocubulary for the plural name, and must return true from this
     *   method for the collected objects.  
     */
    isCollectiveFor(obj) { return nil; }

    /*
     *   "List Group" objects.  This specifies a list of ListGroup objects
     *   that we use to list this object in object listings, such as
     *   inventory lists and room contents lists.
     *   
     *   An object can be grouped in more than one way.  When multiple
     *   groups are specified here, the order is significant:
     *   
     *   - To the extent two groups entirely overlap, which is to say that
     *   one of the pair entirely contains the other (for example, if
     *   every coin is a kind of money, then the "money" listing group
     *   would contain every object in the "coin" group, plus other
     *   objects as well: the coin group is a subset of the money group),
     *   the groups must be listed from most general to most specific (for
     *   our money/coin example, then, money would come before coin in the
     *   group list).
     *   
     *   - When two groups do not overlap, then the earlier one in our
     *   list is given higher priority.
     *   
     *   By default, we return an empty list.
     */
    listWith = []

    /* 
     *   Get the list group for my special description.  This works like
     *   the ordinary listWith, but is used to group me with other objects
     *   showing special descriptions, rather than in ordinary listings.
     */
    specialDescListWith = []

    /*
     *   Our interior room name.  This is the status line name we display
     *   when an actor is within this object and can't see out to the
     *   enclosing room.  Since we can't rely on the enclosing room's
     *   status line name if we can't see the enclosing room, we must
     *   provide one of our own.
     *   
     *   By default, we'll use our regular name.
     */
    roomName = (name)

    /*
     *   Show our interior description.  We use this to generate the long
     *   "look" description for the room when an actor is within this
     *   object and cannot see the enclosing room.
     *   
     *   Note that this is used ONLY when the actor cannot see the
     *   enclosing room - when the enclosing room is visible (because the
     *   nested room is something like a chair that doesn't enclose the
     *   actor, or can enclose the actor but is open or transparent), then
     *   we'll simply use the description of the enclosing room instead,
     *   adding a note to the short name shown at the start of the room
     *   description indicating that the actor is in the nested room.
     *   
     *   By default, we'll show the appropriate "actor here" description
     *   for the posture, so we'll say something like "You are sitting on
     *   the red chair" or "You are in the phone booth."  Instances can
     *   override this to customize the description with something more
     *   detailed, if desired.  
     */
    roomDesc { gActor.listActorPosture(gActor); }

    /* our interior room name when we're in the dark */
    roomDarkName = (libMessages.roomDarkName)

    /* show our interior description in the dark */
    roomDarkDesc { libMessages.roomDarkDesc; }
    
    /*
     *   Look around from the point of view of this object and on behalf
     *   of the given actor.  This can be used to generate a description
     *   as seen from this object by the given actor, and is suitable for
     *   cases where the actor can use this object as a sensing device. 
     */
    lookAround(actor, verbose, explicit)
    {
        /* look around from my point of view */
        lookAroundPov(actor, self, verbose, explicit);
    }

    /*
     *   Look around from within this object, looking from the given point
     *   of view and on behalf of the given actor.  In most cases, this
     *   will eventually defer to a Room or NestedRoom object.  
     */
    lookAroundPov(actor, pov, verbose, explicit)
    {
        /* 
         *   If the actor can see our immediate container, delegate this to
         *   the container.  If we don't have a container, or the actor
         *   can't see out to my container, then we must provide the
         *   description. 
         */
        if (location != nil && actor.canSee(location))
        {
            /* we can see our location, so let the location handle it */
            location.lookAroundPov(actor, pov, verbose, explicit);
        }
        else
        {
            /* 
             *   we have no location, or the actor can't see our location,
             *   so we must provide the description here 
             */
            lookAroundWithin(actor, pov, verbose, explicit);
        }
    }

    /*
     *   Provide a "room description" of the interior of this object.  This
     *   is called when an actor performs a "look around" command, and the
     *   actor is within this object, and the actor cannot see anything
     *   outside of this object; this can happen simply because we're a
     *   top-level room, but it can also happen when we're a closed opaque
     *   container or there's not enough light to see the enclosing
     *   location.
     *   
     *   'actor' is the actor doing the looking, and 'pov' is the point of
     *   view of the description.  These are usually the same, but need not
     *   be; for example, an actor could be looking at a room through a
     *   hole in the wall, in which case the pov would be the object
     *   representing the "side" of the hole in the room being described.
     *   
     *   'explicit' indicates whether or not the description was explicitly
     *   requested by the player.  If this is true, it indicates that the
     *   description was requested by a "look around" command or
     *   equivalent; if not, the description is being shown incidentally to
     *   another command, such as travel from one room to another.
     *   
     *   Note that this method must be overridden if a room uses a
     *   non-conventional contents mechanism (i.e., it doesn't store its
     *   complete set of direct contents in a 'contents' list property).  
     *   
     *   In most cases, this routine will only be called in Room and
     *   NestedRoom objects, because actors can normally only enter those
     *   types of objects.  However, it is possible to try to describe the
     *   interior of other types of objects if (1) the game allows actors
     *   to enter other types of objects, or (2) the game provides a
     *   non-actor point-of-view object, such as a video camera, that can
     *   be placed in ordinary containers and which transmit what they see
     *   for remote viewing.  
     */
    lookAroundWithin(actor, pov, verbose, explicit)
    {
        local illum;
        local infoTab;
        local info;
        local actorList;

#ifdef SENSE_CACHE
        /* turn on the sense cache while we're looking */
        libGlobal.enableSenseCache();
#endif

        /* 
         *   get a table of all of the objects that the viewer can sense
         *   in the location using sight-like senses 
         */
        infoTab = actor.visibleInfoTableFromPov(pov);

        /* get the ambient illumination at the point of view */
        info = infoTab[pov];
        if (info != nil)
        {
            /* get the ambient illumination from the info list item */
            illum = info.ambient;
        }
        else
        {
            /* the actor's not in the list, so it must be completely dark */
            illum = 0;
        }

        /* 
         *   if the actor has never seen this location before, show a
         *   verbose description even if one was not specifically requested
         */
        if (!seenBy(actor))
        {
            /* show the full description the first time we're described */
            verbose = true;
            
            /* 
             *   If we're not in the dark, note that we've been seen.
             *   Don't count this as having seen the location if we're in
             *   the dark, since we won't normally describe any detail in
             *   the dark.  
             */
            if (illum > 1)
                setSeenBy(actor, true);
        }

        /* 
         *   drop the point-of-view object from the list, to ensure that we
         *   don't list the point-of-view object doing the looking among
         *   the room's contents 
         */
        infoTab.removeElement(pov);
        
        /* display the short description */
        "<.roomname>";
        lookAroundWithinName(actor, illum);
        "<./roomname>";

        /* if we're in verbose mode, display the full description */
        if (verbose)
        {
            local specialList;
            
            /* display the full room description */
            "<.roomdesc>";
            lookAroundWithinDesc(actor, illum);
            "<./roomdesc>";

            /* 
             *   Display any special description messages for objects
             *   within the room, other than those carried by the actor.
             *   These messages are part of the verbose description rather
             *   than the portable item listing, because these messages are
             *   meant to look like part of the room's full description and
             *   thus should not be included in a non-verbose listing.
             *   
             *   Note that we only want to show objects that are currently
             *   using special descriptions AND which aren't contained in
             *   the actor doing the looking, so subset the list
             *   accordingly.  
             */
            specialList = specialDescList(
                infoTab, {obj: obj.useSpecialDesc() && !obj.isIn(actor)});
            specialDescLister.showList(actor, nil, specialList, 0, 0,
                                       infoTab, nil);
        }

        /* 
         *   Describe each visible object directly contained in the room
         *   that wants to be listed - generally, we list any mobile
         *   objects that don't have special descriptions.  We list items
         *   in all room descriptions, verbose or not, because the
         *   portable item list is more of a status display than it is a
         *   part of the full description.
         *   
         *   Note that the infoTab has sense data that will ensure that we
         *   don't show anything we shouldn't if the room is dark.  
         */
        lookAroundWithinContents(actor, illum, infoTab);

        /* describe each actor we can see */
        actorList = senseInfoTableSubset(infoTab,
            {obj, info: actorHereLister.isListed(obj)});
        actorHereLister.showList(actor, nil, actorList, 0, 0, infoTab, nil);

        /* show all of the sounds we can hear */
        lookAroundWithinSense(actor, pov, sound, roomListenLister, explicit);

        /* show all of the odors we can smell */
        lookAroundWithinSense(actor, pov, smell, roomSmellLister, explicit);

        /* show exits */
        lookAroundWithinShowExits(actor, illum);

#ifdef SENSE_CACHE
        /* turn off the sense cache now that we're done */
        libGlobal.disableSenseCache();
#endif
    }

    /*
     *   Display the "status line" name of the room.  This is normally a
     *   brief, single-line description.
     *   
     *   By long-standing convention, each location in a game usually has a
     *   distinctive name that's displayed here.  Players usually find
     *   these names helpful in forming a mental map of the game.
     *   
     *   By default, if we have an enclosing location, and the actor can
     *   see the enclosing location, we'll defer to the location.
     *   Otherwise, we'll display our roo interior name.  
     */
    statusName(actor)
    {
        /* 
         *   use the enclosing location's status name if there is an
         *   enclosing location and its visible; otherwise, show our
         *   interior room name 
         */
        if (location != nil && actor.canSee(location))
            location.statusName(actor);
        else
            lookAroundWithinName(actor, actor.getVisualAmbient());
    }

    /*
     *   Show my name for a "look around" command.  This is used to display
     *   the location name when we're providing the room description.
     *   'illum' is the ambient visual sense level at the point of view.
     *   
     *   By default, we show our interior room name or interior dark room
     *   name, as appropriate to the ambient light level at the point of
     *   view.  
     */
    lookAroundWithinName(actor, illum)
    {
        /* 
         *   if there's light, show our name; otherwise, show our
         *   in-the-dark name 
         */
        if (illum > 1)
        {
            /* we can see, so show our interior description */
            "\^<<roomName>>";
        }
        else
        {
            /* we're in the dark, so show our default dark name */
            say(roomDarkName);
        }

        /* show the actor's posture as part of the description */
        actor.statusPosture(self);
    }

    /*
     *   Show our "look around" long description.  This is used to display
     *   the location's full description when we're providing the room
     *   description. 
     */
    lookAroundWithinDesc(actor, illum)
    {
        /* 
         *   if we have any light, show our interior room description;
         *   otherwise, show our in-the-dark description 
         */
        if (illum > 1)
        {
            /* we can see, so show our interior room description */
            roomDesc;
        }
        else
        {
            /* it's too dark to see, so show our in-the-dark description */
            roomDarkDesc;
        }
    }

    /*
     *   Show my room contents for a "look around" description within this
     *   object. 
     */
    lookAroundWithinContents(actor, illum, infoTab)
    {
        local lst;
        local lister;

        /* 
         *   if the illumination is less than 'dim' (level 2), display
         *   only self-illuminating items 
         */
        if (illum != nil && illum < 2)
        {
            /* 
             *   We're in the dark - list only those objects that the
             *   actor can sense via the sight-like senses, which will be
             *   the list of self-illuminating objects.  (To produce this
             *   list, simply make a list consisting of all of the objects
             *   in the sense info table; since the table naturally
             *   includes only objects that are illuminated, the entire
             *   contents of the table will give us the visible objects.)  
             */
            lst = senseInfoTableSubset(infoTab, {obj, info: true});
            
            /* use my dark contents lister */
            lister = darkRoomContentsLister;
        }
        else
        {
            /* start with my contents list */
            lst = contents;

            /* always remove the actor from the list */
            lst -= actor;

            /* use the normal (lighted) lister */
            lister = roomContentsLister;
        }

        /* add a paragraph before the room's contents */
        "<.p>";

        /* show the contents */
        lister.showList(actor, self, lst, ListRecurse, 0, infoTab, nil);
    }

    /*
     *   Add to the room description a list of things we notice through a
     *   specific sense.  This is used to add things we can hear and smell
     *   to a room description.
     *   
     *   'explicit' has the same meaning it does for lookAround: this is
     *   true when the examination of the surroundings was explicitly
     *   requested by a "look around" command or the like, nil if the
     *   examination is incidental to another command (such as travel
     *   between rooms).  
     */
    lookAroundWithinSense(actor, pov, sense, lister, explicit)
    {
        local infoTab;
        local presenceList;

        /* 
         *   get the information table for the desired sense from the
         *   given point of view 
         */
        infoTab = pov.senseInfoTable(sense);

        /* 
         *   Get the list of everything with a presence in this sense.
         *   Include only items that aren't part of the actor's inventory,
         *   since inventory items aren't part of the room and thus
         *   shouldn't be described as part of the room.  
         */
        presenceList = senseInfoTableSubset(infoTab,
            {obj, info: obj.(sense.presenceProp) && !obj.isIn(actor)});

        /* list the items */
        lister.showList(actor, nil, presenceList, 0, 0, infoTab, nil);
    }

    /*
     *   Show the exits from this room as part of a description of the
     *   room, if applicable.  By default, if we have an exit lister
     *   object, we'll invoke it to show the exits.  
     */
    lookAroundWithinShowExits(actor, illum)
    {
        /* if we have an exit lister, have it show a list of exits */
        if (gExitLister != nil)
            gExitLister.lookAroundShowExits(actor, self, illum);
    }

    /* 
     *   Get my lighted room contents lister - this is the Lister object
     *   that we use to display the room's contents when the room is lit.
     *   We'll return the default library room lister.  
     */
    roomContentsLister { return roomLister; }

    /* 
     *   Get my dark room contents lister - this is the Lister object we'll
     *   use to display the room's self-illuminating contents when the room
     *   is dark.  
     */
    darkRoomContentsLister { return darkRoomLister; }

    /*
     *   Get the travel connector from this location in the given
     *   direction.
     *   
     *   Map all of the directional connections to their values in the
     *   enclosing location, except for any explicitly defined in this
     *   object.  (In most cases, ordinary object won't define any
     *   directional connectors directly, since those connections usually
     *   apply only to top-level rooms.)  
     */
    getTravelConnector(dir, actor)
    {
        /* if we explicitly define the link property, use that definition */
        if (propDefined(dir.dirProp))
        {
            /* 
             *   we have a defined link in this direction - return the
             *   value of the link property 
             */
            return self.(dir.dirProp);
        }

        /* 
         *   We don't define the direction link explicitly.  If the actor
         *   can see our container, retrieve the link from our container;
         *   otherwise, stop here, and simply return the default link for
         *   the direction.  
         */
        if (location != nil && actor.canSee(location))
        {
            /* 
             *   we can see out to our location, so get the connector for
             *   the location 
             */
            return location.getTravelConnector(dir, actor);
        }
        else
        {
            /* 
             *   we can't see our location, so there's no way to travel
             *   this way - return the default connector for the direction
             */
            return dir.defaultConnector;
        }
    }

    /* 
     *   When an actor takes me, the actor will assign me a holding index
     *   value, which is simply a serial number indicating the order in
     *   which I was picked up.  This lets the actor determine which items
     *   have been held longest: the item with the lowest holding index
     *   has been held the longest.  
     */
    holdingIndex = 0

    /*
     *   An object has "weight" and "bulk" specifying how heavy and how
     *   large the object is.  These are in arbitrary units, and by
     *   default everything has a weight of 1 and a bulk of 1.
     */
    weight = 1
    bulk = 1

    /*
     *   Calculate the bulk contained within this object.  By default,
     *   we'll simply add up the bulks of all of our contents.
     */
    getBulkWithin()
    {
        local total;
        
        /* add up the bulks of our contents */
        total = 0;
        foreach (local cur in contents)
            total += cur.getBulk();
        
        /* return the total */
        return total;
    }

    /*
     *   Get the destination for an object dropped within me.  Ordinary
     *   objects can't contain actors, so an actor can't directly drop
     *   something within a regular Thing, but the same effect could occur
     *   if an actor throws a projectile, since the projectile might reach
     *   either the target or an intermediate obstruction, then bounce off
     *   and land in whatever contains the object hit.
     *   
     *   By default, objects dropped within us won't stay within us, but
     *   will land in our container's drop destination.
     *   
     *   'obj' is the object being dropped.
     */
    getDropDestination(obj)
    {
        /* by default, the object lands in our container's drop destination */
        return location.getDropDestination(obj);
    }

    /*
     *   Get the nominal destination for an object dropped into this
     *   object.  This is used to report where an object ends up when the
     *   object is moved into me by some indirect route, such as throwing
     *   the object.
     *   
     *   By default, most objects simply return themselves, so we'll
     *   report something like "obj lands {in/on} self".  Some objects
     *   might want to report a different object as the destination; for
     *   example, an indoor room might want to report objects as falling
     *   onto the floor.  
     */
    getNominalDropDestination() { return self; }

    /*
     *   Announce myself as a default object for an action.  By default,
     *   we'll show the standard library message announcing a default.  
     */
    announceDefaultObject(whichObj, action, resolvedAllObjects)
    {
        /* use the standard library message for the announcement */
        return libMessages.announceDefaultObject(
            self, whichObj, action, resolvedAllObjects);
    }
    
    /*
     *   Calculate my total weight.  Weight is generally inclusive of the
     *   weights of contents or components, because anything inside an
     *   object normally contributes to the object's total weight. 
     */
    getWeight()
    {
        local total;

        /* start with my own intrinsic weight */
        total = weight;
                
        /* 
         *   add the weights of my direct contents, recursively
         *   calculating their total weights 
         */
        foreach (local cur in contents)
            total += cur.getWeight();

        /* return the total */
        return total;
    }

    /*
     *   Calculate my total bulk.  Most objects have a fixed external
     *   shape (and thus bulk) that doesn't vary as contents are added or
     *   removed, so our default implementation simply returns our
     *   intrinsic bulk without considering any contents.
     *   
     *   Some objects do change shape as contents are added.  Such objects
     *   can override this routine to provide the appropriate behavior.
     *   (We don't try to provide a parameterized total bulk calculator
     *   here because there are too many possible ways a container's bulk
     *   could be affected by its contents or by other factors.)
     *   
     *   If an object's bulk depends on its contents, the object should
     *   override notifyInsert() so that it calls checkBulkChange() - this
     *   will ensure that an object won't suddenly become too large for
     *   its container (or holding actor).  
     */
    getBulk()
    {
        /* by default, simply return my intrinsic bulk */
        return bulk;
    }

    /*
     *   Calculate the amount of bulk that this object contributes towards
     *   encumbering an actor directly holding the item.  By default, this
     *   simply returns my total bulk.
     *   
     *   Some types of objects will override this to cause the object to
     *   contribute more or less bulk to an actor holding the object.  For
     *   example, an article of clothing being worn by an actor typically
     *   contributes no bulk at all to the actor's encumbrance, because an
     *   actor wearing an item typically doesn't also have to hold on to
     *   the item.  Or, a small animal might encumber an actor more than
     *   its normal bulk because it's squirming around trying to get free.
     */
    getEncumberingBulk(actor)
    {
        /* by default, return our normal total bulk */
        return getBulk();
    }

    /*
     *   Calculate the amount of weight that this object contributes
     *   towards encumbering an actor directly holding the item.  By
     *   default, this simply returns my total weight.
     *   
     *   Note that we don't recursively sum the encumbering weights of our
     *   contents - we simply sum the actual weights.  We do it this way
     *   because items within a container aren't being directly held by
     *   the actor, so any difference between the encumbering and actual
     *   weights should not apply, even though the actor is indirectly
     *   holding the items.  If a subclass does want the sum of the
     *   encumbering weights, it should override this to make that
     *   calculation.  
     */
    getEncumberingWeight(actor)
    {
        /* by default, return our normal total weight */
        return getWeight();
    }

    /*
     *   "What if" test.  Make the given changes temporarily, bypassing
     *   any side effects that would normally be associated with the
     *   changes; invokes the given callback; then remove the changes.
     *   Returns the result of calling the callback function.
     *   
     *   The changes are expressed as pairs of argument values.  The first
     *   value in a pair is a property, and the second is a new value for
     *   the property.  For each pair, we'll set the given property to the
     *   given value.  The setting is direct - we don't invoke any method,
     *   because we don't want to cause any side effects at this point;
     *   we're interested only in what the world would look like if the
     *   given changes were to go into effect.
     *   
     *   A special property value of 'moveInto' can be used to indicate
     *   that the object should be moved into another object for the test.
     *   In this case, the second element of the pair is not a value to be
     *   stored into the moveInto property, but instead the value is a new
     *   location for the object.  We'll call the baseMoveInto method to
     *   move the object to the given new location.
     *   
     *   In any case, after making the changes, we'll invoke the given
     *   callback function, which we'll call with no arguments.
     *   
     *   Finally, on our way out, we'll restore the properties we changed
     *   to their original values.  We once again do this without any side
     *   effects.  Note that we restore the old values even if we exit
     *   with an exception.  
     */
    whatIf(func, [changes])
    {
        local oldList;
        local cnt;
        local i;
        
        /* 
         *   Allocate a vector with one slot per change, so that we have a
         *   place to store the old values of the properties we're
         *   changing.  Note that the argument list has two entries (prop,
         *   value) for each change. 
         */
        cnt = changes.length();
        oldList = new Vector(cnt / 2);

        /* run through the change list and make each change */
        for (i = 1 ; i <= cnt ; i += 2)
        {
            local curProp;
            local newVal;
            
            /* get the current property and new value */
            curProp = changes[i];
            newVal = changes[i+1];
            
            /* see what we have */
            switch(curProp)
            {
            case &moveInto:
                /*
                 *   It's the special moveInto property.  This indicates
                 *   that we are to move self into the given new location.
                 *   First save the current location, then use
                 *   baseMoveInto to make the change without triggering
                 *   any side effects. 
                 */
                oldList.append(saveLocation());
                baseMoveInto(newVal);
                break;

            default:
                /* 
                 *   Anything else is a property to which we want to
                 *   assign the given new value.  Remember the old value
                 *   in our original values vector, then set the new value
                 *   from the argument.  
                 */
                oldList.append(self.(curProp));
                self.(curProp) = newVal;
                break;
            }
        }

        /* 
         *   the changes are now in effect - invoke the given callback
         *   function to check things out under the new conditions, but
         *   protect the code so that we are sure to restore things on the
         *   way out 
         */
        try
        {
            /* invoke the given callback, returning the result */
            return (func)();
        }
        finally
        {
            /* run through the change list and make each change */
            for (i = 1, local j = 1 ; i <= cnt ; i += 2, ++j)
            {
                local curProp;
                local oldVal;
            
                /* get the current property and old value */
                curProp = changes[i];
                oldVal = oldList[j];
            
                /* see what we have */
                switch(curProp)
                {
                case &moveInto:
                    /* restore the location that we saved */
                    restoreLocation(oldVal);
                    break;

                default:
                    /* restore the property value */
                    self.(curProp) = oldVal;
                    break;
                }
            }
        }
    }

    /*
     *   Run a what-if test to see what would happen if this object were
     *   being held directly by the given actor.
     */
    whatIfHeldBy(func, actor)
    {
        /* 
         *   by default, simply run the what-if test with this object
         *   moved into the actor's inventory
         */
        return whatIf(func, &moveInto, actor);
    }

    /*
     *   Check a proposed change in my bulk.  When this is called, the new
     *   bulk should already be in effect (the best way to do this when
     *   just making a check is via whatIf).
     *   
     *   This routine can be called during the 'check' or 'action' stages
     *   of processing a command to determine if a change in my bulk would
     *   cause a problem.  If so, we'll add a failure report and exit the
     *   command.
     *   
     *   By default, notify our immediate container of the change to see
     *   if there's any objection.  A change in an object's bulk typically
     *   only aaffects its container or containers.
     *   
     *   The usual way to invoke this routine in a 'check' or 'action'
     *   routine is with something like this:
     *   
     *      whatIf({: checkBulkChange()}, &inflated, true);
     *   
     *   This checks to see if the change in my bulk implied by changing
     *   self.inflated to true would present a problem for my container,
     *   terminating with a reportFailure+exit if so.  
     */
    checkBulkChange()
    {
        /*
         *   Notify each of our containers of a change in our bulk; if
         *   any container has a problem with our new bulk, it can
         *   report a failure and exit.
         */
        forEachContainer(
            {loc: loc.checkBulkChangeWithin(self)});
    }

    /*
     *   Check a bulk change of one of my direct contents, given by obj.
     *   When this is called, 'obj' will be (tentatively) set to reflect
     *   its proposed new bulk; if this routine doesn't like the new bulk,
     *   it should issue a failure report and exit the command, which will
     *   cancel the command that would have caused the change and will
     *   prevent the proposed change from taking effect.
     *   
     *   By default, we'll do nothing; subclasses that are sensitive to
     *   the bulks of their contents should override this.  
     */
    checkBulkChangeWithin(changingObj)
    {
    }

    /*
     *   Find all of the bags of holding contained within this object and
     *   add them to the given vector.  By default, we'll simply recurse
     *   into our children so they can add their bags of holding.  
     */
    getBagsOfHolding(vec)
    {
        /* 
         *   if my contents can't be touched from outside, there's no
         *   point in traversing our children 
         */
        if (transSensingIn(touch) != transparent)
            return;

        /* add each of my children's bags to the list */
        foreach (local cur in contents)
            cur.getBagsOfHolding(vec);
    }

    /*
     *   The strength of the light the object is giving off, if indeed it
     *   is giving off light.  This value should be one of the following:
     *   
     *   0: The object is giving off no light at all.
     *   
     *   1: The object is self-illuminating, but doesn't give off enough
     *   light to illuminate any other objects.  This is suitable for
     *   something like an LED digital clock.
     *   
     *   2: The object gives off dim light.  This level is bright enough
     *   to illuminate nearby objects, but not enough to reach distant
     *   objects or go through obscuring media, and not enough for certain
     *   activities requiring strong lighting, such as reading.
     *   
     *   3: The object gives off medium light.  This level is bright
     *   enough to illuminate nearby objects, and is enough for most
     *   activities, including reading and the like.  Traveling a distance
     *   or through an obscuring medium reduces this level to dim (2).
     *   
     *   4: The object gives off strong light.  This level is bright
     *   enough to illuminate nearby objects, and travel through an
     *   obscuring medium or over a distance reduces it to medium light
     *   (3).
     *   
     *   Note that the special value -1 is reserved as an invalid level,
     *   used to flag certain events (such as the need to recalculate the
     *   ambient light level from a new point of view).
     *   
     *   Most objects do not give off light at all.  
     */
    brightness = 0
    
    /*
     *   Sense sizes of the object.  Each object has an individual size
     *   for each sense.  By default, objects are medium for all senses;
     *   this allows them to be sensed from a distance or through an obscuring
     *   medium, but doesn't allow their details to be sensed.
     */
    sightSize = medium
    soundSize = medium
    smellSize = medium
    touchSize = medium

    /*
     *   Determine whether or not the object has a "presence" in each
     *   sense.  An object has a presence in a sense if an actor
     *   immediately adjacent to the object could detect the object by the
     *   sense alone.  For example, an object has a "hearing presence" if
     *   it is making some kind of noise, and does not if it is silent.
     *   
     *   Presence in a given sense is an intrinsic (which does not imply
     *   unchanging) property of the object, in that presence is
     *   independent of the relationship to any given actor.  If an alarm
     *   clock is ringing, it has a hearing presence, unconditionally; it
     *   doesn't matter if the alarm clock is sealed inside a sound-proof
     *   box, because whether or not a given actor has a sense path to the
     *   object is a matter for a different computation.
     *   
     *   Note that presence doesn't control access: an actor might have
     *   access to an object for a sense even if the object has no
     *   presence in the sense.  Presence indicates whether or not the
     *   object is actively emitting sensory data that would make an actor
     *   aware of the object without specifically trying to apply the
     *   sense to the object.
     *   
     *   By default, an object is visible and touchable, but does not emit
     *   any sound or odor.  
     */
    sightPresence = true
    soundPresence = nil
    smellPresence = nil
    touchPresence = true

    /*
     *   My "contents lister."  This is a Lister object that we use to
     *   display the contents of this object for room descriptions,
     *   inventories, and the like.  
     */
    contentsLister = thingContentsLister

    /* 
     *   the Lister to use when showing my contents as part of my own
     *   description (i.e., for Examine commands) 
     */
    descContentsLister = thingDescContentsLister

    /* 
     *   the Lister to use when showing my contents in response to a
     *   LookIn command 
     */
    lookInLister = thingLookInLister

    /* 
     *   my "in-line" contents lister - this is the Lister object that
     *   we'll use to display my contents parenthetically as part of my
     *   list entry in a second-level contents listing
     */
    inlineContentsLister = inlineListingContentsLister

    /*
     *   Determine if I can be sensed under the given conditions.  Returns
     *   true if the object can be sensed, nil if not.  If this method
     *   returns nil, this object will not be considered in scope for the
     *   current conditions.
     *   
     *   By default, we return nil if the ambient energy level for the
     *   object is zero.  If the ambient level is non-zero, we'll return
     *   true in 'transparent' conditions, nil for 'opaque', and we'll let
     *   the sense decide via its canObjBeSensed() method for any other
     *   transparency conditions.  Note that 'ambient' as given here is the
     *   ambient level *at the object*, not as seen along my sense path -
     *   so this should NOT be given as the ambient value from a SenseInfo,
     *   which has already been adjusted for the sense path.  
     */
    canBeSensed(sense, trans, ambient)
    {
        /* 
         *   adjust the ambient level for the transparency path, if the
         *   sense uses ambience at all 
         */
        if (sense.ambienceProp != nil)
        {
            /* 
             *   adjust the ambient level for the transparency - if that
             *   leaves a level of zero, the object can't be sensed 
             */
            if (adjustBrightness(ambient, trans) == 0)
                return nil;
        }

        /* check the viewing conditions */
        switch(trans)
        {
        case transparent:
        case attenuated:
            /* 
             *   under transparent or attenuated conditions, I appear as
             *   myself 
             */
            return true;

        case obscured:
        case distant:
            /* 
             *   ask the sense to determine if I can be sensed under these
             *   conditions 
             */
            return sense.canObjBeSensed(self, trans, ambient);

        default:
            /* for any other conditions, I can't be sensed at all */
            return nil;
        }
    }

    /*
     *   Determine if I can be sensed IN DETAIL in the given sense, with
     *   the given SenseInfo description.  By default, an object's details
     *   can be sensed if the sense path is 'transparent' or 'attenuated',
     *   OR the object's scale in the sense is 'large'.
     */
    canDetailsBeSensed(sense, info)
    {
        /* if the sense path is opaque, we can be sensed */
        if (info == nil || info.trans == opaque)
            return nil;

        /* 
         *   if we have 'large' scale in the sense, our details can be
         *   sensed under any conditions 
         */
        if (self.(sense.sizeProp) == large)
            return true;

        /* 
         *   we don't have 'large' scale in the sense, so we can be sensed
         *   in detail only if the sense path is 'transparent' or
         *   'attenuated' 
         */
        return (info.trans is in (transparent, attenuated));
    }

    /*
     *   Call a method on this object from the given point of view.  We'll
     *   push the current point of view, call the method, then restore the
     *   enclosing point of view. 
     */
    fromPOV(pov, propToCall, [args])
    {
        /* push the new point of view */
        pushPOV(pov);

        /* make sure we pop the point of view no matter how we leave */
        try
        {
            /* call the method */
            self.(propToCall)(args...);
        }
        finally
        {
            /* restore the enclosing point of view on the way out */
            popPOV();
        }
    }

    /*
     *   Every Thing has a location, which is the Thing that contains this
     *   object.  A Thing's location can only be a simple object
     *   reference, or nil; it cannot be a list, and it cannot be a method.
     *   
     *   If the location is nil, the object does not exist anywhere in the
     *   simulation's physical model.  A nil location can be used to
     *   remove an object from the game world, temporarily or permanently.
     *   
     *   In general, the 'location' property should be declared for each
     *   statically defined object (explicitly or implicitly via the '+'
     *   syntax).  'location' is a private property - it should never be
     *   evaluated or changed by any subclass or by any other object.
     *   Only Thing methods may evaluate or change the 'location'
     *   property.  So, you can declare a 'location' property when
     *   defining an object, but you should essentially never refer to
     *   'location' directly in any other context; instead, use the
     *   location and containment methods (isIn, etc) when you want to
     *   know an object's containment relationship to another object.  
     */
    location = nil

    /*
     *   Get the direct container we have in common with the given object,
     *   if any.  Returns at most one common container.  Returns nil if
     *   there is no common location.  
     */
    getCommonDirectContainer(obj)
    {
        local found;

        /* we haven't found one yet */
        found = nil;
        
        /* scan each of our containers for one in common with 'loc' */
        forEachContainer(new function(loc) {
            /*
             *   If this location of ours is a direct container of the
             *   other object, it is a common location, so note it.  
             */
            if (obj.isDirectlyIn(loc))
                found = loc;
        });

        /* return what we found */
        return found;
    }

    /*
     *   Get the container (direct or indirect) we have in common with the
     *   object, if any.  
     */
    getCommonContainer(obj)
    {
        local found;

        /* we haven't found one yet */
        found = nil;
        
        /* scan each of our containers for one in common with 'loc' */
        forEachContainer(new function(loc) {
            /*
             *   If this location of ours is a direct container of the
             *   other object, it is a common location, so note it.  
             */
            if (obj.isIn(loc))
                found = loc;
        });

        /* if we found a common container, return it */
        if (found != nil)
            return found;

        /* 
         *   we didn't find obj's container among our direct containers,
         *   so try our direct container's containers 
         */
        forEachContainer(new function(loc) {
            local cur;

            /* try finding for a common container of this container */
            cur = loc.getCommonContainer(obj);

            /* if we found it, note it */
            if (cur != nil)
                found = cur;
        });

        /* return what we found */
        return found;
    }

    /*
     *   General initialization - this will be called during preinit so
     *   that we can set up the initial values of any derived internal
     *   properties.  
     */
    initializeThing()
    {
        /* initialize our location settings */
        initializeLocation();

        /* 
         *   if we're marked as equivalent to everything in our class,
         *   build a listing group for the equivalence 
         */
        if (isEquivalent)
            getSuperclassList()[1].initializeEquivalent();
    }

    /*
     *   Initialize my location's contents list - add myself to my
     *   container during initialization
     */
    initializeLocation()
    {
        if (location != nil)
            location.addToContents(self);
    }

    /*
     *   Initialize this class object for listing its instances that are
     *   marked with isEquivalent.  We'll initialize a list group that
     *   lists our equivalent instances as a group.
     *   
     *   Note that this method should be called on the common superclass
     *   of a group of equivalent items, so that we build the equivalent
     *   grouper on the class object, not on the individual equivalent
     *   instances.  This allows all of the instances of an equivalent
     *   class to share a common grouper.  
     */
    initializeEquivalent()
    {
        /* 
         *   If we don't already have an equivalent item grouper, create
         *   one.  We might have created one already: we only need one per
         *   class of equivalents, and another instance of this same class
         *   could have gotten to it before this one.
         *   
         *   Note that we want to check for an equivalent grouper on this
         *   class object, not on any superclasses.  We might be a
         *   subclass of a more general class that also has equivalent
         *   instances; if so, we want a separate identity, because
         *   equivalence is defined by the immediate superclass.  
         */
        if (!propDefined(&equivalentGrouper, PropDefDirectly))
        {
            local baseGrouper;
            
            /* 
             *   Get our base class grouper, if we have one.  Note that
             *   since we know we haven't defined a grouper in this class
             *   object yet (per the test above), we can simply evaluate
             *   the grouper property to get any inherited value. 
             */
            baseGrouper = equivalentGrouper;
            
            /* create the grouper */
            equivalentGrouper = equivalentGrouperClass.createInstance();

            /* 
             *   Add it to our list, as long as our listWith isn't already
             *   defined as a method.  Note that we intentionally add it
             *   as the last element, because an equivalent group is the
             *   most specific kind of group possible.  
             */
            if (propType(&listWith) != TypeCode)
                listWith += equivalentGrouper;

            /* 
             *   if we have an inherited equivalent grouper, remove it
             *   from our group list - it's the equivalent grouper for a
             *   base class, which doesn't interest us, because we're not
             *   equivalent to any base classes 
             */
            if (baseGrouper != nil)
                listWith -= baseGrouper;
        }
    }

    /* 
     *   my equivalence grouper class - when we initialize, we'll create a
     *   grouper of this class and store it in equivalentGrouper 
     */
    equivalentGrouperClass = ListGroupEquivalent

    /* 
     *   Our equivalent item grouper.  During initialization, we will
     *   create an equivalent grouper and store it in this property for
     *   each class object that has instances marked with isEquivalent.
     *   Note that this is stored with the class, because we want each of
     *   our equivalent instances to share the same grouper object so that
     *   they are listed together as a group.  
     */
    equivalentGrouper = nil

    /*
     *   My contents.  This is a list of the objects that this object
     *   directly contains.
     */
    contents = []

    /*
     *   Get my associated noise object.  By default, this looks for an
     *   item of class Noise directly within me.
     */
    getNoise()
    {
        /* look for a Noise object among my direct contents */
        return contents.valWhich({obj: obj.ofKind(Noise)});
    }

    /*
     *   Get my associated odor object.  By default, this looks for an
     *   item of class Odor directly within me. 
     */
    getOdor()
    {
        /* look for a Noise object among my direct contents */
        return contents.valWhich({obj: obj.ofKind(Odor)});
    }

    /*
     *   Get a vector of all of my contents, recursively including
     *   contents of contents.  
     */
    allContents()
    {
        local vec;
        
        /* start with an empty vector */
        vec = new Vector(32);

        /* add all of my contents to the vector */
        addAllContents(vec);

        /* return the result */
        return vec;
    }

    /* 
     *   add myself and all of my contents, recursively including contents
     *   of contents, to a vector 
     */
    addAllContents(vec)
    {
        /* visit everything in my contents */
        foreach (local cur in contents)
        {
            /* 
             *   if this item is already in the vector, skip it - we've
             *   already visited it and its contents
             */
            if (vec.indexOf(cur) != nil)
                continue;
            
            /* add this item */
            vec.append(cur);

            /* add this item's contents recursively */
            cur.addAllContents(vec);
        }
    }

    /*
     *   Show the contents of this object.  If the object has any
     *   contents, we'll display a listing of the contents.  This is used
     *   to display the object's contents as part of the description of a
     *   room ("look around"), of an object ("examine box"), or of an
     *   object's contents ("look in box").
     *   
     *   'options' is the set of flags that we'll pass to showList(), and
     *   has the same meaning as for that function.
     *   
     *   'infoTab' is a lookup table of SenseInfo objects for the objects
     *   that the actor to whom we're showing the contents listing can see
     *   via the sight-like senses.
     *   
     *   This method should be overridden by any object that doesn't store
     *   its contents using a simple 'contents' list property.  
     */
    showObjectContents(pov, lister, options, indent, infoTab)
    {
        local cont;

        /* if I don't list my contents, there's nothing to do */
        if (!contentsListed)
            return;

        /* get my listable contents */
        cont = lister.getListedContents(self, infoTab);
        
        /* if the surviving list isn't empty, show it */
        if (cont != [])
            lister.showList(pov, self, cont, options, indent, infoTab, nil);
    }

    /*
     *   Show the contents of this object as part of an inventory listing.
     *   By default, we simply use the same listing we do for the normal
     *   contents listing. 
     */
    showInventoryContents(pov, lister, options, indent, infoTab)
    {
        /* by default, use the normal room/object contents listing */
        showObjectContents(pov, lister, options, indent, infoTab);
    }

    /*
     *   Get my listed contents - this is the list of my contents that
     *   we'll show with a contents listing, given the sense information
     *   list.  
     */
    getListedContents(lister, infoTab)
    {
        /* if I don't list my contents at all, there's nothing to list */
        if (!contentsListed)
            return [];
        
        /*
         *   return only my direct contents that are listed in the
         *   infoTab, since these are the objects that can be sensed from
         *   the actor's point of view 
         */
        return contents.subset({x: lister.isListed(x) && infoTab[x] != nil});
    }

    /* 
     *   Is this a "top-level" location?  A top-level location is an
     *   object which doesn't have another container, so its 'location'
     *   property is nil, but which is part of the game universe anyway.
     *   In most cases, a top-level location is simply a Room, since the
     *   network of rooms makes up the game's map.
     *   
     *   If an object has no location and is not itself a top-level
     *   location, then the object is not part of the game world.  It's
     *   sometimes useful to remove objects from the game world, such as
     *   when they're destroyed within the context of the game.  
     */
    isTopLevel = nil

    /*
     *   Determine if I'm is inside another Thing.  Returns true if this
     *   object is contained within 'obj', directly or indirectly (that
     *   is, this returns true if my immediate container is 'obj', OR my
     *   immediate container is in 'obj', recursively defined).
     *   
     *   isIn(nil) returns true if this object is "outside" the game
     *   world, which means that the object is not reachable from anywhere
     *   in the game map and is thus not part of the simulation.  This is
     *   the case if our outermost container is NOT a top-level object, as
     *   indicated by its isTopLevel property.  If we're inside an object
     *   marked as a top-level object, or we're inside an object that's
     *   inside a top-level object (and so on), then we're part of the
     *   game world, so isIn(nil) will return nil.  If our outermost
     *   container is has a nil isTopLevel property, isIn(nil) will return
     *   true.
     *   
     *   Note that our notion of "in" is not limited to enclosing
     *   containment, because the same containment hierarchy is used to
     *   represent all types of containment relationships, including
     *   things being "on" other things and part of other things.  
     */
    isIn(obj)
    {
        /* if we have no location, we're not inside any object */
        if (location == nil)
        {
            /*
             *   We have no container, so there are two possibilities:
             *   either we're not part of the game world at all, or we're
             *   a top-level object, which is an object that's explicitly
             *   part of the game world and at the top of the containment
             *   tree.  Our 'isTopLevel' property determines which it is.
             *   
             *   If they're asking us if we're inside 'nil', then what
             *   they want to know is if we're outside the game world.  If
             *   we're not a top-level object, we are outside the game
             *   world, so isIn(nil) is true; if we are a top-level
             *   object, then we're explicitly part of the game world, so
             *   isIn(nil) is false.
             *   
             *   If they're asking us if we're inside any non-nil object,
             *   then they simply want to know if we're inside that
             *   object.  So, if 'obj' is not nil, we must return nil:
             *   we're not in any object, so we can't be in 'obj'.  
             */
            if (obj == nil)
            {
                /* 
                 *   they want to know if we're outside the simulation
                 *   entirely: return true if we're NOT a top-level
                 *   object, nil otherwise 
                 */
                return !isTopLevel;
            }
            else
            {
                /*
                 *   they want to know if we're inside some specific
                 *   object; we can't be, because we're not in any object 
                 */
                return nil;
            }
        }

        /* if obj is my immediate container, I'm obviously in it */
        if (location == obj)
            return true;

        /* I'm in obj if my container is in obj */
        return location.isIn(obj);
    }

    /*
     *   Determine if I'm directly inside another Thing.  Returns true if
     *   this object is contained directly within obj.  Returns nil if
     *   this object isn't directly within obj, even if it is indirectly
     *   in obj (i.e., its container is directly or indirectly in obj).  
     */
    isDirectlyIn(obj)
    {
        /* I'm directly in obj only if it's my immediate container */
        return location == obj;
    }

    /*
     *   Determine if I'm "nominally" inside the given Thing.  Returns
     *   true if the object is actually within the given object OR if the
     *   object is in its actual container's nominal drop destination. 
     */
    isNominallyIn(obj)
    {
        /* if I'm actually in the given object, I'm nominally in it as well */
        if (isIn(obj))
            return true;

        /* 
         *   if the object is a room part, and we're nominally in the room
         *   part, then we're nominally in the object 
         */
        if (obj.ofKind(RoomPart) && isNominallyInRoomPart(obj))
            return true;

        /* we're not nominally in the object */
        return nil;
    }

    /*
     *   Determine if this object is contained within an item fixed in
     *   place within the given location.  We'll check our container to
     *   see if its contents are within an object fixed in place in the
     *   given location.
     *   
     *   This is a rather specific check that might seem a bit odd, but
     *   for some purposes it's useful to treat objects within fixed
     *   containers in a location as though they were in the location
     *   itself, because fixtures of a location are to some extent parts
     *   of the location.  
     */
    isInFixedIn(loc)
    {
        /* return true if my location's contents are in fixed things */
        return location != nil && location.contentsInFixedIn(loc);
    }

    /*
     *   Are my contents within a fixed item that is within the given
     *   location?  By default, we return nil because we are not ourselves
     *   fixed. 
     */
    contentsInFixedIn(loc) { return nil; }

    /*
     *   Determine if I'm "held" by an actor, for the purposes of being
     *   manipulated in an action.  In most cases, an object is considered
     *   held by an actor if it's directly within the actor's inventory,
     *   because the actor's direct inventory represents the contents of
     *   the actors hands (or equivalent).
     *   
     *   Some classes might override this to change the definition of
     *   "held" to include things not directly in the actor's inventory or
     *   exclude things directly in the inventory.  For example, an item
     *   being worn is generally not considered held even though it might
     *   be in the direct inventory, and a key on a keyring is considered
     *   held if the keyring is being held.  
     */
    isHeldBy(actor)
    {
        /* 
         *   by default, an object is held if and only if it's in the
         *   direct inventory of the actor 
         */
        return isDirectlyIn(actor);
    }

    /*
     *   Add to a vector all of my contents that are directly held when
     *   I'm being directly held.  This is used to add the direct contents
     *   of an item to scope when the item itself is being directly held.
     *   
     *   In most cases, we do nothing.  Certain types of objects override
     *   this because they consider their contents to be held if they're
     *   held.  For example, a keyring considers all of its keys to be
     *   held if the keyring itself is held, because the keys are attached
     *   to the keyring rather than contained within it.  
     */
    appendHeldContents(vec)
    {
        /* by default, do nothing */
    }

    /*
     *   Determine if I'm "owned" by another object.  By default, if we
     *   have an explicit owner, then we are owned by 'obj' if and only if
     *   'obj' is our explicit owner; otherwise, if 'obj' is our immediate
     *   location, and our immediate location can own us (as reported by
     *   obj.canOwn(self)), then 'obj' owns us; otherwise, 'obj' owns us
     *   if our immediate location CANNOT own us AND our immediate
     *   location is owned by 'obj'.  This last case is tricky: it means
     *   that if we're inside something other than 'obj' that can own us,
     *   such as another actor, then 'obj' doesn't own us because our
     *   immediate location does; it also means that if we're inside an
     *   object that has an explicit owner rather than an owner based on
     *   location, we have the same explicit owner, so a dollar bill
     *   inside Bob's wallet which is in turn being carried by Charlie is
     *   owned by Bob, not Charlie.
     *   
     *   This is used to determine ownership for the purpose of
     *   possessives in commands.  Ownership is not always exclusive: it
     *   is possible for a given object to have multiple owners in some
     *   cases.  For example, if Bob and Bill are both sitting on a couch,
     *   the couch could be referred to as "bob's couch" or "bill's
     *   couch", so the couch is owned by both Bob and Bill.  It is also
     *   possible for an object to be unowned.
     *   
     *   In most cases, ownership is a function of location (possession is
     *   nine-tenths of the law, as they say), but not always; in some
     *   cases, an object has a particular owner regardless of its
     *   location, such as "bob's wallet".  This default implementation
     *   allows for ownership by location, as well as explicit ownership,
     *   with explicit ownership (as indicated by the self.owner property)
     *   taking precedence.  
     */
    isOwnedBy(obj)
    {
        /* 
         *   if I have an explicit owner, then obj is my owner if it
         *   matches my explicit owner 
         */
        if (owner != nil)
            return owner == obj;
        
        /*
         *   Check my immediate container to see if it's the owner in
         *   question.  
         */
        if (location == obj)
        {
            /* 
             *   My immediate container is the owner of interest.  If this
             *   container can own me, then I'm owned by it because we
             *   didn't find any other owners first.  Otherwise, I'm not
             *   owned by it because it can't own me in the first place. 
             */
            return obj.canOwn(self);
        }
        else if (location == nil)
        {
            /* 
             *   I have no location, so I'm not inside 'obj', so it's not
             *   my owner. 
             */
            return nil;
        }
        else
        {
            /*
             *   My immediate container is not the object of interest.  If
             *   this container can own me, then I'm owned by this
             *   container and NOT by obj. 
             */
            if (location.canOwn(self))
                return nil;

            /*
             *   My container can't own me, so it's not my owner, so our
             *   owner is my container's owner - thus, I'm owned by obj if
             *   my location is owned by obj.  
             */
            return location.isOwnedBy(obj);
        }
    }

    /*
     *   My explicit owner.  By default, objects do not have explicit
     *   owners, which means that the owner at any given time is
     *   determined by the object's location - my innermost container that
     *   can own me is my owner.
     *   
     *   However, in some cases, an object is inherently owned by a
     *   particular other object (usually an actor), and this is invariant
     *   even when the object moves to a new location not within the
     *   owner.  For such cases, this property can be set to the explicit
     *   owner object, which will cause self.isOwnedBy(obj) to return true
     *   if and only if obj == self.owner.  
     */
    owner = nil

    /*
     *   Get the "nominal owner" of this object.  This is the owner that
     *   we report for the object if asked to distinguish this object from
     *   another via the OwnershipDistinguisher.  Note that the nominal
     *   owner is not necessarily the only owner, because an object can
     *   have multiple owners in some cases; however, the nominal owner
     *   must always be an owner, in that isOwnedBy(getNominalOwner())
     *   should always return true.
     *   
     *   By default, if we have an explicit owner, we'll return that.
     *   Otherwise, if our immediate container can own us, we'll return
     *   our immediate container.  Otherwise, we'll return our immediate
     *   container's nominal owner.  Note that this last case means that a
     *   dollar bill inside Bob's wallet will be Bob's dollar bill, even
     *   if Bob's wallet is currently being carried by another actor.  
     */
    getNominalOwner()
    {
        /* if we have an explicit owner, return that */
        if (owner != nil)
            return owner;

        /* if we have no location, we have no owner */
        if (location == nil)
            return nil;

        /* if our immediate location can own us, return it */
        if (location.canOwn(self))
            return location;

        /* return our immediate location's owner */
        return location.getNominalOwner();
    }

    /*
     *   Can I own the given object?  By default, objects cannot own other
     *   objects.  This can be overridden when ownership is desired.
     *   
     *   This doesn't determine that we *do* own the given object, but
     *   only that we *can* own the given object.
     */
    canOwn(obj) { return nil; }

    /*
     *   Get the carrying actor.  This is the nearest enclosing location
     *   that's an actor. 
     */
    getCarryingActor()
    {
        /* if I don't have a location, there's no carrier */
        if (location == nil)
            return nil;

        /* if my location is an actor, it's the carrying actor */
        if (location.isActor)
            return location;

        /* return my location's carrying actor */
        return location.getCarryingActor();
    }

    /*
     *   Try making the current command's actor hold me.  By default,
     *   we'll simply try a "take" command on the object.  
     */
    tryHolding()
    {
        /*   
         *   Try an implicit 'take' command.  If the actor is carrying the
         *   object indirectly, make the command "take from" instead,
         *   since what we really want to do is take the object out of its
         *   container.  
         */
        if (isIn(gActor))
            return tryImplicitAction(TakeFrom, self, location);
        else
            return tryImplicitAction(Take, self);
    }

    /*
     *   Add an object to my contents.
     *   
     *   Note that this should NOT be overridden to cause side effects -
     *   if side effects are desired when inserting a new object into my
     *   contents, use notifyInsert().  This routine is not allowed to
     *   cause side effects because it is sometimes necessary to bypass
     *   side effects when moving an item.  
     */
    addToContents(obj)
    {
        /* add the object to my contents list */
        contents += obj;
    }

    /*
     *   Remove an object from my contents.
     *   
     *   Do NOT override this routine to cause side effects.  If side
     *   effects are desired when removing an object, use notifyRemove().  
     */
    removeFromContents(obj)
    {
        /* remove the object from my contents list */
        contents -= obj;
    }

    /*
     *   Save my location for later restoration.  Returns a value suitable
     *   for passing to restoreLocation. 
     */
    saveLocation()
    {
        /* 
         *   I'm an ordinary object with only one location, so simply
         *   return the location 
         */
        return location;
    }

    /*
     *   Restore a previously saved location.  Does not trigger any side
     *   effects. 
     */
    restoreLocation(oldLoc)
    {
        /* move myself without side effects into my old container */
        baseMoveInto(oldLoc);
    }

    /*
     *   Move this object to a new container.  Before the move is actually
     *   performed, we notify the items in the movement path of the
     *   change, then we send notifyRemove and notifyInsert messages to
     *   the old and new containment trees, respectively.
     *   
     *   All notifications are sent before the object is actually moved.
     *   This means that the current game state at the time of the
     *   notifications reflects the state before the move.  
     */
    moveInto(newContainer)
    {
        /* notify the path */
        moveIntoNotifyPath(newContainer);

        /* perform the main moveInto operations */
        mainMoveInto(newContainer);
    }

    /*
     *   Move this object to a new container as part of travel.  This is
     *   almost the same as the regular moveInto(), but does not attempt
     *   to calculate and notify the sense path to the new location.  We
     *   omit the path notification because travel is done via travel
     *   connections, which are not necessarily the same as sense
     *   connections.  
     */
    moveIntoForTravel(newContainer)
    {
        /* 
         *   perform the main moveInto operations, omitting the path
         *   notification 
         */
        mainMoveInto(newContainer);
    }

    /*
     *   Main moveInto - this is the mid-level containment changer; this
     *   routine sends notifications to the old and new container, but
     *   doesn't notify anything along the connecting sense path. 
     */
    mainMoveInto(newContainer)
    {
        /* notify my container that I'm being removed */
        sendNotifyRemove(self, newContainer);

        /* notify my new container that I'm about to be added */
        if (newContainer != nil)
            newContainer.sendNotifyInsert(self, newContainer);

        /* perform the basic containment change */
        baseMoveInto(newContainer);

        /* note that I've been moved */
        moved = true;
    }

    /*
     *   Base moveInto - this is the low-level containment changer; this
     *   routine does not send any notifications to any containers, and
     *   does not mark the object as moved.  This form should be used only
     *   for internal library-initiated state changes, since it bypasses
     *   all of the normal side effects of moving an object to a new
     *   container.  
     */
    baseMoveInto(newContainer)
    {
        /* if I have a container, remove myself from its contents list */
        if (location != nil)
            location.removeFromContents(self);

        /* remember my new location */
        location = newContainer;

        /*
         *   if I'm not being moved into nil, add myself to the
         *   container's contents
         */
        if (location != nil)
            location.addToContents(self);
    }

    /*
     *   Notify each element of the move path of a moveInto operation. 
     */
    moveIntoNotifyPath(newContainer)
    {
        local path;
        
        /* calculate the path; if there isn't one, there's nothing to do */
        if ((path = getMovePathTo(newContainer)) == nil)
            return;

        /*
         *   We must fix up the path's final element, depending on whether
         *   we're moving into the new container from the outside or from
         *   the inside.  
         */
        if (isIn(newContainer))
        {
            /*
             *   We're already in the new container, so we're moving into
             *   the new container from the inside.  The final element of
             *   the path is the new container, but since we're stopping
             *   within the new container, we don't need to traverse out
             *   of it.  Simply remove the final two elements of the path,
             *   since we're not going to make this traversal when moving
             *   the object.  
             */
            path = path.sublist(1, path.length() - 2);
        }
        else
        {
            /*   
             *   We're moving into the new container from the outside.
             *   Since we calculated the path to the container, we must
             *   now add a final element to traverse into the container;
             *   the final object in the path doesn't matter, since it's
             *   just a placeholder for the new item inside the container 
             */
            path += [PathIn, nil];
        }

        /* traverse the path, sending a notification to each element */
        traversePath(path, new function(target, op) {
            /* notify this path element */
            target.notifyMoveViaPath(self, newContainer, op);

            /* continue the traversal */
            return true;
        });
    }

    /*
     *   Call a function on each container.  If we have no location, we
     *   won't invoke the function at all.  We'll invoke the function as
     *   follows:
     *   
     *   (func)(location, args...)  
     */
    forEachContainer(func, [args])
    {
        /* call the function on our location, if we have one */
        if (location != nil)
            (func)(location, args...);
    }

    /* 
     *   Call a function on each *connected* container.  For most objects,
     *   this is the same as forEachContainer, but objects that don't
     *   connect their containers for sense purposes would do nothing
     *   here. 
     */
    forEachConnectedContainer(func, [args])
    {
        /* by default, use the standard forEachContainer handling */
        forEachContainer(func, args...);
    }

    /*
     *   Determine if I'm a valid staging location for the given nested
     *   room destination 'dest'.  This is called when the actor is
     *   attempting to enter the nested room 'dest', and the travel
     *   handlers find that we're the staging location for the room.  (A
     *   "staging location" is the location the actor is required to
     *   occupy immediately before moving into the destination.)
     *   
     *   If this object is a valid staging location, the routine should
     *   simply do nothing.  If this object isn't valid as a staging
     *   location, this routine should display an appropriate message and
     *   terminate the command with 'exit'.
     *   
     *   An arbitrary object can't be a staging location, simply because
     *   an actor can't enter an arbitrary object.  So, by default, we'll
     *   explain that we can't enter this object.  If the destination is
     *   contained within us, we'll provide a more specific explanation
     *   indicating that the problem is that the destination is within us.
     */
    checkStagingLocation(dest)
    {
        /* 
         *   if the destination is within us, explain specifically that
         *   this is the problem 
         */
        if (dest.isIn(self))
            reportFailure(&invalidStagingContainer, self, dest);
        else
            reportFailure(&invalidStagingLocation, self);

        /* terminate the command */
        exit;
    }

    /*
     *   by default, objects don't accept commands 
     */
    acceptCommand(issuingActor)
    {
        /* report that we don't accept commands */
        libMessages.cannotTalkTo(self, issuingActor);

        /* tell the caller we don't accept commands */
        return nil;
    }

    /*
     *   Receive greetings from another actor.  This is invoked when the
     *   player types "actor, hello" (where we're the target actor), in
     *   which case the player character is the actor greeting us; or when
     *   the command "talk to actor" is entered (where we're "actor"), in
     *   which case the target actor of the command is the source of the
     *   greeting (because the target actor of the command is the one
     *   talking to us).  
     */
    greetingsFrom(sourceActor)
    {
        mainReport(&noResponseFrom, self);
    }

    /*
     *   by default, most objects are not logical targets for commands 
     */
    isLikelyCommandTarget = nil

    /*
     *   Generate a lookup table of all of the objects connected by
     *   containment to this object.  This table includes all containment
     *   connections, even through closed containers and the like.
     *   
     *   The table is keyed by object; the associated values are
     *   meaningless, as all that matters is whether or not an object is
     *   in the table.  
     */
    connectionTable()
    {
        local tab;

#ifdef SENSE_CACHE
        local cache;

        /* if we already have a cached connection list, return it */
        if ((cache = libGlobal.connectionCache) != nil
            && (tab = cache[self]) != nil)
            return tab;
#endif

        /* create a lookup table for the results */
        tab = new LookupTable(32, 64);

        /* add everything connected to me */
        addDirectConnections(tab);

#ifdef SENSE_CACHE
        /* cache the list, if we caching is active */
        if (cache != nil)
            cache[self] = tab;
#endif

        /* return the table */
        return tab;
    }

    /*
     *   Add this item and its direct containment connections to the
     *   lookup table.  If this object is already in the table, do
     *   nothing.  This should recursively add all connected objects.  
     */
    addDirectConnections(tab)
    {
        /* if I'm not already in the table, add me and my connections */
        if (tab[self] == nil)
        {
            /* add myself */
            tab[self] = true;

            /* add my contents */
            foreach (local cur in contents)
                cur.addDirectConnections(tab);

            /* add my container */
            if (location != nil)
                location.addDirectConnections(tab);
        }
    }

    /*
     *   Try an implicit action that would remove this object as an
     *   obstructor to 'obj' from the perspective of the current actor in
     *   the given sense.  This is invoked when this object is acting as
     *   an obstructor between the current actor and 'obj' for the given
     *   sense, and the caller wants to perform a command that requires a
     *   clear sense path to the given object in the given sense.
     *   
     *   If it is possible to perform an implicit command that would clear
     *   the obstruction, try performing the command, and return true.
     *   Otherwise, simply return nil.  The usual implied command rules
     *   should be followed (which can be accomplished simply by using
     *   tryImplictAction() to execute any implied command).
     *   
     *   The particular type of command that would remove this obstructor
     *   can vary by obstructor class.  For a container, for example, an
     *   "open" command is the usual remedy.  
     */
    tryImplicitRemoveObstructor(sense, obj)
    {
        /* by default, we have no way of clearing our obstruction */
        return nil;
    }

    /*
     *   Display a message explaining why we are obstructing a sense path
     *   to the given object.
     */
    cannotReachObject(obj)
    {
        /* 
         *   Default objects have no particular obstructive capabilities,
         *   so we can't do anything but show the default message.  This
         *   is a last resort that should rarely be used; normally, we
         *   will be able to identify a specific obstructor that overrides
         *   this to explain precisely what kind of obstruction is
         *   involved.  
         */
        libMessages.cannotReachObject(obj);
    }

    /*
     *   Display a message explaining that the source of a sound cannot be
     *   seen because I am visually obstructing it.  By default, we show
     *   nothing at all; subclasses can override this to provide a better
     *   explanation when possible.  
     */
    cannotSeeSoundSource(obj) { }

    /* explain why we cannot see the source of an odor */
    cannotSeeSmellSource(obj) { }

    /*
     *   Get the path for this object reaching out and touching the given
     *   object.  This can be used to determine whether or not an actor
     *   can touch the given object.  
     */
    getTouchPathTo(obj)
    {
        local path;
        
#ifdef SENSE_CACHE
        local key = [self, obj];
        local cache;
        local info;

        /* if we have a cache, try finding the data in the cache */
        if ((cache = libGlobal.canTouchCache) != nil
            && (info = cache[key]) != nil)
        {
            /* we have a cache entry - return the path from the cache */
            return info.touchPath;
        }
#endif

        /* select the path from here to the target object */
        path = selectPathTo(obj, &canTouchViaPath);

#ifdef SENSE_CACHE
        /* if caching is enabled, add a cache entry for the path */
        if (cache != nil)
            cache[key] = new CanTouchInfo(path);
#endif

        /* return the path */
        return path;
    }

    /*
     *   Get the path for moving this object from its present location to
     *   the given new container. 
     */
    getMovePathTo(newLoc)
    {
        /* 
         *   select the path from here to the new location, using the
         *   canMoveViaPath method to discriminate among different path
         *   possibilities. 
         */
        return selectPathTo(newLoc, &canMoveViaPath);
    }

    /*
     *   Get the path for throwing this object from its present location
     *   to the given target object. 
     */
    getThrowPathTo(newLoc)
    {
        /*
         *   select the path from here to the target, using the
         *   canThrowViaPath method to discriminate among different paths 
         */
        return selectPathTo(newLoc, &canThrowViaPath);
    }

    /*
     *   Determine if we can traverse self for moving the given object in
     *   the given manner.  This is used to determine if a containment
     *   connection path can be used to move an object to a new location.
     *   Returns a CheckStatus object indicating success if we can
     *   traverse self, failure if not.
     *   
     *   By default, we'll simply return a success indicator.  Subclasses
     *   might want to override this for particular conditions.  For
     *   example, containers would normally override this to return nil
     *   when attempting to move an object in or out of a closed
     *   container.  Some special containers might also want to override
     *   this to allow moving an object in or out only if the object is
     *   below a certain size threshhold, for example.
     *   
     *   'obj' is the object being moved, and 'dest' is the destination of
     *   the move.
     *   
     *   'op' is one of the pathXxx operations - PathIn, PathOut, PathPeer
     *   - specifying what kind of movement is being attempted.  PathIn
     *   indicates that we're moving 'obj' from outside self to inside
     *   self; PathOut indicates the opposite.  PathPeer indicates that
     *   we're moving 'obj' entirely within self - this normally means
     *   that we've moved the object out of one of our contents and will
     *   move it into another of our contents.  
     */
    checkMoveViaPath(obj, dest, op) { return checkStatusSuccess; }

    /*
     *   Determine if we can traverse this object for throwing the given
     *   object in the given manner.  By default, this returns the same
     *   thing as canMoveViaPath, since throwing is in most cases the same
     *   as ordinary movement.  Objects can override this when throwing an
     *   object through this path element should be treated differently
     *   from ordinary movement.
     */
    checkThrowViaPath(obj, dest, op)
        { return checkMoveViaPath(obj, dest, op); }

    /*
     *   Determine if we can traverse self in the given manner for the
     *   purposes of 'obj' touching another object.  'obj' is usually an
     *   actor; this determines if 'obj' is allowed to reach through this
     *   path element on the way to touching another object.
     *   
     *   By default, this returns the same thing as canMoveViaPath, since
     *   touching is in most cases the same as ordinary movement of an
     *   object from one location to another.  Objects can overridet his
     *   when touching an object through this path element should be
     *   treated differently from moving an object.  
     */
    checkTouchViaPath(obj, dest, op)
        { return checkMoveViaPath(obj, dest, op); }

    /*
     *   Determine if we can traverse this object for moving the given
     *   object via a path.  Calls checkMoveViaPath(), and returns true if
     *   checkMoveViaPath() indicates success, nil if it indicates failure.
     *   
     *   Note that this method should generally not be overridden; only
     *   checkMoveViaPath() should usually need to be overridden.  
     */
    canMoveViaPath(obj, dest, op)
        { return checkMoveViaPath(obj, dest, op).isSuccess; }

    /* determine if we can throw an object via this path */
    canThrowViaPath(obj, dest, op)
        { return checkThrowViaPath(obj, dest, op).isSuccess; }

    /* determine if we can reach out and touch an object via this path */
    canTouchViaPath(obj, dest, op)
        { return checkTouchViaPath(obj, dest, op).isSuccess; }

    /*
     *   Check moving an object through this container via a path.  This
     *   method is called during moveInto to notify each element along a
     *   move path that the movement is about to occur.  We call
     *   checkMoveViaPath(); if it indicates failure, we'll report the
     *   failure encoded in the status object and terminate the command
     *   with 'exit'.  
     *   
     *   Note that this method should generally not be overridden; only
     *   checkMoveViaPath() should usually need to be overridden.  
     */
    notifyMoveViaPath(obj, dest, op)
    {
        local stat;

        /* check the move */
        stat = checkMoveViaPath(obj, dest, op);

        /* if it's a failure, report the failure message and terminate */
        if (!stat.isSuccess)
        {
            /* report the failure */
            reportFailure(stat.msgProp, stat.msgParams...);

            /* terminate the command */
            exit;
        }
    }

    /*
     *   Choose a path from this object to a given object.  If no paths
     *   are available, returns nil.  If any paths exist, we'll find the
     *   shortest usable one, calling the given property on each object in
     *   the path to determine if the traversals are allowed.
     *   
     *   If we can find a path, but there are no good paths, we'll return
     *   the shortest unusable path.  This can be useful for explaining
     *   why the traversal is impossible.  
     */
    selectPathTo(obj, traverseProp)
    {
        local allPaths;
        local goodPaths;
        local minPath;
        
        /* get the paths from here to the given object */
        allPaths = getAllPathsTo(obj);

        /* if we found no paths, the answer is obvious */
        if (allPaths.length() == 0)
            return nil;

        /* start off with an empty vector for the good paths */
        goodPaths = new Vector(allPaths.length());

        /* go through the paths and find the good ones */
        foreach (local path in allPaths)
        {
            local ok;
            
            /* 
             *   traverse the path, calling the traversal check property
             *   on each point in the path; if any check property returns
             *   nil, it means that the traversal isn't allowed at that
             *   point, so we can't use the path 
             */
            ok = true;
            traversePath(path, new function(target, op) {
                /*
                 *   Invoke the check property on this target.  If it
                 *   doesn't allow the traversal, the path is unusable. 
                 */
                if (target.(traverseProp)(self, obj, op))
                {
                    /* we're still okay - continue the path traversal */
                    return true;
                }
                else
                {
                    /* failed - note that the path is no good */
                    ok = nil;

                    /* there's no need to continue the path traversal */
                    return nil;
                }
            });

            /* 
             *   if we didn't find any objections to the path, add this to
             *   the list of good paths 
             */
            if (ok)
                goodPaths.append(path);
        }

        /* if there are no good paths, take the shortest bad path */
        if (goodPaths.length() == 0)
            goodPaths = allPaths;

        /* find the shortest of the paths we're still considering */
        minPath = nil;
        foreach (local path in goodPaths)
        {
            /* if this is the best so far, note it */
            if (minPath == nil || path.length() < minPath.length())
                minPath = path;
        }

        /* return the shortest good path */
        return minPath;
    }

    /*
     *   Traverse a containment connection path, calling the given
     *   function for each element.  In each call to the callback, 'obj'
     *   is the container object being traversed, and 'op' is the
     *   operation being used to traverse it.
     *   
     *   At each stage, the callback returns true to continue the
     *   traversal, nil if we are to stop the traversal.
     *   
     *   Returns nil if any callback returns nil, true if all callbacks
     *   return true.  
     */
    traversePath(path, func)
    {
        local len;

        /* 
         *   if there's no path at all, there's nothing to do - simply
         *   return true in this case, because no traversal callback can
         *   fail when there are no traversal callbacks to begin with 
         */
        len = path.length();
        if (len == 0)
            return true;
        
        /* traverse from the path's starting point */
        if (path[1] != nil && !(func)(path[1], PathFrom))
            return nil;

        /* run through this path and see if the traversals are allowed */
        for (local i = 2 ; i <= len ; i += 2)
        {
            local target;
            local op;
            
            /* get the next traversal operation */
            op = path[i];
            
            /* check the next traversal to see if it's allowed */
            switch(op)
            {
            case PathIn:
                /* 
                 *   traverse in - notify the previous object, since it's
                 *   the container we're entering 
                 */
                target = path[i-1];
                break;

            case PathOut:
                /*
                 *   traversing out of the current container - tell the
                 *   next object, since it's the object we're leaving 
                 */
                target = path[i+1];
                break;

            case PathPeer:
                /*
                 *   traversing from one object to a containment peer -
                 *   notify the container in common to both peers 
                 */
                target = path[i-1].getCommonDirectContainer(path[i+1]);
                break;

            case PathThrough:
                /* 
                 *   traversing through a multi-location connector (the
                 *   previous and next object will always be the same in
                 *   this case, so it doesn't really matter which we
                 *   choose) 
                 */
                target = path[i-1];
                break;
            }

            /* call the traversal callback */
            if (target != nil && !(func)(target, op))
            {
                /* the callback told us not to continue */
                return nil;
            }
        }

        /* traverse to the path's ending point */
        if (path[len] != nil && !(func)(path[len], PathTo))
            return nil;

        /* all callbacks told us to continue */
        return true;
    }

    /*
     *   Build a vector containing all of the possible paths we can
     *   traverse to get from me to the given object.  The return value is
     *   a vector of paths; each path is a list of containment operations
     *   needed to get from here to there.
     *   
     *   Each path item in the vector is a list arranged like so:
     *   
     *   [obj, op, obj, op, obj]
     *   
     *   Each 'obj' is an object, and each 'op' is an operation enum
     *   (PathIn, PathOut, PathPeer) that specifies how to get from the
     *   preceding object to the next object.  The first object in the
     *   list is always the starting object (i.e., self), and the last is
     *   always the target object ('obj').  
     */
    getAllPathsTo(obj)
    {
        local vec;
        
        /* create an empty vector to hold the return set */
        vec = new Vector(10);

        /* look along each connection from me */
        buildContainmentPaths(vec, [self], obj);

        /* return the vector */
        return vec;
    }

    /*
     *   Service routine for getAllPathsTo: build a vector of the
     *   containment paths starting with this object.  
     */
    buildContainmentPaths(vec, pathHere, obj)
    {
        /* scan each of our contents */
        foreach (local cur in contents)
        {
            /*
             *   If this item is the target, we've found what we're
             *   looking for - simply add the path to here to the return
             *   vector.
             *   
             *   Otherwise, if this item isn't already in the path,
             *   traverse into it; if it's already in the path, there's no
             *   need to look at it because we've already looked at it to
             *   get here 
             */
            if (cur == obj)
            {
                /* 
                 *   The path to here is the path to the target.  Append
                 *   the target object itself with an 'in' operation.
                 *   Before adding the path to the result set, normalize
                 *   it.  
                 */
                vec.append(normalizePath(pathHere + [PathIn, obj]));
            }
            else if (pathHere.indexOf(cur) == nil)
            {
                /* 
                 *   look at this item, adding it to the path with an
                 *   'enter child' traversal 
                 */
                cur.buildContainmentPaths(vec, pathHere + [PathIn, cur], obj);
            }
        }

        /* scan each of our locations */
        forEachConnectedContainer(new function(loc) {
            /*
             *   If this item is the target, we've found what we're
             *   looking for.  Otherwise, if this item isn't already in
             *   the path, traverse into it 
             */
            if (loc == obj)
            {
                /* 
                 *   We have the path to the target.  Add the traversal to
                 *   the container to the path, normalize the path, and
                 *   add the path to the result set. 
                 */
                vec.append(normalizePath(pathHere + [PathOut, loc]));
            }
            else if (pathHere.indexOf(loc) == nil)
            {
                /* 
                 *   Look at this container, adding it to the path with an
                 *   'exit to container' traversal.
                 */
                loc.buildContainmentPaths(vec,
                                          pathHere + [PathOut, loc], obj);
            }
        });
    }

    /*
     *   "Normalize" a containment path to remove redundant containment
     *   traversals.
     *   
     *   First, we expand any sequence of in+out operations that take us
     *   out of one root-level containment tree and into another to
     *   include a "through" operation for the multi-location object being
     *   traversed.  For example, if 'a' and 'c' do not share a common
     *   container, then we will turn this:
     *   
     *     [a PathIn b PathOut c]
     *   
     *   into this:
     *   
     *     [a PathIn b PathThrough b Path Out c]
     *   
     *   This will ensure that when we traverse the path, we will
     *   explicitly traverse through the connector material of 'b'.
     *   
     *   Second, we replace any sequence of out+in operations through a
     *   common container with "peer" operations across the container's
     *   contents directly.  For example, a path that looks like this
     *   
     *     [a PathOut b PathIn c]
     *   
     *   will be normalized to this:
     *   
     *     [a PathPeer c]
     *   
     *   This means that we go directly from a to c, traversing through
     *   the fill medium of their common container 'b' but not actually
     *   traversing out of 'b' and back into it.  
     */
    normalizePath(path)
    {
        /* 
         *   Traverse the path looking for items to normalize with the
         *   (in-out)->(through) transformation.  Start at the second
         *   element, which is the first path operation code.  
         */
        for (local i = 2 ; i <= path.length() ; i += 2)
        {
            /*
             *   If we're on an 'in' operation, and an 'out' operation
             *   immediately follows, and the object before the 'in' and
             *   the object after the 'out' do not share a common
             *   container, we must add an explicit 'through' step for the
             *   multi-location connector being traversed. 
             */
            if (path[i] == PathIn
                && i + 2 <= path.length()
                && path[i+2] == PathOut
                && path[i-1].getCommonDirectContainer(path[i+3]) == nil)
            {
                /* we need to add a 'through' operation */
                path = path.sublist(1, i + 1)
                       + PathThrough
                       + path.sublist(i + 1);
            }
        }

        /* 
         *   make another pass, this time applying the (out-in)->peer
         *   transformation 
         */
        for (local i = 2 ; i <= path.length() ; i += 2)
        {
            /*
             *   If we're on an 'out' operation, and an 'in' operation
             *   immediately follows, we can collapse the out+in sequence
             *   to a single 'peer' operation.  
             */
            if (path[i] == PathOut
                && i + 2 <= path.length()
                && path[i+2] == PathIn)
            {
                /* 
                 *   this sequence can be collapsed to a single 'peer'
                 *   operation - rewrite the path accordingly 
                 */
                path = path.sublist(1, i - 1)
                       + PathPeer
                       + path.sublist(i + 3);
            }
        }

        /* return the normalized path */
        return path;
    }
    
    /*
     *   Get the visual sense information for this object from the current
     *   global point of view.  If we have explicit sense information set
     *   with setSenseInfo, we'll return that; otherwise, we'll calculate
     *   the current sense information for the given point of view.
     *   Returns a SenseInfo object giving the information.  
     */
    getVisualSenseInfo()
    {
        local infoTab;
        
        /* if we have explicit sense information already set, use it */
        if (explicitVisualSenseInfo != nil)
            return explicitVisualSenseInfo;

        /* calculate the sense information for the point of view */
        infoTab = getPOV().visibleInfoTable();

        /* find and return the information for myself */
        return infoTab[self];
    }

    /*
     *   Call a description method with explicit point-of-view and the
     *   related point-of-view sense information.  'pov' is the point of
     *   view object, which is usually an actor; 'senseInfo' is a
     *   SenseInfo object giving the sense information for this object,
     *   which we'll use instead of dynamically calculating the sense
     *   information for the duration of the routine called.  
     */
    withVisualSenseInfo(pov, senseInfo, methodToCall, [args])
    {
        local oldSenseInfo;
        
        /* push the sense information */
        oldSenseInfo = setVisualSenseInfo(senseInfo);

        /* push the point of view */
        pushPOV(pov);

        /* make sure we restore the old value no matter how we leave */
        try
        {
            /* 
             *   call the method with the given arguments, and return the
             *   result 
             */
            return self.(methodToCall)(args...);
        }
        finally
        {
            /* restore the old point of view */
            popPOV();
            
            /* restore the old sense information */
            setVisualSenseInfo(oldSenseInfo);
        }
    }

    /* 
     *   Set the explicit visual sense information; if this is not nil,
     *   getVisualSenseInfo() will return this rather than calculating the
     *   live value.  Returns the old value, which is a SenseInfo or nil.  
     */
    setVisualSenseInfo(info)
    {
        local oldInfo;

        /* remember the old value */
        oldInfo = explicitVisualSenseInfo;

        /* remember the new value */
        explicitVisualSenseInfo = info;

        /* return the original value */
        return oldInfo;
    }

    /* current explicit visual sense information overriding live value */
    explicitVisualSenseInfo = nil

    /*
     *   Determine how accessible my contents are to a sense.  Any items
     *   contained within a Thing are considered external features of the
     *   Thing, hence they are transparently accessible to all senses.
     */
    transSensingIn(sense) { return transparent; }

    /*
     *   Determine how accessible peers of this object are to the contents
     *   of this object, via a given sense.  This has the same meaning as
     *   transSensingIn(), but in the opposite direction: whereas
     *   transSensingIn() determines how accessible my contents are from
     *   the outside, this determines how accessible the outside is from
     *   the contents.
     *
     *   By default, we simply return the same thing as transSensingIn(),
     *   since most containers are symmetrical for sense passing from
     *   inside to outside or outside to inside.  However, we distinguish
     *   this as a separate method so that asymmetrical containers can
     *   have different effects in the different directions; for example,
     *   a box made of one-way mirrors might be transparent when looking
     *   from the inside to the outside, but opaque in the other
     *   direction.
     */
    transSensingOut(sense) { return transSensingIn(sense); }

    /*
     *   Get my "fill medium."  This is an object of class FillMedium that
     *   permeates our interior, such as fog or smoke.  This object
     *   affects the transmission of senses from one of our children to
     *   another, or between our interior and exterior.
     *   
     *   Note that the FillMedium object is usually a regular object in
     *   scope, so that the player can refer to the fill medium.  For
     *   example, if a room is filled with fog, the player might want to
     *   be able to refer to the fog in a command.
     *   
     *   By default, our medium is the same as our parent's medium, on the
     *   assumption that fill media diffuse throughout the location's
     *   interior.  Note, though, that Container overrides this so that a
     *   closed Container is isolated from its parent's fill medium -
     *   think of a closed bottle within a room filled with smoke.
     *   However, a fill medium doesn't expand from a child into its
     *   containers - it only diffuses into nested containers, never out.
     *   
     *   An object at the outermost containment level has no fill medium
     *   by default, so we return nil if our location is nil.
     *   
     *   Note that, unlike the "surface" material, the fill medium is
     *   assumed to be isotropic - that is, it has the same sense-passing
     *   characteristics regardless of the direction in which the energy
     *   is traversing the medium.  Since we don't have any information in
     *   our containment model about the positions of our objects relative
     *   to one another, we have no way to express anisotropy in the fill
     *   medium among our children anyway.
     *   
     *   Note further that energy going in or out of this object must
     *   traverse both the fill medium and the surface of the object
     *   itself.  Since we have no other information on the relative
     *   positions of our contents, we can only assume that they're
     *   uniformly distributed through our interior, so it is necessary to
     *   traverse the same amount of fill material to go from one child to
     *   any other or from a child to our inner surface.
     *   
     *   As a sense is transmitted, several consecutive traversals of a
     *   single fill material (i.e., a single object reference) will be
     *   treated as a single traversal of the material.  Since we don't
     *   have a notion of distance in our containment model, we can't
     *   assume that we cover a certain amount of distance just because we
     *   traverse a certain number of containment levels.  So, if we have
     *   three nested containment levels all inheriting a single fill
     *   material from their outermost parent, traversing from the inner
     *   container to the outer container will count as a single traversal
     *   of the material.  
     */
    fillMedium()
    {
        return (location != nil ? location.fillMedium() : nil);
    }

    /* can the given actor see/hear/smell/touch me? */
    canBeSeenBy(actor) { return actor.canSee(self); }
    canBeHeardBy(actor) { return actor.canHear(self); }
    canBeSmelledBy(actor) { return actor.canSmell(self); }
    canBeTouchedBy(actor) { return actor.canTouch(self); }

    /* can the player character see/hear/smell/touch me? */
    canBeSeen = (canBeSeenBy(gPlayerChar))
    canBeHeard = (canBeHeardBy(gPlayerChar))
    canBeSmelled = (canBeSmelledBy(gPlayerChar))
    canBeTouched = (canBeTouchedBy(gPlayerChar))

    /*
     *   Determine how well I can sense the given object.  Returns a
     *   SenseInfo object describing the sense path from my point of view
     *   to the object.
     *   
     *   Note that, because 'distant', 'attenuated', and 'obscured'
     *   transparency levels always compound (with one another and with
     *   themselves) to opaque, there will never be more than a single
     *   obstructor in a path, because any path with two or more
     *   obstructors would be an opaque path, and hence not a path at all.
     */
    senseObj(sense, obj)
    {
#ifdef SENSE_CACHE
        local infoTab;
        local info;

        /* get the sense information table */
        infoTab = senseInfoTable(sense);

        /* get the SenseInfo from the list for the desired object */
        info = infoTab[obj];

        /* if we couldn't find the object, return an opaque indicator */
        if (info == nil)
            info = new SenseInfo(obj, opaque, nil, 0);

        /* return the sense data descriptor */
        return info;
#else
        local objs;
        local trans;
        local obs;
        local ambient;

        /* 
         *   get the list of objects connected to us by containment -
         *   since the only way senses can travel between objects is via
         *   containment relationships, this is the complete set of
         *   objects that could be connected to us by any senses 
         */
        objs = connectionTable();

        /* 
         *   if the subject object isn't even in the connection list, it
         *   definitely can't be sensed 
         */
        if (objs[obj] == nil)
            return new SenseInfo(obj, opaque, nil, 0);

        /* cache the sensory information for all of these objects */
        cacheSenseInfo(objs, sense);

        /* 
         *   Check to see if the object can be sensed from here.  If I'm
         *   inside the object, consider its interior status; otherwise,
         *   consider its exterior status. 
         */
        if (isIn(obj))
        {
            /* we're in the object, so consider its interior data */
            trans = obj.tmpTransWithin_;
            ambient = obj.tmpAmbientWithin_;
            obs = obj.tmpObstructorWithin_;
        }
        else
        {
            /* we're outside the object, so consider its exterior data */
            trans = obj.tmpTrans_;
            ambient = obj.tmpAmbient_;
            obs = obj.tmpObstructor_;
        }

        /* check to see if I can sense the object */
        if (obj.canBeSensed(sense, trans, ambient))
        {
            /* it can be sensed - return the object's cached information */
            return new SenseInfo(obj, trans, obs, ambient);
        }
        else
        {
            /* it can't be sensed - indicate 'opaque' */
            return new SenseInfo(obj, opaque, nil, nil);
        }
#endif
    }

    /*
     *   Find an opaque obstructor.  This can be called immediately after
     *   calling senseObj() when senseObj() indicates that the object is
     *   opaquely obscured.  We will find the nearest (by containment)
     *   object where the sense status is non-opaque, and we'll return
     *   that object.
     *   
     *   senseObj() by itself does not determine the obstructor when the
     *   sense path is opaque, because doing so requires extra work.  The
     *   sense path calculator that senseObj() uses cuts off its search
     *   whenever it reaches an opaque point, because beyond that point
     *   nothing can be sensed.
     *   
     *   This can only be called immediately after calling senseObj()
     *   because we re-use the same cached sense path information that
     *   senseObj() uses.  
     */
    findOpaqueObstructor(sense, obj)
    {
        local path;
        
        /* get all of the paths from here to there */
        path = getAllPathsTo(obj);

        /* 
         *   if there are no paths, we won't be able to find a specific
         *   obstructor - the object simply isn't connected to us at all 
         */
        if (path == nil)
            return nil;

        /* 
         *   Arbitrarily take the first path - there must be an opaque
         *   obstructor on every path or we never would have been called
         *   in the first place.  One opaque obstructor is as good as any
         *   other for our purposes.  
         */
        path = path[1];

        /* 
         *   The last thing in the list that can be sensed is the opaque
         *   obstructor.  Note that the path entries alternate between
         *   objects and traversal operations.  
         */
        for (local i = 3, local len = path.length() ; i <= len ; i += 2)
        {
            local obj;
            local trans;
            local ambient;

            /* get this object */
            obj = path[i];

            /* 
             *   get the appropriate sense direction - if the path takes
             *   us out, look at the interior sense data for the object;
             *   otherwise look at its exterior sense data 
             */
            if (path[i-1] == PathOut)
            {
                /* we're looking outward, so use the interior data */
                trans = obj.tmpTransWithin_;
                ambient = obj.tmpAmbientWithin_;
            }
            else
            {
                /* we're looking inward, so use the exterior data */
                trans = obj.tmpTrans_;
                ambient = obj.tmpAmbient_;
            }

            /* 
             *   if this item cannot be sensed, the previous item is the
             *   opaque obstructor 
             */
            if (!obj.canBeSensed(sense, trans, ambient))
            {
                /* can't sense it - the previous item is the obstructor */
                return path[i-2];
            }
        }

        /* we didn't find any obstructor */
        return nil;
    }
    
    /*
     *   Build a list of full information on all of the objects reachable
     *   from me through the given sense, along with full information for
     *   each object's sense characteristics.  For each object, the
     *   returned list will contain a SenseInfo entry describing the sense
     *   conditions from the point of view of 'self' to the object.  
     */
    senseInfoTable(sense)
    {
        local objs;
        local tab;
        local siz;

#ifdef SENSE_CACHE
        local cache;
        local key = [self, sense];

        /* if we have cached sense information, simply return it */
        if ((cache = libGlobal.senseCache) != nil
            && (tab = cache[key]) != nil)
            return tab;
#endif
        
        /* 
         *   get the list of objects connected to us by containment -
         *   since the only way senses can travel between objects is via
         *   containment relationships, this is the complete set of
         *   objects that could be connected to us by any senses 
         */
        objs = connectionTable();

        /* cache the sensory information for all of these objects */
        cacheSenseInfo(objs, sense);

        /* build a table of all of the objects we can reach */
        siz = objs.getEntryCount();
        tab = new LookupTable(32, siz == 0 ? 32 : siz);
        objs.forEachAssoc(new function(cur, val)
        {
            local trans;
            local ambient;
            local obs;
            
            /* 
             *   consider the appropriate set of data, depending on
             *   whether we're looking out from within this object or in
             *   from outside it 
             */
            if (isIn(cur))
            {
                /* we're in the object, so consider its interior data */
                trans = cur.tmpTransWithin_;
                ambient = cur.tmpAmbientWithin_;
                obs = cur.tmpObstructorWithin_;
            }
            else
            {
                /* we're outside the object, so consider its exterior data */
                trans = cur.tmpTrans_;
                ambient = cur.tmpAmbient_;
                obs = cur.tmpObstructor_;
            }

            /* if we can reach this one, add it to the table */
            if (cur.canBeSensed(sense, trans, ambient))
                tab[cur] = new SenseInfo(cur, trans, obs, ambient);
        });

#ifdef SENSE_CACHE
        /* add the information vector to the sense cache */
        if (cache != nil)
            cache[key] = tab;
#endif

        /* return the result table */
        return tab;
    }

    /*
     *   Build a list of the objects reachable from me through the given
     *   sense and with a presence in the sense. 
     */
    sensePresenceList(sense)
    {
#ifdef SENSE_CACHE
        local infoTab;
        
        /* get the full sense list */
        infoTab = senseInfoTable(sense);

        /* 
         *   return only the subset of items that have a presence in this
         *   sense, and return only the items themselves, not the
         *   SenseInfo objects 
         */
        return senseInfoTableSubset(infoTab,
            {obj, info: obj.(sense.presenceProp)});
#else
        local objs;
        local vec;
        
        /* 
         *   get the table of objects connected to us by containment -
         *   since the only way senses can travel between objects is via
         *   containment relationships, this is the complete set of
         *   objects that could be connected to us by any senses 
         */
        objs = connectionTable();

        /* cache the sensory information for all of these objects */
        cacheSenseInfo(objs, sense);

        /* 
         *   build a list of all of the objects we can reach and which
         *   have a presence in the sense 
         */
        vec = new Vector(objs.getEntryCount());
        objs.forEachAssoc(new function(cur, val)
        {
            local trans;
            local ambient;
            
            /* 
             *   consider the appropriate set of data, depending on
             *   whether we're looking out from within this object or in
             *   from outside it 
             */
            if (isIn(cur))
            {
                /* we're in the object, so consider its interior data */
                trans = cur.tmpTransWithin_;
                ambient = cur.tmpAmbientWithin_;
            }
            else
            {
                /* we're outside the object, so consider its exterior data */
                trans = cur.tmpTrans_;
                ambient = cur.tmpAmbient_;
            }

            /* 
             *   if we can reach this one, and it has a presence in the
             *   sense, add it to the list 
             */
            if (cur.canBeSensed(sense, trans, ambient)
                && cur.(sense.presenceProp))
                vec.append(cur);
        });

        /* return the vector in list format */
        return vec.toList();
#endif
    }

    /*
     *   Determine the highest ambient sense level at this object for any
     *   of the given senses.  
     */
    senseAmbientMax(senses)
    {
        local objs;
        local maxSoFar;

        /* we don't have any level so far */
        maxSoFar = 0;
        
        /* get the table of connected objects */
        objs = connectionTable();

        /* go through each sense */
        foreach (local sense in senses)
        {
            /* 
             *   cache the ambient level for this sense for everything
             *   connected by containment 
             */
            cacheAmbientInfo(objs, sense);

            /* 
             *   if our cached level for this sense is the highest so far,
             *   remember it 
             */
            if (tmpAmbient_ > maxSoFar)
                maxSoFar = tmpAmbient_;
        }

        /* return the highest level we found */
        return maxSoFar;
    }

    /*
     *   Cache sensory information for all objects in the given list from
     *   the point of view of self.  This caches the ambient energy level
     *   at each object, if the sense uses ambient energy, and the
     *   transparency and obstructor on the best path in the sense to the
     *   object.  'objs' is the connection table, as generated by
     *   connectionTable().  
     */
    cacheSenseInfo(objs, sense)
    {
        /* first, calculate the ambient energy level at each object */
        cacheAmbientInfo(objs, sense);

        /* next, cache the sense path from here to each object */
        cacheSensePath(sense);
    }

    /*
     *   Cache the ambient energy level at each object in the table.  The
     *   list must include everything connected by containment.  
     */
    cacheAmbientInfo(objs, sense)
    {
        local aprop;
        
        /* 
         *   if this sense has ambience, transmit energy from sources to
         *   all reachable objects; otherwise, just clear out the sense
         *   data 
         */
        if ((aprop = sense.ambienceProp) != nil)
        {
            local sources;

            /* create a vector to hold the set of ambient energy sources */
            sources = new Vector(16);

            /* 
             *   Clear out any cached sensory information from past
             *   calculations, and note objects that have ambient energy to
             *   propagate.  
             */
            objs.forEachAssoc(new function(cur, val)
            {
                /* clear old sensory information for this object */
                cur.clearSenseInfo();

                /* if it's an energy source, note it */
                if (cur.(aprop) != 0)
                    sources.append(cur);
            });

            /* 
             *   Calculate the ambient energy level at each object.  To do
             *   this, start at each energy source and transmit its energy
             *   to all objects within reach of the sense. 
             */
            sources.forEach(new function(cur)
            {
                /* if this item transmits energy, process it */
                if (cur.(aprop) != 0)
                    cur.transmitAmbient(sense);
            });
        }
        else
        {
            /* 
             *   this sense doesn't use ambience - all we need to do is
             *   clear out any old sense data for this object 
             */
            objs.forEachAssoc({cur, val: cur.clearSenseInfo()});
        }
    }

    /*
     *   Transmit my radiating energy to everything within reach of the
     *   sense.  
     */
    transmitAmbient(sense)
    {
        local ambient;
        
        /* get the energy level I'm transmitting */
        ambient = self.(sense.ambienceProp);

        /* if this is greater than my ambient level so far, take it */
        if (ambient > tmpAmbient_)
        {
            /* 
             *   remember the new settings: start me with my own ambience,
             *   and with no fill medium in the way (there's nothing
             *   between me and myself, so I shine on myself with full
             *   force and with no intervening fill medium) 
             */
            tmpAmbient_ = ambient;
            tmpAmbientFill_ = nil;

            /* 
             *   if the level is at least 2, transmit to adjacent objects
             *   (level 1 is self-illumination only, so we don't transmit
             *   to anything else) 
             */
            if (ambient >= 2)
            {
                /* transmit to my containers */
                shineOnLoc(sense, ambient, nil);

                /* shine on my contents */
                shineOnContents(sense, ambient, nil);
            }
        }
    }

    /*
     *   Transmit ambient energy to my location or locations. 
     */
    shineOnLoc(sense, ambient, fill)
    {
        /* 
         *   shine on my container, if I have one, and its immediate
         *   children 
         */
        if (location != nil)
            location.shineFromWithin(self, sense, ambient, fill);
    }

    /*
     *   Shine ambient energy at my surface onto my contents. 
     */
    shineOnContents(sense, ambient, fill)
    {
        local levelWithin;
        local fillWithin;
        
        /*
         *   Figure the level of energy to transmit to my contents.  To
         *   reach my contents, the ambient energy here must traverse our
         *   surface.  Since we want to know what the ambient light at our
         *   surface looks like when viewed from our interior, we must use
         *   the sensing-out transparency. 
         */
        levelWithin = adjustBrightness(ambient, transSensingOut(sense));

        /*
         *   If there's a new fill material in my interior that the
         *   ambient energy here hasn't already just traversed, we must
         *   further adjust the ambient level in my interior by the fill
         *   transparency. 
         */
        fillWithin = tmpFillMedium_;
        if (fillWithin != fill && fillWithin != nil)
        {
            /* 
             *   we're traversing a new fill material - adjust the
             *   brightness further for the fill material 
             */
            levelWithin = adjustBrightness(levelWithin,
                                           fillWithin.senseThru(sense));
        }

        /* if that leaves any ambient energy for my interior, transmit it */
        if (levelWithin >= 2 && levelWithin > tmpAmbientWithin_)
        {
            /* note my new interior ambience */
            tmpAmbientWithin_ = levelWithin;
            
            /* shine on each object directly within me */
            foreach (local cur in contents)
                cur.shineFromWithout(self, sense, levelWithin, fillWithin);
        }
    }

    /*
     *   Transmit ambient energy from an object within me.  This transmits
     *   to my outer surface, and also to my own immediate children - in
     *   other words, to the peers of the child shining on us.  We need to
     *   transmit to the source's peers right now, because it might
     *   degrade the ambient energy to go out through our surface.  
     */
    shineFromWithin(fromChild, sense, ambient, fill)
    {
        local levelWithout;
        local levelWithin;
        local fillWithin;
        
        /*
         *   Calculate the change in energy as the sense makes its way to
         *   our "inner surface," and to peers of the sender - in both
         *   cases, the energy must traverse our fill medium to get to the
         *   next object.
         *   
         *   As always, energy must never traverse a single fill medium
         *   more than once consecutively, so if the last fill material is
         *   the same as the fill material here, no further adjustment is
         *   necessary for another traversal of the same material.  
         */
        levelWithin = ambient;
        fillWithin = tmpFillMedium_;
        if (fillWithin != fill && fillWithin != nil)
        {
            /* adjust the brightness for the fill traversal */
            levelWithin = adjustBrightness(levelWithin,
                                           fillWithin.senseThru(sense));
        }

        /* if there's no energy left to transmit, we're done */
        if (levelWithin < 2)
            return;

        /* 
         *   Since we're transmitting the energy from within us, calculate
         *   any attenuation as the energy goes from our inner surface to
         *   our outer surface - this is the energy that makes it through
         *   to our exterior and thus is the new ambient level at our
         *   surface.  We must calculate the attenuation that a viewer
         *   from outside sees looking at an energy source within us, so
         *   we must use the sensing-in transparency.
         *   
         *   Note that we start here with the level within that we've
         *   already calculated: we assume that the energy from our child
         *   must first traverse our interior medium before reaching our
         *   "inner surface," at which point it must then further traverse
         *   our surface material to reach our "outer surface," at which
         *   point it's the ambient level at our exterior.  
         */
        levelWithout = adjustBrightness(levelWithin, transSensingIn(sense));

        /* 
         *   The level at our outer surface is the new ambient level for
         *   this object.  The last fill material traversed is the fill
         *   material within me.  If it's the best yet, take it.
         */
        if (levelWithout > tmpAmbient_)
        {
            /* it's the best so far - cache it */
            tmpAmbient_ = levelWithout;
            tmpAmbientFill_ = fillWithin;

            /* transmit to our containers */
            shineOnLoc(sense, levelWithout, fillWithin);
        }

        /* transmit the level within to each peer of the sender */
        if (levelWithin > tmpAmbientWithin_)
        {
            /* note our level within */
            tmpAmbientWithin_ = levelWithin;

            /* transmit to each of our children */
            foreach (local cur in contents)
            {
                /* if it's not the source, shine on it */
                if (cur != fromChild)
                    cur.shineFromWithout(self, sense,
                                         levelWithin, fillWithin);
            }
        }
    }

    /*
     *   Transmit ambient energy from an object immediately containing me.
     */
    shineFromWithout(fromParent, sense, level, fill)
    {
        /* if this is the best level yet, take it and transmit it */
        if (level > tmpAmbient_)
        {
            /* cache this new best level */
            tmpAmbient_ = level;
            tmpAmbientFill_ = fill;

            /* transmit it down to my children */
            shineOnContents(sense, level, fill);
        }
    }

    /*
     *   Cache the sense path for each object reachable from this point of
     *   view.  Fills in tmpTrans_ and tmpObstructor_ for each object with
     *   the best transparency path from the object to me.  
     */
    cacheSensePath(sense)
    {
        /* the view from me to myself is unobstructed */
        tmpTrans_ = transparent;
        tmpTransWithin_ = transparent;
        tmpObstructor_ = nil;
        tmpObstructorWithin_ = nil;

        /* build a path to my containers */
        sensePathToLoc(sense, transparent, nil, nil);

        /* build a path to my contents */
        sensePathToContents(sense, transparent, nil, nil);
    }

    /*
     *   Build a path to my location or locations 
     */
    sensePathToLoc(sense, trans, obs, fill)
    {
        /* 
         *   proceed to my container, if I have one, and its immediate
         *   children 
         */
        if (location != nil)
            location.sensePathFromWithin(self, sense, trans, obs, fill);
    }

    /*
     *   Build a sense path to my contents 
     */
    sensePathToContents(sense, trans, obs, fill)
    {
        local transWithin;
        local obsWithin;
        local fillWithin;

        /*
         *   Figure the transparency to my contents.  To reach my
         *   contents, we must look in through our surface.  If we change
         *   the transparency, we're the new obstructor.  
         */
        transWithin = transparencyAdd(trans, transSensingIn(sense));
        obsWithin = (trans == transWithin ? obs : self);

        /*
         *   If there's a new fill material in my interior that we haven't
         *   already just traversed, we must further adjust the
         *   transparency by the fill transparency. 
         */
        fillWithin = tmpFillMedium_;
        if (fillWithin != fill && fillWithin != nil)
        {
            local oldTransWithin = transWithin;;
            
            /* we're traversing a new fill material */
            transWithin = transparencyAdd(transWithin,
                                          fillWithin.senseThru(sense));
            if (transWithin != oldTransWithin)
                obsWithin = fill;
        }

        /* if the path isn't opaque, proceed to my contents */
        if (transWithin != opaque)
        {
            /* build a path to each child */
            foreach (local cur in contents)
                cur.sensePathFromWithout(self, sense, transWithin,
                                         obsWithin, fillWithin);
        }
    }

    /*
     *   Build a path from an object within me. 
     */
    sensePathFromWithin(fromChild, sense, trans, obs, fill)
    {
        local transWithin;
        local fillWithin;
        local transWithout;
        local obsWithout;

        /* 
         *   Calculate the transparency change along the path from the
         *   child to our "inner surface" and to peers of the sender - in
         *   both cases, we must traverse the fill material. 
         *   
         *   As always, energy must never traverse a single fill medium
         *   more than once consecutively, so if the last fill material is
         *   the same as the fill material here, no further adjustment is
         *   necessary for another traversal of the same material.  
         */
        transWithin = trans;
        fillWithin = tmpFillMedium_;
        if (fillWithin != fill && fillWithin != nil)
        {
            /* adjust for traversing a new fill material */
            transWithin = transparencyAdd(transWithin,
                                          fillWithin.senseThru(sense));
            if (transWithin != trans)
                obs = fillWithin;
        }

        /* if we're opaque at this point, we're done */
        if (transWithin == opaque)
            return;

        /*
         *   Calculate the transparency going from our inner surface to
         *   our outer surface - we must traverse our own material to
         *   travel this segment. 
         */
        transWithout = transparencyAdd(transWithin, transSensingOut(sense));
        obsWithout = (transWithout != transWithin ? self : obs);

        /*
         *   We now have the path to our outer surface.  The last fill
         *   material traversed is the fill material within me.  If this
         *   is the best yet, remember it.  
         */
        if (transparencyCompare(transWithout, tmpTrans_) > 0)
        {
            /* it's the best so far - cache it */
            tmpTrans_ = transWithout;
            tmpObstructor_ = obsWithout;

            /* transmit to our containers */
            sensePathToLoc(sense, transWithout, obsWithout, fillWithin);
        }

        /* 
         *   if this is the best interior transparency yet, build a path
         *   to each peer of the sender 
         */
        if (transparencyCompare(transWithin, tmpTransWithin_) > 0)
        {
            /* it's the best so far - cache it */
            tmpTransWithin_ = transWithin;
            tmpObstructorWithin_ = obs;
            
            /* build a path to each peer of the sender */
            foreach (local cur in contents)
            {
                /* if it's not the source, build a path to it */
                if (cur != fromChild)
                    cur.sensePathFromWithout(self, sense, transWithin,
                                             obs, fillWithin);
            }
        }
    }

    /*
     *   Build a path from an object immediately containing me. 
     */
    sensePathFromWithout(fromParent, sense, trans, obs, fill)
    {
        /* if this is the best level yet, take it and keep going */
        if (transparencyCompare(trans, tmpTrans_) > 0)
        {
            /* remember this new best level */
            tmpTrans_ = trans;
            tmpObstructor_ = obs;

            /* build a path down into my children */
            sensePathToContents(sense, trans, obs, fill);
        }
    }
    

    /*
     *   Clear the sensory scratch-pad properties, in preparation for a
     *   sensory calculation pass. 
     */
    clearSenseInfo()
    {
        tmpAmbient_ = 0;
        tmpAmbientWithin_ = 0;
        tmpAmbientFill_ = nil;
        tmpTrans_ = opaque;
        tmpTransWithin_ = opaque;
        tmpObstructor_ = nil;
        tmpObstructorWithin_ = nil;

        /* pre-calculate my fill medium */
        tmpFillMedium_ = fillMedium();
    }

    /* 
     *   Scratch-pad for calculating ambient energy level - valid only
     *   after calcAmbience and until the game state changes in any way.
     *   This is for internal use within the sense propagation methods
     *   only.  
     */
    tmpAmbient_ = 0

    /*
     *   Last fill material traversed by the ambient sense energy in
     *   tmpAmbient_.  We must keep track of this so that we can treat
     *   consecutive traversals of the same fill material as equivalent to
     *   a single traversal.  
     */
    tmpAmbientFill_ = nil

    /*
     *   Scrach-pad for the best transparency level to this object from
     *   the current point of view.  This is used during cacheSenseInfo to
     *   keep track of the sense path to this object. 
     */
    tmpTrans_ = opaque

    /*
     *   Scratch-pad for the obstructor that contributed to a
     *   non-transparent path to this object in tmpTrans_. 
     */
    tmpObstructor_ = nil

    /*
     *   Scratch-pads for the ambient level, best transparency, and
     *   obstructor to our *interior* surface.  We keep track of these
     *   separately from the exterior data so that we can tell what we
     *   look like from the persepctive of an object within us.  
     */
    tmpAmbientWithin_ = 0
    tmpTransWithin_ = opaque
    tmpObstructorWithin_ = nil

    /*
     *   My fill medium.  We cache this during each sense path
     *   calculation, since the fill medium calculation often requires
     *   traversing several containment levels. 
     */
    tmpFillMedium_ = nil

    /*
     *   Merge two senseInfoTable tables.  Merges the second table into
     *   the first.  If an object appears only in the first table, the
     *   entry is left unchanged; if an object appears only in the second
     *   table, the entry is added to the first table.  If an object
     *   appears in both tables, we'll keep the one with better detail or
     *   brightness, adding it to the first table if it's the one in the
     *   second table.  
     */
    mergeSenseInfoTable(a, b)
    {
        /* if either table is nil, return the other table */
        if (a == nil)
            return b;
        else if (b == nil)
            return a;
        
        /*
         *   Iterate over the second table.  For each item already in the
         *   first table, keep the better of the two items, adding it to
         *   the first table if the better one is the one from the second
         *   table.  For each item from the second table not already in
         *   the first table, simply add the item to the first table. 
         */
        b.forEachAssoc(new function(obj, infoB)
        {
            local infoA;

            /* get the corresponding item in the first table */
            infoA = a[obj];

            /* 
             *   if this item isn't in the first table, add it; otherwise,
             *   compare the items to see which one to keep 
             */
            if (infoA == nil)
            {
                /* it's not in the first table at all, so simply add it */
                a[obj] = infoB;
            }
            else
            {
                /*
                 *   The item exists in both tables, so we must keep the
                 *   better of the two records.  Keep the one with better
                 *   transparency, or better ambient sense energy if the
                 *   transparencies are the same.
                 */
                if (infoA.trans == infoB.trans)
                {
                    /* same transparency - compare energy levels */
                    if (infoA.ambient < infoB.ambient)
                    {
                        /* the item from the second table is better */
                        a[obj] = infoB;
                    }
                }
                else if (transparencyCompare(infoA.trans, infoB.trans) < 0)
                {
                    /* a is less transparent than b - keep the 'b' item */
                    a[obj] = infoB;
                }
            }
        });

        /* return the merged first table */
        return a;
    }

    /*
     *   Receive notification that a command is about to be performed.
     *   This is called on each object not directly involved in a command
     *   but connected by containment with the actor performing the
     *   command, and on any objects explicitly registered with the actor,
     *   the actor's location and its locations up to the outermost
     *   container, or the directly involved objects.  
     */
    beforeAction()
    {
        /* by default, do nothing */
    }

    /*
     *   Receive notification that a command has just been performed.
     *   This is called by the same rules as beforeAction(), but under the
     *   conditions prevailing after the command has been completed. 
     */
    afterAction()
    {
        /* by default, do nothing */
    }

    /*
     *   Get my notification list - this is a list of objects on which we
     *   must call beforeAction and afterAction when this object is
     *   involved in a command as the direct object, indirect object, or
     *   any addition object (other than as the actor performing the
     *   command).
     *   
     *   The general notification mechanism always includes in the
     *   notification list all of the objects connected by containment to
     *   the actor; this method allows for explicit registration of
     *   additional objects that must be notified when commands are
     *   performed on this object even when the other objects are nowhere
     *   nearby.  
     */
    getObjectNotifyList()
    {
        /* return our registration list */
        return objectNotifyList;
    }

    /*
     *   Add an item to our registered notification list for actions
     *   involving this object as the direct object, indirect object, and
     *   so on.
     *   
     *   Items can be added here if they must be notified of actions
     *   involving this object regardless of the physical proximity of
     *   this item and the notification item.  
     */
    addObjectNotifyItem(obj)
    {
        objectNotifyList += obj;
    }

    /* remove an item from the registered notification list */
    removeObjectNotifyItem(obj)
    {
        objectNotifyList -= obj;
    }

    /* our list of registered notification items */
    objectNotifyList = []

    /* -------------------------------------------------------------------- */
    /*
     *   Verify a proposed change of location of this object from its
     *   current container hierarchy to the given new container.  We'll
     *   verify removal from each container up to but not including a
     *   parent that's in common with the new container - we stop upon
     *   reaching the common parent because the object isn't leaving the
     *   common parent, but merely repositioned around within it.  We'll
     *   also verify insertion into each new parent from the first
     *   non-common parent on down to the immediate new container.
     *   
     *   This routine is called any time an actor action would cause this
     *   object to be moved to a new container, so it is the common point
     *   at which to intercept any action that would attempt to move the
     *   object.  
     */
    verifyMoveTo(newLoc)
    {
        /* check removal up to the common parent */
        sendVerifyRemove(self, newLoc);

        /* check insertion into parents up to the common parent */
        if (newLoc != nil)
            newLoc.sendVerifyInsert(self);
    }

    /*
     *   Send verifyRemove notification to each direct parent, each of
     *   their direct parents, and so forth, stopping when we reach
     *   parents that we have in common with our new location.  We don't
     *   notify parents in common with new location (or their parents)
     *   because we're not actually removing the object from the common
     *   parents. 
     */
    sendVerifyRemove(obj, newLoc)
    {
        /* send notification to each container, as appropriate */
        forEachContainer(new function(loc) {
            /*
             *   If this container contains the new location, don't send
             *   it (or its parents) notification, since we're not leaving
             *   it.  Otherwise, send the notification and proceed to its
             *   parents. 
             */
            if (newLoc == nil
                || (loc != newLoc && !newLoc.isIn(loc)))
            {
                /* notify this container of the removal */
                loc.verifyRemove(obj);

                /* recursively notify this container's containers */
                loc.sendVerifyRemove(obj, newLoc);
            }
        });
    }

    /*
     *   Send notifyRemove notification to each parent, following the same
     *   strategy as sendVerifyRemove.  This routine is called when we are
     *   about to be moved to our new location - this is called *before*
     *   our new location is established.  
     */
    sendNotifyRemove(obj, newLoc)
    {
        /* send notification to each container, as appropriate */
        forEachContainer(new function(loc) {
            /*
             *   If this container contains the new location, don't send
             *   it (or its parents) notification, since we're not leaving
             *   it.  Otherwise, send the notification and proceed to its
             *   parents. 
             */
            if (newLoc == nil
                || (loc != newLoc && !newLoc.isIn(loc)))
            {
                /* notify this container of the removal */
                loc.notifyRemove(obj);

                /* recursively notify this container's containers */
                loc.sendNotifyRemove(obj, newLoc);
            }
        });
    }

    /*
     *   Send verifyInsert notification to each direct parent, each of
     *   their direct parents, and so forth, stopping when we reach
     *   parents that we have in common with our new location.  We don't
     *   notify parents in common with new location (or their parents).  
     */
    sendVerifyInsert(obj)
    {
        /* 
         *   if the object is already in me, there's no need to notify
         *   myself of the insertion; otherwise, send the notification
         */
        if (!obj.isIn(self))
        {
            /* 
             *   before we notify ourselves, notify my own parents - this
             *   sends the notifications from the outside in 
             */
            forEachContainer({loc: loc.sendVerifyInsert(obj)});

            /* notify this potential container of the insertion */
            verifyInsert(obj);
        }
    }

    /*
     *   Send notifyInsert notification to each direct parent, following
     *   the same strategy as sendVerifyInsert.  This is called *before*
     *   the change is made, so we will still be in our old container when
     *   this is called.  'obj' is the object being inserted, and
     *   'newCont' is the new direct container of the object.  
     */
    sendNotifyInsert(obj, newCont)
    {
        /* 
         *   If the object is already in me, there's no need to notify
         *   myself of the insertion; otherwise, send the notification.
         *   
         *   If the object is already inside me indirectly, and we've
         *   moving it directly in me, still notify myself, since we're
         *   still picking up a new direct child.  
         */
        if (!obj.isIn(self))
        {
            /* 
             *   before we notify ourselves, notify my own parents - this
             *   sends the notifications from the outside in 
             */
            forEachContainer({loc: loc.sendNotifyInsert(obj, newCont)});

            /* notify this potential container of the insertion */
            notifyInsert(obj, newCont);
        }
        else if (newCont == self && !obj.isDirectlyIn(self))
        {
            /* notify myself of the new direct insertion */
            notifyInsert(obj, self);
        }
    }

    /*
     *   Verify removal of an object from my contents or a child object's
     *   contents.  By default we allow the removal.  This is to be called
     *   during verification only, so gVerifyResult is valid when this is
     *   called.  
     */
    verifyRemove(obj)
    {
    }

    /*
     *   Verify insertion of an object into my contents.  By default we
     *   allow it, unless I'm already inside the other object.  This is to
     *   be called only during verification.  
     */
    verifyInsert(obj)
    {
        /* 
         *   If I'm inside the other object, don't allow it, since this
         *   would create circular containment. 
         */
        if (isIn(obj))
            illogicalNow(obj.circularlyInMessage, self, obj);
    }

    /* 
     *   my message indicating that another object x cannot be put into me
     *   because I'm already in x 
     */
    circularlyInMessage = &circularlyIn

    /*
     *   Receive notification that we are about to remove an object from
     *   this container.
     */
    notifyRemove(obj)
    {
    }

    /*
     *   Receive notification that we are about to insert a new object
     *   into this container.  'obj' is the object being moved, and
     *   'newCont' is the new direct container (which might be a child of
     *   ours).  
     */
    notifyInsert(obj, newCont)
    {
    }

    /* -------------------------------------------------------------------- */
    /*
     *   Determine if one property on this object effectively "hides"
     *   another.  This is a sort of override check for two distinct
     *   properties.
     *   
     *   We look at the object to determine where prop1 and prop2 are
     *   defined in the class hierarchy.  If prop1 isn't defined, it
     *   definitely doesn't hide prop2.  If prop2 isn't defined, prop1
     *   definitely hides it.  If both are defined, then prop1 hides prop2
     *   if and only if it is defined at a point in the class hierarchy
     *   that is "more specialized" than prop2.  That is, for prop1 to
     *   hide prop2, the class that defines prop1 must either be the same
     *   as the class that defines prop2, or the class where prop1 is
     *   defined must inherit from the class that defines prop2, or the
     *   class where prop1 is defined must be earlier in a multiple
     *   inheritance list than the class defining prop2.  
     */
    propHidesProp(prop1, prop2)
    {
        local definer1;
        local definer2;
        
        /* 
         *   get the classes in our hierarchy where the two properties are
         *   defined 
         */
        definer1 = propDefined(prop1, PropDefGetClass);
        definer2 = propDefined(prop2, PropDefGetClass);

        /* if prop1 isn't defined, it definitely doesn't hide prop2 */
        if (definer1 == nil)
            return nil;

        /* if prop1 isn't defined, prop1 definitely hides it */
        if (definer2 == nil)
            return true;

        /* 
         *   They're both defined.  If definer1 inherits from definer2,
         *   then prop1 definitely hides prop2. 
         */
        if (definer1.ofKind(definer2))
            return true;

        /* 
         *   if definer2 inherits from definer1, then prop2 hides prop1,
         *   so prop1 doesn't hide prop2 
         */
        if (definer2.ofKind(definer1))
            return nil;

        /*
         *   The two classes don't have an inheritance relation among
         *   themselves, but they might still be related in our own
         *   hierarchy by multiple inheritance.  In particular, if there
         *   is some point in the hierarchy where we have multiple base
         *   classes, and one class among the multiple bases inherits from
         *   definer1 and the other from definer2, then the one that's
         *   earlier in the multiple inheritance list is the hider. 
         */
        return superHidesSuper(definer1, definer2);
    }

    /*
     *   Determine if a given superclass of ours hides another superclass
     *   of ours, by being inherited (directly or indirectly) in our class
     *   list ahead of the other. 
     */
    superHidesSuper(s1, s2)
    {
        local lst;
        local idx1, idx2;
        
        /* get our superclass list */
        lst = getSuperclassList();

        /* if we have no superclass, there is no hiding */
        if (lst.length() == 0)
            return nil;

        /* 
         *   if we have only one element, there's obviously no hiding
         *   going on at this level, so simply traverse into the
         *   superclass to see if the hiding happens there 
         */
        if (lst.length() == 1)
            return lst[1].superHidesSuper(s1, s2);

        /*
         *   Scan the superclass list to determine the first superclass to
         *   which each of the two superclasses is related.  Stop looking
         *   when we find both superclasses or exhaust our list.  
         */
        for (local i = 1, idx1 = idx2 = nil ;
             i <= lst.length() && (idx1 == nil || idx2 == nil) ; ++i)
        {
            /* 
             *   if we haven't found s1 yet, and this superclass of ours
             *   inherits from s1, this is the earliest at which we've
             *   found s1 
             */
            if (idx1 == nil && lst[i].ofKind(s1))
                idx1 = i;

            /* likewise for the other superclass */
            if (idx2 == nil && lst[i].ofKind(s2))
                idx2 = i;
        }

        /* 
         *   if we found the two superclasses at different points in our
         *   hierarchy, the one that comes earlier hides the other; so, if
         *   idx1 is less than idx2, s1 hides s2, and if idx1 is greater
         *   than idx2, s2 hides s1 (equivalently, s1 doesn't hide s2) 
         */
        if (idx1 != idx2)
            return idx1 < idx2;

        /*
         *   We found both superclasses at the same point in our
         *   hierarchy, so they must be multiply inherited by this base
         *   class or one of its classes.  Keep looking up the hierarchy
         *   for our answer. 
         */
        return lst[idx1].superHidesSuper(s1, s2);
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Examine" action 
     */
    dobjFor(Examine)
    {
        preCond = [objVisible]
        verify()
        {
            /* give slight preference to an object being held */
            if (!isIn(gActor))
                logicalRank(80, 'not held');
        }
        action()
        {
            /* 
             *   call our mainExamine method from the current actor's point
             *   of view 
             */
            fromPOV(gActor, &mainExamine);
        }
    }

    /*
     *   Main examination processing.  This is called with the current
     *   global POV set to the actor performing the 'examine' command. 
     */
    mainExamine()
    {
        /* perform the basic 'examine' action */
        basicExamine();

        /* 
         *   listen to and smell the object, but only show a message if we
         *   have an explicitly associated Noise/Odor object 
         */
        basicExamineListen(nil);
        basicExamineSmell(nil);
        
        /* show special descriptions for any contents */
        examineSpecialContents();
    }

    /*
     *   Perform the basic 'examine' action.  This shows either the normal
     *   or initial long description (the latter only if the object hasn't
     *   been moved yet, and it has a special initial examine
     *   description), and marks the object as having been described at
     *   least once.  
     */
    basicExamine()
    {
        local info;
        local t;
        
        /* check the transparency to viewing this object */
        info = getVisualSenseInfo();
        t = info.trans;

        /*
         *   If the viewing conditions are 'transparent' or 'attenuated',
         *   we can see at full detail, no matter what our visual scale.
         *   If we have 'large' visual scale, we can see full detail,
         *   whatever the viewing conditions.  If we're not at 'large'
         *   scale, we can only see the distantDesc or obscuredDesc under
         *   those respective viewing conditions.  
         */
        if (canDetailsBeSensed(sight, info))
        {
            /* 
             *   Viewing conditions and/or scale are suitable for showing
             *   full details, so show my normal long description.  If I've
             *   never been moved, and I have an initExamineDesc, show
             *   that; otherwise, show the normal "desc" description.  
             */
            if (!moved && propType(&initExamineDesc) != TypeNil)
                initExamineDesc;
            else
                desc;

            /* note that we've examined it */
            described = true;

            /* show any subclass-specific status */
            examineStatus();
        }
        else if (t == obscured)
        {
            /* 
             *   we're obscured, and we're not at large scale, so show our
             *   obscured description 
             */
            obscuredDesc(info.obstructor);
        }
        else if (t == distant)
        {
            /* 
             *   we're distant, and we're not at large scale, so show our
             *   distant description 
             */
            distantDesc;
        }
    }

    /*
     *   Show any status associated with the object as part of the full
     *   description.  This is shown after the basicExamine() message, and
     *   is only displayed if we can see full details of the object
     *   according to the viewing conditions.
     *   
     *   By default, we do nothing.  Subclasses can add any special status
     *   information (for example, an openable could note whether the
     *   object is open or closed).  
     */
    examineStatus() { }

    /*
     *   Basic examination of the object for sound.  If the object has an
     *   associated noise object, we'll describe it.
     *   
     *   If 'explicit' is true, we'll show our soundDesc if we have no
     *   associated Noise object; otherwise, we'll show nothing at all
     *   unless we have a Noise object.  
     */
    basicExamineListen(explicit)
    {
        local obj;
        local info;
        local t;

        /* get my associated Noise object, if we have one */
        obj = getNoise();

        /* 
         *   if we have no Noise object, and we're not showing an explicit
         *   description, show nothing 
         */
        if (!explicit && obj == nil)
            return;

        /* get our sensory information from the actor's point of view */
        info = getPOV().senseObj(sound, self);
        t = info.trans;

        /*
         *   If we have a transparent path to the object, or we have
         *   'large' sound scale, show full details.  Otherwise, show the
         *   appropriate non-detailed message for the listening conditions.
         */
        if (canDetailsBeSensed(sound, info))
        {
            /* 
             *   We can hear full details.  If we have a Noise object, show
             *   its "listen to source" description; otherwise, show our
             *   own default sound description.  
             */
            if (obj != nil)
            {
                /* note the explicit display of the Noise description */
                obj.noteDisplay();
                
                /* show the examine-source description of the Noise */
                obj.sourceDesc;
            }
            else
            {
                /* we have no Noise, so use our own description */
                soundDesc;
            }
        }
        else if (t == obscured)
        {
            /* show our 'obscured' description */
            obscuredSoundDesc(info.obstructor);
        }
        else if (t == distant)
        {
            /* show our 'distant' description */
            distantSoundDesc;
        }
    }

    /*
     *   Basic examination of the object for odor.  If the object has an
     *   associated odor object, we'll describe it.  
     */
    basicExamineSmell(explicit)
    {
        local obj;
        local info;
        local t;

        /* get our associated Odor object, if any */
        obj = getOdor();

        /* 
         *   if we have no Odor object, and we're not showing an explicit
         *   description, show nothing 
         */
        if (!explicit && obj == nil)
            return;
        
        /* get our sensory information from the actor's point of view */
        info = getPOV().senseObj(smell, self);
        t = info.trans;

        /*
         *   If we have a transparent path to the object, or we have
         *   'large' sound scale, show full details.  Otherwise, show the
         *   appropriate non-detailed message for the listening conditions.
         */
        if (canDetailsBeSensed(smell, info))
        {
            /* if we have a Noise object, show its "listen to source" */
            if (obj != nil)
            {
                /* note the explicit display of the Odor description */
                obj.noteDisplay();
                
                /* show the examine-source description of the Odor */
                obj.sourceDesc;
            }
            else
            {
                /* we have no associated Odor; show our default description */
                smellDesc;
            }
        }
        else if (t == obscured)
        {
            /* show our 'obscured' description */
            obscuredSmellDesc(info.obstructor);
        }
        else if (t == distant)
        {
            /* show our 'distant' description */
            distantSmellDesc;
        }
    }

    /*
     *   Basic examination of an object for taste.  Unlike the
     *   smell/listen examination routines, we don't bother using a
     *   separate sensory emanation object for tasting, as tasting is
     *   always an explicit action, never passive.  Furthermore, since
     *   tasting requires direct physical contact with the object, we
     *   don't differentiate levels of transparency or distance.  
     */
    basicExamineTaste()
    {
        /* simply show our taste description */
        tasteDesc;
    }

    /*
     *   Basic examination of an object for touch.  As with the basic
     *   taste examination, we don't use an emanation object or
     *   distinguish transparency levels, because feeling an object
     *   requires direct physical contact.  
     */
    basicExamineFeel()
    {
        /* simply show our touch description */
        feelDesc;
    }

    /*
     *   Show the special descriptions of any contents.  We'll run through
     *   the visible information list for the location; for any visible
     *   item inside me that is using its special description, we'll
     *   display the special description as a separate paragraph.  
     */
    examineSpecialContents()
    {
        local lst;
        local infoTab;
        
        /* get the actor's table of visible items */
        infoTab = gActor.visibleInfoTable();

        /* 
         *   get the objects using special descriptions and contained
         *   within this object 
         */
        lst = specialDescList(infoTab,
                              {obj: obj.useSpecialDesc() && obj.isIn(self)});

        /* show the special descriptions */
        specialDescLister.showList(gActor, nil, lst, 0, 0, infoTab, nil);
    }

    /*
     *   Given a visible object info table (from Actor.visibleInfoTable()),
     *   get the list of objects, filtered by the given condition and
     *   sorted by specialDescOrder.  
     */
    specialDescList(infoTab, cond)
    {
        local lst;
        
        /* 
         *   get a list of all of the objects in the table - the objects
         *   are the keys, so we just want a list of the keys 
         */
        lst = infoTab.keysToList();

        /* subset the list for the given condition */
        lst = lst.subset(cond);

        /* 
         *   sort the list in ascending order of specialDescOrder, and
         *   return the result 
         */
        return lst.sort(SortAsc,
                        {a, b: a.specialDescOrder - b.specialDescOrder});
    }
    

    /* -------------------------------------------------------------------- */
    /*
     *   "Read" 
     */
    dobjFor(Read)
    {
        preCond = [objVisible]
        verify()
        {
            /* 
             *   reduce the likelihood that they want to read an ordinary
             *   item, but allow it 
             */
            logicalRank(50, 'not readable');

            /* give slight preference to an object being held */
            if (!isIn(gActor))
                logicalRank(80, 'not held');
        }
        action()
        {
            /* simply show the ordinary description */
            actionDobjExamine();
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Look in" 
     */
    dobjFor(LookIn)
    {
        preCond = [objVisible]
        verify()
        {
            /* give slight preference to an object being held */
            if (!isIn(gActor))
                logicalRank(80, 'not held');
        }
        action() { mainReport(&nothingInside); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Search".  By default, we make "search obj" do the same thing as
     *   "look in obj".  
     */
    dobjFor(Search) asDobjFor(LookIn)

    /* -------------------------------------------------------------------- */
    /*
     *   "Look under" 
     */
    dobjFor(LookUnder)
    {
        preCond = [objVisible]
        verify() { }
        action() { mainReport(&nothingUnder); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Look behind" 
     */
    dobjFor(LookBehind)
    {
        preCond = [objVisible]
        verify() { }
        action() { mainReport(&nothingBehind); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Look through" 
     */
    dobjFor(LookThrough)
    {
        verify() { }
        action() { mainReport(&nothingThrough); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Listen to"
     */
    dobjFor(ListenTo)
    {
        preCond = [objAudible]
        verify() { }
        action()
        {
            /* show our "listen" description explicitly */
            fromPOV(gActor, &basicExamineListen, true);
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Smell"
     */
    dobjFor(Smell)
    {
        preCond = [objSmellable]
        verify() { }
        action()
        {
            /* 
             *   show our 'smell' description, explicitly showing our
             *   default description if we don't have an Odor association 
             */
            fromPOV(gActor, &basicExamineSmell, true);
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Taste"
     */
    dobjFor(Taste)
    {
        /* to taste an object, we have to be able to touch it */
        preCond = [touchObj]
        verify()
        {
            /* you *can* taste anything, but for most things it's unlikely */
            logicalRank(50, 'not edible');
        }
        action()
        {
            /* show our "taste" description */
            fromPOV(gActor, &basicExamineTaste);
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Feel" 
     */
    dobjFor(Feel)
    {
        /* to feel an object, we have to be able to touch it */
        preCond = [touchObj]
        verify() { }
        action()
        {
            /* show our "feel" description */
            fromPOV(gActor, &basicExamineFeel);
        }
    }
    

    /* -------------------------------------------------------------------- */
    /*
     *   "Take" action
     */

    dobjFor(Take)
    {
        preCond = [touchObj, objNotWorn, roomToHoldObj]
        verify()
        {
            /*      
             *   if the object is already being held by the actor, it
             *   makes no sense at the moment to take it 
             */
            if (isDirectlyIn(gActor))
            {
                /* I'm already holding it, so this is not logical */
                illogicalNow(&alreadyHolding);
            }
            else
            {
                local carrier;
                
                /*
                 *   If the object isn't being held, it's logical to take
                 *   it.  However, rank objects being carried as less
                 *   likely than objects not being carried, because in an
                 *   ambiguous situation, it's more likely that an actor
                 *   would want to take something not being carried at all
                 *   than something already being carried inside another
                 *   object.  
                 */
                if (isIn(gActor))
                    logicalRank(70, 'already in');

                /*
                 *   If the object is being carried by another actor,
                 *   reduce the likelihood, since taking something from
                 *   another actor is usually less likely than taking
                 *   something out of one's own possessions or from the
                 *   location.  
                 */
                carrier = getCarryingActor();
                if (carrier != nil && carrier != gActor)
                    logicalRank(60, 'other owner');
            }

            /* 
             *   verify transfer from the current container hierarchy to
             *   the new container hierarchy 
             */
            verifyMoveTo(gActor);
        }
    
        action()
        {
            /* move me into the actor's direct contents */
            moveInto(gActor);

            /* issue our default acknowledgment of the command */
            defaultReport(&okayTake);
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Take from" processing 
     */
    dobjFor(TakeFrom)
    {
        preCond = [touchObj, objNotWorn, roomToHoldObj]
        verify()
        {
            /* 
             *   we can only take something from something else if the thing
             *   is inside the other thing 
             */
            if (gIobj != nil && !self.isIn(gIobj))
                illogicalNow(gIobj.takeFromNotInMessage);

            /* treat this otherwise like a regular "take" */
            verifyDobjTake();
        }
        check()
        {
            /* treat this like a regular "take" */
            checkDobjTake();
        }
        action()
        {
            /* treat this like a regular "take" */
            actionDobjTake();
        }
    }
    iobjFor(TakeFrom)
    {
        verify()
        {
            /* check what we know about the dobj */
            if (gDobj == nil)
            {
                /* 
                 *   We haven't yet resolved the direct object; check the
                 *   tentative direct object list, and count us as
                 *   illogical if none of the possible direct objects are
                 *   in me.  
                 */
                if (gTentativeDobj.indexWhich({x: x.obj_.isIn(self)}) == nil)
                    illogicalNow(takeFromNotInMessage);
            }
            else if (!gDobj.isIn(self))
            {
                /* 
                 *   the dobj isn't in me, so it's obviously not logical
                 *   to take the dobj out of me 
                 */
                illogicalNow(takeFromNotInMessage);
            }
        }
    }

    /* general message for "take from" when an object isn't in me */
    takeFromNotInMessage = &takeFromNotIn

    /* -------------------------------------------------------------------- */
    /*
     *   "Drop" verb processing 
     */
    dobjFor(Drop)
    {
        preCond = [objHeld]
        verify()
        {
            /* the object must be held by the actor, at least indirectly */
            if (!isIn(gActor))
                illogicalNow(&notCarrying);

            /* 
             *   verify transfer from the current container hierarchy to
             *   the new one 
             */
            verifyMoveTo(gActor.getDropDestination(self));
        }

        action()
        {
            /* move the object to the actor's drop destination */
            moveInto(gActor.getDropDestination(self));
        
            /* issue our default acknowledgment of the command */
            defaultReport(&okayDrop);
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Put In" verb processing.  Default objects cannot contain other
     *   objects, but they can be put in arbitrary containers.  
     */
    dobjFor(PutIn)
    {
        preCond = [objHeld]
        verify()
        {
            /* 
             *   It makes no sense to put us in a container we're already
             *   directly in.  (It's fine to put it in something it's
             *   indirectly in, though - doing so takes it out of the
             *   intermediate container and moves it directly into the
             *   indirect object.) 
             */
            if (gIobj != nil && isDirectlyIn(gIobj))
                illogicalNow(&alreadyPutIn);

            /* verify the transfer */
            verifyMoveTo(gIobj);
        }
    }
    iobjFor(PutIn)
    {
        preCond = [touchObj]
        verify()
        {
            /* by default, objects cannot be put in this object */
            illogical(&notAContainer);
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Put On" processing.  Default objects cannot have other objects
     *   put on them, but they can be put on surfaces.
     */
    dobjFor(PutOn)
    {
        preCond = [objHeld]
        verify()
        {
            /* it makes no sense to put us on a surface we're already on */
            if (gIobj != nil && isDirectlyIn(gIobj))
                illogicalNow(&alreadyPutOn);

            /* verify the transfer */
            verifyMoveTo(gIobj);
        }
    }

    iobjFor(PutOn)
    {
        preCond = [touchObj]
        verify()
        {
            /* by default, objects cannot be put on this object */
            illogical(&notASurface);
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "PutUnder" action 
     */
    dobjFor(PutUnder)
    {
        preCond = [touchObj]
        verify() { }
    }

    iobjFor(PutUnder)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPutUnder); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Wear" action 
     */
    dobjFor(Wear)
    {
        verify() { illogical(&notWearable); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Doff" action 
     */
    dobjFor(Doff)
    {
        verify() { illogical(&notDoffable); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Kiss" 
     */
    dobjFor(Kiss)
    {
        preCond = [touchObj]
        verify() { logicalRank(50, 'not kissable'); }
        action() { mainReport(&cannotKiss); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Ask for" action 
     */
    dobjFor(AskFor)
    {
        verify() { illogical(&notAddressable, self); }
    }
    iobjFor(AskFor)
    {
        /* 
         *   we can ask anyone for anything; let the direct object (the
         *   person being asked) handle the command 
         */
        verify() { }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Talk to" 
     */
    dobjFor(TalkTo)
    {
        verify() { illogical(&notAddressable, self); }
    }

    /* -------------------------------------------------------------------- */
    /* 
     *   "Give to" action 
     */
    dobjFor(GiveTo)
    {
        preCond = [objHeld]
        verify() { }
    }
    iobjFor(GiveTo)
    {
        preCond = [touchObj]
        verify() { illogical(&notInterested, self); }
    }

    /* -------------------------------------------------------------------- */
    /* 
     *   "Show to" action 
     */
    dobjFor(ShowTo)
    {
        preCond = [objVisible]
        verify()
        {
            /* it's more likely that we want to show something held */
            if (isHeldBy(gActor))
            {
                /* I'm being held - use the default logical ranking */
            }
            else if (isIn(gActor))
            {
                /* 
                 *   the actor isn't hold me, but I am in the actor's
                 *   inventory, so reduce the likelihood only slightly 
                 */
                logicalRank(80, 'not held');
            }
            else
            {
                /* 
                 *   the actor isn't even carrying me, so reduce our
                 *   likelihood even more 
                 */
                logicalRank(70, 'not carried');
            }
        }
        check()
        {
            /*
             *   The direct object must be visible to the indirect object
             *   in order for the indirect object to be shown the direct
             *   object. 
             */
            if (!gIobj.canSee(self))
            {
                reportFailure(&actorCannotSee, gIobj, self);
                exit;
            }

            /*
             *   The actor performing the showing must also be visible to
             *   the indirect object, otherwise the actor wouldn't be able
             *   to attract the indirect object's attention to do the
             *   showing.  
             */
            if (!gIobj.canSee(gActor))
            {
                reportFailure(&actorCannotSee, gIobj, gActor);
                exit;
            }
        }
    }
    iobjFor(ShowTo)
    {
        verify() { illogical(&notInterested, self); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Ask about" action 
     */
    dobjFor(AskAbout)
    {
        verify() { illogical(&notAddressable, self); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Tell about" action 
     */
    dobjFor(TellAbout)
    {
        verify() { illogical(&notAddressable, self); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Follow" action 
     */
    dobjFor(Follow)
    {
        verify()
        {
            /* make sure I'm followable to start with */
            if (!verifyFollowable())
                return;

            /* it makes no sense to follow myself */
            if (gActor == gDobj)
                illogical(&cannotFollowSelf);

            /* ask the actor to verify following the object */
            gActor.actorVerifyFollow(self);
        }
        action()
        {
            /* ask the actor to carry out the follow */
            gActor.actorActionFollow(self);
        }
    }

    /*
     *   Verify that I'm a followable object.  By default, it's not
     *   logical to follow an arbitrary object.  If I'm not followable,
     *   this routine will generate an appropriate illogical() explanation
     *   and return nil.  If I'm followable, we'll return true.  
     */
    verifyFollowable()
    {
        illogical(&notFollowable);
        return nil;
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Attack" action.  By default, we'll simply re-route this to an
     *   "attack with" command so that we ask for a weapon.  Some objects
     *   might want to override this to allow for "attack" with no
     *   implement.  
     */
    dobjFor(Attack)
    {
        preCond = [touchObj]
        verify() { }
        action() { askForIobj(AttackWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Attach with" action 
     */
    dobjFor(AttackWith)
    {
        preCond = [touchObj]

        /* 
         *   it makes as much sense to attack any object as any other, but
         *   by default attacking an object has no effect 
         */
        verify() { }
        action() { mainReport(&uselessToAttack); }
    }
    iobjFor(AttackWith)
    {
        preCond = [objHeld]
        verify() { illogical(&notAWeapon); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Throw" action.  By default, we'll simply re-route this to a
     *   "throw at" action.  Objects that can meaningfully be thrown
     *   without any specific target can override this.
     *   
     *   Note that we don't apply an preconditions or verification, since
     *   we don't really do anything with the action ourselves.  If an
     *   object overrides this, it should add any preconditions and
     *   verifications that are appropriate.  
     */
    dobjFor(Throw)
    {
        verify() { }
        action() { askForIobj(ThrowAt); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Throw at" action 
     */
    dobjFor(ThrowAt)
    {
        preCond = [objHeld]
        verify()
        {
            /* by default, we can throw something if we can drop it */
            verifyMoveTo(gActor.getDropDestination(self));

            /* can't throw something at itself */
            if (gIobj == self)
                illogical(&cannotThrowAtSelf);

            /* can't throw dobj at iobj if iobj is in dobj */
            if (gIobj != nil && gIobj.isIn(self))
                illogicalNow(&cannotThrowAtContents);
        }
        action()
        {
            /* 
             *   process a 'throw' operation, finishing with hitting the
             *   target if we get that far 
             */
            processThrow(gIobj, &throwTargetHitWith);
        }
    }
    iobjFor(ThrowAt)
    {
        /* by default, anything can be a target */
        verify() { }
    }

    /*
     *   Process a 'throw' command.  This is common handling that can be
     *   used for any sort of throwing (throw at, throw to, throw in,
     *   etc).  The projectile is self, and 'target' is the thing we're
     *   throwing at or to.  'hitProp' is the property to call on 'target'
     *   if we reach the target.  
     */
    processThrow(target, hitProp)
    {
        local path;
        local stat;
        local prvCont;
        
        /* get the throw path */
        path = getThrowPathTo(target);
        
        /* 
         *   the previous container on the path is initially the
         *   projectile's own initial location 
         */
        prvCont = location;
        
        /* traverse the path with throwViaPath */
        stat = traversePath(path, new function(obj, op)
        {
            local result;
            
            /* process this traversal operation */
            result = throwViaPath(obj, op, prvCont, target);
            
            /* 
             *   Note this object as the previous container (since each
             *   traversal operation is sent to a container).  Also note
             *   the operator we're applying, so we'll know whether we're
             *   traversing from inside or outside the container.
             *   
             *   If this path element is the target, this must be the last
             *   element of the traversal, so don't note the previous
             *   container here - we'll want to keep the actual previous
             *   container in this case.  
             */
            if (obj != target)
                prvCont = obj;
            
            /* return the result */
            return result;
        });
        
        /*
         *   If we made it all the way through the path without complaint,
         *   process hitting the target.  If something along the path
         *   finished the traversal (by returning nil), we'll consider the
         *   action complete - the object along that path that canceled
         *   the traversal is responsible for having displayed an
         *   appropriate message.  
         */
        if (stat)
            target.(hitProp)(self, prvCont);
    }

    /*
     *   Process the effect of throwing the object 'obj' at the target
     *   'self'.  By default, we'll move the projectile to the target's
     *   drop location, and display a message saying that there was no
     *   effect other than the projectile dropping to the floor (or
     *   whatever it drops to).
     */
    throwTargetHitWith(obj, prvCont)
    {
        local dest;
        
        /* 
         *   use the same destination we would have used if we were an
         *   obstacle that had stopped the throw 
         */
        dest = getHitFallDestination(obj, prvCont);

        /* move the object to the drop destination */
        obj.moveInto(dest);

        /* 
         *   generate the default message if we successfully moved the
         *   object to the target destination 
         */
        if (obj.isDirectlyIn(dest))
            mainReport(&throwHit, obj, self,
                       dest.getNominalDropDestination());
    }

    /*
     *   Carry out a 'throw' operation along a path.  'self' is the
     *   projectile; 'obj' is the path element being traversed, and 'op'
     *   is the operation being used to traverse the element.  'target' is
     *   the object we're throwing 'self' at.
     *   
     *   By default, we'll use the standard canThrowViaPath handling
     *   (which invokes the even more basic checkThrowViaPath) to
     *   determine if we can make this traversal.  If so, we'll proceed
     *   with the throw; otherwise, we'll stop the throw by calling
     *   stopThrowViaPath() and returning the result.  
     */
    throwViaPath(obj, op, prvCont, target)
    {
        /*
         *   By default, if we can throw the object through self, return
         *   true to allow the caller to proceed; otherwise, describe the
         *   object as hitting this element and falling to the appropriate
         *   point.  
         */
        if (obj.canThrowViaPath(self, target, op))
        {
            /* no objection - allow the traversal to proceed */
            return true;
        }
        else
        {
            /* can't do it - stop the throw and return the result */
            return obj.stopThrowViaPath(self, prvCont);
        }
    }

    /*
     *   Stop a 'throw' operation along a path.  'self' is the object in
     *   the path that is impassable by 'projectile' according to
     *   canThrowViaPath(), and 'prvCont' is the container previously
     *   traversed along the path.
     *   
     *   The return value is taken as a path traversal continuation
     *   indicator: nil means to stop the traversal, which is to say that
     *   the 'throw' command finishes here.  If we don't really want to
     *   stop the traversal, we can return 'true' to let the traversal
     *   continue.
     *   
     *   By default, we'll stop the throw, moving the projectile (self)
     *   into the previous container in the path and announcing that we
     *   hit 'obj'.  This is the normal handling when we can't throw
     *   through 'obj' because 'obj' is a closed container or is otherwise
     *   impassable by self when thrown.  This can be overridden to
     *   provide different handling if needed.  
     */
    stopThrowViaPath(projectile, prvCont)
    {
        local dest;
        
        /* 
         *   We can't traverse self with the projectile, so stop here.
         *   Get the destination for the projectile - this is the
         *   destination for an object hitting self.  
         */
        dest = getHitFallDestination(projectile, prvCont);

        /* move the projectile to the drop destination */
        projectile.moveInto(dest);
        
        /* 
         *   if that succeeded, report what happened - we hit 'self' and
         *   fall to 'dest' 
         */
        if (projectile.isDirectlyIn(dest))
            mainReport(&throwHit, projectile, self,
                       dest.getNominalDropDestination());
        
        /* stop the traversal - the 'throw' is now finished */
        return nil;
    }
    
    /*
     *   Get the "hit-and-fall" destination for a thrown object.  This is
     *   called when we interrupt a thrown object's trajectory because
     *   we're in the way of its trajectory.
     *   
     *   For example, if the actor is inside a cage, and tries to throw a
     *   projectile at an object outside the cage, and the cage blocks the
     *   projectile's passage, then this routine is called on the cage to
     *   determine where the projectile ends up.  The projectile's
     *   ultimate destination is the hit-and-fall destination for the
     *   cage: it's where the project ends up when it hits me and then
     *   falls to the ground, its trajectory cut short.
     *   
     *   'prvCont' gives the container traversed in the throw path
     *   immediately before this object.  This lets us know which "side"
     *   the projectile was on when it hit us.  In the simple case, this
     *   tells us whether the object was thrown from inside or from
     *   outside of us.  In the case of a multi-location item, this tells
     *   us which of our multiple containers the projectile approached us
     *   from.  
     */
    getHitFallDestination(thrownObj, prvCont)
    {
        /* 
         *   if the previous container is within us, we're throwing from
         *   inside to the outside, so the object falls within me;
         *   otherwise, the object bounces off the outside and falls
         *   outside me 
         */
        if (prvCont.isIn(self))
        {
            /* throwing from within - land directly in me */
            return self;
        }
        else
        {
            /* 
             *   throwing from without - the projectile bounces off me, so
             *   it should end up in our location's 'drop' destination 
             */
            return location.getDropDestination(thrownObj);
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Throw to" action 
     */
    dobjFor(ThrowTo)
    {
        preCond = [objHeld]
        verify()
        {
            /* 
             *   by default, we can throw an object to someone if we can
             *   throw it at them 
             */
            verifyDobjThrowAt();
        }
        action()
        {
            /* 
             *   process a 'throw' operation, finishing with the target
             *   trying to catch the object if we get that far
             */
            processThrow(gIobj, &throwTargetCatch);
        }
    }

    iobjFor(ThrowTo)
    {
        verify()
        {
            /* by default, we don't want to catch anything */
            illogical(&willNotCatch, self);
        }
    }

    /*
     *   Process the effect of throwing the object 'obj' to the catcher
     *   'self'.  By default, we'll catch the object.  
     */
    throwTargetCatch(obj, prvCont)
    {
        /* take the object */
        obj.moveInto(self);

        /* generate the default message if we successfully took the object */
        if (obj.isDirectlyIn(self))
            mainReport(&throwCatch, obj, self);
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Dig" action - by default, simply re-reoute to dig-with, since we
     *   generally need a digging implement to dig in anything.  Some
     *   objects might want to override this to allow digging without any
     *   implement; a sandy beach, for example, might allow digging in the
     *   sand without a shovel.  
     */
    dobjFor(Dig)
    {
        preCond = [touchObj]
        verify() { }
        action() { askForIobj(DigWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "DigWith" action 
     */
    dobjFor(DigWith)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotDig); }
    }
    iobjFor(DigWith)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotDigWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "jump over" 
     */
    dobjFor(JumpOver)
    {
        verify() { illogical(&cannotJumpOver); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "jump off" 
     */
    dobjFor(JumpOff)
    {
        verify() { illogical(&cannotJumpOff); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Push" action 
     */
    dobjFor(Push)
    {
        preCond = [touchObj]
        verify() { logicalRank(50, 'not pushable'); }
        action() { reportFailure(&pushNoEffect); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Pull" action 
     */
    dobjFor(Pull)
    {
        preCond = [touchObj]
        verify() { logicalRank(50, 'not pullable'); }
        action() { reportFailure(&pullNoEffect); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Move" action 
     */
    dobjFor(Move)
    {
        preCond = [touchObj]
        verify() { logicalRank(50, 'not movable'); }
        action() { reportFailure(&moveNoEffect); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "MoveWith" action 
     */
    dobjFor(MoveWith)
    {
        preCond = [touchObj]
        verify() { logicalRank(50, 'not movable'); }
        action() { reportFailure(&moveNoEffect); }
    }
    iobjFor(MoveWith)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotMoveWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "MoveTo" action 
     */
    dobjFor(MoveTo)
    {
        preCond = [touchObj]
        verify() { logicalRank(50, 'not movable'); }
        action() { reportFailure(&moveToNoEffect); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Turn" action 
     */
    dobjFor(Turn)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotTurn); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Turn to" action 
     */
    dobjFor(TurnTo)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotTurn); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "TurnWith" action 
     */
    dobjFor(TurnWith)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotTurn); }
    }
    iobjFor(TurnWith)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotTurnWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Consult" action 
     */
    dobjFor(Consult)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotConsult); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Type on" action 
     */
    dobjFor(TypeOn)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotTypeOn); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Enter on" action 
     */
    dobjFor(EnterOn)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotEnterOn); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Switch" action 
     */
    dobjFor(Switch)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotSwitch); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Flip" action 
     */
    dobjFor(Flip)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotFlip); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "TurnOn" action 
     */
    dobjFor(TurnOn)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotTurnOn); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "TurnOff" action 
     */
    dobjFor(TurnOff)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotTurnOff); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Light" action.  By default, we treat this as equivalent to
     *   "burn".  
     */
    dobjFor(Light) asDobjFor(Burn)

    /* -------------------------------------------------------------------- */
    /*
     *   "Burn".  By default, we ask for something to use to burn the
     *   object, since most objects are not self-igniting.  
     */
    dobjFor(Burn)
    {
        preCond = [touchObj]
        verify()
        {
            /* 
             *   although we can in principle burn anything, most things
             *   are unlikely choices for burning
             */
            logicalRank(50, 'not flammable');
        }
        action()
        {
            /* rephrase this as a "burn with" command */
            askForIobj(BurnWith);
        }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Burn with" 
     */
    dobjFor(BurnWith)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotBurn); }
    }
    iobjFor(BurnWith)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotBurnWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Extinguish" 
     */
    dobjFor(Extinguish)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotExtinguish); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "AttachTo" action 
     */
    dobjFor(AttachTo)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotAttach); }
    }
    iobjFor(AttachTo)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotAttachTo); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "DetachFrom" action 
     */
    dobjFor(DetachFrom)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotDetach); }
    }
    iobjFor(DetachFrom)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotDetachFrom); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Detach" action 
     */
    dobjFor(Detach)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotDetach); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Break" action 
     */
    dobjFor(Break)
    {
        preCond = [touchObj]
        verify() { illogical(&shouldNotBreak); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Cut with" action 
     */
    dobjFor(CutWith)
    {
        preCond = [touchObj]
        verify() { logicalRank(50, 'not cuttable'); }
        action() { reportFailure(&cutNoEffect); }
    }

    iobjFor(CutWith)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotCutWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Climb", "climb up", and "climb down" actions
     */
    dobjFor(Climb)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotClimb); }
    }

    dobjFor(ClimbUp)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotClimb); }
    }

    dobjFor(ClimbDown)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotClimb); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Open" action 
     */
    dobjFor(Open)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotOpen); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Close" action 
     */
    dobjFor(Close)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotClose); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Lock" action 
     */
    dobjFor(Lock)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotLock); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Unlock" action 
     */
    dobjFor(Unlock)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnlock); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "LockWith" action 
     */
    dobjFor(LockWith)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotLock); }
    }
    iobjFor(LockWith)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotLockWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "UnlockWith" action 
     */
    dobjFor(UnlockWith)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnlock); }
    }
    iobjFor(UnlockWith)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotUnlockWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Eat" action 
     */
    dobjFor(Eat)
    {
        /* 
         *   generally, an object must be held to be eaten; this can be
         *   overridden on an object-by-object basis as desired 
         */
        preCond = [objHeld]
        verify() { illogical(&cannotEat); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Drink" action 
     */
    dobjFor(Drink)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotDrink); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Pour" 
     */
    dobjFor(Pour)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPour); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Pour into" 
     */
    dobjFor(PourInto)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPour); }
    }
    iobjFor(PourInto)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPourInto); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Pour onto" 
     */
    dobjFor(PourOnto)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPour); }
    }
    iobjFor(PourOnto)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPourOnto); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Clean" action 
     */
    dobjFor(Clean)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotClean); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "CleanWith" action 
     */
    dobjFor(CleanWith)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotClean); }
    }
    iobjFor(CleanWith)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotCleanWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "SitOn" action 
     */
    dobjFor(SitOn)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotSitOn); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "LieOn" action 
     */
    dobjFor(LieOn)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotLieOn); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "StandOn" action 
     */
    dobjFor(StandOn)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotStandOn); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Board" action 
     */
    dobjFor(Board)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotBoard); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Get out of" (unboard) action 
     */
    dobjFor(GetOutOf)
    {
        verify() { illogical(&cannotUnboard); }
    }

    /*
     *   "Get off of" action 
     */
    dobjFor(GetOffOf)
    {
        verify() { illogical(&cannotGetOffOf); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Fasten" action 
     */
    dobjFor(Fasten)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotFasten); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Fasten to" action 
     */
    dobjFor(FastenTo)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotFasten); }
    }
    iobjFor(FastenTo)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotFastenTo); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Unfasten" action 
     */
    dobjFor(Unfasten)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnfasten); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Unfasten from" action 
     */
    dobjFor(UnfastenFrom)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnfasten); }
    }
    iobjFor(UnfastenFrom)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnfastenFrom); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "PlugIn" action 
     */
    dobjFor(PlugIn)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPlugIn); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "PlugInto" action 
     */
    dobjFor(PlugInto)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPlugIn); }
    }
    iobjFor(PlugInto)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPlugInTo); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Unplug" action 
     */
    dobjFor(Unplug)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnplug); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "UnplugFrom" action 
     */
    dobjFor(UnplugFrom)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnplug); }
    }
    iobjFor(UnplugFrom)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnplugFrom); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Screw" action 
     */
    dobjFor(Screw)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotScrew); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "ScrewWith" action 
     */
    dobjFor(ScrewWith)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotScrew); }
    }
    iobjFor(ScrewWith)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotScrewWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Unscrew" action 
     */
    dobjFor(Unscrew)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnscrew); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "UnscrewWith" action 
     */
    dobjFor(UnscrewWith)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotUnscrew); }
    }
    iobjFor(UnscrewWith)
    {
        preCond = [objHeld]
        verify() { illogical(&cannotUnscrewWith); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   "Enter" 
     */
    dobjFor(Enter)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotEnter); }
    }

    /* 
     *   "Go through" 
     */
    dobjFor(GoThrough)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotGoThrough); }
    }

    /* -------------------------------------------------------------------- */
    /*
     *   Push in Direction action - this is for commands like "push
     *   boulder north" or "drag sled into cave". 
     */
    dobjFor(PushTravel)
    {
        preCond = [touchObj]
        verify() { illogical(&cannotPushTravel); }
    }

    /* 
     *   For all of the two-object forms, map these using our general
     *   push-travel mapping.  We do all of this mapping here, rather than
     *   in the action definition, so that individual objects can change
     *   the meanings of these verbs for special cases as appropriate.  
     */
    mapPushTravelHandlers(PushTravelThrough, GoThrough)
    mapPushTravelHandlers(PushTravelEnter, Enter)
    mapPushTravelHandlers(PushTravelGetOutOf, GetOutOf)
    mapPushTravelHandlers(PushTravelClimbUp, ClimbUp)
    mapPushTravelHandlers(PushTravelClimbDown, ClimbDown)
;

