#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - Status Line
 *   
 *   This module defines the framework for displaying the status line,
 *   which is the area conventionally displayed at the top of the screen
 *   showing information such as the current location, score (if scoring
 *   is used at all), and number of turns.  
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   The OutputStream for the status line.  We use a separate output
 *   stream for this so that we don't affect the main text area when
 *   updating the status line.  
 */
transient statuslineOutputStream: OutputStream
    /* 
     *   Assume we'll be able to use a banner API window, or the simple
     *   text status line.  If we can use either of these, then we'll be
     *   using our own private output stream at the interpreter level,
     *   separate from the main text window's output stream, in which case
     *   we won't affect the main text window's input stream.  Therefore,
     *   assume that we won't need to coordinate with the input manager.  
     */
    myInputManager = nil

    /* we sit atop the system-level main console output stream */
    writeFromStream(txt)
    {
        if (bannerHandle != nil)
        {
            /* write the text to our banner */
            bannerSay(bannerHandle, txt);
        }
        else
        {
            /* no banner window - write the text to the console */
            tadsSay(txt);
        }
    }

    /* begin statusline display */
    beginStatusLine()
    {
        /* flush the main window */
        flushOutput();

        /* if we haven't initialized yet, do so now */
        if (!statusBannerInited)
        {
            /* note that we've tried initializing */
            statusBannerInited = true;

            /* create our statusline window */
            createStatuslineBanner();

            /* 
             *   If we failed to create a banner window, and we're running
             *   on a full HTML interpreter, then we'll have to use
             *   <BANNER> tags to produce the status line.  Since we'll
             *   write our <BANNER> tags to the main text window in this
             *   situation, each status line update will affect the main
             *   text window's input/output streams, so we will have to
             *   coordinate with the main input manager.
             *   
             *   We don't have to coordinate with the input manager if
             *   we're using a true banner API window, because we'll be
             *   able to write directly to the independent output stream
             *   provided by our separate window.  We also don't have to
             *   coordinate if we're using the old-style "status-mode"
             *   system, because that also uses an effectively independent
             *   output stream that doesn't affect the main text window.  
             */
            if (bannerHandle == nil
                && systemInfo(SysInfoInterpClass) == SysInfoIClassHTML)
            {
                /* 
                 *   it's an HTML interpreter - note that we'll use
                 *   <BANNER> tags for the statusline 
                 */
                bannerTagMode = true;
                
                /* 
                 *   our <BANNER> tags go to the main output stream, so we
                 *   need to coordinate our output with the main input
                 *   manager 
                 */
                myInputManager = inputManager;
            }
        }

        /* check what kind of statusline display we're using */
        if (bannerHandle != nil)
        {
            /* we have a banner API window - clear the window */
            bannerClear(bannerHandle);

            /*
             *   If the platform doesn't support size-to-contents, then set
             *   the height to our best estimate for the size.
             *   
             *   If we do support size-to-contents, we'll set the height to
             *   the exact rendered size when we're done, so we don't need
             *   to worry about setting an estimate; indicate this to the
             *   interpreter by setting the is-advisory flag to true.  
             */
            bannerSetSize(bannerHandle, statusLine.getEstimatedHeightHtml(),
                          BannerSizeAbsolute, true);

            /* set up the statusline color in the window */
            writeToStream('<body bgcolor=statusbg text=statustext>');
        }
        else if (bannerTagMode)
        {
            /* HTML mode - use a <BANNER> tag for the statusline */
            writeToStream('<banner id=StatusLine height=previous
                border><body bgcolor=statusbg text=statustext>');
        }
        else
        {
            /* plain text mode - enter text status mode */
            statusMode(StatModeStatus);
        }
    }

    /* end statusline display */
    endStatusLine()
    {
        /* check the type of statusline we're generating */
        if (bannerHandle != nil)
        {
            /* end the last line */
            writeToStream('\n');

            /* 
             *   it's a banner API window - flush the stream for the
             *   window, then size the window to its current contents
             *   (which might not do anything on some platforms, but will
             *   give us the optimal size where it is supported) 
             */
            bannerFlush(bannerHandle);
            bannerSizeToContents(bannerHandle);
        }
        else if (bannerTagMode)
        {
            /* HTML <BANNER> mode - end the <BANNER> tag */
            writeToStream('</banner>');
        }
        else
        {
            /* plain text statusline - end status mode */
            statusMode(StatModeNormal);
        }
    }

    /*
     *   Create the statusline banner window.  This attempts to create a
     *   banner window for the statusline using the banner API.  
     */
    createStatuslineBanner()
    {
        /* 
         *   Try creating a banner window.  Use a top-aligned window;
         *   don't worry about the size for now, since we'll adjust the
         *   size to fit the contents when we've finished generating the
         *   status display.
         *   
         *   Note that we require <TAB> alignment capabilities in the new
         *   window, because we use <TAB> in the default layout to display
         *   the score/turn counters at the right side of the status line.
         *   
         *   Games that completely override the layout of the status line
         *   might not need <TAB> alignment, in which case they might want
         *   to override this method to remove the need-tab-align flag.
         *   The need-tab-align flag might cause some interpreters to use
         *   a fixed-pitch font in the statusline window even if
         *   proportionally-spaced fonts are available, which might not be
         *   desirable to some authors.  If <TAB> alignment isn't needed,
         *   then this style flag can be eliminated if desired.  
         */
        bannerHandle = bannerCreate(BannerFirst, nil,
                                    BannerTypeText, BannerAlignTop,
                                    nil, nil,
                                    BannerStyleBorder | BannerStyleTabAlign);
    }

    /*
     *   My banner window.  If we have banner API support, we'll write our
     *   output to this window; otherwise, we'll write directly to the
     *   main output stream, using either HTML <banner> tags or the
     *   traditional modal statusline generation.  
     */
    bannerHandle = nil

    /* flag: we've initialized our banner window (or tried to) */
    statusBannerInited = nil

    /* flag: we're using HTML <BANNER> tags for the statusline */
    bannerTagMode = nil
;

/*
 *   The OutputStream for the right half of the status line (the
 *   score/turn count area) in text mode.  We use a separate stream for
 *   this because we have to write this text with the special
 *   statusRight() intrinsic in text mode. 
 */
transient statusRightOutputStream: OutputStream
    /*
     *   Write from the stream.  We simply buffer up text until we're
     *   asked to display the final data. 
     */
    writeFromStream(txt)
    {
        /* buffer the text */
        buf_ += txt;
    }

    /*
     *   Flush the buffer.  This writes whatever we've buffered up to the
     *   right half of the text-mode status line. 
     */
    flushStream()
    {
        /* write the text to the system console */
        statusRight(buf_);

        /* we no longer have anything buffered */
        buf_ = '';
    }

    /* our buffered text */
    buf_ = ''
;


/* ------------------------------------------------------------------------ */
/*
 *   Status line - this is an abstract object that controls the status line
 *   display.
 *   
 *   We provide two main methods: showStatusHtml, which shows the status
 *   line in HTML format, and showStatusText, which shows the status line
 *   in plain text mode.  To display the status line, we invoke one or the
 *   other of these methods, according to the current mode, to display the
 *   statusline.  The default implementations of these methods generate the
 *   appropriate formatting codes for a statusline with a left part and a
 *   right part, calling showStatusLeft and showStatusRight, respectively,
 *   to display the text for the parts.
 *   
 *   Games can customize the statusline at two levels.  At the simpler
 *   level, a game can modify showStatusLeft and/or showStatusRight to
 *   change the text displayed on the left and/or right of the statusline.
 *   Since these two methods are used regardless of the statusline style of
 *   the underlying interpreter, games don't have to worry about the
 *   different modes when overriding these.
 *   
 *   At the more complex level, a game can modify showStatusHtml and/or
 *   showStatusText.  Modifying these routines provides complete control
 *   over the formatting of the entir status line.  If a game wants to use
 *   something other than the traditional left/right display, it must
 *   modify these methods.  
 */
statusLine: object
    /* 
     *   Show the status line, in HTML or text mode, as appropriate.  By
     *   default, the library sets this up as a "prompt daemon," which
     *   means that this will be called automatically just before each
     *   command line is read.  
     */
    showStatusLine()
    {
        local oldStr;

#ifdef SENSE_CACHE
        /* 
         *   showing the status line doesn't normally change any game
         *   state, so we can turn on the sense cache while generating the
         *   display 
         */
        libGlobal.enableSenseCache();
#endif

        /* start the status line in the output stream */
        statuslineOutputStream.beginStatusLine();

        /* switch to the status line output stream */
        oldStr = outputManager.setOutputStream(statuslineOutputStream);

        /* make sure we restore statusline mode before we're done */
        try
        {
            /*
             *   Generate a text or HTML status line, as appropriate.
             *   
             *   If we have a banner window, simply display our statusline
             *   contents to our window.
             *   
             *   If we don't have a banner window, we must use the
             *   old-style one-line status display, using status mode
             *   switching.  
             */
            if (statuslineOutputStream.bannerHandle != nil
                || statuslineOutputStream.bannerTagMode)
            {
                /* show the HTML status line */
                showStatusHtml();
            }
            else
            {
                /* show the status line in plain text mode */
                showStatusText();
            }
        }
        finally
        {
            /* restore the original default output stream */
            outputManager.setOutputStream(oldStr);

            /* end the status line in the output stream */
            statuslineOutputStream.endStatusLine();

#ifdef SENSE_CACHE
            /* turn off sense caching */
            libGlobal.disableSenseCache();
#endif
        }
    }

    /*
     *   Show the status line in HTML format.  Our default implementation
     *   shows the traditional two-part (left/right) status line, using
     *   showStatusLeft() and showStatusRight() to display the parts.  
     */
    showStatusHtml()
    {
        /* hyperlink the location name to a "look around" command */
        "<a plain href='<<libMessages.commandLookAround>>'>";
            
        /* show the left part of the status line */
        showStatusLeft();
            
        /* set up for the score part on the right half */
        "</a><tab align=right><a plain
            href='<<libMessages.commandFullScore>>'>";
        
        /* show the right part of the status line */
        showStatusRight();
        
        /* end the score link */
        "</a>";
        
        /* add the status-line exit list, if desired */
        gPlayerChar.location.showStatuslineExits();
    }

    /*
     *   Get the estimated HTML-style banner height, in lines of text.
     *   This is used to set the status line banner size for platforms
     *   where sizing to the exact height of the rendered contents isn't
     *   supported.
     *   
     *   If showStatusHtml() is overridden to display more or fewer lines
     *   of text than the basic implementation here, then this routine must
     *   be overridden as well to reflect the new height.  
     */
    getEstimatedHeightHtml()
    {
        local ht;
        
        /* 
         *   we need one line for the basic display (the location name and
         *   score/turn count) 
         */
        ht = 1;

        /* add in the estimated height of the exits display, if appropriate */
        ht += gPlayerChar.location.getStatuslineExitsHeight();

        /* return the result */
        return ht;
    }

    /*
     *   Show the statusline in text mode.  Our default implementation
     *   shows the traditional two-part (left/right) status line, using
     *   showStatusLeft() and showStatusRight() to display the parts.  
     */
    showStatusText()
    {
        /* show the left part of the display */
        showStatusLeft();

        /* switch to the right-side status stream */
        outputManager.setOutputStream(statusRightOutputStream);

        /* show the right-half text */
        showStatusRight();
        
        /* flush the right-side stream */
        statusRightOutputStream.flushStream();
    }

    /*
     *   Show the left part of a standard left/right statusline.  By
     *   default, we'll show the player character's 
     */
    showStatusLeft()
    {
        local actor;

        /* get the player character actor */
        actor = gPlayerChar;

        "<.statusroom>";

        /* show the actor's location's status name */
        if (actor != nil && actor.location != nil)
            actor.location.statusName(actor);

        "<./statusroom>";
    }

    /*
     *   Show the right part of a standard left/right statusline.  By
     *   default, we'll show the current score, a slash, and the number of
     *   turns. 
     */
    showStatusRight()
    {
        local s;

        /* if there's a score object, show the score */
        if ((s = libGlobal.scoreObj) != nil)
        {
            /* show the score and the number of turns so far */
            "<.statusscore><<s.totalScore>>/<<
            libGlobal.totalTurns>><./statusscore>";
        }
    }
;

/* in case the 'score' module isn't included */
property totalScore;
