#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - Lister class
 *   
 *   This module defines the "Lister" class, which generates formatted
 *   lists of objects, and several subclasses of Lister that generate
 *   special kinds of lists.  
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   Lister.  This is the base class for formatting of lists of objects.
 *   
 *   The external interface consists of the showList() method, which
 *   displays a formatted list of objects according to the rules of the
 *   lister subclass.
 *   
 *   The rest of the methods are an internal interface which lister
 *   subclasses can override to customize the way that a list is shown.
 *   Certain of these methods are meant to be overridden by virtually all
 *   listers, such as the methods that show the prefix and suffix
 *   messages.  The remaining methods are designed to allow subclasses to
 *   customize detailed aspects of the formatting, so they only need to be
 *   overridden when something other than the default behavior is needed.  
 */
class Lister: object
    /*
     *   Show a list, showing all items in the list as though they were
     *   fully visible, regardless of their actual sense status.  
     */
    showListAll(lst, options, indent)
    {
        local infoTab;
    
        /* create a sense information table with each item in full view */
        infoTab = new LookupTable(16, 32);
        foreach (local cur in lst)
        {
            /* add a plain view sensory description to the info list */
            infoTab[cur] = new SenseInfo(cur, transparent, nil, 3);
        }
        
        /* show the list from the current global point of view */
        showList(getPOV(), nil, lst, options, indent, infoTab, nil);
    }

    /*
     *   Display a list of items, grouping according to the 'listWith'
     *   associations of the items.  We will only list items for which
     *   isListed() returns true.
     *   
     *   'pov' is the point of view of the listing, which is usually an
     *   actor (and usually the player character actor).
     *   
     *   'parent' is the parent (container) of the list being shown.  This
     *   should be nil if the listed objects are not all within a single
     *   object.
     *   
     *   'lst' is the list of items to display.
     *   
     *   'options' gives a set of ListXxx option flags.
     *   
     *   'indent' gives the indentation level.  This is used only for
     *   "tall" lists (specified by including ListTall in the options
     *   flags).  An indentation level of zero indicates no indentation.
     *   
     *   'infoTab' is a lookup table of SenseInfo objects for all of the
     *   objects that can be sensed from the perspective of the actor
     *   performing the action that's causing the listing.  This is
     *   normally the table returned from Thing.senseInfoTable() for the
     *   actor from whose point of view the list is being generated.  (We
     *   take this as a parameter rather than generating ourselves for two
     *   reasons.  First, it's often the case that the same information
     *   table will be needed for a series of listings, so we can save the
     *   compute time of recalculating the same table repeatedly by having
     *   the caller obtain the table and pass it to each lister.  Second,
     *   in some cases the caller will want to synthesize a special sense
     *   table rather than using the actual sense information; taking this
     *   as a parameter allows the caller to easily customize the table.)
     *   
     *   'parentGroup' is the ListGroup object that is showing this list.
     *   We will not group the objects we list into the parent group, or
     *   into any group more general than the parent group.  
     *   
     *   This routine is not usually overridden in lister subclasses.
     *   Instead, this method calls a number of other methods that
     *   determine the listing style in more detail; usually those other,
     *   simpler methods are customized in subclasses.  
     */
    showList(pov, parent, lst, options, indent, infoTab, parentGroup)
    {
        local cur;
        local i, cnt;
        local listCount;
        local dispCount;
        local groups;
        local groupTab;
        local singles;
        local sublists;
        local itemOptions;
        local groupOptions;
        local filteredList;
        local itemCount;
        local origLst;

        /* remember the original list */
        origLst = lst;

        /* narrow the list down to those items that will be listed */
        lst = lst.subset({x: isListed(x)});
        
        /* 
         *   If we have an infoTab, build a new list consisting only of
         *   the items in 'lst' that have infoTab entries - we can't sense
         *   anything that doesn't have an infoTab entry, so we don't want
         *   to show any such objects.  
         */
        if (infoTab != nil)
        {
            /* create a vector to build the new filtered list */
            filteredList = new Vector(lst.length());

            /* 
             *   run through our original list and confirm that each one
             *   is in the infoTab
             */
            foreach (local cur in lst)
            {
                /* 
                 *   if this item has an infoTab entry, add this item to
                 *   the filtered list 
                 */
                if (infoTab[cur] != nil)
                    filteredList.append(cur);
            }
        
            /* forget the original list, and use the filtered list instead */
            lst = filteredList;
        }

        /* create a lookup table to keep track of the groups we've seen */
        groupTab = new LookupTable();
        groups = new Vector(10);
        
        /* no singles yet */
        singles = new Vector(10);
        
        /* 
         *   First, scan the list to determine how we're going to group
         *   the objects.
         */
        for (i = 1, cnt = lst.length() ; i <= cnt ; ++i)
        {
            local curGroups;
            local parentIdx;
            
            /* get this object into a local for easy reference */
            cur = lst[i];
            
            /* if the item isn't part of this listing, skip it */
            if (!isListed(cur))
                continue;

            /* get the list of groups with which this object is listed */
            curGroups = listWith(cur);

            /* if there are no groups, we can move on to the next item */
            if (curGroups == nil)
                continue;

            /* 
             *   If we have a parent group, and it appears in the list of
             *   groups for this item, eliminate everything in the item's
             *   group list up to and including the parent group.  If
             *   we're showing this list as part of a group to begin with,
             *   we obviously don't want to show this list grouped into
             *   the same group, and we also don't want to group it into
             *   anything broader than the parent group.  Groups are
             *   listed from most general to most specific, so we can
             *   eliminate anything up to and including the parent group. 
             */
            if (parentGroup != nil
                && (parentIdx = curGroups.indexOf(parentGroup)) != nil)
            {
                /* eliminate everything up to and including the parent */
                curGroups = curGroups.sublist(parentIdx + 1);
            }

            /* if this item has no groups, skip it */
            if (curGroups.length() == 0)
                continue;

            /*
             *   This item has one or more group associations that we must
             *   consider.
             */
            foreach (local g in curGroups)
            {
                local itemsInGroup;
                
                /* find the group table entry for this group */
                itemsInGroup = groupTab[g];

                /* if there's no entry for this group, create a new one */
                if (itemsInGroup == nil)
                {
                    /* create a new group table entry */
                    itemsInGroup = groupTab[g] = new Vector(10);

                    /* add it to the group vector */
                    groups.append(g);
                }

                /* 
                 *   add this item to the list of items that want to be
                 *   grouped with this group 
                 */
                itemsInGroup.append(cur);
            }
        }

        /*
         *   We now have the set of all of the groups that could possibly
         *   be involved in this list display.  We must now choose the
         *   single group we'll use to display each grouped object.
         *   
         *   First, eliminate any groups with only one item.  
         */
        for (i = 1, cnt = groups.length() ; i <= cnt ; ++i)
        {
            /* if this group has only one member, drop it */
            if (groupTab[groups[i]].length() < 2)
            {
                /* remove this group from the group list */
                groups.removeElementAt(i);

                /* 
                 *   adjust the list count, and back up to try the element
                 *   newly at this index on the next iteration 
                 */
                --cnt;
                --i;
            }
        }

        /*
         *   Next, scan for groups with identical member lists, and for
         *   groups with subset member lists.  For each pair of identical
         *   elements we find, eliminate the more general of the two.  
         */
        for (i = 1, cnt = groups.length() ; i <= cnt ; ++i)
        {
            local g1;
            local mem1;

            /* get the current group and its membership list */
            g1 = groups[i];
            mem1 = groupTab[g1];

            /* look for matching items in the list after this one */
            for (local j = i + 1 ; j <= cnt ; ++j)
            {
                local g2;
                local mem2;

                /* get the current item and its membership list */
                g2 = groups[j];
                mem2 = groupTab[g2];
                
                /*
                 *   Compare the membership lists for the two items.  Note
                 *   that we built these membership lists all in the same
                 *   order of objects, so if two membership lists have all
                 *   the same members, those members will be in the same
                 *   order in the two lists; hence, we can simply compare
                 *   the two lists to determine the membership order.  
                 */
                if (mem1 == mem2)
                {
                    local ordList;
                    
                    /* 
                     *   The groups have identical membership, so
                     *   eliminate the more general group.  Groups are
                     *   ordered from most general to least general, so
                     *   keep the one with the higher index in the group
                     *   list for an object in the membership list.  Note
                     *   that we assume that each member has the same
                     *   ordering for the common groups, so we can pick a
                     *   member arbitrarily to find the way a member
                     *   orders the groups.  
                     */
                    ordList = listWith(mem1[1]);
                    if (ordList.indexOf(g1) > ordList.indexOf(g2))
                    {
                        /* 
                         *   group g1 is more specific than group g2, so
                         *   keep g1 and discard g2 - remove the 'j'
                         *   element from the list, and back up in the
                         *   inner loop so we reconsider the element newly
                         *   at this index on the next iteration 
                         */
                        groups.removeElementAt(j);
                        --cnt;
                        --j;
                    }
                    else
                    {
                        /* 
                         *   group g2 is more specific, so discard g1 -
                         *   remove the 'i' element from the list, back up
                         *   in the outer loop, and break out of the inner
                         *   loop, since the outer loop element is no
                         *   longer there for us to consider in comparing
                         *   more elements in the inner loop 
                         */
                        groups.removeElementAt(i);
                        --cnt;
                        --i;
                        break;
                    }
                }
            }
        }

        /*
         *   Scan for subsets.  For each group whose membership list is a
         *   subset of another group in our list, eliminate the subset,
         *   keeping only the larger group.  The group lister will be able
         *   to show the subgroup as grouped within its larger list.  
         */
        for (local i = 1, cnt = groups.length() ; i <= cnt ; ++i)
        {
            local g1;
            local mem1;

            /* get the current group and its membership list */
            g1 = groups[i];
            mem1 = groupTab[g1];

            /* look at the other elements to see if we have any subsets */
            for (local j = 1 ; j <= cnt ; ++j)
            {
                local g2;
                local mem2;

                /* don't bother checking the same element */
                if (j == i)
                    continue;

                /* get the current item and its membership list */
                g2 = groups[j];
                mem2 = groupTab[g2];

                /* 
                 *   if g2's membership is a subset, eliminate g2 from the
                 *   group list 
                 */
                if (isListSubset(mem2, mem1))
                {
                    /* remove g2 from the list */
                    groups.removeElementAt(j);

                    /* adjust the loop counters for the removal */
                    --cnt;
                    --j;

                    /* 
                     *   adjust the outer loop counter if it's affected -
                     *   the outer loop is affected if it's already past
                     *   this point in the list, which means that its
                     *   index is higher than the inner loop index 
                     */
                    if (i > j)
                        --i;
                }
            }
        }

        /*
         *   We now have a final accounting of the groups that we will
         *   consider using.  Reset the membership list for each group in
         *   the surviving list. 
         */
        foreach (local g in groups)
        {
            local itemsInList;

            /* get this group's membership list vector */
            itemsInList = groupTab[g];

            /* clear the vector */
            itemsInList.removeRange(1, itemsInList.length());
        }

         /*
          *   Now, run through our item list again, and assign each item
          *   to the surviving group that comes earliest in the item's
          *   group list.  
          */
        for (i = 1, cnt = lst.length() ; i <= cnt ; ++i)
        {
            local curGroups;
            local winningGroup;

            /* get this object into a local for easy reference */
            cur = lst[i];
            
            /* if the item isn't part of this listing, skip it */
            if (!isListed(cur))
                continue;

            /* get the list of groups with which this object is listed */
            curGroups = listWith(cur);
            if (curGroups == nil)
                curGroups = [];

            /* 
             *   find the first element in the group list that is in the
             *   surviving group list
             */
            winningGroup = nil;
            foreach (local g in curGroups)
            {
                /* if this group is in the surviving list, it's the one */
                if (groups.indexOf(g) != nil)
                {
                    winningGroup = g;
                    break;
                }
            }

            /* 
             *   if we have a group, add this item to the group's
             *   membership; otherwise, add it to the singles list 
             */
            if (winningGroup != nil)
                groupTab[winningGroup].append(cur);
            else
                singles.append(cur);
        }

        /* eliminate any surviving group that has only one member */
        for (i = 1, cnt = groups.length() ; i <= cnt ; ++i)
        {
            local mem;

            /* get this group's membership list */
            mem = groupTab[groups[i]];

            /* 
             *   if this group's membership is singular, eliminate the
             *   group and add the member into the singles pile
             */
            if (mem.length() < 2)
            {
                /* put the item into the singles list */
                if (mem.length() > 0)
                    singles.append(mem[1]);

                /* eliminate this item from the group list */
                groups.removeElementAt(i);

                /* adjust the loop counters */
                --cnt;
                --i;
            }
        }

        /*
         *   We now know how we're grouping everything.  Add up the total
         *   cardinality - count one per single item, plus the cardinality
         *   of each group. 
         */
        itemCount = singles.length();
        foreach (local g in groups)
        {
            /* add the grammatical cardinality of this group */
            itemCount += g.groupCardinality(self, groupTab[g]);
        }

        /*
         *   We now know how many items we're listing (grammatically
         *   speaking), so we can call the list interface object to set up
         *   the list.  If we're displaying nothing at all, just ask the
         *   interface object to display the message for an empty list,
         *   and we're done.  
         */
        if (itemCount == 0)
        {
            /* show the empty list */
            showListEmpty(pov, parent);
        }
        else
        {
            /* 
             *   Check to see if we have one or more group sublists - if
             *   we do, we must use the "long" list format for our overall
             *   list, otherwise we can use the normal "short" list
             *   format.  The long list format uses semicolons to separate
             *   items.  
             */
            for (i = 1, cnt = groups.length(), sublists = nil ;
                 i <= cnt ; ++i)
            {
                /* 
                 *   if this group's lister item displays a sublist, we
                 *   must use the long format 
                 */
                if (groups[i].groupDisplaysSublist)
                {
                    /* note that we are using the long format */
                    sublists = true;
                    
                    /* 
                     *   one is enough to make us use the long format, so
                     *   we need not look any further 
                     */
                    break;
                }
            }
            
            /* generate the prefix message if we're in a 'tall' listing */
            if ((options & ListTall) != 0)
            {
                /* indent the prefix */
                showListIndent(options, indent);
                
                /* 
                 *   Show the prefix.  If this is a contents listing, show
                 *   the contents prefix; otherwise show the full list
                 *   prefix.  
                 */
                if ((options & ListContents) != 0)
                    showListContentsPrefixTall(itemCount, pov, parent);
                else
                    showListPrefixTall(itemCount, pov, parent);
                
                /* go to a new line for the list contents */
                "\n";
                
                /* indent the items one level now, since we showed a prefix */
                ++indent;
            }
            else
            {
                /* show the prefix */
                showListPrefixWide(itemCount, pov, parent);
            }
            
            /* 
             *   regardless of whether we're adding long formatting to the
             *   main list, display the group sublists with whatever
             *   formatting we were originally using 
             */
            groupOptions = options;
            
            /* show each item with our current set of options */
            itemOptions = options;
            
            /* 
             *   if we're using sublists, show "long list" separators in
             *   the main list 
             */
            if (sublists)
                itemOptions |= ListLong;
            
            /* 
             *   calculate the number of items we'll show in the list -
             *   each group shows up as one list entry, so the total
             *   number of list entries is the number of single items plus
             *   the number of groups 
             */
            listCount = singles.length() + groups.length();

            /*
             *   Show the items.  Run through the (filtered) original
             *   list, so that we show everything in the original sorting
             *   order.  
             */
            dispCount = 0;
            foreach (cur in lst)
            {
                local group;
                local displayedCur;
                
                /* presume we'll display this item */
                displayedCur = true;
                
                /* mark the object as having been seen from this POV */
                markAsSeen(cur, pov);
                
                /*
                 *   Figure out how to show this item: if it's in the
                 *   singles list, show it as a single item; if it's in
                 *   the group list, show its group; if it's in a group
                 *   we've previously shown, show nothing, as we showed
                 *   the item when we showed its group.  
                 */
                if (singles.indexOf(cur) != nil)
                {
                    /*
                     *   It's in the singles list, so show it as a single
                     *   item.
                     *   
                     *   If the item has contents that we'll display in
                     *   'tall' mode, show the item with its contents - we
                     *   don't need to show the item separately, since it
                     *   will provide a 'tall' list prefix showing itself.
                     *   Otherwise, show the item singly.  
                     */
                    if ((options & ListTall) != 0
                        && (options & ListRecurse) != 0
                        && getListedContents(cur, infoTab) != [])
                    {
                        /* show the item with its contents */
                        showContentsList(pov, cur, itemOptions | ListContents,
                                         indent, infoTab);
                    }
                    else
                    {
                        /* show the list indent if necessary */
                        showListIndent(itemOptions, indent);
                        
                        /* show the item */
                        showListItem(cur, itemOptions, pov, infoTab);
                        
                        /* 
                         *   if we're in wide recursive mode, show the
                         *   item's contents as an in-line parenthetical 
                         */
                        if ((options & ListTall) == 0
                            && (options & ListRecurse) != 0
                            && !contentsListedSeparately(cur))
                        {
                            /* show the item's in-line contents */
                            showInlineContentsList(pov, cur,
                                itemOptions | ListContents,
                                indent + 1, infoTab);
                        }
                    }
                }
                else if ((group = groups.valWhich(
                    {g: groupTab[g].indexOf(cur) != nil})) != nil)
                {
                    /* show the list indent if necessary */
                    showListIndent(itemOptions, indent);
                    
                    /* we found the item in a group, so show its group */
                    group.showGroupList(pov, self, groupTab[group],
                                        groupOptions, indent, infoTab);

                    /* 
                     *   Forget this group - we only need to show each
                     *   group once, since the group shows every item it
                     *   contains.  Since we'll encounter the groups other
                     *   members as we continue to scan the main list, we
                     *   want to make sure we don't show the group again
                     *   when we reach the other items.  
                     */
                    groups.removeElement(group);
                }
                else
                {
                    /* 
                     *   We didn't find the item in the singles list or in
                     *   a group - it must be part of a group that we
                     *   already showed previously, so we don't need to
                     *   show it again now.  Simply make a note that we
                     *   didn't display it.  
                     */
                    displayedCur = nil;
                }
                
                /* if we displayed the item, show a suitable separator */
                if (displayedCur)
                {
                    /* count another list entry displayed */
                    ++dispCount;
                    
                    /* show an appropriate separator */
                    showListSeparator(itemOptions, dispCount, listCount);
                }
            }

            /* 
             *   if we're in 'wide' mode, finish the listing (note that if
             *   this is a 'tall' listing, we're already done, because a
             *   tall listing format doesn't make provisions for anything
             *   after the item list) 
             */
            if ((options & ListTall) == 0)
            {
                /* show the wide-mode list suffix */
                showListSuffixWide(itemCount, pov, parent);
            }
        }

        /* 
         *   If the list is recursive, mention the contents of any items
         *   that weren't listed in the main list.  Don't do this if we're
         *   already recursively showing such a listing, since if we did so
         *   we could show items at recursive depths more than once.  
         */
        if ((options & ListRecurse) != 0
            && indent == 0
            && (options & ListContents) == 0)
        {
            /* show the contents of each object we didn't list */
            showSeparateContents(pov, origLst,
                                 options | ListContents, infoTab);
        }
    }

    /*
     *   Service routine: show the separately-listed contents of the items
     *   in the given list, and their separately-listed contents, and so
     *   on.  This routine is not normally overridden in subclasses, and is
     *   not usually called except from the Lister implementation.
     *   
     *   For each item in the given list, we show the item's contents if
     *   the item is either marked as unlisted, or it's marked as showing
     *   its contents separately.  In the former case, we know that we
     *   cannot have shown the item's contents in-line in the main list,
     *   since we didn't show the item at all in the main list.  In the
     *   latter case, we know that we didn't show the item's contents in
     *   the main list because it's specifically marked as showing its
     *   contents out-of-line.  
     */
    showSeparateContents(pov, lst, options, infoTab)
    {
        /* 
         *   show the separate contents list for each item in the list
         *   which isn't itself listable or which has its contents listed
         *   separately despite its being listed 
         */
        foreach (local cur in lst)
        {
            /* if we didn't list this item, show its listable contents */
            if (!isListed(cur) || contentsListedSeparately(cur))
            {
                /* 
                 *   Show the item's contents.  Note that even though we're
                 *   showing this list recursively, it's actually a new
                 *   top-level list, so show it at indent level zero.  
                 */
                showContentsList(pov, cur, options, 0, infoTab);
            }

            /* 
             *   if this item's contents are listable, recursively do the
             *   same thing with its contents 
             */
            if (contentsListed(cur))
                showSeparateContents(pov, getContents(cur), options, infoTab);
        }
    }

    /*
     *   Service routine: determine if list a is a subset of list b.  a is
     *   a subset of b if every element of a is in b.  
     */
    isListSubset(a, b)
    {
        /* a can't be a subset if it has more elements than b */
        if (a.length() > b.length())
            return nil;
        
        /* check each element of a to see if it's also in b */
        foreach (local cur in a)
        {
            /* if this element of a is not in b, a is not a subset of b */
            if (b.indexOf(cur) == nil)
                return nil;
        }

        /* 
         *   we didn't find any elements of a not in b, so a is a subset
         *   of b
         */
        return true;
    }

    /*	
     *   Show a list indent if necessary.  If ListTall is included in the
     *   options, we'll indent to the given level; otherwise we'll do
     *   nothing.  
     */
    showListIndent(options, indent)
    {
        /* show the indent only if we're in "tall" mode */
        if ((options & ListTall) != 0)
        {
            for (local i = 0 ; i < indent ; ++i)
                "\t";
        }
    }

    /*
     *   Show a newline after a list item if we're in a tall list; does
     *   nothing for a wide list.  
     */
    showTallListNewline(options)
    {
        if ((options & ListTall) != 0)
            "\n";
    }

    /*
     *   Show a simple list, recursing into contents lists if necessary.
     *   We pay no attention to grouping; we just show the items
     *   individually.
     *   
     *   'prevCnt' is the number of items already displayed, if anything
     *   has already been displayed for this list.  This should be zero if
     *   this will display the entire list.  
     */
    showListSimple(pov, lst, options, indent, prevCnt, infoTab)
    {
        local i;
        local cnt;
        local dispCount;
        local totalCount;

        /* calculate the total number of items in the lis t*/
        totalCount = prevCnt + lst.length();
        
        /* display the items */
        for (i = 1, cnt = lst.length(), dispCount = prevCnt ; i <= cnt ; ++i)
        {
            local cur;
            
            /* get the item */
            cur = lst[i];
            
            /* mark the object as having been seen from this point of view */
            markAsSeen(cur, pov);
            
            /* 
             *   If the item has contents that we'll display in 'tall'
             *   mode, show the item with its contents - we don't need to
             *   show the item separately, since it will provide a 'tall'
             *   list prefix showing itself.  Otherwise, show the item
             *   singly.  
             */
            if ((options & ListTall) != 0
                && (options & ListRecurse) != 0
                && getListedContents(cur, infoTab) != [])
            {
                /* show the item with its contents */
                showContentsList(pov, cur, options | ListContents,
                                 indent + 1, infoTab);
            }
            else
            {
                /* show the list indent if necessary */
                showListIndent(options, indent);
                
                /* show the item */
                showListItem(cur, options, pov, infoTab);

                /* 
                 *   if we're in wide recursive mode, show the item's
                 *   contents as an in-line parenthetical 
                 */
                if ((options & ListTall) == 0
                    && (options & ListRecurse) != 0
                    && !contentsListedSeparately(cur))
                {
                    /* show the item's in-line contents */
                    showInlineContentsList(pov, cur,
                                           options | ListContents,
                                           indent + 1, infoTab);
                }
            }

            /* count the item displayed */
            ++dispCount;

            /* show the list separator */
            showListSeparator(options, dispCount, totalCount);
        }
    }

    /*
     *   List the contents of an item.
     *   
     *   'pov' is the point of view, which is usually an actor (and
     *   usually the player character actor).
     *   
     *   'obj' is the item whose contents we are to display.
     *   
     *   'options' is the set of flags that we'll pass to showList(), and
     *   has the same meaning as for that function.
     *   
     *   'infoTab' is a lookup table of SenseInfo objects giving the sense
     *   information for all of the objects that the actor to whom we're
     *   showing the contents listing can sense.  
     */
    showContentsList(pov, obj, options, indent, infoTab)
    {
        /* 
         *   List the item's contents.  By default, use the contentsLister
         *   property of the object whose contents we're showing to obtain
         *   the lister for the contents.  
         */
        obj.showObjectContents(pov, obj.contentsLister, options,
                               indent, infoTab);
    }

    /*
     *   Determine if an object's contents are listed separately from its
     *   own list entry for the purposes of our type of listing.  If this
     *   returns true, then we'll list the object's contents in a separate
     *   listing (a separate sentence following the main listing sentence,
     *   or a separate tree when in 'tall' mode).
     *   
     *   Note that this only matters for objects listed in the top-level
     *   list.  We'll always show the contents separately for an object
     *   that isn't listed in the top-level list (i.e., an object for which
     *   isListed(obj) returns nil).  
     */
    contentsListedSeparately(obj) { return obj.contentsListedSeparately; }

    /*
     *   Show an "in-line" contents list.  This shows the item's contents
     *   list as a parenthetical, as part of a recursive listing.  This is
     *   pretty much the same as showContentsList(), but uses the object's
     *   in-line contents lister instead of its regular contents lister.  
     */
    showInlineContentsList(pov, obj, options, indent, infoTab)
    {
        /* show the item's contents using its in-line contents lister */
        obj.showObjectContents(pov, obj.inlineContentsLister,
                               options, indent, infoTab);
    }

    /* 
     *   Show the prefix for a 'wide' listing - this is a message that
     *   appears just before we start listing the objects.  'itemCount' is
     *   the number of items to be listed; the items might be grouped in
     *   the listing, so a list that comes out as "three boxes and two
     *   books" will have an itemCount of 5.  (The purpose of itemCount is
     *   to allow the message to have grammatical agreement in number.)
     *   
     *   This will never be called with an itemCount of zero, because we
     *   will instead use showListEmpty() to display an empty list.  
     */
    showListPrefixWide(itemCount, pov, parent) { }

    /* 
     *   show the suffix for a 'wide' listing - this is a message that
     *   appears just after we finish listing the objects 
     */
    showListSuffixWide(itemCount, pov, parent) { }

    /* 
     *   Show the list prefix for a 'tall' listing.  Note that there is no
     *   list suffix for a tall listing, because the format doesn't allow
     *   it. 
     */
    showListPrefixTall(itemCount, pov, parent) { }

    /* 
     *   Show the list prefix for the contents of an object in a 'tall'
     *   listing.  By default, we just show our usual tall list prefix.  
     */
    showListContentsPrefixTall(itemCount, pov, parent)
        { showListPrefixTall(itemCount, pov, parent); }

    /*
     *   Show an empty list.  If the list to be displayed has no items at
     *   all, this is called instead of the prefix/suffix routines.  This
     *   can be left empty if no message is required for an empty list, or
     *   can display the complete message appropriate for an empty list
     *   (such as "You are empty-handed").  
     */
    showListEmpty(pov, parent) { }

    /*
     *   Is this item to be listed in room descriptions?  Returns true if
     *   so, nil if not.  By default, we'll use the object's isListed
     *   method to make this determination.  We virtualize this into the
     *   lister interface to allow for different inclusion rules for the
     *   same object depending on the type of list we're generating.  
     */
    isListed(obj) { return obj.isListed(); }

    /*
     *   Are this item's contents listable?  
     */
    contentsListed(obj) { return obj.contentsListed; }

    /*
     *   Get all contents of this item. 
     */
    getContents(obj) { return obj.contents; }

    /*
     *   Get the listed contents of an object.  'infoTab' is the sense
     *   information table for the enclosing listing.  By default, we call
     *   the object's getListedContents() method, but this is virtualized
     *   in the lister interface to allow for listing other hierarchies
     *   besides ordinary contents.  
     */
    getListedContents(obj, infoTab)
    {
        return obj.getListedContents(self, infoTab);
    }

    /*
     *   Get the list of grouping objects for listing the item.  By
     *   default, we return the object's listWith result.  Subclasses can
     *   override this to specify different groupings for the same object
     *   depending on the type of list we're generating.
     *   
     *   The group list returned is in order from most general to most
     *   specific.  For example, if an item is grouped with coins in
     *   general and silver coins in particular, the general coins group
     *   would come first, then the silver coin group, because the silver
     *   coin group is more specific.  
     */
    listWith(obj) { return obj.listWith; }

    /*
     *   Mark the object as seen for the purposes of this listing from the
     *   given point of view (usually an actor).  By default, we call the
     *   object's setSeenBy method.  
     */
    markAsSeen(obj, pov) { obj.setSeenBy(pov, true); }

    /* show an item in a list */
    showListItem(obj, options, pov, infoTab)
    {
        obj.showListItem(options, pov, infoTab);
    }

    /* 
     *   Show a set of equivalent items as a counted item ("three coins").
     *   The listing mechanism itself never calls this directly; instead,
     *   this is provided so that ListGroupEquivalent can ask the lister
     *   how to describe its equivalent sets, so that different listers
     *   can customize the display of equivalent items.
     *   
     *   'lst' is the full list of equivalent items.  By default, we pick
     *   one of these arbitrarily to show, since they're presumably all
     *   the same for the purposes of the list.  
     */
    showListItemCounted(lst, options, pov, infoTab)
    {
        /* 
         *   by defualt, show the counted name for one of the items
         *   (chosen arbitrarily, since they're all the same) 
         */
        lst[1].showListItemCounted(lst, options, pov, infoTab);
    }

    /*
     *   Show a list separator after displaying an item.  curItemNum is
     *   the number of the item just displayed (1 is the first item), and
     *   totalItems is the total number of items that will be displayed in
     *   the list.
     *   
     *   This generic routine is further parameterized by properties for
     *   the individual types of separators.  This default implementation
     *   distinguishes the following separators: the separator between the
     *   two items in a list of exactly two items; the separator between
     *   adjacent items other than the last two in a list of more than two
     *   items; and the separator between the last two elements of a list
     *   of more than two items.
     */
    showListSeparator(options, curItemNum, totalItems)
    {
        local useLong = ((options & ListLong) != 0);
        
        /* if this is a tall list, the separator is simply a newline */
        if ((options & ListTall) != 0)
        {
            "\n";
            return;
        }
        
        /* if that was the last item, there are no separators */
        if (curItemNum == totalItems)
            return;
        
        /* check to see if the next item is the last */
        if (curItemNum + 1 == totalItems)
        {
            /* 
             *   We just displayed the penultimate item in the list, so we
             *   need to use the special last-item separator.  If we're
             *   only displaying two items total, we use an even more
             *   special separator.  
             */
            if (totalItems == 2)
            {
                /* use the two-item separator */
                if (useLong)
                    longListSepTwo;
                else
                    listSepTwo;
            }
            else
            {
                /* use the normal last-item separator */
                if (useLong)
                    longListSepEnd;
                else
                    listSepEnd;
            }
        }
        else
        {
            /* in the middle of the list - display the normal separator */
            if (useLong)
                longListSepMiddle;
            else
                listSepMiddle;
        }
    }

    /* 
     *   Show the specific types of list separators for this list.  By
     *   default, these will use the generic separators defined in the
     *   library messages object (libMessages).  For English, these are
     *   commas and semicolons for short and long lists, respectively; the
     *   word "and" for a list with only two items; and a comma or
     *   semicolon and the word "and" between the last two items in a list
     *   with more than two items.  
     */

    /* 
     *   normal and "long list" separator between the two items in a list
     *   with exactly two items 
     */
    listSepTwo { libMessages.listSepTwo; }
    longListSepTwo { libMessages.longListSepTwo; }

    /* 
     *   normal and long list separator between items in list with more
     *   than two items 
     */
    listSepMiddle { libMessages.listSepMiddle; }
    longListSepMiddle { libMessages.longListSepMiddle; }

    /* 
     *   normal and long list separator between second-to-last and last
     *   items in a list with more than two items 
     */
    listSepEnd { libMessages.listSepEnd; }
    longListSepEnd { libMessages.longListSepEnd; }

    /*
     *   Get my "top-level" lister.  For a sub-lister, this will return
     *   the parent lister's top-level lister.  The default lister is a
     *   top-level lister, so we just return ourself.  
     */
    getTopLister() { return self; }

    /*
     *   The last custom flag defined by this class.  Lister and each
     *   subclass are required to define this so that each subclass can
     *   allocate its own custom flags in a manner that adapts
     *   automatically to future additions of flags to base classes.  As
     *   the base class, we allocate our flags statically with #define's,
     *   so we simply use the fixed #define'd last flag value here.
     */
    nextCustomFlag = ListCustomFlag
;


/* ------------------------------------------------------------------------ */
/*
 *   Plain lister - this lister doesn't show anything for an empty list,
 *   and doesn't show a list suffix or prefix. 
 */
plainLister: Lister
    /* show the prefix/suffix in wide mode */
    showListPrefixWide(itemCount, pov, parent) { }
    showListSuffixWide(itemCount, pov, parent) { }

    /* show the tall prefix */
    showListPrefixTall(itemCount, pov, parent) { }
;

/*
 *   Sub-lister for listing the contents of a group.  This lister shows a
 *   simple list with no prefix or suffix, and otherwise uses the
 *   characteristics of the parent lister.  
 */
class GroupSublister: object
    construct(parentLister, parentGroup)
    {
        /* remember the parent lister and group objects */
        self.parentLister = parentLister;
        self.parentGroup = parentGroup;
    }

    /* no prefix or suffix */
    showListPrefixWide(itemCount, pov, parent) { }
    showListSuffixWide(itemCount, pov, parent) { }
    showListPrefixTall(itemCount, pov, parent) { }

    /* show nothing when empty */
    showListEmpty(pov, parent) { }

    /*
     *   Show an item in the list.  Rather than going through the parent
     *   lister directly, we go through the parent group, so that it can
     *   customize the display of items in the group.  
     */
    showListItem(obj, options, pov, infoTab)
    {
        /* ask the parent group to handle it */
        parentGroup.showGroupItem(parentLister, obj, options, pov, infoTab);
    }

    /*
     *   Show a counted item in the group.  As with showListItem, we ask
     *   the parent group to do the work, so that it can customize the
     *   display if desired.  
     */
    showListItemCounted(lst, options, pov, infoTab)
    {
        /* ask the parent group to handle it */
        parentGroup.showGroupItemCounted(
            parentLister, lst, options, pov, infoTab);
    }

    /* delegate everything we don't explicitly handle to our parent lister */
    propNotDefined(prop, [args])
    {
        return delegated (getTopLister()).(prop)(args...);
    }

    /* get the top-level lister - returns my parent's top-level lister */
    getTopLister() { return parentLister.getTopLister(); }

    /* my parent lister */
    parentLister = nil

    /* my parent list group */
    parentGroup = nil
;

/*
 *   Lister for objects in a room description with special descriptions. 
 */
specialDescLister: Lister
    /* list everything */
    isListed(obj) { return true; }

    /* show a list item */
    showListItem(obj, options, pov, infoTab)
    {
        /* add a paragraph break */
        "<.p>";

        /* show the object's special description */
        obj.showSpecialDescWithInfo(infoTab[obj]);
    }

    /* use the object's special description grouper */
    listWith(obj) { return obj.specialDescListWith; }

    /* we show no list separators */
    showListSeparator(options, curItemNum, totalItems) { }
;

/*
 *   Lister for actors present in a room, shown as part of the room's
 *   description.  We show each actor's actorHereDesc as a separate
 *   paragraph.  
 */
actorHereLister: Lister
    /* list only actors */
    isListed(obj) { return obj.isActor; }

    /* list an actor by showing its actorHereDesc */
    showListItem(obj, options, pov, infoTab)
    {
        /* add a paragraph break */
        "<.p>";

        /* show the actor's "I am here" description */
        obj.actorHereDesc;
    }

    /* use the actor's 'here' grouper */
    listWith(obj) { return obj.actorHereListWith; }

    /* we show no list separators */
    showListSeparator(options, curItemNum, totalItems) { }
;

/*
 *   Lister for actors within a nested room.  This is used to show the
 *   actors occupying a nested room when an actor looks at the nested room
 *   ("examine chair").  
 */
nestedActorLister: Lister
    showListPrefixWide(itemCount, pov, parent) { }
    showListSuffixWide(itemCount, pov, parent) { }

    /* list actors, even when marked as non-listed */
    isListed(obj) { return true; }

    /* show a list item */
    showListItem(obj, options, pov, infoTab)
        { obj.listActorPosture(pov); }

    /* use the same grouping as for a regular room description listing */
    listWith(obj) { return obj.actorHereListWith; }

    /* show no separator, since we use a complete sentence for each item */
    showListSeparator(options, curItemNum, totalCount) { }
;

/*
 *   Plain lister for actors.  This is the same as an ordinary
 *   plainLister, but ignores each object's isListed flag and lists it
 *   anyway. 
 */
plainActorLister: plainLister
    isListed(obj) { return true; }
;

/*
 *   Grouper for actors in a common posture in a common nested location.
 *   We create one of these per nested room per posture when we discover
 *   actors in nested rooms during "examine" on the nested rooms.  This
 *   grouper lets us group actors like so: "Dan and Jane are sitting on
 *   the couch."  
 */
class NestedActorGrouper: ListGroup
    construct(location, posture)
    {
        self.location = location;
        self.posture = posture;
    }
    
    showGroupList(pov, lister, lst, options, indent, infoTab)
    {
        /* if the location isn't in the sense table, skip the whole list */
        if (infoTab[location] == nil)
            return;

        /* create a sub-lister for the group */
        lister = createGroupSublister(lister);
        
        /* show the list prefix */
        posture.listActorGroupPrefix(location, lst.length());

        /* list the actors' names as a plain list */
        plainActorLister.showList(pov, location, lst, options,
                                  indent, infoTab, self);

        /* add the suffix mentioning that they're sitting here */
        posture.listActorGroupSuffix(location, lst.length());
    }
;


/* 
 *   Base class for inventory listers.  This lister uses a special listing
 *   method to show the items, so that items can be shown with special
 *   notations in an inventory list that might not appear in other types
 *   of listings.  
 */
class InventoryLister: Lister
    /* list items in inventory according to their isListedInInventory */
    isListed(obj) { return obj.isListedInInventory; }

    /*
     *   Show list items using the inventory name, which might differ from
     *   the regular nmae of the object.  
     */
    showListItem(obj, options, pov, infoTab)
    {
        obj.showInventoryItem(options, pov, infoTab);
    }

    showListItemCounted(lst, options, pov, infoTab)
    {
        lst[1].showInventoryItemCounted(lst, options, pov, infoTab);
    }

    /*
     *   Show contents of the items in the inventory.  We customize this
     *   so that we can differentiate inventory contents lists from other
     *   contents lists.  
     */
    showContentsList(pov, obj, options, indent, infoTab)
    {
        /* list the item's contents */
        obj.showInventoryContents(pov, obj.contentsLister, options,
                                  indent, infoTab);
    }

    /*
     *   Show the contents in-line, for an inventory listing. 
     */
    showInlineContentsList(pov, obj, options, indent, infoTab)
    {
        /* list the item's contents using its in-line lister */
        obj.showInventoryContents(pov, obj.inlineContentsLister,
                                  options, indent, infoTab);
    }
;

/*
 *   Base class for inventory lister for items being worn.  We use a
 *   special listing method to show these items, so that items being shown
 *   explicitly in a worn list can be shown differently from the way they
 *   would in a normal inventory list.  (For example, a worn item in a
 *   normal inventory list might show a "(worn)" indication, whereas it
 *   would not want to show a similar indication in a list of objects
 *   explicitly being worn.)  
 */
class WearingLister: Lister
    /* show the list item using the "worn listing" name */
    showListItem(obj, options, pov, infoTab)
        { obj.showWornItem(options, pov, infoTab); }
    showListItemCounted(lst, options, pov, infoTab)
        { lst[1].showWornItemCounted(lst, options, pov, infoTab); }
;

/*
 *   Base class for contents listers.  This is used to list the contents
 *   of the objects that appear in top-level lists (a top-level list is
 *   the list of objects directly in a room that appears in a room
 *   description, or the list of items being carried in an INVENTORY
 *   command, or the direct contents of an object being examined). 
 */
class ContentsLister: Lister
;

/*
 *   Base class for description contents listers.  This is used to list
 *   the contents of an object when we examine the object, or when we
 *   explicitly LOOK IN the object.  
 */
class DescContentsLister: Lister
    /* 
     *   Use the explicit look-in flag for listing contents.  We might
     *   list items within an object on explicit examination of the item
     *   that we wouldn't list in a room or inventory list containing the
     *   item. 
     */
    isListed(obj) { return obj.isListedInContents; }
;

/*
 *   Base class for sense listers, which list the items that can be sensed
 *   for a command like "listen" or "smell". 
 */
class SenseLister: Lister
    /* show everything we're asked to list */
    isListed(obj) { return true; }

    showListPrefixWide(itemCount, pov, parent) { }
    showListSuffixWide(itemCount, pov, parent) { }

    /* show a counted list item */
    showListItemCounted(lst, options, pov, infoTab)
    {
        /* 
         *   simply show one item, without the count - non-visual senses
         *   don't distinguish numbers of items that are equivalent 
         */
        showListItem(lst[1], options, pov, infoTab);
    }

    /* show no list separators */
    showListSeparator(options, curItemNum, totalItems) { }
;

/*
 *   Room contents lister for things that can be heard.  
 */
roomListenLister: SenseLister
    /* list an item in a room if its isSoundListedInRoom is true */
    isListed(obj) { return obj.isSoundListedInRoom; }

    /* list an item */
    showListItem(obj, options, pov, infoTab)
    {
        /* add a paragraph break */
        "<.p>";

        /* show the "listen" list name for the item */
        obj.soundHereDesc();
    }
;

/*
 *   Lister for explicit "listen" action 
 */
listenActionLister: roomListenLister
    /* list everything in response to an explicit general LISTEN command */
    isListed(obj) { return true; }

    /* show an empty list */
    showListEmpty(pov, parent)
    {
        mainReport(&nothingToHear);
    }

;

/*
 *   Room contents lister for things that can be smelled. 
 */
roomSmellLister: SenseLister
    /* list an item in a room if its isSmellListedInRoom is true */
    isListed(obj) { return obj.isSmellListedInRoom; }

    /* list an item */
    showListItem(obj, options, pov, infoTab)
    {
        /* add a paragraph break */
        "<.p>";

        /* show the "smell" list name for the item */
        obj.smellHereDesc();
    }
;

/*
 *   Lister for explicit "smell" action 
 */
smellActionLister: roomSmellLister
    /* list everything in response to an explicit general SMELL command */
    isListed(obj) { return true; }

    /* show an empty list */
    showListEmpty(pov, parent)
    {
        mainReport(&nothingToSmell);
    }

;

/*
 *   Inventory lister for things that can be heard.  
 */
inventoryListenLister: SenseLister
    /* list an item */
    showListItem(obj, options, pov, infoTab)
    {
        /* add a paragraph break */
        "<.p>";

        /* show the "listen" list name for the item */
        obj.soundHereDesc();
    }
;

/*
 *   Inventory lister for things that can be smelled. 
 */
inventorySmellLister: SenseLister
    /* list an item */
    showListItem(obj, options, pov, infoTab)
    {
        /* add a paragraph break */
        "<.p>";

        /* show the "smell" list name for the item */
        obj.smellHereDesc();
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   List Group Interface.  An instance of this object is created for each
 *   set of objects that are to be grouped together.  
 */
class ListGroup: object
    /*
     *   Show a list of items from this group.  All of the items in the
     *   list will be members of this list group; we are to display a
     *   sentence fragment showing the items in the list, suitable for
     *   embedding in a larger list.
     *   
     *   'options', 'indent', and 'infoTab' have the same meaning as they
     *   do for showList().
     *   
     *   Note that we are not to display any separator before or after our
     *   list; the caller is responsible for that.  
     */
    showGroupList(pov, lister, lst, options, indent, infoTab) { }

    /*
     *   Show an item in the group's sublist.  The sublister calls this to
     *   display each item in the group when the group calls the sublister
     *   to display the group list.  By default, we simply let the
     *   sublister handle the request, which gives items in our group
     *   sublist the same appearance they would have had in the sublist to
     *   begin with.  We can customize this behavior to give our list
     *   items a different appearance special to the group sublist.
     *   
     *   Note that the same customization could be accomplished by
     *   creating a specialized subclass of GroupSublister in
     *   createGroupSublister(), and overriding showListItem() in the
     *   specialized GroupSublister subclass.  We use this mechanism as a
     *   convenience, so that a separate group sublister class doesn't
     *   have to be created simply to customize the display of group
     *   items.  
     */
    showGroupItem(sublister, obj, options, pov, infoTab)
    {
        /* by default, list using the regular sublister */
        sublister.showListItem(obj, options, pov, infoTab);
    }

    /*
     *   Show a counted item in our group list.  This is the counted item
     *   equivalent of showGroupItem.  
     */
    showGroupItemCounted(sublister, lst, options, pov, infoTab)
    {
        /* by default, list using the regular sublister */
        sublister.showListItemCounted(lst, options, pov, infoTab);
    }

    /* 
     *   Determine if showing the group list will introduce a sublist into
     *   an enclosing list.  This should return true if we will show a
     *   sublist without some kind of grouping, so that the caller knows
     *   to introduce some extra grouping into its enclosing list.  This
     *   should return nil if the sublist we display will be clearly set
     *   off in some way (for example, by being enclosed in parentheses). 
     */
    groupDisplaysSublist = true

    /*
     *   Determine the cardinality of the group listing, grammatically
     *   speaking.  This is the number of items that the group seems to be
     *   for the purposes of grammatical agreement.  For example, if the
     *   group is displayed as "$1.38 in change", it would be singular for
     *   grammatical agreement, hence would return 1 here; if it displays
     *   "five coins (two copper, three gold)," it would count as five
     *   items for grammatical agreement.
     *   
     *   For languages (like English) that grammatically distinguish
     *   number only between singular and plural, it is sufficient for
     *   this to return 1 for singular and anything higher for plural.
     *   For the sake of languages that make more elaborate number
     *   distinctions for grammatical agreement, though, this should
     *   return as accurate a count as is possible.
     *   
     *   By default, we return the number of items to be displayed in the
     *   list group.  This should be overridden when necessary, such as
     *   when the group message is singular in usage even if the list has
     *   multiple items (as in "$1.38 in change").  
     */
    groupCardinality(lister, lst) { return lst.length(); }

    /*
     *   Create the group sublister.
     *   
     *   In most cases, when a group displays a list of the items in the
     *   group as a sublist, it will not want to use the same lister that
     *   was used to show the enclosing group, because the enclosing
     *   lister will usually have different prefix/suffix styles than the
     *   sublist.  However, the group list and the enclosing list might
     *   share many other attributes, such as the style of name to use
     *   when displaying items in the list.  The default sublister we
     *   create, GroupSublister, is a hybrid that uses the enclosing
     *   lister's attributes except for a few, such as the prefix and
     *   suffix, that usually need to be changed for the sublist.
     */
    createGroupSublister(parentLister)
    {
        /* create the standard group sublister by default */
        return new GroupSublister(parentLister, self);
    }
;

/*
 *   Sorted group list.  This is a list that simply displays its members
 *   in a specific sorting order. 
 */
class ListGroupSorted: ListGroup
    /*
     *   Show the group list 
     */
    showGroupList(pov, lister, lst, options, indent, infoTab)
    {
        /* put the list in sorted order */
        lst = sortListGroup(lst);

        /* create a sub-lister for the group */
        lister = createGroupSublister(lister);

        /* show the list */
        lister.showList(pov, nil, lst, options & ~ListContents,
                        indent, infoTab, self);
    }

    /*
     *   Sort the group list.  By default, if we have a
     *   compareGroupItems() method defined, we'll sort the list using
     *   that method; otherwise, we'll just return the list unchanged. 
     */
    sortListGroup(lst)
    {
        /* 
         *   if we have a compareGroupItems method, use it to sort the
         *   list; otherwise, just return the list in its current order 
         */
        if (propDefined(&compareGroupItems, PropDefAny))
            return lst.sort(SortAsc, {a, b: compareGroupItems(a, b)});
        else
            return lst;
    }

    /*
     *   Compare a pair of items from the group to determine their sorting
     *   order.  Returns 0 if the two items are at the same sorting order,
     *   1 if the first item sorts after the second item, -1 if the first
     *   item sorts before the second item.
     *   
     *   We do not provide a default implementation of this method, so
     *   that sortListGroup will know not to bother sorting the list at
     *   all if the subclass doesn't provide a definition for this method.
     */
    // compareGroupItems(a, b) { return a > b ? 1 : a == b ? 0 : -1; }
;

/*
 *   List Group implementation: parenthesized sublist.  Displays the
 *   number of items collectively, then displays the list of items in
 *   parentheses.
 *   
 *   Note that this is a ListGroupSorted subclass.  If our subclass
 *   defines a compareGroupItems() method, we'll show the list in the
 *   order specified by compareGroupItems().  
 */
class ListGroupParen: ListGroupSorted
    /* 
     *   show the group list 
     */
    showGroupList(pov, lister, lst, options, indent, infoTab)
    {
        /* sort the list group, if we have an ordering method defined */
        lst = sortListGroup(lst);

        /* create a sub-lister for the group */
        lister = createGroupSublister(lister);

        /* show the collective count of the object */
        showGroupCountName(lst);

        /* show the tall or wide sublist */
        if ((options & ListTall) != 0)
        {
            /* tall list - show the items as a sublist */
            "\n";
            lister.showList(pov, nil, lst, options & ~ListContents,
                            indent, infoTab, self);
        }
        else
        {
            /* wide list - add a space and a paren for the sublist */
            " (";

            /* show the sublist */
            lister.showList(pov, nil, lst, options & ~ListContents,
                            indent, infoTab, self);

            /* end the sublist */
            ")";
        }
    }

    /*
     *   Show the collective count for the list of objects.  By default,
     *   we'll simply display the countName of the first item in the list,
     *   on the assumption that each object has the same plural
     *   description.  However, in most cases this should be overridden to
     *   provide a more general collective name for the group. 
     */
    showGroupCountName(lst)
    {
        /* show the first item's countName */
        say(lst[1].countName(lst.length()));
    }

    /* we don't add a sublist, since we enclose our list in parentheses */
    groupDisplaysSublist = nil
;

/*
 *   List Group implementation: simple prefix/suffix lister.  Shows a
 *   prefix message, then shows the list, then shows a suffix message.
 *   
 *   Note that this is a ListGroupSorted subclass.  If our subclass
 *   defines a compareGroupItems() method, we'll show the list in the
 *   order specified by compareGroupItems().  
 */
class ListGroupPrefixSuffix: ListGroupSorted
    showGroupList(pov, lister, lst, options, indent, infoTab)
    {
        /* sort the list group, if we have an ordering method defined */
        lst = sortListGroup(lst);

        /* create a sub-lister for the group */
        lister = createGroupSublister(lister);

        /* show the prefix */
        showGroupPrefix;

        /* if we're in tall mode, start a new line */
        showTallListNewline(options);

        /* show the list */
        lister.showList(pov, nil, lst, options & ~ListContents,
                        indent + 1, infoTab, self);

        /* show the suffix */
        showGroupSuffix;
    }

    /* the prefix - subclasses should override this if desired */
    showGroupPrefix = ""

    /* the suffix */
    showGroupSuffix = ""
;

/*
 *   Equivalent object list group.  This is the default listing group for
 *   equivalent items.  The Thing class creates an instance of this class
 *   during initialization for each set of equivalent items.  
 */
class ListGroupEquivalent: ListGroup
    showGroupList(pov, lister, lst, options, indent, infoTab)
    {
        /* show a count of the items */
        lister.showListItemCounted(lst, options, pov, infoTab);
    }

    /* we display as a single item, so there's no sublist */
    groupDisplaysSublist = nil
;

